/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.HybsDateUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;

// import java.text.SimpleDateFormat;
import java.util.Calendar;
// import java.util.Locale;

/**
 * DATE レンデラーは、カラムのデータを表示パラメータで指定されたフォーマットで
 * 日付表示する場合に使用するクラスです。
 * 6桁未満のデータの場合はフォーマットをかけずにそのまま出力します。
 *
 * 表示パラメータで指定されたフォーマットを、java.text.SimpleDateFormat で
 * 解析してフォーマット変換します。
 * フォーマットルールは、{@link java.text.SimpleDateFormat} を参照願います。
 * フォーマット初期値はシステムリソースのSYS_TIMEです。
 *
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 3.5.0.0 (2003/09/17) 新規作成
 * @og.rev 5.4.3.6 (2012/01/19) コメント修正
 * @og.group データ表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Renderer_DATE extends AbstractRenderer {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.5.7.2 (2012/10/09)" ;

//	private final SimpleDateFormat format ;
	private final String	format ;			// 5.5.7.2 (2012/10/09) HybsDateUtil を利用
	private final String	defValue ;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 */
	public Renderer_DATE() {
		format		= null;
		defValue	= null;
	}

	/**
	 * 引数つきコンストラクター。
	 *
	 * @og.rev 3.5.1.0 (2003/10/03) 初期値が nullの場合は、""、８桁以下はそのまま返す。
	 * @og.rev 3.8.5.3 (2006/08/01) 日付を６桁以上 とし、それ以下はそのまま返す。
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Renderer_DATE( final DBColumn clm ) {

		String fm = clm.getRendererParam();
		if( fm == null || fm.length() == 0 || fm.equals( "_" ) ) {
			fm = HybsSystem.sys( "SYS_TIME" );
		}
//		format = new SimpleDateFormat( fm,Locale.JAPAN );
		format = fm;			// 5.5.7.2 (2012/10/09) HybsDateUtil を利用

		String defv = clm.getDefault();
		if( defv == null || defv.equals( "_" ) ) {
			defValue = "";
		}
		else if( defv.length() < 6 ) {
			defValue = defv;
		}
		else {
			Calendar cal = HybsDateUtil.getCalendar( defv );
			defValue = HybsDateUtil.getDate( cal.getTimeInMillis() , format );

//			defValue = format.format( getCalendar( defv ).getTime() );
		}
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellRendererオブジェクト
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		return new Renderer_DATE( clm );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 3.5.1.0 (2003/10/03) データが nullの場合は、デフォルト値を、８桁以下はそのまま返す。
	 * @og.rev 3.8.5.3 (2006/08/01) 日付を６桁以上 とし、それ以下はそのまま返す。
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 *
	 * @param	value 入力値
	 *
	 * @return	データの表示用文字列
	 */
	@Override
	public String getValue( final String value ) {
		if( value == null ) { return defValue; }
		if( value.length() < 6 ) { return value; }

		Calendar cal = HybsDateUtil.getCalendar( value );
		return HybsDateUtil.getDate( cal.getTimeInMillis() , format );

//		final String rtn ;
//		synchronized( format ) {
//			rtn = format.format( getCalendar( value ).getTime() );
//		}
//
//		return rtn;
	}

	/**
	 * 引数の日付け文字列に応じた、カレンダオブジェクトを返します。
	 *
	 * 引数は、YYYYMMDD形式(８Byte)か、YYYYMMDDHHMMSS形式(１４Byte)の
	 * いづれかです。それ以外の場合は、初期化データが返されます。
	 *
	 * @og.rev 3.5.1.0 (2003/10/03) データが null や、""、の場合の処理を削除する。
	 * @og.rev 3.8.5.3 (2006/08/01) 日付を６桁以上 とし、それ以下はそのまま返す。
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するので廃止する。
	 *
	 * @param	value	カレンダオブジェクトを作成する、日付け文字列
	 *
	 * @return	カレンダオブジェクト
	 */
// 	private Calendar getCalendar( final String value ) {
// 		Calendar cal = Calendar.getInstance();
// 		cal.clear();
// 
// 		int hour=0,minute=0,second=0,date=1;
// 
// 		int year   = Integer.parseInt( value.substring( 0,4 ) );
// 		int month  = Integer.parseInt( value.substring( 4,6 ) ) - 1;
// 		if( value.length() >= 8 ) {
// 			date   = Integer.parseInt( value.substring( 6,8 ) );
// 		}
// 
// 		if( value.length() == 14 ) {
// 			hour   = Integer.parseInt( value.substring( 8,10 ) );
// 			minute = Integer.parseInt( value.substring( 10,12 ) );
// 			second = Integer.parseInt( value.substring( 12,14 ) );
// 		}
// 
// 		cal.set( year,month,date,hour,minute,second );
// 
// 		return cal;
// 	}
}
