/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.mail;


import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;

import org.opengion.fukurou.db.DBUtil;
import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.db.TransactionReal;
import org.opengion.fukurou.db.ConnectionFactory;
import org.opengion.fukurou.db.DBFunctionName;
import org.opengion.fukurou.util.ApplicationInfo;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;

/**
 * メールモジュール関係の機能の一部を他から使用するためのクラスです。
 * 
 * ※MailSenderTagからGE32,34へ履歴を出力する機能を追加する際に、モジュール系の動作を本パッケージに集約しておくために作成。
 *   必要としている箇所のみ実装。
 *   
 * @og.rev 5.9.2.3 (2015/11/27) 新規作成
 *
 * @og.group メールモジュール
 *
 * @version  4.0
 * @author   Takahashi Masakazu
 * @since    JDK1.6
 */
public class MailModuleUtil {

	//  Ver4互換モード対応
	private static final String CONTENTS = HybsSystem.sysBool( "VER4_COMPATIBLE_MODE" ) ? "CONTENT" : "CONTENTS";

//	private static final String	selYkno	= "SELECT GE32S02.NEXTVAL YKNO FROM DUAL";
	private static final String	selYkno		= "GE32S02";
	private static final String	insGE32		= "INSERT INTO GE32(YKNO,PARA_KEY,PTN_ID,FROM_ADDR,TITLE,"+CONTENTS+",ATTACH1,ATTACH2,ATTACH3,ATTACH4,ATTACH5,DYSET,USRSET,PGUPD,SYSTEM_ID,FGJ)"
											+ " VALUES(?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,'1')";
	private static final String insGE34		= "INSERT INTO GE34(YKNO,DST_ID,GROUP_ID,DST_NAME,DST_ADDR,DST_KBN,FGJ,DYSET,USRSET,PGUPD)"
											+ " VALUES(?,?,?,?,?,?,?,?,?,?)";

	// 内部データのカラム番号(履歴テーブル)
	private static final int GE32_YKNO		= 0 ;
	private static final int GE32_PARAKEY	= 1 ;
	private static final int GE32_PTN_ID 	= 2;
	private static final int GE32_FROM_ADDR	= 3;
	private static final int GE32_TITLE 	= 4;
	private static final int GE32_CONTENTS	= 5;		
	private static final int GE32_ATTACH1	= 6;
	private static final int GE32_ATTACH2	= 7;
	private static final int GE32_ATTACH3	= 8;
	private static final int GE32_ATTACH4	= 9;
	private static final int GE32_ATTACH5	= 10;
	private static final int GE32_DYSET		= 11;
	private static final int GE32_USRSET	= 12;
	private static final int GE32_PGUPD	= 13;
	private static final int GE32_SYSTEM_ID	= 14;
	// 内部データのカラム番号(履歴テーブル)
	private static final int GE34_YKNO	 	= 0 ;
	private static final int GE34_DST_ID 	= 1 ;
	private static final int GE34_GROUP_ID 	= 2 ;
	private static final int GE34_DST_NAME	= 3 ;
	private static final int GE34_DST_ADDR	= 4 ;
	private static final int GE34_DST_KBN	= 5 ;
	private static final int GE34_FGJ		= 6 ;
	private static final int GE34_DYSET		= 7 ;
	private static final int GE34_USRSET	= 8 ;
	private static final int GE34_PGUPD	= 9 ;
	
	// アドレスマップ
	private static final int IDX_DST_ADDR	= 0;
	private static final int IDX_DST_KBN	= 1;
	
	/** メール送信区分 {@value} */
	private static final int KBN_TO			= 0 ;	// メール送信区分(TO)
	/** メール送信区分 {@value} */
	private static final int KBN_CC			= 1 ;	// メール送信区分(CC)
	/** メール送信区分 {@value} */
	private static final int KBN_BCC			= 2 ;	// メール送信区分(BCC)

	private Map<String, String[]>	mailDstMap		= null;
	private Map<String,String>		initParamMap	= null;		// パラメータマップ

	protected final String DBID = HybsSystem.sys( "RESOURCE_DBID" );
	protected final DBFunctionName dbName = DBFunctionName.getDBName( ConnectionFactory.getDBName( DBID ) ); // 5.9.31.1 (2018/04/13)

	/** コネクションにアプリケーション情報を追記するかどうか指定 */
	private static final boolean USE_DB_APPLICATION_INFO  = HybsSystem.sysBool( "USE_DB_APPLICATION_INFO" ) ;

	/** アプリケーション情報 */
	public static final ApplicationInfo appInfo;
	static {
		if( USE_DB_APPLICATION_INFO ) {
			appInfo = new ApplicationInfo();
			// ユーザーID,IPアドレス,ホスト名
			appInfo.setClientInfo( "MailModuel", HybsSystem.HOST_ADRS, HybsSystem.HOST_NAME );
			// 画面ID,操作,プログラムID
			appInfo.setModuleInfo( "MailModuel", "MailManager", "MailManager" );
		}
		else {
			appInfo = null;
		}
	}


	/**
	 * 履歴テーブル(GE32)と宛先テーブル(GE34)に登録します。
	 * 登録時に、桁数オーバーにならないように、テーブル定義の桁数を上限として、
	 * 登録前に各項目の桁数整理を行います。
	 * 
	 * @og.rev 5.9.3.0 (2015/12/04) 添付ファイル対応
	 *
	 */
	public void commitMailDB(){
		// 履歴テーブルの追加
		String[] insGE32Args = new String[15];
		String ykno = getYkno();
		String[] attachFiles = StringUtil.csv2Array( initParamMap.get( "FILES" )); // 5.9.3.0
		
		insGE32Args[GE32_YKNO]		= ykno;
		insGE32Args[GE32_PARAKEY] 	= initParamMap.get( "PARAKEY" );
		insGE32Args[GE32_PTN_ID] 	= trim( initParamMap.get( "PTN_ID" ), 20 );
		insGE32Args[GE32_FROM_ADDR] = trim( initParamMap.get( "FROM" ), 100);
		insGE32Args[GE32_TITLE] 	= trim( initParamMap.get( "TITLE" ), 300);
		insGE32Args[GE32_CONTENTS] 	= initParamMap.get( "CONTENT" );	
//		insGE32Args[GE32_ATTACH1] 	= "";
//		insGE32Args[GE32_ATTACH2] 	= "";
//		insGE32Args[GE32_ATTACH3] 	= "";
//		insGE32Args[GE32_ATTACH4] 	= "";
//		insGE32Args[GE32_ATTACH5] 	= "";
		// 5.9.3.0
		if ( attachFiles != null ) { 
			int attSize = attachFiles.length;
			for( int i = 0; i < attSize; i++ ) {
				insGE32Args[6 + i] = trim( attachFiles[i], 256);
			}
		}
		
		insGE32Args[GE32_DYSET]  = HybsSystem.getDate( "yyyyMMddHHmmss" );
		insGE32Args[GE32_USRSET] = initParamMap.get( "LOGIN_USERID" );
		insGE32Args[GE32_PGUPD] = initParamMap.get( "PGID" );
		insGE32Args[GE32_SYSTEM_ID] = initParamMap.get( "SYSTEM_ID" );
		DBUtil.dbExecute( insGE32, insGE32Args, appInfo, DBID );

		// 宛先テーブル追加
		String[] insGE34Args = new String[10];
		insGE34Args[GE34_YKNO]= ykno;
		for( String dstId : mailDstMap.keySet() ) {
			insGE34Args[GE34_DST_ID] 	= trim( mailDstMap.get( dstId )[IDX_DST_ADDR], 10 );
			insGE34Args[GE34_GROUP_ID] 	= "";
			insGE34Args[GE34_DST_NAME] 	= "";
			insGE34Args[GE34_DST_ADDR] 	= trim( mailDstMap.get( dstId )[IDX_DST_ADDR], 100 );
			insGE34Args[GE34_DST_KBN] 	= mailDstMap.get( dstId )[IDX_DST_KBN];
			insGE34Args[GE34_FGJ] 		= "1";
			insGE34Args[GE34_DYSET] 	= HybsSystem.getDate( "yyyyMMddHHmmss" );
			insGE34Args[GE34_USRSET] 	= initParamMap.get( "LOGIN_USERID" );
			insGE34Args[GE34_PGUPD] 	= initParamMap.get( "PGID" );
			DBUtil.dbExecute( insGE34, insGE34Args, appInfo, DBID );
		}
	}

	/**
	 * パラメータマップをセットします。
	 *
	 * @param	params	パラメータのマップ
	 */
//	public void setInitParams( final Map<String, String> params ) {
//		initParamMap = params;
//	}
	
	/**
	 * パラメータからマップをセットします。
	 * 
	 * @og.rev 5.9.3.0 (2015/11/30) files追加
	 * 
	 * @param systemId	 システムID
	 * @param from 		FROMアドレス
	 * @param tos 		TOアドレス（カンマ区切り）
	 * @param ccs 		CCアドレス（カンマ区切り）
	 * @param bccs 		BCCアドレス（カンマ区切り）
	 * @param content 	本文
	 * @param title 	タイトル
	 * @param userid 	登録ユーザ
	 * @param pgid 		登録PG
	 * @param files 	添付ファイル
	 *
	 */
	public void setInitParams( final String systemId, final String from, final String[] tos, final String[] ccs
								,final String[] bccs, final String content, final String title, final String userid, final String pgid
								,final String[] files) {
		Map<String,String>   paramMap = new HashMap<String,String>();;
		
		paramMap.put( "SYSTEM_ID", systemId    );
		paramMap.put( "FROM"   , from );
		paramMap.put( "TO"     , StringUtil.array2csv( tos )   );
		paramMap.put( "CC"     , StringUtil.array2csv( ccs )   );
		paramMap.put( "BCC"    , StringUtil.array2csv( bccs )  );
		paramMap.put( "CONTENT"    , content  );
		paramMap.put( "TITLE"    , title  );
		paramMap.put( "DATE", HybsSystem.getDate("yyyy/MM/dd") );
		paramMap.put( "TIME", HybsSystem.getDate("HH:mm:ss") );
		paramMap.put( "LOGIN_USERID", userid );
		paramMap.put( "PGID", pgid );
		paramMap.put( "FILES", StringUtil.array2csv( files ) ); // 5.9.3.0 (2015/12/04)
		
		getDstMap(tos, ccs, bccs  );
		
		initParamMap = paramMap;
	}

	/**
	 * 指定の長さ以内の文字列を返します。
	 *
	 * @param	src		オリジナルの文字列
	 * @param	maxLen	指定の長さ
	 *
	 * @return	指定の長さに短縮された文字列
	 */
	private String trim( final String src, final int maxLen ) {
		String rtn = src;
		if( src != null && src.length() > maxLen ) {
//			rtn = src.substring( 0, maxLen );
			rtn = StringUtil.cut( src, maxLen );
		}
		return rtn;
	}

	/**
	 * 要求NOを採番します。
	 * この要求NOで履歴テーブル(GE32)と宛先テーブル(GE30)の関連付けを持たせます。
	 * 
	 * @og.rev 5.9.31.1 (2018/04/13) シーケンスの取り方変更
	 *
	 * @return	要求NO
	 */
	private String getYkno() {
//		String[][] tmp = DBUtil.dbExecute( selYkno, new String[0], appInfo, DBID );
//		if( tmp == null || tmp.length == 0 ) {
//			String errMsg = "要求NO採番エラー"
//						+ " SQL=" + selYkno ;
//			throw new RuntimeException( errMsg );
//		}
		Transaction tran = new TransactionReal( appInfo );
		int rtn_ykno = dbName.getSequence(selYkno,tran,DBID);
		return Integer.toString( rtn_ykno );
	}
	
	/**
	 * 送信先のアドレスをセットします。
	 *
	 * @param toId
	 * @param ccId
	 * @param bccId
	 */
	private void getDstMap( final String[] toId, final String[] ccId, final String[] bccId  ){

		// 送信先(TO、CC、BCC)のマップを作成します。
		mailDstMap =  (TreeMap<String, String[]>)getDstAddrMap( bccId, KBN_BCC );
		mailDstMap.putAll( (TreeMap<String, String[]>) getDstAddrMap( ccId, KBN_CC ) );
		mailDstMap.putAll( (TreeMap<String, String[]>) getDstAddrMap( toId, KBN_TO ) );
	}
	
	/**
	 * 送信先のアドレス・マップを作成します。
	 *
	 * @og.rev 5.9.33.0 (2018/06/01) dstBufがnullの場合の処理
	 *
	 * @param	dstBuf	送信先配列
	 * @param	kbn		送信区分[0:TO/1:CC/2:BCC]
	 *
	 * @return	送信先のアドレス・マップ
	 */
	private Map<String, String[]> getDstAddrMap( final String[] dstBuf, final int kbn ){
		Map<String,String[]> dstMap= new TreeMap<String,String[]>();
		 // IDX_DST_ADDR ,IDX_DST_KBN
		if( dstBuf != null ) { // 5.9.33.0 (2018/06/01)
			String[] dstInit = { "", Integer.toString( kbn ) };
			
			int len = dstBuf.length;
			 for( int i=0; i < len; i++ ){
				String[] indMember = dstInit.clone();
				indMember[IDX_DST_ADDR] = dstBuf[i]; 			//メールアドレス
	
				dstMap.put( dstBuf[i], indMember );
			}
		}
		 return dstMap;
	}

}
