/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

import javax.mail.internet.MimeUtility;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.opengion.fukurou.util.LogWriter;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.resource.GUIInfo;

/**
 * 制御を別の画面にフォワードする場合に使用するタグです(通常は、entry.jsp で使用します)。
 *
 * フォワード時に、指定の引数をセットできます。
 *
 * @og.formSample
 * ●形式：&lt;og:forward page="…" gamenId="[…]" /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:forward
 *       gamenId            【TAG】gamenId 属性を登録します
 *       page             ○【TAG】転送先URLを指定します(必須)。
 *       keys               【TAG】キーを CSV 形式でセットします
 *       vals               【TAG】キーに対応する値を CSV 形式でセットします
 *       dbkeys             【TAG】DBキーをCSV 形式でセットします
 *       scope              【TAG】キャッシュする場合のスコープ[request/page/session/applicaton]を指定します(初期値:session)
 *       useRedirect        【TAG】response.sendRedirect するかどうか[true/false]を指定します(初期値:false)
 *       target             【TAG】サブミット先の文書を表示させるフレーム、またはウィンドウの名前を指定します
 *       filename           【TAG】ファイル転送時にファイル名の別名を指定します(初期値:null)
 *       noTransition       【TAG】(通常は使いません)画面遷移を行わない形式の登録方法を使用するかを指定します
 *       useAjaxSubmit      【TAG】(通常は使いません)ajaxを利用したsubmitを利用するかどうかを指定します
 *       redirectEncode     【TAG】(通常は使いません)useRedirect=true時に 日本語を含む転送先(page)をバイナリ変換するときのコード名[UTF-8/Windows-31J]など(初期値:null)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *       &lt;og:forward
 *          page          = "転送先URLを指定します(必須)。"
 *          dbkeys        = "DBキーをCSV 形式でセットします。"
 *          keys          = "キーを CSV 形式でセットします。"
 *          vals          = "キーに対応する値を CSV 形式でセットします。"
 *          scope         = "キャッシュする場合のスコープ(request,page,session,applicaton)を指定します(初期値:session)。"
 *          gamenId       = "gamenId 属性を登録します。"
 *          useRedirect   = "useRedirect 属性を登録します。"
 *          filename      = "ファイル転送時にファイル名を指定する場合に設定します。"
 *          target        = "taregt 属性を登録します。"
 *          noTransition  = "実際にforwardせずに、id="forwardUrl"属性のbody部分に遷移先のURLを出力します。(画面遷移なしモード時に使用します)"
 *       /&gt;
 *
 * @og.group 画面制御
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ForwardTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.7.4.1 (2014/03/14)" ;

	private static final long serialVersionUID = 574120140314L ;

	private	transient DBTableModel	table 	= null;
	private	String		tableId		= HybsSystem.TBL_MDL_KEY;
	private	String		page		= null;
	private	String		keys		= null;
	private	String		dbkeys		= null;
	private	String[]	vals		= null;
	private	String		gamenId		= null;		// 3.5.4.9 (2004/02/25) 追加
	private	boolean		useRedirect = false;	// 3.5.6.6 (2004/08/23) 追加
	private String		target		= null;		// 3.8.0.4 (2005/08/08) 追加
	private String		filename	= null;		// 4.0.0 (2005/11/30) 追加
	private static final String		disposition = "inline";	// 固定
	private boolean 	noTransition= false;	// 4.3.3.0 (2008/10/01) 追加
	private boolean 	useAjaxSubmit		= false;	// 4.3.8.0 (2009/08/01) 追加
//	private boolean 	useEncodeRedirect	= true;		// 5.6.5.0 (2013/06/07) 追加、即廃止
	private String	 	redirectEncode	= null;			// 5.6.5.0 (2013/06/07) 追加

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 2.1.0.0 (2002/10/11) 戻り値の先頭の "?" を追加しないように変更。
	 * @og.rev 3.0.1.3 (2003/03/11) 直接転送した場合の整合性を取る為の対応。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.1.2.0 (2003/04/07) ソースコード中の固定値を、定義ファイルを使用するように変更する。
	 * @og.rev 3.5.4.9 (2004/02/25) gamenId を指定して、フォワードできるようにします。
	 * @og.rev 3.5.6.6 (2004/08/23) useRedirect 属性を追加します。
	 * @og.rev 4.3.3.0 (2008/10/01) noTransition 属性を追加します。
	 * @og.rev 4.3.7.1 (2009/06/08) target属性に絶対パス及び"_"で始まる予約語を指定できるようにします。
	 * @og.rev 4.3.8.0 (2009/08/01) useAjaxSubmit対応、noTransition値取得メソッド名変更
	 * @og.rev 5.0.0.4 (2009/10/01) 画面遷移なし登録で戻った際に最終行が選択されるバグを修正
	 * @og.rev 5.1.3.0 (2010/02/01) noTransition、ajaxSubmitのコントロールは、requestで行う。
	 * @og.rev 5.1.6.0 (2010/05/01) IE以外は、MIMEエンコードする。(RFC2231に従う)
	 * @og.rev 5.1.7.0 (2010/06/01) 画面IDをパラメーターに付加する。
	 * @og.rev 5.1.7.0 (2010/06/01) 遷移先がJSP以外の場合は、フォワードページのキャッシュをしない。
	 * @og.rev 5.1.8.0 (2010/07/01) httpが指定されていた場合に、自身の画面IDが付加されるバグを修正
	 * @og.rev 5.3.8.0 (2011/08/01) ファイル名指定でIEの場合、URLエンコードすると途中で切れるため(IE7のバグ)、Shift_JIS(WIndows-31J)で直接指定する。
	 * @og.rev 5.6.5.0 (2013/06/07) redirectEncode 属性を追加します。
	 * @og.rev 5.7.4.1 (2014/03/14) 日本語ファイル名が正しく処理できない件を修正(IE11のMSIE関連)
	 * @og.rev 5.9.29.0 (2018/01/30) Edge,safariの日本語ファイル名対応(RFC6266方式を併記）
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		// noTransition = isNoTransitionRequest() || noTransition; // 4.3.3.0 (2008/10/01) 追加
//		noTransition = isNoTransitionSession() || noTransition; // 4.3.8.0 (2009/08/01) メソッド名変更
//		useAjaxSubmit	 = isAjaxSubmitSession() || useAjaxSubmit; // 4.3.8.0 (2009/08/01) 追加
		noTransition = isNoTransitionRequest(); // 5.1.3.0 (2010/02/01)
		useAjaxSubmit = isAjaxSubmitRequest(); // 5.1.3.0 (2010/02/01)

		String urlEnc = getUrlEncode();
		String fwdUrl = makeGamenLink();		// 3.5.4.9 (2004/02/25)

		HttpServletResponse response = (HttpServletResponse)pageContext.getResponse();
		try {
			String url = response.encodeRedirectURL( fwdUrl );	// 3.5.4.9 (2004/02/25)

			if( filename != null ) {
				// 5.1.6.0 (2010/05/01) IE以外は、MIMEエンコードする。(RFC2231に従う)
//				if( ( (HttpServletRequest)pageContext.getRequest() ).getHeader( "User-Agent" ).indexOf( "MSIE" ) == -1 ) {
//					filename = MimeUtility.encodeWord( filename, "UTF-8", "B" );
//				}
//				else {
//					// 5.3.8.0 (2011/08/01) IEの場合、URLエンコードすると途中で切れるため(IE7のバグ)、Shift_JIS(WIndows-31J)で直接指定する。
////					filename = StringUtil.urlEncode( filename );
//					filename = new String( filename.getBytes("Windows-31J"), "ISO-8859-1" );
//				}
				String newFilename	  = filename;
				String newFilenameEnc = StringUtil.urlEncode( newFilename );  // 5.9.29.0 (2018/02/02)
				
				// 5.7.4.1 (2014/03/14) 日本語ファイル名が正しく処理できない件を修正(IE11のMSIE関連)
				// 条件を反転させた上でIE11対応を行う
				String reqHeader = ((HttpServletRequest)pageContext.getRequest()).getHeader( "User-Agent" );
				if( reqHeader.indexOf( "MSIE" ) >= 0 || reqHeader.indexOf( "Trident" ) >= 0 ) {
					filename = new String( filename.getBytes("Windows-31J"), "ISO-8859-1" );
				}
				else {
					filename = MimeUtility.encodeWord( filename, "UTF-8", "B" );
				}

//				(response).setHeader( "Content-Disposition",disposition + "; filename=\"" + filename + "\"" );
				(response).setHeader( "Content-disposition",disposition + "; filename=\"" + newFilename + "\"; "
						+ "filename*=UTF-8''" + newFilenameEnc ); // 5.9.29.0 (2018/02/02)　RFC6266方式を併記
			}
			// 5.1.7.0 (2010/06/01) 画面IDをパラメーターに付加する。
			// 5.1.8.0 (2010/07/01) httpが指定されていた場合に、自身の画面IDが付加されるバグを修正
			else {
//				if( page.endsWith( ".jsp" ) ) {
				if( !page.startsWith( "http" ) && page.endsWith( ".jsp" ) ) {
					if( keys == null || keys.length() == 0 || ( "," + keys + "," ).indexOf( ",GAMENID," ) < 0 ) {
						if( gamenId != null && gamenId.length() > 0 ) {
							url = XHTMLTag.addUrlEncode( url, "GAMENID=" + gamenId );
						}
						// useRedirect=trueでGAMENIDを指定せず、../GF9100/index.jsp等と書かれている場合の対応
						else if( !useRedirect ) {
							url = XHTMLTag.addUrlEncode( url, "GAMENID=" + getGUIInfoAttri( "KEY" ) );
						}
					}
				}
			}

			// 3.5.6.6 (2004/08/23) useRedirect 属性を追加
			String url2 = XHTMLTag.addUrlEncode( url , urlEnc );
			// 4.3.3.0 (2008/10/01) 追加
			// if( noTransition ) {
			if( noTransition || useAjaxSubmit ) { // 4.3.8.0 (2009/08/01) useAjaxSubmitと共用
				// 5.0.0.4 (2009/10/01)
				if( noTransition ) {
					int selRow = nval( getRequestValue( "SEL_ROW" ),-1 );
					if( selRow > 0 ) {
						url2 = XHTMLTag.addUrlEncode( url2, "SEL_ROW=" + selRow );
					}
				}
				jspPrint( "<div id=\"noTransitionUrl\">" + url2 + "</div>" );
				jspPrint( "<div id=\"noTransitionTarget\">" + target + "</div>" );
			}
			// 3.8.0.4 (2005/08/08) target 属性を追加します。
//			else if( target != null ) {
			else if( target != null && target.length() > 0 ) {
				// 4.3.7.1 (2009/06/08)
				String src = null;
				if( target.startsWith( "_b" ) ) { // "_blank"
					src = "window.open(\"" + url2 + "\", \"_blank\") ";
				}
				else {
					String frameTop = "";
					if( target.indexOf('.') < 0 ) { frameTop = "top."; }
					src = frameTop + target + ".location.href=\"" + url2 + "\"" + HybsSystem.CR;
				}
//				String script = "<script language=\"JavaScript\" ><!-- " + HybsSystem.CR
				String script = "<script type=\"text/javascript\" ><!-- " + HybsSystem.CR
								+ src + HybsSystem.CR
							+ "// --></script>" ;
				jspPrint( script );
			}
			else if( useRedirect ) {
				// 5.6.5.0 (2013/06/07) redirectEncode 属性追加
				if( redirectEncode != null ) {
					url2 = new String( url2.getBytes( redirectEncode ), "ISO-8859-1" );
				}
				response.sendRedirect( url2  );
			}
			else {
				pageContext.forward( url2 );
			}
		} catch(IOException ex) {
			String errMsg = "フォワード/リダイレクトでIOエラーが発生しました。"
								+ HybsSystem.CR
								+ fwdUrl + "?" + urlEnc ;
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		} catch( ServletException ex) {
			String errMsg = "フォワード/リダイレクトでServletエラーが発生しました。"
								+ HybsSystem.CR
								+ fwdUrl + "?" + urlEnc ;
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}

		// 3.0.1.3 (2003/03/11) forward で直接転送した場合の整合性を取る為の対応。
		// 5.1.7.0 (2010/06/01) 遷移先がJSP以外の場合は、フォワードページのキャッシュをしない。
		if( fwdUrl != null && fwdUrl.endsWith( ".jsp" ) ) {
			setSessionAttribute( HybsSystem.FORWARD_PAGE_KEY, fwdUrl );	// 3.5.4.9 (2004/02/25)
		}

		return(SKIP_PAGE);		// ページの残りの処理を行わない。
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.4.9 (2004/02/25) 画面ID属性(gamenId)を追加
	 * @og.rev 3.8.5.0 (2006/03/06) filename 属性を追加します。
	 * @og.rev 4.3.3.0 (2008/10/01) noTransition 属性を追加します。
	 * @og.rev 5.6.5.0 (2013/06/07) redirectEncode 属性を追加します。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		tableId		= HybsSystem.TBL_MDL_KEY;
		page		= null;
		keys		= null;
		dbkeys		= null;
		vals		= null;
		table 		= null;
		gamenId		= null;
		useRedirect = false;	// 3.5.6.6 (2004/08/23) 追加
		target		= null;		// 3.8.0.4 (2005/08/08) 追加
		filename	= null;
		noTransition= false;	// 4.3.3.0 (2008/10/01) 追加
		useAjaxSubmit		= false;	// 4.3.8.0 (2009/08/01) 追加
		redirectEncode	= null;			// 5.6.5.0 (2013/06/07) 追加
	}

	/**
	 * keys 属性 を元に、request情報より values の値を取り込む。
	 *
	 * keys属性は キー情報がカンマ区切りになっている為,ばらして
	 * values属性の配列に一つづつ設定していきます。
	 *
	 * @og.rev 3.1.7.0 (2003/05/02) 戻るボタン用のBACK_GAMENID を、引数に追加します。
	 * @og.rev 3.1.8.0 (2003/05/16) 戻るボタン用のBACK_GAMENID の自動登録を廃止します。
	 * @og.rev 4.0.0.0 (2005/01/31) getParameterRows() を使用するように変更
	 *
	 * @return	URLエンコードされた文字列
	 */
	private String getUrlEncode() {
		String[] dbkey = StringUtil.csv2Array( dbkeys );
		String[] key   = StringUtil.csv2Array( keys );

//		int dbCount = (dbkey != null) ? dbkey.length : 0 ;
//		int count   = (key   != null) ? key.length   : 0 ;
		int dbCount = dbkey.length;
		int count   = key.length  ;

		String[] allKey = new String[ dbCount + count ];
		String[] allval = new String[ dbCount + count ];

		int[] rowNo = getParameterRows();		// 4.0.0 (2005/01/31)
		if( dbCount > 0 && dbkey[0] != null && dbkey[0].length() > 0 && rowNo.length > 0 ) {
			table = (DBTableModel)getObject( tableId );
			if( table != null ) {
				for( int i=0; i<dbCount; i++ ) {
					allKey[i] = dbkey[i];
					allval[i] = table.getValue(rowNo[0],table.getColumnNo( dbkey[i] ));
				}
			}
		}

		if( count > 0 ) {
			for( int i=0; i<count; i++ ) {
				allKey[i+dbCount] = key[i];
				allval[i+dbCount] = vals[i];
			}
		}

		return XHTMLTag.urlEncode( allKey,allval );
	}

	/**
	 * 【TAG】転送先URLを指定します。
	 *
	 * @og.tag 転送先URLを指定します。
	 *
	 * @param	url 転送先URL
	 */
	public void setPage( final String url ) {
		page = getRequestParameter( url );
	}

	/**
	 * 【TAG】DBキーをCSV 形式でセットします。
	 *
	 * @og.tag
	 * URI の引数にセットするキーを CSV 形式でセットします。
	 * ここの指定は,DBTableModel 上のデータを取り込みます。
	 *
	 * @param	key DBキー(CSV 形式)
	 */
	public void setDbkeys( final String key ) {
		dbkeys = getRequestParameter( key ) ;
	}

	/**
	 * 【TAG】キーを CSV 形式でセットします。
	 *
	 * @og.tag
	 * URI の引数にセットするキーを CSV 形式でセットします。
	 *
	 * @param	key キー(CSV 形式)
	 */
	public void setKeys( final String key ) {
		keys = getRequestParameter( key ) ;
	}

	/**
	 * 【TAG】キーに対応する値を CSV 形式でセットします。
	 *
	 * @og.tag
	 * URI の引数にセットする値を CSV 形式でセットします。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) CommonTagSupport#getCSVParameter を使用
	 *
	 * @param	val 値(CSV 形式)
	 */
	public void setVals( final String val ) {
		vals = getCSVParameter( val );

		// 先に分解しないとデータ自身にカンマを持っている場合に分解をミスる。
	}

	/**
	 * 画面ＩＤとpage から、フォワードするＵＲＬを作成します。
	 * page が null の場合は、index.jsp に転送されます。
	 *
	 * @og.rev 3.5.4.9 (2004/02/25) 新規追加
	 * @og.rev 3.5.5.0 (2004/03/12) URLを求めるのに、GUIInfo#getRealAddress() を使用する。
	 * @og.rev 4.0.0.0 (2005/01/31) GUIInfoの実アドレスのパラメータを考慮する。
	 *
	 * @return	フォワードするＵＲＬ(転送先URL)
	 */
	private String makeGamenLink() {
		if( page == null ) { page = "index.jsp"; }

		if( gamenId == null || gamenId.length() == 0 ) { return page; }

		GUIInfo guiInfo = getGUIInfo( gamenId );
		if( guiInfo == null ) {	// 見つからない場合は、転送しない
			LogWriter.log( "Forward[" + gamenId + "]先にはアクセス不可です。" );
			return page;
		}

		String address = guiInfo.getRealAddress( page );
		String link = getRequestParameter( address );

		// Ver 4.0.0 (2007/06/12) gamenId を使用する場合は、useRedirect = true をセット
		useRedirect = true;
		return link;
	}

	/**
	 * 【TAG】gamenId 属性を登録します。
	 *
	 * @og.tag
	 * gamenId 属性は、画面IDをキーに、実アドレスを求めるのに使用します。
	 * 画面IDが指定された場合は、実アドレスに変換する。指定されない場合は、
	 * page 属性をそのままリダイレクトアドレスとします。
	 * gamenId を指定した場合は、このuseRedirect属性に関係なく、Redirect されます。
	 *
	 * @og.rev 3.5.4.9 (2004/02/25) 新規追加
	 *
	 * @param   id gamenId 属性
	 * @see		#setUseRedirect( String )
	 */
	public void setGamenId( final String id ) {
		gamenId = nval( getRequestParameter( id ),gamenId );
	}

	/**
	 * 【TAG】response.sendRedirect するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * useRedirect 属性は、URLの転送方式を指定できます。
	 * 通常は、pageContext.forward しますが、この属性に true を
	 * 設定すると、response.sendRedirect します。
	 * gamenId を指定した場合は、この属性に関係なく、Redirect されます。
	 * 初期値は、false (forward) です。
	 *
	 * @og.rev 3.5.6.6 (2004/08/23) 新規追加
	 *
	 * @param   flag useRedirect 属性 [false:forwrd/true:redirect]
	 * @see		#setGamenId( String )
	 */
	public void setUseRedirect( final String flag ) {
		useRedirect = nval( getRequestParameter( flag ),useRedirect );
	}

	/**
	 * 【TAG】サブミット先の文書を表示させるフレーム、またはウィンドウの名前を指定します。
	 *
	 * @og.tag サブミット先のフレーム名(ターゲット属性)を設定します。
	 *
	 * @og.rev 3.8.0.4 (2005/08/08) 新規追加
	 *
	 * @param	flag サブミット先の文書のフレーム名(ターゲット属性)
	 */
	public void setTarget( final String flag ) {
		target = nval( getRequestParameter( flag ),target );
	}

	/**
	 * 【TAG】ファイル転送時にファイル名の別名を指定します(初期値:null)。
	 *
	 * @og.tag
	 * ファイルを転送する場合に、オープンさせるファイル名を指定します。
	 * 指定しない場合は、フォラード前に呼び出されたアドレスがそのまま使用されます。
	 *
	 * @param   name ファイル名
	 */
	public void setFilename( final String name ) {
		filename = nval( getRequestParameter( name ),filename );
	}

	/**
	 * 【TAG】(通常は使いません)画面遷移を行わない形式の登録方法を使用するかを指定します。
	 *
	 * @og.tag
	 * 画面遷移なしの登録を行うかどうかを指定します。
	 * trueが指定された場合、forwardタグでは、実際の画面遷移は行わずに、forward先の
	 * URLを<div id="noTransitionUrl">[URL]</div>の形でHTMLとして出力し、JavaScript側で
	 * 実際の画面遷移を発生させます。target属性は<div id="noTransitionTarget">[TARGET]</div>で
	 * 出力されます。
	 *
	 * この値は、og:headタグで設定値、または前画面からの値を継承するため、通常、この属性ではセットしません。
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 新規追加
	 * @og.rev 5.1.3.0 (2010/02/01) noTransition、ajaxSubmitのコントロールは、requestで行う。
	 *
	 * @param   noTrnstn 画面遷移を行わない形式の登録方法を使用するか
	 */
	public void setNoTransition( final String noTrnstn ) {
//		noTransition = nval( getRequestParameter( noTrnstn ), noTransition );
		setNoTransitionRequest( nval( getRequestParameter( noTrnstn ), isNoTransitionRequest() ) );
	}

	/**
	 * 【TAG】(通常は使いません)ajaxを利用したsubmitを利用するかどうかを指定します。
	 *
	 * @og.tag
	 * ajaxを利用したSubmit処理を行うかどうかを指定します。
	 * setNoTransitionをtrueにした場合と同様の処理を行います。
	 * (forwardせずにURLをDIVタグで出力する)
	 *
	 * この値は、og:headタグで設定値、または前画面からの値を継承するため、通常、この属性ではセットしません。
	 *
	 * @og.rev 4.3.8.0 (2009/08/01) 新規追加
	 * @og.rev 5.1.3.0 (2010/02/01) noTransition、ajaxSubmitのコントロールは、requestで行う。
	 *
	 * @param   ajaxSubmit ajaxを利用したsubmitを利用するかどうか
	 */
	public void setUseAjaxSubmit( final String ajaxSubmit ) {
//		useAjaxSubmit = nval( getRequestParameter( ajaxSubmit ), useAjaxSubmit );
		setAjaxSubmitRequest( nval( getRequestParameter( ajaxSubmit ), isAjaxSubmitRequest() ) );
	}

	/**
	 * 【TAG】(通常は使いません)useRedirect=true時に 日本語を含む転送先(page)をバイナリ変換するときのコード名[UTF-8/Windows-31J]など(初期値:null)
	 *
	 * @og.tag
	 * 日本語ファイル名を持つ URL を useRedirect で転送する場合に、転送先(page)をバイナリ変換しないと、転送できないケースがあります。
	 * このあたりは、ブラウザにも依存するので、既存の処理に影響しない様に、このパラメータを使用する場合のみ
	 * 処理を実行します。
	 * IEでの日本語ファイル転送で、うまくいかない場合の、暫定処置です。
	 * IEの場合は、redirectEncode="UTF-8" で、日本語ファイルを転送できます。
	 * ただし、useRedirect=true時に filename 属性によるファイル名の書き換え処理はできませんので、ご注意ください。
	 * 初期値は、null で、バイナリエンコードしません。（互換性ありの状態です。）
	 *
	 * @og.rev 5.6.5.0 (2013/06/07) 新規追加
	 *
	 * @param   redEnc URL を 転送する場合に、日本語を含む転送先(page)をバイナリ変換するときのコード名
	 */
	public void setRedirectEncode( final String redEnc ) {
		redirectEncode = nval( getRequestParameter( redEnc ), redirectEncode );
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @serialData 一部のオブジェクトは、シリアライズされません。
	 *
	 * @param	strm	ObjectOutputStreamオブジェクト
	 * @throws IOException	シリアライズに関する入出力エラーが発生した場合
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @serialData 一部のオブジェクトは、シリアライズされません。
	 *
	 * @param	strm	ObjectInputStreamオブジェクト
	 * @see #release2()
	 * @throws IOException	シリアライズに関する入出力エラーが発生した場合
	 * @throws ClassNotFoundException	クラスを見つけることができなかった場合
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "tableId"		,tableId	)
				.println( "page"		,page		)
				.println( "keys"		,keys		)
				.println( "dbkeys"		,dbkeys		)
				.println( "vals"		,vals		)
				.println( "gamenId"		,gamenId	)
				.println( "useRedirect" ,useRedirect)
				.println( "target"		,target		)
				.println( "filename"	,filename	)
				.println( "disposition"		,disposition	)
				.println( "noTransition"	,noTransition	)
				.println( "useAjaxSubmit"	,useAjaxSubmit	)
				.println( "redirectEncode"	,redirectEncode	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
