package org.opengion.fukurou.model;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;

/**
 * ファイル操作のインタフェース
 * 
 * ローカルサーバ、クラウドストレージ(AWS,AZURE,BLUEMIX,ORACLE)のファイル操作用です。
 * FileOperationFactoryを通して、インスタンスを生成可能です。
 * 
 * @og.group ファイル操作
 *
 * @og.rev 5.10.8.0 (2019/02/01) 新規作成
 * @author oota
 * @since	JDK7.0
 */
public interface FileOperation {
	/**
	 * InputStreamのデータを書き込みます。
	 * 
	 * @param is 書き込みデータのInputStream
	 * @throws IOException
	 */
	public void write(InputStream is) throws IOException;

	/**
	 * データを読み込み、InputStreamとして、返します。
	 * 
	 * @return 読み込みデータのInputStream
	 * @throws FileNotFoundException
	 */
	public InputStream read() throws FileNotFoundException;

	/**
	 * ファイルを削除します。
	 * 
	 * @return 成否フラグ
	 */
	public boolean delete();

	/**
	 * ファイルを指定先に、コピーします。
	 * 
	 * @param afPath コピー先
	 * @return 成否フラグ
	 */
	public boolean copy(String afPath);

	/**
	 * ファイルを指定先に、移動します。
	 * 
	 * @param afPath 移動先
	 * @return 成否フラグ
	 */
	public boolean move(String afPath);

	/**
	 * 設定パスを取得します。
	 * 
	 * @return 設定パス
	 */
	public String getPath();

	/**
	 * 絶対パスを取得します。
	 * 
	 * @return 絶対パス
	 */
	public String getAbsolutePath();

	/**
	 * 名称を取得します。
	 * 
	 * @return 名称
	 */
	public String getName();

	/**
	 * 親のパスを取得します。
	 * 
	 * @return 親のパス
	 */
	public String getParent();

	/**
	 * ファイルサイズを返します
	 * 
	 * @return ファイルサイズ
	 */
	public long length();

	/**
	 * 最終更新時刻を取得します。
	 * 
	 * @return 最終更新時刻
	 */
	public long lastModified();

	/**
	 * ファイルの場合は、trueを返します。
	 * 
	 * @return ファイルフラグ
	 */
	public boolean isFile();

	/**
	 * ディレクトリの場合は、trueを返します。
	 * 
	 * @return ディレクトリフラグ
	 */
	public boolean isDirectory();

	/**
	 * 存在する場合は、trueを返します。
	 * 
	 * @return 存在フラグ
	 */
	public boolean exists();

	/**
	 * パスのファイルとディレクトリ一覧を取得します。
	 * 
	 * @return ファイルとティレクトリ一覧
	 */
	public FileOperation[] listFiles();

	/**
	 * パスのファイルとディレクトリ一覧を取得して、
	 * 引数でフィルターを行います。
	 * 
	 * @param filter フィルター
	 * @return	ファイルとディレクトリ一覧
	 */
	public FileOperation[] listFiles(FileOperationFileFilter filter);

	/**
	 * ディレクトリを作成します。
	 * 
	 * ※１つのディレクトリのみ作成します。
	 * (クラウドストレージにはディレクトリの概念が無いため、
	 * 作成は行わず、trueを返します)
	 * 
	 * @return 成否フラグ
	 */
	public boolean mkdir();

	/**
	 * ディレクトリを作成します。
	 * 
	 * ※複数のディレクトリを作成します。
	 * (クラウドストレージにはディレクトリの概念が無いため、
	 * 作成は行わず、trueを返します)
	 * 
	 * @return 成否フラグ
	 */
	public boolean mkdirs();

	/**
	 * 指定のファイル情報のファイル名に変更します。
	 * 
	 * @param dest 変更後のファイル情報
	 * @return 成否フラグ
	 */
	public boolean renameTo(FileOperation dest);

	/**
	 * 親のディレクトリを返します。
	 * 
	 * @return 親のディレクトリ
	 */
	public FileOperation getParentFile();

	/**
	 * 書き込み可能フラグ
	 * 
	 * ※クラウドストレージの場合は、
	 * 必ずtrueを返します。
	 * 
	 * @return 書き込み可能フラグ
	 */
	public boolean canWrite();

	/**
	 * 読み取り可能フラグ
	 * 
	 * ※クラウドストレージの場合は、
	 * 必ずtrueを返します。
	 * 
	 * @return 読み取り可能フラグ
	 */
	public boolean canRead();

	/**
	 * 隠しファイルフラグ
	 * 
	 * ※クラウドストレージの場合は、
	 * 必ずfalseを返します。
	 * 
	 * @return 隠しファイルフラグ
	 */
	public boolean isHidden();

	/**
	 * 新規ファイル作成
	 * 
	 * 既にファイルが存在しない場合のみ、
	 * 空のファイルを作成します。
	 *
	 * @return 成否フラグ
	 * @throws IOException
	 */
	public boolean createNewFile() throws IOException;

	/**
	 * 最終更新時刻の更新
	 * 
	 * 最終更新時刻の更新を行います。
	 * ※クラウドストレージの場合は、
	 * 最終更新時刻の更新を行えません。
	 * 
	 * @param time 更新する最終更新時刻
	 * @return 成否フラグ
	 */
	public boolean setLastModified(long time);

	/**
	 * カノニカルファイル情報を取得します。
	 * 
	 * ※ローカルサーバのみ通常ファイルと、
	 * カノニカルファイルで異なります。
	 * 
	 * @return カノニカルファイル情報
	 * @throws IOException
	 */
	public FileOperation getCanonicalFile() throws IOException;
}