package org.opengion.plugin.cloud;

import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.opengion.fukurou.model.AbstractFileOperation;
import org.opengion.fukurou.model.FileOperationFileFilter;
import org.opengion.fukurou.model.FileOperationInfo;
import org.opengion.fukurou.model.FileOperation;
import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;

import com.ibm.cloud.objectstorage.SDKGlobalConfiguration;
import com.ibm.cloud.objectstorage.auth.AWSCredentials;
import com.ibm.cloud.objectstorage.auth.AWSStaticCredentialsProvider;
import com.ibm.cloud.objectstorage.client.builder.AwsClientBuilder.EndpointConfiguration;
import com.ibm.cloud.objectstorage.oauth.BasicIBMOAuthCredentials;
import com.ibm.cloud.objectstorage.services.s3.AmazonS3;
import com.ibm.cloud.objectstorage.services.s3.AmazonS3ClientBuilder;
import com.ibm.cloud.objectstorage.services.s3.model.AmazonS3Exception;
import com.ibm.cloud.objectstorage.services.s3.model.ListObjectsV2Request;
import com.ibm.cloud.objectstorage.services.s3.model.ListObjectsV2Result;
import com.ibm.cloud.objectstorage.services.s3.model.ObjectListing;
import com.ibm.cloud.objectstorage.services.s3.model.ObjectMetadata;
import com.ibm.cloud.objectstorage.services.s3.model.PutObjectRequest;
import com.ibm.cloud.objectstorage.services.s3.model.S3Object;
import com.ibm.cloud.objectstorage.services.s3.model.S3ObjectSummary;

/**
 * FileOperation_IBM.javaは、Bluemixのストレージ(S3と互換性があります)の、
 * ファイル操作を行うクラスです。
 * 
 * IBMCloud(Bluemix)のダッシュボードから、下記の値を取得して、
 * システムリソースに登録を行う必要があります。
 * 
 * CLOUD_STORAGE_S3_APIKEY、CLOUD_STORAGE_S3_SERVICEINSTANCEID、CLOUD_STORAGE_S3_SERVICE_END_POINT、CLOUD_STORAGE_S3_REGION
 * 
 * @og.rev 5.10.8.0 (2019/02/01) 新規作成
 *
 * @version 5
 * @author  oota
 * @since   JDK7.0
 *
 */
public class FileOperation_IBM extends AbstractFileOperation {
	/** クラス変数 */
	private final AmazonS3 amazonS3;
	private final String conBuket;
	private static final String COS_AUTH_ENDPOINT = "https://iam.ng.bluemix.net/oidc/token";

	/**
	 * コンストラクター
	 * @param buket バケット
	 * @param inPath パス
	 */
	public FileOperation_IBM(String buket, String inPath) {
		super(StringUtil.nval( buket, HybsSystem.sys("CLOUD_BUCKET") ),inPath);
		conBuket = buket;

		// 公式のマニュアルにより、下記のENDPOINTを設定
		SDKGlobalConfiguration.IAM_ENDPOINT = COS_AUTH_ENDPOINT;

		final String apiKey = HybsSystem.sys("CLOUD_STORAGE_S3_APIKEY");
		final String serviceInstanceId = HybsSystem.sys("CLOUD_STORAGE_S3_SERVICEINSTANCEID");
		AWSCredentials credentials = new BasicIBMOAuthCredentials(apiKey, serviceInstanceId);

		final String serviceEndpoint = HybsSystem.sys("CLOUD_STORAGE_S3_SERVICE_END_POINT");
		final String signingRegion = HybsSystem.sys("CLOUD_STORAGE_S3_REGION");
		EndpointConfiguration endpointConfiguration = new EndpointConfiguration(serviceEndpoint, signingRegion);

		amazonS3 = AmazonS3ClientBuilder.standard()
				.withCredentials(new AWSStaticCredentialsProvider(credentials))
				.withPathStyleAccessEnabled(true)
				.withEndpointConfiguration(endpointConfiguration)
				.build();

		try {
			if (!amazonS3.doesBucketExist(bucket)) {
				amazonS3.createBucket(bucket);
			}
		} catch (AmazonS3Exception ase) {
			StringBuilder errMsg = new StringBuilder(HybsSystem.BUFFER_MIDDLE);
			errMsg.append("アクセスキーによる認証が失敗しました。");
			errMsg.append(" CLOUD_STORAGE_S3_APIKEY：").append(apiKey);
			errMsg.append(" CLOUD_STORAGE_S3_SERVICEINSTANCEID：").append(serviceInstanceId);
			errMsg.append(" CLOUD_STORAGE_S3_SERVICE_END_POINT：").append(serviceEndpoint);
			errMsg.append(" CLOUD_STORAGE_S3_REGION：").append(signingRegion);
			errMsg.append(" システムエラー情報：").append(ase.getMessage());
			throw new HybsSystemException(errMsg.toString());
		}
	}

	/**
	 * InputStreamのデータを書き込みます。
	 * 
	 * @param is 書き込みデータのInputStream
	 * @throws IOException
	 */
	@Override
	public void write(InputStream is) throws IOException {
		ByteArrayInputStream bais = null;
		try {
			ObjectMetadata om = new ObjectMetadata();

			byte[] bytes = toByteArray(is);
			om.setContentLength(bytes.length);
			bais = new ByteArrayInputStream(bytes);

			PutObjectRequest request = new PutObjectRequest(bucket, path, bais, om);

			amazonS3.putObject(request);
		} catch (Exception e) {
			StringBuilder errMsg = new StringBuilder(HybsSystem.BUFFER_MIDDLE);
			errMsg.append("BLUEMIXバケットに書き込みが失敗しました。path：").append(path);
			errMsg.append(" システムエラー情報：").append(e.getMessage());
			throw new IOException(errMsg.toString());
		} finally {
			Closer.ioClose(bais);
		}
	}

	/**
	 * データを読み込み、InputStreamとして、返します。
	 * 
	 * @return 読み込みデータのInputStream
	 * @throws FileNotFoundException
	 */
	@Override
	public InputStream read() throws FileNotFoundException {
		S3Object object = null;

		try {
			object = amazonS3.getObject(bucket, path);
		} catch (Exception e) {
			StringBuilder errMsg = new StringBuilder(HybsSystem.BUFFER_MIDDLE);
			errMsg.append("BLUEMIXバケットから読み込みが失敗しました。path：").append(path);
			errMsg.append(" システムエラー情報：").append(e.getMessage());
			throw new FileNotFoundException(errMsg.toString());
		}
		return object.getObjectContent();
	}

	/**
	 * ファイルを削除します。
	 * 
	 * @return 成否フラグ
	 */
	@Override
	public boolean delete() {
		boolean flgRtn = false;

		try {
			if (isFile()) {
				// ファイル削除
				amazonS3.deleteObject(bucket, path);
			} else if (isDirectory()) {
				// ディレクトリ削除
				// 一括削除のapiが無いので、繰り返しで削除を行う
				ObjectListing objectList = amazonS3.listObjects(bucket, path);
				List<S3ObjectSummary> list = objectList.getObjectSummaries();
				for (S3ObjectSummary obj : list) {
					amazonS3.deleteObject(bucket, obj.getKey());
				}

			}
			flgRtn = true;
		} catch (Exception e) {
			// エラーはスルーして、falseを返す
		}

		return flgRtn;
	}

	/**
	 * ファイルを指定先に、コピーします。
	 * 
	 * @param afPath コピー先
	 * @return 成否フラグ
	 */
	@Override
	public boolean copy(String afPath) {
		boolean flgRtn = false;

		try {
			amazonS3.copyObject(bucket, path, bucket, afPath);
			flgRtn = true;
		} catch (Exception e) {
			// エラーはスルーして、falseを返す
		}

		return flgRtn;
	}

	/**
	 * ファイルサイズを返します
	 * 
	 * @return ファイルサイズ
	 */
	@Override
	public long length() {
		long rtn = 0;

		try {
			ObjectMetadata meta = amazonS3.getObjectMetadata(bucket, path);
			rtn = meta.getContentLength();
		} catch (Exception e) {
			// エラーはスルーして、0を返す。
		}
		return rtn;
	}

	/**
	 * 最終更新時刻を取得します。
	 * 
	 * @return 最終更新時刻
	 */
	@Override
	public long lastModified() {
		long rtn = 0;

		try {
			ObjectMetadata meta = amazonS3.getObjectMetadata(bucket, path);
			rtn = meta.getLastModified().getTime();
		} catch (Exception e) {
			// エラーはスルーして、0を返す
		}
		return rtn;
	}

	/**
	 * ファイルの場合は、trueを返します。
	 * 
	 * @return ファイルフラグ
	 */
	@Override
	public boolean isFile() {
		return amazonS3.doesObjectExist(bucket, path);
	}

	/**
	 * ディレクトリの場合は、trueを返します。
	 * 
	 * @return ディレクトリフラグ
	 */
	@Override
	public boolean isDirectory() {
		boolean flgRtn = false;

		if (StringUtils.isEmpty(path)) {
			return true;
		}

		// S3にはディレクトリの概念はないので、「/」で続くデータが存在するかで、判定
		ObjectListing objectList = amazonS3.listObjects(bucket, setDirTail(path));
		List<S3ObjectSummary> list = objectList.getObjectSummaries();
		flgRtn = list.size() == 0 ? false : true;

		return flgRtn;
	}

	/**
	 * パスのファイルとディレクトリ一覧を取得します。
	 * 
	 * @return ファイルとティレクトリ一覧
	 */
	@Override
	public FileOperation[] listFiles(FileOperationFileFilter filter) {
		if (!exists()) {
			return new FileOperationInfo[0];
		}

		String search = path;
		if (isDirectory()) {
			search = setDirTail(path);
		}

		List<FileOperationInfo> rtnList = new ArrayList<FileOperationInfo>();

		// 検索処理
		ListObjectsV2Request request = new ListObjectsV2Request()
				.withBucketName(bucket)
				.withPrefix(search)
				.withDelimiter("/");
		ListObjectsV2Result list = amazonS3.listObjectsV2(request);
		List<S3ObjectSummary> objects = list.getObjectSummaries();

		// ファイル情報の取得
		for (S3ObjectSummary obj : objects) {
			String key = obj.getKey();

			FileOperationInfo file = new FileOperationInfo();
			file.setPath(key);
			file.setName(drawName(key));
			file.setParent(drawParent(key));
			file.setLastModified(obj.getLastModified().getTime());
			file.setFile(true);
			file.setSize(obj.getSize());
			rtnList.add(file);
		}

		// サブディレクトリ情報の取得
		List<String> folders = list.getCommonPrefixes();
		for (String str : folders) {
			String key = rTrim(str, '/');

			FileOperationInfo file = new FileOperationInfo();
			file.setPath(key);
			file.setName(drawName(key));
			file.setParent(drawParent(key));
			file.setDirectory(true);
			rtnList.add(file);
		}

		// フィルタ処理
		FileOperation[] filterList = filter(rtnList, filter);

		return filterList;
	}

	/**
	 * 親のディレクトリを返します。
	 * 
	 * @return 親のディレクトリ
	 */
	@Override
	public FileOperation getParentFile() {
		return new FileOperation_IBM(conBuket,getParent());
	}

	/** Bluemixの場合はローカル環境でのテストは出来ないようです。(proxyを設定しても接続できません) */
}
