/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.transfer;

import java.io.File;
import java.io.PrintWriter;

import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.HybsDateUtil;
import org.opengion.fukurou.util.StringUtil;

/**
 * 伝送要求に対してのデータをファイルに書込みします。
 * 但し、書き込まれるデータについては、旧伝送システムの形式と互換性を持たせるため、
 * データの前30Byteに空白で埋め、さらに全体で標準では500Byteになるように行末にも空白埋めをします。
 * 500byte以外にしたい場合は、書き込みパラメータの第２引数に整数で指定してください。
 * 
 * 先頭データに限っては、スペース以外で埋める事も可能です。
 * その場合は実行対象の第２引数から順番に
 * 状況コード(1byte)、データコード(1byte)、送り先(2byte)、テキスト種別(4byte) 送り元(2byte)
 * を指定してください。
 * 送り元を指定した場合はテキストの後に8byteスペース+送り元+26byteスペース+連番　を付加した上で
 * 行末の空白埋めを行います（後ろ70byteという指定ではないので注意）
 *
 * 書込みするファイル名は、実行対象で指定します。ファイル名は絶対パスで指定して下さい。
 * また、書込するテキストファイルのエンコードは書込パラメーターの第一引数で指定することができます。
 * 指定しない場合、UTF-8が適用されます。
 * 
 * 出力したファイルを扱う場合に異なるOSで改行コードが問題となる場合がある場合は第三引数で指定します。
 * 文字列でCRLFかLFを指定してください。標準はOSの改行コードです。
 * 
 * 
 * 第一引数：エンコード(UTF-8)
 * 第二引数：バイト数(負数で標準の500を利用) 
 * 第三引数：末尾改行コード(標準はOS依存 CRLForLF設定可)
 * 
 * @og.rev 5.10.13.1 (2019/07/12) 改行コード追加
 *
 * @og.group 伝送システム
 *
 * @version  5.0
 * @author   Hiroki.Nakamura
 * @since    JDK1.6
 */
public class TransferExec_SAMCB implements TransferExec {
	private final int fill = 500;
	private final  String BR = System.getProperty("line.separator"); 


	// 書込ファイルオブジェクト
//	private File fileWrite = null;			// 5.5.2.4 (2012/05/16) ローカル変数化

	// 書込ファイルのエンコード
//	private String fileEncode = null;		// 5.5.2.4 (2012/05/16) ローカル変数化

	/**
	 * ファイルに書込みします。
	 *
	 * @param vals 伝送データ(配列)
	 * @param config 伝送設定オブジェクト
	 * @param tran トランザクションオブジェクト
	 *
	 * @og.rev 5.5.3.3 (2012/06/15) close処理
	 * @og.rev 5.8.1.1 (2014/11/14) パラメータで桁数指定可能にする
	 * @og.rev 5.8.1.2 (2014/11/21) 前方,後方のスペース埋めの箇所の互換性を上げる
	 * @og.rev 5.10.13.1 (2019/07/12) 改行コード指定と第二引数の標準設定
	 */
	@Override
	public void execute( final String[] vals, final TransferConfig config, final Transaction tran ) {
		String fileEncode = "UTF-8";
		String outBR = BR;
		
		// 5.8.1.2
		String[] obj = StringUtil.csv2Array( config.getExecObj(), ' ' );
//		File fileWrite = new File( config.getExecObj() );
		File fileWrite = new File( obj[0] );
		String preText = "";
		String suffText = "";
		preText +=  ( obj.length > 1 ) ? obj[1] : ""; // 状況コード
		preText +=  ( obj.length > 2 ) ? obj[2] : ""; // データコード
		preText +=  ( obj.length > 3 ) ? StringUtil.stringFill( obj[3], 8, fileEncode ) : ""; // ホストNO
		preText +=  ( obj.length > 4 ) ? StringUtil.stringFill( obj[4], 4, fileEncode ) : ""; // テキスト種別
		preText = StringUtil.stringFill( preText, 14, fileEncode );
		if(obj.length > 1){
			preText += HybsDateUtil.getDate( "yyMMddHHmmss" );
			preText += HybsDateUtil.getDate( "MMdd" );
		}
		preText +=  ( obj.length > 4 ) ? StringUtil.stringFill( obj[4], 4, fileEncode ) : ""; // テキスト種別
		
		suffText +=  ( obj.length > 5 ) ? StringUtil.stringFill( "", 8, fileEncode ) + StringUtil.stringFill( obj[5], 28, fileEncode ) : ""; // 送り元
		
		// 5.8.1.1 (2014/11/14) fillByte追加
		int fillByte = fill;
		String execPrm = config.getExecPrm();
		if( execPrm != null && execPrm.length() > 0 ){
			String[] prm = StringUtil.csv2Array( execPrm, ' ' );
			fileEncode =  prm[0];
			if( prm.length > 1 ) {
				fillByte = Integer.parseInt( prm[1] );
				if( fillByte < 0) { fillByte=fill; } // 5.10.13.1 (2019/07/12) 改行コード指定
			}
			// 5.10.13.1 (2019/07/12) 改行コード指定
			if( prm.length > 2 ) {
				switch (prm[2]) {
					case "CRLF"	: outBR = "\r\n"; break;
					case "LF" 	: outBR = "\n"; break;
					default 	: outBR=BR;					
				}
			}
		}

//		String fileEncode = config.getExecPrm();
//		if( fileEncode == null || fileEncode.length() == 0 ) {
//			fileEncode = "UTF-8";
//		}

		PrintWriter writer = FileUtil.getPrintWriter( fileWrite,fileEncode );
		String line = null;
		// 5.8.1.2 (2014/11/21) iが欲しいので書き方を変更する
//		for( String s : vals ) {
		for (int i = 0; i < vals.length; i++){
			// 前30Byteを空白埋め 
			// 5.8.1.2 (2014/11/21) 前方をスペース以外で埋められるようにする
//			String preSpace = StringUtil.stringFill( "", 30, fileEncode );
			String preSpace = StringUtil.stringFill( preText, 30, fileEncode );
			// 全体で500Byteになるように後ろに空白埋め
			// 5.8.1.1 500Byte以外も指定可能とする
//			line = StringUtil.stringFill( preSpace + s, 500, fileEncode );
//			line = StringUtil.stringFill( preSpace + s, fillByte, fileEncode );
			// 5.8.1.2 (2014/11/21)
			if(obj.length > 5){
				line = StringUtil.stringFill( preSpace + vals[i] + suffText + StringUtil.intFill( Integer.toString( i ), 14 ), fillByte, fileEncode );
			}
			else{
				line = StringUtil.stringFill( preSpace + vals[i] + suffText, fillByte, fileEncode );
			}
			
//			writer.println( line );
			writer.print( line + outBR ); // 5.10.13.1 (2019/07/12) 改行コード指定
		}
		writer.close(); // 5.5.3.3 (2012/06/15)
	}
}
