/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import java.io.File;
import java.io.PrintWriter;

import org.opengion.fukurou.system.OgBuilder ;					// 6.4.4.2 (2016/04/01)
import org.opengion.fukurou.db.DBUtil;
import org.opengion.fukurou.db.Transaction;	
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.FixLengthData;
import org.opengion.fukurou.util.StringUtil;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;

/**
 * TableFilter_DBSRC_OUT は、TableFilter インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。
 *
 * ここでは、オブジェクト一覧(GF82)の検索結果より、オブジェクト明細テーブル(GF83)から
 * 必要な情報を取得し、各種オブジェクトソースを抜き出します。
 * 出力ファイルは、オブジェクト名＋".sql" という命名規則で作成します。
 * 
 * ここで、PACKAGE と、PACKAGE BODY が同じオブジェクト名の場合、同じファイルに追加(append=true)されます。
 * 本来は、処理フォルダを先に削除しておかないと、上書きされてしまいます。
 * ここでは、フォルダ削除ではなく、できるだけローカル処理するように、PACKAGE の場合だけ、
 * 先に、ファイルを削除する処理を実行します。
 * 
 * また、オブジェクトタイプによって、出力フォルダを変える場合は、個々に指定してください。
 * 以下のコメントは参考ですので、詳細は、jsp 側の抜出プログラムの仕様をご確認ください。
 * 
 *   view			04_VIEW
 *   function		05_SRC
 *   package		05_SRC
 *   package body	05_SRC
 *   procedure		05_SRC
 *   trigger		06_TRG
 * 
 * オブジェクト一覧(GF82)の検索では、(SYSTEM_ID,TBLSYU,OBJ_TYPE,OBJ_NAME,NAME_JA)
 * の項目を取得する必要があります。
 *
 * パラメータは、tableFilterタグの keys, vals にそれぞれ記述するか、BODY 部にCSS形式で記述します。
 * 【パラメータ】
 *  {
 *       DIR : {&#064;BASE_DIR}/sql/install/05_SRC ;   出力ファイルの基準フォルダ(必須)
 *       XML : false ;                                 XML出力を行うかどうか[true/false]を指定します(初期値:false)。
 *  }
 *
 * @og.formSample
 * ●形式：
 *      select SYSTEM_ID,TBLSYU,OBJ_TYPE,OBJ_NAME,NAME_JA from GF82
 *      ① &lt;og:tableFilter classId="DBSRC_OUT" keys="DIR" vals="{&#064;BASE_DIR}/sql/install/05_SRC" /&gt;
 *
 *      ② &lt;og:tableFilter classId="DBSRC_OUT" &gt;
 *               {
 *                   DIR : {&#064;BASE_DIR}/sql/install/05_SRC ;
 *                   XML : false ;
 *               }
 *         &lt;/og:tableFilter&gt;
 *
 * @og.rev 5.6.7.0 (2013/07/27) 新規作成
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public class TableFilter_DBSRC_OUT extends AbstractTableFilter {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.5.0.1 (2016/10/21)" ;

	private static final String[] KEYS = { "SYSTEM_ID","TBLSYU","OBJ_TYPE","OBJ_NAME","NAME_JA" };

	private static final int SYSTEM_ID		= 0;
	private static final int TBLSYU			= 1;
	private static final int OBJ_TYPE		= 2;
	private static final int OBJ_NAME		= 3;
 //	private static final int NAME_JA		= 4;

	private static final String ENCODE = "UTF-8" ;

	// オブジェクト明細テーブル(GF83) の検索SQL
	private static final String GF83_SEL = "select NOLINE,SRC_TEXT"
											+ " from GF83"
											+ " where SYSTEM_ID=? and TBLSYU=? and OBJ_TYPE=? and OBJ_NAME=?"
											+ " and   FGJ='1'"
											+ " order by NOLINE" ;

	// 5.6.6.0 (2013/07/05) ヘッダー部作成用
	private static final String CMNT  = "************************************************************************" ;

	private static final int X = FixLengthData.X ;		// type 定数
	private static final int K = FixLengthData.K ;		// type 定数

	/** 5.6.7.0 (2013/07/27) 各種定数  */
	// 6.0.2.3 (2014/10/10) AbstractTableFilter へ移動

	/** XML形式かどうか */

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.1.1 (2016/01/16) keysMap を、サブクラスから設定させるように変更。
	 */
	public TableFilter_DBSRC_OUT() {
		super();
		initSet( "DIR"	, "出力ファイルの基準フォルダ(必須)"						);
		initSet( "XML"	, "XML出力を行うかどうか[true/false]を指定(初期値:false)"	);
	}

	/**
	 * DBTableModel処理を実行します。
	 *
	 * @og.rev 5.8.2.2 (2014/12/19) XML時エスケープと、BODYの場合の追記はタグ制御難しいのでファイルを分ける
	 * @og.rev 6.3.7.0 (2015/09/04) AutoCloseableを使用したtry-with-resources構築に対応。
	 * @og.rev 6.5.0.1 (2016/10/21) ErrorMessage をまとめるのと、直接 Throwable を渡します。
	 *
	 * @return 処理結果のDBTableModel
	 */
	public DBTableModel execute() {
		isXml = StringUtil.nval( getValue( "XML" ), false );
		// 6.0.2.3 (2014/10/10) ※ このクラスでは、EXEC_END_TAG は、キャッシュしません。

		final File dir = new File( getValue( "DIR" ) );
		if( ! dir.exists() && !dir.mkdirs() ) {
			final String errMsg = "所定のフォルダが作成できませんでした。[" + dir + "]" ;
			throw new HybsSystemException( errMsg );
		}

		String[]	data	= null;
		final Transaction tran = getTransaction();

		final int[] clmNo = getTableColumnNo( KEYS );
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		final DBTableModel table = getDBTableModel();
		final int rowCnt = table.getRowCount();
		for( int row=0; row<rowCnt; row++ ) {
			String objType	= null;
			String objName	= null;
			String fileName	= null;		// 5.8.2.2. (2014/12/19)
			try {
				data = table.getValues( row );
				objType			= data[clmNo[OBJ_TYPE]];
				objName			= data[clmNo[OBJ_NAME]];
				fileName 		= objName;	// 5.8.2.2. (2014/12/19)

				// 5.8.2.2. (2014/12/19) BODYの場合の追記はタグ制御難しいのでファイルを分ける
				if( "PACKAGE BODY".equalsIgnoreCase( objType ) ){ fileName = fileName + "_BODY"; }

				// パッケージの場合は、既存のファイルを削除します。(PACKAGE BODY がappendされるため)
				final File objFile = new File( dir,fileName + ( isXml ? ".xml" : ".sql" ) );	// 5.8.2.2. (2014/12/19)
				// 6.0.0.1 (2014/04/25) These nested if statements could be combined
				if( "PACKAGE".equalsIgnoreCase( objType ) && objFile.exists() && !objFile.delete() ) {
					// このExceptionは、catchでErrorMessageにセットされます。
					final String errMsg = "所定のファイルが削除できませんでした。[" + objFile + "]" ;
					throw new HybsSystemException( errMsg );
				}

				// 出力ファイル名は、オブジェクト名と同じ
				// PACKAGE と、PACKAGE BODY が同じオブジェクト名の場合、同じファイルに追加(append=true)されます。
				// 5.8.2.2. (2014/12/19) 出力ファイルは原則オブジェクト名と同じだが、PachageBodyのみ後ろに＿BODYを付けている
				// 6.3.7.0 (2015/09/04) AutoCloseableを使用したtry-with-resources構築に対応。
				try( final PrintWriter writer = FileUtil.getPrintWriter( objFile,ENCODE,false ) ) {
					if( isXml ) { writer.println( XML_START_TAG ); }
					writer.println( makeHeadLine( clmNo,data ) );

					// オブジェクト明細テーブル(GF83) の検索
					final String[] vals = new String[] { data[clmNo[SYSTEM_ID]],data[clmNo[TBLSYU]],objType,objName };
					final String[][] gf83 = DBUtil.dbExecute( GF83_SEL,vals,tran );
					if( gf83.length == 0 ) {
						System.out.println( "OBJ_TYPE=[" + objType + "], OBJ_NAME=[" + objName + "] is Not Found!" );
						continue;
					}

					// ソースの出力
					for( int j=0; j<gf83.length; j++ ) {
						// 5.8.8.2 (2014/12/19) XML時は内容エスケープする
						if( isXml ) { writer.println( StringUtil.htmlFilter( gf83[j][1] ) ); }
						else {		  writer.println( gf83[j][1] ); }
					}

					writer.println( makeEndLine() );
					if( isXml ) { writer.println( XML_END_TAG ); }
				}
			}
			catch( final RuntimeException ex ) {
				// 6.5.0.1 (2016/10/21) ErrorMessage をまとめるのと、直接 Throwable を渡します。
				makeErrorMessage( "TableFilter_DBSRC_OUT Error",ErrorMessage.NG )
					.addMessage( row+1,ErrorMessage.NG,"DBSRC_OUT"
						, "OBJ_TYPE=[" + objType + "], OBJ_NAME=[" + objName + "]"
						, StringUtil.array2csv( data )
					)
					.addMessage( ex );
			}
		}

		return table;
	}

	/**
	 * ヘッダーとして使用する文字列を作成します。
	 *
	 * @og.rev 5.7.2.0 (2014/01/10) 構文の見直し
	 * @og.rev 5.8.8.2 (2014/12/19) View以外の場合に不正Create文となるので修正
	 * @og.rev 6.2.2.1 (2015/03/31) View以外の場合に不正Create文となる箇所の文法修正
	 * @og.rev 6.4.4.2 (2016/04/01) StringBuilderの代わりに、OgBuilderを使用する。
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	ヘッダーとして使用する文字列
	 * @og.rtnNotNull
	 */
	protected String makeHeadLine( final int[] clmNo,final String[] data ) {
		// 5.7.2.0 (2014/01/10) objType,objName の再利用と、VIEWの場合は、AS を追加します。
		final String objType  = data[clmNo[OBJ_TYPE]];
		final String objName  = data[clmNo[OBJ_NAME]];

		final String LINE1 = "SYSTEM_ID : " + data[clmNo[SYSTEM_ID]] ;
		final String LINE2 = objName + " ( " + objType + " )" ;						// 5.7.2.0 (2014/01/10)
		final String LINE3 = "Created : " + HybsSystem.getDate() ;

		final int[] addLen = new int[] { 0,0,0 };	// 各データ間のスペース
		final int[] type   = new int[] { X,K,X };	// 各データの種別 X:半角 S:空白前埋め K:全角混在
		final FixLengthData fixData = new FixLengthData( addLen,type );

		final String[][] outData = new String[][] {
			{ "/**",	CMNT ,	"**/" },
			{ "/* ",	LINE1,	" */" },
			{ "/* ",	LINE2,	" */" },
			{ "/* ",	LINE3,	" */" },
			{ "/**",	CMNT ,	"**/" },
		};

		// 5.6.6.0 (2013/07/05) 簡易メソッドを利用
		fixData.addAllListData( outData );

		// 5.8.8.2 (2014/12/19) View以外の場合に不正Create文となるので修正
		// 6.2.2.1 (2015/03/31) View以外の場合に不正Create文となる箇所の文法修正

		// 6.4.4.2 (2016/04/01)
		final OgBuilder buf = new OgBuilder();
		fixData.getAllFixData( buf.getBuilder() );				// OgBuilder の内部 Builder に、fixData のデータを書き込む。
		return buf.appendIfCR( isXml , EXEC_START_TAG )
					.append(   "CREATE " )
					.appendIf( "VIEW".equalsIgnoreCase( objType )
								, objType , " " , objName , " AS " )
					.toString();
	}

	/**
	 * 最後の行に相当する文字列を作成します。
	 *
	 * @og.rev 6.0.2.3 (2014/10/10) 処理を簡素化します。
	 * @og.rev 6.2.2.1 (2015/03/31) PostgreSQL対応。最後の記号を、"/" から、";" に変更します。
	 *
	 * @return	最後の行に相当する文字列
	 * @og.rtnNotNull
	 */
	private String makeEndLine() {
		return isXml ? CR + EXEC_END_TAG : ";" ;
	}
}
