/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.io;

import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.fukurou.util.HybsEntry;

import java.io.PrintWriter;
import java.util.List;

/**
 * DBTableModel インターフェース のオブジェクトをPrintWriter を用いて出力する為の，共通インターフェースです。
 *
 * @og.group ファイル出力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public interface TableWriter {

	/** タブ項目区切り文字	 */
	String TAB_SEPARATOR = "\t";		// タブ項目区切り文字

	/** カンマ項目区切り文字	 */
	String CSV_SEPARATOR = ",";			// カンマ項目区切り文字  3.5.6.0 (2004/06/18)

	/**
	 * DBTableModel から 各形式のデータを作成して,PrintWriter に書き出します。
	 * このメソッドは、EXCEL 書き出し時に使用します。
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @see #isExcel()
	 */
	void writeDBTable() ;

	/**
	 * DBTableModel から 各形式のデータを作成して,PrintWriter に書き出します。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 引数に PrintWriter を受け取るように変更します。
	 *
	 * @param	writer PrintWriterオブジェクト
	 */
	void writeDBTable( final PrintWriter writer ) ;

	/**
	 * DBTableModel をセットします。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) lang 引数も同時に設定します。
	 *
	 * @param	table	DBTableModelオブジェクト
	 * @param	lang	言語
	 */
	void setDBTableModel( final DBTableModel table, final String lang ) ;

	/**
	 * 内部の DBTableModel を返します。
	 *
	 * @return  DBTableModelオブジェクト
	 */
	DBTableModel getDBTableModel() ;

	/**
	 * DBTableModelの出力順をセットします。
	 * Label,Name,Size,Class,Data の各フィールドの頭文字のアルファベットで
	 * 出力順を設定します。
	 *
	 * @param   headerSequence 出力順 (LNSCD など)
	 */
	void setHeaderSequence( final String headerSequence ) ;

	/**
	 * DBTableModelの出力順を返します。
	 * Label,Name,Size,Class,Data の各フィールドの頭文字のアルファベットで
	 * 出力順を設定します。
	 *
	 * @return  出力順 (LNSCD など)
	 */
	String getHeaderSequence() ;

	/**
	 * データを書き込む場合の,区切り文字をセットします。
	 * なお，このメソッドは,サブクラスによっては,使用しない場合があります。
	 * もし，使用しないサブクラスを作成する場合は, UnsupportedOperationException
	 * を throw するように,サブクラスで実装して下さい。
	 *
	 * @param   separator 区切り文字
	 */
	void setSeparator( final String separator ) ;

	/**
	 * DBTableModelのデータとして書き込むときに、追加モードで書き込むかどうか[true/false]を設定します。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @param   flag 追加モードで書き込むかどうか[true:追加モード/false:通常モード]
	 */
	void setAppend( final boolean flag ) ;

	/**
	 * DBTableModelのデータとして書き込むときに、追加モードで書き込むかどうかを取得します。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @return	追加モードで書き込むかどうか[true:追加モード/false:通常モード]
	 */
	boolean isAppend() ;

	/**
	 * DBTableModelのデータとして書き込むときのシート名を設定します。
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @param   sheetName シート名
	 */
	void setSheetName( final String sheetName ) ;

	/**
	 * EXCEL雛型参考ファイルのシート名を設定します。
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * EXCELファイルを書き出す時に、雛型として参照するシート名を指定します。
	 * これにより、複数の形式の異なるデータを順次書き出したり(appendモードを併用)する
	 * ことや、シートを指定して新規にEXCELを作成する場合にフォームを設定する事が可能になります。
	 * 初期値は、null(第一シート) です。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @param   sheetName シート名
	 */
	void setRefSheetName( final String sheetName ) ;

	/**
	 * このクラスが、EXCEL対応機能を持っているかどうかを返します。
	 *
	 * EXCEL対応機能とは、シート名のセット、雛型参照ファイル名のセット、
	 * 書き込み元ファイルのFileオブジェクト取得などの、特殊機能です。
	 * 本来は、インターフェースを分けるべきと考えますが、taglib クラス等の
	 * 関係があり、問い合わせによる条件分岐で対応します。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @return	EXCEL対応機能を持っているかどうか
	 */
	boolean isExcel() ;

	/**
	 * 出力先ディレクトリとファイル名をセットします。
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 * @og.rev 6.0.2.0 (2014/09/19) ディレクトリとファイルを分けて管理します。
	 *
	 * @param   directory 出力先ディレクトリ名
	 * @param   filename  出力先ファイル名
	 */
	void setFilename( final String directory , final String filename ) ;

	/**
	 * EXCEL雛型参考ファイル名をセットします。(DIR + Filename)
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 *
	 * @param   filename EXCEL雛型参考ファイル名
	 */
	void setRefFilename( final String filename ) ;

	/**
	 * 雛形のシート名を、そのまま使用する(true)か、新規、または、外部指定のシート名を使用する(false)を指定します。(初期値:false[外部指定のシート名を使用])。
	 *
	 * ※ Ver5では、追記モード時に、指定シートが存在した場合は上書きします(初期値:false[上書きしない])。5.9.12.1 (2016/09/09)
	 *    Ver6では、追記モード時に、雛形を指定できないため、雛形のシート名を、そのまま使用する(true)か、
	 *    新規、または、外部指定のシート名を使用する(false)を指定する属性になります。
	 * 
	 * @og.rev 6.5.0.0 (2016/09/30) sheetOverwrite で、雛形シートの使用時に、元のシート名を使用します。
	 *
	 * @param   flag 元のシート名を使用するかどうか[true:使用する/false:新規、または、外部指定のシート名を使用]
	 */
	void setSheetOverwrite( final boolean flag ) ;

	/**
	 * EXCELで、出力処理の最後にセルの計算式の再計算をさせるシート名をCSV形式で指定します。
	 *
	 * @og.rev 6.5.0.0 (2016/09/30) recalcSheetName で、セル内の計算式を再計算させるシート名を指定。5.9.12.1 (2016/09/09)
	 *
	 * @param  sheet 対象シート名をCSV形式で指定
	 */
	void setRecalcSheetName( final String sheet ) ;

	/**
	 * 読み取り元ファイルのエンコード文字列を指定します。
	 * ファイルは、BufferedReader で受け取る為、本来は、エンコードは不要ですが、
	 * 固定長ファイルの読み取り時のバイトコード分割時に、指定のエンコードで
	 * 分割する必要があります。(例えば、半角文字は、Shift_JIS では、１バイト)
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) 新規作成
	 *
	 * @param   enc ファイルのエンコード文字列
	 */
	void setEncode( final String enc ) ;

	/**
	 * 行番号情報を、出力する(true)/しない(false)を指定します。
	 *
	 * 通常のフォーマットでは、各行の先頭に行番号を出力します。
	 * これは、#NAME 属性を使用する場合には、必ず出力する必要があります。
	 * (#NAME 属性は、読み取り時には、必須です。)
	 * この、先頭の行番号が不要な場合(つまり、他のシステムへのデータ出力、
	 * このシステムでは、#NAME 属性が出力されないため、読み込みできません。)
	 * この行番号を出力しないようにできます。
	 * 初期値は、true(出力する) です。
	 *
	 * @og.rev 3.7.0.2 (2005/02/14) 新規追加
	 *
	 * @param   useNumber 行番号情報 [true:出力する/false:しない]
	 */
	void setUseNumber( final boolean useNumber ) ;

	/**
	 * パラメーターリストをセットします。
	 * 内部は、HybsEntry クラスを持っています。
	 * 引数が、null の場合は、何もしません。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @param	listParam	パラメーターリスト
	 */
	void setParam( final List<HybsEntry> listParam ) ;

	/**
	 * 出力先ファイルのカラム列を、外部(タグ)より指定します。
	 * ただし、指定のカラム名は、DBTableModel上に存在している必要があります。
	 *
	 * @og.rev 4.0.0.0 (2005/11/30) 新規追加
	 *
	 * @param   clms 出力先ファイルのカラム列(CSV形式)
	 */
	void setColumns( final String clms ) ;

	/**
	 * 書き込み対象外のカラム列を、外部(タグ)よりCSV形式で指定します。
	 *
	 * 指定するカラム名に対して、書き込み処理を行いません。
	 * ここで指定するカラム名は、検索したDBTableModel上に含まれる必要はありません。
	 * その場合は、ここでの指定は無視されます。
	 *
	 * @og.rev 6.1.0.0 (2014/12/26) omitNames 属性を追加
	 *
	 * @param   clms 書き込み対象外のカラム列(CSV形式)
	 */
	void setOmitNames( final String clms ) ;

	/**
	 * EXCEL出力時のデフォルトフォント名を設定します。
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * EXCELファイルを書き出す時に、デフォルトフォント名を指定します。
	 * フォント名は、EXCELのフォント名をそのまま使用してください。
	 * 内部的に、POI の org.apache.poi.hssf.usermodel.HSSFFont#setFontName( String )
	 * に設定されます。
	 * 初期値は、システムリソース の TABLE_WRITER_DEFAULT_FONT_NAME です。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規追加
	 *
	 * @param   fontName デフォルトフォント名
	 */
	void setFontName( String fontName ) ;

	/**
	 * EXCEL出力時のデフォルトフォントポイント数を設定します。
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * EXCELファイルを書き出す時に、デフォルトポイント数を指定します。
	 * 内部的に、POI の org.apache.poi.hssf.usermodel.HSSFFont#setFontHeightInPoints( short )
	 * に設定されます。
	 * 初期値は、システムリソース の TABLE_WRITER_DEFAULT_FONT_POINTS です。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規追加
	 *
	 * @param   point フォントポイント数
	 */
	void setFontPoint( short point ) ;

	/**
	 * データの書き込み開始位置を設定します(初期値:0)。
	 *
	 * TAB区切りテキストやEXCEL等のデータの書き込みの開始位置を指定します。
	 * 属性名は、行を飛ばす処理ということで、readTable タグと同じ名称です。
	 * ファイルの先頭行が、０行としてカウントしますので、設定値は、読み飛ばす
	 * 件数になります。(１と指定すると、１件読み飛ばし、２行目から読み込みます。)
	 * 行の読み飛ばしと、カラムの読み飛ばし(columns)、refFileURL、refFilename、
	 * refSheetName とともに使用すれば、ある程度のレイアウト設定が可能です。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 5.7.9.0 (2014/08/08) 新規作成
	 *
	 * @param	skipRowCount 書き込み開始位置
	 */
	void setSkipRowCount( int skipRowCount );

	/**
	 * EXCEL出力時に、データを書き込んだ範囲に罫線を入れるかどうかを指定します。
	 *
	 * データを書き込んでEXCELを作成しても、ノーマルのセルに値がセットされている
	 * だけなので、ある程度加工が必要です。
	 * そこで、データのセットされたセルに罫線を入れることで、それなりのデータが
	 * 出力された感じになります。
	 * この設定と、useAutoCellSize="true" で、セルの幅を自動調整すれば、見栄えが良くなります。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	useCellStyle 罫線を入れるかどうか(true:入れる/false:入れない)
	 * @see		#setUseAutoCellSize( boolean )
	 */
	void setUseCellStyle( final boolean useCellStyle );

	/**
	 * EXCEL出力時に、セルの幅をデータの幅に自動的に合わせるかどうかを指定します。
	 *
	 * データを書き込んでEXCELを作成しても、ノーマルのセルに値がセットされている
	 * だけなので、ある程度加工が必要です。
	 * そこで、データのセットされたセルの幅を自動調整することで、それなりのデータが
	 * 出力された感じになります。
	 * この設定と、useCellStyle="true" で、セルの罫線を自動設定すれば、見栄えが良くなります。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	useAutoCellSize データの幅に自動的に合わせるかどうか(true:自動調整/false:何もしない)
	 * @see		#setUseCellStyle( boolean )
	 */
	void setUseAutoCellSize( final boolean useAutoCellSize );

	/**
	 * EXCEL出力時に、セルの有効範囲を設定するかどうかを指定します。
	 *
	 * セルの有効範囲というのは、EXCELでの 空行、空列の存在しない範囲を指します。
	 * 通常、空行でも、データとして残っている場合は、EXCELのセルオブジェクトは存在します。
	 * ここで、useActiveWorkbook="true" とすると、空行、空列を削除します。
	 * 
	 * 雛形を使用した場合は、データより多めに設定した計算などは、この処理で
	 * 削除されますので、データサイズにフィットさせることができます。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	useActiveWorkbook セルの有効範囲を設定するかどうか(true:設定する/false:そのまま)
	 */
	void setUseActiveWorkbook( final boolean useActiveWorkbook );

	/**
	 * EXCEL出力時に、シート変更するキーとなるカラム名を指定します(このカラムの値がシート名になります)。
	 *
	 * EXCEL帳票では、帳票雛形に、PAGE_BRAKE キーを設定しましたが、TableWriterでは、
	 * メモリ上のカラムの値が変更したときに、シート変更させることができます。
	 * このカラムの値がキーブレイクすると、新しいシートに書き出し始めます。
	 * シート名は、このカラムの値（キーブレイクする値）です。
	 * 
	 * 雛形ファイルを使用する場合、雛形シートもキーブレイクに伴って、＋１されます。
	 * つまり、雛形シートとデータシートは同時に変更されます。
	 * ただし、雛形シートは、最後の雛形シートで止まります。
	 * これは、雛形シートにヘッダー雛形とボディ雛形を用意しておき、最初のキーブレイクで
	 * ヘッダーからボディの書き込みにチェンジするイメージで使用できます。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	pageBreakColumn シート変更するキーとなるカラム名を指定
	 * @see		#setFileBreakColumn( String )
	 */
	void setPageBreakColumn( final String pageBreakColumn );

	/**
	 * EXCEL出力時に、ファイル名を変更するキーとなるカラム名を指定します(このカラムの値がファイル名になります)。
	 *
	 * EXCEL帳票では、メモリ上のカラムの値が変更したときに、ファイル名を変更することができます。
	 * このカラムの値がキーブレイクすると、新しいファイルに書き出し始めます。
	 * ファイル名は、このカラムの値（キーブレイクする値）＋ 元の出力ファイル名の拡張子(.xlsなど)です。
	 * この設定を使用する場合は、出力ファイル名は無視されますが、拡張子だけは使用されます。
	 * 
	 * 雛形ファイルを使用する場合、雛形ファイルもキーブレイクに伴って、再利用されます。
	 * 例えば、pageBreakColumn と併用する場合、キーブレイクで雛形シートも最初から適用になります。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	fileBreakColumn ファイル名を変更するキーとなるカラム名を指定
	 * @see		#setPageBreakColumn( String )
	 */
	void setFileBreakColumn( final String fileBreakColumn );

	/**
	 * EXCEL出力時に、Hyperlinkを作成するキーとなるカラム名と値となるカラム名を指定します。
	 *
	 * ここで、作成するハイパーリンクは、EXCELのシートに対するハイパーリンクです。
	 * それ以外のリンク(本当のURLやファイル等)のリンクは(今は)作成できません。
	 * ハイパーリンクを作成するには、①作成するカラム と ②作成する値 が必要です。
	 * このメソッドで設定するのは、「①:②」という形式でカラム名を指定します。
	 * ②がなければ、①と同じとします。
	 * ②の値のシートの存在有無は、無視します。ハイパーリンクを作成するシートを作成する前に
	 * ハイパーリンクを作成するケースが存在します。
	 * (例えば、各シートへのリンクを持った一覧を作成してから、明細の各シートを作成する様なケース)
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	hyperLinkColumn Hyperlinkを作成するキーとなるカラム名と値となるカラム名を指定
	 */
	void setHyperLinkColumn( final String hyperLinkColumn ) ;

	/**
	 * EXCEL出力時に、Sheet一覧を先頭Sheetに作成する場合のSheet名を指定します。
	 *
	 * これは、Workbook に含まれる Sheet 一覧を作成する場合に、利用可能です。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	sheetName EXCELファイルのシート名
	 */
	void setAddTitleSheet( final String sheetName );

	/**
	 * 書込処理でコードリソースのラベル変換を行うかどうかを指定します。
	 *
	 * コードリソースをそのままの値で出力すると、数字や記号になり何が書かれているのか
	 * 不明になります。
	 * これは、コードリソースをラベルに変換して出力するかどうかを指定します。
	 * 当然、コードはユニークですが、ラベルはユニークになるかどうか保障はされていませんので
	 * TableReader 系で読み込む場合には、リスクが発生します。
	 * また、TableReader 系で読み込む場合にも、ラベルからコードを求める逆変換を行うように、
	 * setUseRenderer メソッドで指定する必要があります。
	 *
	 * 従来は、TableWriter 系に、TableWriter_Renderer 系のクラスを作って対応していましたが、
	 * このメソッドの属性値のフラグで、制御します。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) 新規作成
	 *
	 * @param	useRenderer	コードリソースのラベル変換を行うかどうかを指定
	 */
	void setUseRenderer( final boolean useRenderer ) ;

	/**
	 * デバッグ情報を出力するかどうかを指定します。
	 *
	 * EXCELなどを書き出す場合、シートブレイクやファイルブレイク時の行番号が、検索時の行番号と
	 * 異なる為、エラー時の判定が難しくなります。
	 * そこで、どうしてもわからなくなった場合に備えて、デバッグ情報を出力できるようにします。
	 * 通常は使用しませんので、設定を無視します。
	 * 初期値は、false:デバッグ情報を出力しない です。
	 *
	 * @og.rev 6.1.0.0 (2014/12/26) デバッグ情報を出力するかどうかを指定
	 *
	 * @param	useDebug	デバッグ情報を出力するかどうかを指定
	 */
	void setDebug( final boolean useDebug ) ;
}
