/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

import java.util.concurrent.ConcurrentMap;							// 6.4.3.3 (2016/03/04)
import java.util.concurrent.ConcurrentHashMap;						// 6.4.3.1 (2016/02/12) refactoring
import java.util.Locale;

import static org.opengion.fukurou.system.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

/**
 * Attributes.java は、String 型キーにString型値を Map するクラスです。
 *
 * HTMLのPOST/GET等の受け渡しや、String型の引数が多い場合に効果があります。
 * 特に、getAttributes( String[] param ) による属性リスト作成は、
 * HTMLタグの属性定義を行う上で，非常に便利に利用できます。
 *
 * ※ 6.1.1.0 (2015/01/17)
 *    StringBuilder と同様、set メソッド , add メソッドの戻り値に、自分自身を戻します。
 *    これにより、連結処理できるようにします。
 *
 * この実装は同期化されません。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class Attributes {
	/** 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。  */
	private final ConcurrentMap<String,String> attMap ;		// Map系変数は、Mapと判る様に命名する。

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。
	 */
	public Attributes() {
		attMap = new ConcurrentHashMap<>();
	}

	/**
	 * Attributesオブジェクト を与えて新しく作成するコンストラクター
	 *
	 * @og.rev 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。
	 *
	 * @param att Attributesオブジェクト
	 */
	public Attributes( final Attributes att ) {
		attMap = new ConcurrentHashMap<>( att.attMap );			// AttributesのattMapは、ConcurrentHashMap なので、not null保障
	}

	/**
	 * マップからマッピングをすべて削除します 。
	 *
	 */
	public void clear()  {
		attMap.clear() ;
	}

	/**
	 * マップが指定のキーをマップする値を返します。
	 * マップがこのキーのマッピングを保持していない場合は null
	 * を返します。戻り値の null は、マップがキーのマッピングを
	 * 保持していないことを示すとはかぎりません。つまり、マップが
	 * 明示的にキーを null にマップすることもあります。
	 *
	 * @param    key 関連付けられた値が返されるキー(大文字小文字は同値)
	 *
	 * @return   マップが、指定されたキーにマッピングしている値。
	 *           このキーに対するマッピングがマップにない場合は null
	 */
	public String get( final String key ) {
		return attMap.get( key.toLowerCase( Locale.JAPAN ) ) ;
	}

	/**
	 * 指定された値と指定されたキーをこのマップに関連付けます
	 * 指定されたキーに、null を関連付けることはできません。
	 * (もちろん、""：ゼロストリング は登録できます。)
	 * なぜなら、getAttribute( String[] keys ) 等で値が null の
	 * キーは、取得できない為です。
	 * また、すでに何らかの値がセットされている所に､null をセットした
	 * 場合は､前の値をなにも変更しません。
	 * 通常、値をクリアしたい場合は､ remove( String key ) を利用してください。
	 *
	 * @og.rev 6.1.1.0 (2015/01/17) 戻り値に、自分自身を返すようにします。
	 *
	 * @param    key 指定される値が関連付けられるキー(大文字小文字は同値)
	 * @param    value 指定されるキーに関連付けられる値
	 *
	 * @return	自分自身
	 * @og.rtnNotNull
	 */
	public Attributes set( final String key,final String value ) {
		if( value != null ) {
			attMap.put( key.toLowerCase( Locale.JAPAN ),value ) ;
		}
		return this ;		// 6.1.1.0 (2015/01/17) 戻り値に、自分自身を返す
	}

	/**
	 * 指定された値と指定されたキーをこのマップに関連付けます
	 * set( String key,String value ) との違いは､value が null
	 * の場合に、def を代わりにセットすることです。
	 * ただし、value が null で、def も null の場合は､
	 * なにもセットされません。
	 *
	 * @og.rev 6.1.1.0 (2015/01/17) 戻り値に、自分自身を返すようにします。
	 *
	 * @param    key 指定される値が関連付けられるキー(大文字小文字は同値)
	 * @param    value 指定されるキーに関連付けられる値
	 * @param    def value が null の場合にキーに関連付けられる値
	 *
	 * @return	自分自身
	 * @og.rtnNotNull
	 */
	public Attributes set( final String key,final String value,final String def ) {
		if( value != null )    { attMap.put( key.toLowerCase( Locale.JAPAN ),value ) ; }
		else if( def != null ) { attMap.put( key.toLowerCase( Locale.JAPAN ),def )   ; }

		return this ;		// 6.1.1.0 (2015/01/17) 戻り値に、自分自身を返す
	}

	/**
	 * Attributes 属性を、既存の属性に上書き追加します。
	 *
	 * 引数 att が null の場合は，何もしません。
	 * 内部の Map に直接書き込みますので、すでに同一キーの属性が存在している場合は，
	 * 上書きで置き換えます。
	 * つまり、初期値を設定する場合は、最初に呼び出します。引数で設定された最新の値を
	 * 使用する場合は、最後に設定します。
	 * ただし、#add(String,String) は、既存のキーに値を追記していきますので、これらより
	 * 後に設定しないと、上書きされてしまいますので、ご注意ください。
	 *
	 * 従来の、#addAttributes( Attributes ) の代替えメソッドです。
	 *
	 * @og.rev 6.1.1.0 (2015/01/17) 戻り値に、自分自身を返すようにします。
	 *
	 * @param att Attributes属性
	 *
	 * @return	自分自身
	 * @og.rtnNotNull
	 */
	public Attributes set( final Attributes att ) {
		if( att != null && !att.attMap.isEmpty() ) {
			attMap.putAll( att.attMap );				// AttributesのattMapは、ConcurrentHashMap なので、not null保障
		}

		return this ;		// 6.1.1.0 (2015/01/17) 戻り値に、自分自身を返す
	}

	/**
	 * 指定された値と指定されたキーをこのマップに追加します
	 *
	 * マップ自身のキーは、ユニークである為、既存の値に対して、
	 * 新しく値を追加します。
	 * 追加する方法は、値の文字列の結合です。このメソッドでは、
	 * デフォルトのスペースで結合します。
	 *
	 * 値が null または、すでにそのキーに同一の値が関連付けられている場合は、
	 * 何もしません。
	 *
	 * @og.rev 6.1.1.0 (2015/01/17) 戻り値に、自分自身を返すようにします。
	 *
	 * @param    key   指定される値が関連付けられるキー(大文字小文字は同値)
	 * @param    value 指定されるキーの値に、追加される値
	 *
	 * @return	自分自身
	 * @og.rtnNotNull
	 */
	public Attributes add( final String key,final String value ) {
		return add( key,value," " ) ;		// 6.1.1.0 (2015/01/17) 戻り値に、自分自身を返す
	}

	/**
	 * 指定された値と指定されたキーをこのマップに追加します
	 *
	 * class属性や、style属性など、同一キーに対して、複数の値をつなげる場合に
	 * 使用します。
	 *
	 * マップ自身のキーは、ユニークである為、既存の値に対して、
	 * 新しく値を追加します。
	 * 追加する方法は、値の文字列の結合です。このメソッドでは、
	 * 引数 sepa で文字列を結合します。
	 *
	 * 値が null または、sepa が null または、すでにそのキーに
	 * 同一の値が関連付けられている場合は、何もしません。
	 *
	 * @og.rev 6.1.1.0 (2015/01/17) 戻り値に、自分自身を返すようにします。
	 *
	 * @param    key   指定される値が関連付けられるキー(大文字小文字は同値)
	 * @param    value 指定されるキーの値に、追加される値
	 * @param    sepa  値を連結するときの文字列
	 *
	 * @return	自分自身
	 * @og.rtnNotNull
	 */
	public Attributes add( final String key,final String value,final String sepa ) {
		if( value != null && sepa != null ) {
			final String lkey = key.toLowerCase( Locale.JAPAN );

			String temp = attMap.get( lkey );
			// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
			if( temp == null ) {
				attMap.put( lkey,value );
			}
			else {
				temp = temp.trim();
				if( temp.indexOf( value ) < 0 ||			// 存在しない または、
					  ! temp.equals( value ) &&				// 一致しない
					  ! temp.startsWith( value + sepa ) &&	// 先頭にない
					  ! temp.endsWith( sepa + value ) &&	// 最終にない
					  temp.indexOf( sepa + value + sepa ) < 0 ) {	// 途中にない			// 6.9.7.0 (2018/05/14) PMD
						if( temp.endsWith( sepa ) ) {
							attMap.put( lkey,temp + value );
						}
						else {
							attMap.put( lkey,temp + sepa + value );
						}
				}
			}
		}

		return this ;		// 6.1.1.0 (2015/01/17) 戻り値に、自分自身を返す
	}

	/**
	 * このキーにマッピングがある場合に、そのマッピングをマップから削除します。
	 *
	 * @param    key マッピングがマップから削除されるキー(大文字小文字は同値)
	 *
	 * @return   このキーにマッピングがある場合に、そのマッピングをマップから削除します
	 *           指定されたキーに関連した以前の値。key にマッピングがなかった場合は null。
	 */
	public String remove( final String key ) {
		return attMap.remove( key.toLowerCase( Locale.JAPAN ) );
	}

	/**
	 * マップ内のキーと値のマッピングの数を返します。
	 *
	 * @return   インタフェース Map 内の size
	 */
	public int size() {
		return attMap.size() ;
	}

	/**
	 * マップに含まれているキーと属性のペアを タグの属性リストの形式で返します。
	 * key1="value1" key2="value2" key3="value3" .... の形式で、value が null の
	 * 場合は，key そのもののペアを出力しません。
	 * value が空文字列 "" の場合は，key="" で出力します。
	 *
	 * 引数には，key として出力したい値を配列文字列で渡します。
	 * これは、拡張性に乏しい(すべて出せば，属性項目の追加に対応できる。)
	 * 方法ですが、タグ毎に異なる属性のみを管理するには，厳格に出力
	 * タグ属性を定義したいという思いから，導入しました。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) 内部処理見直し。値の取得は、Mapから直接取得する。
	 * @og.rev 6.4.3.3 (2016/03/04) ConcurrentHashMap の not null制限のチェック追加
	 *
	 * @param    keys 指定 key の文字列配列(可変長引数)(大文字小文字は同値)
	 *
	 * @return   キーと属性のペアをタグの属性リストの形式で返します
	 * @og.rtnNotNull
	 */
//	public String getAttribute( final String[] keys ) {
	public String getAttribute( final String... keys ) {
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

		if( keys != null ) {
			for( final String key : keys ) {
				if( key != null ) {
					final String value = attMap.get( key.toLowerCase( Locale.JAPAN ) );	// 6.0.4.0 (2014/11/28) Mapから直接取得する。
					if( value != null ) {
						// 6.0.2.5 (2014/10/31) char を append する。
						buf.append( key ).append("=\"").append( value ).append("\" ");
					}
				}
			}
		}

		return buf.toString();
	}

	/**
	 * マップに含まれているキーと属性のペアを タグの属性リストの形式ですべて返します。
	 * なお、value が null の場合は，key そのもののペアを出力しません。
	 * value が空文字列 "" の場合は，key="" で出力します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) 内部処理見直し。値の取得は、Mapから直接取得する。
	 * @og.rev 6.4.3.4 (2016/03/11) forループを、forEach メソッドに置き換えます。
	 *
	 * @return   キーと属性のペアをタグの属性リストの形式で返します
	 * @og.rtnNotNull
	 */
	public String getAttribute() {
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

		// 内部に持っているすべてのデータを出力する。
		// よって、toLowerCase も、null チェックも不要。
		attMap.forEach( (k,v) -> buf.append( k ).append("=\"").append( v ).append("\" ") );

		return buf.toString();
	}

	/**
	 * マップに含まれているキーと属性のペアを タグの属性リストの形式ですべて返します。
	 * なお、value が nullや、空文字列("") の場合は，key そのもののペアを出力しません。
	 * この文字列は、key:value; 形式で、文字列を作成します。value に、ダブルコーテーション
	 * は付けません。
	 *
	 * @og.rev 7.0.1.0 (2018/10/15) 新規作成
	 *
	 * @return   キーと属性のペアをCSS形式で返します
	 * @og.rtnNotNull
	 */
	public String getCssFormat() {
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

		// 内部に持っているすべてのデータを出力する。
		// よって、toLowerCase も、null チェックも不要。
		attMap.forEach( (k,v) -> { if( StringUtil.isNotNull(k) ) { buf.append( k ).append(':').append( v ).append("; "); } } );

		return buf.toString();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return	オブジェクトの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return getAttribute() ;
	}
}
