/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

/**
 * DBCell で共通的に使用される フォーマッタークラスです。
 * フォーマットは、$1,$2,$3,$4･･･$9という文字列を含んだ入力テキストです。
 * これに、AAA:BBB:CCC:DDD という値(value)を、コロン(:)で分割し、
 * おのおのの、$1,$2,$3,$4 に割り当てなおして、文字列を合成します。
 * また、$1 は、本来の値として使用しますので、getValut()メソッドで、
 * 取り出せるようになっています。
 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てが
 * ない変数は、""(ゼロ文字列)として、扱われます。
 *
 * @og.rev 3.4.0.2 (2003/09/05) 新規作成
 * @og.rev 5.2.2.0 (2010/11/01) パッケージ移動(hayabusa.html ⇒ fukurou.util)
 * @og.group データ表示
 * @og.group データ編集
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class StringFormat {
	private static final String[] FROM = { "$1","$2","$3","$4","$5","$6","$7","$8","$9" } ;
	/** 初期セパレータ {@value} */
	public static final char SEPARATOR = ':' ;
	private final String inText	;
	private final String inValue;
	private final String inName	;	// 4.3.4.0 (2008/12/01) $Cの置換え追加
	private String outText	;
	private String outValue	;

	/**
	 * コンストラクター
	 * テキストとコロン(:)で区切られた引数を指定してオブジェクトを構築します。
	 * テキストには、$1,$2,$3,$4･･･$9という文字列を含んだ入力テキストです。
	 * 値は、コロン(:)で区切られて、$1,$2等に順番に割り当てられます。
	 * nameは$Cで置き換える文字列です。
	 *
	 * @og.rev 4.3.4.0 (2008/12/01) $C対応追加
	 *
	 * @param text	$1,$2,$3,$4･･･$9という文字列を含んだ入力テキスト
	 * @param value	コロン(:)で区切られた引数(AAA:BBB:CCC:DDD)
	 * @param name	$Cと置き換える文字列
	 */
	public StringFormat( final String text, final String value, final String name ) {
		inText  = text;
		inValue = value;
		inName  = name;
	}

	/**
	 * フォーマット変換を行い結果を返します。
	 * 変換時に、$1,$2･･･等に割り当てられない変数には、ゼロ文字列("")が割り当てられます。
	 *
	 * @og.rev 3.8.8.2 (2007/01/26) 自分自身を、$0 に割り当てる。
	 * @og.rev 4.3.4.0 (2008/12/01) $Cの置換え追加
	 * @og.rev 6.8.2.4 (2017/11/20) FROM配列($1～$9)の制約で、イベント名が9個しか使えなかったのを修正します。
	 *
	 * @return フォーマット変換結果
	 */
	public String format() {

		final String[] to;
		if( inValue != null && inValue.indexOf( SEPARATOR ) >= 0 ) {
			to = StringUtil.csv2Array( inValue, SEPARATOR );
		}
		else {
			to = new String[] { inValue };
		}

		String newText = inText;
		final int toLen = Math.min( FROM.length , to.length );		// 6.8.2.4 (2017/11/20) ループを回す際に、少ないほうで回します。
		int i = 0;
		for( ; i<toLen; i++ ) {
			newText = StringUtil.replace( newText, FROM[i], to[i] );
		}
		for( ; i<FROM.length; i++ ) {
			newText = StringUtil.replace( newText, FROM[i], "" );
		}

		// 3.8.8.2 (2007/01/26) 自分自身を、$0 に割り当てる。
		newText = StringUtil.replace( newText, "$0", inValue );

		// 4.3.4.0 (2008/12/01) $Cの置換え
		newText = StringUtil.replace( newText, "$C", inName );

		outValue = to[0];
		outText = newText;
		return outText;
	}

	/**
	 * 第一引数($1に相当)を返します。
	 * 引数はコロン(:)で区切られて渡されています。内部で使用する本当の引数は
	 * 第一引数です。これは、フォーマット時の$1に割り当てられます。
	 * フォーマット変換前に取得すると、null が返ります。
	 *
	 * @return 第一引数($1に相当)
	 */
	public String getValue() {
		return outValue;
	}

	/**
	 * フォーマット変換結果を返します。
	 * これは、#format() 処理を実行した結果を内部でキャッシュしています。
	 * 何度も結果だけを取得したい場合に使用します。(変換処理は実行しません)
	 * フォーマット変換前に取得すると、null が返ります。
	 *
	 * @return フォーマット変換結果
	 */
	public String getText() {
		return outText;
	}

	/**
	 * 第一引数($1に相当)を返します。
	 * 引数はコロン(:)で区切られて渡されています。内部で使用する本当の引数は
	 * 第一引数です。これは、フォーマット時の$1に割り当てられます。
	 * フォーマット変換前に取得すると、null が返ります。
	 *
	 * @og.rev 6.4.5.3 (2016/05/13) 簡易処理用の static メソッド作成
	 *
	 * @param  oldVal 元の値
	 * @return 第一引数($1に相当)
	 */
	public static String getValue( final String oldVal ) {
		if( oldVal != null ) {
			final int ad = oldVal.indexOf( ':' );
			return ad < 0 ? oldVal : oldVal.substring( 0,ad );
		}
		return oldVal;
	}
}
