/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.filter;

import org.opengion.fukurou.system.Closer;

import java.io.PrintWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponseWrapper;
import javax.servlet.http.HttpServletResponse;

/**
 * FileFilter で使用する、File圧縮されたレスポンスのラッパクラスです。
 *
 * @og.group フィルター処理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class FileResponseWrapper extends HttpServletResponseWrapper {
	/** レスポンスオブジェクト */
	protected HttpServletResponse origResponse ;
	/** サーブレット出力ストリーム */
	protected ServletOutputStream stream ;
	/** 出力ライターオブジェクト */
	protected PrintWriter writer ;

	private final String filename ;

	/**
	 * コンストラクター
	 *
	 * @param	response	レスポンス
	 * @param	filename	ファイル名
	 */
	public FileResponseWrapper(final HttpServletResponse response,final String filename ) {
		super(response);
		this.filename = filename;
		origResponse = response;
	}

	/**
	 * ServletOutputStream の実体である FileResponseStream を作成して返します。
	 *
	 * @return	ServletOutputStreamオブジェクト
	 * @og.rtnNotNull
	 * @throws IOException 入出力エラーが発生したとき
	 */
	public ServletOutputStream createOutputStream() throws IOException {
		return new FileResponseStream(origResponse,filename);
	}

	/**
	 * 内部ストリーム を クローズします。
	 *
	 */
	public void finishResponse() {
		Closer.ioClose( writer );
		Closer.ioClose( stream );
	}

	/**
	 * 内部ストリームの flush() メソッドを呼び出します。
	 *
	 * @og.rev 6.3.9.0 (2015/11/06) コンストラクタで初期化されていないフィールドを null チェックなしで利用している(findbugs)
	 */
	@Override
	public void flushBuffer() throws IOException {
		if( stream != null ) {
			stream.flush();
		}
	}

	/**
	 * 内部ServletOutputStreamを返します。
	 *
	 * 内部オブジェクトが存在しない場合は、新規に作成します。
	 *
	 * @return ServletOutputStreamオブジェクト
	 */
	@Override
	public ServletOutputStream getOutputStream() throws IOException {
		if( writer != null ) {
			throw new IllegalStateException("getWriter() has already been called!");
		}

		if( stream == null ) {
			stream = createOutputStream();
		}
		return stream;
	}

	/**
	 * 内部PrintWriterを返します。
	 *
	 * 内部オブジェクトが存在しない場合は、新規に作成します。
	 *
	 * @return PrintWriterオブジェクト
	 */
	@Override
	public PrintWriter getWriter() throws IOException {
		if( writer != null ) {
			return writer;
		}

		if( stream != null ) {
			throw new IllegalStateException("getOutputStream() has already been called!");
		}

		stream = createOutputStream();
		writer = new PrintWriter(new OutputStreamWriter(stream, "UTF-8"));
		return writer;
	}

	/**
	 * 内部ストリームのデータ長を設定します(何もしません)。
	 *
	 * @param	length	データ長
	 */
	@Override
	public void setContentLength(final int length) {
		// ここでは処理を行いません。
	}
}
