/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.resource.GUIInfo;
import org.opengion.hayabusa.resource.RoleMode ;
import org.opengion.hayabusa.db.DBLastSql;
import org.opengion.fukurou.system.OgBuilder ;				// 6.4.4.1 (2016/03/18)
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.StringUtil ;
import org.opengion.fukurou.util.ToString;					// 6.1.1.0 (2015/01/17)

import static org.opengion.fukurou.util.StringUtil.nval ;

/**
 * ファイルアップロード/ダウンロードリンクを作成するタグです(通常は result_info.jsp に組込み)。
 *
 * 画面検索した結果を、ダウンロードしたり、アップロードする機能は、画面リソースの
 * ロール制御で管理しています。
 * ただし、デフォルトで、以下の制限が掛かっています。
 *
 * ファイルダウンロード
 *   ・command="NEW|RENEW|FIRST|LAST|PREV|NEXT|VIEW|RESET" のみ
 * ファイルアップロード
 *   ・command="COPY|INSERT" のみ
 *
 * 5.7.8.1 (2014/07/18)
 *  addDownCmnd と、addUpldCmnd を指定すれば、ダウンロード、アップロードできる
 *  command を増やすことが可能です。
 *
 * また、typeDownload="true" にするとDownload 処理を行う為の前処理を実施します。
 * Download 処理を行うには、指定の画面(呼び出し元)と、抜き出した時の
 * DBTableModel がマッチしているか判定します。
 * また、DBTableModel が存在しない、または、不備な場合(オーバーフロー)は、
 * 最終SQLを実行する必要があり、そもそも、抜出処理が出来ない状態では、
 * エラーを返す必要があります。
 * それら、一連のDownload 処理を行うかどうかを指定します。
 * true の場合、session の HybsSystem.DB_LAST_SQL_KEY に格納された
 * DBLastSql オブジェクトを取得し、scope が request または、overflow が
 * true の場合に、BODY部を実行します。(通常、queryタグを記述します。)
 * query の出力先は、scope を request にします。
 * それ以外は、tableId から取り出した DBTableModel を、scope = "request"
 * に(アドレスのみ)コピーします。
 * writeTable タグでは、scope = "request" から、DBTableModel を取り出します。

 * @og.formSample
 * ●形式：一般ユーザーが直接組み込むことはありません。
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:fileUD
 *       command            【TAG】command を指定します
 *       downloadImg        【TAG】ダウンロード時のアイコンを指定します(初期値:"{&#064;SYS.JSP}/image/fileDownload.gif")
 *       downloadJsp        【TAG】ダウンロード時の呼び出しJSPを指定します(初期値:"{&#064;SYS.JSP}/common/fileDownload.jsp")
 *       uploadImg          【TAG】アップロード時のアイコンを指定します(初期値:"{&#064;SYS.JSP}/image/fileUpload.gif")
 *       uploadJsp          【TAG】アップロード時の呼び出しJSPを指定します(初期値:"{&#064;SYS.JSP}/common/fileUpload.jsp")
 *       imgWidth           【TAG】アイコン幅サイズ(px)を指定します(初期値:20px)
 *       imgHeight          【TAG】アイコン高さサイズ(px)を指定します(初期値:20px)
 *       filename           【TAG】ダウンロード時ファイル名を指定します(初期値:"{&#064;GUI.LABEL}")  6.8.1.4 (2017/08/25) 拡張子はつけません。
 *       addDownCmnd        【TAG】ダウンロード時のコマンドを追加指定できます(初期値:null)
 *       addUpldCmnd        【TAG】アップロード時のコマンドを追加指定できます(初期値:null)
 *       roles              【TAG】ロールズをセットします
 *       target             【TAG】ダウンロード文書を表示させるフレーム、またはウィンドウの名前を指定します
 *       errorMsg           【TAG】ダウンロード出来ない場合のエラーメッセージIDを指定します(初期値:ERR0028)
 *       typeDownload       【TAG】Download 処理を行うかどうかを指定します(初期値:false)
 *       useEditLink        【TAG】編集リンク自体を出すかどうかどうか(初期値:true)
 *       editImg            【TAG】編集アイコンを指定します(初期値:"{&#064;SYS.JSP}/image/edit.png")
 *       clmEdit            【TAG】編集時にカラム編集タブを表示するかどうか(初期値:true)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:fileUD&gt;
 *
 * ●使用例
 *     &lt;og:fileUD command="{&#064;command}" /&gt;
 *
 *     &lt;og:fileUD
 *         command        = command設定 (ダウンロード="NEW|RENEW|FIRST|LAST|PREV|NEXT|VIEW" 、   アップロード="COPY|INSERT")
 *       [ downloadImg ]  = "{&#064;SYS.JSP}/image/fileDownload.gif"   ダウンロード時のアイコン
 *       [ downloadJsp ]  = "{&#064;SYS.JSP}/common/fileDownload.jsp"  ダウンロード時の呼び出しJSP
 *       [ uploadImg   ]  = "{&#064;SYS.JSP}/image/fileUpload.gif"     アップロード時のアイコン
 *       [ uploadJsp   ]  = "{&#064;SYS.JSP}/common/filePopup.jsp"     アップロード時の呼び出しJSP
 *       [ imgWidth    ]  = "20px"                                アイコン幅サイズ(px)
 *       [ imgHeight   ]  = "20px"                                アイコン高さサイズ(px)
 *       [ filename    ]  = "{&#064;GUI.LABEL}.xls"               ダウンロード時ファイル名
 *       [ roles       ]  = "GE|DOC"                              アクセスロールズ
 *       [ target      ]  = "_blank"                              ダウンロードファイルを開くターゲット
 *       [ errorMsg    ]  = "ERR0042"                             エラーメッセージID
 *       [ typeDownload]  = "false/true"                          Download 処理を行うかどうか(初期値:false)
 *     /&gt;
 *
 * @og.group ファイルアップロード/ダウンロード
 *
 * @og.rev 4.3.0.0 (2008/07/04) 新規追加
 * @og.rev 5.7.4.1 (2014/03/15) エラーメッセージ変更
 * @og.rev 5.9.32.1 (2018/05/08) URLチェック対応
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class FileUDTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.8.1.4 (2017/08/25)" ;
	private static final long serialVersionUID = 681420170825L ;

	private String	sysJsp			= HybsSystem.sys( "JSP" );

	private String	command			;			// ダウンロード="NEW|RENEW" 、アップロード="COPY|INSERT"
	private String	downloadImg		= sysJsp + "/image/fileDownload.gif";	// ダウンロード時のアイコン
	private String	downloadJsp		= sysJsp + "/common/fileDownload.jsp";	// ダウンロード時の呼び出しJSP
	private String	uploadImg		= sysJsp + "/image/fileUpload.gif";		// アップロード時のアイコン
	private String	uploadJsp		= sysJsp + "/common/filePopup.jsp";		// アップロード時の呼び出しJSP
	private String	imgWidth		= "20px";					// アイコン幅サイズ(px)
	private String	imgHeight		= "20px";					// アイコン高さサイズ(px)
	private String	filename		;							// ダウンロード時ファイル名("{@GUI.LABEL}")  6.8.1.4 (2017/08/25) 拡張子はつけません。
	private String	roles			;							// アクセスロールズ
	private String	target			;							// ダウンロードファイルを開くターゲット
	private String	errorMsg		= "ERR0042";				// 出力対象データが見つかりませんでした。再度検索して下さい。 5.7.4.1 (2014/03/15)
	private boolean isError			;
	private boolean	typeDownload	;							// Download 処理を行うかどうか(初期値:false)
	private boolean isEdit			= true;						// 5.6.0.2 (2013/01/19) ロールのisEditとほぼ同等。clmEdit用。
	private boolean useEditLink		= true;						// 5.7.5.0 (2014/04/04) 編集リンク自体を出すかどうかどうか(初期値:true)

	private String	addDownCmnd		;							// 5.7.8.1 (2014/07/18) ダウンロード時のコマンドを追加
	private String	addUpldCmnd		;							// 5.7.8.1 (2014/07/18) アップロード時のコマンドを追加

	private static final String	EDIT_GAMENID= "14_EDIT";					// 編集画面の画面ID
	private String	editImg			= sysJsp + "/image/edit.png";	// 編集時のアイコン

	// 5.9.32.1 (2018/05/08) URLチェック対応
	private boolean	useURLCheck		=  HybsSystem.sysBool( "USE_FILEUD_URL_CHECK" );	// Ver5 では、FILEUD_URL_CHECK
	private static final long	URL_CHECK_TIME	= 1000*60*60*24;						// 初期値：２４時間(とりあえず固定値)
	// private String	urlCheckCrypt	= HybsSystem.sys( "URL_CHECK_CRYPT" );			// 将来使用するかも

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public FileUDTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 画面項目並べ替え対応
	 * @og.rev 6.2.0.1 (2015/03/06) 画面IDのキーを"GAMENID" に統一します。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doStartTag() {

		if( typeDownload ) {
			final DBLastSql lastSql = (DBLastSql)getSessionAttribute( HybsSystem.DB_LAST_SQL_KEY );
			if( lastSql != null ) {
				final String guikey = getRequest().getParameter( "GAMENID" );		// 6.2.0.1 (2015/03/06) 画面IDのキーを"GAMENID" に統一します。
				if( lastSql.isGuiMatch( guikey ) ) {
					// view タグで仕込んだ clmNames を、Request 変数に設定します。
					setRequestAttribute( "DB.CLM_NAMES"			, lastSql.getClmNames() );
					// 5.1.6.0 (2010/05/01) 画面項目並べ替え対応
					setRequestAttribute( "DB.VIEW_CLM_NAMES"	, lastSql.getViewClmNames() );
					setRequestAttribute( "DB.VIEW_EDITABLE"		, Boolean.toString( lastSql.isViewEditable() ) );

					// 6.3.9.0 (2015/11/06) Deeply nested if..then statements are hard to read (PMD)
					if( lastSql.isRequest() && getUserInfo( "LASTSQL" ) != null ) {
						return EVAL_BODY_BUFFERED ;		// Body を評価する。( extends BodyTagSupport 時)
					}

					final String tableId = lastSql.getTableId();
					// セッションのDBTableModelを、そのまま、リクエストに入れているだけなので、実態は同じ。
					setRequestAttribute( HybsSystem.TBL_MDL_KEY, getSessionAttribute( tableId ) );
					return SKIP_BODY ;					// Body を評価しない
				}
			}

			// 処理が実行できない場合、エラーを出力します。
			jspPrint( getResource().getLabel( errorMsg ) );
			isError = true;
		}

		return SKIP_BODY ;				// Body を評価しない
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 4.3.0.1 (2008/08/11) callBack パラメータの設定漏れ修正、引数から見直し
	 * @og.rev 5.1.6.0 (2010/05/01) 帳票画面定義機能対応
	 * @og.rev 5.1.7.0 (2010/06/01) 進む・戻るボタンでもExcel出力ボタンを出す
	 * @og.rev 5.3.6.0 (2011/06/01) Edit機能見直し
	 * @og.rev 5.3.7.0 (2011/07/01) Excel取込で画面IDとアドレスが異なる場合にエラーとなるバグを修正
	 * @og.rev 5.3.7.0 (2011/07/01) 編集画面をリサイズできるように対応します。
	 * @og.rev 5.3.7.0 (2011/07/01) PL/SQLを利用した検索でファイルダウンロードが出来るように対応します。
	 * @og.rev 5.4.2.0 (2011/12/01) 画面リソースで画面編集機能のON/OFFを設定できるようにする。
	 * @og.rev 5.5.2.4 (2012/05/16) nullに対する冗長な比較
	 * @og.rev 5.5.5.2 (2012/08/20) 画面リソースのoモード対応
	 * @og.rev 5.5.6.2 (2012/09/24) makeTagのtarget対応
	 * @og.rev 5.6.0.2 (2013/01/19) editモードのisEditコントロール対応
	 * @og.rev 5.7.5.0 (2014/04/04) 編集リンク自体を出すかどうかどうか(初期値:true)
	 * @og.rev 5.7.8.1 (2014/07/18) command="RESET" 時に、ダウンロードリンクを表示
	 * @og.rev 5.7.8.1 (2014/07/18) addDownCmnd , addUpldCmndを追加
	 * @og.rev 6.2.0.1 (2015/03/06) 画面IDのキーを"GAMENID" に統一します。
	 * @og.rev 6.2.2.0 (2015/03/27) ファイルの拡張子を、".xls" から、システムリソースの TABLE_WRITER_EXCEL_TYPE に変更
	 * @og.rev 6.3.1.0 (2015/06/28) 4:個人設定可のリソースの取得方法に修正(バグFix)。TABLE_WRITER_EXCEL_TYPE
	 * @og.rev 6.3.4.0 (2015/08/01) 14_EDIT の指定で、フォルダではなく、index.jspまで指定します。
	 * @og.rev 6.4.4.1 (2016/03/18) StringBuilderの代わりに、OgBuilderを使用する。
	 * @og.rev 6.8.1.3 (2017/08/04) 日本語のeditNameが、ネイティブモードでHTTP400 エラーになる。･･･ 5.10.0.2 (2018/06/22)
	 * @og.rev 6.8.1.4 (2017/08/25) filename に、拡張子はつけません。
	 * @og.rev 5.9.32.1 (2018/05/11) makeTagに、URLチェック対応のための引数を追加
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();

		if( isError      ) { return SKIP_PAGE ; }		// ページの残りの処理を行わない。
		if( typeDownload ) { return EVAL_PAGE ;	}		// ページの残りを評価する。

		boolean isDownload ;
		// 5.7.8.1 (2014/07/18) command="RESET" 時に、ダウンロードリンクを表示
		if(    "NEW".equals(   command ) || "RENEW".equals( command )
			|| "FIRST".equals( command ) || "LAST".equals(  command )
			|| "PREV".equals(  command ) || "NEXT".equals(  command )
			|| "VIEW".equals(  command ) || "RESET".equals( command ) ) {
			isDownload = true;
		}
		else if( "COPY".equals( command ) || "INSERT".equals( command ) ) {
			isDownload = false;
		}
		// 5.7.8.1 (2014/07/18) addDownCmnd , addUpldCmndを追加
		else if( addDownCmnd != null && addDownCmnd.equals( command ) ) {
			isDownload = true;
		}
		else if( addUpldCmnd != null && addUpldCmnd.equals( command ) ) {
			isDownload = false;
		}
		else {
			return EVAL_PAGE ;
		}

		byte bitMode ;
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		if( roles == null ) {
			final GUIInfo guiInfo = (GUIInfo)getSessionAttribute( HybsSystem.GUIINFO_KEY );
			bitMode = guiInfo.getBitMode();
		}
		else {
			final RoleMode roleMode = RoleMode.newInstance( roles );
			final RoleMode userRoleMode = getUser().getRoleMode();
			bitMode = userRoleMode.getAccessBitMode( roleMode );
		}

		if( filename == null ) {
			// 6.2.2.0 (2015/03/27) ファイルの拡張子を、".xls" から、システムリソースの TABLE_WRITER_EXCEL_TYPE に変更
			// 6.3.1.0 (2015/06/28) 4:個人設定可のリソースの取得方法に修正(バグFix)
			// 6.8.1.4 (2017/08/25) filename に、拡張子はつけません。
			filename = getGUIInfoAttri( "LABEL" );
		}
		final String guikey = getGUIInfoAttri( "KEY" );	// 4.3.0.1 (2008/08/11)
		if( isDownload ) {
			if( RoleMode.isDownload( bitMode ) ) {
				// 4.3.0.1 (2008/08/11) callBack パラメータの設定漏れ修正、引数から見直し
				// <og:link href="{@SYS.JSP}/common/fileDownload.jsp" target="_blank" useGamenId="false"
				//		keys="filename,h_clms" vals="{@GUI.LABEL}.xls,{@h_clms}" >
				// 5.3.7.0 (2011/07/01) PL/SQLを利用した検索でファイルダウンロードが出来るように対応します。
				// 6.2.0.1 (2015/03/06) 画面IDのキーを"GAMENID" に統一します。
				final String href = downloadJsp + "?command=RENEW&GAMENID=" + guikey + "&filename=" + StringUtil.urlEncode( filename ) ;

//				jspPrint( makeTag( downloadImg,href,"Download File", target ) );		// 5.5.6.2 (2012/09/24)
				jspPrint( makeTag( downloadImg,href,"Download File", target,true ) );	// 5.9.32.1 (2018/05/11) URLチェック対応

				// 5.1.6.0 (2010/05/01) 帳票画面定義機能対応
				// 5.3.6.0 (2011/06/01) Edit機能見直し
				final GUIInfo editGui = getUser().getGUIInfo( EDIT_GAMENID );
				// 5.7.5.0 (2014/04/04) 編集リンク自体を出すかどうかどうか(初期値:true)
				// 6.1.0.0 (2014/12/26) refactoring:Prefer StringBuffer over += for concatenating strings
				if( editGui != null && useEditLink ) {
					final String editName = getRequestValue( "editName" );

					final OgBuilder buf = new OgBuilder()
						.append( "javascript:ogOpen('" , sysJsp )						// 前後に分断されるので、注意
						.append( "/" , editGui.getAddress() )							// 6.3.4.0 (2015/08/01)
						.append( "/index.jsp?command=NEW&GAMENID=" , guikey )			// 6.2.0.1 (2015/03/06) 画面IDのキーを"GAMENID" に統一します。
						.appendIf( editName != null										// 5.5.2.4 (2012/05/16) nullに対する冗長な比較
								, "&editName=" , StringUtil.urlEncode( editName ) )		// 6.8.1.3 (2017/08/04) 日本語のeditNameが、ネイティブモードでHTTP400 エラーになる。
						.appendIf( RoleMode.isEdit( bitMode ) && isEdit					// 5.6.0.2(2013/01/09)
								, "&edit=true" )
						.append( "&fullEdit=" , String.valueOf( RoleMode.isFullEdit( bitMode ) ) )
						.append( "',null,null,'width=800px,height=350px,top=350,left=100,resizable=yes',false);" );
					// 6.8.1.3 (2017/08/04) a href ではなく、a onClick にする。
//					final String editLink = makeTag( editImg,buf.toString(),"Edit",null );
					final String editLink = makeTag( editImg,buf.toString(),"Edit",null,false );	// 5.9.32.1 (2018/05/11) URLチェック対応
					jspPrint( editLink.replace( "href","onClick" ) );			// 6.8.1.3 (2017/08/04) makeTag を修正したくなかった。
				}
			}
		}
		else {
			if( RoleMode.isUpload( bitMode ) ) {
				// 4.3.0.1 (2008/08/11) callBack パラメータの設定漏れ修正、引数から見直し
				// <og:link href="{@SYS.JSP}/common/filePopup.jsp" keys="callBack" vals="{@GUI.KEY}/{@SESSION.JSPID}" >

				final String jspId  = (String)getSessionAttribute( "JSPID" );
				// 5.3.7.0 (2011/07/01) Excel取込で画面IDとアドレスが異なる場合にエラーとなるバグを修正
				String callBack = getGUIInfoAttri( "ADDRESS" ) + "/" + jspId + "?GAMENID=" + guikey;
				callBack = StringUtil.urlEncode( callBack );
				final String href = uploadJsp + "?callBack=" +  callBack;

//				jspPrint( makeTag( uploadImg,href,"Upload File",target ) );			// 5.5.6.2 (2012/09/24)
				jspPrint( makeTag( uploadImg,href,"Upload File",target,false ) );	// 5.9.32.1 (2018/05/11) URLチェック対応
			}
		}

		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 5.7.4.1 (2014/03/15) エラーメッセージ変更
	 * @og.rev 5.7.5.0 (2014/04/04) useEditLink 追加
	 * @og.rev 5.7.8.1 (2014/07/18) addDownCmnd,addUpldCmnd 追加
	 * @og.rev 5.9.32.1 (2018/05/08) URLチェック対応
	 */
	@Override
	protected void release2() {
		super.release2();
		sysJsp			= HybsSystem.sys( "JSP" );
		command			= null;		// ダウンロード="NEW|RENEW|FIRST|LAST|PREV|NEXT|VIEW|RESET" 、アップロード="COPY|INSERT"
		downloadImg		= sysJsp + "/image/fileDownload.gif";	// ダウンロード時のアイコン
		downloadJsp		= sysJsp + "/common/fileDownload.jsp";	// ダウンロード時の呼び出しJSP
		uploadImg		= sysJsp + "/image/fileUpload.gif";		// アップロード時のアイコン
		uploadJsp		= sysJsp + "/common/filePopup.jsp";		// アップロード時の呼び出しJSP
		imgWidth		= "20px";						// アイコン幅サイズ(px)
		imgHeight		= "20px";						// アイコン高さサイズ(px)
		filename		= null;							// ダウンロード時ファイル名("{@GUI.LABEL}") 6.8.1.4 (2017/08/25) 拡張子はつけません。
		roles			= null;							// アクセスロールズ
		target			= null;							// ダウンロードファイルを開くターゲット
		errorMsg		= "ERR0042";					// 出力対象データが見つかりませんでした。再度検索して下さい。 5.7.4.0 (2014/03/15)
		typeDownload	= false;						// Download 処理を行うかどうか(初期値:false)
		isError			= false;
		editImg			= sysJsp + "/image/edit.png";	// 編集時のアイコン
		isEdit			= true;							// 5.6.0.2 (2013/01/19)
		useEditLink		= true;							// 5.7.5.0 (2014/04/04) 追加
		addDownCmnd		= null;							// 5.7.8.1 (2014/07/18) ダウンロード時のコマンドを追加
		addUpldCmnd		= null;							// 5.7.8.1 (2014/07/18) アップロード時のコマンドを追加
		useURLCheck		= HybsSystem.sysBool( "USE_FILEUD_URL_CHECK" );		// 5.9.32.1 (2018/05/11)
	//	urlCheckCrypt	= HybsSystem.sys( "URL_CHECK_CRYPT" );				// 5.9.32.1 (2018/05/11)
	}

	/**
	 * リンクを作成します。
	 *
	 * @og.rev 4.3.0.1 (2008/08/11) callBack パラメータの設定漏れ修正、引数から見直し
	 * @og.rev 5.5.6.2 (2012/09/24) tgt引数追加
	 * @og.rev 6.4.2.0 (2016/01/29) alt属性にtitle属性を追記。
	 * @og.rev 5.9.32.1 (2018/05/08) URLチェック対応
	 *
	 * @param	imgsrc	イメージソース
	 * @param	href	リンクのURL
	 * @param	alt		イメージの説明文書
	 * @param 	tgt		HTMLのtarget
	 * @param 	useChk	urlCheck対象 [true:useURLCheckの条件に応じて、ﾁｪｯｸ用URLを作成する。/false:常につけない]
	 *
	 * @return	リンク文字列
	 * @og.rtnNotNull
	 */
//	private String makeTag( final String imgsrc, final String href,  final String alt, final String tgt ) {							// 5.5.6.2 (2012/09/24)
	private String makeTag( final String imgsrc, final String href,  final String alt, final String tgt, final boolean useChk ) {	// 5.9.32.1 (2018/05/08)
		// linkのBODY部にセットするイメージタグの作成
		// 6.1.1.0 (2015/01/17) Attributesの連結記述
		final String img = XHTMLTag.img(
				new Attributes()
					.set( "width"	, imgWidth )
					.set( "height"	, imgHeight )
					.set( "src"		, imgsrc )
					.set( "alt"		, alt )
					.set( "title"	, alt )					// 6.4.2.0 (2016/01/29)
			);

		// 5.9.32.1 (2018/05/11) urlCheck
		final String tag = useURLCheck && useChk
							? XHTMLTag.addURLCheckKey( href, HybsSystem.URL_CHECK_KEY, getUser().getUserID(), System.currentTimeMillis() + URL_CHECK_TIME )
							: href ;

		// 6.1.1.0 (2015/01/17) Attributesの連結記述
		return XHTMLTag.link(
				new Attributes()
//					.set( "href"	, href )
					.set( "href"	, tag )					// 5.9.32.1 (2018/05/11) urlCheck
					.set( "body"	, img )
					.set( "target"	, tgt )					// 5.5.6.2 (2012/09/24)
			);
	}

	/**
	 * 【TAG】command を指定します。
	 *
	 * @og.tag
	 * command を指定します。
	 * ダウンロードは、"NEW|RENEW|FIRST|LAST|PREV|NEXT|VIEW|RESET" のみ有効です。
	 * アップロードは、"COPY|INSERT" のみ有効です。
	 * それ以外のコマンドでは、アイコンは表示されません。
	 * 例えば、NEXT/PREV など。
	 *
	 * @param	cmd コマンド [NEW/RENEW/FIRST/LAST/PREV/NEXT/VIEW/RESET/COPY/INSERT]
	 * @see		#setAddDownCmnd( String )
	 * @see		#setAddUpldCmnd( String )
	 * @see		#setCommand( String )
	 */
	public void setCommand( final String cmd ) {
		command = nval( getRequestParameter( cmd ),command );
	}

	/**
	 * 【TAG】ダウンロード時のアイコンを指定します(初期値:"{&#064;SYS.JSP}/image/fileDownload.gif")。
	 *
	 * @og.tag
	 * ダウンロード時に表示されるアイコンを指定します。
	 * 初期値:"{&#064;SYS.JSP}/image/fileDownload.gif" です。
	 *
	 * @param	img ダウンロード時のアイコン
	 */
	public void setDownloadImg( final String img ) {
		downloadImg = nval( getRequestParameter( img ),downloadImg );
	}

	/**
	 * 【TAG】ダウンロード時の呼び出しJSPを指定します(初期値:"{&#064;SYS.JSP}/common/fileDownload.jsp")。
	 *
	 * @og.tag
	 * ダウンロード時に呼び出されるJSPファイル名を指定します。
	 * 初期値:"{&#064;SYS.JSP}/common/fileDownload.jsp" です。
	 *
	 * @param	file ダウンロード時の呼び出しJSP
	 */
	public void setDownloadJsp( final String file ) {
		downloadJsp = nval( getRequestParameter( file ),downloadJsp );
	}

	/**
	 * 【TAG】アップロード時のアイコンを指定します(初期値:"{&#064;SYS.JSP}/image/fileUpload.gif")。
	 *
	 * @og.tag
	 * アップロード時に表示されるアイコンを指定します。
	 * 初期値:"{&#064;SYS.JSP}/image/fileUpload.gif" です。
	 *
	 * @param	img アップロード時のアイコン
	 */
	public void setUploadImg( final String img ) {
		uploadImg = nval( getRequestParameter( img ),uploadImg );
	}

	/**
	 * 【TAG】アップロード時の呼び出しJSPを指定します(初期値:"{&#064;SYS.JSP}/common/fileUpload.jsp")。
	 *
	 * @og.tag
	 * アップロード時に呼び出されるJSPファイル名を指定します。
	 * 初期値:"{&#064;SYS.JSP}/common/fileUpload.jsp" です。
	 *
	 * @param	file アップロード時の呼び出しJSP
	 */
	public void setUploadJsp( final String file ) {
		uploadJsp = nval( getRequestParameter( file ),uploadJsp );
	}

	/**
	 * 【TAG】アイコン幅サイズ(px)を指定します(初期値:20px)。
	 *
	 * @og.tag
	 * アップロード/ダウンロードのアイコンを表示する際の幅サイズ(px)を指定します。
	 * ファイルサイズは、アップロード/ダウンロード共通です。
	 * 初期値は、20px です。
	 *
	 * @param	width	アイコン幅サイズ(px)
	 */
	public void setImgWidth( final String width ) {
		imgWidth = nval( getRequestParameter( width ),imgWidth );
	}

	/**
	 * 【TAG】アイコン高さサイズ(px)を指定します(初期値:20px)。
	 *
	 * @og.tag
	 * アップロード/ダウンロードのアイコンを表示する際の高さサイズ(px)を指定します。
	 * ファイルサイズは、アップロード/ダウンロード共通です。
	 * 初期値は、20px です。
	 *
	 * @param	height	アイコン高さサイズ(px)
	 */
	public void setImgHeight( final String height ) {
		imgHeight = nval( getRequestParameter( height ),imgHeight );
	}

	/**
	 * 【TAG】ダウンロード時ファイル名を指定します(初期値:"{&#064;GUI.LABEL}")。
	 *
	 * @og.tag
	 * ダウンロード時のファイル名を指定します
	 * 通常は、画面のラベル(言語対応)で、拡張子は付けません。ファイルの種類に応じて、拡張子を自動設定します。
	 *
	 * @og.rev 6.8.1.4 (2017/08/25) 拡張子はつけません。
	 *
	 * @param	file	ダウンロード時ファイル名 ("{&#064;GUI.LABEL}")
	 */
	public void setFilename( final String file ) {
		filename = nval( getRequestParameter( file ),filename );
	}

	/**
	 * 【TAG】ロールズをセットします。
	 *
	 * @og.tag
	 * ここで指定したロールを元に、ユーザー毎のアクセス許可がチェックされます。
	 * アクセス許可されないと、表示されません。
	 * このロールを指定しない場合は、画面のロールが使用されます。
	 *
	 * @param	rls パラメータ
	 */
	public void setRoles( final String rls ) {
		roles = nval( getRequestParameter( rls ),roles );
	}

	/**
	 * 【TAG】ダウンロード文書を表示させるフレーム、またはウィンドウの名前を指定します。
	 *
	 * @og.tag フレーム名(ターゲット属性)を設定します。
	 *
	 * @param	name ダウンロード文書のフレーム名(ターゲット属性)
	 */
	public void setTarget( final String name ) {
		target = nval( getRequestParameter( name ),target );
	}

	/**
	 * 【TAG】Download 処理を行うかどうか[true:実行する/false:リンク作成]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * Download 処理を行うには、指定の画面(呼び出し元)と、抜き出した時の
	 * DBTableModel がマッチしているか判定します。
	 * また、DBTableModel が存在しない、または、不備な場合(オーバーフロー)は、
	 * 最終SQLを実行する必要があり、そもそも、抜出処理が出来ない状態では、
	 * エラーを返す必要があります。
	 * それら、一連のDownload 処理を行うかどうかを指定します。
	 * true の場合、session の HybsSystem.DB_LAST_SQL_KEY に格納された
	 * DBLastSql オブジェクトを取得し、scope が request または、overflow が
	 * true の場合に、BODY部を実行します。(通常、queryタグを記述します。)
	 * query の出力先は、scope を request にします。
	 * それ以外は、tableId から取り出した DBTableModel を、scope = "request"
	 * に(アドレスのみ)コピーします。
	 * writeTable タグでは、scope="request" から、DBTableModel を取り出します。
	 *
	 * 初期値は、false(抜き出しでは無くリンク作成) です。
	 *
	 * @param	type	Download処理 [true:実行する/false:リンク作成]
	 */
	public void setTypeDownload( final String type ) {
		typeDownload = nval( getRequestParameter( type ),typeDownload );
	}

	/**
	 * 【TAG】ダウンロード出来ない場合のエラーメッセージIDを指定します(初期値:ERR0042)。
	 *
	 * @og.tag
	 * DBTableModel が、最後に作成された画面と、メモリ中の DBLastSql オブジェクトが
	 * 管理している情報が異なる場合は、ダウンロード出来ません。
	 * 同様に、DBLastSql オブジェクト 自身が作成されていない場合も出来ません。
	 * そのようなケースに表示するエラーメッセージのメッセージIDを指定します。
	 * 引数は、指定できません。
	 * 初期値は、『ERR0042：出力対象データが見つかりませんでした。再度検索して下さい。』 です。
	 *
	 * @param	msg ダウンロード文書のフレーム名(ターゲット属性)
	 */
	public void setErrorMsg( final String msg ) {
		errorMsg = nval( getRequestParameter( msg ),errorMsg );
	}

	/**
	 * 【TAG】編集アイコンを指定します(初期値:"{&#064;SYS.JSP}/image/edit.png")。
	 *
	 * @og.tag
	 * 編集アイコンを指定します。
	 * 初期値:"{&#064;SYS.JSP}/image/edit.png" です。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 *
	 * @param	img アップロード時のアイコン
	 */
	public void setEditImg( final String img ) {
		editImg = nval( getRequestParameter( img ),editImg );
	}

	/**
	 * 【TAG】Edit画面でカラム編集タブを出すかどうか[true:出す/false:出さない]設定します(初期値:true:出す)。
	 *
	 * @og.tag
	 * falseを指定するとカラム編集タブを出さない。
	 *
	 * @param	type	編集タブ出力 [true:出す/false:出さない]
	 */
	public void setClmEdit( final String type ) {
		isEdit = nval( getRequestParameter( type ),isEdit );
	}

	/**
	 * 【TAG】Edit画面でカラム編集リンクを出力するかどうか[true:する/false:しない]設定します(初期値:true:する)。
	 *
	 * @og.tag
	 * falseを指定するとカラム編集リンクを出さない。
	 * 従来は、カラム編集画面の内、【表示項目設定】タブを表示しないだけでした。
	 * この設定は、RWMODE を、 ##d , ##u , ##f の場合にも同様の設定が可能です。
	 * カラム編集リンク自体を出さない機能は、現時点ではリソース設定できないため、
	 * タグのパラメータとして指定できるようにします。
	 * 基本的には、このパラメータはほとんど使われていないハズなので、
	 * 機能を置き換えます。
	 * 初期値は、カラム編集リンクを出力する(=true)です。
	 *
	 * @og.rev 5.7.5.0 (2014/04/04) 新規追加
	 *
	 * @param	flag	リンク出力 [true:する/false:しない]
	 * @see		org.opengion.hayabusa.taglib.QueryButtonTag#setUseEditMenu(String)
	 */
	public void setUseEditLink( final String flag ) {
		useEditLink = nval( getRequestParameter( flag ),useEditLink );
	}

	/**
	 * 【TAG】ダウンロード時のコマンドを追加指定できます(初期値:null)。
	 *
	 * @og.tag
	 * ダウンロードリンクが表示される command は、
	 * "NEW|RENEW|FIRST|LAST|PREV|NEXT|VIEW|RESET" のみです。
	 * これ以外の command で、ダウンロードリンクを表示させたい場合は、
	 * この、addDownCmnd に、記述します。
	 * 初期値は、null(追加コマンドはない)です。
	 *
	 * @og.rev 5.7.8.1 (2014/07/18) 新規追加
	 *
	 * @param	cmd	追加するコマンド
	 * @see		#setCommand( String )
	 */
	public void setAddDownCmnd( final String cmd ) {
		addDownCmnd = nval( getRequestParameter( cmd ),addDownCmnd );
	}

	/**
	 * 【TAG】アップロード時のコマンドを追加指定できます(初期値:null)。
	 *
	 * @og.tag
	 * アップロードリンクが表示される command は、"COPY|INSERT" のみです。
	 * これ以外の command で、アップロードリンクを表示させたい場合は、
	 * この、addUpldCmnd に、記述します。
	 * 初期値は、null(追加コマンドはない)です。
	 *
	 * @og.rev 5.7.8.1 (2014/07/18) 新規追加
	 *
	 * @param	cmd	追加するコマンド
	 * @see		#setCommand( String )
	 */
	public void setAddUpldCmnd( final String cmd ) {
		addUpldCmnd = nval( getRequestParameter( cmd ),addUpldCmnd );
	}

	/**
	 * 【TAG】リンクアドレスにURLチェック用の確認キーを付加するかどうか[true/false]を指定します。
	 *
	 * @og.tag
	 * この属性は、URLCheckFilter(org.opengion.hayabusa.filter.URLCheckFilter)と合わせて使用します。
	 * trueが指定された場合、リンクURLのパラメーターに、URL確認用のパラメーターが出力されます。
	 * このパラメーターを、URLCheckFilterが復号化し、正式なアクセスであるかどうかの判定を行います。
	 * 
	 * この属性はjsp/fileDownloadに対してチェックをかける場合にtrueにする必要があります。
	 * 通常はシステムリソースのFILEUD_URL_CHECKで設定します。
	 *
	 * 初期値は、 "false"(確認キーを出力しない) です。
	 *
	 * @og.rev 5.9.32.1 (2018/05/08) 追加
	 *
	 * @param	flag 暗号化するかかどうか [true:暗号化する/false:しない]
	 * @see org.opengion.hayabusa.filter.URLCheckFilter
	 */
	public void setUseURLCheck( final String flag ) {
		useURLCheck = nval( getRequestParameter( flag ),useURLCheck );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION		)
				.println( "command"		,command		)
				.println( "downloadImg"	,downloadImg	)
				.println( "downloadJsp"	,downloadJsp	)
				.println( "uploadImg"	,uploadImg		)
				.println( "uploadJsp"	,uploadJsp		)
				.println( "imgWidth"	,imgWidth		)
				.println( "imgHeight"	,imgHeight		)
				.println( "filename"	,filename		)
				.println( "roles"		,roles			)
				.println( "target"		,target			)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
