/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ToString;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.Attributes;

import org.opengion.fukurou.util.QrcodeImage;
import static org.opengion.fukurou.util.QrcodeImage.ErrCrct;
import static org.opengion.fukurou.util.QrcodeImage.EncMode;
import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.Locale ;

/**
 * QRコードに対応したイメージファイルを作成するタグです。
 *
 * QRコードで表示できる文字数は、バージョン、エンコードモード、エラー訂正レベル に依存します。
 * また、イメージの大きさは、文字数と１セル辺りのピクセルに依存します。
 * fileURLは、通常、システムリソースのFILE_URL(=filetemp)なので、fileURL="{&#064;USER.ID}" と
 * するのが一般的です。
 * ファイル名は、初期値:rqcode ですが、拡張子はimageType(初期値:PNG)を小文字化して追加します。
 * 拡張子が付いている場合は、そのまま使用されます。
 * また、同一ファイル名の場合、ブラウザキャッシュのため、画像が更新されないことがあるため
 * imgタグのsrc属性に、キャッシュの無効化のための引数を追加しています。
 *
 * @og.formSample
 * ●形式：&lt;og:qrCode fileURL="{&#064;USER.ID}" &gt;
 *             エンコードする文字列
 *         &lt;/og:qrCode &gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * または、
 *
 * ●形式：&lt;og:qrCode value="エンコードする文字列" /&gt;
 *
 * ●Tag定義：
 *   &lt;og:qrCode
 *       value              【TAG】エンコードする文字列(または、BODY部に記述)
 *       version            【TAG】バージョン (2から40の整数)(初期値:5)
 *       encodeMode         【TAG】エンコードモード('N':数字モード 'A':英数字モード　'B':8bit byteモード)(初期値:B)
 *       errCorrect         【TAG】エラー訂正レベル ('L','M','Q','H')(初期値:M)
 *       imageType          【TAG】イメージファイル形式(PNG/JPEG)(初期値:PNG)
 *       pixel              【TAG】１セル辺りの塗りつぶしピクセル(初期値:3)
 *       fileURL            【TAG】QRイメージファイルを出力するディレクトリ(初期値:FILE_URL)
 *       filename           【TAG】QRイメージファイル名 (初期値:rqcode)
 *       textEncode         【TAG】byteモード時のテキスト文字エンコード(初期値:Charset.defaultCharset()) 7.2.3.0 (2020/04/10)
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:qrCode&gt;
 *
 * @og.rev 7.2.1.0 (2020/03/13) 新規作成
 * @og.group 画面表示
 *
 * @version  7.2
 * @author	 Kazuhiko Hasegawa
 * @since    JDK11.0,
 */
public class QRcodeTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.2.1.0 (2020/03/13)" ;
	private static final long serialVersionUID = 721020200313L ;

	private static final int MAX_ALT_SIZE = 50 ;						// ｲﾒｰｼﾞに表示させる alt属性の文字数制限

	private String	value		;										// エンコードする文字列(または、BODY部に記述)
	private int		version		= QrcodeImage.DEF_VERSION ;				// バージョン (2から40の整数)(初期値:5)
	private EncMode	encMode		= EncMode.DEF;							// エンコードモード('N':数字モード 'A':英数字モード　'B':8bit byteモード)(初期値:B)
	private ErrCrct	errCrct		= ErrCrct.DEF;							// エラー訂正レベル ('L','M','Q','H')(初期値:M)
	private String	imgType		= QrcodeImage.IMAGE_TYPE;				// イメージファイル形式(PNG/JPEG)(初期値:PNG)
	private int		pixel		= QrcodeImage.PIXEL ;					// １セル辺りの塗りつぶしピクセル(初期値:3)
	private String	fileURL		= HybsSystem.sys( "FILE_URL" );			// QRイメージファイルを出力するディレクトリ(初期値:FILE_URL/{@USER.ID})
	private String	filename	= "rqcode";								// QRイメージファイル名 (初期値:rqcode)
	private String	textEncode	;										// byteモード時のテキスト文字エンコード(初期値:Charset.defaultCharset()) 7.2.3.0 (2020/04/10)

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 7.2.1.0 (2020/03/13) 新規作成
	 */
	public QRcodeTag() { super(); }			// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		// caseKey 、caseVal 属性対応
		if( useTag() ) {
			if( value == null || value.length() <= 0 ) {
				return EVAL_BODY_BUFFERED ;		// Body を評価する
			}
		}
		return SKIP_BODY ;				// Body を評価しない
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		value = getBodyString();

		return SKIP_BODY ;
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 7.2.3.0 (2020/04/10) textEncode byteモード時のテキスト文字エンコード追加
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();
		// caseKey 、caseVal 属性対応
		if( useTag() ) {
			if( filename.indexOf( '.' ) < 0 ) {				// 拡張子が存在しない
				filename = filename + "." + imgType.toLowerCase(Locale.JAPAN);
			}

			try {
				final String saveFile = HybsSystem.url2dir( fileURL,filename );

				final QrcodeImage qrcode = new QrcodeImage();
//				qrcode.init( value,saveFile,version,encMode,errCrct,imgType,pixel );
				qrcode.init( value,saveFile,version,encMode,errCrct,imgType,pixel,textEncode );		// 7.2.3.0 (2020/04/10)
				qrcode.saveImage();
			}
			catch( final Throwable th ) {
				final String errMsg = "QRコードが作成できませんでした。"					+ CR
								+ "指定のパラメータが正しいかどうかご確認ください。"		+ CR
								+ "  version    = [" + version	 + "] バージョン (2から40の整数)"						+ CR
								+ "  encodeMode = [" + encMode	 + "] エンコードモード('N':数字 'A':英数字'B':byte)"	+ CR
								+ "  errCorrect = [" + errCrct	 + "] エラー訂正レベル ('L','M','Q','H')"	+ CR
								+ "  imageType  = [" + imgType	 + "] イメージファイル形式(PNG/JPEG)"		+ CR
								+ "  pixel      = [" + pixel	 + "] １セル辺りの塗りつぶしピクセル"		;
				throw new HybsSystemException( errMsg,th );
			}

			filename = filename + "?val=" + System.currentTimeMillis() ;		// 同じファイル名の場合のキャッシュの無効化
			final String src = StringUtil.urlAppend( getContextPath() , fileURL , filename );
			final String alt = value.substring( 0,Math.min( value.length() , MAX_ALT_SIZE ) );

			// 作成された QRコードのイメージを表示します。
			final String img = XHTMLTag.img(
					new Attributes()
						.set( "src"		, src )
						.set( "alt"		, alt )
						.set( "title"	, alt )
			);

			jspPrint( img );
		}
		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 7.2.3.0 (2020/04/10) textEncode byteモード時のテキスト文字エンコード追加
	 */
	@Override
	protected void release2() {
		super.release2();
		value		= null;										// エンコードする文字列(または、BODY部に記述)
		version		= QrcodeImage.DEF_VERSION;					// バージョン (2から40の整数)(初期値:5)
		encMode		= EncMode.DEF;								// エンコードモード('N':数字モード 'A':英数字モード　'B':8bit byteモード)(初期値:B)
		errCrct		= ErrCrct.DEF ;								// エラー訂正レベル ('L','M','Q','H')(初期値:M)
		imgType		= QrcodeImage.IMAGE_TYPE;					// イメージファイル形式(PNG/JPEG)(初期値:PNG)
		pixel		= QrcodeImage.PIXEL ;						// １セル辺りの塗りつぶしピクセル(初期値:3)
		fileURL		= HybsSystem.sys( "FILE_URL" );				// QRイメージファイルを出力するディレクトリ(初期値:FILE_URL/{@USER.ID})
		filename	= "rqcode";									// QRイメージファイル名 (初期値:rqcode)
		textEncode	= null;										//  7.2.3.0 (2020/04/10) byteモード時のテキスト文字エンコード
	}

	/**
	 * 【TAG】エンコードする文字列(または、BODY部に記述)を指定します。
	 *
	 * @og.tag
	 * エンコードする文字列のバイト数は、バージョン、エンコードモード、エラー訂正レベルに依存します。
	 * また、イメージの大きさは、それらプラス１セル辺りのピクセルも影響します。
	 *
	 * @param	val エンコードする文字列(または、BODY部に記述)
	 */
	public void setValue( final String val ) {
		value = nval( getRequestParameter( val ),value );
	}

	/**
	 * 【TAG】バージョン (2から40の整数)を指定します(初期値:5)。
	 *
	 * @og.tag
	 * エンコードする文字列のバイト数は、エラー訂正レベル、バージョン に依存します。
	 * 文字列のバイト数を増やす場合は、バージョンを適切に設定します。
	 *
	 * @param	ver バージョン
	 */
	public void setVersion( final String ver ) {
		version = nval( getRequestParameter( ver ),version );
	}

	/**
	 * 【TAG】エンコードモード('N':数字モード 'A':英数字モード　'B':8bit byteモード)を指定します(初期値:B)。
	 *
	 * @og.tag
	 * エンコードする文字列の種類に応じて設定します。
	 * 日本語等を含む場合は、'B':8bit byteモード にしてください。
	 *
	 * @param	mode エンコードモード
	 */
	public void setEncodeMode( final String mode ) {
		final String em = nval( getRequestParameter( mode ),null );
		if( em != null ) {
			encMode = EncMode.get( em.charAt(0) );
		}
	}

	/**
	 * 【TAG】エラー訂正レベル ('L','M','Q','H')を指定します(初期値:M)。
	 *
	 * @og.tag
	 * エンコードする文字列のバイト数は、エラー訂正レベル、バージョン に依存します。
	 * 通常、初期値のままで問題ありません。
	 *
	 * @param	crct エラー訂正レベル
	 */
	public void setErrCorrect( final String crct ) {
		final String ec = nval( getRequestParameter( crct ),null );
		if( ec != null ) {
			errCrct = ErrCrct.get( ec.charAt(0) );
		}
	}

	/**
	 * 【TAG】イメージファイル形式(PNG/JPEG)を指定します(初期値:PNG)。
	 *
	 * @og.tag
	 * QRコードのイメージファイルの形式を指定します。
	 * 拡張子は自動的にイメージファイル形式(の小文字)がセットされます。
	 *
	 * @param	type イメージファイル形式
	 */
	public void setImageType( final String type ) {
		imgType = nval( getRequestParameter( type ),imgType );
	}

	/**
	 * 【TAG】１セル辺りの塗りつぶしピクセルを指定します(初期値:3)。
	 *
	 * @og.tag
	 * QRコードのイメージファイルの形式を指定します。
	 * 拡張子は自動的にイメージファイル形式(の小文字)がセットされます。
	 *
	 * @param	px ピクセル数
	 */
	public void setPixel( final String px ) {
		pixel = nval( getRequestParameter( px ),pixel );
	}

	/**
	 * 【TAG】 QRイメージファイルを出力するディレクトリを指定します(初期値:FILE_URL)。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリに、QRイメージファイルをセーブします。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' (UNIX) または、２文字目が、
	 * ":" (Windows)の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * FILE_URL 属性で指定のフォルダの下に、フォルダを作成します。
	 * 初期値は、FILE_URL になるため、通常は{&#064;USER.ID}を指定する必要があります。
	 *
	 * @param	url 保存先ディレクトリ名
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_URL
	 */
	public void setFileURL( final String url ) {
		final String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】QRイメージファイル名をセットします(初期値:rqcode)。
	 *
	 * @og.tag
	 * ファイルを作成するときのファイル名をセットします。
	 * ファイル名の拡張子は、imageType属性の小文字を追加します。
	 * 拡張子付きで指定した場合(ファイル名に、ピリオドを含む場合)は、
	 * そのままの値を使用します。
	 *
	 * @param   fname ファイル名
	 */
	public void setFilename( final String fname ) {
		filename = nval( getRequestParameter( fname ),filename );
	}

	/**
	 * 【TAG】byteモード時のテキスト文字エンコードをセットします(初期値:環境依存)。
	 *
	 * @og.tag
	 * ﾃｷｽﾄのｴﾝｺｰﾄﾞの指定がない場合は、プラットフォーム依存のデフォルトの Charset です。
	 * java.nio.charset.Charset#defaultCharset()
	 * QRｺｰﾄﾞで、機種依存文字(①など)は、Windows-31J を指定しても読み取り側が対応していません。
	 * その場合は、UTF-8 を指定します。(必要なﾊﾞｲﾄ数は当然増えます)
	 *
	 * 通常、何も指定しないか、UTF-8 を指定するかのどちらかになります。
	 *
	 * @og.rev 7.2.3.0 (2020/04/10) textEncode byteモード時のテキスト文字エンコード追加
	 *
	 * @param   txtEnc テキスト文字エンコード
	 */
	public void setTextEncode( final String txtEnc ) {
		textEncode = nval( getRequestParameter( txtEnc ),textEncode );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			, VERSION			)
				.println( "value"			, value				)
				.println( "version"			, version			)
				.println( "encMode"			, encMode			)
				.println( "errCrct"			, errCrct			)
				.println( "imgType"			, imgType			)
				.println( "pixel"			, pixel				)
				.println( "fileURL"			, fileURL			)
				.println( "filename"		, filename			)
				.fixForm().toString() ;
	}
}

