/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.TagBuffer;

import org.opengion.hayabusa.db.Selection;								// 6.4.4.0 (2016/03/11)
import org.opengion.hayabusa.db.SelectionFactory;						// 6.4.4.0 (2016/03/11)

/**
 * CHBOX2 エディターは、カラムのデータをチェックボックスで編集する場合に使用するクラスです。
 *
 * このエディターは、CHBOXとは異なりチェックボックス特有の制御を全く行いません。
 * 特性としては、typeがcheckboxであるという1点を除いて、TEXT エディターと同じです
 *
 * ラベル表示が必要な場合は、編集パラメータに、"useLabel"と
 * 記述しておくことで、ラベルを出力することが可能です。
 *
 * カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * 6.4.4.0 (2016/03/11)
 * チェックボックスとして、チェックした値を送信する機能に、コードリソースの値を
 * 指定できるようにします。これは、プルダウンメニューのマルチプルと同じ効果が
 * 期待できます。
 * コードリソースは、無くてもかまいません。
 *
 * @og.group データ編集
 *
 * @version  6.4
 * @author	 Kazuhiko Hasegawa
 * @since    JDK8.0,
 */
public class Editor_CHBOX2 extends AbstractEditor {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.0.5.1 (2019/09/27)" ;

	// 6.4.4.0 (2016/03/11) CHBOX2 は、Selection は、必須ではなく、定義されていれば使う程度
	private final Selection	selection ;		// 6.4.4.0 (2016/03/11)
	private final boolean writable	;
	private final boolean useLabel	;		// 4.3.3.0 (2008/10/01)

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 6.4.4.0 (2016/03/11) CHBOX2は、コードリソースも使用できるように変更。
	 */
	public Editor_CHBOX2() {
		super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		selection = null;
		writable  = false;
		useLabel  = false;
	}

	/**
	 * DBColumnオブジェクトを指定したprivateコンストラクター。
	 *
	 * @og.rev 6.3.3.0 (2015/07/25) maxlength は不要なので削除
	 * @og.rev 6.4.4.0 (2016/03/11) CHBOX2は、コードリソースも使用できるように変更。
	 * @og.rev 7.0.5.1 (2019/09/27) optionAttributes が二重に設定されていたため、削除
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Editor_CHBOX2( final DBColumn clm ) {
		super( clm );
		attributes.set( "type"		, "checkbox"	);
		attributes.set( "NO_MAXLEN"	, "true"		);			// 6.3.3.0 (2015/07/25) maxlength は不要なので削除

		tagBuffer.add( XHTMLTag.inputAttri( attributes ) );

		// CHBOX2 は、コードリソース(selection)が存在しない場合もありうる。
		final String addKeyLabel = clm.getAddKeyLabel();		// 6.2.0.0 (2015/02/27) キー:ラベル形式
		selection = SelectionFactory.newSelection( "CHBOX" , clm.getCodeData(), addKeyLabel );

		// 6.4.4.0 (2016/03/11) CHBOX2は、コードリソースも使用できるように変更。
		writable = clm.isWritable();

		// 6.1.1.0 (2015/01/17) Attributesの連結記述
		attributes = new Attributes()
					.set( clm.getEditorAttributes() )				// #addAttributes( Attributes ) の代替え
					.add( "class","CHBOX" );

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		// 7.0.5.1 (2019/09/27) optionAttributes が二重に設定されていたため、削除
		tagBuffer.add( XHTMLTag.inputAttri( attributes ) );
//				 .add( attributes.get( "optionAttributes" ) );		// 6.0.4.0 (2014/11/28)

		useLabel = "useLabel".equalsIgnoreCase( clm.getEditorParam() );
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_CHBOX2( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * @og.rev 6.4.4.0 (2016/03/11) チェックボックスの横に値をラベル表示します
	 *
	 * @param	value 値
	 *
	 * @return	データの編集用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		final String chbox ;

		if( selection == null ) {
			chbox = new StringBuilder( BUFFER_MIDDLE )
							.append( "<label>" )
							.append( super.getValue( value ) )
							.append( value )
							.append( "</label>" )
							.toString();
		}
		else {
			if( writable ) {
				chbox = selection.getOption( name,value,true );			// 6.2.2.4 (2015/04/24)
			}
			else {
				chbox = selection.getValueLabel( value,true );			// 6.2.2.4 (2015/04/24) useLabel
			}
		}

		return new TagBuffer( "pre" )
					.add( tagBuffer.makeTag() )
					.addBody( chbox )
					.makeTag();
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @og.rev 6.4.4.0 (2016/03/11) チェックボックスの横に値をラベル表示します。
	 *
	 * @param	row   行番号
	 * @param	value 値
	 *
	 * @return	データ表示/編集用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {
		final String chbox ;

		if( selection == null ) {
			chbox = new StringBuilder( BUFFER_MIDDLE )
						.append( "<label>" )
						.append( super.getValue( row,value ) )
						.append( useLabel ? value : "" )
						.append( "</label>" )
						.toString();
		}
		else {
			if( writable ) {
				chbox = selection.getOption( name + HybsSystem.JOINT_STRING + row,value,useLabel );		// 6.2.2.4 (2015/04/24)
		}
		else {
				chbox = selection.getValueLabel( value,useLabel );		// 6.2.2.4 (2015/04/24)
			}
		}

		return new TagBuffer( "pre" )
					.add( tagBuffer.makeTag() )
					.addBody( chbox )
					.makeTag( row,value );
	}
}
