/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.HybsFileFilter;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)

import static org.opengion.fukurou.util.StringUtil.nval ;

/**
 * fileOption でファイルを検索する場合の条件を指定するタグです。
 *
 * @og.formSample
 * ●形式：&lt;og:fileWhere name="…" startsWith="…" endsWith="…" … /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:fileWhere
 *       startsWith         【TAG】指定された接頭辞で始まる場合、スルー(選択)されます(初期値:null)
 *       unStartsWith       【TAG】(追加：7.2.6.0 (2020/06/30))指定された接頭辞で始まらない場合、スルー(選択)されます(初期値:null)
 *       startsDir          【TAG】(追加：7.2.6.0 (2020/06/30))指定された接頭辞で始まるフォルダ場合、スルー(選択)されます(初期値:null)
 *       unStartsDir        【TAG】(追加：7.2.6.0 (2020/06/30))指定された接頭辞で始まらないフォルダ場合、スルー(選択)されます(初期値:null)
 *       endsWith           【TAG】指定された接頭辞で終わる場合、スルー(選択)されます(初期値:null)
 *       unEndsWith         【TAG】(追加：7.2.6.0 (2020/06/30))指定された接頭辞で終わらない場合、スルー(選択)されます(初期値:null)
 *       instr              【TAG】指定された文字列がファイル名に含まれる場合、スルー(選択)されます(初期値:null)
 *       unInstr            【TAG】(追加：7.2.6.0 (2020/06/30))指定された文字列がファイル名に含まれない場合、スルー(選択)されます(初期値:null)
 *       instrDir           【TAG】(追加：7.2.6.0 (2020/06/30))指定された文字列がフォルダ名に含まれる場合、スルー(選択)されます(初期値:null)
 *       unInstrDir         【TAG】(追加：7.2.6.0 (2020/06/30))指定された文字列がフォルダ名に含まれない場合、スルー(選択)されます(初期値:null)
 *       equals             【TAG】ファイル名が一致する場合、スルー(選択)されます(初期値:null)
 *       unEquals           【TAG】(追加：7.2.6.0 (2020/06/30))ファイル名が一致しない場合、スルー(選択)されます(初期値:null)
 *       matches            【TAG】ファイル名が、指定された正規表現と一致する場合、スルー(選択)されます(初期値:null)
 *       unMatches          【TAG】ファイル名が、指定された正規表現と一致しない場合、スルー(選択)されます(初期値:null)
 *       matchDir           【TAG】(追加：7.2.6.0 (2020/06/30))フォルダ名が、指定された正規表現と一致する場合、スルー(選択)されます(初期値:null)
 *       unMatchDir         【TAG】(追加：7.2.6.0 (2020/06/30))フォルダ名が、指定された正規表現と一致しない場合、スルー(選択)されます(初期値:null)
 *       lastModified       【TAG】指定のタイムスタンプ以後に変更されている場合、スルー(選択)されます(初期値:null)
 *       unLastModified     【TAG】(追加：7.2.6.0 (2020/06/30))指定のタイムスタンプ以前に変更されている場合、スルー(選択)されます(初期値:null)
 *       isLarger           【TAG】指定の大きさより大きいファイルの場合、スルー(選択)されます(初期値:null)
 *       isSmaller          【TAG】指定の大きさより小さいファイルの場合、スルー(選択)されます(初期値:null)
 *       isHidden           【TAG】true:HIDDENファイルのみ/false:NORMALファイルのみスルー(選択)されます(初期値:null)
 *       useDir             【廃止】(廃止：7.2.6.0 (2020/06/30))判定をファイルだけでなく、ディレクトリでも行うかどうかを指定[false:File/true:File+Dir/only:Dir](初期値:false)
 *       ignoreCase         【TAG】大文字/小文字を区別しないかどうか[true/false]を指定します(初期値:false(区別する))
 *       notEquals          【TAG】判定結果を反転させるかどうか[true/false]を指定します(初期値:false) 6.8.0.0 (2017/06/02)
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null) 6.8.0.0 (2017/06/02)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null) 6.8.0.0 (2017/06/02)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない) 6.8.0.0 (2017/06/02)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない) 6.8.0.0 (2017/06/02)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない) 6.8.0.0 (2017/06/02)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *     &lt;og:fileOption select="NAME" from="./jsp" &gt;
 *         &lt;og:fileWhere
 *             startsWith = "File"          :指定された接頭辞で始まる場合、選択される
 *             endsWith   = ".java"         :指定された接頭辞で終わる場合、選択される
 *             instr      = "Tag"           :指定された文字列がファイル名に含まれる場合、選択される
 *             equals     = "FileWhereTag.java" :ファイル名が一致する場合、選択される(大文字小文字は区別しない)
 *             matches    = "File*Tag"      :ファイル名が、指定された正規表現と一致する場合、選択される(大文字小文字は区別しない)
 *             unMatches  = "File*Tag"      :ファイル名が、指定された正規表現と一致しない場合、選択される(大文字小文字は区別しない)
 *             lastModified = "20050101"    :指定のタイムスタンプ以後に変更された場合、選択される
 *                                           日付けの指定は、YYYYMMDD 形式, TODAY , YESTERDAY が使用できます。
 *             isLarger   = "1024"          :指定の大きさ(バイト単位)より大きいファイルの場合、選択される
 *             isSmaller  = "1024"          :指定の大きさ(バイト単位)より小さいファイルの場合、選択される
 *             isHidden   = "true"          :true:HIDDENファイルのみ選択される
 *             useDir     = "false"         :(廃止)フィルタによる対象の絞込みをディレクトリにも適用するかどうか
 *         /&gt;
 *     &lt;/og:fileOption&gt;
 *
 *       ・検索条件が入力された時
 *         条件を満足して全てのファイルを列挙します。
 *
 *       ・検索条件が入力されなかった時
 *         該当ディレクトリ内の全てのファイルを列挙します。
 *
 * @og.rev 2.1.1.0 (2002/11/11) 新規作成
 * @og.rev 4.0.0.0 (2005/01/31) 内部ロジックの全面見直し
 * @og.rev 7.2.6.0 (2020/06/30) 内部ロジックの全面見直し
 * @og.group その他入力
 *
 * @version  7.2
 * @author   Kazuhiko Hasegawa
 * @since    JDK11.0,
 */
public class FileWhereTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.8.0.0 (2017/06/02)" ;
	private static final long serialVersionUID = 680020170602L ;

	private String	startsWith		;		// 指定された接頭辞で始まる場合、選択される
	private String	unStartsWith	;		// (追加：7.2.6.0 (2020/06/30))指定された接頭辞で始まらない場合、スルー(選択)されます(初期値:null)
	private String	startsDir		;		// (追加：7.2.6.0 (2020/06/30))指定された接頭辞で始まるフォルダ場合、スルー(選択)されます(初期値:null)
	private String	unStartsDir		;		// (追加：7.2.6.0 (2020/06/30))指定された接頭辞で始まらないフォルダ場合、スルー(選択)されます(初期値:null)
	private String	endsWith		;		// 指定された接頭辞で終わる場合、選択される
	private String	unEndsWith		;		// (追加：7.2.6.0 (2020/06/30))指定された接頭辞で終わらない場合、スルー(選択)されます(初期値:null)
	private String	instr			;		// 指定された文字列がファイル名に含まれる場合、選択される
	private String	unInstr			;		// (追加：7.2.6.0 (2020/06/30))指定された文字列がファイル名に含まれない場合、スルー(選択)されます(初期値:null)
	private String	instrDir		;		// (追加：7.2.6.0 (2020/06/30))指定された文字列がフォルダ名に含まれる場合、スルー(選択)されます(初期値:null)
	private String	unInstrDir		;		// (追加：7.2.6.0 (2020/06/30))指定された文字列がフォルダ名に含まれない場合、スルー(選択)されます(初期値:null)
	private String	fileEquals		; 		// ファイル名が一致する場合、選択される(大文字小文字は区別しない)
	private String	matches			;		// ファイル名が、指定された正規表現と一致する場合、選択される(大文字小文字は区別しない)
	private String	unMatches		;		// ファイル名が、指定された正規表現と一致しない場合、選択される(大文字小文字は区別しない)
	private String	matchDir		;		// (追加：7.2.6.0 (2020/06/30))フォルダ名が、指定された正規表現と一致する場合、スルー(選択)されます(初期値:null)
	private String	unMatchDir		;		// (追加：7.2.6.0 (2020/06/30))フォルダ名が、指定された正規表現と一致しない場合、スルー(選択)されます(初期値:null)
	private String	lastModified	;		// 指定のタイムスタンプ以後に変更された場合、選択される
	private String	unLastModified	;		// (追加：7.2.6.0 (2020/06/30))指定のタイムスタンプ以前に変更されている場合、スルー(選択)されます(初期値:null)
	private String	isLarger		;		// 指定の大きさ(バイト単位)より大きいファイルの場合、選択される
	private String	isSmaller		;		// 指定の大きさ(バイト単位)より小さいファイルの場合、選択される
	private String	isHidden		;		// 5.7.5.0 (2014/04/04) true:HIDDENファイルのみ/false:NORMALファイルのみスルー(選択)されます
//	private String	useDir	= "false";		// (廃止：7.2.6.0 (2020/06/30))6.4.0.2 (2015/12/11) useDIR 属性に、onlyパラメータを追加します[false:File/true:File+Dir/only:Dir]。
	private boolean	ignoreCase		;		// 6.3.1.1 (2015/07/10) 大文字/小文字の区別なしで検索可能にするかどうか(初期値:false:区別する)
	private boolean notEquals		;		// 6.8.0.0 (2017/06/02) 判定結果を反転させて処理します。

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public FileWhereTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 5.1.2.0 (2010/01/01) unMatches メソッドで反転属性の追加
	 * @og.rev 5.3.9.0 (2011/09/01) useDir属性を追加
	 * @og.rev 5.6.6.0 (2013/07/05) unMatchesの引数間違いの訂正
	 * @og.rev 5.7.5.0 (2014/04/04) isHidden属性を追加します。
	 * @og.rev 6.3.1.1 (2015/07/10) メソッドチェーン化と、大文字/小文字の区別なし(ignoreCase=true)対応
	 * @og.rev 6.4.0.2 (2015/12/11) useDIR 属性に、onlyパラメータを追加します[false:File/true:File+Dir/only:Dir]。
	 * @og.rev 6.8.0.0 (2017/06/02) caseKey,caseVal,caseNN,caseNull 属性を追加
	 * @og.rev 7.2.6.0 (2020/06/30) ディレクトリ処理を統一します。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		// 6.8.0.0 (2017/06/02) caseKey,caseVal,caseNN,caseNull 属性を追加
		if( !useTag() ) { return EVAL_PAGE ; }

		// 6.3.1.1 (2015/07/10) メソッドチェーン化と、大文字/小文字の区別なし(ignoreCase=true)対応
		// 6.8.0.0 (2017/06/02) notEquals で、判定結果を反転させて処理します。
//		final HybsFileFilter filter = new HybsFileFilter( useDir,ignoreCase,notEquals )
		final HybsFileFilter filter = new HybsFileFilter( ignoreCase,notEquals )			// 7.2.6.0 (2020/06/30) useDIR 属性 廃止
			.startsWith(	startsWith		)
			.startsWith(	unStartsWith, true  )	// 7.2.6.0 (2020/06/30)
			.startsDir(		startsDir		)		// 7.2.6.0 (2020/06/30)
			.startsDir(		unStartsDir	, true	)	// 7.2.6.0 (2020/06/30)
			.endsWith(		endsWith		)
			.endsWith(		unEndsWith	, true  )	// 7.2.6.0 (2020/06/30)
			.instr(			instr			)
			.instr(			unInstr		, true  )	// 7.2.6.0 (2020/06/30)
			.instrDir(		instrDir	)			// 7.2.6.0 (2020/06/30)
			.instrDir(		unInstrDir	, true  )	// 7.2.6.0 (2020/06/30)
			.fileEquals(	fileEquals		)
			.matches(		matches			)
			.matches(		unMatches	, true )	// 7.2.6.0 (2020/06/30)
			.matchDir(		matchDir		)		// 7.2.6.0 (2020/06/30)
			.matchDir(		unMatchDir	, true  )	// 7.2.6.0 (2020/06/30)
			.lastModified(	lastModified	)
			.lastModified(	unLastModified	, true	)	// 7.2.6.0 (2020/06/30)
			.isLarger(	 	isLarger		)
			.isSmaller(	 	isSmaller		)
			.isHidden(	 	isHidden		);		// 5.7.5.0 (2014/04/04) 追加

		// ファイルフィルタを追加
		final FileQueryTag fileQuery = (FileQueryTag)findAncestorWithClass( this, FileQueryTag.class );
		if( fileQuery != null ) {
			fileQuery.setFileFilter( filter );
			return EVAL_PAGE ;
		}

		final FileOptionTag fileOption = (FileOptionTag)findAncestorWithClass( this, FileOptionTag.class );
		if( fileOption != null ) {
			fileOption.setFileFilter( filter );
			return EVAL_PAGE ;
		}

		// どちらにも属していなかった場合は、エラーになる。
		final String errMsg = "この、FileWhereTag タグは、FileQueryTag か、FileOption の"
					+ "BODY 部にのみ、配置できます。(それ以外の箇所での使用はできません。)" ;
		throw new HybsSystemException( errMsg );
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 4.0.0.0 (2005/01/31) 内部ロジックの全面見直し
	 * @og.rev 5.3.9.0 (2011/09/01) useDir属性追加
	 * @og.rev 5.7.4.3 (2014/03/28) isLarger,isSmaller属性を文字列に変更
	 * @og.rev 5.7.5.0 (2014/04/04) isHidden属性を追加
	 * @og.rev 6.3.1.1 (2015/07/10) 大文字/小文字の区別なし(ignoreCase=true)対応
	 * @og.rev 6.4.0.2 (2015/12/11) useDIR 属性に、onlyパラメータを追加します[false:File/true:File+Dir/only:Dir]。
	 * @og.rev 6.8.0.0 (2017/06/02) notEquals 属性追加 , useDir="false" の初期値設定
	 * @og.rev 7.2.6.0 (2020/06/30) ディレクトリ処理を統一します。
	 */
	@Override
	protected void release2() {
		super.release2();
		startsWith		= null;
		unStartsWith	= null;		// 7.2.6.0 (2020/06/30)
		startsDir		= null;		// 7.2.6.0 (2020/06/30)
		unStartsDir		= null;		// 7.2.6.0 (2020/06/30)
		endsWith		= null;
		unEndsWith		= null;		// 7.2.6.0 (2020/06/30)
		instr			= null;
		unInstr			= null;		// 7.2.6.0 (2020/06/30)
		instrDir		= null;		// 7.2.6.0 (2020/06/30)
		unInstrDir		= null;		// 7.2.6.0 (2020/06/30)
		fileEquals		= null;
		matches			= null;
		unMatches		= null;
		matchDir		= null;		// 7.2.6.0 (2020/06/30)
		unMatchDir		= null;		// 7.2.6.0 (2020/06/30)
		lastModified	= null;
		unLastModified	= null;		// 7.2.6.0 (2020/06/30)
		isLarger		= null;		// 5.7.4.3 (2014/03/28) isLarger,isSmaller属性を文字列に変更
		isSmaller		= null;		// 5.7.4.3 (2014/03/28) isLarger,isSmaller属性を文字列に変更
		isHidden		= null;		// 5.7.5.0 (2014/04/04) isHidden属性を追加
//		useDir			= "false";	// 6.4.0.2 (2015/12/11) useDIR 属性に、onlyパラメータを追加	// 7.2.6.0 (2020/06/30) useDIR 属性 廃止
		ignoreCase		= false;	// 大文字/小文字の区別
		notEquals		= false;	// 6.8.0.0 (2017/06/02) notEquals 属性追加
	}

	/**
	 * 【TAG】指定された接頭辞で始まる場合、スルー(選択)されます(初期値:null)。
	 * @og.tag
	 * 引数が null の場合は、追加しません。
	 *
	 * @param    prefix 接頭辞
	 * @see java.lang.String#startsWith(String)
	 */
	public void setStartsWith( final String prefix ) {
		startsWith = nval( getRequestParameter( prefix ),startsWith );
	}

	/**
	 * 【TAG】指定された接頭辞で始まらない場合、スルー(選択)されます(初期値:null)。
	 * @og.tag
	 * 引数が null の場合は、追加しません。
	 *
	 * @og.rev 7.2.6.0 (2020/06/30) ディレクトリ処理を統一します。
	 *
	 * @param    prefix 接頭辞
	 * @see java.lang.String#startsWith(String)
	 */
	public void setUnStartsWith( final String prefix ) {
		unStartsWith = nval( getRequestParameter( prefix ),unStartsWith );
	}

	/**
	 * 【TAG】指定された接頭辞で始まるフォルダ場合、スルー(選択)されます(初期値:null)。
	 * @og.tag
	 * 引数が null の場合は、追加しません。
	 *
	 * @og.rev 7.2.6.0 (2020/06/30) ディレクトリ処理を統一します。
	 *
	 * @param    prefix 接頭辞
	 * @see java.lang.String#startsWith(String)
	 */
	public void setStartsDir( final String prefix ) {
		startsDir = nval( getRequestParameter( prefix ),startsDir );
	}

	/**
	 * 【TAG】指定された接頭辞で始まらないフォルダ場合、スルー(選択)されます(初期値:null)。
	 * @og.tag
	 * 引数が null の場合は、追加しません。
	 *
	 * @og.rev 7.2.6.0 (2020/06/30) ディレクトリ処理を統一します。
	 *
	 * @param    prefix 接頭辞
	 * @see java.lang.String#startsWith(String)
	 */
	public void setUnStartsDir( final String prefix ) {
		unStartsDir = nval( getRequestParameter( prefix ),unStartsDir );
	}

	/**
	 * 【TAG】指定された接頭辞で終わる場合、スルー(選択)されます(初期値:null)。
	 * @og.tag
	 * 引数が null の場合は、追加しません。
	 *
	 * @param    suffix 接尾辞
	 * @see java.lang.String#endsWith(String)
	 */
	public void setEndsWith( final String suffix ) {
		endsWith = nval( getRequestParameter( suffix ),endsWith );
	}

	/**
	 * 【TAG】指定された接頭辞で終わらない場合、スルー(選択)されます(初期値:null)。
	 * @og.tag
	 * 引数が null の場合は、追加しません。
	 *
	 * @og.rev 7.2.6.0 (2020/06/30) ディレクトリ処理を統一します。
	 *
	 * @param    suffix 接尾辞
	 * @see java.lang.String#endsWith(String)
	 */
	public void setUuEndsWith( final String suffix ) {
		unEndsWith = nval( getRequestParameter( suffix ),unEndsWith );
	}

	/**
	 * 【TAG】指定された文字列がファイル名に含まれる場合、スルー(選択)されます(初期値:null)。
	 * @og.tag
	 * 引数が null の場合は、追加しません。
	 *
	 * @param    str 指定の部分文字列
	 */
	public void setInstr( final String str ) {
		instr = nval( getRequestParameter( str ),instr );
	}

	/**
	 * 【TAG】指定された文字列がファイル名に含まれない場合、スルー(選択)されます(初期値:null)。
	 * @og.tag
	 * 引数が null の場合は、追加しません。
	 *
	 * @og.rev 7.2.6.0 (2020/06/30) ディレクトリ処理を統一します。
	 *
	 * @param    str 指定の部分文字列
	 * @see java.lang.String#endsWith(String)
	 */
	public void setUuInstr( final String str ) {
		unInstr = nval( getRequestParameter( str ),unInstr );
	}

	/**
	 * 【TAG】指定された文字列がフォルダ名に含まれる場合、スルー(選択)されます(初期値:null)。
	 * @og.tag
	 * 引数が null の場合は、追加しません。
	 *
	 * @og.rev 7.2.6.0 (2020/06/30) ディレクトリ処理を統一します。
	 *
	 * @param    str 指定の部分文字列
	 * @see java.lang.String#endsWith(String)
	 */
	public void setIinstrDir( final String str ) {
		instrDir = nval( getRequestParameter( str ),instrDir );
	}

	/**
	 * 【TAG】指定された文字列がフォルダ名に含まれない場合、スルー(選択)されます(初期値:null)。
	 * @og.tag
	 * 引数が null の場合は、追加しません。
	 *
	 * @og.rev 7.2.6.0 (2020/06/30) ディレクトリ処理を統一します。
	 *
	 * @param    str 指定の部分文字列
	 * @see java.lang.String#endsWith(String)
	 */
	public void setUnInstrDir( final String str ) {
		unInstrDir = nval( getRequestParameter( str ),unInstrDir );
	}

	/**
	 * 【TAG】ファイル名が一致する場合、スルー(選択)されます(初期値:null)。
	 * @og.tag
	 * 大文字小文字は区別しません。
	 * 引数が null の場合は、追加しません。
	 *
	 * @param    str ファイル名文字列
	 * @see java.lang.String#equals(Object)
	 */
	public void setEquals( final String str ) {
		fileEquals = nval( getRequestParameter( str ),fileEquals );
	}

	/**
	 * 【TAG】ファイル名が、指定された正規表現と一致する場合、スルー(選択)されます(初期値:null)。
	 * @og.tag
	 *
	 * ignoreCase を使用する場合、Pattern.CASE_INSENSITIVE を適用します。
	 * Pattern.compile( str,Pattern.CASE_INSENSITIVE ) ;
	 * pattern.matcher( pathname.getName() ).find() == true と同じ結果が得られます。
	 * 引数が null の場合は、追加しません。
	 *
	 * @param    str ファイル名文字列(正規表現)
	 * @see java.util.regex.Pattern#compile(String,int)
	 * @see java.util.regex.Matcher#find()
	 */
	public void setMatches( final String str ) {
		matches = nval( getRequestParameter( str ),matches );
	}

	/**
	 * 【TAG】ファイル名が、指定された正規表現と一致しない場合、スルー(選択)されます(初期値:null)。
	 * @og.tag
	 *
	 * ignoreCase を使用する場合、Pattern.CASE_INSENSITIVE を適用します。
	 * Pattern.compile( str,Pattern.CASE_INSENSITIVE ) ;
	 * pattern.matcher( pathname.getName() ).find() == false と同じ結果が得られます。
	 * 引数が null の場合は、追加しません。
	 *
	 * @param    str ファイル名文字列(不一致させる正規表現)
	 * @see java.util.regex.Pattern#compile(String,int)
	 * @see java.util.regex.Matcher#find()
	 */
	public void setUnMatches( final String str ) {
		unMatches = nval( getRequestParameter( str ),unMatches );
	}

	/**
	 * 【TAG】フォルダ名が、指定された正規表現と一致する場合、スルー(選択)されます(初期値:null)。
	 * @og.tag
	 *
	 * ignoreCase を使用する場合、Pattern.CASE_INSENSITIVE を適用します。
	 * Pattern.compile( str,Pattern.CASE_INSENSITIVE ) ;
	 * pattern.matcher( pathname.getName() ).find() == true と同じ結果が得られます。
	 * 引数が null の場合は、追加しません。
	 *
	 * @og.rev 7.2.6.0 (2020/06/30) ディレクトリ処理を統一します。
	 *
	 * @param    str ファイル名文字列(正規表現)
	 * @see java.util.regex.Pattern#compile(String,int)
	 * @see java.util.regex.Matcher#find()
	 */
	public void setMmatchDir( final String str ) {
		matchDir = nval( getRequestParameter( str ),matchDir );
	}

	/**
	 * 【TAG】フォルダ名が、指定された正規表現と一致しない場合、スルー(選択)されます(初期値:null)。
	 * @og.tag
	 *
	 * ignoreCase を使用する場合、Pattern.CASE_INSENSITIVE を適用します。
	 * Pattern.compile( str,Pattern.CASE_INSENSITIVE ) ;
	 * pattern.matcher( pathname.getName() ).find() == true と同じ結果が得られます。
	 * 引数が null の場合は、追加しません。
	 *
	 * @og.rev 7.2.6.0 (2020/06/30) ディレクトリ処理を統一します。
	 *
	 * @param    str ファイル名文字列(正規表現)
	 * @see java.util.regex.Pattern#compile(String,int)
	 * @see java.util.regex.Matcher#find()
	 */
	public void setUnMatchDir( final String str ) {
		unMatchDir = nval( getRequestParameter( str ),unMatchDir );
	}

	/**
	 * 【TAG】指定のタイムスタンプ以後に変更されている場合、スルー(選択)されます(初期値:null)。
	 * @og.tag
	 * ディレクトリは、ここの判定では無視します。(必ず true を返します)
	 * 日付けの指定に、YYYYMMDD 形式の ８文字数字文字列以外に、
	 * TODAY や YESTERDAY なども使用できます。
	 * TODAY は、実行日の 00:00:00 を基準時刻とし、YESTERDAY は、その前日になります。
	 * 引数が null の場合は、追加しません。
	 *
	 *     YYYYMMDD   YYYYMMDD形式の指定日の 00:00:00 を基準時刻
	 *     TODAY      実行日の 00:00:00 を基準時刻
	 *     YESTERDAY  実行日前日の 00:00:00 を基準時刻
	 *     LAST_WEEK  実行日の先週(7日前) 00:00:00 を基準時刻
	 *     MONTH      実行月の 1日 00:00:00 を基準時刻
	 *     LAST_MONTH 実行前月の 同日 00:00:00 を基準時刻
	 *     LAST_YEAR  実行前年の 同月同日 00:00:00 を基準時刻
	 *
	 * @param    modify YYYYMMDD形式の指定日
	 */
	public void setLastModified( final String modify ) {
		lastModified = nval( getRequestParameter( modify ),lastModified );
	}

	/**
	 * 【TAG】指定のタイムスタンプ以前に変更されている場合、スルー(選択)されます(初期値:null)。
	 * @og.tag
	 * ディレクトリは、ここの判定では無視します。(必ず true を返します)
	 * 日付けの指定のフォーマットは、lastModifiedと同じ記号が使用できます。
	 *     LAST_YEAR  実行前年の 同月同日 00:00:00 を基準時刻
	 *
	 * @param    modify YYYYMMDD形式の指定日
	 */
	public void setUnLastModified( final String modify ) {
		unLastModified = nval( getRequestParameter( modify ),unLastModified );
	}

	/**
	 * 【TAG】指定の大きさより大きいファイルの場合、スルー(選択)されます(初期値:null)。
	 *
	 * @og.tag
	 * ファイルの大きさ(バイト単位)は同値を含みます。
	 * 未指定の場合は、スルー(選択)されます。
	 *
	 * 指定はバイト単位ですが、**KB , **MB , **GB などの単位を付ける事も可能です。
	 * 現時点では、K , KB , M , MB , G , GB のみ指定可能です。
	 *
	 * @param	len 	ファイルの大きさ(バイト単位)
	 */
	public void setIsLarger( final String len ) {
		isLarger = nval( getRequestParameter( len ),isLarger );
	}

	/**
	 * 【TAG】指定の大きさより小さいファイルの場合、スルー(選択)されます(初期値:null)。
	 *
	 * @og.tag
	 * ファイルの大きさ(バイト単位)は同値を含みません。
	 * 未指定の場合は、スルー(選択)されます。
	 *
	 * 指定はバイト単位ですが、**KB , **MB , **GB などの単位を付ける事も可能です。
	 * 現時点では、K , KB , M , MB , G , GB のみ指定可能です。
	 *
	 * @param	len	ファイルの大きさ(バイト単位)
	 */
	public void setIsSmaller( final String len ) {
		isSmaller = nval( getRequestParameter( len ),isSmaller );
	}

	/**
	 * 【TAG】対象に隠しファイル(hidden)を、スルー(選択)する場合、true を指定します(初期値:null)。
	 * @og.tag
	 * trueを指定すると、隠しファイル(hidden)のみ選択します。
	 * falseを指定すると、隠しファイル(hidden)以外を選択します。
	 * 隠しファイルかどうかの定義はOSに従います。
	 * 初期値は、両方とも選択する為、null (trueでもfalseでもない)です。
	 *
	 * @og.rev 5.7.5.0 (2014/04/04) 新規作成
	 *
	 * @param    flag hidden対象 [true:する/false:しない]
	 */
	public void setIsHidden( final String flag ) {
		isHidden = nval( getRequestParameter( flag ), isHidden );
	}

	/**
	 * 【廃止】フィルタによる対象の絞込みをディレクトリにも適用するかどうか[false:File/true:File+Dir/only:Dir]を指定します(初期値:false:しない)。
	 *
	 * @og.tag
	 * このﾌﾗｸﾞは、設定したすべての属性に対して一律に作用します。
	 * 個々に適用するには、それぞれの属性のxxxDir などを使用してください。
	 * 初期値は、false(ディレクトリはフィルタによる絞込みをしない)です。
	 *
	 * @og.rev 5.3.9.0 (2011/09/01) 新規作成
	 * @og.rev 6.4.0.2 (2015/12/11) useDIR 属性に、onlyパラメータを追加します[false:File/true:File+Dir/only:Dir]。
	 * @og.rev 7.2.6.0 (2020/06/30) ディレクトリ処理を統一します。
	 *
	 * @param	flag	ディレクトリ適用 [true:する/false:しない]
	 */
	public void setUseDir( final String flag ) {
//		useDir = nval( getRequestParameter( flag ),useDir );
	}

	/**
	 * 【TAG】大文字/小文字を区別しないかどうか[true/false]を指定します(初期値:false(区別する))。
	 *
	 * @og.tag
	 * startsWith , contains , endsWith , equalsIgnoreCase での比較時に、比較対象の
	 * 大文字/小文字を区別しないかどうかを指定します。
	 * 区別しない ("true") 場合、aaa と AAA は、一致したとみなされます。
	 * 初期値は、区別する ("false") です。
	 *
	 * @og.rev 6.3.1.1 (2015/07/10) 大文字/小文字の区別なし(ignoreCase=true)対応
	 *
	 * @param   flag 大文字/小文字を区別しないかどうか [true:しない/それ以外:する]
	 */
	public void setIgnoreCase( final String flag ) {
		ignoreCase = nval( getRequestParameter( flag ),ignoreCase );
	}

	/**
	 * 【TAG】判定結果を反転させるかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 通常の判定結果において、結果の正反対の処理を行います。
	 * 初期値は、通常 (false)です。
	 *
	 * @og.rev 6.8.0.0 (2017/06/02) 新規追加
	 *
	 * @param   flag 判定結果反転 [true:反転する/それ以外:通常]
	 */
	public void setNotEquals( final String flag ) {
		notEquals = nval( getRequestParameter( flag ),notEquals );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return String型。このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "startsWith"		,startsWith		)
				.println( "endsWith"		,endsWith		)
				.println( "instr"			,instr			)
				.println( "equals"			,fileEquals		)
				.println( "matches"			,matches		)
				.println( "unMatches"		,unMatches		)
				.println( "lastModified"	,lastModified	)
				.println( "isLarger"		,isLarger		)
				.println( "isSmaller"		,isSmaller		)
//				.println( "useDir"			,useDir			)
				.println( "ignoreCase"		,ignoreCase		)		// 6.3.1.1 (2015/07/10)
				.println( "Other..."		,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
