/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;

import org.opengion.hayabusa.resource.ResourceFactory;
import org.opengion.hayabusa.resource.ResourceManager;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;

/**
 * TableFilter_LABEL は、TableFilter インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。
 *
 * ここでは、キーのCLM,LBL と、LANG より、ラベルリソースに変換した値を設定します。
 * ラベルの設定は、設定先の値が NULL/空文字か、カラム名と同じ(無変換セット)の場合のみセットされます。
 *
 * パラメータは、tableFilterタグの keys, vals にそれぞれ記述するか、BODY 部にCSS形式で記述します。
 * 【パラメータ】
 *  {
 *       SYSTEM_ID  : ラベルリソースの作成システムIDを指定します。無指定時は、ログイン時のリソースになります。
 *       LANG       : ラベルリソースの言語を指定します。無指定時は、日本語になります。
 *       CLM        : ラベルリソースのキーとなる値が設定されているカラム名を指定します。
 *       LBL        : CLMで指定されたカラム名から値を取り出し、ラベルリソースに変換した結果をセットします。
 *       LLBL       : 同上（ラベル長）					7.2.9.0 (2020/10/12)
 *       SLBL       : 同上（ラベル短）					7.2.9.0 (2020/10/12)
 *       DISC       : 同上（概要）						7.2.9.0 (2020/10/12)
 *       LRAW       : 同上（長生ラベル）				7.2.9.0 (2020/10/12)
 *       SRAW       : 同上（短生ラベル）				7.2.9.0 (2020/10/12)
 *       MSG        : 同上（引数付きラベル）			7.2.9.1 (2020/10/23)
 *  }
 *
 * 5.7.8.0 (2014/07/04) 解除
 * CLM と LBL に同じカラムを指定すると、このフィルターを通過するたびに、変換後のラベルが
 * 再びキーとして変換しようと試みることになります。不測の事態に陥る可能性があるため、
 * その様な設定は出来なくなっています。
 *
 * 5.7.8.0 (2014/07/04) CLM と LBL に同じカラムを指定する事ができます。
 *
 * また、CLM または LBL で指定したカラムが DBTableModel に存在しない場合は、処理そのものを
 * 無視します。その場合は、警告も出力されませんので、ご注意ください。
 *
 * 7.2.9.0 (2020/10/12) LLBL,SLBL,DISC,RLBL 属性追加。
 *
 * 7.2.9.1 (2020/10/23) MSG 属性追加。
 *   引数付きなので、VALS 指定時に、セットするカラム名に引数として渡したいパラメータをスペースで繋げます。
 *   パラメータは、@カラム名 か、そのままの値で指定します。
 *
 * @og.formSample
 * ●形式：
 *      ① &lt;og:tableFilter classId="LABEL" keys="CLM,LBL" vals="COLUMN,LABEL_CLM" /&gt;
 *
 *      ② &lt;og:tableFilter classId="LABEL" &gt;
 *               {
 *                      SYSTEM_ID  : GF ;
 *                      LANG       : ja ;
 *                      CLM        : COLUMN ;
 *                      LBL        : LABEL_CLM ;
 *               }
 *         &lt;/og:tableFilter&gt;
 *
 * @og.rev 5.6.6.0 (2013/07/05) keys の整合性チェックを追加
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public class TableFilter_LABEL extends AbstractTableFilter {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.2.9.1 (2020/10/23)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.1.1 (2016/01/16) keysMap を、サブクラスから設定させるように変更。
	 */
	public TableFilter_LABEL() {
		super();
		initSet( "SYSTEM_ID" 	, "システムIDを指定します(初期値:ログイン時のリソース)"		);
		initSet( "LANG"			, "言語を指定(初期値:日本語)"								);
		initSet( "CLM"			, "キーとなる値が設定されているカラム名を指定"				);
		initSet( "LBL"			, "ラベルリソースに変換した結果をセットするカラム名を指定"	);
		initSet( "LLBL"			, "同上（ラベル長）"										);		// 7.2.9.0 (2020/10/12)
		initSet( "SLBL"			, "同上（ラベル短）"										);		// 7.2.9.0 (2020/10/12)
		initSet( "DISC"			, "同上（概要）	"											);		// 7.2.9.0 (2020/10/12)
		initSet( "LRAW"			, "同上（長生ラベル）"										);		// 7.2.9.0 (2020/10/12)
		initSet( "SRAW"			, "同上（短生ラベル）"										);		// 7.2.9.0 (2020/10/12)
		initSet( "MSG"			, "同上（引数付きラベル）"									);		// 7.2.9.1 (2020/10/23)
	}

	/**
	 * DBTableModel処理を実行します。
	 *
	 * @og.rev 5.5.2.2 (2012/05/09) SYSTEM_ID を受け取るように変更
	 * @og.rev 5.5.2.6 (2012/05/25) protected変数を、private化したため、getterメソッドで取得するように変更
	 * @og.rev 5.7.8.0 (2014/07/04) CLM と LBL に同じカラムを指定できるように変更
	 * @og.rev 6.4.0.5 (2016/01/09) LBLが、nullか、または、CLMと同じ場合は、ﾘｿｰｽから、そうでない場合は、LBLの値を使用します。
	 * @og.rev 6.5.0.1 (2016/10/21) ErrorMessage をまとめるのと、直接 Throwable を渡します。
	 * @og.rev 7.2.9.0 (2020/10/12) LLBL,SLBL,DISC,RLBL 属性追加
	 * @og.rev 7.2.9.1 (2020/10/23) MSG 属性追加
	 *
	 * @return 処理結果のDBTableModel
	 */
	public DBTableModel execute() {
		final DBTableModel table = getDBTableModel();			// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加

		if( table == null ) { return table; }					// 6.4.0.5 (2016/01/09)

		final String systemId	= getValue( "SYSTEM_ID" );		// 5.5.2.2 (2012/05/09)
		final String lang		= getValue( "LANG" );
		final ResourceManager resource = ResourceFactory.newInstance( systemId,lang,false );		// 5.5.2.2 (2012/05/09)

		final String clm  = getValue( "CLM" );
		final String lbl  = getValue( "LBL" );
		final String llbl = getValue( "LLBL" );						// 7.2.9.0 (2020/10/12)
		final String slbl = getValue( "SLBL" );						// 7.2.9.0 (2020/10/12)
		final String disc = getValue( "DISC" );						// 7.2.9.0 (2020/10/12)
		final String lraw = getValue( "LRAW" );						// 7.2.9.0 (2020/10/12)
		final String sraw = getValue( "SRAW" );						// 7.2.9.0 (2020/10/12)

		final int clmNo   = table.getColumnNo( clm,false );			// 存在しない場合は、-1 を返す。
		final int lblNo   = table.getColumnNo( lbl,false );
		final int llblNo  = table.getColumnNo( llbl,false );		// 7.2.9.0 (2020/10/12)
		final int slblNo  = table.getColumnNo( slbl,false );		// 7.2.9.0 (2020/10/12)
		final int discNo  = table.getColumnNo( disc,false );		// 7.2.9.0 (2020/10/12)
		final int lrawNo  = table.getColumnNo( lraw,false );		// 7.2.9.0 (2020/10/12)
		final int srawNo  = table.getColumnNo( sraw,false );		// 7.2.9.0 (2020/10/12)

		// 7.2.9.1 (2020/10/23) MSGの値は、セットするカラム名の後ろに、パラメータをスペースで繋げています。
		final String msg_prm = getValue( "MSG" );
		final int   msgNo ;
		final int[] prmNo ;
		final String[] prmVal ;

		if( msg_prm != null ) {
			final String[] msgs = msg_prm.split( "[\\s]+" );		// 連続スペース、タブを区切り文字にする。
			msgNo = table.getColumnNo( msgs[0],false );
			prmNo  = new int[msgs.length-1];
			prmVal = new String[msgs.length-1];
			for( int i=1; i<msgs.length; i++ ) {
				prmVal[i-1] = msgs[i];								// 元の引数をそのまま使う可能性があるので、セットしておく。

				if( '@' == msgs[i].charAt(0) ) {					// ｶﾗﾑ指定の意味
					prmNo[i-1] = table.getColumnNo( msgs[i].substring(1),false );
				}
				else {
					prmNo[i-1] = -1;
				}
			}
		}
		else {
			msgNo  = -1;
			prmNo  = null;
			prmVal = null;
		}

		// 5.7.8.0 (2014/07/04) CLM と LBL に同じカラムを指定できるように変更
//		if( clmNo >= 0 && lblNo >= 0 ) {
//		if( clmNo >= 0 && ( lblNo >= 0 || llblNo >= 0 || slblNo >= 0 || discNo >= 0 || lrawNo >= 0 || srawNo >= 0 ) ) {					// 7.2.9.0 (2020/10/12)
		if( clmNo >= 0 && ( lblNo >= 0 || llblNo >= 0 || slblNo >= 0 || discNo >= 0 || lrawNo >= 0 || srawNo >= 0 || msgNo >= 0 ) ) {	// 7.2.9.1 (2020/10/23)
			String[] data  = null;
			final int rowCnt = table.getRowCount();
			for( int row=0; row<rowCnt; row++ ) {
				String clmVal = null;
				try {
					data   = table.getValues( row );
					clmVal = data[clmNo];		// 変換する元のカラム名

					if( lblNo >= 0 ) {
						final String lblVal = data[lblNo];		// 変換する元のラベル		6.4.0.5 (2016/01/09)
						// 6.4.0.5 (2016/01/09) LBLが、nullか、または、CLMと同じ場合は、ﾘｿｰｽから、そうでない場合は、LBLの値を使用します。
						// つまり、上記の条件が成立した場合は、LBLを書き換えないということです。
						if( StringUtil.isNull( lblVal ) || lblVal.equals( clmVal ) ) {
							data[lblNo] = resource.getLabel( clmVal ) ;
						}
					}

					// 7.2.9.0 (2020/10/12) LLBL,SLBL,DISC,RLBL 属性追加
					if( llblNo >= 0 ) {
						final String llblVal = data[llblNo];		// 変換する元のラベル		6.4.0.5 (2016/01/09)
						if( StringUtil.isNull( llblVal ) || llblVal.equals( clmVal ) ) {
							data[llblNo] = resource.getLongLabel( clmVal ) ;
						}
					}

					if( slblNo >= 0 ) {
						final String slblVal = data[slblNo];		// 変換する元のラベル		6.4.0.5 (2016/01/09)
						if( StringUtil.isNull( slblVal ) || slblVal.equals( clmVal ) ) {
							data[slblNo] = resource.getShortLabel( clmVal ) ;
						}
					}

					// 7.2.9.0 (2020/10/12) SLBL,DISC,RLBL 属性追加
					if( discNo >= 0 ) {
						final String discVal = data[discNo];		// 変換する元のラベル		6.4.0.5 (2016/01/09)
						if( StringUtil.isNull( discVal ) || discVal.equals( clmVal ) ) {
							data[discNo] = resource.getDescription( clmVal ) ;
						}
					}

					// 7.2.9.0 (2020/10/12) SLBL,DISC,RLBL 属性追加
					if( lrawNo >= 0 ) {
						final String lrawVal = data[lrawNo];		// 変換する元のラベル		6.4.0.5 (2016/01/09)
						if( StringUtil.isNull( lrawVal ) || lrawVal.equals( clmVal ) ) {
							final String val = resource.getRawLongLabel( clmVal ) ;
							data[lrawNo] = val == null || val.equals( clmVal ) ? "" : val ;		// null時はkeyが返ってくるので 空文字 を返す。
						}
					}

					// 7.2.9.0 (2020/10/12) SLBL,DISC,RLBL 属性追加
					if( srawNo >= 0 ) {
						final String srawVal = data[srawNo];		// 変換する元のラベル		6.4.0.5 (2016/01/09)
						if( StringUtil.isNull( srawVal ) || srawVal.equals( clmVal ) ) {
							final String val = resource.getRawShortLabel( clmVal,false ) ;		// null時は 空文字 を返す。
							data[srawNo] = val == null || val.equals( clmVal ) ? "" : val;		// null時は 空文字 を返す。

//							data[srawNo] = resource.getRawShortLabel( clmVal,false ) ;			// null時は 空文字 を返す。
						}
					}

					// 7.2.9.1 (2020/10/23) MSG 属性追加
					if( msgNo >= 0 ) {
						final String msgVal = data[msgNo];
						if( StringUtil.isNull( msgVal ) || msgVal.equals( msgVal ) ) {
							final String[] prmVals = new String[prmNo.length];
							for( int i=0; i<prmNo.length; i++ ) {
								if( prmNo[i] >= 0 ) {
									prmVals[i] = StringUtil.nval( data[prmNo[i]] , "" );
								}
								else {
									prmVals[i] = prmVal[i];
								}
							}

							data[msgNo] = resource.getLabel( clmVal,prmVals ) ;
						}
					}
				}
				catch( final RuntimeException ex ) {
					// 6.5.0.1 (2016/10/21) ErrorMessage をまとめるのと、直接 Throwable を渡します。
					makeErrorMessage( "TableFilter_LABEL Error",ErrorMessage.NG )
						.addMessage( row+1,ErrorMessage.NG,"LABEL"
							, "CLM=[" + clm + "],VAL=[" + clmVal + "]"
							, StringUtil.array2csv( data )
						)
						.addMessage( ex );
				}
			}
		}

		return table;
	}
}
