package org.opengion.fukurou.model;

import java.io.File;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;

/**
 * CloudFileOperation用のファイル情報の格納クラス
 * 
 * listFilesで取得した、ディレクトリとファイル一覧情報を格納します。
 * 
 * パフォーマンスや分かりやすさを考慮してCloudFileOperationからは分離しています
 * 
 * @og.group ファイル操作
 * 
 * @og.rev 5.10.8.0 (2019/02/01) 新規作成
 * @og.rev 5.10.9.0 (2019/03/01) 変更対応
 * @author oota
 * @since JDK7.0
 */
public class FileOperationInfo extends CloudFileOperation
		{
	/** クラス変数 */
	private String plugin;
	private long size;
	private long lastModified;
	private boolean isFile;
	private boolean isDirectory;
	private FileOperation file;
	
	/**
	 * コンストラクタ
	 * 
	 * 生成時の初期処理。
	 * 
	 * @param plugin プラグイン名
	 * @param bucket バケット名
	 * @param path ファイルパス
	 */
	public FileOperationInfo(String plugin, String bucket, String path) {
		super(bucket, path);
		this.plugin = plugin;
		size = 0;
		lastModified = 0;
		isFile = false;
		isDirectory = false;
		file = null;
	}

	/**
	 * FileOperationクラスの生成
	 * 
	 * 呼び出し時に、FileOperationインスタンスが未生成の場合は、
	 * 生成を行います。
	 */
	private void setFileOperation() {
		if(file == null) {
			file = FileOperationFactory.newStorageOperation( plugin, conBucket, conPath );
		}
	}
	
	/** Method */
	/**
	 * 書き込み処理
	 * 
	 * InputStreamのデータを書き込みます。
	 * 
	 * @param is 書き込みデータのInputStream
	 * @throws IOException ファイル関連エラー情報
	 */
	@Override
	public void write(InputStream is) throws IOException {
		setFileOperation();
		file.write(is);
	}

	/**
	 * 読み込み処理
	 * 
	 * データを読み込み、InputStreamとして、返します。
	 * 
	 * @return 読み込みデータのInputStream
	 * @throws FileNotFoundException ファイル非存在エラー情報
	 */
	@Override
	public InputStream read() throws FileNotFoundException {
		setFileOperation();
		return file.read();
	}

	/**
	 * 削除処理
	 * 
	 * ファイルを削除します。
	 * 
	 * @return 成否フラグ
	 */
	@Override
	public boolean delete() {
		setFileOperation();
		return file.delete();
	}
	
	/**
	 * コピー処理
	 * 
	 * ファイルを指定先に、コピーします。
	 * 
	 * @param afPath コピー先
	 * @return 成否フラグ
	 */
	@Override
	public boolean copy(String afPath) {
		setFileOperation();
		return file.copy(afPath);
	}

	/**
	 * 一覧取得
	 * 
	 * １つ下の、ディレクトリ・ファイル一覧を取得します。
	 * 
	 * @param filter フィルタ情報
	 * @return ファイル一覧
	 */
	@Override
	public File[] listFiles(FileFilter filter) {
		setFileOperation();
		return file.listFiles(filter);
	}

	/**
	 * ファイルサイズ取得
	 * 
	 * ファイルサイズを取得します。
	 * 
	 * @return ファイルサイズ
	 */
	@Override
	public long length() {
		return size;
	}

	/**
	 * ファイルサイズ設定
	 * 
	 * ファイルサイズを設定します。
	 * 
	 * @param size ファイルサイズ
	 */
	public void setSize(long size) {
		this.size = size;
	}

	/**
	 * 最終更新時刻の取得
	 * 
	 * 最終更新時刻を取得します。
	 * 
	 * @return 最終更新時刻
	 */
	@Override
	public long lastModified() {
		return lastModified;
	}

	/**
	 * 最終更新時刻の設定
	 * 
	 * 最終更新時刻を設定します。
	 * 
	 * @param lastModified 最終更新時刻
	 */
	public void setLastModifiedValue(long lastModified) {
		this.lastModified = lastModified;
	}

	/**
	 * ファイル判定取得
	 * 
	 * ファイルであるかの判定を返します。
	 * 
	 * @return ファイル判定
	 */
	@Override
	public boolean isFile() {
		return isFile;
	}

	/**
	 * ファイル判定設定
	 * 
	 * ファイルであるかの判定を設定します。
	 * 
	 * @param isFile ファイル判定
	 */
	public void setFile(boolean isFile) {
		this.isFile = isFile;
	}

	/**
	 * ディレクトリ判定取得
	 * 
	 * ディレクトリであるかの判定を返します。
	 * 
	 * @return ディレクトリ判定
	 */
	@Override
	public boolean isDirectory() {
		return isDirectory;
	}

	/**
	 * ディレクトリ判定設定
	 * 
	 * ディレクトリであるかの判定を設定します。
	 * 
	 * @param isDirectory ディレクトリ判定
	 */
	public void setDirectory(boolean isDirectory) {
		this.isDirectory = isDirectory;
	}
	
	
	/**
	 * 親情報の取得
	 * 
	 * 親情報を返します。
	 * 
	 * @return 親情報
	 */
	@Override
	public File getParentFile() {
		 return  FileOperationFactory.newStorageOperation( file , this.getParent() );
	}
	
//	// テスト用メソッドです
//	public static void main(String[] args) {
//		System.out.println("start");
//		
//		FileOperation file = new FileOperationInfo("aws", "otest20190205", "sample/test.txt");
//		
//		File parent = file.getParentFile();
//		System.out.println(parent.getPath());
//		System.out.println(parent.isDirectory());
//		System.out.println(parent.isFile());
//		
//		System.out.println("end");
//	}
}
