package org.opengion.hayabusa.io;

import java.io.File;

import org.opengion.fukurou.model.FileOperation;
import org.opengion.fukurou.model.FileOperationFactory;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;

/**
 * クラウドを含むファイル操作クラスの生成
 * 
 * 直接fukurouをCallしてもよいのですが、hayabusaからの呼び出しではシステムリソースを参照する必要があるため
 * ラッパー的にこのクラスを経由してCallする事でシステムリソースが使われるようにしておきます。
 * （タグ以外からも呼び出されるため、commonTagSupportではなく専用クラスをioパッケージに作成しています）
 * 
 * ローカルのファイルを扱いたい場合は、pluginにDEFAULTを指定してください。
 * 
 * @og.rev 5.10.8.0 (2019/02/01) 新規作成
 * @og.group
 *
 * @version 5.0
 * @author Takahashi Masakazu
 * @since JDK7.0
 */
public final class HybsFileOperationFactory {
	private static String defPlugin=HybsSystem.sys("CLOUD_TARGET");
	private static String defBucket=HybsSystem.sys("CLOUD_BUCKET");
	
	/**
	 * コンストラクタはprivate化しておきます。
	 */
	private HybsFileOperationFactory(){
		// コンストラクタ
	}

	/**
	 * fukurouのFileOperationFactoryを呼び出してFOInterfaceを取得します。
	 * plugin,buketを指定しない場合はシステムリソースを利用します。
	 * 
	 * @param plugin プラグイン名
	 * @param bucket バケット名
	 * @param path ファイルパス
	 * @return FileOperationインスタンス
	 */
	public static FileOperation create(final String plugin, final String bucket, final String path) {
		return FileOperationFactory.newStorageOperation( StringUtil.nval(plugin, defPlugin), StringUtil.nval(bucket, defBucket), path );
	}
	
	/**
	 * ディレクトリとファイル名を指定用です。
	 * 
	 * @param plugin プラグイン名
	 * @param bucket バケット名
	 * @param dir ディレクトリパス
	 * @param file ファイル名
	 * @return FileOperationインスタンス
	 */
	public static FileOperation create(final String plugin, final String bucket, final String dir, final String file) {
		final StringBuilder sb = new StringBuilder(HybsSystem.BUFFER_SMALL);
		sb.append(dir).append(HybsSystem.FS).append(file);
		return create(plugin, bucket, sb.toString());
	}
	
	/**
	 * FileOperation(ディレクトリ)とファイル名を指定用です。
	 * 
	 * @param plugin プラグイン名
	 * @param bucket バケット名
	 * @param dir ファイル(ディレクトリパス取得)
	 * @param file ファイル名
	 * @return FileOperationインスタンス
	 */
	public static FileOperation create(final String plugin, final String bucket, final File dir, final String file) {
		final StringBuilder sb = new StringBuilder(HybsSystem.BUFFER_SMALL);
		sb.append(dir.getPath()).append(HybsSystem.FS).append(file);
		return create(plugin, bucket, sb.toString());
	}
	
}
