/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.daemon;

import java.util.Date;

import org.opengion.fukurou.util.HybsTimerTask;
import org.opengion.fukurou.util.LogWriter;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.plugin.cloud.MailManager_DB_SendGridAPI;


/**
 * メールパラメータテーブルを監視して、SengGridAPIを利用したメール送信プログラムを呼び出します。
 * このクラスは、HybsTimerTask を継承した タイマータスククラスです。
 * startDaemon() がタイマータスクによって、呼び出されます。
 * 
 * 通常のメールモジュールはSMTPを利用した送信で、hayabusa.mail内のクラスをCallしていますが、
 * こちらはplugin.cloud内のsendGridのAPIをCallしてメールを送信します。
 * そのため、sendGridを利用出来る状態(jarファイルの配備等)となっている必要があります。
 * 
 * システムリソースにMAIL_SENDGRID_APIKEYが登録されていない場合は動作しません。
 *
 * @og.group メールモジュール
 *
 * @og.rev 5.9.26.0 (2017/11/02) 新規作成
 * @author   T.OTA
 * @since    JDK1.7
 */
public class MailDaemon_SendGridAPI extends HybsTimerTask {

	private int loopCnt = 0;

	private static final int LOOP_COUNTER = 24; // カウンタを24回に設定

	/**
	 * このタイマータスクによって初期化されるアクションです。
	 * パラメータを使用した初期化を行います。
	 *
	 */
	@Override
	public void initDaemon() {
		// 何もありません。(PMD エラー回避)
	}

	/**
	 * タイマータスクのデーモン処理の開始ポイントです。
	 *
	 */
	@Override
	protected void startDaemon() {
		// SendGridのAPIキー
		String sendGridApiKey = HybsSystem.sys("MAIL_SENDGRID_APIKEY");
		if(sendGridApiKey == null || sendGridApiKey.length() == 0){
			// SendGridのAPIキーが設定されていない場合は終了
			return;
		}

		if( loopCnt % LOOP_COUNTER == 0 ) {
			loopCnt = 1;
			System.out.println( toString() + " " + new Date()  + " " );
		}
		else {
			String systemId = null;
			try {
				systemId = getValue( "SYSTEM_ID" );
				// SengGridAPI利用のメール送信機能呼び出し
				MailManager_DB_SendGridAPI manager = new MailManager_DB_SendGridAPI();
				manager.sendDBMail( systemId );
			}
			catch( RuntimeException rex ) {
				String errMsg = "メール送信失敗しました。"
							+ " SYSTEM_ID=" + systemId 
							+ " " + rex.getMessage();			// 5.10.7.0 (2019/01/11) 追加
				LogWriter.log( errMsg );
			}
			loopCnt++ ;
		}
	}
}
