/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.resource.UserInfo;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.Query;
import org.opengion.hayabusa.db.QueryFactory;
import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.util.ArraySet;						// 6.4.3.4 (2016/03/11)

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.Set;											// 6.4.3.4 (2016/03/11)
import java.util.Locale ;

/**
 * ユーザー情報を管理しているタグです。
 *
 * 設定した値は、{&#064;USER.XXXX} 形式で 取り出すことができます。
 * また、command 属性で 直接画面に値を書き出すことも可能です。
 *
 * 以下の値は UserInfo オブジェクトの項目から取得します。
 * ・USER.JNAME     ユーザー日本語名称
 * ・USER.ID        ユーザーＩＤ
 * ・USER.IDNO      USER.ID が５Byte以上の時のみ先頭１文字を除いたユーザーＩＤ
 * ・USER.INFO      ユーザー情報(ユーザーID：日本語名称)
 * ・USER.LANG      言語
 * ・USER.ROLES     ロール
 * ・USER.IPADDRESS IPアドレス
 * ・USER.LOGINTIME ログイン時刻
 *
 * 以下の値はあらかじめ、動的に作成されます。
 * ・USER.YMD       ８byte の今日のシステム日付
 * ・USER.YMDH    １４byte の今日のシステム日時
 *
 * それ以外は、外部より設定された値です。
 *
 * ※ このタグは、Transaction タグの対象です。
 *
 * @og.formSample
 * ●形式：&lt;og:userInfo command="[…]" key="[…]" value="[…]" /&gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:userInfo
 *       command            【TAG】コマンド (SET,GET,NVLGET,REMOVE,SQL)をセットします(初期値:SET)
 *       key                【TAG】UserInfo に登録するキーをセットします(予約語:JNAME,ID,PASSWD,IDNO,INFO,LANG,ROLE,IPADDRESS,LOGINTIME)
 *       value              【TAG】UserInfo に登録する値をセットします
 *       dbid               【TAG】(通常は使いません)Queryオブジェクトを作成する時のDB接続IDを指定します
 *       save               【TAG】UserInfo に登録した値を永続化するかを指定します(初期値:false)
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:userInfo&gt;
 *
 * ●使用例
 *      &lt;og:userInfo
 *         command       = &quot;コマンド (SET,GET,NVLGET,REMOVE,SQL)をセットします(初期値:SET)。&quot;
 *         key           = &quot;UserInfo に登録するキーをセットします(予約語:JNAME,ID,PASSWD,IDNO,INFO,LANG,ROLE,IPADDRESS,LOGINTIME)。&quot;
 *         value         = &quot;UserInfo に登録する値をセットします。&quot;
 *         dbid          = &quot;(通常は使いません)Queryオブジェクトを作成する時のDB接続IDを指定します。&quot;
 *      &gt;
 *
 * @og.group その他部品
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class UserInfoTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.3.4 (2016/03/11)" ;
	private static final long serialVersionUID = 643420160311L ;

	/** command 引数に渡す事の出来る コマンド  セット {@value} */
	public static final String CMD_SET   = "SET" ;
	/** command 引数に渡す事の出来る コマンド  ゲット {@value} */
	public static final String CMD_GET   = "GET" ;
	/** command 引数に渡す事の出来る コマンド  ゲット {@value} */
	public static final String CMD_REMOVE   = "REMOVE" ;
	/** command 引数に渡す事の出来る コマンド  ゲット {@value} */
	public static final String CMD_SQL   = "SQL" ;
	// 3.5.5.3 (2004/04/09) 値が null の場合は、""(ゼロ文字列)を返すNVLGETを採用
	/** command 引数に渡す事の出来る コマンド  NVLゲット {@value} */
	public static final String CMD_NVLGET   = "NVLGET" ;

	// 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	private static final Set<String> COMMAND_SET = new ArraySet<>( CMD_SET , CMD_GET , CMD_NVLGET , CMD_REMOVE , CMD_SQL );

	// 3.5.6.0 (2004/06/18) すべてを protected から private に変更します。
	private String			command		= CMD_SET;
	private String			key			;
	private String			value		;
	private transient DBTableModel table;
	// 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
	private String			dbid		;
	private String			sql			;
	// 4.3.4.0 (2008/12/01) save属性追加
	private boolean			save		;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public UserInfoTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doStartTag() {
		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		return useTag() && CMD_SQL.equals( command )
					? EVAL_BODY_BUFFERED				// Body を評価する
					: SKIP_BODY ;						// Body を評価しない
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) ボディの内容を取得する処理を、CommonTagSupport で行う。
	 * @og.rev 3.6.0.8 (2004/11/19) エラー発生時に確実にリリースされるように try finally 追加
	 * @og.rev 3.8.6.3 (2006/11/30) SQL 文の前後のスペースを取り除きます。
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0.0 (2005/01/31) lang ⇒ ResourceManager へ変更
	 * @og.rev 5.1.9.0 (2010/08/01) TransactionTag 対応。上位に TransactionTag があれば、そこからConnection をもらう。
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更
	 * @og.rev 5.3.8.0 (2011/08/01) Transaction発生箇所でclose()
	 * @og.rev 6.3.1.1 (2015/07/10) BodyString,BodyRawStringは、CommonTagSupport で、trim() します。
	 * @og.rev 6.3.6.1 (2015/08/28) Transaction でAutoCloseableを使用したtry-with-resources構築に対応。
	 * @og.rev 6.3.6.1 (2015/08/28) QueryFactory.close( Query ) 廃止。Queryはキャッシュしません。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		sql = getBodyString();

		// 6.3.6.1 (2015/08/28) Transaction でAutoCloseableを使用したtry-with-resources構築に対応。
		try( Transaction tran = getTransaction() ) {
			final Query query = QueryFactory.newInstance();		// 4.0.0 (2005/01/31)
			query.setConnection( tran.getConnection( dbid ) );			// 6.3.6.1 (2015/08/28)
			query.setResourceManager( getResource() );	// 4.0.0 (2005/01/31)
			query.setStatement( sql );
			query.execute();

			table = query.getDBTableModel();
			tran.commit();				// 6.3.6.1 (2015/08/28)
		}
		return SKIP_BODY ;
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.5.3 (2004/04/09) 値が null の場合は、""(ゼロ文字列)を返すフラグを採用
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
	 * @og.rev 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		// 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
		if( useTag() && check( command, COMMAND_SET ) ) {
			commandExec( command );

			if( CMD_GET.equals( command ) ) {
				jspPrint( value );
			}
			else if( CMD_NVLGET.equals( command ) ) {	// 3.5.5.3 (2004/04/09)
				jspPrint( nval(value,"") );
			}
		}

		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
	 * @og.rev 4.3.4.0 (2008/12/01) save属性対応
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		command		= CMD_SET;
		key			= null;
		value		= null;
		table		= null;
		dbid		= null;
		sql			= null;
		save		= false;
	}

	/**
	 * コマンドを実行します。
	 *
	 * コマンドは,HTMLから(get/post)指定されますので,setCommand()メソッドで
	 * 登録します。
	 * コマンドを登録すると同時に,実行も行ないます。
	 *
	 * @og.rev 3.5.5.3 (2004/04/09) 値が null の場合は、""(ゼロ文字列)を返すフラグを採用
	 *
	 * @param   command コマンド (public static final 宣言されている文字列)
	 * @see		<a href="../../../../constant-values.html#org.opengion.hayabusa.taglib.UserInfoTag.CMD_GET">コマンド定数</a>
	 */
	protected void commandExec( final String command ) {
		if( CMD_SQL.equals( command ) ) { setSQLAttribute() ; }
		else if( CMD_SET.equals(    command ) ) { setAttribute() ;    }
		else if( CMD_GET.equals(    command ) ) { getAttribute() ;    }
		else if( CMD_NVLGET.equals( command ) ) { getAttribute() ;    }	// 3.5.5.3 (2004/04/09)
		else if( CMD_REMOVE.equals( command ) ) { removeAttribute() ; }
	}

	/**
	 * UserInfoの文字列を登録します。
	 *
	 * @og.rev 4.3.4.0 (2008/12/01) GE20(ユーザー定数)へ登録するかのフラグを追加
	 *
	 */
	private void setAttribute() {
		setUserInfo( key, value, save );
	}

	/**
	 * UserInfoの文字列を作成します。
	 *
	 */
	private void getAttribute() {
		value = getUserInfo( key );
	}

	/**
	 * UserInfoの文字列を削除します。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) GE20(ユーザー定数)から削除するかのフラグを追加
	 */
	private void removeAttribute() {
		final UserInfo userInfo = getUser();
		userInfo.removeAttribute( key, save );
	}

	/**
	 * UserInfoの文字列を指定のSQL文より作成します。
	 *
	 * @og.rev 4.3.4.0 (2008/12/01) GE20(ユーザー定数)へ登録するかのフラグを追加
	 *
	 */
	private void setSQLAttribute() {
		if( table == null || table.getRowCount() == 0 ) { return ; }

		final int row    = 0;
		final int clmCnt = table.getColumnCount();
		for( int clm=0; clm<clmCnt; clm++ ) {
			final String clmkey = table.getColumnName( clm );
			final String clmval = table.getValue( row,clm );
			setUserInfo( clmkey, clmval, save );
		}
	}

	/**
	 * 【TAG】コマンド (SET,GET,NVLGET,REMOVE,SQL)をセットします(初期値:SET)。
	 *
	 * @og.tag
	 * コマンドは,HTMLから(get/post)指定されますので,CMD_xxx で設定される
	 * フィールド定数値のいずれかを、指定できます。
	 * 何も設定されない、または、null の場合は、"SET" が初期値にセットされます。
	 * SQL の場合、検索結果の戻り値が複数存在する場合は、最初の１件目のみ使用します。
	 *
	 * @param	cmd コマンド (public static final 宣言されている文字列)
	 * @see		<a href="../../../../constant-values.html#org.opengion.hayabusa.taglib.UserInfoTag.CMD_SET">コマンド定数</a>
	 */
	public void setCommand( final String cmd ) {
		final String cmd2 = getRequestParameter( cmd );
		if( cmd2 != null && cmd2.length() > 0 ) { command = cmd2.toUpperCase(Locale.JAPAN); }
	}

	/**
	 * 【TAG】UserInfo に登録するキーをセットします(予約語:JNAME,ID,PASSWD,IDNO,INFO,LANG,ROLE,IPADDRESS,LOGINTIME)。
	 *
	 * @og.tag UserInfo に登録するキーをセットします。
	 *
	 * @param	inkey 登録するキー
	 */
	public void setKey( final String inkey ) {
		key = getRequestParameter( inkey ).toUpperCase(Locale.JAPAN);
		if( key.startsWith( "USER." ) ) {
			key = key.substring( 5 );
		}
	}

	/**
	 * 【TAG】UserInfo に登録する値をセットします。
	 *
	 * @og.tag UserInfo に登録する値をセットします。
	 *
	 * @param	val	登録値
	 */
	public void setValue( final String val ) {
		value = getRequestParameter( val );
	}

	/**
	 * 【TAG】UserInfo に登録した値を永続化するかを指定します(初期値:false)。
	 *
	 * @og.tag
	 * UserInfo に登録した値を永続化するかを指定します。
	 * trueが指定された場合、UserInfoに設定された値は、GE20(ユーザー定数)に保存され、
	 * UserInfoが再作成されるタイミングで自動的に復元されます。
	 * ここで、登録された値は、そのユーザーの全てのロールに対して有効となります。
	 * 初期値は、false(永続化しない)です。
	 *
	 * @param	sv	登録値を永続化するか
	 */
	public void setSave( final String sv ) {
		save = nval( getRequestParameter( sv ),save );
	}

	/**
	 * 【TAG】(通常は使いません)Queryオブジェクトを作成する時のDB接続IDを指定します。
	 *
	 * @og.tag Queryオブジェクトを作成する時のDB接続IDを指定します。
	 *
	 * @param	id データベース接続ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION	)
				.println( "command"			,command	)
				.println( "key"				,key		)
				.println( "value"			,value		)
				.println( "dbid"			,dbid		)
				.println( "sql"				,sql		)
				.println( "save"			,save		)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
