package org.opengion.hayabusa.taglib;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;											// 6.4.3.4 (2016/03/11)
import java.util.Iterator;										// 6.7.7.0 (2017/03/31)
import java.util.function.IntFunction;							// 7.0.1.3 (2018/11/12)

import org.opengion.fukurou.system.BuildNumber;					// 7.0.1.2 (2018/11/04)
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModelSorter;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBColumn;						// 6.8.3.1 (2017/12/01)
import org.opengion.hayabusa.io.JsChartData;					// 7.0.1.1 (2018/10/22)

import org.opengion.fukurou.util.ArraySet;						// 6.4.3.4 (2016/03/11)
import org.opengion.fukurou.util.ToString;
import org.opengion.fukurou.util.StringUtil;					// 6.8.3.1 (2017/12/01)
import org.opengion.fukurou.util.ColorMap;						// 7.0.1.3 (2018/11/12)
import static org.opengion.fukurou.util.StringUtil.nval ;

/**
 * JsChart は、JavascriptのjsChart用のｽｸﾘﾌﾟﾄを出力するクラスです。
 * 複数の JsChartData オブジェクトを合成することも、ここで行っています。
 * ChartJSを利用しているため、標準属性以外の項目をセットする場合はoptionAttributesで行ってください。
 * 例えばアニメーションをOFFにする場合はanimation:falseをセットします。
 *
 * 出力されるスクリプトでは、idを指定しない場合はhybscanvas[tableId]が利用されます。
 * 複数のグラフを同一画面で出力する場合はidかtableIdを変えてください。
 * チャートオブジェクトはchart_[id]という名前で作成されるため、ajax等でコントロールが必要な場合は利用してください。
 *
 * @og.formSample
 * ●形式：&lt;og:jsChart chartType="…" ... /&gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{$#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *  &lt;og:jsChart
 *      chartType       ○【TAG】ﾁｬｰﾄの種類(type属性)を指定します[line/bar/horizontalBar/radar/polarArea/pie/doughnut](必須)。
 *      labelColumn     ○【TAG】ﾗﾍﾞﾙのｶﾗﾑ名(data:labels属性)を指定します(表示名称）(必須)。
 *      id                【TAG】canvasﾀｸﾞのidを指定します(初期値:hybscanvas)。
 *      height            【TAG】ﾁｬｰﾄ(canvasﾀｸﾞ)の高さを指定します(初期値:400)。
 *      width             【TAG】ﾁｬｰﾄ(canvasﾀｸﾞ)の幅を指定します(初期値:400)。
 *      title             【TAG】ﾀｲﾄﾙを指定します(options:title:text)。
 *      titlePosition     【TAG】ﾀｲﾄﾙの表示位置[top/right/bottom/left]を指定します(options:title:position)。
 *      legendDisplay     【TAG】凡例を表示するか[true/false]を指定します(options:legend:display)。
 *      legendPosition    【TAG】凡例の表示位置[top/right/bottom/left]を指定します(options:legend:position)。
 *      usePointStyle     【TAG】凡例のスタイル属性を使用するかどうか[true/false]を指定します(options:legend:labels:usePointStyle)。	// 6.8.5.0 (2018/01/09)
 *      barWidthPer       【TAG】棒線の横幅を指定します(初期値:0.8, typeがbar,horizontalBarの場合に有効)(options:xAxes:categoryPercentage)。
 *      onClick           【TAG】ﾁｬｰﾄｸﾘｯｸ時のｲﾍﾞﾝﾄを指定します(options:onClick)。
 *      plugins           【TAG】プラグイン定義された関数を指定します(plugins)。	// 6.9.9.2 (2018/09/18)
 *  ===================
 *      xposition         【TAG】x軸の表示位置[top/right/bottom/left]を指定します(初期値:bottom)(options:scales:xAxes:position)。		// 7.0.1.2 (2018/11/04)
 *      xscaleType        【TAG】x軸のｽｹｰﾙﾀｲﾌﾟ[category/linear/time/realtime]を指定します(初期値:category)(options:scales:xAxes:type)。
 *      xlabel            【TAG】x軸のラベルを指定します(options:scales:xAxes:scaleLabel:labelString)。
 *      xscaleCallback    【TAG】x軸ｺｰﾙﾊﾞｯｸを指定します(options:scales:xAxes:ticks:callback)。
 *      xbeginAtZero      【TAG】x軸を０から書き始まるかどうか(xscaleTypeがlinearの場合に有効)(options:scales:xAxes:ticks:beginAtZero)	// 7.0.1.1 (2018/10/22) 初期値 null
 *      xmax              【TAG】x軸の最大値を指定します(xscaleTypeがlinearの場合に有効)(options:scales:xAxes:ticks:max)。
 *      xmin              【TAG】x軸の最小値を指定します(xscaleTypeがlinearの場合に有効)(options:scales:xAxes:ticks:min)。
 *      xstepSize         【TAG】x軸のメモリ幅を指定します(xscaleTypeがlinearの場合に有効)(options:scales:xAxes:ticks:stepSize)。
 *      optChart          【TAG】chartの属性に、TLDで未定義の属性を追加指定します。							// 7.0.1.2 (2018/11/04)
 *      optOptions        【TAG】optionsの属性に、その他ｵﾌﾟｼｮﾝを追加指定します。							// 7.0.1.2 (2018/11/04)
 *      optAxis           【TAG】その他options:scales:xAxesのｵﾌﾟｼｮﾝを指定します。							// 7.0.1.2 (2018/11/04)
 *      optTicks          【TAG】その他options:scales:xAxes:ticksのｵﾌﾟｼｮﾝを指定します。						// 7.0.1.2 (2018/11/04)
 *      optScaleLabel     【TAG】その他options:scales:xAxes:scaleLabelのｵﾌﾟｼｮﾝを指定します。				// 7.0.1.2 (2018/11/04)
 *      optGridLines      【TAG】その他options:scales:xAxes:gridLinesのｵﾌﾟｼｮﾝを指定します。					// 7.0.1.2 (2018/11/04)
 *  =================== options:scales:xAxes:time 以下の属性(xscaleTypeがtimeの場合に有効)
 *      timeUnit          【TAG】x軸のﾀｲﾑの単位[year/quarter/month/week/day/hour/minute/second]を指定(time:unit)します(指定しない場合は自動)。
 *      timeMax           【TAG】x軸のﾀｲﾑの最大値を指定します(time:max)。
 *      timeMin           【TAG】x軸のﾀｲﾑの最小値を指定します(time:min)。
 *      timeUnitStepSize  【TAG】x軸のﾀｲﾑの単位幅を指定します(time:unitStepSize)。
 *      timeSetFormat     【TAG】x軸の設定するﾀｲﾑ(入力ﾃﾞｰﾀ)のﾌｫｰﾏｯﾄを指定します(time:format)。
 *      timeLblFormat     【TAG】x軸の表示するﾀｲﾑのﾌｫｰﾏｯﾄを指定します(time:displayFormats:year～secondまで同じ値を設定します)。
 *      tooltipFormat     【TAG】時間のﾂｰﾙﾁｯﾌﾟに使用するﾌｫｰﾏｯﾄ(time:tooltipFormat)							// 7.0.1.0 (2018/10/15)
 *  ===================
 *      markValues        【TAG】y軸に横マーカーラインの設定値をCSV形式で複数指定します	// 6.8.5.0 (2018/01/09)
 *      markColors        【TAG】y軸に横マーカーラインの色をCSV形式で複数指定します		// 6.8.5.0 (2018/01/09)
 *      markLbls          【TAG】y軸に横マーカーラインのラベルをCSV形式で複数指定します(未指定時はラベルを表示しません)	// 6.8.5.0 (2018/01/09)
 *      markAdjust        【TAG】y軸に横マーカーラインのラベル表示位置の上下方向を調整します(＋でラインの下側、－で上側に位置を移動します。初期値:-6)。	// 6.8.5.0 (2018/01/09)
 *      xmarkValues       【TAG】x軸に縦マーカーラインの設定値をCSV形式で複数指定します	// 7.0.1.1 (2018/10/22)
 *      xmarkColors       【TAG】x軸に縦マーカーラインの色をCSV形式で複数指定します		// 7.0.1.1 (2018/10/22)
 *      markWidth         【TAG】x軸,y軸全マーカーライン共通のラインの幅を指定します:borderWidth(初期値:2)					// 7.0.1.1 (2018/10/22)
 *      markDash          【TAG】x軸,y軸全マーカーライン共通のラインに点線を指定([5,2]など)します:borderDash(初期値:null)	// 7.0.1.1 (2018/10/22)
 *      markFontSize      【TAG】x軸,y軸全マーカーライン共通のラベルのフォントサイズを指定します:fontSize(初期値:10)		// 7.0.1.1 (2018/10/22)
 *  ===================
 *      useZeroDataOmit   【TAG】ﾃﾞｰﾀが０の場合、使用しない（除外する）かどうかを指定します[true:０ﾃﾞｰﾀを除外する](初期値:false)
 *      useRenderer       【TAG】データ出力でレンデラを利用するかどうか[true/false]を指定します(初期値:false)
 *      sortColumn        【TAG】検索結果をこのカラムでソートしなおします(初期値:null)。
 *      valueQuot         【TAG】値の前後にクオートをはさむかどうか[true/false]指定します。
 *      useZoom           【TAG】ズーム処理を使用するかどうか[true/false]を指定します。	// 6.8.5.0 (2018/01/09)
 *      varColumns        【TAG】TableModelの指定のカラムをvarの配列変数として出力します。	// 7.0.1.2 (2018/11/04)
 *  ===================
 * X    widthEventColumn  【廃止】横幅を動機に設定するｶﾗﾑのIDを指定します。
 * X    heightEventColumn 【廃止】縦幅を動的に設定するｶﾗﾑのIDを指定します。
 * X    minEventColumn    【廃止】minを動的に設定するｶﾗﾑのIDを指定します。
 * X    maxEventColumn    【廃止】maxを動的に設定するｶﾗﾑのIDを指定します。
 *  ===================
 * X    optionAttributes  【廃止】optionsの属性に、その他ｵﾌﾟｼｮﾝを追加指定します。(7.0.1.2 (2018/11/04) 廃止)
 *      tableId           【TAG】(通常は使いません)sessionから所得する DBTableModelオブジェクトの ID
 *      scope             【TAG】キャッシュする場合のスコープ[request/page/session/application]を指定します(初期値:session)
 *      caseKey           【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *      caseVal           【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *      caseNN            【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない)
 *      caseNull          【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない)
 *      caseIf            【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *      debug             【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *  &gt;   ... Body ...
 *  &lt;/og:jsChart&gt;
 *
 * ●使用例
 *      &lt;og:jsChart
 *          chartType      = "[line/bar/horizontalBar/radar/polarArea/pie/doughnut]"
 *          labelColumn    = "LDATA"
 *          id             = "hybscanvas"
 *          height         = "400"
 *          width          = "400"
 *          title          = "タイトル"
 *          titlePosition  = "top"				[top/right/bottom/left]
 *          xlabel         = "名称"
 *          legendPosition = "right"			[top/right/bottom/left]
 *          legendDisplay  = "true"				[true/false]
 *          xsclaeCallback = "function(value){return value + ' 様';}"
 *          xscaleType     = "linear"
 *          xmax           = "1000000"
 *          xmin           = "100000"
 *          xstepSize      = "10000"
 *          barWidthPer    = "0.4"
 *      &gt;
 *          &lt;og:jsChartData ... /&gt;
 *      &lt;/og:jsChart&gt;
 *
 * @og.group 画面表示
 *
 * @version	5.9.17.2		2017/02/08
 * @author	T.OTA
 * @since	JDK7.0
 *
 */
public class JsChartTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。{@value} */
	private static final String VERSION = "7.0.1.3 (2018/11/12)" ;
	private static final long serialVersionUID = 701320181112L ;

	// xscaleType="realtime" のときのみ、ｲﾝｸﾙｰﾄﾞします。
	private static final String STREAMING_JS	= "common/option/chartjs-plugin-streaming.min.js";		// 7.0.1.2 (2018/11/04)
	private static final String SCRIPT_STREAMING_JS
							= "<script type=\"text/javascript\" src=\""
								+ "/" + HybsSystem.getContextName() + "/jsp/"
								+ STREAMING_JS
								+ "?v=" + BuildNumber.BUILD_ID
								+ "\" ><!-- --></script>"
								+ CR ;

	/** chartType 引数に渡す事の出来る タイプ 折れ線 {@value} **/
	public static final String		CTYPE_LINE			= "line";
	/** chartType 引数に渡す事の出来る タイプ 棒線 {@value} **/
	public static final String		CTYPE_BAR			= "bar";
	/** chartType 引数に渡す事の出来る タイプ 横棒線 {@value} **/
	public static final String		CTYPE_HBAR			= "horizontalBar";
	/** chartType 引数に渡す事の出来る タイプ レイダー {@value} **/
	public static final String		CTYPE_RADAR			= "radar";
	/** chartType 引数に渡す事の出来る タイプ ポーラエリア {@value} **/
	public static final String		CTYPE_POLAR			= "polarArea";
	/** chartType 引数に渡す事の出来る タイプ 円 {@value} **/
	public static final String		CTYPE_PIE			= "pie";
	/** chartType 引数に渡す事の出来る タイプ ドーナツ {@value} **/
	public static final String		CTYPE_DOUGHNUT		= "doughnut";
	/** chartType 引数に渡す事の出来る タイプ リスト {@value} */

	private static final Set<String> CTYPE_SET		= new ArraySet<>( CTYPE_LINE,CTYPE_BAR,CTYPE_HBAR,CTYPE_RADAR,CTYPE_POLAR,CTYPE_PIE,CTYPE_DOUGHNUT );

	// 6.9.9.4 (2018/10/01) String配列から、Set に変更
	/** chartType が円形のリスト */
	private static final Set<String> SET_CI_TYPE	= new ArraySet<>( CTYPE_RADAR, CTYPE_POLAR, CTYPE_PIE, CTYPE_DOUGHNUT );

	private static final String TYPE_CATEGORY		= "category";
	private static final String TYPE_LINEAR			= "linear";
	private static final String TYPE_TIME			= "time";
	private static final String TYPE_REALTIME		= "realtime";		// 7.0.1.2 (2018/11/04)

	private static final Set<String> SET_POSITION	= new ArraySet<>( "top", "right", "bottom", "left" );
	private static final Set<String> SET_TIMEUNIT	= new ArraySet<>( "year", "quarter", "month", "week", "day", "hour", "minute", "second" );					// 7.0.0.1 (2018/10/09)
	private static final Set<String> SET_XSCALE		= new ArraySet<>( TYPE_CATEGORY, TYPE_TIME, TYPE_LINEAR, TYPE_REALTIME );
	private static final Set<String> SET_BOOLEAN	= new ArraySet<>( "true", "false" );

	private static final String 	CANVAS_NAME		= "hybscanvas";

//	private static final String		MARK_DEF_COLOR	= "BLUE";		// 6.8.5.0 (2018/01/09) y軸に横マーカーラインの色の初期値
	private static final String		MARK_DEF_ADJUST	= "-6";			// 6.8.5.0 (2018/01/09) y軸に横マーカーラインの位置調整の初期値

	// 7.0.1.3 (2018/11/12) バッファキー検索処理追加
	private static final String		TIME_FORMAT_JA	= "{year:'YYYY年',quarter:'YYYY年M月',month:'YYYY年M月',week:'M月D日',day:'M月D日',hour:'D日 H時',minute:'H時m分',second:'m分s秒'}" ;
	private static final String		TIME_FORMAT		= "{year:'YYYY',quarter:'YYYY/M',month:'YYYY/M',week:'M/D',day:'M/D',hour:'D HH',minute:'HH:mm',second:'HH:mm:ss'}" ;

	// 7.0.1.3 (2018/11/12) true/false なので、記号化します。
	private static final boolean	USE_QUOTE		= false;
	private static final boolean	NO_QUOTE		= true;		// IS_NUMBER か、!USE_QUOTE か、

	// 変数宣言
	// 6.9.8.0 (2018/05/28) FindBugs:直列化可能クラスの非 transient で非直列化可能なインスタンスフィールド
	private final transient	List<JsChartData>	jsChartData = new ArrayList<JsChartData>() ;		// 6.7.5.0 (2017/03/10) jsChartDataのﾘｽﾄ

	private transient	JsChartData jsXAxis  = new JsChartData();	// xAxes の設定用(datasetは使いません)

	private String	chartBody			;							// ﾁｬｰﾄタグのBODY部分に書かれた文字列

	private String	chartType			;							// ﾁｬｰﾄﾀｲﾌﾟ(必須)
	private String	labelColumn			;							// ﾗﾍﾞﾙｶﾗﾑ(必須)
	private String	id					;							// canvasﾀｸﾞのid
	private String	height				= "400";					// canvasﾀｸﾞのheight
	private String	width				= "400";					// canvasﾀｸﾞのwidth
	private String	title				;							// ﾀｲﾄﾙ
	private String	titlePosition		;							// ﾀｲﾄﾙ位置
	private String	legendPosition		;							// 凡例位置
	private String	legendDisplay		;							// 凡例表示ﾌﾗｸﾞ
	private boolean	usePointStyle		;							// 6.8.5.0 (2018/01/09) 点のスタイル属性を使用するかどうか(初期値:false)
	private boolean	useLegend			;							// Legend関連属性(legendPosition,legendDisplay,usePointStyle) のどれかがセットされれば、true
	private String	barWidthPer			= "0.8";					// 棒線の横幅(ﾊﾟｰｾﾝﾄ)
	private String	onClick				;							// ｸﾘｯｸｲﾍﾞﾝﾄ
	private String	plugins				;							// 6.9.9.2 (2018/09/18) プラグイン定義された関数を指定します。
	private String	xscaleType			= TYPE_CATEGORY ;			// x軸のｽｹｰﾙﾀｲﾌﾟ
	private String	xposition			;							// x軸の表示位置[top/right/bottom/left]	 // 7.0.1.2 (2018/11/04)
	private boolean	valueQuot			;							// 値の前後にクオートをはさむかどうか[true/false]指定します。
	private String	tableId				= HybsSystem.TBL_MDL_KEY;	// ﾃｰﾌﾞﾙid
	private String	markValues			;							// 6.8.5.0 (2018/01/09) y軸に横マーカーラインの設定値をCSV形式で複数指定します
	private String	markColors			;							// 6.8.5.0 (2018/01/09) y軸に横マーカーラインの色をCSV形式で複数指定します
	private String	markLbls			;							// 6.8.5.0 (2018/01/09) y軸に横マーカーラインのラベルをCSV形式で複数指定します(未指定時はラベルを表示しません)
	private String	markAdjust			;							// 6.8.5.0 (2018/01/09) y軸に横マーカーラインのラベル表示位置の上下方向を調整します(初期値:-6)。
	private String	xmarkValues			;							// 7.0.1.1 (2018/10/22) x軸に縦マーカーラインの設定値をCSV形式で複数指定します
	private String	xmarkColors			;							// 7.0.1.1 (2018/10/22) x軸に縦マーカーラインの色をCSV形式で複数指定します
	private String	markWidth			= "2";						// 7.0.1.1 (2018/10/22) マーカーライン共通のラインの幅を指定します:borderWidth(初期値:2)
	private String	markDash			;							// 7.0.1.1 (2018/10/22) マーカーライン共通のラインに点線を指定([5,2]など)します:borderDash(初期値:null)
	private String	markFontSize  		= "10";						// 7.0.1.1 (2018/10/22) マーカーライン共通のラベルのフォントサイズを指定します:fontSize(初期値:10)
	private boolean	useZoom				;							// 6.8.5.0 (2018/01/09) ズーム処理を使用するかどうか(初期値:false)
	private String	varColumns  		;							// 7.0.1.2 (2018/11/04) TableModelの指定のカラムをvarの配列変数として出力します。
//	private String	widthEventColumn	;							// (廃止)横幅の動的参照ｶﾗﾑ	2017/03/28 ADD
//	private String	heightEventColumn	;							// (廃止)縦幅の動的参照ｶﾗﾑ	2017/03/28 ADD
//	private String	minEventColumn		;							// (廃止)最小値の動的参照ｶﾗﾑ	2017/03/28 ADD
//	private String	maxEventColumn		;							// (廃止)最大値の動的参照ｶﾗﾑ	2017/03/28 ADD
	private boolean useZeroDataOmit		;							// 6.7.7.0 (2017/03/31) ﾃﾞｰﾀが０の場合、使用しない（除外する）かどうか
	private boolean useRenderer			;							// 6.7.9.0 (2017/04/28) useRenderer 追加
	private String	sortColumn			;							// 6.8.0.0 (2017/06/02) 検索結果をこのカラムでソートしなおします(初期値:null)。
//	private String	optionAttributes	;							// ｵﾌﾟｼｮﾝ
//	private String	chartAttributes		;							// 6.9.9.2 (2018/09/18) chartの属性に、TLDで未定義の属性を追加指定します。
	private String	optChart			;							// 7.0.1.2 (2018/11/04) chartの属性に、TLDで未定義の属性を追加指定します。
	private String	optOptions			;							// 7.0.1.2 (2018/11/04) optionsの属性に、その他ｵﾌﾟｼｮﾝを追加指定します。

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.9.7.0 (2018/05/14) PMD Each class should declare at least one constructor
	 */
	public JsChartTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 6.7.5.0 (2017/03/10) jsChartData属性の初期化もれ
	 * @og.rev 5.9.19.0 (2017/04/07) T.OTA 61200-170316-02	ﾁｬｰﾄｻｲｽﾞ・max・minの動的変更対応
	 * @og.rev 6.7.7.0 (2017/03/31) useZeroDataOmit属性の追加
	 * @og.rev 6.7.9.0 (2017/04/28) useRenderer 追加
	 * @og.rev 6.8.0.0 (2017/06/02) sortColumn 追加
	 * @og.rev 6.8.3.0 (2017/11/27) useEqValOmit属性の追加
	 * @og.rev 6.8.5.0 (2018/01/09) xbeginAtZero,ybeginAtZero,markValues,markColors,markLbls,markAdjust,rangeMin,rangeMax,usePointStyle属性の追加
	 * @og.rev 6.9.9.2 (2018/09/18) plugins,chartAttributes属性の追加
	 * @og.rev 6.9.9.3 (2018/09/25) xscaleType の time 属性廃止。
	 * @og.rev 6.9.9.4 (2018/10/01) ﾘﾆｱ対応,time 属性復活
	 * @og.rev 6.9.9.4 (2018/10/01) 7.0.1.0 (2018/10/15) time 属性修正、tooltipFormat属性の追加
	 * @og.rev 7.0.1.1 (2018/10/22) xmarkValues,xmarkColors属性の追加
	 * @og.rev 7.0.1.1 (2018/10/22) ylabelColor,y2label,y2labelColor属性の追加
	 * @og.rev 7.0.1.2 (2018/11/04) ylabelColor,y2label,y2labelColor属性の追加
	 */
	@Override
	protected void release2() {
		super.release2();
		jsChartData.clear();				// 6.7.5.0 (2017/03/10)
		jsXAxis				= new JsChartData();				// xAxes の設定用(datasetは使いません)

		chartBody			= null;		// 7.0.1.1 (2018/10/22) ﾁｬｰﾄタグのBODY部分に書かれた文字列
		chartType			= null;
		id					= null;
		height				= "400";
		width				= "400";
		labelColumn			= null;
		title				= null;
		titlePosition		= null;
		legendPosition		= null;
		legendDisplay		= null;
		usePointStyle		= false;	// 6.8.5.0 (2018/01/09) 点のスタイル属性を使用するかどうか
		useLegend			= false;	// 7.0.1.1 (2018/10/22) Legend関連属性(legendPosition,legendDisplay,usePointStyle) のどれかがセットされれば、true
		barWidthPer			= "0.8";
		onClick				= null;
		plugins				= null;		// 6.9.9.2 (2018/09/18) プラグイン定義された関数を指定します。
		xscaleType			= TYPE_CATEGORY;
		xposition			= null;		// 7.0.1.2 (2018/11/04) x軸の表示位置[top/right/bottom/left]
		valueQuot			= false;	// 7.0.1.1 (2018/10/22) 値の前後にクオートをはさむかどうか[true/false]指定します。
		tableId				= HybsSystem.TBL_MDL_KEY;
		markValues			= null;		// 6.8.5.0 (2018/01/09) y軸に横マーカーラインの設定値をCSV形式で複数指定します
		markColors			= null;		// 6.8.5.0 (2018/01/09) y軸に横マーカーラインの色をCSV形式で複数指定します
		markLbls			= null;		// 6.8.5.0 (2018/01/09) y軸に横マーカーラインのラベルをCSV形式で複数指定します(未指定時はラベルを表示しません)
		markAdjust			= null;		// 6.8.5.0 (2018/01/09) y軸に横マーカーラインのラベル表示位置の上下方向を調整します(初期値:-6)。
		xmarkValues			= null;		// 7.0.1.1 (2018/10/22) x軸に縦マーカーラインの設定値をCSV形式で複数指定します
		xmarkColors			= null;		// 7.0.1.1 (2018/10/22) x軸に縦マーカーラインの色をCSV形式で複数指定します
		markWidth			= "2";		// 7.0.1.1 (2018/10/22) マーカーライン共通のラインの幅を指定します:borderWidth(初期値:2)
		markDash			= null;		// 7.0.1.1 (2018/10/22) マーカーライン共通のラインに点線を指定([5,2]など)します:borderDash(初期値:null)
		markFontSize  		= "10";		// 7.0.1.1 (2018/10/22) マーカーライン共通のラベルのフォントサイズを指定します:fontSize(初期値:10)
		useZoom				= false;	// 6.8.5.0 (2018/01/09) ズーム処理を使用するかどうか(初期値:false)
		varColumns  		= null;		// 7.0.1.2 (2018/11/04) TableModelの指定のカラムをvarの配列変数として出力します。
//		widthEventColumn	= null;		// 5.9.19.0
//		heightEventColumn	= null;		// 5.9.19.0
//		maxEventColumn		= null;		// 5.9.19.0
//		minEventColumn		= null;		// 5.9.19.0
		useZeroDataOmit		= false;	// 6.7.7.0 (2017/03/31) ﾃﾞｰﾀが０の場合、使用しない（除外する）かどうか
		useRenderer			= false;	// 6.7.9.0 (2017/04/28) useRenderer 追加
		sortColumn			= null;		// 6.8.0.0 (2017/06/02) 検索結果をこのカラムでソートしなおします(初期値:null)。
//		optionAttributes	= null;
//		chartAttributes		= null;		// 6.9.9.2 (2018/09/18) chartの属性に、TLDで未定義の属性を追加指定します。
		optChart			= null;		// 7.0.1.2 (2018/11/04) chartの属性に、TLDで未定義の属性を追加指定します。
		optOptions			= null;		// 7.0.1.2 (2018/11/04) optionsの属性に、その他ｵﾌﾟｼｮﾝを追加指定します。
	}

	/**
	 * Taglibの開始タグが見つかった時に処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 6.7.5.0 (2017/03/10) タグの使用を決める共通属性の追加
	 *
	 * @return 後続処理の指示
	 */
	@Override
	public int doStartTag() {
		if( !useTag() ) { return SKIP_BODY ; }		// 6.7.5.0 (2017/03/10)

		return EVAL_BODY_BUFFERED; // Bodyを評価する
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 7.0.1.1 (2018/10/22) このスクリプトの中に入れたい文字があれば、登録できるようにします。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		chartBody = getBodyString();

		if( chartBody != null ) {
			chartBody = chartBody.trim();
		}

		return SKIP_BODY ;
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 6.7.5.0 (2017/03/10) タグの使用を決める共通属性の追加
	 * @og.rev 6.9.9.4 (2018/10/01) idの振り方、データの持ち方変更
	 *
	 * @return 後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();
		if( !useTag() ) { return EVAL_PAGE ; }			// 6.7.5.0 (2017/03/10)

		id = (id==null ? tableId : id );				// 6.9.9.4 (2018/10/01) id指定なしの場合はtableId

		// jsChart出力
		jspPrint( jsChartOutput() );

		return EVAL_PAGE;
	}

	/**
	 * jsChart出力用
	 * jsChartTag と jsChartData を使用して、jsChart情報を出力します。
	 *
	 * @og.rev 5.9.19.0 (2017/04/07) T.OTA 61200-170316-02	ﾁｬｰﾄｻｲｽﾞ・max・minの動的変更対応
	 * @og.rev 6.7.7.0 (2017/03/31) ﾁｬｰﾄﾃﾞｰﾀで、ｾﾞﾛ、null カラムを非表示にします。
	 * @og.rev 6.7.9.0 (2017/04/28) useRenderer 追加
	 * @og.rev 6.8.0.0 (2017/06/02) sortColumn 追加
	 * @og.rev 6.8.3.0 (2017/11/27) useZeroDataOmit属性で、nullOmit属性もセットします。
	 * @og.rev 6.8.3.0 (2017/11/27) useEqValOmit属性の追加
	 * @og.rev 6.8.3.1 (2017/12/01) 不要なデータを出力しないようにします。
	 * @og.rev 5.9.27.0 2017/12/01	T.OTA 61200-170831-04	max,minの小数点対応
	 * @og.rev 6.8.5.0 (2018/01/09) xbeginAtZero,ybeginAtZero,markValues,markColors,markLbls,markAdjust,rangeMin,rangeMax,usePointStyle属性の追加
	 * @og.rev 6.9.9.2 (2018/09/18) chart.jsが2.4.0から2.7.2にバージョンアップにより、廃止された属性対応
	 * @og.rev 6.9.9.2 (2018/09/18) plugins,chartAttributes属性の追加
	 * @og.rev 6.9.9.3 (2018/09/25) xscaleType の time 属性廃止。
	 * @og.rev 6.9.9.4 (2018/10/01) ﾘﾆｱ対応,time 属性復活
	 * @og.rev 6.9.9.4 (2018/10/01) idの振り方、データの持ち方変更
	 * @og.rev 7.0.1.1 (2018/10/22) xmarkValues,xmarkColors属性の追加
	 * @og.rev 7.0.1.3 (2018/11/12) バッファキー検索処理追加、markColors,xmarkColors属性に、VIVID,PASTELカラー指定に対応します。
	 *
	 * @return jsChert用文字列
	 */
	private String jsChartOutput() {
		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE );

		// 各JavaScriptの変数名
//		final String qd			= "qd_" + id;		//queryData
		final String cd			= "cd_" + id;		//chartData
		final String myChart	= "chart_"+id;

		// JSON形式でﾃｰﾌﾞﾙ情報を取得
		// ﾃｰﾌﾞﾙ情報の取得
		// 6.8.0.0 (2017/06/02) sortColumn 追加
		DBTableModel table = (DBTableModel)getObject( tableId ) ;
		if( StringUtil.isNotNull( sortColumn ) ) {						// 6.8.5.0 (2018/01/09)
			final int clmNo = table.getColumnNo( sortColumn,false );	// エラーを出さない。

			final DBTableModelSorter temp = new DBTableModelSorter();
			temp.setModel( (DBTableModel)getObject( tableId ) );
			temp.sortByColumn( clmNo,true );							// 順方向のソート
			table = temp;
		}

		final int rowCount = table.getRowCount();						// 6.9.2.0 (2018/03/05)

		// 7.0.1.3 (2018/11/12) DBTableModelに存在しないカラムのChartDataは無視します。
		// これは、動的にグラフを生成する場合に、カラムの増減に合わせて、JsChartDataTagを生成しなおすのが手間だからです。
		final Iterator<JsChartData> itr = jsChartData.iterator();		// 個々のグラフ
		while( itr.hasNext() ) {
			final JsChartData jcData = itr.next();
			final String chtClm = jcData.getChartColumn();
			final int    clmNo  = table.getColumnNo( chtClm, false );	// エラーを出さない。

			if( clmNo < 0 ) {
				itr.remove();											// 7.0.1.3 (2018/11/12) カラムがDBTableModelに存在しない。
			}
			else if( useZeroDataOmit ) {	// ｾﾞﾛﾃﾞｰﾀを使用しない設定
				// 6.8.3.1 (2017/12/01) ループ処理の判定は、ChartColumn のみでよい。
				boolean isRemove = true;
				for( int row=0; row<rowCount; row++ ) {
					final String val = table.getValue( row,clmNo );
					if( StringUtil.isNotNull( val ) && !"0".equals( val ) && !"0.0".equals( val ) && !"0.00".equals( val )  ) {	// 6.8.5.0 (2018/01/09)
						isRemove = false;
						break;		// 判定処理打ち切り
					}
				}

				if( isRemove ) {
					itr.remove();										// すべてが、ｾﾞﾛ、null カラムを、削除します。
				}
			}
		}

//		// ｾﾞﾛﾃﾞｰﾀを使用しない設定
//		if( useZeroDataOmit ) {
//			final Iterator<JsChartData> itr = jsChartData.iterator();	// 個々のグラフ
//			while( itr.hasNext() ) {
//				final JsChartData jcData = itr.next();
//				final String chtClm = jcData.getChartColumn();
//				final int    clmNo  = table.getColumnNo( chtClm, false );	// エラーを出さない。
//
//				// 6.8.3.1 (2017/12/01) ループ処理の判定は、ChartColumn のみでよい。
//				boolean isRemove = true;
//				if( clmNo >= 0 ) {
//					for( int row=0; row<rowCount; row++ ) {
//						final String val = table.getValue( row,clmNo );
//						if( StringUtil.isNotNull( val ) && !"0".equals( val ) && !"0.0".equals( val ) && !"0.00".equals( val )  ) {	// 6.8.5.0 (2018/01/09)
//							isRemove = false;
//							break;	// 判定処理打ち切り
//						}
//					}
//				}
//				if( isRemove ) {
//					itr.remove();											// すべてが、ｾﾞﾛ、null カラムを、削除します。
//				}
//			}
//		}

		// 6.8.3.1 (2017/12/01) 不要なデータを出力しないようにします。
		final int clmSize = jsChartData.size();								// JsChartTag の 値部分のみの配列

		final String[] clmNms = new String[clmSize];						// 6.9.9.4 (2018/10/01) カラム名の配列
		final int[]    clmNos = new int[clmSize];
		final int      lblNos = table.getColumnNo( labelColumn, false );	// エラーを出さない。 6.9.2.0 (2018/03/05)
		final DBColumn dbClm  = table.getDBColumn( lblNos );				// 6.9.2.0 (2018/03/05)

		// jsChartDataﾀｸﾞの変数宣言
		for( int j=0; j<clmSize; j++ ) {
			final String chtClm = jsChartData.get(j).getChartColumn();
			clmNms[j] = chtClm;												// 6.9.9.4 (2018/10/01) カラム名の配列
			clmNos[j] = table.getColumnNo( chtClm, false );					// エラーを出さない。
		}

		// 7.0.1.2 (2018/11/04) 原因調査中。chartjs-plugin-streaming.min.js を組み込むと xscaleType="time" が表示しない。
		// 通常、HeadTag で、useChartJS=true で、一括include しているが、streaming だけ、
		// xscaleType="realtime" 時だけ組み込むように変更します。
		if( TYPE_REALTIME.equals( xscaleType ) ) {
			rtn.append( SCRIPT_STREAMING_JS );
		}

		// canvasﾀｸﾞの設定
		rtn.append( "<canvas class=\""	).append( CANVAS_NAME )
			.append( "\" id=\""			).append( id		)
			.append( "\" width=\""		).append( width		)
			.append( "\" height=\""		).append( height	)
			.append( "\"><!-- --></canvas>" ).append( CR	)
			.append( "<script>" ).append( CR )									// 6.9.9.4 (2018/10/01) query情報の取得(JSON)
			.append( chartBody );												// 7.0.1.1 (2018/10/22) BODY部分の文字列の組み込み

		final boolean isXcateType  = TYPE_CATEGORY.equals(	xscaleType );		// 6.9.9.4 (2018/10/01)
		final boolean isXlinerType = TYPE_LINEAR.equals(	xscaleType );		// 6.8.5.0 (2018/01/09) xscaleType が linear かどうか。
		final boolean isXtimeType  = TYPE_TIME.equals(		xscaleType );		// 6.8.5.0 (2018/01/09) xscaleType が time かどうか。

		// 7.0.1.3 (2018/11/12) var 変数に設定する配列情報を、bufに追加します。
		final DBTableModel fcTable = table;			// ラムダ式で使えるのは、final宣言された変数のみ。根本は、Sorterを組み込んでfinalすべき。
		final IntFunction<String> lcFunc = (row) -> {
											final String lval = fcTable.getValue( row,lblNos );
											return useRenderer && !isXlinerType ? StringUtil.jsonFilter( dbClm.getRendererValue(row,lval) ) : lval ;
										};
		setVarArray( rtn,labelColumn,rowCount,isXcateType || isXtimeType || useRenderer,lcFunc );

//		// 6.9.9.4 (2018/10/01) ラベル部の出力
//		rtn.append( " var " ).append( labelColumn ).append( "=[" );
//		for( int row=0; row<rowCount; row++ ) {
//			if( row > 0 ) { rtn.append( ',' ); }		// カンマは、最初のデータ以降の最初につける。
//
//			final String lval = table.getValue( row,lblNos );
//			final String lblVal = useRenderer && !isXlinerType ? StringUtil.jsonFilter( dbClm.getRendererValue(row,lval) ) : lval ;
//			if( isXcateType || isXtimeType || useRenderer ) {
//				rtn.append( '"' ).append( lblVal ).append( '"' );
//			}
//			else {
//				rtn.append( lblVal );
//			}
//		}
//		rtn.append( "];" ).append( CR );

		// 6.9.9.4 (2018/10/01) データ部の出力
		for( int j=0; j<clmSize; j++ ) {
			final int clmNo = clmNos[j];		// finalしか参照できないため。
			setVarArray( rtn,clmNms[j],rowCount,valueQuot,(row) -> fcTable.getValue( row,clmNo ) );

//			rtn.append( " var " ).append( clmNms[j] ).append( "=[" );
//			for( int row=0; row<rowCount; row++ ) {
//				if( row > 0 ) { rtn.append( ',' ); }		// カンマは、最初のデータ以降の最初につける。
//
//				final String val = table.getValue( row,clmNos[j] );
//
//				if( valueQuot ) {						// 6.8.5.0 (2018/01/09) yscaleType が category
//					rtn.append( '"' ).append( val ).append( '"' );
//				}
//				else {
//					rtn.append( val );
//				}
//			}
//			rtn.append( "];" ).append( CR );
		}

		// x軸がlinearｽｹｰﾙの場合
		// [{x:値1,y:値2},{x:値1,y:値2},･･･] 形式のデータが必要
		if( isXlinerType ) {
			for( int j=0; j<clmSize; j++ ) {
				final String chtClm = clmNms[j];
				rtn.append( "var LI_" ).append( chtClm ).append( "=[];" ).append( CR );

				// 6.9.9.4 (2018/10/01) x軸がlinearｽｹｰﾙの場合、カラム名が、変わるので、再設定している。(超特殊処理)
				jsChartData.get(j).setChartColumn( "LI_" + chtClm );
			}

			rtn.append( "for(var i=0; i<").append( labelColumn ).append( ".length; i++){" );
			for( int j=0; j<clmSize; j++ ) {
				final String chtClm = clmNms[j];
					// {x:ﾗﾍﾞﾙ, y:値}の形式で値を設定
				rtn.append( "LI_" ).append( chtClm ).append( "[i]={x:" ).append( labelColumn )
					.append( "[i],y:" ).append( chtClm ).append( "[i]};" );
			}
			rtn.append( "};" ).append( CR );
		}

		// 7.0.1.2 (2018/11/04) varColumns 追加
		final String[] varClms  = StringUtil.csv2Array( varColumns );	// 独自に出力しておきたいカラム列の値
		for( int j=0; j<varClms.length; j++ ) {
			final int varNos = table.getColumnNo( varClms[j], false );	// エラーを出さない。
			final boolean isNumType = table.getDBColumn( varNos ).isNumberType();			// 6.4.6.0 (2016/05/27)

			setVarArray( rtn,varClms[j],rowCount,!isNumType,(row) -> fcTable.getValue( row,varNos ) );

//			rtn.append( " var " ).append( varClms[j] ).append( "=[" );
//			for( int row=0; row<rowCount; row++ ) {
//				if( row > 0 ) { rtn.append( ',' ); }					// カンマは、最初のデータ以降の最初につける。
//
//				final String val = table.getValue( row,varNos );
//
//				// 注意：valueQuot ではなく、isNumType で判定する。
//				if( isNumType ) {
//					rtn.append( val );
//				}
//				else {
//					rtn.append( '"' ).append( val ).append( '"' );
//				}
//			}
//			rtn.append( "];" ).append( CR );
		}

		// ==================================================================================
		// 7.0.1.1 (2018/10/22) jsChartData(X軸)の設定
		if( !SET_CI_TYPE.contains( chartType ) ) {
			jsXAxis.setId( "x0" );						// X軸のid
			jsXAxis.setUseTime( isXtimeType );			// x軸の時間表示を使用するかどうか

			// 7.0.1.3 (2018/11/12) バッファキー検索処理追加
			if( isXtimeType && !jsXAxis.contains( JsChartData.TIME , "displayFormats" ) ) {		// ｷｰﾜｰﾄﾞが無ければ追加
				// ほんとはリソースに入れるべきでしょう。
				if( "ja".equalsIgnoreCase( getLanguage() ) ) {						// 'ja' なら日本
					jsXAxis.addTime( "displayFormats" , TIME_FORMAT_JA , NO_QUOTE );	// 標準ﾀｲﾑﾌｫｰﾏｯﾄ適用。ｵﾌﾞｼﾞｪｸﾄなので、クオートなし
				}
				else {
					jsXAxis.addTime( "displayFormats" , TIME_FORMAT , NO_QUOTE );		// 標準ﾀｲﾑﾌｫｰﾏｯﾄ適用。ｵﾌﾞｼﾞｪｸﾄなので、クオートなし
				}
			}

			// 6.9.9.4 (2018/10/01) 不要な属性は出さない。
			if( CTYPE_BAR.equals( chartType ) || CTYPE_HBAR.equals( chartType ) ) {
				jsXAxis.addAxis( "categoryPercentage" , barWidthPer , NO_QUOTE );		// 数値
			}

			// x軸にﾘﾆｱｽｹｰﾙを設定した場合(これは残す)
			final String xpos = xposition != null ? xposition
												  : isXlinerType ? "bottom"
																 : CTYPE_HBAR.equals( chartType ) ? "left" : null ;
			jsXAxis.addAxis( "position" , xpos , USE_QUOTE );	// 文字

			rtn.append( jsXAxis.getAxis() ).append( CR );
		}

		// horizontalBar 時は、X軸とY軸を入れ替える
		final char[] yx = CTYPE_HBAR.equals( chartType ) ? new char[] { 'x','y' }		// 入替
														 : new char[] { 'y','x' } ;		// 通常の並び順

		// ==================================================================================
		// jsChartData(Y軸)の設定
		// 7.0.1.1 (2018/10/22) data:dataset と、options:scales:yAxes: を変数化して出力します。
		for( final JsChartData chData : jsChartData ) {
//			rtn.append(  chData.getDataset()		).append( CR )
			rtn.append(  chData.getDataset( yx[0])	).append( CR )		// 横棒線の場合は、'x'が、それ以外は、'y'
				.append( chData.getAxis()			).append( CR );
		}

		rtn.append( "var ").append( cd ).append( "={labels:" ).append( labelColumn ).append( ",datasets:[" );
		for( final JsChartData chData : jsChartData ) {
			rtn.append( chData.getDatasetKey() ).append( ',' );
		}
		rtn.append( "]};" ).append( CR )
			.append(  "var "    ).append( myChart   ).append( "=new Chart(" ).append( id )	// jsChartの生成
			.append( ",{type:'" ).append( chartType )
			.append( "',data:"  ).append( cd );

		// 6.9.9.2 (2018/09/18) plugins,chartAttributes属性の追加
		if( StringUtil.isNotNull( plugins ) ) {
			rtn.append( ",plugins: [" ).append( plugins ).append( ']' );
		}
//		if( StringUtil.isNotNull( chartAttributes ) ) {
//			rtn.append( ',' ).append( chartAttributes );
//		}

		rtn.append( ",options:{responsive:false" );	// ﾚｽﾎﾟﾝｼﾌﾞ OFF

		// ｸﾘｯｸｲﾍﾞﾝﾄの設定
		// clickLink 変数を使用する場合、内部でﾏｽﾀﾃﾞｰﾀを使用します。ｷｰとして、渡しておく必要があります。
		if( StringUtil.isNotNull( onClick ) ) {							// 6.8.5.0 (2018/01/09)
			rtn.append( ",onClick:function(event,obj){" ).append( onClick ).append( '}' );		// 6.9.9.4 (2018/10/01)
		}

		// ﾀｲﾄﾙ属性の設定
		if( StringUtil.isNotNull( title ) ) {						// 6.8.5.0 (2018/01/09)
			rtn.append( ",title:{display:true" );
			setProp( rtn, ",text:'"    , title        , "'" );
			setProp( rtn, ",position:'", titlePosition, "'" );
			rtn.append( '}' );
		}

		// 凡例属性の設定
		if( useLegend ) {											// 7.0.1.1 (2018/10/22)
			rtn.append( ",legend:{" );
			setProp( rtn, "display:"   , legendDisplay  , ","  );
			setProp( rtn, "position:'" , legendPosition , "'," );

			// 凡例のスタイルを、pointStyle にあわせるかどうか
			if( usePointStyle ) {									// 7.0.1.1 (2018/10/22)
				rtn.append( "labels:{usePointStyle: true}" );
			}
			rtn.append( '}' );
		}

		// 6.9.9.2 (2018/09/18) chart.jsが2.4.0から2.7.2にバージョンアップにより、廃止された属性対応
		if( !SET_CI_TYPE.contains( chartType ) ) {
			// 円形以外の場合はscales属性に設定
			rtn.append( ",scales:{" )
				.append( yx[0] ).append( "Axes:[" );		// 横棒線の場合は、'x'が、それ以外は、'y'

//			if( CTYPE_HBAR.equals( chartType ) ) {
//				// 横棒線の場合はx軸の設定(axisの複数形)
//				rtn.append( "xAxes:[" );
//			}
//			else {
//				// それ以外はy軸の設定
//				rtn.append( "yAxes:[" );
//			}

			// 7.0.1.1 (2018/10/22) options:scales:yAxes: を変数化して出力しているので、その設定のみでよい。
			for( final JsChartData chData : jsChartData ) {
				if( chData.isUseAxis() ) {
					rtn.append( chData.getAxisKey() ).append( ',' );
				}
			}
			rtn.append( "]," )
				.append( yx[1] ).append( "Axes:[" );		// 横棒線の場合は、'y'が、それ以外は、'x'

//			if( CTYPE_HBAR.equals( chartType ) ) {
//				// 横棒線の場合はy軸の設定
//				rtn.append( "yAxes:[" );
//			}
//			else {
//				// それ以外はx軸の設定(axisの複数形)
//				rtn.append( "xAxes:[" );
//			}
			rtn.append( jsXAxis.getAxisKey() ).append( "]}" );		// 7.0.1.1 (2018/10/22)

			// 6.8.5.0 (2018/01/09) markValues,markColors,markLbls,markAdjust 属性対応
			// 7.0.1.1 (2018/10/22) xmarkValues,xmarkColors属性の追加
			// 7.0.1.3 (2018/11/12) 色指定に、VIVID,PASTEL を使えるようにします。

			final String[] mkVals  = StringUtil.csv2Array( markValues );	// y軸の値で、横のﾏｰｶｰ
			final String[] xmkVals = StringUtil.csv2Array( xmarkValues );	// x軸の値で、縦のﾏｰｶｰ
			final int vCnt = mkVals.length;
			final int xCnt = xmkVals.length;
			if( vCnt > 0 || xCnt > 0 ) {
				rtn.append( ",annotation:{annotations:[" );

				// 従来の markValues,markColors,markLbls,markAdjust 属性対応
				if( vCnt > 0 ) {
//					final String[] mkCols = StringUtil.csv2Array( markColors	, ',' , vCnt , MARK_DEF_COLOR );
					final String[] mkLbls = StringUtil.csv2Array( markLbls		, ',' , vCnt );
					final String[] mkAjst = StringUtil.csv2Array( markAdjust	, ',' , vCnt , MARK_DEF_ADJUST );
					final String[] mkCols = colorCsv( markColors ,  vCnt );							// 7.0.1.3 (2018/11/12)

					// 7.0.1.1 (2018/10/22) 'y-axis-0' → 'y0Ax' これは、JsChartData#getAxisKey() で取得できる値だが、決め打ち
					for( int i=0; i<vCnt; i++ ) {
						rtn.append( "{type:'line',scaleID:'y0Ax',mode:'horizontal'" );
						setProp( rtn, ",borderWidth:"	, markWidth		);
						setProp( rtn, ",borderDash:"	, markDash		);
						setProp( rtn, ",value:"			, mkVals[i]		);
						setProp( rtn, ",borderColor:'"	, mkCols[i] , "'" );
						if( !mkLbls[i].isEmpty() ) {
							rtn.append( ",label:{enabled:'true',position:'left',backgroundColor:'rgba(0,0,0,0)'" );
							setProp( rtn, ",yAdjust:"	, mkAjst[i]		);
							setProp( rtn, ",content:'"	, mkLbls[i] , "'" );
							setProp( rtn, ",fontColor:'", mkCols[i] , "'" );
							setProp( rtn, ",fontSize:"	, markFontSize	);
							rtn.append( '}' );
						}
						rtn.append( "}," );
					}
				}

				// 7.0.1.1 (2018/10/22) xmarkValues,xmarkColors属性対応
				if( xCnt > 0 ) {
//					final String[] xmkCols = StringUtil.csv2Array( xmarkColors	, ',' , xCnt , MARK_DEF_COLOR );
					final String[] xmkCols = colorCsv( xmarkColors ,  xCnt );		// 7.0.1.3 (2018/11/12)

					// 7.0.1.1 (2018/10/22) 'x-axis-0' → 'x0Ax' これは、JsChartData#getAxisKey() で取得できる値だが、決め打ち
					for( int i=0; i<xCnt; i++ ) {
						rtn.append( "{type:'line',scaleID:'x0Ax',mode:'vertical'" );
						setProp( rtn, ",borderWidth:"	, markWidth		);
						setProp( rtn, ",borderDash:"	, markDash		);
						setProp( rtn, ",value:'"		, xmkVals[i] , "'" );		// 横軸はﾗﾍﾞﾙなので、文字列として対応
						setProp( rtn, ",borderColor:'"	, xmkCols[i] , "'" );
						rtn.append( "}," );
					}
				}
				rtn.append( "]}" );
			}

			// 6.8.5.0 (2018/01/09) ズーム処理を使用するかどうか
			if( useZoom ) {
				rtn.append( ",pan:{enabled:true,mode:'xy'},zoom:{enabled:true,drag:false,mode:'xy'}" );
			}
		}
//		setProp( rtn, ",", optionAttributes );
		setProp( rtn, ",", optOptions );			// 7.0.1.2 (2018/11/04)
		rtn.append( '}' );
//		setProp( rtn, ",", chartAttributes );
		setProp( rtn, ",", optChart );				// 7.0.1.2 (2018/11/04)
		rtn.append( "});" ).append( CR );

//		// ｲﾍﾞﾝﾄ設定用 5.9.19.0
//		// 5.9.27.0 (2017/12/01) MODIFY ｲﾍﾞﾝﾄにkeyupを追加
//		// 7.0.1.1 (2018/10/22) 複雑すぎるとの、ﾚｲｱｳﾄが難しいのと、zoomがあるので、廃止予定にします。
//		// widthEventColumn設定
//		if( StringUtil.isNotNull( widthEventColumn ) ) {						// 6.8.5.0 (2018/01/09)
//			rtn.append( "$(document).delegate('#" ).append( widthEventColumn ).append( "','mouseup keyup',function(){" )
//				.append( "var width=$(this).val();" )
//				.append( "$('#" ).append( id ).append( "').attr('width',width);" )
//				.append( myChart ).append( ".chart.width=width;" )
//				.append( myChart ).append( ".update();" )
//				.append( "});" ).append( CR )
//				.append( "$(function(){" )
//				.append( "var chartWidth=$('#" ).append( id ).append("').attr('width');" )
//				.append( "$('#" ).append( widthEventColumn ).append( "').val(chartWidth);" )		// 初期値を設定
//				.append( "});" ).append( CR );
//		}
//		// heightEventColumn設定
//		if( StringUtil.isNotNull( heightEventColumn ) ) {						// 6.8.5.0 (2018/01/09)
//			rtn.append( "$(document).delegate('#" ).append( heightEventColumn ).append( "','mouseup keyup',function(){" )
//				.append( "var height=$(this).val();" )
//				.append( "$('#" ).append( id ).append( "').attr('height',height);" )
//				.append( myChart ).append( ".chart.height=height;" )
//				.append( myChart ).append( ".update();" )
//				.append( "});").append( CR )
//				.append( "$(function(){")
//				.append( "var chartHeight=$('#" ).append( id ).append("').attr('height');" )
//				.append( "$('#" ).append( heightEventColumn ).append( "').val(chartHeight);" )	// 初期値を設定
//				.append( "});" ).append( CR );
//		}
//		// minEventColumn設定
//		if( StringUtil.isNotNull( minEventColumn ) ) {						// 6.8.5.0 (2018/01/09)
//			rtn.append( "$(document).delegate('#" ).append( minEventColumn ).append( "','mouseup keyup',function(){" )
//				// 5.9.27.0 (2017/12/01) MODIFY IntからFloat型に変更
//				.append( "var min=parseFloat($(this).val());")
//				.append( myChart ).append( ".options.scales.yAxes[0].ticks.min = min;" )
//				.append( myChart ).append( ".update();")
//				.append( "});" ).append( CR )
//				.append( "$(function(){" )
//		//		// 7.0.1.1 (2018/10/22) 'y-axis-0' → 'y0Ax' これは、JsChartData#getAxisKey() で取得できる値だが、決め打ち
//				.append( "var chartMax=" ).append( myChart ).append( ".scales['y0Ax'].max;" )
//				.append( "var chartMin=" ).append( myChart ).append( ".scales['y0Ax'].min;" )
//				.append( "$('#" ).append( minEventColumn ).append( "').val(chartMin);" )					// 初期値を設定
//				.append( "$('#" ).append( minEventColumn ).append( "').attr({'max':chartMax});" )			// 初期値を設定
//				.append( "$('#" ).append( minEventColumn ).append( "').attr({'min':chartMin});" )			// 初期値を設定
//				.append( "});" ).append( CR );
//		}
//		// maxEventColumn設定
//		if( StringUtil.isNotNull( maxEventColumn ) ) {						// 6.8.5.0 (2018/01/09)
//			rtn.append( "$(document).delegate('#").append( maxEventColumn ).append( "','mouseup keyup',function(){" )
//				// 5.9.27.0 (2017/12/01) MODIFY IntからFloat型に変更
//				.append( "var max=parseFloat($(this).val());")
//				.append( myChart ).append( ".options.scales.yAxes[0].ticks.max = max;")
//				.append( myChart ).append( ".update();")
//				.append( "});").append( CR )
//				.append( "$(function(){" )
//		//		// 7.0.1.1 (2018/10/22) 'y-axis-0' → 'y0Ax' これは、JsChartData#getAxisKey() で取得できる値だが、決め打ち
//				.append( "var chartMax=" ).append( myChart ).append( ".scales['y0Ax'].max;" )
//				.append( "var chartMin=" ).append( myChart ).append( ".scales['y0Ax'].min;" )
//				.append( "$('#" ).append( maxEventColumn ).append( "').val(chartMax);" )					// 初期値を設定
//				.append( "$('#" ).append( maxEventColumn ).append( "').attr({'max':chartMax});" )			// 初期値を設定
//				.append( "$('#" ).append( maxEventColumn ).append( "').attr({'min':chartMin});" )			// 初期値を設定
//				.append( "});" ).append( CR );
//		}

		// 6.9.2.0 (2018/03/05) ズーム処理を使用する場合、ｸﾀﾞﾌﾞﾙｸﾘｯｸで元に戻すためのｲﾍﾞﾝﾄを発生させます。
		if( useZoom ) {
			rtn.append( "window.onload=function(){$('#").append( id ).append( "').dblclick(function(){window." )
				.append( myChart ).append( ".resetZoom();});}" );
		}

		rtn.append( CR ).append( "</script>" );

		return rtn.toString();
	}

	/**
	 * setに値が存在する場合、sbにstr + setの形で値を追加する。
	 *
	 * @param buf	ベースとなるStringBuilder
	 * @param str	文字列１(必須)
	 * @param set	文字列２(nullかゼロ文字列の場合は、追加しません)
	 */
	private void setProp( final StringBuilder buf, final String str, final String set ) {
		if( StringUtil.isNotNull( set ) ) {						// 6.8.5.0 (2018/01/09)
			buf.append( str ).append( set );
		}
	}

	/**
	 * setに値が存在する場合、sbにstr + set + endの形で値を追加する。
	 *
	 * @param buf	ベースとなるStringBuilder
	 * @param str	文字列１(必須)
	 * @param set	文字列２(nullかゼロ文字列の場合は、追加しません)
	 * @param end	文字列３
	 */
	private void setProp( final StringBuilder buf, final String str, final String set, final String end ) {
		if( StringUtil.isNotNull( set ) ) {						// 6.8.5.0 (2018/01/09)
			buf.append( str ).append( set ).append( end );
		}
	}

	/**
	 * var 変数に設定する配列情報を、bufに追加します。
	 *
	 * var 変数名が key で、cnt分の繰返しで、IntFunction を呼びます。
	 * isQuote=trueの場合は、前後にクォーテーションをつけます。
	 *
	 * @og.rev 7.0.1.3 (2018/11/12) var 変数に設定する配列情報を、bufに追加します。
	 *
	 * @param buf	ベースとなるStringBuilder
	 * @param key	キー
	 * @param cnt	ループする個数(通常は行数：rowCount)
	 * @param isQuote	クォーテーションで括るかどうか [true:括る/false:括らない]
	 * @param func	数値を引数に取る関数型インタフェース
	 */
	private void setVarArray( final StringBuilder buf, final String key, final int cnt, final boolean isQuote, final IntFunction<String> func ) {
		buf.append( " var " ).append( key ).append( "=[" );
		for( int row=0; row<cnt; row++ ) {
			if( row > 0 ) { buf.append( ',' ); }		// カンマは、最初のデータ以降の最初につける。

			final String val = func.apply( row );

			if( isQuote ) {
				buf.append( '"' ).append( val ).append( '"' );
			}
			else {
				buf.append( val );
			}
		}
		buf.append( "];" ).append( CR );
	}

	/**
	 * ﾊﾟﾗﾒｰﾀﾁｪｯｸ用ﾒｿｯﾄﾞ。
	 *
	 * @param trg		ターゲット
	 * @param set		使用可能なキーワードのSet
	 * @param trgStr	ターゲットの名称
	 */
	private void checkPara( final String trg, final Set<String> set, final String trgStr ) {
		if( StringUtil.isNotNull( trg ) && !check( trg, set ) ) {						// 6.8.5.0 (2018/01/09)
			final StringBuilder errMsg = new StringBuilder( BUFFER_MIDDLE )
				.append( "指定の" ).append( trgStr ).append( "は指定できません。" ).append( CR )
				.append( trgStr ).append( "=[" ).append( trg ).append( ']' ).append( CR )
				.append( set );		// org.opengion.fukurou.util.ArraySet の toStringメソッド
			throw new HybsSystemException( errMsg.toString() );
		}
	}

	/**
	 * 色コードの配列を返すメソッドです。
	 *
	 * これは、普通のCSV形式のﾃﾞｰﾀなら、そのまま分割します。
	 * 配列は、lenの数だけ作成します。
	 * nullやゼロ文字列の場合は、ColorMapのOLOR_KEYすべてを対象にします。
	 * 1色の場合も、すべて同じ色をlen の数だけセットします。
	 *
	 * VIVIDとPASTEL はｷｰﾜｰﾄﾞで、org.opengion.fukurou.util.ColorMap のビビッドカラーと
	 * パステルカラーの配列を指定したことと同じになります。
	 * また、色番号として、ビビッドを、(V0～V11) , パステルを、(P0～P11)
	 * に割当てていますので、配列に分解後一旦すべてのｷｰﾜｰﾄﾞを色番号検索に使用します。
	 *
	 * @og.rev 7.0.1.3 (2018/11/12) 色コードの配列を返すメソッド追加
	 *
	 * @param colCsv	色コードのCSV形式文字列
	 * @param len		作成する配列の個数
	 * @return	色コードに変換後の配列
	 */
	private String[] colorCsv( final String colCsv, final int len ) {
		// 色の数を、len にあわせる必要があります。
		final String[] mkCols = new String[len];

		// cols を元に、ColorMap から色配列を取得します。
		final String[] cols = ColorMap.getColorKeys( colCsv );

		// 色配列に順番に割り当てますが、色が足りない場合は、初期値の色をセットします。
		final int min = Math.min( mkCols.length , cols.length );
		for( int i=0; i<min; i++ ) {
			mkCols[i] = cols[i];
		}
		for( int i=min; i<mkCols.length; i++ ) {
			mkCols[i] = cols[0];					// 色コードの最初の色
		}

		return mkCols ;

//		final String[] mkCols ;
//		if( "VIVID".equalsIgnoreCase( cols ) ) {
//			mkCols = new String[len];
//			for( int i=0; i<len; i++ ) {
//				mkCols[i] = ColorMap.getVividKey( i );
//			}
//		}
//		else if( "PASTEL".equalsIgnoreCase( cols ) ) {
//			mkCols = new String[len];
//			for( int i=0; i<len; i++ ) {
//				mkCols[i] = ColorMap.getPastelKey( i );
//			}
//		}
//		else {
//			mkCols = StringUtil.csv2Array( cols , ',' , len , MARK_DEF_COLOR );
//
//			// 無条件で回します。
//			for( int i=0; i<len; i++ ) {
//				mkCols[i] = ColorMap.getColorKey( mkCols[i],mkCols[i] );	// 自身で検索して無ければ自身を初期値で戻してます。
//			}
//		}
//
//		return mkCols ;
	}

	/**
	 * jsChartData情報をﾘｽﾄに追加します。
	 *
	 * @og.rev 6.7.5.0 (2017/03/10) ﾘｽﾄの初期化方法を変更します。
	 *
	 * @param jsData jsChartData情報
	 */
	protected void addJsChartData( final JsChartData jsData ) {
		jsChartData.add( jsData );
	}

	/**
	 * 登録済みのjsChartData情報の個数を返します。
	 *
	 * @og.rev 6.7.7.0 (2017/03/31) 新規登録
	 *
	 * @return 登録済みのjsChartData情報の個数
	 */
	protected int getJsChartDataSize() {
		return jsChartData.size();
	}

	/**
	 * borderColorとbackgroundColorに色を１色しか使用できないかどうかを返します。
	 *
	 * chartType に応じて、色配列が使用できないタイプがあります。
	 *    line/radar が true  （１色しか使用できない）
	 *    それ以外（bar/horizontalBar/polarArea/pie/doughnut）が false （色配列が使用できる）
	 *
	 * @og.rev 7.0.1.1 (2018/10/22) 新規登録
	 *
	 * @return 登録済みのjsChartData情報の個数
	 */
	protected boolean isOneColor() {
		// line/radar が true  （１色しか使用できない）
		return CTYPE_LINE.equalsIgnoreCase( chartType ) || CTYPE_RADAR.equalsIgnoreCase( chartType );
	}

	/**
	 * 【TAG】ﾁｬｰﾄの種類を指定します[line/bar/horizontalBar/radar/polarArea/pie/doughnut](必須)。
	 *
	 * なお、複合グラフ時には、この値を、"bar" にしておかないと、きちんと表示しないようです。
	 * @og.tag
	 *
	 * @param cType ﾁｬｰﾄﾀｲﾌﾟ [line/bar/horizontalBar/radar/polarArea/pie/doughnut]
	 */
	public void setChartType( final String cType ) {
		chartType = nval( getRequestParameter( cType ) , null );

		if( !check( chartType, CTYPE_SET ) ) {
			final StringBuilder errMsg = new StringBuilder( BUFFER_MIDDLE )
				.append( "指定のチャートタイプは実行できません。"        ).append( CR )
				.append( "chartType=[" ).append( chartType ).append( ']' ).append( CR )
				.append( CTYPE_SET );	// org.opengion.fukurou.util.ArraySet の toStringメソッド
			throw new HybsSystemException( errMsg.toString() );
		}
	}

	/**
	 * 【TAG】ﾗﾍﾞﾙのｶﾗﾑ名を指定します(表示名称）(必須)。
	 *
	 * @og.tag
	 *
	 * @param lblclm ﾗﾍﾞﾙｶﾗﾑ
	 */
	public void setLabelColumn( final String lblclm ) {
		labelColumn = nval( getRequestParameter( lblclm ),labelColumn );
	}

	/**
	 * 【TAG】canvasﾀｸﾞのidを指定します(初期値:hybscanvas)。
	 *
	 * @og.tag
	 * canvasﾀｸﾞのidに設定します。
	 *
	 * @param id canvasﾀｸﾞのid
	 */
	@Override
	public void setId( final String id ) {
		this.id = nval( getRequestParameter( id ),this.id );
	}

	/**
	 * 【TAG】ﾁｬｰﾄの高さを指定します(初期値:400)。
	 *
	 * @og.tag
	 * canvasﾀｸﾞの高さに設定します。
	 *
	 * @param hei 設定する高さ
	 */
	public void setHeight( final String hei ) {
		height = nval( getRequestParameter( hei ),height );
	}

	/**
	 * 【TAG】ﾁｬｰﾄの幅を指定します(初期値:400)。
	 *
	 * @og.tag
	 * canvasﾀｸﾞの横幅を設定します。
	 *
	 * @param wid 設定する横幅
	 */
	public void setWidth( final String wid ) {
		width = nval( getRequestParameter( wid ),width );
	}

	/**
	 * 【TAG】ﾀｲﾄﾙを指定します。
	 *
	 * @og.tag
	 *
	 * @og.rev 6.9.9.3 (2018/09/25) nvalを入れて、初期値を設定します。
	 *
	 * @param title ﾀｲﾄﾙ
	 */
	public void setTitle( final String title ) {
		this.title = nval( getRequestParameter( title ),this.title );
	}

	/**
	 * 【TAG】ﾀｲﾄﾙの表示位置[top/right/bottom/left]を指定します(初期値:top)。
	 *
	 * @og.tag
	 *
	 * @og.rev 6.9.9.3 (2018/09/25) nvalを入れて、初期値を設定します。
	 *
	 * @param titlePosition ﾀｲﾄﾙの表示位置 [top/right/bottom/left]
	 */
	public void setTitlePosition( final String titlePosition ) {
		this.titlePosition = nval( getRequestParameter( titlePosition ),this.titlePosition );

		checkPara( this.titlePosition, SET_POSITION, "titlePosition" );
	}


	/**
	 * 【TAG】凡例を表示するか[true/false]を指定します。
	 *
	 * @og.tag
	 *
	 * @og.rev 6.9.9.3 (2018/09/25) nvalを入れて、初期値を設定します。
	 *
	 * @param display 凡例を表示するか [true/false]
	 */
	public void setLegendDisplay( final String display ) {
		legendDisplay = nval( getRequestParameter( display ),legendDisplay );

		if( legendDisplay != null ) {
			checkPara( legendDisplay, SET_BOOLEAN, "legendDisplay" );
			useLegend = true;
		}
	}

	/**
	 * 【TAG】凡例の表示位置[top/right/bottom/left]を指定します(初期値:top)。
	 *
	 * @og.tag
	 *
	 * @og.rev 6.9.9.3 (2018/09/25) nvalを入れて、初期値を設定します。
	 *
	 * @param position 凡例の表示位置 [top/right/bottom/left]
	 */
	public void setLegendPosition( final String position ) {
		legendPosition = nval( getRequestParameter( position ),legendPosition );

		if( legendPosition != null ) {
			checkPara( legendPosition, SET_POSITION, "legendPosition" );
			useLegend = true;
		}
	}

	/**
	 * 【TAG】凡例のスタイル属性を使用するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 凡例のスタイルを、jsChartDataタグのpointStyle属性で指定した形状に変更します。
	 * 複数データの場合、片方だけ指定したい場合は、usePointStyle="true" にしておき、
	 * 指定したいほうだけ、jsChartDataタグ側で、pointStyle属性を設定してください。
	 * options:legend:labels属性のusePointStyle です。
	 *
	 * @og.rev 6.8.5.0 (2018/01/09) 新規登録
	 *
	 * @param usePstyle 凡例のスタイル属性を使用するかどうか [true:使用する/false:使用しない]
	 */
	public void setUsePointStyle( final String usePstyle ) {
		final String useStyle = nval( getRequestParameter( usePstyle ),null );

		if( useStyle != null ) {
			usePointStyle = Boolean.parseBoolean( useStyle );
			useLegend = true;		// パラメータの設定が行われた場合のみ、設定します。
		}
	}

	/**
	 * 【TAG】棒線の横幅を指定します(初期値:0.8, typeがbar,horizontalBarの場合に有効)。
	 *
	 * @og.tag
	 *
	 * @og.rev 6.9.9.3 (2018/09/25) nvalを入れて、初期値を設定します。
	 *
	 * @param barWidthPer 棒線の横幅
	 */
	public void setBarWidthPer( final String barWidthPer ) {
		this.barWidthPer = nval( getRequestParameter( barWidthPer ) , this.barWidthPer );
	}

	/**
	 * 【TAG】ﾁｬｰﾄｸﾘｯｸ時のｲﾍﾞﾝﾄを指定します。
	 *
	 * @og.tag
	 * 下記の値が引数として渡されます。
	 *
	 * event:ｲﾍﾞﾝﾄ情報
	 * obj:ｸﾘｯｸされたｵﾌﾞｼﾞｪｸﾄの情報
	 *
	 * @param click チャートクリック時のイベントを指定
	 */
	public void setOnClick( final String click ) {
		onClick = nval( getRequestParameter( click ),onClick );
	}

	/**
	 * 【TAG】プラグイン定義された関数を指定します。
	 *
	 * @og.tag
	 * プラグインは、plugins: [pinFunc], 形式で追加されます。
	 * この属性での指定時は、[]は、不要で、CSV形式の関数名を並べます。
	 * 外部に、var pinFunc = { afterDatasetsDraw: function(chart, options) { ･･･ } };
	 * 形式のプラグインを指定することで、個別に読ませることが可能です。
	 * なお、すべてのチャートに、同一のプラグインを指定する場合は、この属性ではなく、
	 * Chart.plugins.register({ afterDatasetsDraw: function(chart, options) { ･･･ } });
	 * 形式で、プラグイン登録
	 *
	 * @og.rev 6.9.9.2 (2018/09/18) プラグイン定義された関数を指定します。
	 *
	 * @param attri 追加属性の値
	 */
	public void setPlugins( final String attri ) {
		plugins = nval( getRequestParameter( attri ),plugins );
	}

	//========================================================================================

	/**
	 * 【TAG】x軸の表示位置[top/right/bottom/left]を指定します(初期値:bottom)。
	 *
	 * @og.tag
	 * horizontalBar を指定した場合は、left になります。
	 * 初期値(null)は、下(bottom)に表示されます。
	 * options:scales:xAxes の 要素の属性です。
	 *
	 * @og.rev 7.0.1.2 (2018/11/04) 新規登録
	 *
	 * @param pos x軸の表示位置 [top/right/bottom/left]
	 */
	public void setXposition( final String pos ) {
		xposition = nval( getRequestParameter( pos ),null );

		checkPara( xposition, SET_POSITION, "position" );
	}

	/**
	 * 【TAG】x軸のｽｹｰﾙﾀｲﾌﾟ[category/linear/time]を指定します(初期値:category)。
	 *
	 * @og.tag
	 *
	 * @og.rev 6.9.9.3 (2018/09/25) nvalを入れて、初期値を設定します。
	 * @og.rev 7.0.1.1 (2018/10/22) JsChartDataｵﾌﾞｼﾞｪｸﾄを使用。
	 *
	 * @param xscaleType x軸のｽｹｰﾙﾀｲﾌﾟ [category/linear/time]
	 */
	public void setXscaleType( final String xscaleType ) {
		this.xscaleType = nval( getRequestParameter( xscaleType ) , this.xscaleType );

		// プラグインなどで独自の type を指定することがあるため、警告だけにします。
		try {
			checkPara( this.xscaleType, SET_XSCALE, "xscaleType" );
		}
		catch( final HybsSystemException ex ) {
			System.err.println( ex.getMessage() );
		}

		jsXAxis.addAxis( "type" , this.xscaleType , USE_QUOTE );		// 文字
	}

	/**
	 * 【TAG】x軸のラベルを指定します。
	 *
	 * @og.tag
	 *
	 * @og.rev 6.9.9.3 (2018/09/25) nvalを入れて、初期値を設定します。
	 * @og.rev 7.0.1.1 (2018/10/22) JsChartDataｵﾌﾞｼﾞｪｸﾄを使用。
	 *
	 * @param xlabel x軸のラベル
	 */
	public void setXlabel( final String xlabel ) {
		final String lbl = nval( getRequestParameter( xlabel ),null );
		if( lbl != null ) {
			final String scLbl = "{display: true,labelString:'" + lbl + "'}" ;
			jsXAxis.addAxis( "scaleLabel" , scLbl , NO_QUOTE );		// カンマが不要なのは判っている
		}
	}

	/**
	 * 【TAG】x軸ｺｰﾙﾊﾞｯｸを指定します。
	 *
	 * @og.tag
	 * x軸のメモリ編集用スケールバックを設定します。
	 *
	 * @og.rev 6.9.9.3 (2018/09/25) nvalを入れて、初期値を設定します。
	 * @og.rev 7.0.1.1 (2018/10/22) JsChartDataｵﾌﾞｼﾞｪｸﾄを使用。
	 *
	 * @param callback x軸ｺｰﾙﾊﾞｯｸ
	 */
	public void setXscaleCallback( final String callback ) {
		jsXAxis.addTicks( "callback" , nval( getRequestParameter( callback ),null ) , NO_QUOTE );	// ファンクションは、クオートしない
	}

	/**
	 * 【TAG】x軸を０から書き始まるかどうか(初期値:null)。
	 *
	 * @og.tag
	 *
	 * @og.rev 6.8.5.0 (2018/01/09) 新規登録
	 * @og.rev 7.0.1.1 (2018/10/22) JsChartDataｵﾌﾞｼﾞｪｸﾄを使用。
	 *
	 * @param xZero x軸のゼロ開始
	 */
	public void setXbeginAtZero( final String xZero ) {
		jsXAxis.addTicks( "beginAtZero" , nval( getRequestParameter( xZero ) , null ) , NO_QUOTE );		// Boolean
	}

	/**
	 * 【TAG】x軸の最大値を指定します(xscaleTypeがlinearの場合に有効)。
	 *
	 * @og.tag
	 *
	 * @og.rev 6.9.9.3 (2018/09/25) nvalを入れて、初期値を設定します。
	 * @og.rev 7.0.1.1 (2018/10/22) JsChartDataｵﾌﾞｼﾞｪｸﾄを使用。
	 *
	 * @param xmax x軸の最大値
	 */
	public void setXmax( final String xmax ) {
		jsXAxis.addTicks( "max" , nval( getRequestParameter( xmax ),null ) , NO_QUOTE );		// 数値
	}

	/**
	 * 【TAG】x軸の最小値を指定します(xscaleTypeがlinearの場合に有効)。
	 *
	 * @og.tag
	 *
	 * @og.rev 6.9.9.3 (2018/09/25) nvalを入れて、初期値を設定します。
	 * @og.rev 7.0.1.1 (2018/10/22) JsChartDataｵﾌﾞｼﾞｪｸﾄを使用。
	 *
	 * @param xmin x軸の最小値
	 */
	public void setXmin( final String xmin ) {
		jsXAxis.addTicks( "min" , nval( getRequestParameter( xmin ),null ) , NO_QUOTE );		// 数値
	}

	/**
	 * 【TAG】x軸のメモリ幅を指定します(xscaleTypeがlinearの場合に有効)。
	 *
	 * @og.tag
	 *
	 * @og.rev 6.9.9.3 (2018/09/25) nvalを入れて、初期値を設定します。
	 * @og.rev 7.0.1.1 (2018/10/22) JsChartDataｵﾌﾞｼﾞｪｸﾄを使用。
	 *
	 * @param xstepSize x軸のメモリ幅
	 */
	public void setXstepSize( final String xstepSize ) {
		jsXAxis.addTicks( "stepSize" , nval( getRequestParameter( xstepSize ),null ) , NO_QUOTE );		// 数値
	}

	/**
	 * 【TAG】chartの属性に、TLDで未定義の属性を追加指定します。
	 *
	 * @og.tag
	 * chartの属性に、TLDで未定義の属性を追加指定します。
	 * これは、TLDで未定義の属性を、chart.js で使用する場合に、引数の属性をそのまま、追加します。
	 *
	 * @og.rev 7.0.1.2 (2018/11/04) 属性名変更
	 *
	 * @param attri 追加属性の値
	 */
	public void setOptChart( final String attri ) {
		optChart = nval( getRequestParameter( attri ),optChart );
	}

	/**
	 * 【TAG】optionsの属性に、その他ｵﾌﾟｼｮﾝを追加指定します。
	 *
	 * @og.tag
	 * optionsの属性に、その他ｵﾌﾟｼｮﾝを追加指定します。
	 *
	 * @og.rev 7.0.1.2 (2018/11/04) 属性名変更
	 *
	 * @param attri オプションの値
	 */
	public void setOptOptions( final String attri ) {
		optOptions = nval( getRequestParameter( attri ),optOptions );
	}

	/**
	 * 【TAG】その他options:scales:xAxesのｵﾌﾟｼｮﾝを指定します。
	 *
	 * @og.tag
	 * options:scales:xAxes の 要素の属性です。
	 *  ※ chartJS上は、Axes(axisの複数形)と、Axis を使い分けていますが、属性は、axis で統一します。
	 *
	 * @og.rev 7.0.1.2 (2018/11/04) 属性の追加。
	 *
	 * @param attri その他options:scales:xAxesのｵﾌﾟｼｮﾝ
	 */
	public void setOptAxis( final String attri ) {
		jsXAxis.addOptions( JsChartData.AXIS , nval( getRequestParameter( attri ),null ) );
	}

	/**
	 * 【TAG】その他options:scales:xAxes:ticksのｵﾌﾟｼｮﾝを指定します。
	 *
	 * @og.tag
	 * options:scales:xAxes:ticks の 要素の属性です。
	 *
	 * @og.rev 7.0.1.2 (2018/11/04) 属性の追加。
	 *
	 * @param attri その他options:scales:xAxes:ticksのｵﾌﾟｼｮﾝ
	 */
	public void setOptTicks( final String attri ) {
		jsXAxis.addOptions( JsChartData.TICKS , nval( getRequestParameter( attri ),null ) );
	}

	/**
	 * 【TAG】その他options:scales:xAxes:scaleLabelのｵﾌﾟｼｮﾝを指定します。
	 *
	 * @og.tag
	 * options:scales:xAxes:scaleLabel の 要素の属性です。
	 *
	 * @og.rev 7.0.1.2 (2018/11/04) 属性の追加。
	 *
	 * @param attri その他options:scales:xAxes:scaleLabelのｵﾌﾟｼｮﾝ
	 */
	public void setOptScaleLabel( final String attri ) {
		jsXAxis.addOptions( JsChartData.SCALE_LABEL , nval( getRequestParameter( attri ),null ) );
	}

	/**
	 * 【TAG】その他options:scales:xAxes:gridLinesのｵﾌﾟｼｮﾝを指定します。
	 *
	 * @og.tag
	 * options:scales:xAxes:gridLines の 要素の属性です。
	 *
	 * @og.rev 7.0.1.2 (2018/11/04) 属性の追加。
	 *
	 * @param attri その他options:scales:xAxes:gridLinesのｵﾌﾟｼｮﾝ
	 */
	public void setOptGridLines( final String attri ) {
		jsXAxis.addOptions( JsChartData.GRID_LINES , nval( getRequestParameter( attri ),null ) );
	}

	/**
	 * 【TAG】x軸のﾀｲﾑの単位[year/quarter/month/week/day/hour/minute/second]を指定します。
	 *
	 * @og.tag
	 * (xscaleTypeがtimeの場合に有効。指定しない場合は自動)
	 *
	 * @og.rev 6.9.9.3 (2018/09/25) xscaleType の time 属性廃止。
	 * @og.rev 6.9.9.4 (2018/10/01) nvalを入れて、属性復活。
	 * @og.rev 7.0.1.1 (2018/10/22) JsChartDataｵﾌﾞｼﾞｪｸﾄを使用。
	 *
	 * @param tunit x軸のﾀｲﾑの単位 [year/quarter/month/week/day/hour/minute/second]
	 */
	public void setTimeUnit( final String tunit ) {
		final String timeUnit = nval( getRequestParameter( tunit ),null );

		checkPara( timeUnit, SET_TIMEUNIT, "timeUnit" );

		jsXAxis.addTime( "unit" , timeUnit , USE_QUOTE );	// 文字列
	}

	/**
	 * 【TAG】x軸のﾀｲﾑの最大値を指定します(xscaleTypeがtimeの場合に有効)。
	 *
	 * @og.tag
	 *
	 * @param tmax x軸のﾀｲﾑの最大値
	 * @og.rev 6.9.9.3 (2018/09/25) xscaleType の time 属性廃止。
	 * @og.rev 6.9.9.4 (2018/10/01) nvalを入れて、属性復活。
	 * @og.rev 7.0.1.1 (2018/10/22) JsChartDataｵﾌﾞｼﾞｪｸﾄを使用。
	 */
	public void setTimeMax( final String tmax ) {
		jsXAxis.addTime( "max" , nval( getRequestParameter( tmax ),null ) , USE_QUOTE );	// 時間の最大、最小は、文字列として扱う
	}

	/**
	 * 【TAG】x軸のﾀｲﾑの最小値を指定します(xscaleTypeがtimeの場合に有効)。
	 *
	 * @og.tag
	 * @og.rev 6.9.9.3 (2018/09/25) xscaleType の time 属性廃止。
	 * @og.rev 6.9.9.4 (2018/10/01) nvalを入れて、属性復活。
	 * @og.rev 7.0.1.1 (2018/10/22) JsChartDataｵﾌﾞｼﾞｪｸﾄを使用。
	 *
	 * @param tmin x軸のﾀｲﾑの最小値
	 */
	public void setTimeMin( final String tmin ) {
		jsXAxis.addTime( "min" , nval( getRequestParameter( tmin ),null ) , USE_QUOTE );	// 時間の最大、最小は、文字列として扱う
	}

	/**
	 * 【TAG】x軸のﾀｲﾑの単位幅を指定します(xscaleTypeがtimeの場合に有効)。
	 *
	 * @og.tag
	 *
	 * @og.rev 6.9.9.3 (2018/09/25) xscaleType の time 属性廃止。
	 * @og.rev 6.9.9.4 (2018/10/01) nvalを入れて、属性復活。
	 * @og.rev 7.0.1.1 (2018/10/22) JsChartDataｵﾌﾞｼﾞｪｸﾄを使用。
	 *
	 * @param tunitStepSize x軸のﾀｲﾑの単位幅
	 */
	public void setTimeUnitStepSize( final String tunitStepSize ) {
		jsXAxis.addTime( "unitStepSize" , nval( getRequestParameter( tunitStepSize ),null ) , NO_QUOTE );	// 数値
	}

	/**
	 * 【TAG】x軸の設定するﾀｲﾑ(入力ﾃﾞｰﾀ)のﾌｫｰﾏｯﾄを指定します(xscaleTypeがtimeの場合に有効)。
	 *
	 * @og.tag
	 * ﾌｫｰﾏｯﾄは、Moment.js の定義を使用します。<a href="http://momentjs.com/docs/#/parsing/">Moment.js Documentation</a>
	 * 例：YYYY/MM/DD HH:mm:ss
	 *
	 * <table class="plain">
	 *   <caption>ボタンのタイプ説明</caption>
	 *	 <tr><th>Input		</th><th>Example		</th><th>Description																	</th></tr>
	 *	 <tr><td>YYYY		</td><td>2014 			</td><td>4 or 2 digit year																</td></tr>
	 *	 <tr><td>YY 		</td><td>14				</td><td>2 digit year 																	</td></tr>
	 *	 <tr><td>Y			</td><td>-25			</td><td>Year with any number of digits and sign										</td></tr>
	 *	 <tr><td>Q			</td><td>1..4 			</td><td>Quarter of year. Sets month to first month in quarter.							</td></tr>
	 *	 <tr><td>M MM		</td><td>1..12			</td><td>Month number 																	</td></tr>
	 *	 <tr><td>MMM MMMM	</td><td>Jan..December	</td><td>Month name in locale set by moment.locale()									</td></tr>
	 *	 <tr><td>D DD		</td><td>1..31			</td><td>Day of month 																	</td></tr>
	 *	 <tr><td>Do 		</td><td>1st..31st		</td><td>Day of month with ordinal														</td></tr>
	 *	 <tr><td>DDD DDDD	</td><td>1..365			</td><td>Day of year																	</td></tr>
	 *	 <tr><td>X			</td><td>1410715641		</td><td>Unix timestamp																	</td></tr>
	 *	 <tr><td>x			</td><td>1.41072E+12	</td><td>Unix ms timestamp																</td></tr>
	 *	 <tr><td>			</td><td> 				</td><td> 																				</td></tr>
	 *	 <tr><td>gggg		</td><td>2014 			</td><td>Locale 4 digit week year 														</td></tr>
	 *	 <tr><td>gg 		</td><td>14				</td><td>Locale 2 digit week year 														</td></tr>
	 *	 <tr><td>w ww		</td><td>1..53			</td><td>Locale week of year															</td></tr>
	 *	 <tr><td>e			</td><td>0..6 			</td><td>Locale day of week																</td></tr>
	 *	 <tr><td>ddd dddd	</td><td>Mon...Sunday	</td><td>Day name in locale set by moment.locale()										</td></tr>
	 *	 <tr><td>GGGG		</td><td>2014 			</td><td>ISO 4 digit week year															</td></tr>
	 *	 <tr><td>GG 		</td><td>14				</td><td>ISO 2 digit week year															</td></tr>
	 *	 <tr><td>W WW		</td><td>1..53			</td><td>ISO week of year 																</td></tr>
	 *	 <tr><td>E			</td><td>1..7 			</td><td>ISO day of week																</td></tr>
	 *	 <tr><td>			</td><td> 				</td><td> 																				</td></tr>
	 *	 <tr><td>H HH		</td><td>0..23			</td><td>Hours (24 hour time) 															</td></tr>
	 *	 <tr><td>h hh		</td><td>1..12			</td><td>Hours (12 hour time used with a A.)											</td></tr>
	 *	 <tr><td>k kk		</td><td>1..24			</td><td>Hours (24 hour time from 1 to 24)												</td></tr>
	 *	 <tr><td>a A		</td><td>am pm			</td><td>Post or ante meridiem (Note the one character a p are also considered valid) 	</td></tr>
	 *	 <tr><td>m mm		</td><td>0..59			</td><td>Minutes																		</td></tr>
	 *	 <tr><td>s ss		</td><td>0..59			</td><td>Seconds																		</td></tr>
	 *	 <tr><td>S SS SSS	</td><td>0..999			</td><td>Fractional seconds																</td></tr>
	 *	 <tr><td>Z ZZ		</td><td>+12:00			</td><td>Offset from UTC as +-HH:mm, +-HHmm, or Z 										</td></tr>
	 * </table>
	 *
	 * @og.rev 6.9.9.3 (2018/09/25) xscaleType の time 属性廃止。
	 * @og.rev 6.9.9.4 (2018/10/01) nvalを入れて、属性復活。
	 * @og.rev 7.0.1.1 (2018/10/22) JsChartDataｵﾌﾞｼﾞｪｸﾄを使用。
	 *
	 * @param tFormat x軸の設定するﾀｲﾑのﾌｫｰﾏｯﾄ
	 */
	public void setTimeSetFormat( final String tFormat ) {
		jsXAxis.addTime( "format" , nval( getRequestParameter( tFormat ),null ) , USE_QUOTE );	// 文字列
	}

	/**
	 * 【TAG】x軸の表示するﾀｲﾑのﾌｫｰﾏｯﾄを指定します(xscaleTypeがtimeの場合に有効)。
	 *
	 * @og.tag
	 * ﾌｫｰﾏｯﾄは、Moment.js の定義を使用します。<a href="http://momentjs.com/docs/#/parsing/">Moment.js Documentation</a>
	 * 例：YYYY/MM/DD HH:mm:ss
	 *
	 * timeLblFormatが指定されている場合、全てのdisplayFormatsにtimeLblFormatを設定する
	 *
	 * @og.rev 6.9.9.3 (2018/09/25) xscaleType の time 属性廃止。
	 * @og.rev 6.9.9.4 (2018/10/01) nvalを入れて、属性復活。
	 * @og.rev 7.0.1.1 (2018/10/22) JsChartDataｵﾌﾞｼﾞｪｸﾄを使用。
	 *
	 * @param tLblFormat x軸の表示するﾀｲﾑのﾌｫｰﾏｯﾄ
	 * @see		#setTimeSetFormat(String)
	 */
	public void setTimeLblFormat( final String tLblFormat ) {
		final String timeFmt = nval( getRequestParameter( tLblFormat ),null );
		if( timeFmt != null ) {
			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
					.append(  "{year:'"		).append( timeFmt )
					.append( "',quarter:'"	).append( timeFmt )
					.append( "',month:'"	).append( timeFmt )
					.append( "',week:'"		).append( timeFmt )
					.append( "',day:'"		).append( timeFmt )
					.append( "',hour:'"		).append( timeFmt )
					.append( "',minute:'"	).append( timeFmt )
					.append( "',second:'"	).append( timeFmt )
					.append( "'}" );

			jsXAxis.addTime( "displayFormats" , buf.toString() , NO_QUOTE );	// ｵﾌﾞｼﾞｪｸﾄなので、クオートなし
		}
	}

	/**
	 * 【TAG】x軸の時間のﾂｰﾙﾁｯﾌﾟに使用するﾌｫｰﾏｯﾄ(ﾀｲﾑｽｹｰﾙ用)を指定します(xscaleTypeがtimeの場合に有効)。
	 *
	 * @og.tag
	 * ﾌｫｰﾏｯﾄは、Moment.js の定義を使用します。<a href="http://momentjs.com/docs/#/parsing/">Moment.js Documentation</a>
	 * 例：YYYY/MM/DD HH:mm:ss
	 *
	 * @og.rev 7.0.1.0 (2018/10/15) 時間のﾂｰﾙﾁｯﾌﾟに使用するﾌｫｰﾏｯﾄ(ﾀｲﾑｽｹｰﾙ用)
	 * @og.rev 7.0.1.1 (2018/10/22) JsChartDataｵﾌﾞｼﾞｪｸﾄを使用。
	 *
	 * @param tipFormat x軸の表示するﾀｲﾑのﾌｫｰﾏｯﾄ
	 * @see		#setTimeSetFormat(String)
	 */
	public void setTooltipFormat( final String tipFormat ) {
		jsXAxis.addTime( "tooltipFormat" , nval( getRequestParameter( tipFormat ),null ) , USE_QUOTE );	// 文字列
	}

	//========================================================================================

	/**
	 * 【TAG】y軸に横マーカーラインの設定値をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * annotation オプションに値を設定します。
	 * X軸に平行に固定値の線を引きます。線の値を、CSV形式で指定します。
	 * type: 'line',scaleID: 'y0Ax',mode: 'horizontal',borderWidth: 2 固定です。
	 *
	 * @og.rev 6.8.5.0 (2018/01/09) 新規登録
	 *
	 * @param mkVals y軸に横マーカーラインの設定値(CSV形式)
	 */
	public void setMarkValues( final String mkVals ) {
		markValues = nval( getRequestParameter( mkVals ) , markValues );
	}

	/**
	 * 【TAG】y軸に横マーカーラインの色をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * annotation オプションに値を設定します。
	 * X軸に平行に固定値の線を引きます。線の色を、CSV形式で指定します。
	 * markValues が指定されており、markColorsが指定されていない場合は、青色(BLUE)になります。
	 * 色指定に、VIVID,PASTEL を使えるようにします。
	 *
	 * @og.rev 6.8.5.0 (2018/01/09) 新規登録
	 * @og.rev 7.0.1.3 (2018/11/12) 色指定に、VIVID,PASTEL を使えるようにします。
	 *
	 * @param mkCols y軸に横マーカーラインの色(CSV形式)
	 */
	public void setMarkColors( final String mkCols ) {
		markColors = nval( getRequestParameter( mkCols ) , markColors );
	}

	/**
	 * 【TAG】y軸に横マーカーラインのラベルをCSV形式で複数指定します。
	 *
	 * @og.tag
	 * annotations の label 属性 の content 属性に値をセットします。
	 * label 属性は、enabled: 'true',position: 'left',backgroundColor: 'rgba(0,0,0,0)',
	 * fontSize: 10, は固定で、fontColor は、markColors 属性で指定した
	 * y軸に横マーカーラインの色を使用します。
	 * 色指定に、VIVID,PASTEL を使えるようにします。
	 *
	 * @og.rev 6.8.5.0 (2018/01/09) 新規登録
	 *
	 * @param mklbls y軸に横マーカーラインのラベル(CSV形式)
	 */
	public void setMarkLbls( final String mklbls ) {
		markLbls = nval( getRequestParameter( mklbls ) , markLbls );
	}

	/**
	 * 【TAG】y軸に横マーカーラインのラベル表示位置の上下方向を調整します(初期値:-6)。
	 *
	 * @og.tag
	 * annotation オプションに値を設定します。
	 * annotations の label 属性の yAdjust に値をセットします。
	 * これは、ラインに対するラベルの位置を表します。＋で、下側、－で上側に表示します。
	 * 初期値は、－６ で、ラインの上側に来るように調整しています。
	 *
	 * @og.rev 6.8.5.0 (2018/01/09) 新規登録
	 *
	 * @param mkAjst y軸に横マーカーラインのラベル表示位置の上下方向調整
	 */
	public void setMarkAdjust( final String mkAjst ) {
		markAdjust = nval( getRequestParameter( mkAjst ) , markAdjust );
	}

	/**
	 * 【TAG】x軸に縦マーカーラインの設定値をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * annotation オプションに値を設定します。
	 * Y軸に平行に固定値の縦線を引きます。線の値を、CSV形式で指定します。
	 * type: 'line',borderDash:[5,2],scaleID: 'x0Ax',mode:'vertical',borderWidth:0.5 固定です。
	 *
	 * @og.rev 7.0.1.1 (2018/10/22) xmarkValues,xmarkColors属性の追加
	 *
	 * @param mkVals x軸に縦マーカーラインの設定値(CSV形式)
	 */
	public void setXmarkValues( final String mkVals ) {
		xmarkValues = nval( getRequestParameter( mkVals ) , xmarkValues );
	}

	/**
	 * 【TAG】x軸に縦マーカーラインの色をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * annotation オプションに値を設定します。
	 * Y軸に平行に固定値の縦線を引きます。線の色を、CSV形式で指定します。
	 * xmarkValues が指定されており、markColorsが指定されていない場合は、青色(BLUE)になります。
	 *
	 * @og.rev 7.0.1.1 (2018/10/22) xmarkValues,xmarkColors属性の追加
	 * @og.rev 7.0.1.3 (2018/11/12) 色指定に、VIVID,PASTEL を使えるようにします。
	 *
	 * @param mkCols x軸に縦マーカーラインの色(CSV形式)
	 */
	public void setXmarkColors( final String mkCols ) {
		xmarkColors = nval( getRequestParameter( mkCols ) , xmarkColors );
	}

	/**
	 * 【TAG】x軸,y軸全マーカーライン共通のラインの幅を指定します:borderWidth(初期値:2)。
	 *
	 * @og.tag
	 * annotation オプションに値を設定します。
	 * この値は、x軸,y軸関係なく、マーカーラインの順番も関係なく、共通設定になります。
	 *
	 * @og.rev 7.0.1.1 (2018/10/22) markWidth,markDash,markFontSize属性の追加
	 *
	 * @param width マーカーライン共通のラインの幅
	 */
	public void setMarkWidth( final String width ) {
		markWidth = nval( getRequestParameter( width ) , markWidth );
	}

	/**
	 * 【TAG】x軸,y軸全マーカーライン共通のラインに点線を指定([5,2]など)します:borderDash(初期値:null)。
	 *
	 * @og.tag
	 * annotation オプションに値を設定します。
	 * この値は、x軸,y軸関係なく、マーカーラインの順番も関係なく、共通設定になります。
	 * markDash="[5,2]" とすれば、線の長さが5px , 線と線の間が2px になります。
	 *
	 * @og.rev 7.0.1.1 (2018/10/22) markWidth,markDash,markFontSize属性の追加
	 *
	 * @param dash マーカーライン共通のラインの点線の形状
	 */
	public void setMarkDash( final String dash ) {
		markDash = nval( getRequestParameter( dash ) , markDash );
	}

	/**
	 * 【TAG】x軸,y軸全マーカーライン共通のラベルのフォントサイズを指定します:fontSize(初期値:10)。
	 *
	 * @og.tag
	 * annotation オプションに値を設定します。
	 * この値は、x軸,y軸関係なく、マーカーラインの順番も関係なく、共通設定になります。
	 *
	 * @og.rev 7.0.1.1 (2018/10/22) markWidth,markDash,markFontSize属性の追加
	 *
	 * @param size マーカーライン共通のフォントサイズ
	 */
	public void setMarkFontSize( final String size ) {
		markFontSize = nval( getRequestParameter( size ) , markFontSize );
	}

	//========================================================================================

	/**
	 * 【TAG】すべてのﾃﾞｰﾀが０の場合、使用しないかどうか[true:除外する/false:除外しない]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * JSON データを作成して、JsChartに渡しますが、このフラグを true に設定すると
	 * 0 または、null（空文字列）のデータを出力しません。  6.8.3.0 (2017/11/27)
	 * グラフ系で、0 が、ありえない値として設定されている場合に、使用すると、
	 * 出力するデータ量を抑えることが出来ます。
	 *
	 * @og.rev 6.7.7.0 (2017/03/31) useZeroDataOmit属性の追加
	 *
	 * @param useZeroOmit ﾃﾞｰﾀが０の場合の使用可否 [true:除外する/false:除外しない]
	 */
	public void setUseZeroDataOmit( final String useZeroOmit ) {
		useZeroDataOmit = nval( getRequestParameter( useZeroOmit ) , useZeroDataOmit );
	}

	/**
	 * 【TAG】JSON出力で、値出力にレンデラを利用するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * JSONのデータのレンデラー変換を行うかどうか。
	 * 数値部分にはレンデラー変換は行いません。ラベル文字に行います。
	 * 指定しない場合は使用しない(false)です。
	 *
	 * @og.rev 6.7.9.0 (2017/04/28) useRenderer 追加
	 *
	 * @param	usernd レンデラーを利用するかどうか [true/false]
	 */
	public void setUseRenderer( final String usernd ) {
		useRenderer = nval( getRequestParameter( usernd ) , useRenderer );
	}

	/**
	 * 【TAG】検索結果をこのカラムでソートし直します(初期値:null)。
	 *
	 * @og.tag
	 * query で検索した結果を、JsChartで利用する場合、チャート上のソート順と、
	 * リストや、別のチャートでの表示準が異なる場合に、このカラムで、ソートしなおします。
	 * 通常は、labelColumn と同じ値でソートすることで、X軸の順番に表示されます。
	 *
	 * @og.rev 6.8.0.0 (2017/06/02) sortColumn 追加
	 *
	 * @param	sortClm このカラムでソートし直す
	 */
	public void setSortColumn( final String sortClm ) {
		sortColumn = nval( getRequestParameter( sortClm ) , sortColumn );
	}

	/**
	 * 【TAG】値の前後にクオートをはさむかどうか[true/false]指定します(初期値:false)。
	 *
	 * @og.tag
	 * 以前は、yscaleType="category" のときに、値が、文字列のため、クオートで囲う判断をしていました。
	 * その属性は、JsChartDataTag に移ったため、新たなパラメータを用意します。
	 * 将来的に、自動判定にするか、JsChartDataTag から情報を取得するかもしれません。
	 *
	 * @og.rev 7.0.1.1 (2018/10/22) 新規登録
	 *
	 * @param flag 値の前後にクオートをはさむかどうか [true/false]
	 */
	public void setValueQuot( final String flag ) {
		valueQuot = nval( getRequestParameter( flag ),valueQuot );
	}

	/**
	 * 【TAG】ズーム処理を使用するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * annotation オプションにpan と zoom を設定します。
	 * これは、chartjs-plugin-zoom.js を使用します。
	 * 初期値は、false:使用しないです。
	 *
	 * <ul>
	 *   <li>ﾎｲｰﾙでxy軸の拡大、縮小			</li>
	 *   <li>canvasをクリックでzoomリセット	</li>
	 *   <li>ｸﾘｯｸした状態で移動で、ﾊﾟﾝ動作	</li>
	 *   <li>数値(日付)ｽｹｰﾙの方向のみ可能	</li>
	 * </ul>
	 *
	 * @og.rev 6.8.5.0 (2018/01/09) 新規登録
	 *
	 * @param zoom ズーム処理を使用するかどうか [true:使用する/false:使用しない]。
	 */
	public void setUseZoom( final String zoom ) {
		useZoom = nval( getRequestParameter( zoom ) , useZoom );
	}

	/**
	 * 【TAG】TableModelの指定のカラム(CSV形式)をvarの配列変数として出力します。
	 *
	 * @og.tag
	 * これは、指定のカラムのデータをJavaScriptのvar変数定義で配列として出力します。
	 * labelColumn や、JsChartDataTag の chartColumn と同じ方法です。
	 *
	 * 例えば、TableModelを、sortColumn でソートすると、JsChartDataTag の pointBGColor
	 * の配列順も変わりますので、ソートされた状態で出力したいことがあると思います。
	 *
	 * @og.rev 7.0.1.2 (2018/11/04) 新規登録
	 *
	 * @param clms 指定のカラム(CSV形式)をvarの配列変数として出力
	 */
	public void setVarColumns( final String clms ) {
		varColumns = nval( getRequestParameter( clms ) , varColumns );
	}

	//========================================================================================

//	/**
//	 * 【廃止】横幅の動的設定カラムを設定します。
//	 *
//	 * @og.tag
//	 * 7.0.1.1 (2018/10/22) 複雑すぎるとの、ﾚｲｱｳﾄが難しいのと、zoomがあるので、廃止予定にします。
//	 *
//	 * @og.rev 5.9.19.0 (2017/04/07) 追加
//	 * @og.rev 6.9.9.3 (2018/09/25) nvalを入れて、初期値を設定します。
//	 *
//	 * @param widthEventColumn 横幅の動的設定カラム
//	 */
//	public void setWidthEventColumn( final String widthEventColumn ) {
//		this.widthEventColumn = nval( getRequestParameter( widthEventColumn ),this.widthEventColumn );
//	}

//	/**
//	 * 【廃止】縦幅の動的設定カラムを設定します。
//	 *
//	 * @og.tag
//	 * 7.0.1.1 (2018/10/22) 複雑すぎるとの、ﾚｲｱｳﾄが難しいのと、zoomがあるので、廃止予定にします。
//	 *
//	 * @og.rev 5.9.19.0 (2017/04/07) 追加
//	 * @og.rev 6.9.9.3 (2018/09/25) nvalを入れて、初期値を設定します。
//	 *
//	 * @param heightEventColumn 縦幅の動的設定カラム
//	 */
//	public void setHeightEventColumn( final String heightEventColumn ) {
//		this.heightEventColumn = nval( getRequestParameter( heightEventColumn ),this.heightEventColumn );
//	}

//	/**
//	 * 【廃止】minの動的設定カラムを設定します。
//	 *
//	 * @og.tag
//	 * 7.0.1.1 (2018/10/22) 複雑すぎるとの、ﾚｲｱｳﾄが難しいのと、zoomがあるので、廃止予定にします。
//	 *
//	 * @og.rev 5.9.19.0 (2017/04/07) 追加
//	 * @og.rev 6.9.9.3 (2018/09/25) nvalを入れて、初期値を設定します。
//	 *
//	 * @param minEventColumn minの動的設定カラム
//	 */
//	public void setMinEventColumn( final String minEventColumn ) {
//		this.minEventColumn = nval( getRequestParameter( minEventColumn ),this.minEventColumn );
//	}

//	/**
//	 * 【廃止】maxの動的設定カラムを設定します。
//	 *
//	 * @og.tag
//	 * 7.0.1.1 (2018/10/22) 複雑すぎるとの、ﾚｲｱｳﾄが難しいのと、zoomがあるので、廃止予定にします。
//	 *
//	 * @og.rev 5.9.19.0 (2017/04/07) 追加
//	 * @og.rev 6.9.9.3 (2018/09/25) nvalを入れて、初期値を設定します。
//	 *
//	 * @param maxEventColumn maxの動的設定カラム
//	 */
//	public void setMaxEventColumn( final String maxEventColumn ) {
//		this.maxEventColumn = nval( getRequestParameter( maxEventColumn ),this.maxEventColumn );
//	}

//	/**
//	 * 【TAG】optionsの属性に、その他ｵﾌﾟｼｮﾝを追加指定します。
//	 *
//	 * @og.tag
//	 * optionsの属性に、その他ｵﾌﾟｼｮﾝを追加指定します。
//	 *
//	 * @param attri オプションの値
//	 */
//	public void setOptionAttributes( final String attri ) {
//		optionAttributes = nval( getRequestParameter( attri ),optionAttributes );
//	}
//
//	/**
//	 * 【TAG】chartの属性に、TLDで未定義の属性を追加指定します。
//	 *
//	 * @og.tag
//	 * chartの属性に、TLDで未定義の属性を追加指定します。
//	 * これは、TLDで未定義の属性を、chart.js で使用する場合に、引数の属性をそのまま、追加します。
//	 *
//	 * @og.rev 6.9.9.2 (2018/09/18) chartの属性に、TLDで未定義の属性を追加指定します。
//	 *
//	 * @param attri 追加属性の値
//	 */
//	public void setChartAttributes( final String attri ) {
//		chartAttributes = nval( getRequestParameter( attri ),chartAttributes );
//	}

	/**
	 * 【TAG】(通常は使いません)sessionから所得する DBTableModelオブジェクトの ID。
	 *
	 * @og.tag
	 *
	 * @og.rev 6.9.9.3 (2018/09/25) nvalを入れて、初期値を設定します。
	 *
	 * @param tableId テーブルID
	 */
	public void setTableId( final String tableId ) {
		this.tableId = nval( getRequestParameter( tableId ) , this.tableId );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
				.append( "X_AXIS=" ).append( jsXAxis ).append( CR );

		jsChartData.forEach( js -> buf.append( "Y_AXIS=" ).append( jsXAxis ).append( CR ) );

		return ToString.title( this.getClass().getName() )
			.println( "VERSION"				, VERSION				)
			.println( "id"					, id					)
			.println( "tableId"				, tableId				)
			.println( "chartType"			, chartType				)
			.println( "width"				, width					)
			.println( "height"				, height				)
			.println( "barWidthPer"			, barWidthPer			)
			.println( "title"				, title					)
			.println( "titlePosition"		, titlePosition			)
			.println( "legendPosition"		, legendPosition		)
			.println( "legendDisplay"		, legendDisplay			)
			.println( "xscaleType"			, xscaleType			)
	//		.println( "widthEventColumn"	, widthEventColumn		)	// 2017/03/28 追加
	//		.println( "heightEventColumn"	, heightEventColumn		)	// 2017/03/28 追加
	//		.println( "minEventColumn"		, minEventColumn		)	// 2017/03/28 追加
	//		.println( "maxEventColumn"		, maxEventColumn		)	// 2017/03/28 追加
			.println( "optOptions"			, optOptions			)	// 7.0.1.2 (2018/11/04)
			.println( "optChart"			, optChart				)	// 7.0.1.2 (2018/11/04)
	//		.fixForm().toString()
			.fixForm().println()
			.println( buf ).toString();
	}
}
