/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.view;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.fukurou.util.HybsDateUtil;						// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.html.TableFormatter;
import org.opengion.hayabusa.html.ViewStackTableParam;

import java.util.Calendar;
import java.util.Date;
import java.util.List;

/**
 * 積上ガント表示専用のViewFormです。
 * stackParamTagを利用する事でスタックガント用の行を出力する事が可能です。
 * stackParamTagによりstackColumnsが指定された場合は、そのカラム毎にブレークして、
 * stacklink属性により積上げ行の判別が可能なtbody行を出力します。
 * その際、stackColumnsで指定されたカラム以外の[xxx]は処理されません（空白として出力）
 * [xxx]以外で書かれた箇所、例えば&lt;iGantBar&gt;タグの本体部分等は出力されます。
 *
 * ヘッダの表示にはstackHeaderタグを利用します。
 *
 * [エンジン内部積上げを行わない場合]
 * 積上の表示はJavaScriptによってiGantBarタグの箇所に作成されます。
 * 積上げそのものもiGantBarによって出力されるガントを利用してJavaScriptで行っているため、
 * 最大検索行数と表示行数に注意して下さい。
 *
 * [エンジン内部積上げを行う場合]
 * 工数積上げをエンジン内部で行いdivタグとして出力します。
 * その後の描画（位置調整や色等）はJavaScriptで行います。
 * ガント部分は出力されません。
 * スタック部分はbody部分の最後尾に新たにtd作成するため、注意してください。
 * paramタグでの指定で、costColumnが必須です。
 *
 *
 * AbstractViewForm により、setter/getterメソッドのデフォルト実装を提供しています。
 * 各HTMLのタグに必要な setter/getterメソッドのみ，追加定義しています。
 *
 * AbstractViewForm を継承している為,ロケールに応じたラベルを出力させる事が出来ます。
 *
 * @og.rev 5.5.7.0 (2012/10/01) 新規作成
 * @og.rev 5.5.8.3 (2012/11/17) 内部積上げ対応
 * @og.rev 5.6.1.2 (2013/02/22) キャパシティ対応
 * @og.group 画面表示
 *
 * @version  5.0
 * @author	 Takahashi Masakazu
 * @since    JDK5.0,
 */
public class ViewForm_HTMLStackedGanttTable extends ViewForm_HTMLTable	{
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.0.4.0 (2019/05/31)" ;

	/** ボディーフォーマット最大数 初期値:{@value} */
	protected static final int BODYFORMAT_MAX_COUNT = 10;
	/** stack行の判定出力用 {@value} */
	protected static final String STACK_TBODY		= " stackline='true'";
	/** stack行の判定出力用 {@value} */
	protected static final String GANTT_TBODY		= " stackline='false'";
	/** stackのIDﾌﾟﾚﾌｨｯｸｽ {@value} */
	protected static final String STACK_ID_PREFIX	= " id='stack_";
	/** stackの行ﾌﾟﾚﾌｨｯｸｽ {@value} */
	protected static final String STACK_ROW_PREFIX	= " stackrow='";

	/** ヘッダーフォーマット変数 */
	protected TableFormatter 		headerFormat	;
	/** ボディーフォーマット配列変数 */
	protected TableFormatter[]		bodyFormats		;
	/** フッターフォーマット変数 */
	protected TableFormatter		footerFormat	;
	/** ボディーフォーマット数 */
	protected int					bodyFormatsCount;

	// stack,gantt用
	private int[]  stackCols		;

	// 5.5.8.3 (2012/11/17)
	private int[]	costCols		;		// 工数カラム、開始日カラム、終了日カラム
	private boolean	innerStack		= Boolean.parseBoolean( ViewStackTableParam.INNER_STACK_VALUE );
	private boolean	stackHoliday	= Boolean.parseBoolean( ViewStackTableParam.STACK_HOLIDAY_KEY );

	private String[][] calArray				;		// headで作成されたカレンダーデータ			// 6.3.9.1 (2015/11/27) 修飾子を、なし → private に変更。
	private int capCol = -1					;		// 5.6.1.2 (2013/02/22) 能力値カラム		// 6.3.9.1 (2015/11/27) 修飾子を、なし → private に変更。

	/**
	 * DBTableModel から HTML文字列を作成して返します。
	 * startNo(表示開始位置)から、pageSize(表示件数)までのView文字列を作成します。
	 * 表示残りデータが pageSize 以下の場合は,残りのデータをすべて出力します。
	 *
	 *
	 * @og.rev 5.5.8.3 (2012/11/17) 内部積上げ対応
	 * @og.rev 5.6.1.2 (2013/02/22) キャパシティ対応
	 * @og.rev 5.6.2.1 (2013/06/13) 積上不具合修正
	 * @og.rev 6.2.0.0 (2015/02/27) フォーマット系の noDisplay 対応
	 * @og.rev 6.4.2.0 (2016/01/29) HybsDateUtil.getCalendar( String ) を直接利用するように修正します。
	 * @og.rev 6.4.3.4 (2016/03/11) tdに、[カラム]が無いケースで、次の[カラム]のクラス属性が、前方すべてのtdにセットされてしまう対応。
	 * @og.rev 6.4.4.2 (2016/04/01) TableFormatterのタイプ別値取得処理の共通部をまとめる。
	 * @og.rev 6.4.5.0 (2016/04/08) メソッド変更( getColumnDbType(int) → getClassName(int) )
	 * @og.rev 6.8.1.1 (2017/07/22) ckboxTD変数は、&lt;td&gt; から &lt;td に変更します(タグの最後が記述されていない状態でもらう)。
	 * @og.rev 6.8.2.0 (2017/10/13) makeNthChildの廃止と、makeCheckboxで、個別にclass指定するように変更。
	 *
	 * @param  sttNo	  表示開始位置
	 * @param  pgSize   表示件数
	 *
	 * @return	DBTableModelから作成された HTML文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String create( final int sttNo, final int pgSize )  {
		// ガントは、キーブレイクがあるため、全件表示します。
		final int pageSize = getRowCount() ;
		if( pageSize == 0 ) { return ""; }	// 暫定処置

		// 4.3.1.0 (2008/09/08)
		if( headerFormat == null ) {
			final String errMsg = "ViewTagで canUseFormat() = true の場合、Formatter は必須です。";
			throw new HybsSystemException( errMsg );
		}

		headerLine	 = null;		// 3.5.3.1 (2003/10/31) キャッシュクリア

		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		// ガントは、キーブレイクがあるため、全件表示します。
		final int startNo  = 0;

		final int lastNo = getLastNo( startNo, pageSize );
		final int blc = getBackLinkCount();

		// このビューの特有な属性を初期化
		paramInit();

		headerFormat.makeFormat( getDBTableModel() );	// 3.5.6.2 (2004/07/05) 移動
		// 6.2.0.0 (2015/02/27) フォーマット系の noDisplay 対応
		setFormatNoDisplay( headerFormat );

		final StringBuilder out = new StringBuilder( BUFFER_LARGE )
			.append( getCountForm( startNo,pageSize ) )
			.append( getHeader() );

		if( bodyFormatsCount == 0 ) {
			bodyFormats[0] = headerFormat ;
			bodyFormatsCount ++ ;
		}
		else {
			for( int i=0; i<bodyFormatsCount; i++ ) {
				bodyFormats[i].makeFormat( getDBTableModel() );
				// 6.2.0.0 (2015/02/27) フォーマット系の noDisplay 対応
				setFormatNoDisplay( bodyFormats[i] );
			}
		}

		String[] astrOldStackKeys = new String[stackCols.length];
		for( int nIndex=0; nIndex<astrOldStackKeys.length; nIndex++ ) {
			astrOldStackKeys[nIndex] = "";
		}

		int bgClrCnt = 0;
		int stackRow = 0;

		// 5.5.8.3 (2012/11/17)
		double[] costAry = null;
		// 6.3.9.1 (2015/11/27) Found 'DD'-anomaly for variable(PMD)
		final Calendar firstCalday;
		final String   calZoom ;		// 6.3.9.1 (2015/11/27)
		if( innerStack ){
			costAry = new double[calArray.length];
			final String[] firstCal = calArray[0];
			firstCalday = HybsDateUtil.getCalendar(firstCal[0]);							// 6.4.2.0 (2016/01/29)
			final Calendar fstCalEnd = HybsDateUtil.getCalendar(firstCal[2]);				// 6.4.2.0 (2016/01/29)

			if( differenceDays(firstCalday.getTime(),fstCalEnd.getTime()) == 1 ){
				calZoom = ViewStackTableParam.STACK_ZOOM_DAY;
			}
			else if( differenceDays(firstCalday.getTime(),fstCalEnd.getTime()) == 7 ){
				calZoom = ViewStackTableParam.STACK_ZOOM_WEEK;
			}
			else{
				calZoom = ViewStackTableParam.STACK_ZOOM_MONTH;
			}
		}
		else {							// 6.3.9.1 (2015/11/27)
			firstCalday = null;
			calZoom     = null;
		}

		String capacity = null;			// 5.6.1.2 (2013/02/22)
		for( int row=startNo; row<lastNo; row++ ) {
			// データのスキップは行わない

			// ガントのブレイク
		//	if(! isSameGroup(row, astrOldGroupKeys)) {
				// 6.9.7.0 (2018/05/14) PMD These nested if statements could be combined
//				if( !isSameStack(row, astrOldStackKeys) && stackCols.length > 0 ) { // 積上のブレイク
//					if( !(innerStack && row == startNo) ) {							// 5.5.8.3 (2012/11/17) 内部積上げは後から積上げるので、初回は出力しない
				if( !isSameStack(row, astrOldStackKeys) && stackCols.length > 0		// 積上のブレイク
					&& !(innerStack && row == startNo ) ) {							// 5.5.8.3 (2012/11/17) 内部積上げは後から積上げるので、初回は出力しない
						stackRow = row;

						makeBodyTable( out,innerStack ? row -1 : row, stackRow, bgClrCnt, blc, costAry, capacity );			// 6.1.0.0 (2014/12/26) refactoring

						if( innerStack ){
							costAry = new double[calArray.length];
						}
//					}
				}
				// 6.1.0.0 (2014/12/26) refactoring : Avoid if(x != y) ..; else ..;
				if( innerStack ){	// 5.5.8.3 (2012/11/17) 内部積上げをする場合
					final double costDbl = Double.parseDouble( getValue(row,costCols[0]) ); //工数
					final Calendar startDay = HybsDateUtil.getCalendar(getValue(row,costCols[1]));		// 6.4.2.0 (2016/01/29)
					final Calendar endDay   = HybsDateUtil.getCalendar(getValue(row,costCols[2]));		// 6.4.2.0 (2016/01/29)

					final Date startDayDate = startDay.getTime();
					Date endDayDate = endDay.getTime();

					// 5.6.1.2 (2013/02/22)
					if( capCol > -1 ){
						capacity = getValue(row,capCol);
					}
					else{
						capacity = "1";
					}

					// 枠はそのままで計算
					final int fromCell = calNumber(startDayDate,calZoom,firstCalday.getTime());
					final int toCell   = calNumber(endDayDate,calZoom,firstCalday.getTime());

					endDay.add(Calendar.DATE, 1); // 終了日は範囲に入るので１つ進める
					endDayDate = endDay.getTime();

					int stackMother = differenceDays(startDayDate,endDayDate);
					if( !stackHoliday ){
						for( int cel=fromCell; cel<=toCell; cel++  ){
							if( "1".equals( calArray[cel][1] ) ){
								stackMother--;
							}
						}
					}

					Date calFrom;
					Date calTo;

					for( int cel=fromCell; cel<=toCell; cel++ ){
						calFrom = HybsDateUtil.getCalendar(calArray[cel][0]).getTime();			// 6.4.2.0 (2016/01/29)
						calTo   = HybsDateUtil.getCalendar(calArray[cel][2]).getTime();			// 6.4.2.0 (2016/01/29)
						if( calFrom.compareTo( startDayDate ) < 0 ){
							calFrom = startDayDate;
						}
						if( endDayDate.compareTo( calTo ) < 0 ){
							calTo = endDayDate;
						}
						// 6.3.9.1 (2015/11/27) Found 'DD'-anomaly for variable(PMD)
						final int cellDays = differenceDays( calFrom, calTo );
						if( stackHoliday ){
							costAry[cel] += (costDbl / stackMother) * cellDays;
						}
						else{
							// 休日のみの場合は積上げられない！
							if( !"1".equals( calArray[cel][1] ) ){
								costAry[cel] += (costDbl / stackMother) * cellDays;
							}
						}
					}
				}
				else{	// 5.5.8.3 (2012/11/17) 内部積上げの場合はガント部分は出力せずに積上げだけする。
					for( int i=0; i<bodyFormatsCount; i++ ) {
						final TableFormatter bodyFormat = bodyFormats[i];
						if( ! bodyFormat.isUse( row,getDBTableModel() ) ) { continue; }		// 3.5.4.0 (2003/11/25)
						out.append("<tbody").append( getBgColorCycleClass( bgClrCnt++,row ) );
						if( isNoTransition() ) {	// 4.3.3.0 (2008/10/01)
							out.append( getHiddenRowValue( row ) );
						}
						out.append( GANTT_TBODY )
							.append( STACK_ROW_PREFIX ).append( stackRow ).append("'>")
							.append( bodyFormat.getTrTag() );

						// 3.5.5.0 (2004/03/12) No 欄そのものの作成判断追加
						if( isNumberDisplay() ) {
							final String ckboxTD = "<td" + bodyFormat.getRowspan();			// 6.8.1.1 (2017/07/22)
							out.append( makeCheckbox( ckboxTD,row,blc,true ) );				// 6.8.2.0 (2017/10/13)
						}

						int cl = 0;
						for( ; cl<bodyFormat.getLocationSize(); cl++ ) {
							String fmt = bodyFormat.getFormat(cl);
							final int loc = bodyFormat.getLocation(cl);		// 3.5.5.0
							if( ! bodyFormat.isNoClass() && loc >= 0 ) {	// 3.5.5.7 (2004/05/10)
								// 6.4.3.4 (2016/03/11) tdに、[カラム]が無いケースで、次の[カラム]のクラス属性が、前方すべてのtdにセットされてしまう対応。
								final int idx = fmt.lastIndexOf( "<td" );
								if( idx >= 0 ) {	// matchしてるので、あるはず
									final String tdclass = " class=\"" + getClassName(loc) + "\" ";			// 6.4.5.0 (2016/04/08)
									fmt = fmt.substring( 0,idx+3 ) + tdclass + fmt.substring( idx+3 ) ;
								}
							}
							out.append( fmt );			// 3.5.0.0
							// 3.5.5.7 (2004/05/10) #,$ 対応
							if( loc >= 0 ) {
								// 6.4.4.2 (2016/04/01) 処理の共通部をまとめる。
								out.append( getTypeCaseValue( bodyFormat.getType(cl),row,loc ) );
							}
							else {
								out.append( bodyFormat.getSystemFormat(row,loc) );
							}
						}
						out.append( bodyFormat.getFormat(cl) )
							.append("</tbody>").append( CR );
					}
				}
		}

		// 6.0.2.5 (2014/10/31) たぶん、カッコのコメントする位置間違いで使われてないようなので、一旦コメントする。
	//	} // 5.6.5.2 (2013/06/21) 括弧の位置間違いのため修正

		// 内部積上げ時は最終行の出力を行う
		if( innerStack ){
			makeBodyTable( out, lastNo-1, stackRow, bgClrCnt, blc, costAry, capacity );			// 6.1.0.0 (2014/12/26) refactoring
		}

		if( footerFormat != null ) {
			// 6.3.9.0 (2015/11/06) 引数にTableFormatterを渡して、処理の共有化を図る。
			out.append( getTableFoot( footerFormat ) );
		}

		out.append("</table>").append( CR )
			.append( getScrollBarEndDiv() );	// 3.8.0.3 (2005/07/15)

		return out.toString();
	}

	/**
	 * 内容をクリア(初期化)します。
	 *
	 * @og.rev 5.5.8.3 (2012/11/17) 内部積上げのための修正
	 * @og.rev 5.6.1.2 (2013/02/22) キャパシティ対応
	 *
	 */
	@Override
	public void clear() {
		super.clear();
		headerFormat		= null;
		bodyFormats			= null;
		footerFormat		= null;
		bodyFormatsCount	= 0;
		stackCols			= null;			// 5.5.8.3 (2012/11/17)
		costCols			= null;			// 5.5.8.3 (2012/11/17)
		innerStack 			= Boolean.parseBoolean( ViewStackTableParam.INNER_STACK_VALUE ); // 5.5.8.3 (2012/11/17)
		calArray			= null;			// 5.5.8.3 (2012/11/17)
		stackHoliday		= Boolean.parseBoolean( ViewStackTableParam.STACK_HOLIDAY_KEY ); // 5.5.8.3 (2012/11/17)
		capCol				= -1;			// 5.6.1.2 (2013/02/22)
	}

	/**
	 * このビューに対する特別な初期化を行う。
	 *
	 * @og.rev 5.5.8.3 (2012/11/17)
	 * @og.rev 5.5.9.0 (2012/12/03) objectではなくArrayList化
	 * @og.rev 5.6.1.2 (2013/02/22) キャパシティ対応
	 * @og.rev 5.6.2.1 (2013/03/08) stackHolidayが抜けていたので追加
	 */
	private void paramInit() {
		final String costCol		= getParam( ViewStackTableParam.COST_COLUMNS_KEY,	ViewStackTableParam.COST_COLUMNS_VALUE	); // 5.5.8.3 (2012/11/17)
		innerStack					= getBoolParam( ViewStackTableParam.INNER_STACK_KEY	); // 5.5.8.3 (2012/11/17)
		stackHoliday				= getBoolParam( ViewStackTableParam.STACK_HOLIDAY_KEY	); // 5.6.2.1 (2013/03/08)

		if( innerStack ){
			// 6.1.0.0 (2014/12/26) findBugs: Bug type ITA_INEFFICIENT_TO_ARRAY (click for details)
			// 長さが0の配列の引数で Collection.toArray() を使用しています。
			final List<String[]> lst = getViewArrayList();
			calArray = lst.toArray( new String[lst.size()][3] );
			if( calArray == null || costCol == null){
				final String errMsg = "ヘッダのカレンダデータ、costColumnsの設定は必須です。"+costCol;
				throw new HybsSystemException( errMsg );
			}
		}

		final DBTableModel table = getDBTableModel();

		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		final String strStackCols	= getParam( ViewStackTableParam.STACK_COLUMNS_KEY,	ViewStackTableParam.STACK_COLUMNS_VALUE	);
		final String[] stackKeys = StringUtil.csv2Array(strStackCols);
		stackCols = new int[stackKeys.length];
		for( int nIndex=0; nIndex<stackCols.length ; nIndex++ ) {
			stackCols[nIndex] = table.getColumnNo( stackKeys[nIndex] );
		}

		final String[] costKeys = StringUtil.csv2Array(costCol);
		costCols = new int[costKeys.length];
		for( int nIndex=0; nIndex<costCols.length ; nIndex++ ) {
			costCols[nIndex] = table.getColumnNo( costKeys[nIndex] );
		}

		// 5.6.1.2 (2013/02/22) キャパシティ
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		final String capColName		= getParam( ViewStackTableParam.CAP_COLUMN_KEY,	ViewStackTableParam.CAP_COLUMN_VALUE	); // 5.6.1.2 (2013/02/22)
		if( capColName != null && capColName.length() > 0 ){
			capCol = table.getColumnNo(capColName);
		}
	}

	/**
	 * DBTableModel から テーブルのタグ文字列を作成して返します。
	 *
	 * @og.rev 5.9.1.2 (2015/10/23) 自己終了警告対応
	 * @og.rev 6.4.4.1 (2016/03/18) NUMBER_DISPLAYを、static final 定数化します。
	 * @og.rev 6.4.9.0 (2016/07/23) colgroupのHTML5対応(No欄)
	 * @og.rev 6.4.9.1 (2016/08/05) colgroupのHTML5対応(No欄)時の対応ミス修正
	 * @og.rev 6.8.1.0 (2017/07/14) HTML5対応ヘッダー出力設定時に、ブラウザを互換設定したときの対応。
	 * @og.rev 6.8.2.0 (2017/10/13) makeNthChildの廃止と、makeCheckboxで、個別にclass指定するように変更。
	 * @og.rev 7.0.4.0 (2019/05/31) colgroup 廃止
	 *
	 * @return	テーブルのタグ文字列
	 * @og.rtnNotNull
	 */
	@Override
	protected String getTableHead() {
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

//		// 3.5.5.0 (2004/03/12) No 欄そのものの作成判断追加
//		// 7.0.4.0 (2019/05/31) colgroup 廃止
//		if( isNumberDisplay() ) {
//			// 6.4.9.0 (2016/07/23) colgroupのHTML5対応(No欄)
//				buf.append( NUMBER_DISPLAY );		// 6.8.1.0 (2017/07/14) HTML5ネイティブ時でも、出力します。
//			// 6.8.1.0 (2017/07/14) HTML5対応ヘッダー出力設定時に、ブラウザを互換設定したときの対応。
//			// 6.8.2.0 (2017/10/13) makeNthChildの廃止と、makeCheckboxで、個別にclass指定するように変更。
//	//		if( !useIE7Header ) {
//	//			buf.append( "<style type=\"text/css\">" ).append( CR );
//	//			makeNthChild( buf,2,"BIT" );
//	//			makeNthChild( buf,3,"S9"  );
//	//			buf.append( "</style>" ).append( CR );		// 6.4.9.1 (2016/08/05)
//	//		}
//		}

		// 3.5.2.0 (2003/10/20) ヘッダー繰り返し部をgetHeadLine()へ移動
		buf.append("<thead id=\"header\">").append( CR )	// 3.5.6.5 (2004/08/09)
			.append( getHeadLine() )
			.append("</thead>").append( CR );

		return buf.toString();
	}

	/**
	 * ヘッダー繰り返し部を、getTableHead()メソッドから分離。
	 *
	 * @og.rev 6.1.2.0 (2015/01/24) キャッシュを返すのを、#getHeadLine() に移動。
	 *
	 * @return	テーブルのタグ文字列
	 * @og.rtnNotNull
	 */
	@Override
	protected String getHeadLine() {
		if( headerLine == null ) {					// キャッシュになければ、設定する。
			headerLine = getHeadLine( "<th" + headerFormat.getRowspan() ) ;
		}

		return headerLine ;
	}

	/**
	 * ヘッダー繰り返し部を、getTableHead()メソッドから分離。
	 *
	 * @og.rev 6.1.2.0 (2015/01/24) キャッシュを返すのを、#getHeadLine() に移動。
	 * @og.rev 6.4.3.4 (2016/03/11) ヘッダーでもTableFormatterのType(#,$,!)に対応した値を出すようにする。
	 * @og.rev 6.4.4.2 (2016/04/01) TableFormatterのタイプ別値取得処理の共通部をまとめる。
	 *
	 * @param	thTag タグの文字列
	 *
	 * @return	テーブルのタグ文字列
	 * @og.rtnNotNull
	 */
	@Override
	protected String getHeadLine( final String thTag ) {

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
				.append( headerFormat.getTrTag() ).append( CR );

		// 3.5.5.0 (2004/03/12) No 欄そのものの作成判断追加
		if( isNumberDisplay() ) {
			// 6.1.2.0 (2015/01/24) thTag に、headerFormat.getRowspan() を含ませて受け取る。
			if( isUseCheckControl() && "checkbox".equals( getSelectedType() ) ) {
				buf.append( thTag ).append( "></th>" )
					.append( thTag ).append( '>' ).append( getAllCheckControl() ).append( "</th>" )		// 6.0.2.5 (2014/10/31) char を append する。
					.append( thTag ).append( '>' ).append( getNumberHeader()    ).append( "</th>" );	// 6.0.2.5 (2014/10/31) char を append する。
			}
			else {
				buf.append( thTag ).append( " colspan=\"3\">" ).append( getNumberHeader() ).append( "</th>" );	// 6.0.2.5 (2014/10/31) char を append する。
			}
		}

		int cl = 0;
		for( ; cl<headerFormat.getLocationSize(); cl++ ) {
			buf.append( StringUtil.replace( headerFormat.getFormat(cl) ,"td","th" ));
			final int loc = headerFormat.getLocation(cl);
			// 6.4.3.4 (2016/03/11) ヘッダーでもTableFormatterのType(#,$,!)に対応した値を出すようにする。
			if( loc >= 0 ) {
				// 6.4.4.2 (2016/04/01) 処理の共通部をまとめる。
				buf.append( getTypeCaseValue( headerFormat.getType(cl),-1,loc ) );
			}
		}
		buf.append( StringUtil.replace( headerFormat.getFormat(cl) ,"td","th" ) ).append( CR );

		return buf.toString();				// 6.1.2.0 (2015/01/24)
	}

	/**
	 * フォーマットを設定します。
	 *
	 * @param	list	TableFormatterのリスト
	 */
	@Override
	public void setFormatterList( final List<TableFormatter> list ) {		// 4.3.3.6 (2008/11/15) Generics警告対応
		bodyFormats = new TableFormatter[BODYFORMAT_MAX_COUNT];

		bodyFormatsCount = 0;
		// 7.2.9.4 (2020/11/20) PMD:This for loop can be replaced by a foreach loop
		for( final TableFormatter format : list ) {
//		for( int i=0; i<list.size(); i++ ) {
//			final TableFormatter format = list.get( i );		// 4.3.3.6 (2008/11/15) Generics警告対応

			switch( format.getFormatType() ) {
				case TYPE_HEAD : headerFormat = format; break;
				case TYPE_BODY : bodyFormats[bodyFormatsCount++] = format; break;
				case TYPE_FOOT : footerFormat = format; break;
				default : final String errMsg = "FormatterType の定義外の値が指定されました。";
				// 4.3.4.4 (2009/01/01)
						  throw new HybsSystemException( errMsg );
			}
		}

		// 3.5.5.5 (2004/04/23) headerFormat が定義されていない場合はエラー
		if( headerFormat == null ) {
			final String errMsg = "h:thead タグの、フォーマットの指定は必須です。";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * フォーマットメソッドを使用できるかどうかを問い合わせます。
	 *
	 * @return  使用可能(true)/ 使用不可能 (false)
	 */
	@Override
	public boolean canUseFormat() {
		return true;
	}

	/**
	 * ビューで表示したカラムの一覧をCSV形式で返します。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 * @og.rev 6.2.0.1 (2015/03/06) TableFormatter#getLocation(int)の有効判定
	 *
	 * @return	ビューで表示したカラムの一覧
	 * @og.rtnNotNull
	 */
	@Override
	public String getViewClms() {
		final DBTableModel table = getDBTableModel();
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
		for( int i=0; i<headerFormat.getLocationSize(); i++ ) {
			if( buf.length() > 0 ) { buf.append( ',' ); }
			// 6.2.0.1 (2015/03/06) TableFormatter#getLocation(int)の有効判定
			final int loc = headerFormat.getLocation(i);
			if( loc >= 0 ) { buf.append( table.getColumnName( loc ) ); }
		}
		return buf.toString();
	}

	/**
	 * 上下行のデータが同じ積上かどうかをチェックする。
	 *
	 * @param   nRowIndex テーブルモデルの行番号
	 * @param   astrOldValues 古いグルプーデータ配列
	 *
	 * @return  使用可能(true)/ 使用不可能 (false)
	 */
	private boolean isSameStack(final int nRowIndex, final String[] astrOldValues) {
		boolean bRet = stackCols.length > 0 ;
		if( bRet ) {
			for( int nIndex=0; bRet && nIndex<stackCols.length ; nIndex++ ) {
				bRet = astrOldValues[nIndex].equals( getValue( nRowIndex, stackCols[nIndex] ) ) ;
			}

			// 不一致時に astrOldValues に 新しい値を設定しておきます。
			if( !bRet ) {
				for( int nIndex=0; nIndex<stackCols.length; nIndex++ ) {
					astrOldValues[nIndex] = getValue(nRowIndex, stackCols[nIndex]);
				}
			}
		}
		return bRet;
	}

	/**
	 * 対象カラムが積上げカラムかどうか。
	 *
	 * @param   loc 列番号
	 *
	 * @return  対象(true)/ 非対象 (false)
	 */
	private boolean isStackClm(final int loc) {
		boolean rtn = false;
		// 7.2.9.4 (2020/11/20) PMD:This for loop can be replaced by a foreach loop
		for( final int stCol : stackCols ) {
			if( stCol == loc ) {
				rtn = true;
				break;				// 6.3.9.1 (2015/11/27)
			}
		}

//		for( int nIndex=0; nIndex<stackCols.length ; nIndex++ ) {
//			if( stackCols[nIndex] == loc ) {
//				rtn = true;
//				break;				// 6.3.9.1 (2015/11/27)
//			}
//		}
		return rtn;
	}

	/**
	 * 2つの日付の差を求めます。
	 * java.util.Date 型の日付 date1 - date2 が何日かを返します。
	 *
	 * @og.rev 5.5.8.3 (2012/11/17) 新規
	 *
	 * @param date1    日付
	 * @param date2    日付
	 * @return    2つの日付の差(日数 2-1) 同日なら０
	 */
	public static int differenceDays(final Date date1,final Date date2) {
		final long datetime1 = date1.getTime();
		final long datetime2 = date2.getTime();
		final long one_date_time = 1000 * 60 * 60 * 24L;
		return (int)((datetime2 - datetime1) / one_date_time);
	}

	/**
	 * 日付から枠番号を返す。
	 *
	 * @og.rev 5.5.8.3 (2012/11/17) 新規
	 *
	 * @param   date 日付(YYYY/MM/DD)
	 * @param 	zoom Zoom設定値
	 * @param	calFD ヘッダ初日
	 *
	 * @return  枠番号
	 */
	private int calNumber(final Date date, final String zoom, final Date calFD ) {
		// 6.3.9.1 (2015/11/27) Found 'DD'-anomaly for variable(PMD)
		final int rtn ;
		if( zoom.equals( ViewStackTableParam.STACK_ZOOM_MONTH ) ){
			// 月だけは別の計算が必要
			final Calendar cal1 = Calendar.getInstance();
			cal1.setTime( calFD );
			final Calendar cal2 = Calendar.getInstance();
			cal2.setTime( date );
			rtn = ( cal2.get( Calendar.YEAR )-cal1.get( Calendar.YEAR ) ) * 12
					+ ( cal2.get( Calendar.MONTH ) - cal1.get( Calendar.MONTH ) );
		}
		else{
			final int diff = differenceDays( calFD, date );
			if( zoom.equals( ViewStackTableParam.STACK_ZOOM_WEEK )){
				rtn = diff/7;
			}
			else{
				rtn = diff;
			}
		}
		return rtn;
	}

	/**
	 * テーブル本体の作成。
	 *
	 * @og.rev 5.5.8.3 (2012/11/17) 繰り返し利用するため分離
	 * @og.reb 5.6.1.2 (2013/02/22) td終了が抜けていたので追加、キャパシティ対応
	 * @og.reb 6.1.0.0 (2014/12/26) 引数に StringBuffer を追加し、それに、データを追記していく。
	 * @og.rev 6.4.3.4 (2016/03/11) tdに、[カラム]が無いケースで、次の[カラム]のクラス属性が、前方すべてのtdにセットされてしまう対応。
	 * @og.rev 6.4.4.2 (2016/04/01) TableFormatterのタイプ別値取得処理の共通部をまとめる。
	 * @og.rev 6.4.5.0 (2016/04/08) メソッド変更( getColumnDbType(int) → getClassName(int) )
	 * @og.rev 6.8.1.1 (2017/07/22) ckboxTD変数は、&lt;td&gt; から &lt;td に変更します(タグの最後が記述されていない状態でもらう)。
	 * @og.rev 6.8.2.0 (2017/10/13) makeNthChildの廃止と、makeCheckboxで、個別にclass指定するように変更。
	 *
	 * @param   outBuf	データを追加するStringBuffer
	 * @param   row		テーブルモデルのrow
	 * @param 	stackRow スタック行保存用
	 * @param	bgClrCnt 背景色カウンタ
	 * @param	blc		チェックボックス用
	 * @param	costAry コスト集計配列
	 * @param	cap		能力
	 *
	 * @return  テーブル本体のHTML(入力の out オブジェクトそのもの)
	 * @og.rtnNotNull
	 */
	private StringBuilder makeBodyTable( final StringBuilder outBuf, final int row, final int stackRow,
											 final int bgClrCnt, final int blc, final double[] costAry, final String cap ) {
		int bcCnt = bgClrCnt;		// 6.0.0.1 (2014/04/25) 引数を直接変更できなくする。
		for( int i=0; i<bodyFormatsCount; i++ ) {
			final TableFormatter bodyFormat = bodyFormats[i];
			if( ! bodyFormat.isUse( row,getDBTableModel() ) ) { continue; }
			outBuf.append("<tbody").append( getBgColorCycleClass( bcCnt++,row ) );
			if( isNoTransition() ) {
				outBuf.append( getHiddenRowValue( row ) );
			}
			outBuf.append( STACK_TBODY )
				.append( STACK_ROW_PREFIX ).append( stackRow ).append( '\'' )
				.append( STACK_ID_PREFIX  ).append( stackRow ).append( "'>" )
				.append( bodyFormat.getTrTag() );

			//  No 欄そのものの作成判断追加
			if( isNumberDisplay() ) {
				final String ckboxTD = "<td" + bodyFormat.getRowspan();				// 6.8.1.1 (2017/07/22)
				outBuf.append( makeCheckbox( ckboxTD,row,blc,true ) );				// 6.8.2.0 (2017/10/13)
			}

			int cl = 0;
			for( ; cl<bodyFormat.getLocationSize(); cl++ ) {
				String fmt = bodyFormat.getFormat(cl);
				final int loc = bodyFormat.getLocation(cl);		// 3.5.5.0 (2004/03/12)
				if( ! bodyFormat.isNoClass() && loc >= 0 ) {	// 3.5.5.7 (2004/05/10)
					// 6.4.3.4 (2016/03/11) tdに、[カラム]が無いケースで、次の[カラム]のクラス属性が、前方すべてのtdにセットされてしまう対応。
					final int idx = fmt.lastIndexOf( "<td" );
					if( idx >= 0 ) {	// matchしてるので、あるはず
						final String tdclass = " class=\"" + getClassName(loc) + "\" ";			// 6.4.5.0 (2016/04/08)
						fmt = fmt.substring( 0,idx+3 ) + tdclass + fmt.substring( idx+3 ) ;
					}
				}
				outBuf.append( fmt );

				// locがstackに入っていれば出力
				if( isStackClm(loc) ){
					if( loc >= 0 ) {
						// 6.4.4.2 (2016/04/01) 処理の共通部をまとめる。
						outBuf.append( getTypeCaseValue( bodyFormat.getType(cl),row,loc ) );
					}
					else {
						outBuf.append( bodyFormat.getSystemFormat(row,loc) );
					}
				}
			}
			// 5.5.8.3 (2012/11/17)内部積上げの結果は出力場所の特定が難しいため一番最後尾にtd付きで出力しておきます
			if( innerStack ){
				outBuf.append("</td><td><div class='stackDivParent' capacity='"+ cap + "' style='width:100%; position:relative;'>"); // 5.6.1.2 (2013/02/22) td終了追加
				for( int cs=0; cs<costAry.length; cs++ ){
					outBuf.append("<div class='stackDiv' style='position:absolute; top:0px;' num='")
						.append( cs)
						.append("' stackedCost='")
						.append( costAry[cs] )
						.append( "'></div>");
				}
				outBuf.append("</div>");
			}

			outBuf.append( bodyFormat.getFormat(cl) )
				.append("</tbody>").append( CR );
		}
		return outBuf;
	}
}
