/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.Attributes;

import java.util.Map;											// 7.4.2.0 (2021/05/08)

/**
 * UPLOAD2 エディターは、ドラッグ＆ドロップでファイルアップロードを行う場合に
 * 使用する編集用クラスです。
 *
 * 行ごとの対応も、ファイル名の書き換えもサポートしていません。
 * 現状できるのは、filetemp/ログインID フォルダに、そのままのファイル名で
 * アップロードするだけです。
 *
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * 編集ﾊﾟﾗﾒｰﾀに指定できる項目は、『;』で
 *  width=300px;						初期値
 *  height=180px;						初期値
 *  CALL_JSP=/common/dragFiles.jsp;		初期値
 *  UPLOAD_DIR=							初期値 (filetemp/{@USER.ID})
 *
 *
 * @og.rev 7.4.2.0 (2021/05/08) 新規追加
 *
 * @og.group データ編集
 *
 * @version  7.4
 * @author   Kazuhiko Hasegawa
 * @since    JDK11.0,
 */
public class Editor_UPLOAD2 extends AbstractEditor {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.4.2.0 (2021/05/08)" ;

	private static final String JSP = HybsSystem.sys( "JSP" ) ;

	private static final String CALL_JSP   = "/common/dragFiles.jsp" ;
	private static final String UPLOAD_DIR = "" ;		// 指定がない場合は、ﾃﾞﾌｫﾙﾄ( filetemp/{@USER.ID} )
														// dragFiles.jsp 内で、設定。jsp/ 以下のﾌｫﾙﾀﾞを指定

	private static final String DEF_WIDTH  = "300px" ;
	private static final String DEF_HEIGHT = "180px" ;

	private  String		name;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 */
	public Editor_UPLOAD2() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * DBColumnオブジェクトを指定したprivateコンストラクター。
	 *
	 * @og.rev 7.4.2.0 (2021/05/08) optionAttributesが使えるように修正。
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Editor_UPLOAD2( final DBColumn clm ) {
		super( clm );

		final String name = clm.getName();

		final Attributes attri = clm.getEditorAttributes()
					.set( "width"	, DEF_WIDTH  )			// 初期値として渡したい
					.set( "height"	, DEF_HEIGHT );

		// 7.4.2.0 (2021/05/08)
		String srcURL = JSP + CALL_JSP ;
		final String param = clm.getEditorParam();
		if( param != null ) {
			final Map<String,String> prmMap = attri.getParamMap( param );		// param をMapに分解した物

			attri.set( "width"  , prmMap.get( "width"  )  );	// width 指定があれば使う
			attri.set( "height" , prmMap.get( "height" ) );		// width 指定があれば使う

			final String callJsp = prmMap.get( "CALL_JSP" ) ;
			if( callJsp != null && !callJsp.isEmpty() ) { srcURL = JSP + callJsp ; }

			final String urlOpt  = prmMap.get( "UPLOAD_DIR" ) ;
			if( urlOpt != null && !urlOpt.isEmpty() ) { srcURL += "?UPLOAD_DIR=" + urlOpt ; }
		}

		attributes = new Attributes()
					.set( "name"	, name	)
					.set( "id"		, name	)
					.set( "src"		, srcURL  )
		//			.set( "width"	, "300px" )			// 初期値として渡したい
		//			.set( "height"	, "180px" )
		//			.set( clm.getEditorAttributes() );	// #addAttributes( Attributes ) の代替え
					.set( attri );						// #addAttributes( Attributes ) の代替え

		tagBuffer.add( XHTMLTag.iframe( attributes , "" ) );
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 7.4.2.0 (2021/05/08) ドラッグ＆ドロップのファイルアップロードを追加
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_UPLOAD2( clm );
	}
	/**
	 * データの編集用文字列を返します。
	 *
	 * @og.rev 7.4.2.0 (2021/05/08) ドラッグ＆ドロップのファイルアップロードを追加
	 *
	 * @param	value 値
	 *
	 * @return	データの編集用文字列
	 * @og.rtnNotNull
	 */
	public String getValue( final String value ) {
		return tagBuffer.makeTag();
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @og.rev 7.4.2.0 (2021/05/08) ドラッグ＆ドロップのファイルアップロードを追加
	 *
	 * @param	row   行番号
	 * @param	value 値
	 *
	 * @return	データ表示/編集用の文字列
	 * @og.rtnNotNull
	 */
	public String getValue( final int row,final String value ) {
		return tagBuffer.makeTag();
	}
}
