/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.resource;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.db.ApplicationInfo;
import org.opengion.fukurou.db.DBUtil;
// import org.opengion.hayabusa.db.DBColumnConfig;		// 7.0.7.0 (2019/12/13)
// import org.opengion.fukurou.util.StringUtil;			// 7.2.6.0 (2020/06/30)

import java.util.Map;
import java.util.WeakHashMap;
import java.util.Collections ;

/**
 * systemId に対応したカラムデータを作成するデータロードクラスです。
 *
 * カラムデータは、項目(CLM)に対して、各種カラム情報を持っています。
 * エンジン内部で使用している DBColumn オブジェクトは、RENDERER や EDITOR など
 * 実際にはオブジェクトで管理していますが、この ColumnData では、それらのキーとなる
 * 文字列を持っています。実際に DBColumn オブジェクトの構築時に、各属性オブジェクトを
 * 生成(または、キャッシュから取り出し)ます。
 *
 * カラムデータを作成する場合は、同一カラムで、作成区分(KBSAKU)違いの場合は、
 * 最も大きな作成区分を持つコードを使用します。
 * 作成区分(KBSAKU)='0' のデータは、マスタリソースとして、エンジンとともに
 * 配布されるリソースになります。
 *
 * カラムデータには、３つのレベルのオブジェクト作成方法が適用されます。
 * エンジン内部のカラムリソースファイル(org.opengion.hayabusa.common.data.ColumnResource)は、
 * 初期作成されるカラムリソースです。エンジンの更新に対応して、このリソースも同時に
 * 更新されます。このカラムは、最も優先順位の低いリソースで、同一キー情報で他の形式の
 * カラムがあれば、そちらが使用されます。
 *
 * 読込フラグ(FGLOAD)='1'のカラムリソースは、すべて初期起動時に一括読み込みされます。
 * 読込フラグが、'1' 以外のデータは、初期起動時には、メモリにキャッシュされず
 * 実際に使用されるまで、オブジェクトが作成されません。
 * これは、使用されるかどうか判らないカラムデータを、予め作成しないことで、メモリの
 * 節約を図っています。
 *
 * SYSTEM_ID='**' は、共通リソースです。
 * これは、システム間で共通に使用されるリソース情報を登録しておきます。
 *
 * @og.rev 4.0.0.0 (2004/12/31) 新規作成
 * @og.group リソース管理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
final class ColumnDataLoader {
	// リソースの接続先を、取得します。
	private final String DBID = HybsSystem.sys( "RESOURCE_DBID" );

	// ＤＢリソースの初期一括読込のクエリー
// private static final String QUERY = "select CLM,CLS_NAME,USE_LENGTH,VIEW_LENGTH,"
//									+ "RENDERER,EDITOR,DBTYPE,DATA_DEFAULT,LABEL_CLM,CODE_CLM,"
//									+ "CLM_PARAM,RENDERER_PARAM,EDITOR_PARAM,TYPE_PARAM,ROLES"
//									+ " from GEA03 where SYSTEM_ID in ( ?,'**')"
//									+ " and FGJ='1' and FGLOAD = '1'"
//									+ " order by SYSTEM_ID,CLM,KBSAKU" ;

//	// 4.3.5.7 (2009/03/22) FGLOADの影響で個別システムのリソースが読まれない問題の対応
//	// 6.2.0.0 (2015/02/27) フィールドサイズ(FIELD_SIZE) 追加
//	private static final String QUERY = "select CLM,CLS_NAME,USE_LENGTH,VIEW_LENGTH,"
//									+ "RENDERER,EDITOR,DBTYPE,DATA_DEFAULT,LABEL_CLM,CODE_CLM,"
//									+ "CLM_PARAM,RENDERER_PARAM,EDITOR_PARAM,TYPE_PARAM,ROLES,'' AS FIELD_SIZE"			// 6.4.9.5 (2016/09/09) javaDB対応
//									+ ",FGLOAD"
//									+ " from GEA03 where SYSTEM_ID in ( ?,'**')"
//									+ " and FGJ='1'"
//									+ " order by SYSTEM_ID,CLM,KBSAKU" ;

//	// 7.2.6.0 (2020/06/30) "**"以外にベースとなるSYSTEM_ID(RESOURCE_BASE_SYSTEM_ID)設定の対応
//	private static final String QUERY = "select CLM,CLS_NAME,USE_LENGTH,VIEW_LENGTH,"
//									+ "RENDERER,EDITOR,DBTYPE,DATA_DEFAULT,LABEL_CLM,CODE_CLM,"
//									+ "CLM_PARAM,RENDERER_PARAM,EDITOR_PARAM,TYPE_PARAM,ROLES"
//									+ ",'' AS FIELD_SIZE,FGLOAD,UNIQ,SYSTEM_ID"
//									+ ",1 as SNO,KBSAKU"
//									+ " from GEA03 where SYSTEM_ID in ( ?,'**') and FGJ='1'"		// RESOURCE_BASE_SYSTEM_ID
//									+ " union all"
//									+ " select CLM,CLS_NAME,USE_LENGTH,VIEW_LENGTH,"
//									+ "RENDERER,EDITOR,DBTYPE,DATA_DEFAULT,LABEL_CLM,CODE_CLM,"
//									+ "CLM_PARAM,RENDERER_PARAM,EDITOR_PARAM,TYPE_PARAM,ROLES"
//									+ ",'' AS FIELD_SIZE,FGLOAD,UNIQ,SYSTEM_ID"
//									+ ",2 as SNO,KBSAKU"
//									+ " from GEA03 where SYSTEM_ID = ? and FGJ='1'"					// 従来のSYSTEM_ID
//									+ " order by SNO,SYSTEM_ID,CLM,KBSAKU" ;

//	// 7.2.6.1 (2020/07/17) "**"以外にベースとなるSYSTEM_ID(RESOURCE_BASE_SYSTEM_ID)設定の対応
//	private static final String QUERY = "select CLM,CLS_NAME,USE_LENGTH,VIEW_LENGTH,"
//									+ "RENDERER,EDITOR,DBTYPE,DATA_DEFAULT,LABEL_CLM,CODE_CLM,"
//									+ "CLM_PARAM,RENDERER_PARAM,EDITOR_PARAM,TYPE_PARAM,ROLES"
//									+ ",'' AS FIELD_SIZE,FGLOAD,UNIQ,SYSTEM_ID"
//									+ ",SNO,KBSAKU"
//									+ " from ("
//									+ "  select 0 as SNO,B.* from GEA03 B where SYSTEM_ID='**' and FGJ='1'"	// ｴﾝｼﾞﾝ共通
//									+ "  union all"
//									+ "  select 1 as SNO,B.* from GEA03 B where SYSTEM_ID=? and FGJ='1'"	// RESOURCE_BASE_SYSTEM_ID
//									+ "  union all"
//									+ "  select 2 as SNO,B.* from GEA03 B where SYSTEM_ID=? and FGJ='1'"	// 最上位ののSYSTEM_ID
//									+ " ) A"
//									+ " order by SNO,SYSTEM_ID,KBSAKU,CLM" ;

//	// 7.2.8.0 (2020/09/04)  "**"以外にベースとなるSYSTEM_ID(RESOURCE_BASE_SYSTEM_ID)設定の対応。SQL文手直し
//	private static final String QUERY = "select A.CLM,A.CLS_NAME,A.USE_LENGTH,A.VIEW_LENGTH"
//										+ ",A.RENDERER,A.EDITOR,A.DBTYPE,A.DATA_DEFAULT,A.LABEL_CLM,A.CODE_CLM"
//										+ ",A.CLM_PARAM,A.RENDERER_PARAM,A.EDITOR_PARAM,A.TYPE_PARAM,A.ROLES"
//										+ ",'' AS FIELD_SIZE,A.FGLOAD,A.UNIQ,A.SYSTEM_ID"
//										+ ",A.SNO,A.KBSAKU"
//										+ " from ("
//										+ "  select 0 as SNO,B.* from GEA03 B where B.SYSTEM_ID='**'"	// ｴﾝｼﾞﾝ共通
//										+ "  union all"
//										+ "  select 1 as SNO,B.* from GEA03 B where B.SYSTEM_ID=?"		// RESOURCE_BASE_SYSTEM_ID
//										+ "  union all"
//										+ "  select 2 as SNO,B.* from GEA03 B where B.SYSTEM_ID=?"		// 最上位ののSYSTEM_ID
//										+ " ) A"
//										+ " where A.FGJ='1'"
//										+ " order by A.SNO,A.KBSAKU,A.CLM" ;

	// 7.3.1.3 (2021/03/09)
	private static final String SEL_CLM = "select CLM,CLS_NAME,USE_LENGTH,VIEW_LENGTH"
										+ ",RENDERER,EDITOR,DBTYPE,DATA_DEFAULT,LABEL_CLM,CODE_CLM"
										+ ",CLM_PARAM,RENDERER_PARAM,EDITOR_PARAM,TYPE_PARAM,ROLES"
										+ ",'' AS FIELD_SIZE,FGLOAD,UNIQ,SYSTEM_ID"
										+ ",KBSAKU" ;

	// 7.3.1.3 (2021/03/09)
//	private static final String QUERY = SEL_CLM + ",0 as SNO"
	// 7.4.5.0 (2021/08/31) Firebird 対応
	private static final String QUERY = "select a.* from ("
									+	SEL_CLM + ",0 as SNO"
									+ " from GEA03 where SYSTEM_ID='**' and FGJ='1'"	// ｴﾝｼﾞﾝ共通
									+ " union all "
									+  SEL_CLM + ",1 as SNO"
									+ " from GEA03 where SYSTEM_ID=? and FGJ='1'"		// RESOURCE_BASE_SYSTEM_ID
									+ " union all "
									+  SEL_CLM + ",2 as SNO"
									+ " from GEA03 where SYSTEM_ID=? and FGJ='1'"		// 最上位ののSYSTEM_ID
//									+ " order by SNO,KBSAKU,CLM" ;
									+ " ) a order by a.SNO,a.KBSAKU,a.CLM" ;					// 7.4.5.0 (2021/08/31)

//	// ＤＢリソースの個別読込時のクエリー
//	// 6.2.0.0 (2015/02/27) フィールドサイズ(FIELD_SIZE) 追加
//	// 6.3.1.1 (2015/07/10) FGLOAD,UNIQ 追加
//	private static final String QUERY2 = "select CLM,CLS_NAME,USE_LENGTH,VIEW_LENGTH,"
//									+ "RENDERER,EDITOR,DBTYPE,DATA_DEFAULT,LABEL_CLM,CODE_CLM,"
//									+ "CLM_PARAM,RENDERER_PARAM,EDITOR_PARAM,TYPE_PARAM,ROLES,'' AS FIELD_SIZE"		// 6.4.9.5 (2016/09/09) javaDB対応
//									+ ",FGLOAD,UNIQ,SYSTEM_ID"				// 6.3.1.1 (2015/07/10)
//									+ " from GEA03 where SYSTEM_ID in ( ?,'**')"
//									+ " and CLM=? and FGJ='1'"
//									+ " order by SYSTEM_ID,KBSAKU" ;

//	// 7.2.6.0 (2020/06/30) "**"以外にベースとなるSYSTEM_ID(RESOURCE_BASE_SYSTEM_ID)設定の対応
//	private static final String QUERY2 = "select CLM,CLS_NAME,USE_LENGTH,VIEW_LENGTH,"
//									+ "RENDERER,EDITOR,DBTYPE,DATA_DEFAULT,LABEL_CLM,CODE_CLM,"
//									+ "CLM_PARAM,RENDERER_PARAM,EDITOR_PARAM,TYPE_PARAM,ROLES"			// 6.4.9.5 (2016/09/09) javaDB対応
//									+ ",'' AS FIELD_SIZE,FGLOAD,UNIQ,SYSTEM_ID"							// 6.3.1.1 (2015/07/10)
//									+ ",1 as SNO,KBSAKU"
//									+ " from GEA03 where SYSTEM_ID in ( ?,'**') and CLM=? and FGJ='1'"	// RESOURCE_BASE_SYSTEM_ID
//									+ " union all"
//									+ " select CLM,CLS_NAME,USE_LENGTH,VIEW_LENGTH,"
//									+ "RENDERER,EDITOR,DBTYPE,DATA_DEFAULT,LABEL_CLM,CODE_CLM,"
//									+ "CLM_PARAM,RENDERER_PARAM,EDITOR_PARAM,TYPE_PARAM,ROLES"			// 6.4.9.5 (2016/09/09) javaDB対応
//									+ ",'' AS FIELD_SIZE,FGLOAD,UNIQ,SYSTEM_ID"							// 6.3.1.1 (2015/07/10)
//									+ ",2 as SNO,KBSAKU"
//									+ " from GEA03 where SYSTEM_ID = ? and CLM=? and FGJ='1'"			// 従来のSYSTEM_ID
//									+ " order by SNO,SYSTEM_ID,KBSAKU" ;

//	// 7.2.6.0 (2020/06/30) "**"以外にベースとなるSYSTEM_ID(RESOURCE_BASE_SYSTEM_ID)設定の対応
//	private static final String QUERY2 = "select CLM,CLS_NAME,USE_LENGTH,VIEW_LENGTH,"
//									+ "RENDERER,EDITOR,DBTYPE,DATA_DEFAULT,LABEL_CLM,CODE_CLM,"
//									+ "CLM_PARAM,RENDERER_PARAM,EDITOR_PARAM,TYPE_PARAM,ROLES"			// 6.4.9.5 (2016/09/09) javaDB対応
//									+ ",'' AS FIELD_SIZE,FGLOAD,UNIQ,SYSTEM_ID"							// 6.3.1.1 (2015/07/10)
//									+ ",1 as SNO,KBSAKU"
//									+ " from ("
//									+ "  select 0 as SNO,B.* from GEA03 B where SYSTEM_ID='**' and CLM=? and FGJ='1'"	// ｴﾝｼﾞﾝ共通
//									+ "  union all"
//									+ "  select 1 as SNO,B.* from GEA03 B where SYSTEM_ID=? and CLM=? and FGJ='1'"	// RESOURCE_BASE_SYSTEM_ID
//									+ "  union all"
//									+ "  select 2 as SNO,B.* from GEA03 B where SYSTEM_ID=? and CLM=? and FGJ='1'"	// 最上位ののSYSTEM_ID
//									+ " ) A"
//									+ " order by SNO,SYSTEM_ID,KBSAKU" ;

//	// 7.2.8.0 (2020/09/04)  "**"以外にベースとなるSYSTEM_ID(RESOURCE_BASE_SYSTEM_ID)設定の対応。SQL文手直し
//	private static final String QUERY2 = "select A.CLM,A.CLS_NAME,A.USE_LENGTH,A.VIEW_LENGTH,"
//										+ "A.RENDERER,A.EDITOR,A.DBTYPE,A.DATA_DEFAULT,A.LABEL_CLM,A.CODE_CLM"
//										+ ",A.CLM_PARAM,A.RENDERER_PARAM,A.EDITOR_PARAM,A.TYPE_PARAM,A.ROLES"		// 6.4.9.5 (2016/09/09) javaDB対応(null AS エラー)
//										+ ",'' AS FIELD_SIZE,A.FGLOAD,A.UNIQ,A.SYSTEM_ID"							// 6.3.1.1 (2015/07/10)
//										+ ",A.SNO,A.KBSAKU"
//										+ " from ("
//										+ "  select 0 as SNO,B.* from GEA03 B where B.SYSTEM_ID='**'"	// ｴﾝｼﾞﾝ共通
//										+ "  union all"
//										+ "  select 1 as SNO,B.* from GEA03 B where B.SYSTEM_ID=?"		// RESOURCE_BASE_SYSTEM_ID
//										+ "  union all"
//										+ "  select 2 as SNO,B.* from GEA03 B where B.SYSTEM_ID=?"		// 最上位ののSYSTEM_ID
//										+ " ) A"
//										+ " where A.CLM=? and A.FGJ='1'"
//										+ " order by A.SNO,A.KBSAKU" ;

	// 注意：CLMを unionする前に条件として入れたのでパラメータの順番が変わる。
	// 7.3.1.3 (2021/03/09)
//	private static final String QUERY2 = SEL_CLM + ",0 as SNO"
	// 7.4.5.0 (2021/08/31) Firebird 対応
	private static final String QUERY2 = "select a.* from ("
									+ SEL_CLM + ",0 as SNO"
									+ " from GEA03 where SYSTEM_ID='**' and CLM=? and FGJ='1'"	// ｴﾝｼﾞﾝ共通
									+ " union all "
									+  SEL_CLM + ",1 as SNO"
									+ " from GEA03 where SYSTEM_ID=? and CLM=? and FGJ='1'"		// RESOURCE_BASE_SYSTEM_ID
									+ " union all "
									+  SEL_CLM + ",2 as SNO"
									+ " from GEA03 where SYSTEM_ID=? and CLM=? and FGJ='1'"		// 最上位ののSYSTEM_ID
//									+ " order by SNO,KBSAKU,CLM" ;
									+ " ) a order by a.SNO,a.KBSAKU,a.CLM" ;							// 7.4.5.0 (2021/08/31)

	// 6.3.1.1 (2015/07/10) 読込フラグ(FGLOAD) のマーカー設定追加。
	private static final boolean IS_FGLOAD_AUTOSET = HybsSystem.sysBool( "USE_FGLOAD_AUTOSET" );		// 6.4.1.1 (2016/01/16) useFgloadAutoset → IS_FGLOAD_AUTOSET  refactoring

	// 6.3.1.1 (2015/07/10) FGLOAD更新(UNIQ だけで指定可能だが、万一を想定して、SYSTEM_IDとCLMを条件に追記)
	// 7.2.6.0 (2020/06/30) "**"以外にベースとなるSYSTEM_ID(RESOURCE_BASE_SYSTEM_ID)設定の対応したため、where条件から、SYSTEM_ID は削除します。
//	private static final String UPDATE2 = "update GEA03 set FGLOAD='2' where UNIQ=? and SYSTEM_ID=? and CLM=?";
	private static final String UPDATE2 = "update GEA03 set FGLOAD='2' where UNIQ=? and CLM=?";

	/** 6.4.3.1 (2016/02/12) Collections.synchronizedMap で同期処理を行います。  */
	private final Map<String,ColumnData> columnMap = Collections.synchronizedMap( new WeakHashMap<>() );	// キャッシュ用プール
	private final String SYSTEM_ID ;			// システムID
//	private       String BASE_SYS_ID = "**" ;	// 7.2.6.0 (2020/06/30) ベースシステムID
//	private final String BASE_SYS_ID = StringUtil.nval( HybsSystem.sys( "RESOURCE_BASE_SYSTEM_ID", false ),"**" ) ;	// 7.2.9.1 (2020/10/23) ベースシステムID
	private final String BASE_SYS_ID ;			// 7.2.9.2 (2020/10/30) ベースシステムID

	/** コネクションにアプリケーション情報を追記するかどうか指定 */
	public static final boolean USE_DB_APPLICATION_INFO  = HybsSystem.sysBool( "USE_DB_APPLICATION_INFO" ) ;

	// 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	private final ApplicationInfo appInfo;

	/**
	 *  SystemId 毎に ファクトリオブジェクトを作成します。
	 *
	 * @og.rev 7.2.9.2 (2020/10/30) ベースとなるSYSTEM_ID(RESOURCE_BASE_SYSTEM_ID)の取得
	 *
	 * @param systemId システムID
	 * @param baseSys ベースとなるSYSTEM_ID
	 * @param initLoad リソースデータの先読み可否(true:先読みする)
	 */
//	ColumnDataLoader( final String systemId,final boolean initLoad ) {
	ColumnDataLoader( final String systemId,final String baseSys,final boolean initLoad ) {
		SYSTEM_ID   = systemId;
		BASE_SYS_ID = baseSys ;			// 7.2.9.2 (2020/10/30)

		// 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
		if( USE_DB_APPLICATION_INFO ) {
			appInfo = new ApplicationInfo();
			// ユーザーID,IPアドレス,ホスト名
			appInfo.setClientInfo( SYSTEM_ID,HybsSystem.HOST_ADRS,HybsSystem.HOST_NAME );
			// 画面ID,操作,プログラムID
			appInfo.setModuleInfo( "ColumnDataLoader",null,null );
		}
		else {
			appInfo = null;
		}

		// ApplicationInfo の設定が終わってから実行します。
		if( initLoad ) { loadDBResource(); }
	}

	/**
	 * ＤＢリソースより カラムデータを取得、設定します。
	 * 同一キー(CLM)に対して、複数の作成区分(KBSAKU)を持つデータが
	 * 検索される場合は、作成区分(KBSAKU)の大きな値が使用されます。
	 * つまり、より、ローカライズなキーほど、作成区分(KBSAKU)に大きな値を
	 * 使用するようにします。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.3.5.7 (2009/03/22) FGLOADの影響でシステム個別リソースが読まれない問題対応
	 * @og.rev 7.0.7.0 (2019/12/13) 読み取り件数の評価を、破棄分も考慮する。
	 * @og.rev 7.2.6.0 (2020/06/30) "**"以外にベースとなるSYSTEM_ID(RESOURCE_BASE_SYSTEM_ID)設定の対応
	 *
	 */
	private void loadDBResource() {
//		// 7.2.6.0 (2020/06/30)
//		final String baseSystemId = HybsSystem.sys( "RESOURCE_BASE_SYSTEM_ID", false );
//		BASE_SYS_ID = StringUtil.isEmpty( baseSystemId ) ? "**" : baseSystemId;

//		final String[] args = new String[] { SYSTEM_ID };
		final String[] args = new String[] { BASE_SYS_ID,SYSTEM_ID };	// 7.2.6.1 (2020/07/17)

		final String[][] vals = DBUtil.dbExecute( QUERY,args,appInfo,DBID );
		final int[] cnt = new int[3];	// **,BASE_SYS_ID,SYSTEM_ID の個数

		final int len = vals.length;
		for( int i=0; i<len; i++ ) {
			final String clm = vals[i][0];
			final int idx = Integer.parseInt( vals[i][ColumnData.SNO] );
//			final String sys = vals[i][ColumnData.SYSTEM_ID];
//			final int idx = "**".equals( sys ) ? 0 : BASE_SYS_ID.equals( sys ) ? 1 : 2 ;

			if( "1".equals( vals[i][ColumnData.FG_LOAD] ) ) {			// 4.3.5.7 (2009/03/22) 1:一括読込
				columnMap.put( clm,new ColumnData( vals[i] ) );
				cnt[idx]++ ;
			}
			// より上の作成区分で、FGLOAD='1'(一括読込)以外の場合は、破棄する。
			// order by SYSTEM_ID,CLM,KBSAKU
			else if( columnMap.get( clm ) != null ){
				columnMap.remove( clm );
			}
		}

		// 7.0.7.0 (2019/12/13) 読み取り件数の評価を、破棄分も考慮する。
//		System.out.println( "  ColumnDataLoader [" + len + "] loaded" );
		System.out.println( "  ColumnDataLoader [" + len + "] select [" + columnMap.size() + "] "
			+	" ** [" + cnt[0] + "] " + BASE_SYS_ID + " [" + cnt[1] + "] " + SYSTEM_ID + " [" + cnt[2] + "] loaded"  );
	}

	/**
	 * ColumnDataオブジェクトを取得します。
	 * 作成したColumnDataオブジェクトは，内部にプールしておき，同じリソース要求が
	 * あったときは，プールの ColumnDataを返します。
	 * 読込フラグ(FGLOAD)が '1' のデータは、起動時に先読みします。
	 * それ以外のデータは、ここでキー要求が発生した時点で読み込みます。
	 * 読込フラグ(FGLOAD) のマーカー設定モード(USE_FGLOAD_AUTOSET)を使用する(true)場合は、
	 * 追加読み込み(先読みされていないカラム)に対して、読込フラグ(FGLOAD)を 2:使用実績 に
	 * 設定します。(次回起動時の、初期読み込みは行いません。)
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 6.3.1.1 (2015/07/10) 読込フラグ(FGLOAD) のマーカー設定追加。
//	 * @og.rev 7.0.7.0 (2019/12/13) FG_LOADが、3:使用確認 , 8:未使用 の場合に使用されると、2:使用実績 をセットする。	7.0.7.2 元に戻す
	 * @og.rev 7.0.7.0 (2019/12/13) キーだけからﾘｿｰｽ無しのColumnDataを作成します。
	 * @og.rev 7.0.7.2 (2019/12/28) ﾘｿｰｽ無しのｷｬｯｼｭは行わない。(DBからSELECTした際、ｽｷｰﾏから自動で文字か数字を判定しているから)
	 * @og.rev 7.2.6.0 (2020/06/30) "**"以外にベースとなるSYSTEM_ID(RESOURCE_BASE_SYSTEM_ID)設定の対応
	 * @og.rev 7.3.1.3 (2021/03/09) QUERY文字列を変更。それに伴って、引数の並び順を変更。
	 *
	 * @param   key         カラムのキー
	 *
	 * @return	ColumnDataオブジェクト
	 */
	public ColumnData getColumnData( final String key ) {
		ColumnData column = columnMap.get( key ) ;
		if( column == null ) {
//			final String[] args = new String[] { SYSTEM_ID,key };
//			final String[] args = new String[] { key,BASE_SYS_ID,key,SYSTEM_ID,key };	// 7.2.6.1 (2020/07/17)
//			final String[] args = new String[] { BASE_SYS_ID,SYSTEM_ID,key };			// 7.2.8.0 (2020/09/04)
			final String[] args = new String[] { key,BASE_SYS_ID,key,SYSTEM_ID,key };	// 7.3.1.3 (2021/03/09)
			final String[][] vals = DBUtil.dbExecute( QUERY2,args,appInfo,DBID );		// SYSTEM_ID='**' も含む
			if( vals.length > 0 ) {
				final int row=vals.length-1;						// 最後の検索結果
				column = new ColumnData( vals[row] );				// 最後の検索結果が有効
				columnMap.put( key,column );

				// 6.3.1.1 (2015/07/10) 読込フラグ(FGLOAD) のマーカー設定追加。
				if( IS_FGLOAD_AUTOSET ) {
					// 1:一括読込 と、2:使用実績 以外のﾘｿｰｽは、2:使用実績 をセットする。(SYSTEM_ID='**'は含まない)
					final String fgld  = vals[row][ColumnData.FG_LOAD];
					final String sysld = vals[row][ColumnData.SYSTEM_ID];
					if( !"1".equals( fgld ) && !"2".equals( fgld ) && !"**".equals( sysld ) ) {
	//				// 7.0.7.0 (2019/12/13) FG_LOADが、3:使用確認 , 8:未使用 の場合に使用されると、2:使用実績 をセットする。
	//				if( "3".equals( fgld ) || "8".equals( fgld ) ) {
						// 7.2.6.0 (2020/06/30) RESOURCE_BASE_SYSTEM_ID 追加したため、where条件から、SYSTEM_ID は削除します。
//						final String[] args2 = new String[] { vals[row][ColumnData.UNIQ],SYSTEM_ID,key };
						final String[] args2 = new String[] { vals[row][ColumnData.UNIQ],key };
						DBUtil.dbExecute( UPDATE2,args2,appInfo,DBID );		// FGLOAD を、2:使用実績 にセット
					}
				}
			}

//			//
//			// 7.0.7.0 (2019/12/13) キーだけからﾘｿｰｽ無しのColumnDataを作成します。
//			else {
//				final DBColumnConfig config = new DBColumnConfig( key );
//				column = config.getColumnData();
//				columnMap.put( key,column );
//			}
		}
		return column ;
	}

	/**
	 * ColumnData オブジェクトのキャッシュを個別にクリアします。
	 * リソースデータの更新など、一部分の更新時に、すべてのキャッシュを
	 * 破棄するのではなく、指定の分のみ破棄できる機能です。
	 *
	 * @og.rev 6.9.0.1 (2018/02/05) どのシステムIDのﾘｿｰｽがクリアされたかを表示します。
	 *
	 * @param   key         カラムのキー
	 */
	public void clear( final String key ) {
		System.out.println( "SYSTEM_ID=[" + SYSTEM_ID + "] , Key=[" + key + "] の部分リソースクリアを実施しました。" );
		columnMap.remove( key );
	}

	/**
	 * ColumnData オブジェクトのキャッシュをクリアします。
	 *
	 * @og.rev 6.9.0.1 (2018/02/05) どのシステムIDのﾘｿｰｽがクリアされたかを表示します。
	 *
	 */
	public void clear() {
		System.out.println( "SYSTEM_ID=[" + SYSTEM_ID + "] の全リソースをクリアしました。" );
		columnMap.clear();
	}
}
