/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;

/**
 * RICHLABEL レンデラは、特殊文字のエスケープを元に戻して表示する場合に
 * 使用するクラスです。
 * readonlyのテキストエリアでname属性は付けません。(データは送信されません)
 * 
 * 属性値に1以上の数値を指定した場合はSLABELと同等の処理を行います。
 * その際に、有効なタグについては除外して表示します。（一覧表示用）
 *
 * カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 *  @og.rev 5.9.33.0 (2018/06/01) 新規作成
 *  @og.rev 5.10.1.0 (2018/06/29) ignoreStrに値を追加。
 * @og.group データ編集
 *
 * @version  5.9.33.0
 * @author   T.OTA
 * @since    JDK7.0,
 */
public class Renderer_RICHLABEL extends AbstractRenderer {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.9.33.0 (2018/06/01)" ;

	private static final CellRenderer dbCell = new Renderer_RICHLABEL();

	// 2018/06/28 MODIFY pre,img,table,thead,tbody,tr,th,td,ul,li,h[数値]を追加
	private static final String ignoreStr = "strong|font|a|br|p|span|div|pre|img|table|thead|tbody|tr|th|td|ul|li|h\\d";
	private final int cutSize;	

	/**
	 * デフォルトコンストラクタ
	 * 
	 * @og.rev 5.10.4.2 (2018/10/19) 出力文字数指定対応
	 */
	public Renderer_RICHLABEL() {
		cutSize = 0;	// oota tmp add
	}

	// oota tmp add start
	/**
	 * デフォルトコンストラクタ
	 * 
	 * @og.rev 5.10.4.2 (2018/10/19) 出力文字数指定対応
	 * @param clm
	 */
	public Renderer_RICHLABEL(final DBColumn clm) {
		String param = clm.getRendererParam();
		int i;
		try {
			i = Integer.parseInt( param );
		}
		catch( Exception e) {
			i = 0;
		}
		cutSize = i;
	}
	// oota tmp add end

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 * 
	 * @og.rev 5.10.4.2 (2018/10/19) 出力文字数指定対応
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		// return dbCell;
		String param = clm.getRendererParam();

		if( param != null && param.length() > 0 ) {
			return new Renderer_RICHLABEL(clm);
		}else {
			return dbCell;
		}
	}

	/**
	 * データの表示用文字列を返します。
	 * 
	 * @og.rev 5.10.4.2 (2018/10/19) 出力文字数指定対応
	 *
	 * @param   value 入力値
	 *
	 * @return  データの表示用文字列
	 */
	@Override
	public String getValue( final String value ) {
		// 通常
//		return StringUtil.escapeFilter(value,ignoreStr);

		// タグ情報を除去して、表示する
		String escapeStr = StringUtil.escapeFilter(value,ignoreStr);
		if(cutSize <= 0) {
			// paramが未設定(0以下)の場合は、リッチラベルをhtmlに表示する形式で返す。
			return escapeStr;
		}else {
			// paramに1以上の数値が設定されている場合、SLABELと同様の表示形式で返す。
			// 改行は改行コードに変換してお
			String trg = StringUtil.tagCut(
					escapeStr
					.replaceAll("(?i)<br[ /]*>",System.lineSeparator())
					);

			// 簡易的処理。すべてが全角であっても、制限以内である。
			// 以下はSLABELと同様の計算式
			int len = trg.length();
			if( len*2 <= cutSize) { return trg; };

			int byteSize = 0;
			int adrs;
			for( adrs=0; adrs<len && byteSize<cutSize; adrs++) {
				char ch = trg.charAt(adrs);
				if( ch <= 0x7f || ( ch >= 0xff61 && ch <= 0xff9f )) {
					byteSize ++;
				}else {
					byteSize += 2;
				}
			}

			// 正確にカウントした結果、制限以内であったため。
			if( adrs==len && byteSize<=cutSize ) {
				return trg;
			}
			else if( byteSize>cutSize ) {		// オーバーした場合
				adrs-- ;
			}

			StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_SMALL );
			buf.append("<span title=\"");
			buf.append( trg );
			buf.append("\">");
			buf.append(trg.substring(0,adrs));
			buf.append("...</span>");

			return   buf.toString();
		}
	}
}
