/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.model;

import java.io.File;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;

/**
 * CloudFileOperation用のファイル情報の格納クラス
 *
 * listFilesで取得した、ディレクトリとファイル一覧情報を格納します。
 *
 * パフォーマンスや分かりやすさを考慮してCloudFileOperationからは分離しています
 *
 * @og.group ファイル操作
 *
 * @og.rev 5.10.8.0 (2019/02/01) 新規作成
 * @og.rev 5.10.9.0 (2019/03/01) 変更対応
 * @author oota
 * @since JDK7.0
 */
public class FileOperationInfo extends CloudFileOperation {
	//* このプログラムのVERSION文字列を設定します。{@VALUE} */
	private static final String VERSION = "8.0.0.0 (2021/07/31)" ;
	private static final long serialVersionUID = 800020210731L ;

	/** クラス変数 */
	private final String plugin;

	private long size;
	// 8.0.0.0 (2021/07/31) Field ** has the same name as a method
//	private long lastModified;
//	private boolean isFile;
//	private boolean isDirectory;
	private long lastTime;				// 最終時刻
	private boolean isFil;				// ファイルか？
	private boolean isDir;				// フォルダか？
	private FileOperation file;			// ファイルオペレータ

	/**
	 * コンストラクタ
	 *
	 * 生成時の初期処理。
	 *
	 * @og.rev 8.0.0.0 (2021/07/31) Field ** has the same name as a method
	 *
	 * @param plugin プラグイン名
	 * @param bucket バケット名
	 * @param path ファイルパス
	 */
	public FileOperationInfo(final String plugin, final String bucket, final String path) {
		super(bucket, path);
		this.plugin	= plugin;
		size		= 0;
//		lastModified = 0;
//		isFile = false;
//		isDirectory = false;
		lastTime	= 0L;
		isFil		= false;
		isDir		= false;
		file		= null;
	}

	/**
	 * FileOperationクラスの生成
	 *
	 * 呼び出し時に、FileOperationインスタンスが未生成の場合は、
	 * 生成を行います。
	 */
	private void setFileOperation() {
		if(file == null) {
			file = FileOperationFactory.newStorageOperation( plugin, conBucket, conPath );
		}
	}

	/** Method */
	/**
	 * 書き込み処理
	 *
	 * InputStreamのデータを書き込みます。
	 *
	 * @param is 書き込みデータのInputStream
	 * @throws IOException ファイル関連エラー情報
	 */
	@Override
	public void write(final InputStream is) throws IOException {
		setFileOperation();
		file.write(is);
	}

	/**
	 * 読み込み処理
	 *
	 * データを読み込み、InputStreamとして、返します。
	 *
	 * @return 読み込みデータのInputStream
	 * @throws FileNotFoundException ファイル非存在エラー情報
	 */
	@Override
	public InputStream read() throws FileNotFoundException {
		setFileOperation();
		return file.read();
	}

	/**
	 * 削除処理
	 *
	 * ファイルを削除します。
	 *
	 * @return 成否フラグ
	 */
	@Override
	public boolean delete() {
		setFileOperation();
		return file.delete();
	}

	/**
	 * コピー処理
	 *
	 * ファイルを指定先に、コピーします。
	 *
	 * @param afPath コピー先
	 * @return 成否フラグ
	 */
	@Override
	public boolean copy(final String afPath) {
		setFileOperation();
		return file.copy(afPath);
	}

	/**
	 * 一覧取得
	 *
	 * １つ下の、ディレクトリ・ファイル一覧を取得します。
	 *
	 * @param filter フィルタ情報
	 * @return ファイル一覧
	 */
	@Override
	public File[] listFiles(final FileFilter filter) {
		setFileOperation();
		return file.listFiles(filter);
	}

	/**
	 * ファイルサイズ取得
	 *
	 * ファイルサイズを取得します。
	 *
	 * @return ファイルサイズ
	 */
	@Override
	public long length() {
		return size;
	}

	/**
	 * ファイルサイズ設定
	 *
	 * ファイルサイズを設定します。
	 *
	 * @param size ファイルサイズ
	 */
	public void setSize(final long size) {
		this.size = size;
	}

	/**
	 * 最終更新時刻の取得
	 *
	 * 最終更新時刻を取得します。
	 *
	 * @og.rev 8.0.0.0 (2021/07/31) Field ** has the same name as a method
	 *
	 * @return 最終更新時刻
	 */
	@Override
	public long lastModified() {
//		return lastModified;
		return lastTime;
	}

	/**
	 * 最終更新時刻の設定
	 *
	 * 最終更新時刻を設定します。
	 *
	 * @og.rev 8.0.0.0 (2021/07/31) Field ** has the same name as a method
	 *
	 * @param lastModified 最終更新時刻
	 */
	public void setLastModifiedValue(final long lastModified) {
//		this.lastModified = lastModified;
		lastTime = lastModified;
	}

	/**
	 * ファイル判定取得
	 *
	 * ファイルであるかの判定を返します。
	 *
	 * @og.rev 8.0.0.0 (2021/07/31) Field ** has the same name as a method
	 *
	 * @return ファイル判定
	 */
	@Override
	public boolean isFile() {
//		return isFile;
		return isFil;
	}

	/**
	 * ファイル判定設定
	 *
	 * ファイルであるかの判定を設定します。
	 *
	 * @og.rev 8.0.0.0 (2021/07/31) Field ** has the same name as a method
	 *
	 * @param isFile ファイル判定
	 */
	public void setFile(final boolean isFile) {
//		this.isFile = isFile;
		isFil = isFile;
	}

	/**
	 * ディレクトリ判定取得
	 *
	 * ディレクトリであるかの判定を返します。
	 *
	 * @og.rev 8.0.0.0 (2021/07/31) Field ** has the same name as a method
	 *
	 * @return ディレクトリ判定
	 */
	@Override
	public boolean isDirectory() {
//		return isDirectory;
		return isDir;
	}

	/**
	 * ディレクトリ判定設定
	 *
	 * ディレクトリであるかの判定を設定します。
	 *
	 * @og.rev 8.0.0.0 (2021/07/31) Field ** has the same name as a method
	 *
	 * @param isDirectory ディレクトリ判定
	 */
	public void setDirectory(final boolean isDirectory) {
//		this.isDirectory = isDirectory;
		isDir = isDirectory;
	}

	/**
	 * 親情報の取得
	 *
	 * 親情報を返します。
	 *
	 * @return 親情報
	 */
	@Override
	public File getParentFile() {
		return  FileOperationFactory.newStorageOperation( file , this.getParent() );
	}

//	/**
//	 * このオブジェクトと他のオブジェクトが等しいかどうかを示します。
//	 * インタフェース Comparable の 実装に関連して、再定義しています。
//	 *
//	 * @og.rev 7.2.9.4 (2020/11/20) spotbugs:スーパークラスの equals メソッドをオーバーライドしていないクラス
//	 *
//	 * @param   object 比較対象の参照オブジェクト
//	 *
//	 * @return	引数に指定されたオブジェクトとこのオブジェクトが等しい場合は true、そうでない場合は false
//	 */
//	@Override
//	public boolean equals( final Object object ) {
//		return object instanceof File && super.equals( object );
//	}

//	/**
//	 * オブジェクトのハッシュコード値を返します。
//	 * このメソッドは、java.io.File のハッシュ値を返すことで、equals メソッドとの整合性を取っています。
//	 *
//	 * @og.rev 7.2.9.4 (2020/11/20) spotbugs:equals メソッドは定義していますが hashCode メソッドは定義していないクラス
//	 * @og.rev 8.0.0.0 (2021/07/31) Overriding method merely calls super
//	 *
//	 * @return  このオブジェクトのハッシュコード値
//	 */
//	@Override
//	public int hashCode() {
//		return super.hashCode() ;				// PMD:Overriding method merely calls super が出る。多分定義不要
//	}

//	// テスト用メソッドです
//	public static void main(String[] args) {
//		System.out.println("start");
//
//		FileOperation file = new FileOperationInfo("aws", "otest20190205", "sample/test.txt");
//
//		File parent = file.getParentFile();
//		System.out.println(parent.getPath());
//		System.out.println(parent.isDirectory());
//		System.out.println(parent.isFile());
//
//		System.out.println("end");
//	}
}
