/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.resource;

import org.opengion.fukurou.db.DBUtil;
import org.opengion.fukurou.db.ApplicationInfo;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import static org.opengion.fukurou.system.HybsConst.BUFFER_MIDDLE;	// 8.0.0.0 (2021/10/01)

import java.util.Collections;
import java.util.Map;												// 7.2.6.0 (2020/06/30)
import java.util.LinkedHashMap;										// 7.2.6.0 (2020/06/30)

/**
 * systemId と lang に対応した画面データを作成するデータロードクラスです。
 *
 * 画面データは、画面ID(GUIKEY)に対して、各種画面情報を持っています。
 * 従来と異なるのは、同一画面IDに対して、アドレスやロールズを変えた情報を持てると言う
 * 事です。これは、カスタマイズ時に、画面IDは変えずに、実際のアクセスされるアドレスを
 * 変える事で、他のアプリケーションへの影響を最小限にして開発できます。
 * linkタグや、submit などの gamenID を指定するカスタムタグでは、実際のアクセス先は、
 * ログインユーザーのロールズでアクセス可能な画面のアドレスに転送されます。
 * 作番毎のカスタマイズや、ユーザーロールに応じた飛び先変更などにも使用できます。
 *
 * 画面データでは、複数階層持てるように、画面階層(GUILVL)を持っています。このレベルに
 * 応じて、分類(CLASSIFY)の表示方法が変わります。(擬似階層構造)
 *
 * 画面データでは、言語(LANG)は、条件から消えました。実際に名称を表示させる時は、
 * 画面カラムID(LABEL_CLM)に対応する ラベル定義より、言語に応じたラベルを取得します。
 * エンジン内部で使用している GUIInfo オブジェクト構築時に割り当てます。
 * 分類(CLASSIFY)は、コードリソースに登録します。
 *
 * 画面データを作成する場合は、同一画面IDで、作成区分(KBSAKU)違いの場合は、
 * 最も大きな作成区分を持つ画面情報を使用します。
 * 作成区分(KBSAKU)='0' のデータは、マスタリソースとして、エンジンとともに
 * 配布されるリソースになります。
 *
 * 画面データは、カラム定義のような、読込フラグ(FGLOAD)はありません。
 * 画面情報(GUIInfo)は、ユーザーログイン毎に作成されます。(キャッシュは
 * セッション情報に登録されます。)
 * これは、画面アクセス条件を、ログイン時に済ますことで、高速化を図っています。
 * 画面IDの件数が少ないことと、画面IDを自動作成した場合でも、
 * ほとんどのケースで、すべて使用される可能性が非常に高い為です。
 *
 * SYSTEM_ID='**' は、共通リソースです。
 * これは、システム間で共通に使用されるリソース情報を登録しておきます。
 *
 * @og.rev 4.0.0.0 (2004/12/31) 新規作成
 * @og.group リソース管理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
final class GUIDataLoader {
	// リソースの接続先を、取得します。
	private final String DBID = HybsSystem.sys( "RESOURCE_DBID" );

	// ＤＢリソースの初期一括読み込みのクエリー
	// ソート順は、画面IDオブジェクトの優先順(後優先)で、画面表示順ではありません。
	// 5.6.4.3 (2013/05/24) FAQ追加 現段階ではシステムコードは考慮しない
	// 6.3.8.4 (2015/10/09) GE80(FAQﾃｰﾌﾞﾙ)の取得は廃止。(helpタグで行う)
	// 6.3.9.0 (2015/11/06) コンパイル時に静的な値に初期化されるフィールドは static フィールドにしてください(findbugs)。

	// 7.3.1.3 (2021/03/09)
//	private static final String SEL_CLM = "select GUIKEY,GUILVL,LABEL_CLM,ADDRESS,SEQNO,GROUPS"
//									+ ",'' as CLASSIFY,ROLES,RWMODE,TARGET,PARAM,KBLINK,DYUPD,SYSTEM_ID"
//									+ ",KBSAKU" ;

	// 7.3.1.3 (2021/03/09)
	// 7.4.5.0 (2021/08/31) Firebird 対応
//	private static final String QUERY = "select a.* from ("
//									+	SEL_CLM + ",0 as SNO"
//									+ " from GEA11 where SYSTEM_ID='**' and FGJ='1'"		// ｴﾝｼﾞﾝ共通
//									+ " union all "
//									+  SEL_CLM + ",1 as SNO"
//									+ " from GEA11 where SYSTEM_ID IN (?,?) and FGJ='1'"	// RESOURCE_BASE_SYSTEM_ID , 最上位ののSYSTEM_ID
//									+ " ) a "		// 8.0.0.0 (2021/08/31)
//									+ " order by a.SNO,a.KBSAKU,a.SEQNO,a.GUIKEY" ;

	// 8.0.0.0 (2021/10/01) RESOURCE_BASE_SYSTEM_ID は、SYSTEM_IDの配列で複数指定できる。
	private static final String QUERY = "select GUIKEY,GUILVL,LABEL_CLM,ADDRESS,SEQNO,GROUPS"
									+ ",'' as CLASSIFY,ROLES,RWMODE,TARGET,PARAM,KBLINK,DYUPD,SYSTEM_ID,KBSAKU"
									+ " from GEA11 where SYSTEM_ID = ? and FGJ='1'"			// 8.0.0.0 注意 IN (?,?) → = ? に変更
									+ " order by KBSAKU,SEQNO,GUIKEY" ;


	// 7.2.6.0 (2020/06/30) "**"以外にベースとなるSYSTEM_ID(RESOURCE_BASE_SYSTEM_ID)設定の対応
	/** 7.2.9.1 (2020/10/23) Collections.synchronizedMap で同期処理を行います。  */
	private final Map<String,GUIData> guiMap = Collections.synchronizedMap( new LinkedHashMap<>() );	// 集約するｷｰは、GUIKEY＋ROLES 	// 7.2.9.4 (2020/11/20) private 追加

	// 8.0.0.0 (2021/10/01) RESOURCE_BASE_SYSTEM_ID は、SYSTEM_IDの配列で複数指定できる。
//	private final String SYSTEM_ID ;			// システムID
//	private final String BASE_SYS_ID ;			// 7.2.9.2 (2020/10/30) ベースシステムID
	private final String[] SYS_ARRAY;			// 8.0.0.0 (2021/10/01)

	/** コネクションにアプリケーション情報を追記するかどうか指定 */
	public static final boolean USE_DB_APPLICATION_INFO  = HybsSystem.sysBool( "USE_DB_APPLICATION_INFO" ) ;

	// 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	private final ApplicationInfo appInfo;

	/**
	 *  SystemId 毎に ファクトリオブジェクトを作成します。
	 *
	 * @og.rev 7.2.9.2 (2020/10/30) ベースとなるSYSTEM_ID(RESOURCE_BASE_SYSTEM_ID)の取得
	 * @og.rev 8.0.0.0 (2021/10/01) RESOURCE_BASE_SYSTEM_ID は、SYSTEM_IDの配列で複数指定できる。
	 *
//	 * @param systemId システムID
//	 * @param baseSys ベースとなるSYSTEM_ID
	 * @param sysAry 階層ﾘｿｰｽの元となるSYSTEM_IDの配列(前方優先)
	 */
//	GUIDataLoader( final String systemId,final String baseSys ) {
	GUIDataLoader( final String[] sysAry ) {
//		SYSTEM_ID   = systemId;
//		BASE_SYS_ID = baseSys ;			// 7.2.9.2 (2020/10/30)
		SYS_ARRAY	= sysAry ;			// 8.0.0.0 (2021/10/01)

		// 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
		if( USE_DB_APPLICATION_INFO ) {
			appInfo = new ApplicationInfo();
			// ユーザーID,IPアドレス,ホスト名
//			appInfo.setClientInfo( SYSTEM_ID,HybsSystem.HOST_ADRS,HybsSystem.HOST_NAME );
			appInfo.setClientInfo( SYS_ARRAY[0],HybsSystem.HOST_ADRS,HybsSystem.HOST_NAME );
			// 画面ID,操作,プログラムID
			appInfo.setModuleInfo( "GUIDataLoader",null,null );
		}
		else {
			appInfo = null;
		}

		// ApplicationInfo の設定が終わってから実行します。
		loadDBResource();
	}

	/**
	 * ＤＢリソースより 画面データを取得、設定します。
	 * ＤＢリソースは、GUIKEY,GUILVL,LABEL_CLM,ADDRESS,SEQNO,GROUPS,
	 * CLASSIFY,ROLES,RWMODE,TARGET,PARAM,KBLINK,DYUPD の順番で、GUIKEY の重複を許します。
	 * 重複している場合(ロール違い等)は、一つのオブジェクトとして作成され、
	 * 個々のログインユーザー毎にユニークになるように、設定する必要があります。
	 *
	 * ※ 以下のロジックは、後方優先であり、SYSTEM_IDの配列は前方優先なので逆順で回します。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0.0 (2007/10/31) ロールの継承機能の追加・分類の取得を追加(暫定対応)
	 * @og.rev 5.3.1.0 (2011/01/01) 通常画面に対してアドレスを設定しない場合にロールが効かないバグを修正します。
	 * @og.rev 5.3.1.0 (2011/01/01) ロール継承機能廃止
	 * @og.rev 7.2.6.1 (2020/07/17) "**"以外にベースとなるSYSTEM_ID(RESOURCE_BASE_SYSTEM_ID)設定の対応
	 */
	private void loadDBResource() {
		final int size = SYS_ARRAY.length;

		final int[] cnt = new int[size];	// 各SYSTEM_ID の個数
		int selCnt = 0;

		for( int j=size-1; j>=0; j-- ) {	// SYSTEM_IDの配列は、前方優先なので、逆順で回す必要がある。
			final String sysId = SYS_ARRAY[j];
//			final String[] args = new String[] { BASE_SYS_ID,SYSTEM_ID };	// 7.2.6.1 (2020/07/17)
			final String[] args = new String[] { sysId } ;

			final String[][] gea11 = DBUtil.dbExecute( QUERY,args,appInfo,DBID );
//			final int[] cnt = new int[3];	// **,BASE_SYS_ID,SYSTEM_ID の個数

			// 7.2.6.0 (2020/06/30) "**"以外にベースとなるSYSTEM_ID(RESOURCE_BASE_SYSTEM_ID)設定の対応
			final int len = gea11.length;
			selCnt += len;
			String classify = "";
			for( int i=0; i<len; i++ ) {
				final String[] vals = gea11[i];
//				final int idx = Integer.parseInt( vals[GUIData.SNO] );

				// ロールの継承対応
				final int level = Integer.parseInt( vals[GUIData.GUILVL] );
				if( level == 2 ) {			// 小分類
					classify = vals[GUIData.GUIKEY];		// 暫定対応
				}
				else if( level >= 3 ) {		// 通常
					vals[GUIData.CLASSIFY] = classify;		// 暫定対応
				}

				// 5.3.1.0 (2011/01/01) 通常画面に対してアドレスを設定しない場合にロールが効かないバグを修正します。
				if( ( level == 1 || level == 2 ) && StringUtil.isEmpty( vals[GUIData.ADDRESS] ) ) {
					vals[GUIData.ROLES] = null;
				}

				final String key = vals[GUIData.GUIKEY] + "_" + vals[GUIData.ROLES] ;
				guiMap.put( key,new GUIData( vals ) );		// GUIKEY＋ROLES が同一の画面ﾘｿｰｽは、後設定が有効となる。
//				cnt[idx]++ ;
				cnt[j]++ ;
			}
		}
//		final int guiSize = guiMap.size();

//		System.out.println( "  GUIDataLoader [" + guiSize + "] "
//			+	" ** [" + cnt[0] + "] " + BASE_SYS_ID + " [" + cnt[1] + "] " + SYSTEM_ID + " [" + cnt[2] + "] loaded"  );

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
		buf.append( "  " ).append( SYS_ARRAY[0] ).append( "  GUIDataLoader [" ).append( selCnt )
			.append( "] Map=[" ).append( guiMap.size() ).append( "] " );
		for( int j=0; j<size; j++ ) {
			buf.append( SYS_ARRAY[j] ).append( "=[" ).append( cnt[j] ).append( "] " );
		}
		buf.append( "loaded." );
		System.out.println( buf );
	}

	/**
	 * すべてのGUIData オブジェクト配列を取得します。
	 * プールに持っているすべてのキャッシュを、GUIData オブジェクト配列
	 * にして返します。
	 * このリソースは、List で管理しており、読み込み時にすべてキャッシュされます。
	 *
	 * @og.rev 7.2.6.0 (2020/06/30) "**"以外にベースとなるSYSTEM_ID(RESOURCE_BASE_SYSTEM_ID)設定の対応
	 *
	 * @return	すべてのGUIDataオブジェクト配列
	 */
	public GUIData[] getAllData() {
		synchronized( guiMap ) {						// 7.2.6.0 (2020/06/30)
			if( guiMap.isEmpty() ) { loadDBResource(); }
			return guiMap.values().toArray( new GUIData[guiMap.size()] );
		}
	}

	/**
	 * GUIData オブジェクトのキャッシュをクリアします。
	 *
	 * @og.rev 7.2.6.0 (2020/06/30) "**"以外にベースとなるSYSTEM_ID(RESOURCE_BASE_SYSTEM_ID)設定の対応
	 */
	public void clear() {
		synchronized( guiMap ) {						// 7.2.6.0 (2020/06/30)
			guiMap.clear();
		}
	}
}
