/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.io;

import java.io.File;
import java.util.function.Supplier;									// 8.0.0.2 (2021/10/15)

import org.opengion.fukurou.model.FileOperation;
import org.opengion.fukurou.model.FileOperationFactory;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.FileUtil;							// 8.0.0.2 (2021/10/15)
import org.opengion.hayabusa.common.HybsSystem;
// import org.opengion.fukurou.system.HybsConst;					// 5.10.9.0 (2019/03/01)

/**
 * クラウドを含むファイル操作クラスの生成
 *
 * システムリソース参照のみで、クラウドクラスを生成します。(8.0.0.1 (2021/10/08)で固定)
 * 引数付きの場合は、直接、org.opengion.fukurou.model.FileOperationFactory をご利用ください。
 *
 * 直接fukurouをCallしてもよいのですが、hayabusaからの呼び出しではシステムリソースを参照する必要があるため
 * ラッパー的にこのクラスを経由してCallする事でシステムリソースが使われるようにしておきます。
 * （タグ以外からも呼び出されるため、commonTagSupportではなく専用クラスをioパッケージに作成しています）
 *
 * ローカルのファイルを扱いたい場合は、pluginにDEFAULTを指定してください。
 *
 * @og.rev 5.10.8.0 (2019/02/01) 新規作成
 * @og.rev 8.0.0.1 (2021/10/08) 修正対応
 * @og.group
 *
 * @version 5.0
 * @author Takahashi Masakazu
 * @since JDK7.0
 */
public final class HybsFileOperationFactory {
	private static String PLUGIN = HybsSystem.sys("CLOUD_TARGET");
	private static String BUCKET = HybsSystem.sys("CLOUD_BUCKET");

	/**
	 * コンストラクタはprivate化しておきます。
	 */
	private HybsFileOperationFactory(){
		// コンストラクタ
	}

	/**
	 * fukurouのFileOperationFactoryを呼び出してFileOperationを取得します。
	 * plugin,buketを指定しない場合はシステムリソースを利用します。
	 *
	 * @param path ファイルパス
	 * @return FileOperationインスタンス
	 */
	public static FileOperation create( final String path ) {
		return FileOperationFactory.newStorageOperation( PLUGIN,BUCKET,path );
	}

	/**
	 * ディレクトリとファイル名を指定用です。
	 *
	 * @param dir ディレクトリパス
	 * @param fileName ファイル名
	 * @return FileOperationインスタンス
	 */
	public static FileOperation create( final String dir, final String fileName ) {
//		return create( dir + HybsConst.FS + file );
		return FileOperationFactory.newStorageOperation( PLUGIN,BUCKET,dir,fileName );		// 連結方法を統一します。
	}

	/**
	 * FileOperation(ディレクトリ)とファイル名を指定用です。
	 *
	 * @param dir ファイル(ディレクトリパス取得)
	 * @param fileName ファイル名
	 * @return FileOperationインスタンス
	 */
	public static FileOperation create( final File dir, final String fileName ) {
//		return create( dir.getPath() + HybsConst.FS + file );
		return create( new File( dir,fileName ).getPath() );
	}

	/**
	 * fukurouのFileOperationFactoryを呼び出してFileOperationを取得します。
	 * plugin,buketを指定しない場合はシステムリソースを利用します。
	 *
	 * @param plugin プラグイン名
	 * @param bucket バケット名
	 * @param path ファイルパス
	 * @return FileOperationインスタンス
	 */
	public static FileOperation create(final String plugin, final String bucket, final String path) {
		return FileOperationFactory.newStorageOperation(
				StringUtil.nval(plugin, PLUGIN), StringUtil.nval(bucket, BUCKET), path );
	}

	/**
	 * FileOperation(ディレクトリ)とファイル名を指定用です。
	 *
	 * @param plugin プラグイン名
	 * @param bucket バケット名
	 * @param dir ファイル(ディレクトリパス取得)
	 * @param filename ファイル名
	 * @return FileOperationインスタンス
	 */
	public static FileOperation create(final String plugin, final String bucket, final File dir, final String filename) {
		return create(plugin, bucket, new File( dir,filename ).getPath() );
	}

	/**
	 * ディレクトリとファイル名を指定用です。
	 *
	 * @param plugin プラグイン名
	 * @param bucket バケット名
	 * @param dir ディレクトリパス
	 * @param filename ファイル名
	 * @return FileOperationインスタンス
	 */
	public static FileOperation create(final String plugin, final String bucket, final String dir, final String filename) {
		return create(plugin, bucket, new File( dir,filename ).getPath() );
	}

	/**
	 * ｼｽﾃﾑ定数で、ｸﾗｳﾄﾞ設定されているかどうか
	 *
	 * ｼｽﾃﾑ定数の、CLOUD_TARGET か、CLOUD_BUCKET のどちらかが、
	 * null(ゼロ文字列、タブ、空白のみ)の場合、false です。
	 * それ以外は、true を返しますが、正常にｸﾗｳﾄﾞにアクセス出来る保証はありません。
	 * あくまで、ﾘｿｰｽ設定がされているかどうかのみで、判定しています。
	 *
	 * @og.rev 8.0.0.1 (2021/10/08) ｸﾗｳﾄﾞ修正
	 *
	 * @return ｸﾗｳﾄﾞ設定されていれば true
	 */
	public static boolean useCloud() {
		return ! StringUtil.isNull( PLUGIN,BUCKET );	// どれか一つでも null なら、false
	}

	/**
	 * ﾛｰｶﾙﾌｧｲﾙをｸﾗｳﾄﾞに移動後、ﾛｰｶﾙﾌｧｲﾙを削除します。
	 *
	 * ｸﾗｳﾄﾞ設定されている場合、指定のｻﾌﾟﾗｲﾔを実行してﾛｰｶﾙﾌｧｲﾙを取得し、
	 * ｸﾗｳﾄﾞにｺﾋﾟｰ後、ﾛｰｶﾙﾌｧｲﾙを削除します。
	 * ｸﾗｳﾄﾞ設定されていなければ、何もしません。
	 *
	 * @og.rev 8.0.0.2 (2021/10/15) ﾛｰｶﾙﾌｧｲﾙとｸﾗｳﾄﾞﾌｧｲﾙ間の移動
	 *
	 * @param supp ﾛｰｶﾙﾌｧｲﾙを生成するｻﾌﾟﾗｲﾔ
	 */
	public static void local2cloud( final Supplier<File> supp ) {
		if( useCloud() ) {
			final File localFile = supp.get();
			final FileOperation cloudFile = create( localFile.getPath() );
			FileUtil.copy( localFile, cloudFile );
			localFile.delete();
		}
	}

	/**
	 * ｸﾗｳﾄﾞﾌｧｲﾙをﾛｰｶﾙに移動後、ｸﾗｳﾄﾞﾌｧｲﾙを削除します。
	 *
	 * ｸﾗｳﾄﾞ設定されている場合、指定のｻﾌﾟﾗｲﾔを実行してﾛｰｶﾙﾌｧｲﾙを取得し、
	 * まず、ﾛｰｶﾙﾌｧｲﾙを削除後、ﾛｰｶﾙﾌｧｲﾙの親ﾌｫﾙﾀﾞを作成して、
	 * ｸﾗｳﾄﾞﾌｧｲﾙをﾛｰｶﾙに移動後、ｸﾗｳﾄﾞﾌｧｲﾙを削除します。
	 * ｸﾗｳﾄﾞ設定されていなければ、何もしません。
	 *
	 * @og.rev 8.0.0.2 (2021/10/15) ﾛｰｶﾙﾌｧｲﾙとｸﾗｳﾄﾞﾌｧｲﾙ間の移動
	 *
	 * @param supp ﾛｰｶﾙﾌｧｲﾙを生成するｻﾌﾟﾗｲﾔ
	 */
	public static void cloud2local( final Supplier<File> supp ) {
		if( useCloud() ) {
			final File localFile = supp.get();
			final FileOperation cloudFile = create( localFile.getPath() );

			localFile.delete();
			final File localParent = localFile.getParentFile();
			if( localParent != null ) { localParent.mkdirs(); }

			FileUtil.copy( cloudFile, localFile );
			cloudFile.delete();							// もしかして、移動完了前に削除してしまうかも？
		}
	}

	/**
	 * #create(String,String,String)を呼び出してFileOperationを取得します。
	 * plugin,buketを指定しない場合はシステムリソースを利用します。
	 *
	 * 引数をﾃﾞｨﾚｸﾄﾘとして処理します。
	 * ﾛｰｶﾙﾌｫﾙﾀﾞの場合、ﾃﾞｨﾚｸﾄﾘの作成、ﾃﾞｨﾚｸﾄﾘかどうかのﾁｪｯｸ、書き込みﾁｪｯｸを行います。
	 *
	 * @og.rev 8.0.0.2 (2021/10/15) ﾛｰｶﾙﾌｧｲﾙとｸﾗｳﾄﾞﾌｧｲﾙ間の移動
	 *
	 * @param plugin プラグイン名
	 * @param bucket バケット名
	 * @param path ファイルパス
	 * @return FileOperationインスタンス
	 * @throws IllegalArgumentException 引数のﾃﾞｨﾚｸﾄﾘが作成できない、ﾃﾞｨﾚｸﾄﾘでない、書き込めない場合
	 * @see	#create(String,String,String)
	 */
	public static FileOperation createDir(final String plugin, final String bucket, final String dir) {
		final FileOperation cloudDir = create( plugin,bucket,dir );

		if( !useCloud() ) {
			// ｾｰﾌﾞﾃﾞｨﾚｸﾄﾘ 作成
			if( ! cloudDir.exists() && ! cloudDir.mkdirs() ) {
				throw new IllegalArgumentException( "Not make directory: " + dir );
			}

			// Check saveDirectory is truly a directory
			if(!cloudDir.isDirectory()) {
				throw new IllegalArgumentException("Not a directory: " + dir);
			}

			// Check saveDirectory is writable
			if(!cloudDir.canWrite()) {
				throw new IllegalArgumentException("Not writable: " + dir);
			}
		}

		return cloudDir ;
	}
}
