/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.report2;

import org.opengion.fukurou.system.OgRuntimeException ;		// 6.4.2.0 (2016/01/29)
import java.io.File;

//import org.opengion.fukurou.util.StringUtil;				// 6.9.8.1 (2018/06/11) 7.4.4.0 (2021/06/30) Delete
import org.opengion.hayabusa.db.DBTableModel;
import static org.opengion.fukurou.system.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

/**
 * 帳票処理要求を管理するｷｭｰｵﾌﾞｼﾞｪｸﾄです｡
 * このｵﾌﾞｼﾞｪｸﾄでは､帳票の定義及びﾃﾞｰﾀと､処理中に発生したｴﾗｰﾒｯｾｰｼﾞを管理します｡
 * また､このｷｭｰを生成したｵﾌﾞｼﾞｪｸﾄもこのｵﾌﾞｼﾞｪｸﾄにｾｯﾄされます｡
 *
 * @og.group 帳票ｼｽﾃﾑ
 *
 * @version  4.0
 * @author   Hiroki.Nakamura
 * @since    JDK1.6
 */
public class ExecQueue {

	/** 実行方法 {@value} */
	protected static final String OUT_ODS_ONLY		= "1";		// 変換のみ
	/** 実行方法 {@value} */
	protected static final String OUT_PRINT_ONLY	= "2";		// (印刷のみ)
	/** 実行方法 {@value} */
	protected static final String OUT_ODS_PRINT		= "3";		// ﾌﾟﾘﾝﾀ出力
	/** 実行方法 {@value} */
	protected static final String OUT_ODS_PDF		= "P";		// PDF出力
	/** 実行方法 {@value} */
	protected static final String OUT_ODS_PRINT_PDF	= "Q";		// PDF+印刷
	/** 実行方法 {@value} */
	protected static final String OUT_ODS_EXCEL		= "E";		// Excel出力
	/** 実行方法 {@value} */
	protected static final String OUT_ODS_XSLX		= "F";		// 8.0.1.0 (2021/10/29) Excel2007形式
	/** 実行方法 {@value} */
	protected static final String OUT_ODS_ODS		= "S";		// ODS出力 4.3.3.4 (2008/11/01) 追加

	// 7.4.4.0 (2021/06/30) openGionV8事前準備(EXCEL取込廃止､RFID出力廃止)
	// /** 実行方法 {@value} */
	// protected static final String IN_INPUT_ONLY	= "5";		// (取込のみ)
	// /** 実行方法 {@value} */
	// protected static final String IN_EXEC_ONLY	= "6";		// (PG起動)
	// /** 実行方法 {@value} */
	// protected static final String IN_INPUT_EXEC	= "7";		// 取込･PG起動
	// /** 実行方法 {@value} */
	// protected static final String RFID_PRINT		= "A";		// RFID(通常印刷)
	// /** 実行方法 {@value} */
	// protected static final String RFID_ALLPRINT	= "B";		// RFID(全印刷)
	// /** 実行方法 {@value} */
	// protected static final String RFID_ALLERASE	= "C";		// RFID(全消去)
	// /** 実行方法 {@value} */
	// protected static final String RFID_SEQERASE	= "D";		// RFID(連続消去)

	//	// 8.0.1.0 (2021/10/29) CSV出力対応(ﾕﾘﾆﾀ帳票)は､一旦保留にします｡
	//	// 5.9.0.0 (2015/09/04) CSV出力対応
	//	/** 実行方法 {@value} */
	//	protected static final String CSV_PRINT			= "G";
	//	/** 実行方法 {@value} */
	//	protected static final String CSV_PRINT_EXCEL	= "H";
	//	/** 実行方法 {@value} */
	//	protected static final String CSV_PRINT_PDF		= "I";
	//	/** 実行方法 {@value} */
	//	protected static final String CSV_PRINT_EXCEL2	= "J";		// 5.9.4.2 (2016/01/15) EXCEL2追加

	/** 最大ｼｰﾄ数 {@value} */
//	protected static final int MAX_SHEETS_PER_FILE	= 256;		// 5.1.2.0 (2010/01/01)
	protected static final int MAX_SHEETS_PER_FILE	= 999;		// 7.1.0.0 (2020/01/27) 256ｼｰﾄ制限を999にしておく｡

	private String		ykno		;
	private String		systemId	;
	private DBTableModel body		;
	private DBTableModel header		;
	private DBTableModel footer		;
	private String		listId		;
	private String		pdfPasswd	;
	private String		lang		;
	private String		threadId	;
	private String		templateName;
	private String		outputType	;
	private String		printerName	;
	private String		outputName	;
	private boolean		fglocal		;
	private boolean		fgcut		;
	private QueueManager manager	;
	private String		prgdir		;		// 4.3.3.0 (2008/10/01) 板金RFID対応｡
	private String		prgfile		;		// 4.3.3.0 (2008/10/01)
	private String		prtid		;		// 4.3.3.0 (2008/10/01)

	private String		grpid		;		// 5.9.2.2 (2015/11/20)
	private String		dmngrp		;		// 5.9.2.2 (2015/11/20)
	private int			pageCnt		;		// 5.1.2.0 (2010/01/01) 処理したﾍﾟｰｼﾞ数
	private int			rowCnt		;		// 5.1.2.0 (2010/01/01) 処理した行数
	private boolean		isDataEnd	;		// 5.1.2.0 (2010/01/01) 全ﾃﾞｰﾀが処理されたか (ﾒｿｯﾄﾞ名と同じ変数名変更)

	private boolean		useSheetName;		// 5.7.6.2 (2014/05/16) PAGEBREAKｶﾗﾑの値を､ｼｰﾄ名として使うかどうか｡
//	private String		fgnoml		= "0";	// 5.10.0.0 (2018/06/08) ﾒｰﾙ不要ﾌﾗｸﾞ
	//private boolean	fgnoml		;		// 6.9.8.1 (2018/06/11) 帳票ｴﾗｰﾒｰﾙの改修(ﾒｰﾙ不要ﾌﾗｸﾞ=trueで不要､falseが必要) 7.4.4.0 (2021/06/30) Delete

	private final StringBuilder errMsg = new StringBuilder( BUFFER_MIDDLE );

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰ
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public ExecQueue() { super(); }		// これも､自動的に呼ばれるが､空のﾒｿｯﾄﾞを作成すると警告されるので､明示的にしておきます｡

	/**
	 * 要求NOをｾｯﾄします｡
	 *
	 * @param ykno 要求NO
	 */
	public void setYkno( final String ykno ) {
		this.ykno = ykno;
	}

	/**
	 * 要求NOを取得します｡
	 *
	 * @return	要求NO
	 */
	public String getYkno() {
		return ykno;
	}

	/**
	 * ｼｽﾃﾑIDをｾｯﾄします｡
	 *
	 * @param systemId ｼｽﾃﾑID
	 */
	public void setSystemId( final String systemId ) {
		this.systemId = systemId;
	}

	/**
	 * ｼｽﾃﾑIDを取得します｡
	 *
	 * @return  StringｼｽﾃﾑID
	 */
	public String getSystemId() {
		return systemId;
	}

	/**
	 * ﾎﾞﾃﾞｨｰ部分のDBTableModelをｾｯﾄします｡
	 *
	 * @param body DBTableModelｵﾌﾞｼﾞｪｸﾄ
	 */
	public void setBody( final DBTableModel body ) {
		this.body = body;
	}

	/**
	 * ﾎﾞﾃﾞｨｰ部分のDBTableModelを取得します｡
	 *
	 * @return	ﾎﾞﾃﾞｨｰ部分のDBTableModelｵﾌﾞｼﾞｪｸﾄ
	 */
	public DBTableModel getBody() {
		return body;
	}

	/**
	 * ﾍｯﾀﾞｰ部分のDBTableModelをｾｯﾄします｡
	 *
	 * @param header DBTableModelｵﾌﾞｼﾞｪｸﾄ
	 */
	public void setHeader( final DBTableModel header ) {
		this.header = header;
	}

	/**
	 * ﾍｯﾀﾞｰ部分のDBTableModelを取得します｡
	 *
	 * @return	ﾍｯﾀﾞｰ部分のDBTableModelｵﾌﾞｼﾞｪｸﾄ
	 */
	public DBTableModel getHeader() {
		return header;
	}

	/**
	 * ﾌｯﾀｰ部分のDBTableModelをｾｯﾄします｡
	 *
	 * @param footer DBTableModelｵﾌﾞｼﾞｪｸﾄ
	 */
	public void setFooter( final DBTableModel footer ) {
		this.footer = footer;
	}

	/**
	 * ﾌｯﾀｰ部分のDBTableModelを取得します｡
	 *
	 * @return	ﾌｯﾀｰ部分のDBTableModelｵﾌﾞｼﾞｪｸﾄ
	 */
	public DBTableModel getFooter() {
		return footer;
	}

	/**
	 * 帳票IDをｾｯﾄします｡
	 *
	 * @param listId 帳票ID
	 */
	public void setListId( final String listId ) {
		this.listId = listId;
	}

	/**
	 * 帳票IDを取得します｡
	 *
	 * @return 帳票ID
	 */
	public String getListId() {
		return listId;
	}

	/**
	 * PDFﾊﾟｽﾜｰﾄﾞをｾｯﾄします｡
	 *
	 * @param pdfPasswd PDFﾊﾟｽﾜｰﾄﾞ
	 */
	public void setPdfPasswd( final String pdfPasswd ) {
		this.pdfPasswd = pdfPasswd;
	}

	/**
	 * PDFﾊﾟｽﾜｰﾄﾞを取得します｡
	 *
	 * @return PDFﾊﾟｽﾜｰﾄﾞ
	 */
	public String getPdfPasswd() {
		return pdfPasswd;
	}

	/**
	 * 言語をｾｯﾄします｡
	 *
	 * @param lang 言語
	 */
	public void setLang( final String lang ) {
		this.lang = lang;
	}

	/**
	 * 言語を取得します｡
	 *
	 * @return 言語
	 */
	public String getLang() {
		return lang;
	}

	/**
	 * 雛形ﾌｧｲﾙ名をｾｯﾄします｡
	 *
	 * @param templateName 雛形ﾌｧｲﾙ名
	 */
	public void setTemplateName( final String templateName ) {
		this.templateName = templateName;
	}

	/**
	 * 雛形ﾌｧｲﾙ名を取得します｡
	 *
	 * @return 帳票雛形ﾌｧｲﾙ名
	 */
	public String getTemplateName() {
		return templateName;
	}

	/**
	 * 実行方法をｾｯﾄします｡
	 *
	 * @param outputType 実行方法
	 */
	public void setOutputType( final String outputType ) {
		this.outputType = outputType;
	}

	/**
	 * 出力ﾀｲﾌﾟを取得します｡
	 *
	 * @return 出力ﾀｲﾌﾟ
	 */
	public String getOutputType() {
		return outputType;
	}

	/**
	 * ﾌﾟﾘﾝﾀｰ名をｾｯﾄします｡
	 *
	 * @param  printerName ﾌﾟﾘﾝﾀｰ名
	 */
	public void setPrinterName( final String printerName ) {
		this.printerName = printerName;
	}

	/**
	 * ﾌﾟﾘﾝﾀｰ名を取得します｡
	 *
	 * @return ﾌﾟﾘﾝﾀ名
	 */
	public String getPrinterName() {
		return printerName;
	}

	/**
	 * 処理要求を処理するｽﾚｯﾄﾞIDをｾｯﾄします｡
	 *
	 * @param threadId ｽﾚｯﾄﾞID
	 */
	public void setThreadId( final String threadId ) {
		this.threadId = threadId;
	}

	/**
	 * 処理要求を処理するｽﾚｯﾄﾞIDを取得します｡
	 *
	 * @return ｽﾚｯﾄﾞID
	 */
	public String getThreadId() {
		return threadId;
	}

	/**
	 * 出力ﾌｧｲﾙ名をｾｯﾄします｡
	 *
	 * @param outputName 出力ﾌｧｲﾙ名
	 */
	public void setOutputName( final String outputName ) {
		this.outputName = outputName;
	}

	/**
	 * 出力ﾌｧｲﾙ名を設定します｡
	 * GE50に設定されていない場合は第四引数(要求番号)を利用する｡
	 * その場合､ﾀｲﾌﾟに応じた拡張子が自動設定される｡
	 *
	 * ".xls" : OUT_ODS_EXCEL
	 * ".xlsx" : OUT_ODS_XSLX			// 8.0.1.0 (2021/10/29)
	 * ".pdf" : OUT_ODS_PDF , OUT_ODS_PRINT_PDF
	 * ".ods" : OUT_ODS_ODS
// * ".xml" : RFID_PRINT , RFID_ALLPRINT , RFID_ALLERASE , RFID_SEQERASE 7.4.4.0 (2021/06/30) Delete
	 * ".csv" : CSV_PINT , CSV_PRINT_EXCEL , CSV_PRINT_PDF
	 *
	 * @og.rev 4.3.3.4 (2008/11/01) ODS出力対応
	 * @og.rev 5.4.3.0 (2011/12/26) RFIDﾃﾞﾌｫﾙﾄ対応
	 * @og.rev 5.4.4.1 (2012/02/03) RFID拡張子変更
	 * @og.rev 5.9.0.0 (2015/09/04) CSV対応
	 * @og.rev 7.4.4.0 (2021/06/30) openGionV8事前準備(RFID出力廃止)
	 * @og.rev 8.0.1.0 (2021/10/29) CSV出力対応(ﾕﾘﾆﾀ帳票)は､一旦保留にします｡
	 * @og.rev 8.0.1.0 (2021/10/29) OUT_ODS_XSLX (Excel2007形式) 評価用
	 *
	 * @param	outputDir	出力ﾃﾞｨﾚｸﾄﾘ名
	 * @param	outputFile	出力ﾌｧｲﾙ名
	 * @param	type		ﾀｲﾌﾟ
	 * @param	yokyu		要求番号(ﾌｧｲﾙ名が指定されていない場合のﾌｧｲﾙ名)
	 *
	 */
	public void setOutputName( final String outputDir, final String outputFile, final String type, final String yokyu ){
		final StringBuilder filePath = new StringBuilder( BUFFER_MIDDLE );
		filePath.append( outputDir ).append( File.separator );		// 6.0.2.5 (2014/10/31) char を append する｡

		if( outputFile == null || outputFile.isEmpty() ){ // ﾌｧｲﾙ名が指定されていない場合は要求番号を利用する｡
			if( OUT_ODS_EXCEL.equals( type ) ){
				filePath.append( yokyu );
				filePath.append( ".xls" );
			}
			else if( OUT_ODS_XSLX.equals( type ) ){			// 8.0.1.0 (2021/10/29)
				filePath.append( yokyu );
				filePath.append( ".xlsx" );
			}
			else if( OUT_ODS_PDF.equals( type ) || OUT_ODS_PRINT_PDF.equals( type ) ){
				filePath.append( yokyu );
				filePath.append( ".pdf" );
			}
			// 4.3.3.4 (2008/11/01) 追加
			else if( OUT_ODS_ODS.equals ( type ) ){
				filePath.append( yokyu );
				filePath.append( ".ods" );
			}
			// 5.4.3.0 (2011/12/26) 追加
			// 5.4.4.2 (2012/02/03) .txtではなく.xml
			// 7.4.4.0 (2021/06/30) Delete
			//else if( ExecQueue.RFID_PRINT.equals( type ) || ExecQueue.RFID_ALLPRINT.equals( type )
			//		|| ExecQueue.RFID_ALLERASE.equals( type ) || ExecQueue.RFID_SEQERASE.equals( type ) ) {
			//	filePath.append( yokyu );
			//	filePath.append( ".xml" ); //txt-xml
			//}
			// 5.9.0.0 (2015/09/04) 追加
			// 5.9.4.2 (2016/01/13) EXCEL2追加
			// 8.0.1.0 (2021/10/29) CSV出力対応(ﾕﾘﾆﾀ帳票)は､一旦保留にします｡
		//	else if( ExecQueue.CSV_PRINT.equals( type ) || ExecQueue.CSV_PRINT_EXCEL.equals( type )
		//			|| ExecQueue.CSV_PRINT_PDF.equals( type ) || ExecQueue.CSV_PRINT_EXCEL2.equals( type ) ) {
		//		filePath.append( yokyu );
		//		filePath.append( ".csv" );
		//	}
		}
		else {
			filePath.append( outputFile );
		}

		this.outputName = filePath.toString();
	}

	/**
	 * 出力ﾌｧｲﾙ名を取得します｡
	 *
	 * @og.rev 5.1.2.0 (2010/01/01) 256ｼｰﾄを超える場合に対応｡2ﾌｧｲﾙ目以降は､_1､_2･･･をﾌｧｲﾙ名の後ろにつける
	 * @og.rev 6.3.9.0 (2015/11/06) ｺﾝｽﾄﾗｸﾀで初期化されていないﾌｨｰﾙﾄﾞを null ﾁｪｯｸなしで利用している(findbugs)
	 *
	 * @return 出力先ﾌｧｲﾙ名
	 * @og.rtnNotNull
	 */
	public String getOutputName() {
		// 6.3.9.0 (2015/11/06) ｺﾝｽﾄﾗｸﾀで初期化されていないﾌｨｰﾙﾄﾞを null ﾁｪｯｸなしで利用している(findbugs)
		if( outputName == null ) {
			final String errMsg = "#setOutputName(String) または､#setOutputName(String,String,String,String)を先に実行しておいてください｡" ;
			throw new OgRuntimeException( errMsg );
		}

		if( pageCnt <= MAX_SHEETS_PER_FILE ) {
			return outputName;
		}
		else {
			final StringBuilder fileName = new StringBuilder( BUFFER_MIDDLE );

			final int idx = outputName.lastIndexOf( '.' );
			final String name = outputName.substring( 0, idx );
			final String suffix = outputName.substring( idx );
			final int addNo = (int)Math.ceil( (double)pageCnt/(double)MAX_SHEETS_PER_FILE ) - 1;

			fileName.append( name ).append( '_' ).append( addNo ).append( suffix );		// 6.0.2.5 (2014/10/31) char を append する｡

			return fileName.toString();
		}
	}

	/**
	 * 実行ﾌｧｲﾙﾃﾞｨﾚｸﾄﾘを指定します｡
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 板金RFID対応
	 *
	 * @param dir ﾃﾞｨﾚｸﾄﾘ
	 */
	public void setPrgDir( final String dir ) {
		this.prgdir = dir;
	}

	/**
	 * 実行ﾌｧｲﾙﾃﾞｨﾚｸﾄﾘを取得します｡
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 板金RFID対応
	 *
	 * @return ﾌﾟﾛｸﾞﾗﾑﾃﾞｨﾚｸﾄﾘ
	 */
	public String getPrgDir() {
		return prgdir;
	}

	/**
	 * 実行ﾌｧｲﾙ名をｾｯﾄします｡
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 板金RFID対応
	 * @param	file	ﾌｧｲﾙ名
	 */
	public void setPrgFile( final String file ) {
		this.prgfile = file;
	}

	/**
	 * 実行ﾌｧｲﾙ名を取得します｡
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 板金RFID対応
	 *
	 * @return ﾌﾟﾛｸﾞﾗﾑﾌｧｲﾙ名
	 */
	public String getPrgFile() {
		return prgfile;
	}

	/**
	 * ﾌﾟﾘﾝﾀIDをｾｯﾄします｡
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 板金RFID対応
	 * @param	id	ﾌﾟﾘﾝﾀID
	 */
	public void setPrtId( final String id ) {
		this.prtid = id;
	}

	/**
	 * ﾌﾟﾘﾝﾀIDを取得します｡
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 板金RFID対応
	 *
	 * @return ﾌﾟﾘﾝﾀID
	 */
	public String getPrtId() {
		return prtid;
	}

	/**
	 * ｸﾞﾙｰﾌﾟIDをｾｯﾄします｡
	 *
	 * @og.rev 5.9.2.2 (2015/11/20)
	 *
	 * @param	id	ｸﾞﾙｰﾌﾟID
	 */
	public void setGrpId( final String id ) {
		this.grpid = id;
	}

	/**
	 * ｸﾞﾙｰﾌﾟIDを取得します｡
	 *
	 * @og.rev 5.9.2.2 (2015/11/20)
	 *
	 * @return ｸﾞﾙｰﾌﾟID
	 */
	public String getGrpId() {
		return grpid;
	}

	/**
	 * ﾃﾞｰﾓﾝｸﾞﾙｰﾌﾟをｾｯﾄします｡
	 *
	 * @og.rev 5.9.2.2 (2015/11/20)
	 *
	 * @param	name	ﾃﾞｰﾓﾝｸﾞﾙｰﾌﾟ
	 */
	public void setDmnGrp( final String name ) {
		this.dmngrp = name;
	}

	/**
	 * ﾃﾞｰﾓﾝｸﾞﾙｰﾌﾟを取得します｡
	 *
	 * @og.rev 5.9.2.2 (2015/11/20)
	 *
	 * @return ﾃﾞｰﾓﾝｸﾞﾙｰﾌﾟ
	 */
	public String getDmnGrp() {
		return dmngrp;
	}

	/**
	 * ﾛｰｶﾙﾘｿｰｽ使用ﾌﾗｸﾞをｾｯﾄします(初期値:false)｡
	 *
	 * @param	fglocal	ﾛｰｶﾙﾘｿｰｽ使用ﾌﾗｸﾞ[true:使用する/false:使用しない]
	 */
	public void setFglocal( final boolean fglocal ) {
		this.fglocal = fglocal;
	}

	/**
	 * ﾛｰｶﾙﾘｿｰｽ使用ﾌﾗｸﾞを取得します｡
	 *
	 * @return ﾛｰﾙﾘｿｰｽ使用ﾌﾗｸﾞ[true:使用する/false:使用しない]
	 */
	public boolean isFglocal() {
		return fglocal;
	}

	/**
	 * ﾍﾟｰｼﾞｴﾝﾄﾞｶｯﾄﾌﾗｸﾞをｾｯﾄします(初期値:false)｡
	 *
	 * @param fgcut ﾍﾟｰｼﾞｴﾝﾄﾞｶｯﾄの使用可否[true:使用/false:通常]
	 */
	public void setFgcut( final boolean fgcut ) {
		this.fgcut = fgcut;
	}

	/**
	 * ﾍﾟｰｼﾞｴﾝﾄﾞｶｯﾄﾌﾗｸﾞを取得します｡
	 *
	 * @return ﾍﾟｰｼﾞｴﾝﾄﾞｶｯﾄﾌﾗｸﾞ
	 */
	public boolean isFgcut() {
		return fgcut;
	}

	/**
	 * PAGEBREAKｶﾗﾑの値を､ｼｰﾄ名として使うかどうかをｾｯﾄします(初期値:false)｡
	 *
	 * @og.rev 5.7.6.2 (2014/05/16) 新規追加
	 *
	 * @param useSheetName PAGEBREAKｶﾗﾑのｼｰﾄ名使用可否[true:使用/false:使用しない]
	 */
	public void setUseSheetName( final boolean useSheetName ) {
		this.useSheetName = useSheetName;
	}

	/**
	 * PAGEBREAKｶﾗﾑの値を､ｼｰﾄ名として使うかどうかを取得します｡
	 *
	 * @og.rev 5.7.6.2 (2014/05/16) 新規追加
	 *
	 * @return PAGEBREAKｶﾗﾑのｼｰﾄ名使用可否[true:使用/false:使用しない]
	 */
	public boolean isUseSheetName() {
		return useSheetName;
	}

	/**
	 * ｷｭｰﾏﾈｰｼﾞｬｰをｾｯﾄします｡
	 *
	 * @param manager ｷｭｰﾏﾈｰｼﾞｬｰ
	 */
	public void setManager( final QueueManager manager ) {
		this.manager = manager;
	}

	/**
	 * 帳票処理ﾃﾞｰﾀをｾｯﾄします｡
	 * 既にﾃｰﾌﾞﾙﾓﾃﾞﾙがｾｯﾄされている場合は､再ｾｯﾄしません｡
	 *
	 */
	public void setData() {
		if( body == null && manager != null ) {
			manager.set( this );
		}
	}

	/**
	 * ｷｭｰを実行中の状態に更新します｡
	 *
	 */
	public void setExecute() {
		if( manager != null ) {
			manager.execute( this );
		}
	}

	/**
	 * ｷｭｰを完了済の状態に更新します｡
	 *
	 */
	public void setComplete() {
		if( manager != null ) {
			manager.complete( this );
		}
	}

	/**
	 * ｷｭｰをｴﾗｰの状態に更新します｡
	 */
	public void setError() {
		if( manager != null ) {
			manager.error( this );
		}
	}

	/**
	 * ｴﾗｰﾒｯｾｰｼﾞをｾｯﾄします｡
	 *
	 * @param msg ｴﾗｰﾒｯｾｰｼﾞ
	 */
	public void addMsg( final String msg ) {
		errMsg.append( msg );
	}

	/**
	 * ｴﾗｰﾒｯｾｰｼﾞを取得します｡
	 *
	 * @return ｴﾗｰﾒｯｾｰｼﾞ
	 * @og.rtnNotNull
	 */
	public String getMsg() {
		return errMsg.toString();
	}

	/**
	 * 処理したﾍﾟｰｼﾞ数を引数の分だけｶｳﾝﾄｱｯﾌﾟします｡
	 *
	 * これは､256ｼｰﾄ以上を印刷する場合､ﾌｧｲﾙ名に"_"を付けて
	 * 出力する機能と､GE50のSUPAGEｶﾗﾑに印刷ﾍﾟｰｼﾞ数をｾｯﾄするのに
	 * 使用しています｡
	 * openOfficeのｼｰﾄ制限は解除されていますが､SUPAGEｶﾗﾑが
	 * NUMBER(3) でDB定義されているため､1000以上のﾍﾟｰｼﾞを､GE50 に
	 * ｾｯﾄすると､ｴﾗｰになります｡
	 * DB定義を変更するまでの間､最大ﾍﾟｰｼﾞ数を､999 としておきます｡
	 *
	 * @og.rev 5.1.2.0 (2010/01/01) 新規追加
	 * @og.rev 7.1.0.0 (2020/01/27) 256ｼｰﾄ制限を999にしておく｡
	 *
	 * @param pgs ｶｳﾝﾄｱｯﾌﾟするﾍﾟｰｼﾞ数
	 */
	public void addExecPageCnt( final int pgs ) {
		pageCnt += pgs;

		// 7.1.0.0 (2020/01/27) 256ｼｰﾄ制限を999にしておく｡
		if( pageCnt > MAX_SHEETS_PER_FILE ) { pageCnt = MAX_SHEETS_PER_FILE; }
	}

	/**
	 * 処理したﾍﾟｰｼﾞ数を返します｡
	 *
	 * @og.rev 5.1.2.0 (2010/01/01) 新規追加
	 *
	 * @return 処理したﾍﾟｰｼﾞ数
	 */
	public int getExecPagesCnt() {
		return pageCnt;
	}

	/**
	 * 処理した行数をｾｯﾄします｡
	 *
	 * @og.rev 5.1.2.0 (2010/01/01) 新規追加
	 *
	 * @param rws 処理した行数
	 */
	public void setExecRowCnt( final int rws ) {
		rowCnt = rws;
	}

	/**
	 * 処理した行数を返します｡
	 *
	 * @og.rev 5.1.2.0 (2010/01/01) 新規追加
	 *
	 * @return 処理した行数
	 */
	public int getExecRowCnt() {
		return rowCnt;
	}

	/**
	 * 全ての行が処理されたかをｾｯﾄします(初期値:false)｡
	 *
	 * これは､処理結果が､256ｼｰﾄを超えていた場合､再度残りのﾃﾞｰﾀについて
	 * 処理を行うかどうかの判定するために､利用します｡
	 *
	 * @og.rev 5.1.2.0 (2010/01/01) 新規追加
	 *
	 * @param flag 全ての行が処理されたか
	 */
	public void setEnd( final boolean flag ) {
		isDataEnd = flag;
	}

	/**
	 * 全ての行が処理されているかを返します｡
	 *
	 * これは､処理結果が､256ｼｰﾄを超えていた場合､再度残りのﾃﾞｰﾀについて
	 * 処理を行うかどうかの判定するために､利用します｡
	 *
	 * @og.rev 5.1.2.0 (2010/01/01) 新規追加
	 *
	 * @return 全ての行が処理されたか
	 */
	public boolean isEnd() {
		return isDataEnd;
	}

	// /**
	// * FGNOMI(ﾒｰﾙ不要ﾌﾗｸﾞ)を設定します(初期値=false:送信する)｡
	// *
	// * 帳票ｴﾗｰ時のﾒｰﾙ送信は帳票IDの正規表現でｺﾝﾄﾛｰﾙ可能でしたが､
	// * ID単位に未送信ﾌﾗｸﾞを持たせてｺﾝﾄﾛｰﾙ可能にします｡
	// * 帳票の動作にDBのｶﾗﾑ追加が伴うためﾊﾞｰｼﾞｮﾝを5.10系に上げます｡
	// * ｴﾗｰﾒｰﾙが不要の場合は､"true", もしくは､１文字の場合のみ､"0" 以外の場合が､
	// * ﾒｰﾙ不要として､ｴﾗｰﾒｰﾙを送信しません｡
	// *
	// * @og.rev 5.10.0.0 (2018/06/08) 新規追加
	// * @og.rev 6.9.8.1 (2018/06/11) 帳票ｴﾗｰﾒｰﾙの改修(ﾒｰﾙ不要ﾌﾗｸﾞ=trueで不要､falseが必要)
	// * @og.rev 7.4.4.0 (2021/06/30) openGionV8事前準備(GE54.FGNOML廃止)
	// *
//	// * @param fgnoml
	// * @param flag ﾒｰﾙ不要ﾌﾗｸﾞ
	// */
//	// public void setFgnoml( final String fgnoml) {
	// 7.4.4.0 (2021/06/30) Delete
	// public void setFgnoml( final String flag ) {
//	// 	this.fgnoml = fgnoml;
	// 	fgnoml = StringUtil.nval( flag,fgnoml );		// true,TRUE,0以外の１文字の場合､true､それ以外は､初期値
	// }

	// /**
	// * FGNOMI(ﾒｰﾙ不要ﾌﾗｸﾞ)を取得します｡
	// *
	// * @og.rev 5.10.0.0 (2018/06/08) 新規追加
	// * @og.rev 6.9.8.1 (2018/06/11) 帳票ｴﾗｰﾒｰﾙの改修(ﾒｰﾙ不要ﾌﾗｸﾞ=trueで不要､falseが必要)
	// * @og.rev 7.4.4.0 (2021/06/30) openGionV8事前準備(GE54.FGNOML廃止)
	// *
//	// * @return fgnomi
	// * @return ﾒｰﾙ不要ﾌﾗｸﾞ [true:ﾒｰﾙ不要/false:ｴﾗｰﾒｰﾙ送信する]
	// */
//	// public String getFgnoml() {
	// 7.4.4.0 (2021/06/30) Delete
	// public boolean isFgnoml() {
//	// 	return this.fgnoml;
	// 	return fgnoml;
	// }
}
