package gg11;

// import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.business.BizLogic_TABLE;
import org.opengion.fukurou.model.DataModel;

import java.util.List;
import org.opengion.fukurou.fileexec.DBUtil;
import org.opengion.fukurou.fileexec.StringUtil;
import org.opengion.fukurou.util.HybsDateUtil;
import org.opengion.fukurou.db.Transaction;

/**
 * GG11(予定実績ﾃｰﾌﾞﾙ)に対してﾃﾞｰﾀ更新、追加処理を行います。
 *
 * GG11を検索して、予約の更新(UPDATE)と次回予約の登録(INSETRT)を行います。
 *
 * 予約データが存在しない場合は、初期ﾃﾞｰﾀをINSERTする。
 * 予約データが存在すれば、既存行の更新と次回予約のINSERT
 *
 * @version 7.0
 * @author T.OTA
 * @since JDK11

 */
public class GG0100A extends BizLogic_TABLE {
	private static final String SELECT = "select DY_HATU,NEXTDY,USRYOTE,BG_KEY,VIEW_XY,UNIQ from GG11"
										+ " where DY_YOTE=? and SET_KEY=? and SET_GRP=?";

	private static final String UPDATE = "update GG11 set CDJISI='1',DY_HATU=?,JUDG=?,USRJISI=?"
										+ " where UNIQ=?";

	private static final String INSERT = "insert into GG11"
										+ " (DY_YOTE,SET_KEY,SET_GRP,NEXTDY,CDJISI,JUDG,USRYOTE,USRJISI,BG_KEY,VIEW_XY)"
										+ " values (?,?,?,?,?,?,?,?,?,?)";

	private static final int DY_HATU = 0;
	private static final int NEXTDY  = 1;
	private static final int USRYOTE = 2;
	private static final int BG_KEY  = 3;
	private static final int VIEW_XY = 4;
	private static final int UNIQ    = 5;

	/**
	 * 処理のメインロジックの前処理を記述します。
	 *
	 * ループを回すかどうかを指定する #setUseLoop(boolean) で、回さないを指定します。
	 *
	 * @og.rev 6.8.5.0 (2018/01/09) ループを回すかどうかを引数で指定します。
	 */
	@Override
	protected void init() {
		setUseLoop( false );
	}

	/**
	 * 最後の行の終わりで呼び出し
	 *
	 * @return 処理が正常終了したか
	 */
	@Override
	protected boolean last() {
		final DataModel<String> table = getTable();

		final int dyNo  = table.getColumnNo( "DY_YOTE" );	// 予定日
		final int keyNo = table.getColumnNo( "SET_KEY" );	// 登録ｷｰ
		final int grpNo = table.getColumnNo( "SET_GRP" );	// 設定ｸﾞﾙｰﾌﾟ

		final String judg    = var( "JUDG" );
		final String usrjisi = var( "USRJISI" );

		final String[] tbls = table.getValues( 0 );			// 最初の行の値を取得。どの行も同じ値
		final String yote = tbls[dyNo];
		final String key  = tbls[keyNo];
		final String grp  = tbls[grpNo];

		// DBUtil が、ﾃﾞﾌｫﾙﾄでは、Tomcatの起動時の環境変数から、DataSource を作っているため。
		final Transaction tran = getTransaction();

		// GG11を検索して、予約の更新(UPDATE)と次回予約の登録(INSETRT)を行う。

		// select … from GG11 where DY_YOTE=? and SET_KEY=? and SET_GRP=?
		final String[] wh = new String[] { yote,key,grp };
		final List<String[]> rowData = DBUtil.dbQuery( tran,SELECT , wh );	// NEXTDY,USRYOTE,BG_KEY,VIEW_XY,UNIQ

		final String today = StringUtil.getTimeFormat( "yyyyMMdd" );

		if( rowData.isEmpty() || rowData.get(0).length == 0 ) {
			// 予約データが存在しない場合は、初期ﾃﾞｰﾀをINSERTする。
			// (DY_YOTE,SET_KEY,SET_GRP,NEXTDY,CDJISI,JUDG,USRYOTE,USRJISI,BG_KEY,VIEW_XY)
			final String[] in0 = new String[] { today,key,grp,"0","1",judg,"",usrjisi,"","" };
			DBUtil.execute( tran,INSERT , in0 );	// １行だけ処理
		}
		else {
			// 予約データが存在すれば、既存行の更新と次回予約のINSERT
			for( final String[] row : rowData ) {
				// DY_HATU は、null(または空文字)の場合のみ、本日を登録
				final String dyhatu = row[DY_HATU] == null || row[DY_HATU].isEmpty() ? today : row[DY_HATU] ;

				// update GE11 set CDJISI='1',DY_HATU=?,JUDG=?,USRJISI=? where UNIQ=?
				final String[] up = new String[] { dyhatu,judg,usrjisi,row[UNIQ] };
				DBUtil.execute( tran,UPDATE , up );	// １行づつ処理

				// INSERT(次回予約)は、NEXTDY が、"0" 以外で、ﾃﾞｰﾀが存在しない場合のみ
				if( !"0".equals( row[NEXTDY] ) ) {
					final String nextDy = nextDay( row[NEXTDY],yote,today );

					final String[] wh2 = new String[] { nextDy,key,grp };
					final List<String[]> nxData = DBUtil.dbQuery( tran,SELECT , wh2 );	// NEXTDY,USRYOTE,BG_KEY,VIEW_XY,UNIQ
					if( nxData.isEmpty() || nxData.get(0).length == 0 ) {
						// (DY_YOTE,SET_KEY,SET_GRP,NEXTDY,CDJISI,JUDG,USRYOTE,USRJISI,BG_KEY,VIEW_XY)
						final String[] in1 = new String[] { nextDy,key,grp,row[NEXTDY],"0","0",row[USRYOTE],usrjisi,row[BG_KEY],row[VIEW_XY] };
						DBUtil.execute( tran,INSERT , in1 );	// １行づつ処理
					}
				}
			}
		}

		return true;
	}

	/**
	 * 次回の予定日の指定ｷｰﾜｰﾄﾞと、予定日、発生日から次回の予定日を計算します。
	 *
	 * NEXTDY
	 * 0:単発 …次回予定なし(空文字列を返す)
	 * D1:日次 D7:週次 M1:月次 M2:2か月後 M3:3か月後 Y1:1年後
	 * 　… 予定日からの次回予定日算出
	 *  N+上記記号(ND1:翌日 NM1:1か月後 …)
	 * 　… 発生日からの次回予定日算出
	 *
	 * @param nextdy 次回の予定日の指定ｷｰﾜｰﾄﾞ
	 * @param dyYote 今回の予定日
	 * @param dyHatu 今回の発生日(基本的には本日)
	 * @return 次回の予定日
	 */
	private String nextDay( final String nextdy ,final String dyYote ,final String dyHatu ) {
		if( "0".equals( nextdy ) ) {	// 0:単発 の場合は、空文字列を返す
			return "" ;
		}
		// N で始まる場合は、発生日基準での次回予定日の加算を行います。
		else if( 'N' == nextdy.charAt(0) ) {
			return HybsDateUtil.getDateFormat( "YMD",dyHatu,nextdy.substring(1),0 );
		}
		else {
			return HybsDateUtil.getDateFormat( "YMD",dyYote,nextdy,0 );
		}
	}
}
