/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.ToString;

import static org.opengion.fukurou.util.StringUtil.nval;

/**
 * IorQueryTag にパラメーターを渡す為のタグクラスです。
 *
 * 値は、value 属性で指定するか、なければ BODY 部に記述します。
 *
 * @og.formSample
 * ●形式：
 *     &lt;og:iorQueryParam key="where" value="{'PN':'{&#064;PN}%','TANI':'{&#064;TANI}'}" /&gt;
 *
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:iorQueryParam
 *       key              ○【TAG】パラメータとして渡すキー情報を指定します (必須)
 *       value              【TAG】パラメータとして渡す設定値を指定します (初期値:null)
 *       quotCheck          【TAG】リクエスト情報の シングルクォート(') 存在チェックを実施するかどうか[true/false]を設定します (初期値:USE_SQL_INJECTION_CHECK[=true])
 *       xssCheck           【TAG】リクエスト情報の HTMLTag開始/終了文字(&gt;&lt;) 存在チェックを実施するかどうか[true/false]を設定します (初期値:USE_XSS_CHECK[=true])
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します (初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します (初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます (初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます (初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます (初期値:判定しない)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します (初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:iorQueryParam&gt;
 *
 * ●使用例
 *     &lt;og:IorQuery
 *         url           = "http://･･･ "
 *         authURL       = "http://･･･ "
 *         authUserPass  = "admin:******"
 *         appliName     = "データテーブル名"
 *         callMethod    = "getReportInfo"
 *     &gt;
 *         &lt;og:iorQueryParam
 *             key  = "select"  value  = "PN,TANI,count(*) as DD"  /&gt;
 *         &lt;og:iorQueryParam
 *             key  = "where"  &gt;
 *             {'PN':'{&#064;PN}%','TANI':'{&#064;TANI}'}
 *         &lt;/og:iorQueryParam
 *     &lt;/og:IorQuery ･････ &gt;
 *
 * @og.rev 8.0.2.0 (2021/11/30) 新規作成
 * @og.group その他部品
 *
 * @version  8.0
 * @author   LEE.M
 * @since    JDK17.0,
 */
public class IorQueryParamTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。 {@value} */
	private static final String VERSION = "8.0.2.0 (2021/11/30)" ;
	private static final long serialVersionUID = 802020211130L ;

	private String key						;
	private String value					;

	private boolean	quotCheck				= HybsSystem.sysBool( "USE_SQL_INJECTION_CHECK" );
	private boolean	xssCheck				= HybsSystem.sysBool( "USE_XSS_CHECK" );

	/**
	 * デフォルトコンストラクター
	 */
	// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	public IorQueryParamTag() { super(); }

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doStartTag() {
		if( useTag() ) {
			useQuotCheck( quotCheck );
			useXssCheck( xssCheck );

			value = getRequestParameter( value );

			if( value == null || value.isEmpty() ) {
				return EVAL_BODY_BUFFERED;										// Body を評価する。( extends BodyTagSupport 時)
			}
		}
		return SKIP_BODY;														// Body を評価しない
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		value = getBodyString();
		return SKIP_BODY ;
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();
		if( useTag() ) {
			final IorQueryTag IorQuery = (IorQueryTag)findAncestorWithClass( this,IorQueryTag.class );
			if( IorQuery == null ) {
				final String errMsg = "<b>" + getTagName() + "タグは、IorQueryTagの内側(要素)に記述してください。</b>";
				throw new HybsSystemException( errMsg );
			}
			IorQuery.addParam( key, value );
		}
		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		key					= null;
		value				= null;
		quotCheck			= HybsSystem.sysBool( "USE_SQL_INJECTION_CHECK" );
		xssCheck			= HybsSystem.sysBool( "USE_XSS_CHECK" );
	}

	/**
	 * 【TAG】パラメータとして渡すキー情報を指定します。
	 *
	 * @og.tag
	 * IorQuery に渡すパラメータのキー情報です。
	 * 値は、value 属性で指定するか、なければ BODY 部に記述します。
	 *
	 * @param	prmKey	キー情報
	 */
	public void setKey( final String prmKey ) {
		key = nval( getRequestParameter( prmKey ),key );
	}

	/**
	 * 【TAG】パラメータとして渡す設定値を指定します(初期値:null)。
	 *
	 * @og.tag
	 * IorQuery に渡すパラメータの設定値です。
	 * 値は、value 属性で指定するか、なければ BODY 部に記述します。
	 * BODY 部に記述された場合は、文字列を trim() します。
	 * 設定値は、value 属性が優先です。ここの値が、null の場合は、
	 * BODY 要素を値として使用します。
	 *
	 * @param	val	設定値
	 */
	public void setValue( final String val ) {
		value = nval( getRequestParameter( val ),value );
	}

	/**
	 * 【TAG】リクエスト情報の シングルクォート(') 存在チェックを実施するかどうか[true/false]を設定します
	 *        (初期値:USE_SQL_INJECTION_CHECK[={@og.value SystemData#USE_SQL_INJECTION_CHECK}])。
	 *
	 * @og.tag
	 * SQLインジェクション対策の一つとして、暫定的ではありますが、SQLのパラメータに
	 * 渡す文字列にシングルクォート(') を許さない設定にすれば、ある程度は防止できます。
	 * 数字タイプの引数には、 or 5=5 などのシングルクォートを使用しないコードを埋めても、
	 * 数字チェックで検出可能です。文字タイプの場合は、必ず (')をはずして、
	 * ' or 'A' like 'A のような形式になる為、(')チェックだけでも有効です。
	 * (') が含まれていたエラーにする(true)／かノーチェックか(false)を指定します。
	 * (初期値:システム定数のUSE_SQL_INJECTION_CHECK[={@og.value SystemData#USE_SQL_INJECTION_CHECK}])。
	 *
	 * @param	flag クォートチェック [true:する/それ以外:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_SQL_INJECTION_CHECK
	 */
	public void setQuotCheck( final String flag ) {
		quotCheck = nval( getRequestParameter( flag ),quotCheck );
	}

	/**
	 * 【TAG】リクエスト情報の HTMLTag開始/終了文字(&gt;&lt;) 存在チェックを実施するかどうか[true/false]を設定します
	 *        (初期値:USE_XSS_CHECK[={@og.value SystemData#USE_XSS_CHECK}])。
	 *
	 * @og.tag
	 * クロスサイトスクリプティング(XSS)対策の一環としてless/greater than signについてのチェックを行います。
	 * (&gt;&lt;) が含まれていたエラーにする(true)／かノーチェックか(false)を指定します。
	 * (初期値:システム定数のUSE_XSS_CHECK[={@og.value SystemData#USE_XSS_CHECK}])。
	 *
	 * @param	flag	XSSチェック [true:する/false:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_XSS_CHECK
	 */
	public void setXssCheck( final String flag ) {
		xssCheck = nval( getRequestParameter( flag ),xssCheck );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return	このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "key"			,key		)
				.println( "value"		,value		)
				.println( "quotCheck"	,quotCheck	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
