/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import java.io.File;											// 8.0.0.1 (2021/10/08)

import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.hayabusa.report2.QueueManager_DIRECT;
import org.opengion.hayabusa.io.HybsFileOperationFactory;		// 8.0.0.1 (2021/10/08)
// import org.opengion.fukurou.model.FileOperation;				// 8.0.0.1 (2021/10/08)
// import org.opengion.fukurou.util.FileUtil;					// 8.0.0.1 (2021/10/08)

import static org.opengion.fukurou.util.StringUtil.nval;

/**
 * 検索結果の DBTableModelｵﾌﾞｼﾞｪｸﾄをﾚﾎﾟｰﾄ形式に変換するﾀｸﾞです｡
 *
 * ﾃﾞｰﾀ(DBTableModel)と､ｺﾝﾄﾛｰﾗ(QueueManager_DIRECT ｸﾗｽ)を与えて､
 * 外部からｺﾝﾄﾛｰﾙすることで､各種形式で ﾃﾞｰﾀ(DBTableModel)を表示させることが
 * 可能です｡
 * このﾀｸﾞを使用するには､OpenOffice.orgのﾓｼﾞｭｰﾙがｲﾝｽﾄｰﾙされてている必要があります｡
 * また､出力するために帳票ｼｽﾃﾑ関連のﾃﾞｰﾀ設定やﾏｽﾀ設定は一切必要ありません｡
 *
 * @og.formSample
 * ●形式：&lt;og:report fileURL="[･･･]" listId="[･･･]" ･･･ /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:report2
 *       fileURL            【TAG】雛型のHTMLﾌｧｲﾙの保存してある ﾃﾞｨﾚｸﾄﾘを指定します
 *       listId           ○【TAG】帳票IDを指定します(必須)｡
 *       outFileURL         【TAG】出力HTMLﾌｧｲﾙの保存してあるﾃﾞｨﾚｸﾄﾘを指定します(初期値:FILE_URL[={@og.value SystemData#FILE_URL}])
 *       outFilename      ○【TAG】ﾌｧｲﾙを作成するときの出力ﾌｧｲﾙ名をｾｯﾄします(必須)｡
 *       headerKeys         【TAG】固定部の{&#064;KEY} の KEY 部分をCSV形式で複数指定します
 *       headerVals         【TAG】固定部のKEY に対応する値をCSV形式で複数指定します
 *       footerKeys         【TAG】繰り返し部の終了後に表示する key 部分をCSV形式で複数指定します
 *       footerVals         【TAG】固定部のKEY に対応する値をCSV形式で複数指定します
 *       pageEndCut         【TAG】ﾎﾞﾃﾞｨｰ部(繰り返し部)がなくなったときに､それ以降を表示するかどうか[true/false]を指定します(初期値:true)
 *       useLocalResource   【TAG】各ｼｽﾃﾑのﾘｿｰｽ(ﾛｰｶﾙﾘｿｰｽ)を使用するか[true/false]を指定します(初期値:true)
 *       useSheetName       【TAG】PAGEBREAKｶﾗﾑの値を､ｼｰﾄ名として使うかどうか[true/false]を指定します(初期値:false)
 *       fgrun              【TAG】出力方法を指定します(初期値:P(PDF出力))
 *       printerName        【TAG】ﾌﾟﾘﾝﾀｰ名を指定します
 *       useLocal           【TAG】ｼｽﾃﾑ定数でｸﾗｳﾄﾞ設定されていても､ｸﾗｳﾄﾞ環境を使用しない場合､trueを指定します(初期値:false) 8.0.1.0 (2021/10/29)
 *       language           【TAG】ﾀｸﾞ内部で使用する言語ｺｰﾄﾞ[ja/en/zh/…]を指定します
 *       scope              【TAG】ｷｬｯｼｭする場合のｽｺｰﾌﾟ[request/page/session/application]を指定します(初期値:session)
 *       tableId            【TAG】(通常は使いません)sessionから所得する DBTableModelｵﾌﾞｼﾞｪｸﾄの ID
 *       debug              【TAG】ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *
 * @og.group その他出力
 *
 * @version  4.0
 * @author   Hiroki Nakamura
 * @since    JDK5.0,
 */
public class ReportTableTag2 extends CommonTagSupport {
	/** このﾌﾟﾛｸﾞﾗﾑのVERSION文字列を設定します｡	{@value} */
	private static final String VERSION = "8.0.1.0 (2021/10/29)" ;
	private static final long serialVersionUID = 801020211029L ;

	private final String  BASE_URL	= HybsSystem.sys( "FILE_URL" );

	private String		fileURL			= BASE_URL;	// 雛形ﾌｧｲﾙURL
	private String		listId			;			// 帳票ID
	private String		outFileURL		= BASE_URL;	// 出力ﾌｧｲﾙURL
	private String		outFilename		;			// 出力ﾌｧｲﾙ名
	private String[]	headerKeys		;			// 固定部の{@KEY} の KEY 部分を指定する｡ｶﾝﾏで複数指定できる｡
	private String[]	headerVals		;			// 固定部のKEY に対応する値を指定する｡ {@KEY} に置き換わる｡
	private String[]	footerKeys		;			// 繰り返し部の終了後に表示する key 部分を指定する｡ｶﾝﾏで複数指定できる｡
	private String[]	footerVals		;			// 繰り返し部の終了後に表示する key に対する値を指定する｡
	private boolean		pageEndCut		= true;		// ﾍﾟｰｼﾞｴﾝﾄﾞｶｯﾄをするか
	private boolean		useLocalResource= true;		// ﾛｰｶﾙﾘｿｰｽを使用するか
	private boolean		useSheetName	;			// 5.7.6.2 (2014/05/16) PAGEBREAKｶﾗﾑの値を､ｼｰﾄ名として使うかどうか｡
	private String		fgrun			= "P";		// PDF出力
	private String		printerName		;			// ﾌﾟﾘﾝﾀ名
	private boolean		useLocal		;			// 8.0.1.0 (2021/10/29) ｸﾗｳﾄﾞ設定を使用しない場合は､true

	private String		tableId			= HybsSystem.TBL_MDL_KEY ;

	private transient DBTableModel body		;
	private transient DBTableModel header	;
	private transient DBTableModel footer	;

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰ
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public ReportTableTag2() { super(); }		// これも､自動的に呼ばれるが､空のﾒｿｯﾄﾞを作成すると警告されるので､明示的にしておきます｡

	/**
	 * Taglibの終了ﾀｸﾞが見つかったときに処理する doEndTag() を ｵｰﾊﾞｰﾗｲﾄﾞします｡
	 *
	 * @og.rev 8.0.0.2 (2021/10/15) ﾛｰｶﾙﾌｧｲﾙとｸﾗｳﾄﾞﾌｧｲﾙ間の移動
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();

		final int rtnCode;

		body = (DBTableModel)getObject( tableId );
		if( body == null || body.getRowCount() == 0 ) {
			rtnCode = SKIP_PAGE ; // ﾍﾟｰｼﾞの残りの処理を行わない｡
		}
		else {
			exec();

			// 8.0.0.2 (2021/10/15) ﾛｰｶﾙﾌｧｲﾙとｸﾗｳﾄﾞﾌｧｲﾙ間の移動
			// 5.10.9.0 (2019/03/01) ADD ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ指定の場合は､作成されたﾌｧｲﾙをｱｯﾌﾟﾛｰﾄﾞします｡
			final String outFile = HybsSystem.url2dir( outFileURL , outFilename );
			HybsFileOperationFactory.local2cloud( useLocal,() -> new File( outFile ) );

//			final FileOperation cloudFile = HybsFileOperationFactory.create( outFile );
//			if( cloudFile.isCloud() ) {
//				final File localFile = new File( outFile );
//				FileUtil.copy( localFile, cloudFile );
//				localFile.delete();
//			}

			rtnCode = EVAL_PAGE ;
		}

		return rtnCode ;
	}

	/**
	 * ﾀｸﾞﾘﾌﾞｵﾌﾞｼﾞｪｸﾄをﾘﾘｰｽします｡
	 * ｷｬｯｼｭされて再利用されるので､ﾌｨｰﾙﾄﾞの初期設定を行います｡
	 *
	 * @og.rev 5.7.6.2 (2014/05/16) PAGEBREAKｶﾗﾑの値を､ｼｰﾄ名として使うかどうか｡
	 * @og.rev 8.0.1.0 (2021/10/29) useLocal 属性を追加｡
	 */
	@Override
	protected void release2() {
		super.release2();
		fileURL			= BASE_URL;
		listId			= null;
		outFileURL		= BASE_URL;
		outFilename		= null;
		headerKeys		= null;
		headerVals		= null;
		footerKeys		= null;
		footerVals		= null;
		pageEndCut		= true;
		useLocalResource= true;
		useSheetName	= false;	// 5.7.6.2 (2014/05/16) PAGEBREAKｶﾗﾑの値を､ｼｰﾄ名として使うかどうか｡
		fgrun			= "P";
		printerName		= null;
		useLocal		= false;	// 8.0.1.0 (2021/10/29) ｸﾗｳﾄﾞ設定を使用しない場合は､true

		tableId			= HybsSystem.TBL_MDL_KEY ;
		body			= null;
		header			= null;
		footer			= null;
	}

	/**
	 * 帳票処理を行います｡
	 *
	 * @og.rev 4.3.3.4 (2008/11/01) ﾍｯﾀﾞｰ､ﾌｯﾀｰ値が設定されていない場合にNullPointerExceptionが出るﾊﾞｸﾞを修正
	 * @og.rev 4.3.3.4 (2008/11/01) 雛形のﾊﾟｽ及び､出力先のﾊﾟｽを実ﾃﾞｨﾚｸﾄﾘのﾊﾟｽに変換
	 * @og.rev 5.7.6.2 (2014/05/16) PAGEBREAKｶﾗﾑの値を､ｼｰﾄ名として使うかどうか｡
	 * @og.rev 6.4.2.1 (2016/02/05) HybsSystem.url2dir に引数追加｡
	 *
	 */
	private void exec()  {
		final QueueManager_DIRECT manager = new QueueManager_DIRECT();
		manager.setListId( listId );
		manager.setLang( getLanguage() );
		manager.setOutputName( HybsSystem.url2dir( outFileURL , outFilename ) );	// 6.4.2.1 (2016/02/05)
		manager.setOutputType( fgrun );
		manager.setTemplateName( HybsSystem.url2dir( fileURL , listId ) );			// 6.4.2.1 (2016/02/05)
		manager.setPrinterName( printerName );
		manager.setFgcut( pageEndCut );
		manager.setFglocal( useLocalResource );
		manager.setUseSheetName( useSheetName );		// 5.7.6.2 (2014/05/16) PAGEBREAKｶﾗﾑの値を､ｼｰﾄ名として使うかどうか｡

		manager.setBody( body );

		// 4.3.3.4 (2008/11/01)
		if( headerVals != null && headerVals.length > 0 ) {
			String[][] hvals = new String[headerVals.length][1];
			hvals[0] = headerVals;
			header = DBTableModelUtil.makeDBTable( headerKeys, hvals, getResource() );
			manager.setHeader( header );
		}

		// 4.3.3.4 (2008/11/01)
		if( footerVals != null && footerVals.length > 0 ) {
			String[][] fvals = new String[footerVals.length][1];
			fvals[0] = footerVals;
			footer = DBTableModelUtil.makeDBTable( footerKeys, fvals, getResource() );
			manager.setFooter( footer );
		}

		manager.create();
		manager.waitExec();
	}

	/**
	 * 【TAG】雛型のHTMLﾌｧｲﾙの保存してある ﾃﾞｨﾚｸﾄﾘを指定します｡
	 *
	 * @og.tag
	 * この属性で指定されるﾃﾞｨﾚｸﾄﾘのﾌｧｲﾙを読み取ります｡
	 * 指定方法は､通常の fileURL 属性と同様に､先頭が､'/' (UNIX) または､２文字目が､
	 * ":" (Windows)の場合は､指定のURLそのままのﾃﾞｨﾚｸﾄﾘに､そうでない場合は､
	 * ｼｽﾃﾑﾊﾟﾗﾒｰﾀ の FILE_URL 属性で指定のﾌｫﾙﾀﾞの下に､作成されます｡
	 * fileURL = "{&#064;USER.ID}" と指定すると､FILE_URL 属性で指定のﾌｫﾙﾀﾞの下に､
	 * さらに､各個人ID別のﾌｫﾙﾀﾞを作成して､そこを操作します｡
	 *
	 * @og.rev 6.4.2.1 (2016/02/05) URLの最後に､"/" を追加する処理を廃止｡
	 *
	 * @param	url 雛型のHTMLﾌｧｲﾙのﾃﾞｨﾚｸﾄﾘ
	 */
	public void setFileURL( final String url ) {
		final String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】帳票IDを指定します｡
	 *
	 * @og.tag
	 * 帳票IDを指定します｡
	 *
	 * @param	listId	帳票ID
	 */
	public void setListId( final String listId ) {
		this.listId = nval( getRequestParameter( listId ), this.listId );
	}

	/**
	 * 【TAG】出力HTMLﾌｧｲﾙの保存してあるﾃﾞｨﾚｸﾄﾘを指定します
	 * 		(初期値:FILE_URL[={@og.value SystemData#FILE_URL}])｡
	 *
	 * @og.tag
	 * この属性で指定されるﾃﾞｨﾚｸﾄﾘにﾌｧｲﾙを出力します｡
	 * 指定方法は､通常の fileURL 属性と同様に､先頭が､'/' (UNIX) または､２文字目が､
	 * ":" (Windows)の場合は､指定のURLそのままのﾃﾞｨﾚｸﾄﾘに､そうでない場合は､
	 * ｼｽﾃﾑﾊﾟﾗﾒｰﾀ の FILE_URL 属性で指定のﾌｫﾙﾀﾞの下に､作成されます｡
	 * fileURL = "{&#064;USER.ID}" と指定すると､FILE_URL 属性で指定のﾌｫﾙﾀﾞの下に､
	 * さらに､各個人ID別のﾌｫﾙﾀﾞを作成して､そこに出力します｡
	 * (初期値:FILE_URL[={@og.value SystemData#FILE_URL}])｡
	 *
	 * @og.rev 6.4.2.1 (2016/02/05) URLの最後に､"/" を追加する処理を廃止｡
	 *
	 * @param	url 出力HTMLﾌｧｲﾙのﾃﾞｨﾚｸﾄﾘ
	 */
	public void setOutFileURL( final String url ) {
		final String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			outFileURL = StringUtil.urlAppend( outFileURL,furl );
		}
	}

	/**
	 * 【TAG】ﾌｧｲﾙを作成するときの出力ﾌｧｲﾙ名をｾｯﾄします｡
	 *
	 * @og.tag
	 * ﾌｧｲﾙを作成するときの出力ﾌｧｲﾙ名をｾｯﾄします｡
	 * 紙に印字する場合などﾌｧｲﾙに出力しない場合は不要です｡
	 *
	 * @param   filename 出力ﾌｧｲﾙ名
	 */
	public void setOutFilename( final String filename ) {
		this.outFilename = nval( getRequestParameter( filename ),this.outFilename );
	}

	/**
	 * 【TAG】固定部の{&#064;KEY} の KEY 部分をCSV形式で複数指定します｡
	 *
	 * @og.tag
	 * ｶﾝﾏで複数指定できます｡
	 * 分解方法は､CSV変数を先に分解してから､getRequestParameter で値を取得します｡
	 * こうしないとﾃﾞｰﾀ自身にｶﾝﾏを持っている場合に分解をﾐｽる為です｡
	 *
	 * @param   hKeys 固定部のkey
	 */
	public void setHeaderKeys( final String hKeys ) {
		headerKeys = getCSVParameter( hKeys );
	}

	/**
	 * 【TAG】固定部のKEY に対応する値をCSV形式で複数指定します｡
	 *
	 * @og.tag
	 * ｶﾝﾏで複数指定で､ﾘｸｴｽﾄ情報でも設定できます｡
	 * 分解方法は､CSV変数を先に分解してから､getRequestParameter で値を取得します｡
	 * こうしないとﾃﾞｰﾀ自身にｶﾝﾏを持っている場合に分解をﾐｽる為です｡
	 *
	 * @param   hVals 固定部の値
	 */
	public void setHeaderVals( final String hVals ) {
		headerVals = getCSVParameter( hVals );
	}

	/**
	 * 【TAG】繰り返し部の終了後に表示する key 部分をCSV形式で複数指定します｡
	 *
	 * @og.tag
	 * ｶﾝﾏで複数指定できます｡
	 * 分解方法は､CSV変数を先に分解してから､getRequestParameter で値を取得します｡
	 * こうしないとﾃﾞｰﾀ自身にｶﾝﾏを持っている場合に分解をﾐｽる為です｡
	 *
	 * @param   ftKeys 繰り返し部の終了後に表示するkey
	 */
	public void setFooterKeys( final String ftKeys ) {
		footerKeys = getCSVParameter( ftKeys );
	}

	/**
	 * 【TAG】固定部のKEY に対応する値をCSV形式で複数指定します｡
	 *
	 * @og.tag
	 * ｶﾝﾏで複数指定で､ﾘｸｴｽﾄ情報でも設定できます｡
	 * 分解方法は､CSV変数を先に分解してから､getRequestParameter で値を取得します｡
	 * こうしないとﾃﾞｰﾀ自身にｶﾝﾏを持っている場合に分解をﾐｽる為です｡
	 *
	 * @param   ftVals 繰り返し部の終了後に表示する値
	 */
	public void setFooterVals( final String ftVals ) {
		footerVals = getCSVParameter( ftVals );
	}

	/**
	 * 【TAG】ﾍﾟｰｼﾞｴﾝﾄﾞｶｯﾄするかどうか[true:する/false:しない]を指定します(初期値:true:ｶｯﾄする)｡
	 *
	 * @og.tag
	 * ﾍﾟｰｼﾞｴﾝﾄﾞｶｯﾄとは､ﾎﾞﾃﾞｨｰ部(繰り返し部)がなくなったときに､それ以降の表示を打ち切る（ｶｯﾄする）
	 * 機能で､(true:ｶｯﾄする) を指定すると､それ以降を出力しません｡
	 * 初期値は "true" (なくなった時点で､出力しない)です｡
	 *
	 * @param   peCut ﾍﾟｰｼﾞ終了ｶｯﾄ [true:する/false:しない]
	 */
	public void setPageEndCut( final String peCut ) {
		pageEndCut = nval( getRequestParameter( peCut ),pageEndCut );
	}

	/**
	 * 【TAG】各ｼｽﾃﾑのﾘｿｰｽ(ﾛｰｶﾙﾘｿｰｽ)を使用するかどうか[true:する/false:しない]を指定します(初期値:true)｡
	 *
	 * @og.tag
	 * true の場合､各ｼｽﾃﾑに登録されたﾘｿｰｽ情報を使用して帳票ﾃﾞｰﾀが変換されます｡
	 * false の場合は､帳票ﾃﾞｰﾓﾝが起動しているｼｽﾃﾑ(通常は'GE')のﾘｿｰｽが適用されます｡
	 * 初期値は "true" (ﾛｰｶﾙﾘｿｰｽを使用する)です｡
	 *
	 * @param   fgl  ﾛｰｶﾙﾘｿｰｽの使用 [true:する/false:しない]
	 */
	public void setUseLocalResource( final String fgl ) {
		useLocalResource = nval( getRequestParameter( fgl ),useLocalResource );
	}

	/**
	 * 【TAG】PAGEBREAKｶﾗﾑの値を､ｼｰﾄ名として使うかどうかをｾｯﾄします(初期値:false)｡
	 *
	 * @og.tag
	 * PAGEBREAK で､ｼｰﾄﾁｪﾝｼﾞを行う場合､ｼｰﾄ名も指定したい場合があります｡
	 * その場合､この､useSheetName="true" とすることで､PAGEBREAKｶﾗﾑの値を､ｼｰﾄ名として
	 * 使用します｡
	 * useSheetName="false" の場合は､"Page"+ﾍﾟｰｼﾞ番号+"_Row"+現在行番号 がｼｰﾄ名になります｡
	 *
	 * PAGEBREAK は､FIRSTｼｰﾄ雛形にも適用されます｡
	 * ちなみに､FIRSTｼｰﾄ雛形は､特殊で､useSheetName="false" の場合でも､
	 * FIRST_**** などと記述した場合は､**** 文字列をｼｰﾄ名に使用します｡
	 * FIRST だけの場合は､従来と同じｼｰﾄ名になります｡
	 * 初期値は､互換性を考慮し､false:ｼｰﾄ名として使用しない です｡
	 *
	 * @og.rev 5.7.6.2 (2014/05/16) 新規追加
	 *
	 * @param useSName PAGEBREAKｶﾗﾑのｼｰﾄ名使用可否 [true:使用/false:使用しない]
	 */
	public void setUseSheetName( final String useSName ) {
		useSheetName = nval( getRequestParameter( useSName ),useSheetName );
	}

	/**
	 * 【TAG】出力方法を指定します(初期値:P(PDF出力))｡
	 *
	 * @og.tag
	 * 出力方法のｺｰﾄﾞは､FGRUNのｺｰﾄﾞﾘｿｰｽと同じものが指定できます｡
	 * 初期値は "P" (PDF出力)です｡
	 *
	 * @param   code 出力方法(FGRUNのｺｰﾄﾞﾘｿｰｽと同じもの)
	 */
	public void setFgrun( final String code ) {
		fgrun= nval( getRequestParameter( code ),fgrun );
	}

	/**
	 * 【TAG】ﾌﾟﾘﾝﾀｰ名を指定します｡
	 *
	 * @og.tag
	 * ﾌﾟﾘﾝﾀｰ名を指定します｡このﾌﾟﾘﾝﾀｰ名は帳票ｻｰﾊﾞｰ上でのﾌﾟﾘﾝﾀ名です｡
	 * ﾌｧｲﾙ出力等､紙に印刷しない場合は不要です｡
	 *
	 * @param   ptnm ﾌﾟﾘﾝﾀｰ名
	 */
	public void setPrinterName( final String ptnm ) {
		printerName = nval( getRequestParameter( ptnm ),printerName );
	}

	/**
	 * 【TAG】ｼｽﾃﾑ定数でｸﾗｳﾄﾞ設定されていても､ｸﾗｳﾄﾞ環境を使用しない場合､trueを指定します(初期値:false)｡
	 *
	 * @og.tag
	 * ｸﾗｳﾄﾞ設定は､ｼｽﾃﾑ定数の『CLOUD_TARGET』と『CLOUD_BUCKET』の設定で自動的に使用しますが､
	 * どうしてもﾛｰｶﾙでのみ使いたい場合は､この属性を true に設定します｡
	 * 標準はfalse:設定どおりとなります｡
	 *
	 * true/false以外を指定した場合はfalse扱いとします｡
	 *
	 * @og.rev 8.0.1.0 (2021/10/29) useLocal 属性を追加｡
	 *
	 * @param flag ﾛｰｶﾙ環境のみ [true:ﾛｰｶﾙのみ/false:設定どおり]
	 */
	public void setUseLocal( final String flag ) {
		useLocal = nval( getRequestParameter( flag ),useLocal );
	}

	/**
	 * 【TAG】(通常は使いません)結果のDBTableModelを､sessionに登録するときのｷｰを指定します
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])｡
	 *
	 * @og.tag
	 * 検索結果より､DBTableModelｵﾌﾞｼﾞｪｸﾄを作成します｡これを､下流のviewﾀｸﾞ等に
	 * 渡す場合に､通常は､session を利用します｡その場合の登録ｷｰです｡
	 * query ﾀｸﾞを同時に実行して､結果を求める場合､同一ﾒﾓﾘに配置される為､
	 * この tableId 属性を利用して､ﾒﾓﾘ空間を分けます｡
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])｡
	 *
	 * @param	id ﾃｰﾌﾞﾙID (sessionに登録する時のID)
	 */
	public void setTableId( final String id ) {
		tableId = nval( getRequestParameter( id ), tableId );
	}

	/**
	 * ﾀｸﾞの名称を､返します｡
	 * 自分自身のｸﾗｽ名より､自動的に取り出せないため､このﾒｿｯﾄﾞをｵｰﾊﾞｰﾗｲﾄﾞします｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @return  ﾀｸﾞの名称
	 * @og.rtnNotNull
	 */
	@Override
	protected String getTagName() {
		return "report2" ;
	}

	/**
	 * このｵﾌﾞｼﾞｪｸﾄの文字列表現を返します｡
	 * 基本的にﾃﾞﾊﾞｯｸﾞ目的に使用します｡
	 *
	 * @return このｸﾗｽの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "fileURL"			,fileURL			)
				.println( "listId"			,listId				)
				.println( "outFileURL"		,outFileURL			)
				.println( "outFilename"		,outFilename		)
				.println( "headerKeys"		,headerKeys			)
				.println( "headerVals"		,headerVals			)
				.println( "footerKeys"		,footerKeys			)
				.println( "footerVals"		,footerVals			)
				.println( "pageEndCut"		,pageEndCut			)
				.println( "useLocalResource",useLocalResource	)
				.println( "fgrun"			,fgrun				)
				.println( "printerName"		,printerName		)
				.println( "tableId"			,tableId			)
				.println( "BASE_URL"		,BASE_URL			)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
