/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBTableModelSorter;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.fukurou.system.DateSet;						// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.security.HybsCryptography ;			// 5.7.4.3 (2014/03/28)
import org.opengion.fukurou.model.POIUtil;						// 6.2.2.0 (2015/03/27)
import org.opengion.fukurou.util.FileInfo;						// 6.2.2.0 (2015/03/27)
import org.opengion.fukurou.util.FileUtil;						// 6.7.4.1 (2017/02/17)
import org.opengion.fukurou.util.ArraySet;						// 6.4.3.4 (2016/03/11)
import org.opengion.hayabusa.io.HybsFileOperationFactory;		// 8.0.0.1 (2021/10/08)

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.Arrays;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;

/**
 * ﾌｧｲﾙを検索し､DBTableModel にｾｯﾄするﾀｸﾞです｡
 *
 * ﾌｧｲﾙの検索結果は､[WRITABLE],LEVEL,FILE_TYPE,PARENT,NAME,LASTMODIFIED,FILE_LENGTH,RWH,[MD5],[TO_PARENT,TO_NAME],[････]
 * のｶﾗﾑを持つ DBTableModel にｾｯﾄ されます｡このｶﾗﾑは､固定です｡
 * 並び替えについては､このﾀｸﾞで指定しますが､ﾌｧｲﾙの選別(where 条件)は､
 * BODY 部に記述する fileWhere ﾀｸﾞで指定します｡(複数指定可能))
 *
 * [ｶﾗﾑ名]      検索するﾌｧｲﾙの属性は､以下のｶﾗﾑ名で作成されます｡
 *     [WRITABLE]       useWritable=trueで､先頭ｶﾗﾑに､WRITABLE ｶﾗﾑが挿入されます｡
 *      LEVEL           ﾃﾞｨﾚｸﾄﾘを展開する場合のﾚﾍﾞﾙ｡
 *      FILE_TYPE       ﾌｧｲﾙ(F)かﾃﾞｨﾚｸﾄﾘ(D)であるか判定｡
 *      PARENT          この抽象ﾊﾟｽ名の親のﾊﾟｽ名文字列を返します｡
 *      NAME            この抽象ﾊﾟｽ名が示すﾌｧｲﾙまたはﾃﾞｨﾚｸﾄﾘの名前を返します｡
 *      LASTMODIFIED    最後に変更された時刻を返します｡
 *      FILE_LENGTH     ﾌｧｲﾙの長さを返します｡
 *      RWH             読み込み､書き込み､隠し属性をそれぞれ､ｒ,w,h で表します｡
 *     [MD5]            useMD5=trueで､MD5 というｶﾗﾑを追加したうえで､ﾌｧｲﾙのMD5計算を行います｡
 *     [TEXT]           useText=trueで､ﾌｧｲﾙの内容を文字列にして､TEXTというｶﾗﾑに設定します｡
 *     [TO_PARENT]      useUpdateClms=trueで､fileUpdateﾀｸﾞでCOPYやMOVEを行う時に使用する必須となるｶﾗﾑ(TO_PARENT,TO_NAME)を追加します｡
 *     [TO_NAME]        同上
 *     [････]           addClms属性で指定されたｶﾗﾑを追加します｡
 *
 * @og.formSample
 * ●形式：&lt;og:fileQuery from="…" multi="true/false" &gt;
 *             &lt;og:fileWhere … /&gt;
 *              …
 *         &lt;/og:fileQuery&gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し､{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:fileQuery
 *       from             ○【TAG】検索を開始するﾌｧｲﾙまたはﾃﾞｨﾚｸﾄﾘを指定します(必須)｡
 *       multi              【TAG】多段階展開するか､１ﾚﾍﾞﾙ展開するかどうか[true:多段階/false:１ﾚﾍﾞﾙ]を指定します(初期値:false:１ﾚﾍﾞﾙ)｡
 *       level              【TAG】多段階展開するﾚﾍﾞﾙを指定します(初期値:100)｡
 *       orderBy            【TAG】ｿｰﾄするｶﾗﾑ名を指定します(一つのみ)｡
 *       desc               【TAG】表示順を逆転するかどうか[true/false]を指定します(初期値:false)｡
 *       useWritable        【TAG】先頭ｶﾗﾑに､WRITABLE ｶﾗﾑを追加するかどうか[true/false]を指定します(初期値:false)｡
 *       useMD5             【TAG】MD5ｶﾗﾑを追加したうえで､MD5計算を行うかどうか[true/false]を指定します(初期値:false)｡
 *       useText            【TAG】TEXTｶﾗﾑを追加したうえで､ﾌｧｲﾙの内容を読み込むかどうか[true/false]を指定します(初期値:false)｡
 *       useUpdateClms      【TAG】TO_PARENT､TO_NAMEｶﾗﾑを追加するかどうか[true/false]を指定します(初期値:false)｡
 *       addClms            【TAG】検索結果のｶﾗﾑ列に追加するｶﾗﾑ名を､CSV形式で指定します｡
 *       nameOnly           【TAG】ﾌｧｲﾙの拡張子を除いた名前部分のみの値で行います｡7.2.6.0 (2020/06/30)
 *       fileType           【TAG】選択対象[FILE/DIR]を指定します｡下位展開は考慮(multi 属性準拠)されます｡
 *       addFrom            【TAG】from属性で指定された基準ﾌｧｲﾙ/ﾌｫﾙﾀﾞ自体をﾘｽﾄに追加するかどうか[true/false]を指定します(初期値:true)｡
 *       fromBase           【TAG】検索結果のPARENT列から､fromBase指定のﾊﾟｽを削除した相対ﾊﾟｽを作成します｡  (7.0.2.1 (2019/03/04))
 *       command            【TAG】ｺﾏﾝﾄﾞ (NEW,RENEW)をｾｯﾄします("NEW" と "RENEW" 時のみ実行する(初期値:NEW))｡
 *       maxRowCount        【TAG】(通常は使いません)ﾃﾞｰﾀの最大読み込み件数を指定します (初期値:DB_MAX_ROW_COUNT[=1000])(0:[無制限])｡
 *       displayMsg         【TAG】検索結果を画面上に表示するﾒｯｾｰｼﾞﾘｿｰｽIDを指定します (初期値:VIEW_DISPLAY_MSG[=])
 *       overflowMsg        【TAG】検索ﾃﾞｰﾀが最大検索数をｵｰﾊﾞｰした場合に表示するﾒｯｾｰｼﾞﾘｿｰｽIDを指定します
 *                                    (初期値:MSG0007[検索結果が､制限行数を超えましたので､残りはｶｯﾄされました])｡
 *       notfoundMsg        【TAG】検索結果がｾﾞﾛ件の場合に表示するﾒｯｾｰｼﾞﾘｿｰｽIDを指定します(初期値:MSG0077[対象ﾃﾞｰﾀはありませんでした])｡
 *       stopZero           【TAG】検索結果が０件のとき処理を続行するかどうか[true/false]を指定します(初期値:false[続行する])
 *       tableId            【TAG】(通常は使いません)結果のDBTableModelを､sessionに登録するときのｷｰを指定します
 *       scope              【TAG】ｷｬｯｼｭする場合のｽｺｰﾌﾟ[request/page/session/application]を指定します(初期値:session)｡
 *       useSLabel          【TAG】7.0.7.0 (2019/12/13) ｴﾗｰﾒｯｾｰｼﾞにSLABELを利用するかどうか[true/false]を指定します(初期値:false)
 *       useLocal           【TAG】ｼｽﾃﾑ定数でｸﾗｳﾄﾞ設定されていても､ｸﾗｳﾄﾞ環境を使用しない場合､trueを指定します(初期値:false) 8.0.1.0 (2021/10/29)
 *       caseKey            【TAG】このﾀｸﾞ自体を利用するかどうかの条件ｷｰを指定します(初期値:null) 6.8.0.0 (2017/06/02)
 *       caseVal            【TAG】このﾀｸﾞ自体を利用するかどうかの条件値を指定します(初期値:null) 6.8.0.0 (2017/06/02)
 *       caseNN             【TAG】指定の値が､null/ｾﾞﾛ文字列 でない場合(Not Null=NN)は､このﾀｸﾞは使用されます(初期値:判定しない) 6.8.0.0 (2017/06/02)
 *       caseNull           【TAG】指定の値が､null/ｾﾞﾛ文字列 の場合は､このﾀｸﾞは使用されます(初期値:判定しない) 6.8.0.0 (2017/06/02)
 *       caseIf             【TAG】指定の値が､true/TRUE文字列の場合は､このﾀｸﾞは使用されます(初期値:判定しない) 6.8.0.0 (2017/06/02)
 *       debug              【TAG】ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうか[true/false]を指定します(初期値:false)｡
 *   &gt;   ... Body ...
 *   &lt;/og:fileQuery&gt;
 *
 * ●使用例
 *    ･一般的な属性でﾌｧｲﾙの検索を行います｡
 *         &lt;og:fileQuery
 *                from    = "d:/webapps/dbdef/jsp/"
 *                multi   = "true"
 *                command = "{&#064;command}"  &gt;
 *            &lt;og:fileWhere endWith=".jsp" /&gt;
 *      &lt;/og:fileQuery&gt;
 *
 *    ･最終変更日で逆順ｿｰﾄする｡対象は､2002/10/01 以降に変更されたﾌｧｲﾙ｡
 *        &lt;og:fileQuery
 *                from    = "d:/webapps/dbdef/jsp/"
 *                multi   = "true"
 *                orderBy = "LASTMODIFIED"
 *                desc    = "true"
 *                command = "{&#064;command}"  &gt;
 *            &lt;og:fileWhere lastModified="20021001000000" /&gt;
 *        &lt;/og:fileQuery&gt;
 *
 * @og.rev 4.0.0.0 (2005/01/31) 内部ﾛｼﾞｯｸ改定
 * @og.group その他入力
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class FileQueryTag extends QueryTag {
	/** このﾌﾟﾛｸﾞﾗﾑのVERSION文字列を設定します｡	{@value} */
	private static final String VERSION = "8.0.1.0 (2021/10/29)" ;
	private static final long serialVersionUID = 801020211029L ;

	// 6.4.3.4 (2016/03/11) String配列 から､Setに置き換えます｡
	private static final ArraySet<String> SELECT_SET = new ArraySet<>( "LEVEL","FILE_TYPE","PARENT","NAME","LASTMODIFIED","FILE_LENGTH","RWH" );

	private static final String[] USE_UPDATE_CLM = { "TO_PARENT","TO_NAME" };	// 5.3.4.0 (2011/04/01)

	private transient FileFilter filter	;		// FileWhere で指定したﾌｨﾙﾀｰ

	private boolean	multi			;			// 下位層展開ﾌﾗｸﾞ
	private int		level			= 100;		// 展開ﾚﾍﾞﾙ
	private String	from			= HybsSystem.sys( "FILE_URL" );	// 検索起点ﾌｧｲﾙ
	private String	orderBy			;			// 5.3.4.0 (2011/04/01) ｿｰﾄｶﾗﾑ
	private boolean	desc			;			// 5.3.4.0 (2011/04/01) ｿｰﾄの方向(true:逆順)
	private String[]	addClms		= new String[0];				// 5.3.4.0 (2011/04/01) 追加ｶﾗﾑ配列
	private String[]	defClms		;			// 5.7.4.3 (2014/03/28) 初期値のｶﾗﾑ配列
	private String	fileType		;			// 5.3.4.0 (2011/04/01) 選択対象を指定(FILE,DIR)
	private boolean	useWritable		;			// 5.7.4.3 (2014/03/28) 先頭ｶﾗﾑに､WRITABLE ｶﾗﾑを追加するかどうか[true/false](初期値:false)
	private boolean	useMD5			;			// 5.7.4.3 (2014/03/28) MD5ｶﾗﾑを追加したうえで､MD5計算を行うかどうか[true/false](初期値:false)
	private boolean	useText			;			// 6.2.2.0 (2015/03/27) TEXTｶﾗﾑを追加したうえで､ﾌｧｲﾙの内容を読み込むかどうか[true/false]を指定します(初期値:false)｡
	private boolean	useUpdateClms	;			// 5.3.4.0 (2011/04/01) TO_PARENT､TO_NAMEｶﾗﾑを追加(true:追加)
	private boolean	addFrom			= true;		// 5.3.9.0 (2011/09/01) from属性で指定された基準ﾌｧｲﾙ/ﾌｫﾙﾀﾞ自体をﾘｽﾄに追加するかどうか
	private int		fromLen			= -1;		// 7.0.2.1 (2019/03/04) fromBase の文字数をPARENTから削除するための数
	private boolean	nameOnly		;			// 7.2.6.0 (2020/06/30) ﾌｧｲﾙの拡張子を除いた名前部分のみの値で行います｡
	private boolean	useLocal		;			// 8.0.1.0 (2021/10/29) ｸﾗｳﾄﾞ設定を使用しない場合は､true

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰ
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public FileQueryTag() { super(); }		// これも､自動的に呼ばれるが､空のﾒｿｯﾄﾞを作成すると警告されるので､明示的にしておきます｡

	/**
	 * Taglibのﾀｸﾞ本体を処理する doAfterBody() を ｵｰﾊﾞｰﾗｲﾄﾞします｡
	 *
	 * ﾀｸﾞ本体が空の場合は､呼ばれないので､従来の doAfterBody() 処理を､
	 * doEndTag() に持っていきます｡
	 * よって､親ｸﾗｽの doAfterBody() を実行させないため､このﾒｿｯﾄﾞを用意しています｡
	 *
	 * @og.rev 6.7.4.1 (2017/02/17) 親ｸﾗｽの処理を､実行させない｡
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		return SKIP_BODY ;
	}

	/**
	 * Taglibの終了ﾀｸﾞが見つかったときに処理する doEndTag() を ｵｰﾊﾞｰﾗｲﾄﾞします｡
	 *
	 * ﾀｸﾞ本体が空の場合は､呼ばれないので､従来の doAfterBody() 処理を､
	 * doEndTag() に持っていきます｡
	 *
	 * @og.rev 6.7.4.1 (2017/02/17) 従来の doAfterBody() 処理を､doEndTag() に持ってくる｡
	 * @og.rev 6.8.0.0 (2017/06/02) caseKey,caseVal,caseNN,caseNull 属性を追加
	 * @og.rev 8.0.0.1 (2021/10/08) cloud対応
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		if( !useTag() ) { return SKIP_BODY ; }	// 6.8.0.0 (2017/06/02)

		executeCount = 0;

		table = initDBTable();
		if( maxRowCount < 0 ) {
			maxRowCount	= sysInt( "DB_MAX_ROW_COUNT" ) ;
		}

		// 5.3.5.0 (2011/05/01) 最初のﾌｧｲﾙが存在する場合のみ､実行する｡
//		final File fin = new File( from );
		final File fin = HybsFileOperationFactory.create( useLocal,from );		// 8.0.0.1 (2021/10/08)
		if( fin.exists() ) {
			// 7.0.2.1 (2019/03/04) 検索結果のPARENT列から､from指定のﾊﾟｽを削除した相対ﾊﾟｽを作成

			execute( fin,0 ) ;

			// 5.3.4.0 (2011/04/01) 指定ｶﾗﾑのｿｰﾄ処理
			if( orderBy != null ) {
				final int clmNo = table.getColumnNo( orderBy );
				final DBTableModelSorter temp = new DBTableModelSorter();
				temp.setModel( table );
				temp.sortByColumn( clmNo,!desc );	// 注意 desc の値と ｿｰﾄ正逆は､反対です｡
				table = temp;
			}
		}
		return super.doEndTag();
	}

	/**
	 * ﾀｸﾞﾘﾌﾞｵﾌﾞｼﾞｪｸﾄをﾘﾘｰｽします｡
	 * ｷｬｯｼｭされて再利用されるので､ﾌｨｰﾙﾄﾞの初期設定を行います｡
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 指定ｶﾗﾑのｿｰﾄ処理機能､ｶﾗﾑ追加機能､fileType追加
	 * @og.rev 5.3.9.0 (2011/09/01) addFrom属性追加
	 * @og.rev 5.7.4.3 (2014/03/28) useWritable,useMD5属性追加｡valClms を defClms に置き換え｡
	 * @og.rev 6.2.2.0 (2015/03/27) useText属性追加｡
	 * @og.rev 7.0.2.1 (2019/03/04) fromBase属性追加に伴い､fromLen変数を用意｡
	 * @og.rev 7.2.6.0 (2020/06/30) nameOnly 属性 を追加します｡
	 * @og.rev 8.0.1.0 (2021/10/29) useLocal 属性を追加｡
	 */
	@Override
	protected void release2() {
		super.release2();
		filter			= null;
		multi			= false;
		level			= 100;
		from			= HybsSystem.sys( "FILE_URL" );
		orderBy			= null;				// 5.3.4.0 (2011/04/01) ｿｰﾄｶﾗﾑ
		desc			= false;			// 5.3.4.0 (2011/04/01) 降順ﾌﾗｸﾞ
		addClms			= new String[0];	// 5.3.4.0 (2011/04/01) 追加ｶﾗﾑ配列
		defClms			= null;				// 5.7.4.3 (2014/03/28) 初期値のｶﾗﾑ配列
		fileType		= null;				// 5.3.4.0 (2011/04/01) 選択対象を指定(FILE,DIR,ALL)
		useWritable		= false;			// 5.7.4.3 (2014/03/28) 先頭ｶﾗﾑに､WRITABLE ｶﾗﾑを追加するかどうか[true/false](初期値:false)
		useMD5			= false;			// 5.7.4.3 (2014/03/28) MD5ｶﾗﾑを追加したうえで､MD5計算を行うかどうか[true/false](初期値:false)
		useText			= false;			// 6.2.2.0 (2015/03/27) TEXTｶﾗﾑを追加したうえで､ﾌｧｲﾙの内容を読み込むかどうか[true/false]を指定します(初期値:false)｡
		useUpdateClms	= false;			// 5.3.4.0 (2011/04/01) TO_PARENT､TO_NAMEｶﾗﾑを追加(true:追加)
		addFrom			= true;				// 5.3.9.0 (2011/09/01) addFrom属性追加
		fromLen			= -1;				// 7.0.2.1 (2019/03/04) fromBase の文字数をPARENTから削除するための数
		nameOnly		= false;			// 7.2.6.0 (2020/06/30) ﾌｧｲﾙの拡張子を除いた名前部分のみの値で行います｡
		useLocal		= false;			// 8.0.1.0 (2021/10/29) ｸﾗｳﾄﾞ設定を使用しない場合は､true
	}

	/**
	 * FileQuery を実行します｡
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) fileType の条件に合致する場合だけ､ﾃﾞｰﾀを作成する｡
	 * @og.rev 5.3.7.0 (2011/07/01) ﾌｫﾙﾀﾞにｱｸｾｽできない場合は､null となるのでその対応
	 * @og.rev 7.0.2.1 (2019/03/04) fromBase属性追加に伴い､fromLen変数を用意｡
	 *
	 * @param	fin	検索を開始するﾌｧｲﾙ/ﾃﾞｨﾚｸﾄﾘ
	 * @param	lvl	階層展開ﾚﾍﾞﾙ
	 */
	protected void execute( final File fin,final int lvl ) {
		if( !multi && lvl > 1 || lvl > level ) { return; }			// 階層展開する､しない	// 6.9.7.0 (2018/05/14) PMD Useless parentheses.
		if( executeCount > maxRowCount ) { table.setOverflow( true ); return; }

		final boolean isDIR = fin.isDirectory();

		if(		fileType == null
			||  isDIR &&  "DIR".equalsIgnoreCase( fileType )
			|| !isDIR && "FILE".equalsIgnoreCase( fileType ) ) {
				// 6.0.2.4 (2014/10/17) RpC:条件ﾃｽﾄを繰り返しています｡
				if( addFrom || lvl > 0 ) {
					addFileData( executeCount++,lvl,fin );
				}
		}
		if( isDIR ) {
			final File[] list = fin.listFiles( filter );
			// 5.3.7.0 (2011/07/01) ﾌｫﾙﾀﾞにｱｸｾｽできない場合は､null となる｡
			if( list != null ) {
				for( int i=0; i<list.length; i++ ) {
					execute( list[i],lvl+1 );
				}
			}
		}
	}

	/**
	 * 初期化された DBTableModel を返します｡
	 *
	 * ここでは､useWritable､useMD5､useUpdateClms､addClms を加味した
	 * DBTableModel と初期値ﾃﾞｰﾀ(defClms)を作成します｡
	 * 以前は､TO_PARENT､TO_NAMEと､addClms 分のみ初期値を持っていましたが､
	 * 5.7.4.3 (2014/03/28)で､先頭ｶﾗﾑのWRITABLEの初期値を､DBColumn の初期値ではなく
	 * 手動設定する必要がある為､すべてのｶﾗﾑ列の初期値を持っておきます｡
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 指定ｶﾗﾑ追加機能追加
	 * @og.rev 5.7.4.3 (2014/03/28) useWritable,useMD5属性追加
	 * @og.rev 6.2.2.0 (2015/03/27) TEXTｶﾗﾑを追加したうえで､ﾌｧｲﾙの内容を読み込むかどうか[true/false]を指定します(初期値:false)｡
	 * @og.rev 6.4.3.4 (2016/03/11) String配列 から､Setに置き換えます｡
	 * @og.rev 7.0.2.1 (2019/03/04) 追加ｶﾗﾑの位置のｱﾄﾞﾚｽ指定が間違っていた｡
	 *
	 * @return	 ﾃｰﾌﾞﾙﾓﾃﾞﾙ
	 */
	private DBTableModel initDBTable() {
		final DBTableModel tbl = DBTableModelUtil.newDBTable();

		// 5.7.4.3 (2014/03/28) 以下の処理は､ほぼ全面見直し
		int size = SELECT_SET.size() ;							// 6.4.3.4 (2016/03/11) 基本ｶﾗﾑの数

		if( useWritable   ) { size++ ; }						// WRITABLE ｶﾗﾑを追加
		if( useMD5        ) { size++ ; }						// MD5 ｶﾗﾑを追加
		if( useText       ) { size++ ; }						// 6.2.2.0 (2015/03/27) TEXT ｶﾗﾑを追加
		if( useUpdateClms ) { size += USE_UPDATE_CLM.length; }	// TO_PARENT､TO_NAMEｶﾗﾑを追加
		size += addClms.length ;								// addClms(追加ｶﾗﾑ)数を追加

		// DBTableModel の初期化と､初期値配列の確保
		tbl.init( size );
		defClms = new String[size];

		int ad=0;
		// 先頭は､WRITABLE
		if( useWritable ) {
			final DBColumn dbColumn = getDBColumn( "WRITABLE" );
			defClms[ad] = "1";									// WRITABLE を設定するときは､とりあえず 書き込み許可
			tbl.setDBColumn( ad++,dbColumn );
		}

		// SELECT の 基本ｶﾗﾑの設定｡(初期値は不要)
		// 6.4.3.4 (2016/03/11) String配列 から､Setに置き換えます｡
		// ｵﾘｼﾞﾅﾙの forEach｡ｶｳﾝﾀ初期値とﾗﾑﾀﾞ式を与えると､ﾗﾑﾀﾞ式の引数に､ｶｳﾝﾀと値が設定される｡
		SELECT_SET.forEach( ad , (i,v) -> {
			final DBColumn dbColumn = getDBColumn( v );
			tbl.setDBColumn( i,dbColumn );
		} );

		ad += SELECT_SET.size();								// 7.0.2.1 (2019/03/04)

		// MD5 ｶﾗﾑを追加｡
		if( useMD5 ) {
			final DBColumn dbColumn = getDBColumn( "MD5" );
			defClms[ad] = "";									// ﾃﾞｨﾚｸﾄﾘの場合は､MD5計算しません｡
			tbl.setDBColumn( ad++,dbColumn );
		}

		// 6.2.2.0 (2015/03/27) TEXT ｶﾗﾑを追加
		if( useText ) {
			final DBColumn dbColumn = getDBColumn( "TEXT" );
			defClms[ad] = "";									// ﾃﾞｨﾚｸﾄﾘの場合は､TEXT計算しません｡
			tbl.setDBColumn( ad++,dbColumn );
		}

		// TO_PARENT､TO_NAMEｶﾗﾑを追加
		if( useUpdateClms ) {
			for( int i=0; i<USE_UPDATE_CLM.length; i++ ) {
				final DBColumn dbColumn = getDBColumn( USE_UPDATE_CLM[i] );
				defClms[ad] = dbColumn.getDefault();			// 初期値を指定しておく
				tbl.setDBColumn( ad++,dbColumn );
			}
		}

		// 追加ｶﾗﾑのaddClmsｶﾗﾑを追加
		for( int i=0; i<addClms.length; i++ ) {
			final DBColumn dbColumn = getDBColumn( addClms[i] );
			defClms[ad] = dbColumn.getDefault();					// 初期値を指定しておく
			tbl.setDBColumn( ad++,dbColumn );
		}

		return tbl ;
	}

	/**
	 * DBTableModel に､ﾌｧｲﾙ情報をｾｯﾄします｡
	 * ﾌｧｲﾙ情報は､[WRITABLE],LEVEL,FILE_TYPE,PARENT,NAME,LASTMODIFIED,FILE_LENGTH,RWH,[MD5],[TO_PARENT,TO_NAME],[････] です｡
	 *
	 * useWritable=true の場合､先頭ｶﾗﾑに､WRITABLE ｶﾗﾑを追加します｡
	 * useMD5=true の場合､MD5ｶﾗﾑを追加したうえで､MD5計算を行います(ﾌｧｲﾙのみ計算します)｡
	 * useUpdateClms=true の場合TO_PARENT､TO_NAMEｶﾗﾑを追加します｡
	 * addClms で指定されたｶﾗﾑをその後ろに追加します｡
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 指定ｶﾗﾑ追加機能追加
	 * @og.rev 5.7.4.3 (2014/03/28) useWritable,useMD5属性追加
	 * @og.rev 6.2.2.0 (2015/03/27) TEXTｶﾗﾑを追加したうえで､ﾌｧｲﾙの内容を読み込むかどうか[true/false]を指定します(初期値:false)｡
	 * @og.rev 6.2.3.0 (2015/05/01) POIUtil のﾒｿｯﾄﾞ名変更に伴う対応
	 * @og.rev 6.2.4.2 (2015/05/29) POIUtil#extractor の判定方法変更
	 * @og.rev 6.4.2.0 (2016/01/29) DateSet.getDate( String ) を利用するように修正します｡
	 * @og.rev 7.0.2.1 (2019/03/04) fromLen属性追加｡
	 * @og.rev 7.1.0.0 (2020/01/20) fromLen属性で､CanonicalFileで区切り文字'￥'が消えるため､そのため､PARENTに'￥'が残る現象の対応｡
	 * @og.rev 7.2.6.0 (2020/06/30) nameOnly 属性 を追加します｡
	 *
	 * @param	rowNo	ｾｯﾄする行番号
	 * @param	lvl  	ｾｯﾄするﾚﾍﾞﾙ
	 * @param	fin		ﾌｧｲﾙ情報の元となるﾌｧｲﾙｵﾌﾞｼﾞｪｸﾄ
	 */
	private void addFileData( final int rowNo,final int lvl,final File fin ) {
		try {
			final File file = fin.getCanonicalFile();

			final String rwh = (    file.canRead()  ? "r" : "-" )
						+	( file.canWrite() ? "w" : "-" )
						+	( file.isHidden() ? "h" : "-" );

			final String lastModified = DateSet.getDate( file.lastModified(),"yyyyMMddHHmmss" );				// 6.4.2.0 (2016/01/29)

			final boolean isF = file.isFile();							// File=true,それ以外=false

			final int size = table.getColumnCount() ;
			String[] data = Arrays.copyOf( defClms,size );				// JDK1.6

			int ad=0;
			if( useWritable ) { ad++ ; }		// 単にひとつ進める｡初期値はｾｯﾄ済み｡

			// SELECT の 基本ｶﾗﾑの設定
			data[ad++] = String.valueOf( lvl ) ;						// LEVEL
			data[ad++] = isF ? "F" : "D" ;								// FILE_TYPE
			// 7.0.2.1 (2019/03/04) 検索結果のPARENT列から､from指定のﾊﾟｽを削除した相対ﾊﾟｽを作成
			final String parent = file.getParent() ;							// PARENTの元(正規ﾊﾟｽ)
//			data[ad++] = fromLen > 0 ? parent.substring( fromLen ) : parent ;	// PARENT
			data[ad++] = fromLen > 0 ? ( parent.length() < fromLen ? "" : parent.substring( fromLen ) ) : parent ;	// PARENT 7.1.0.0 (2020/01/20)

			// 7.2.6.0 (2020/06/30) nameOnly 属性 を追加します｡
			String fname = file.getName() ;								// NAME
			if( nameOnly ) {
				final int idx = fname.lastIndexOf( '.' );
				if( idx >= 0 ) { fname = fname.substring( 0,idx ); }
			}
//			data[ad++] = file.getName() ;								// NAME
			data[ad++] = fname ;										// NAME
			data[ad++] = lastModified ;									// LASTMODIFIED
			data[ad++] = String.valueOf( FileUtil.length( file ) );		// FILE_LENGTH 6.7.4.1 (2017/02/17) DIRの容量も計算する
			data[ad++] = rwh ;											// RWH

			// MD5 ｶﾗﾑを追加(ﾌｧｲﾙの場合のみ計算します)
			if( useMD5 && isF ) {
				data[ad++] = HybsCryptography.getMD5( file );
			}

			// 6.2.2.0 (2015/03/27) TEXT ｶﾗﾑを追加(ﾌｧｲﾙの場合のみ取り込みます)
			if( useText && isF ) {
				final String sufix = FileInfo.getSUFIX( file ) ;
				String val = "";
				try {
					if( POIUtil.isPOI( file ) ) {						// 6.2.4.2 (2015/05/29)
						val = POIUtil.extractor( file );
					}
					else if( "txt,jsp,java,xml,css,js".contains( sufix ) ) {
						val = POIUtil.extractor( file , "UTF-8");				// 6.2.3.0 (2015/05/01)
					}
					else {
						val = POIUtil.extractor( file , "Windows-31J");			// 6.2.3.0 (2015/05/01)
					}
				}
				catch( final RuntimeException ex ) {
					// 変換に失敗しても､処理は継続する｡
					final String errMsg = "ﾌｧｲﾙのﾃｷｽﾄ変換に失敗しました｡[" + fin + "]"
								+ " ROW=[" + rowNo + "]"
								+ CR + ex.getMessage();
					System.err.println( errMsg );
				}
				data[ad++] = val;
			}

			// useUpdateClms=true 時の TO_PARENT､TO_NAMEｶﾗﾑや､addClmsの追加ｶﾗﾑは､初期値のみｾｯﾄ
			// 初期値ｾｯﾄは､Arrays.copyOf で､defClms のｺﾋﾟｰで完了｡

			table.addColumnValues( data );
		}
		catch( final IOException ex ) {
			final String errMsg = "正式なﾌｧｲﾙ名の取得に失敗しました｡[" + fin + "]"
						+ " ROW=[" + rowNo + "]"
						+ CR + ex.getMessage();
			throw new HybsSystemException( errMsg,ex );
		}
	}

	/**
	 * 【TAG】ﾌｧｲﾙの検索元となるﾃﾞｨﾚｸﾄﾘを指定します
	 *		(初期値:FILE_URL[={@og.value SystemData#FILE_URL}])｡
	 *
	 * @og.tag ﾌｧｲﾙの検索元となるﾃﾞｨﾚｸﾄﾘを指定します｡
	 *
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたﾃﾞｨﾚｸﾄﾘ名の最後が"\"or"/"で終わっていない場合に､"/"を付加する｡
	 * @og.rev 6.4.2.1 (2016/02/05) URLの最後に､"/" を追加する処理を廃止｡
	 * @og.rev 6.4.2.1 (2016/02/05) HybsSystem.url2dir に引数追加｡
	 *
	 * @param	url ﾌｧｲﾙの検索元となるﾃﾞｨﾚｸﾄﾘ
	 */
	public void setFrom( final String url ) {
		final String furl = nval( getRequestParameter( url ),null );
		from = HybsSystem.url2dir( from,furl,"." );			// 6.4.2.1 (2016/02/05)
	}

	/**
	 * 【TAG】多段階展開するか､１ﾚﾍﾞﾙ展開するかどうか[true/false]を指定します(初期値:false)｡
	 *
	 * @og.tag
	 * 初期値は､false (１ﾚﾍﾞﾙ) です｡
	 *
	 * @param	mlti 多段階展開するか [true:する/false:１ﾚﾍﾞﾙ]
	 */
	public void setMulti( final String mlti ) {
		multi = nval( getRequestParameter( mlti ),multi );
	}

	/**
	 * 【TAG】多段階展開するﾚﾍﾞﾙを指定します(初期値:100)｡
	 *
	 * @og.tag
	 * 多段階展開するﾚﾍﾞﾙを指定します(初期値:100)｡
	 *
	 * @param	lvl 多段階展開するﾚﾍﾞﾙ
	 */
	public void setLevel( final String lvl ) {
		level = nval( getRequestParameter( lvl ),level );
	}

	/**
	 * 【TAG】ｿｰﾄするｶﾗﾑ名を指定します(一つのみ)｡
	 *
	 * @og.tag
	 * ｿｰﾄするｶﾗﾑ名を､"LEVEL","FILE_TYPE","PARENT","NAME","LASTMODIFIED","FILE_LENGTH","RWH"
	 * から一つ選びます｡
	 * これは､複数ｶﾗﾑでのｿｰﾄはできません｡
	 * 逆順にｿｰﾄする場合は､desc属性を true にｾｯﾄください｡
	 * + をつけても､無効(ｶﾗﾑ名がないということ)でｴﾗｰになります｡
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 新規追加
	 * @og.rev 6.3.4.0 (2015/08/01) Arrays.toString から String.join に置き換え｡
	 * @og.rev 6.4.3.4 (2016/03/11) String配列 から､Setに置き換えます｡
	 *
	 * @param	clm ｿｰﾄするｶﾗﾑ名 (一つのみ､逆順はﾏｲﾅｽを付ける)
	 * @see		#setDesc( String )
	 */
	public void setOrderBy( final String clm ) {
		orderBy = nval( getRequestParameter( clm ),orderBy );

		if( orderBy != null && ! check( orderBy, SELECT_SET ) ) {
			final String errMsg = "指定の orderBy は､指定できません｡" + CR
							+ "orderBy=[" + orderBy + "] "				+ CR
							+ "orderByList=" + String.join( ", " , SELECT_SET ) ;
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】表示順を逆転するかどうか[true/false]を指定します(初期値:false)｡
	 *
	 * @og.tag
	 * orderBy 属性で指定した表示順を､逆順にするかどうかを指定できます｡
	 * 初期値は､false (昇順) です｡
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 新規追加
	 *
	 * @param	flag 表示順逆順 [逆順:true/正順:false]
	 * @see		#setOrderBy( String )
	 */
	public void setDesc( final String flag ) {
		desc = nval( getRequestParameter( flag ),desc );
	}

	/**
	 * 【TAG】先頭ｶﾗﾑに､WRITABLE ｶﾗﾑを追加するかどうか[true/false]を指定します(初期値:false)｡
	 *
	 * @og.tag
	 * ﾌｧｲﾙ検索結果の１ﾚｺｰﾄﾞ単位に､書き込み許可/禁止属性を付けるには､
	 * ｶﾗﾑ列の先頭に､WRITABLE ｶﾗﾑを追加する必要があります｡
	 * 初期値は､false (追加しない) です｡
	 *
	 * @og.rev 5.7.4.3 (2014/03/28) 新規追加
	 *
	 * @param	flag WRITABLEｶﾗﾑ追加 [true:する/false:しない]
	 */
	public void setUseWritable( final String flag ) {
		useWritable = nval( getRequestParameter( flag ),useWritable );
	}

	/**
	 * 【TAG】MD5ｶﾗﾑを追加したうえで､MD5計算を行うかどうか[true/false]を指定します(初期値:false)｡
	 *
	 * @og.tag
	 * ﾌｧｲﾙの改変等をﾁｪｯｸするには､ﾌｧｲﾙのﾊｯｼｭ値を拾う必要があります｡
	 * ﾀｲﾑｽﾀﾝﾌﾟとｻｲｽﾞ（LASTMODIFIED,FILE_LENGTH）でも､類似の処理は可能ですが､
	 * より､厳密な一致をみるなら､MD5でﾊｯｼｭした結果を突き合わせるのがﾍﾞｽﾄです｡
	 * useMD5=true に設定すると､MD5 というｶﾗﾑを追加したうえで､MD5計算結果をｾｯﾄします｡
	 * 初期値は､false (追加しない) です｡
	 *
	 * @og.rev 5.7.4.3 (2014/03/28) 新規追加
	 *
	 * @param	flag MD5ｶﾗﾑ追加 [true:する/false:しない]
	 */
	public void setUseMD5( final String flag ) {
		useMD5 = nval( getRequestParameter( flag ),useMD5 );
	}

	/**
	 * 【TAG】TEXTｶﾗﾑを追加したうえで､ﾌｧｲﾙの内容を読み込むかどうか[true/false]を指定します(初期値:false)｡
	 *
	 * @og.tag
	 * ﾌｧｲﾙの内容を取得する場合に､true に設定します｡
	 * 初期値は､false (追加しない) です｡
	 *
	 * @og.rev 6.2.2.0 (2015/03/27) TEXTｶﾗﾑを追加したうえで､ﾌｧｲﾙの内容を読み込むかどうか[true/false]を指定します(初期値:false)｡
	 *
	 * @param	flag TEXTｶﾗﾑ追加 [true:する/false:しない]
	 */
	public void setUseText( final String flag ) {
		useText = nval( getRequestParameter( flag ),useText );
	}

	/**
	 * 【TAG】TO_PARENT､TO_NAMEｶﾗﾑを追加するかどうか[true/false]を指定します(初期値:false)｡
	 *
	 * @og.tag
	 * fileUpdateﾀｸﾞでは､ﾌｧｲﾙのCOPYやMOVEが出来ますが､そのｺﾋﾟｰ先､移動先の
	 * ﾌｧｲﾙを行ごとに指定する場合､TO_PARENT､TO_NAMEｶﾗﾑという固定名のｶﾗﾑが
	 * 必要です｡
	 * これを､addClms 属性で指定する代わりに､この属性で､true をｾｯﾄすることで､
	 * 自動的に追加されます｡
	 * 初期値は､false (追加しない) です｡
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 新規追加
	 *
	 * @param	flag TO_PARENT､TO_NAMEｶﾗﾑ追加 [true:追加する/false:追加しない]
	 * @see		#setAddClms( String )
	 * @see		org.opengion.hayabusa.taglib.FileUpdateTag
	 */
	public void setUseUpdateClms( final String flag ) {
		useUpdateClms = nval( getRequestParameter( flag ),useUpdateClms );
	}

	/**
	 * 【TAG】検索結果のｶﾗﾑ列に追加するｶﾗﾑ名を､CSV形式で指定します｡
	 *
	 * @og.tag
	 * ﾃﾞﾌｫﾙﾄのｶﾗﾑ名､[WRITABLE],LEVEL,FILE_TYPE,PARENT,NAME,LASTMODIFIED,FILE_LENGTH,RWH,[MD5],[TO_PARENT,TO_NAME]
	 * 以外に､指定のｶﾗﾑ名を追加することが可能です｡
	 * これは､ﾌｧｲﾙ検索結果以外の項目を追加して､ﾃﾞｰﾀﾍﾞｰｽに書き込む場合に､利用できます｡
	 * 並び順は､ﾃﾞﾌｫﾙﾄｶﾗﾑの後ろに､指定のｶﾗﾑの順番で付きます｡
	 * ここで追加したｶﾗﾑには､ｶﾗﾑﾘｿｰｽの初期値がｾｯﾄされます｡
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 新規追加
	 *
	 * @param	clms 追加するｶﾗﾑ名(CSV形式)
	 * @see		#setUseUpdateClms( String )
	 */
	public void setAddClms( final String clms ) {
		final String tmpClms = nval( getRequestParameter( clms ),null );
		if( tmpClms != null && tmpClms.length() > 0 ) {
			addClms = StringUtil.csv2Array( tmpClms );
		}
	}

	/**
	 * 【TAG】ﾌｧｲﾙ名が､指定されたﾌｧｲﾙﾀｲﾌﾟ[DIR/FILE]と一致した場合､ｽﾙｰ(選択)されます(初期値:null)｡
	 * @og.tag
	 * 大文字小文字は区別しません｡
	 * ﾌｧｲﾙﾀｲﾌﾟ は､DIR,FILE が指定できます｡
	 * DIR は､ﾃﾞｨﾚｸﾄﾘのみ検索します｡(階層下がりも行います)
	 * FILEは､ﾌｧｲﾙのみ検索します｡(階層下がりも行います)
	 * 引数が null の場合は､追加しません｡(つまり､すべてｽﾙｰされます｡)
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) fileType ﾒｿｯﾄﾞで選択対象指定の追加
	 *
	 * @param    str ﾌｧｲﾙﾀｲﾌﾟ [null:ｽﾙｰ/DIR:ﾃﾞｨﾚｸﾄﾘのみ検索/FILE:ﾌｧｲﾙのみ検索]
	 */
	public void setFileType( final String str ) {
		final String tmp = nval( getRequestParameter( str ),fileType );
		if( tmp == null						||
			"DIR".equalsIgnoreCase( tmp )	||
			"FILE".equalsIgnoreCase( tmp ) ) {
				fileType = tmp;
		}
		else {
			// ﾌｧｲﾙﾀｲﾌﾟに不正な値が設定された場合は､ｴﾗｰになる｡
			final String errMsg = "この､fileType 属性には､DIR,FILE 以外は指定できません｡["
						+ tmp + "]";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】from属性で指定された基準ﾌｧｲﾙ/ﾌｫﾙﾀﾞ自体をﾘｽﾄに追加するかどうか[true/false]を指定します(初期値:true)｡
	 * @og.tag
	 * 初期値はtrue(追加する)です｡
	 *
	 * @og.rev 5.3.9.0 (2011/09/01) 新規作成
	 *
	 * @param    flag 基準をﾘｽﾄに追加するかどうか [true:追加する/false:追加しない]
	 */
	public void setAddFrom( final String flag ) {
		addFrom = nval( getRequestParameter( flag ),addFrom );
	}

	/**
	 * 【TAG】検索結果のPARENT列から､fromBase指定のﾊﾟｽを削除した相対ﾊﾟｽを作成します｡
	 * @og.tag
	 * 実ﾌｧｲﾙをURL化する場合に､階層をｽｷｬﾝしたPARENTから､fromBase分の文字列を削除します｡
	 * PARENTに相対ﾊﾟｽを指定することが可能になります｡
	 * 元となるﾌｧｲﾙﾊﾟｽは､getCanonicalFile() で作成した正規ﾊﾟｽ名になるため､
	 * fromBase のﾊﾟｽの文字数も､同様に正規ﾊﾟｽ名から作成します｡
	 *
	 * @og.rev 7.0.2.1 (2019/03/04) fromBase属性追加に伴い､fromLen変数を用意｡
	 * @og.rev 7.0.5.0 (2019/09/16) fromBase のﾊﾟｽの文字数も､正規ﾊﾟｽ名から作成｡
	 * @og.rev 7.1.0.0 (2020/01/20) fromLen属性で､CanonicalFileで区切り文字'￥'が消えるため､そのため､PARENTに'￥'が残る現象の対応｡
	 *
	 * @param    base PARENT列から､fromBase指定のﾊﾟｽを削除した相対ﾊﾟｽを作成
	 */
	public void setFromBase( final String base ) {
		final String fromBase = nval( getRequestParameter( base ),null );
		if( fromBase != null ) {
			// 7.0.5.0 (2019/09/16) fromBase のﾊﾟｽの文字数も､正規ﾊﾟｽ名から作成｡
//			fromLen = fromBase.length();
			try {
				fromLen = new File(fromBase).getCanonicalPath().length();		// 7.0.5.0 (2019/09/16)
				// 7.1.0.0 (2020/01/20)
				final char ch = fromBase.charAt( fromBase.length()-1 );
				if( ch == '\\' || ch == '/' ) { fromLen++; }					// getCanonicalPathでは､最後の区切り記号が消える
			}
			catch( final IOException ex ) {
				final String errMsg = "fromBaseの正式なﾌｧｲﾙ名の取得に失敗しました｡[" + base + "]"
							+ CR + ex.getMessage();
				throw new HybsSystemException( errMsg,ex );
			}
		}
	}

	/**
	 * 【TAG】ﾌｧｲﾙの拡張子を除いた名前部分のみの値で行います(初期値:false)｡
	 *
	 * @og.tag
	 * ﾌｧｲﾙ検索の値を､ﾌｧｲﾙの拡張子を取り除いた値のみで､作成します｡
	 * 初期値は､false (拡張子付きﾌｧｲﾙ名でﾘｽﾄ) です｡
	 *
	 * @og.rev 7.2.6.0 (2020/06/30) nameOnly 属性 を追加します｡
	 *
	 * @param	flag ﾌｧｲﾙの拡張子を除いた名前部分のみで作成するかどうか [true:名前部分のみ/false:ﾌｧｲﾙ名]
	 */
	public void setNameOnly( final String flag ) {
		nameOnly = nval( getRequestParameter( flag ),nameOnly );
	}

	/**
	 * 【TAG】ｼｽﾃﾑ定数でｸﾗｳﾄﾞ設定されていても､ｸﾗｳﾄﾞ環境を使用しない場合､trueを指定します(初期値:false)｡
	 *
	 * @og.tag
	 * ｸﾗｳﾄﾞ設定は､ｼｽﾃﾑ定数の『CLOUD_TARGET』と『CLOUD_BUCKET』の設定で自動的に使用しますが､
	 * どうしてもﾛｰｶﾙでのみ使いたい場合は､この属性を true に設定します｡
	 * 標準はfalse:設定どおりとなります｡
	 *
	 * true/false以外を指定した場合はfalse扱いとします｡
	 *
	 * @og.rev 8.0.1.0 (2021/10/29) useLocal 属性を追加｡
	 *
	 * @param flag ﾛｰｶﾙ環境のみ [true:ﾛｰｶﾙのみ/false:設定どおり]
	 */
	public void setUseLocal( final String flag ) {
		useLocal = nval( getRequestParameter( flag ),useLocal );
	}

	/**
	 * FileFilterｵﾌﾞｼﾞｪｸﾄをｾｯﾄします｡
	 * これは､BODY 部に登録した､FileWhereﾀｸﾞによって設定された
	 * ﾌｧｲﾙﾌｨﾙﾀｰです｡
	 *
	 * @param	filter	ｵﾌﾞｼﾞｪｸﾄ
	 */
	protected void setFileFilter( final FileFilter filter ) {
		this.filter = filter;
	}

	/**
	 * このｵﾌﾞｼﾞｪｸﾄの文字列表現を返します｡
	 * 基本的にﾃﾞﾊﾞｯｸﾞ目的に使用します｡
	 *
	 * @return このｸﾗｽの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "multi"			,multi			)
				.println( "level"			,level			)
				.println( "from"			,from			)
				.println( "orderBy"			,orderBy		)
				.println( "desc"			,desc			)
				.println( "addClms"			,Arrays.toString( addClms )	)
				.println( "defClms"			,Arrays.toString( defClms )	)
				.println( "fileType"		,fileType		)
				.println( "useWritable"		,useWritable	)
				.println( "useMD5"     		,useMD5			)
				.println( "useText"			,useText		)
				.println( "useUpdateClms"	,useUpdateClms	)
				.println( "addFrom"			,addFrom		)
				.println( "filter"			,filter			)				// 6.8.0.0 (2017/06/02)
				.fixForm().toString()
			+ CR
			+ super.toString() ;
	}
}
