/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.io;

import static org.opengion.fukurou.util.HybsConst.CR ;		// 6.1.0.0 (2014/12/26)
import org.opengion.fukurou.util.Closer ;
import org.opengion.fukurou.util.LogWriter;
import org.opengion.fukurou.util.ColorMap;			// 6.0.2.2 (2014/10/03)
// import org.opengion.fukurou.db.DBUtil;			// 6.0.4.0 (2014/11/28)
import org.opengion.fukurou.db.ResultSetValue;		// 6.0.4.0 (2014/11/28)
// import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.DBTableModel;

import java.sql.Connection;
import java.sql.ResultSet;
// import java.sql.ResultSetMetaData;				// 6.0.4.0 (2014/11/28)
import java.sql.SQLException;
import java.sql.Statement;

import java.util.List;
import java.util.ArrayList;
import java.util.Arrays;
// import java.util.Locale;
import java.util.Set;
import java.util.HashSet;

import java.awt.Color;								// 6.0.2.2 (2014/10/03)

import org.jfree.data.Range;
import org.jfree.data.category.DefaultCategoryDataset;

/**
 * HybsCategoryDataset は、org.jfree.data.category.DefaultCategoryDataset を継承したサブクラスで、
 * HybsDataset インターフェースの実装クラスになっています。
 * これは、JDBCCategoryDatasetの データベース機能と、DBTableModel から Dataset を作成する機能を
 * 兼ね備えています。
 * HybsDataset インターフェースは、シリーズのラベル指定、カテゴリカラーバー、パレート図用積上げ
 * 計算などの処理を行うための、インターフェースで、それらの処理も、HybsCategoryDataset に実装します。
 * 
 * このクラスでは、検索結果を内部で持っておき、getValue(int row, int column)
 * メソッドで直接値を返します。
 * 
 * select category,series1,series2,series3,･･･ from ･･･
 * series の横持ち(標準と同じ) 対応です。
 * category カラムの値は、カテゴリのラベルになり、series1,2,3 のラベルがシリーズラベル、値が
 * seriesの値になります。
 *
 * カテゴリのカラー名の指定を行う場合、最後のカラムが、カラー名の文字列になります。
 * select category,series1,series2,series3,･･･,color from ･･･
 * color文字列の検索結果は、Dataset には含まれません。
 *
 * その場合、color カラムがシリーズとして認識されない様に、ChartDatasetTag で、useCategoryColor="true"
 * を指定しておく必要があります。このフラグは、HybsCategoryDataset を使う処理以外では効果が
 * ありません（シリーズとして使用されてしまう）のでご注意ください。
 * このフラグは、カテゴリカラーバーを使う場合には必要ですが、カテゴリカラーバーと（例えばパレート図）
 * を合成する場合に、パレート図側にも useCategoryColor="true" を設定しておけば、同じSQL または、
 * DBTableModel を使う事ができるというためのフラグです。
 *
 * なお、Colorコードは、このクラスで作成しますが、Renderer に与える必要があります。
 * 通常のRenderer には、categoryにカラーを指定する機能がありませんので、HybsBarRenderer に
 * setCategoryColor( Color[] ) メソッドを用意します。(正確には、HybsDrawItem インターフェース)
 * このRenderer で、getItemPaint( int  , int )メソッドをオーバーライドすることで、カテゴリごとの
 * 色を返します。
 *
 * @og.rev 5.8.5.0 (2015/02/06) 6.0.2.2 (2014/10/03) からの逆移植
 *
 * @version  5.8.5.0 (2015/02/06)
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.6,
 */
public class HybsCategoryDataset extends DefaultCategoryDataset implements HybsDataset {
	private static final long serialVersionUID = 602220141003L ;

	private final Set<String> cateCheck	= new HashSet<String>();		// category の重複チェック
	private final int	hsCode	= Long.valueOf( System.nanoTime() ).hashCode() ;	// 5.1.9.0 (2010/08/01) equals,hashCode

	private String[]	seriesLabels		;
	private boolean		isColorCategory		;				// 6.0.2.2 (2014/10/03)
	private boolean		isParetoData		;				// 6.0.2.2 (2014/10/03)

	private Number[][]	numdata				;
	private Color[]		categoryColor		;
	private Range		range				;

	/**
	 * CategoryDataset を構築するに当たり、初期パラメータを設定します。
	 *
	 * @og.rev 6.0.2.2 (2014/10/03) 新規追加
	 *
	 * @param lbls  シリーズのラベル名配列
	 * @param isColCate  カテゴリのカラー名の指定有無(true:使用する)
	 * @param isPareto   パレート図用のDatasetとして処理するかどうか(true:処理する)
	 */
	public void initParam( final String[] lbls , final boolean isColCate , final boolean isPareto ) {
		// 6.0.2.5 (2014/10/31) refactoring
//		seriesLabels	= lbls;
		if( lbls != null ) { seriesLabels = lbls.clone(); }
		isColorCategory	= isColCate;
		isParetoData	= isPareto;
	}

	/**
	 * コネクションと、SQL文字列から、CategoryDataset のデータを作成します。
	 * 元となる処理は、org.jfree.data.jdbc.JDBCCategoryDataset#executeQuery( Connection,String ) です。
	 *
	 * このメソッドでは、先に #initParam(String[],boolean,isPareto) のパラメータを使用して
	 * 検索した結果のデータを加工、処理します。
	 * また、内部的に、データをキャッシュする事と、データ範囲を示す レンジオブジェクト を作成します。
	 *
	 * @og.rev 6.0.2.2 (2014/10/03) 新規追加
	 * @og.rev 6.0.2.3 (2014/10/19) パレート図は、100分率にする。
	 * @og.rev 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
	 * @og.rev 6.9.7.0 (2018/05/14) データ０件のときは、処理を中断します。 5.10.15.0(2019/08/30)で追加
	 *
	 * @param con  コネクション
	 * @param query  SQL文字列
	 *
	 * @throws SQLException データベースアクセス時のエラー
	 * @see		org.jfree.data.jdbc.JDBCCategoryDataset#executeQuery( Connection,String )
	 * @see		org.opengion.fukurou.db.ResultSetValue
	 */
	public void execute( final Connection con, final String query ) throws SQLException {

		// Range を予め求めておきます。
		double minimum = Double.POSITIVE_INFINITY;
		double maximum = Double.NEGATIVE_INFINITY;
		double sum     = 0.0d;					// 6.0.2.3 (2014/10/19) パレート図用合計

		List<Color> colorList = null;			// 6.0.2.2 (2014/10/03) カテゴリカラー

		Statement statement = null;
		ResultSet resultSet = null;
		try {
			statement = con.createStatement();
			resultSet = statement.executeQuery(query);

			// 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
//			ResultSetMetaData metaData = resultSet.getMetaData();
			final ResultSetValue rsv = new ResultSetValue( resultSet );

//			int dataSize = metaData.getColumnCount() -1;	// series の個数は、category 分を引いた数。
			int dataSize = rsv.getColumnCount() -1;			// series の個数は、category 分を引いた数。
			if( isColorCategory ) {							// ColorCategory使用時
				colorList	= new ArrayList<Color>();		// カテゴリカラー
				dataSize--;									// 最終カラムが Colorコードなので、マイナスする。
			}

			if( dataSize<1 ) {
				final String errMsg = "JDBCCategoryDataset.executeQuery() : insufficient columns "
							+ "returned from the database. \n"
							+ " SQL=" + query ;
				throw new SQLException( errMsg );
			}

			// 6.0.2.0 (2014/09/19) シリーズのラベル名配列を使うときは、シリーズ数必要。
			if( seriesLabels != null && seriesLabels.length < dataSize ) {
				final String errMsg = "seriesLabels を使用する場合は、必ずシリーズ数以上指定してください。"
								+ CR
								+ " seriesLabels=" + Arrays.toString( seriesLabels )
								+ CR
								+ " seriesLabels.length=" + seriesLabels.length
								+ " dataSize=" + dataSize
								+ CR ;
				throw new IllegalArgumentException( errMsg );
			}

			String[] series  = new String[dataSize];
			// 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
			final String[] names   = rsv.getNames();
//			int[] columnType = new int[dataSize];
			// ORACLEの引数は、配列＋１から始まるので、metaDataはi＋2から取得。series と、seriesLabels は０から始まる。
			for( int i=0; i<dataSize; i++ ) {
				series[i] = seriesLabels != null && seriesLabels[i] != null
									? seriesLabels[i]
//									: metaData.getColumnLabel(i+2).toUpperCase( Locale.JAPAN );
									: names[i+1] ;
//				columnType[i] = metaData.getColumnType(i+2);
			}

			final List<Number[]> rowList = new ArrayList<Number[]>();
			// 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
//			while (resultSet.next()) {
			while (rsv.next()) {
				Number[] clmList = new Number[dataSize];
				// first column contains the row key...
	 			// 6.0.2.0 (2014/09/19) columnKeyは、series , rowKey は、category に変更する。
//				String category = resultSet.getString(1);					// 4.3.3.6 (2008/11/15) Generics警告対応
				final String category = uniqCategory( resultSet.getString(1) );	// 6.0.2.3 (2014/10/10) categoryの重複回避

				for( int i=0; i<dataSize; i++ ) {					// 6.0.2.2 (2014/10/03) dataSize 分回す。
					Number value = null;
					// 6.0.2.1 (2014/09/26) org.opengion.fukurou.db.DBUtil に、移動
					try {
						// JDBCのアドレス指定は、+2 する。(category 分と、ｱﾄﾞﾚｽが１から始まる為。)
//						value = DBUtil.getNumber( columnType[i],resultSet.getObject(i+2) );
						// ResultSetValueのカラム番号は、+1 する。(category 分があるため)
						value = rsv.getNumber( i+1 );
					}
					catch( SQLException ex ) {		// 6.0.4.0 (2014/11/28) ResultSetValue を使用するので。
						LogWriter.log( ex );
					}
					catch( RuntimeException ex ) {
						LogWriter.log( ex );
					}

					clmList[i] = value;
					addValue(value, series[i], category);		// 6.0.2.0 (2014/09/19) columnKeyは、series , rowKey は、category に変更する。
					// Range 求め
					if( value != null ) {
						final double dbl = value.doubleValue();
						if( isParetoData ) {					// 6.0.2.3 (2014/10/19) パレート図用合計
							sum += dbl ;
						} else {
							if( dbl     < minimum ) { minimum = dbl; }
							if( maximum < dbl     ) { maximum = dbl; }
						}
					}
				}
				rowList.add( clmList );
				// 6.0.2.2 (2014/10/03) ColorCategory は、最後のカラム
				if( isColorCategory ) {
					// 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
//					String colStr = resultSet.getString(dataSize+2);		// 最後のカラム
					final String colStr = rsv.getValue(dataSize+1);				// 最後のカラム
					final Color color   = ColorMap.getColorInstance( colStr );	// 6.0.2.1 (2014/09/26) StringUtil → ColorMap
					colorList.add( color );
				}
			}
			// 6.9.7.0 (2018/05/14) データ０件のときは、処理を中断します。
			if( rowList.isEmpty() ) { return; }
			
			numdata = rowList.toArray( new Number[dataSize][rowList.size()] );
		}
		finally {
			Closer.resultClose( resultSet );
			Closer.stmtClose( statement );
		}

		// colorList が null でないかどうかで判定。
		if( isColorCategory && colorList != null ) {
			categoryColor = colorList.toArray( new Color[colorList.size()] );
		}

		// 6.0.2.3 (2014/10/19) パレート図は、100分率にする。
//		if( isParetoData ) { maximum = changeParetoData(); }
		if( isParetoData ) {
			changeParetoData( sum );
			minimum = 0.0;
			maximum = 100.0;
		}

		range = new Range( minimum, maximum );
	}

	/**
	 * DBTableModelオブジェクトから、CategoryDataset のデータを作成します。
	 * openGionの独自処理メソッドです。
	 *
	 * このメソッドでは、先に #initParam(String[],boolean,isPareto) のパラメータを使用して
	 * 検索した結果のデータを加工、処理します。
	 * また、内部的に、データをキャッシュする事と、データ範囲を示す レンジオブジェクト を作成します。
	 *
	 * @og.rev 6.0.2.2 (2014/10/03) 新規追加
	 * @og.rev 6.0.2.3 (2014/10/19) パレート図は、100分率にする。
	 * @og.rev 6.9.7.0 (2018/05/14) データ０件のときは、処理を中断します。 5.10.15.0(2019/08/30)追加
	 *
	 * @param table DBTableModelオブジェクト
	 * @see		#execute( Connection,String )
	 */
	public void execute( final DBTableModel table ) {
		// 6.9.7.0 (2018/05/14) データ０件のときは、処理を中断します。
		if( table == null || table.getRowCount() == 0 ) { return; }
		
 		final int clmNo = table.getColumnCount();
		final int rowNo = table.getRowCount();

		// Range を予め求めておきます。
		double minimum = Double.POSITIVE_INFINITY;
		double maximum = Double.NEGATIVE_INFINITY;
		double sum     = 0.0d;					// 6.0.2.3 (2014/10/19) パレート図用合計

		int dataSize = clmNo -1;						// series の個数は、category 分を引いた数。
		List<Color> colorList = null;			// 6.0.2.2 (2014/10/03) カテゴリカラー
		if( isColorCategory ) {							// ColorCategory使用時
			colorList	= new ArrayList<Color>();		// カテゴリカラー
			dataSize--;									// 最終カラムが Colorコードなので、マイナスする。
		}

		numdata = new Number[rowNo][clmNo];

		// ※ DBTableModel の row,col と、Dataset の row,col は、逆になっています。
		for( int row=0; row<rowNo; row++ ) {
//			String   category = table.getValue( row,0 );					// １番目(アドレス=0)はカラムの設定値
			final String   category = uniqCategory( table.getValue( row,0 ) );	// 6.0.2.3 (2014/10/10) categoryの重複回避
			final String[] vals     = table.getValues( row );
			for( int clm=0; clm<dataSize; clm++ ) {
				final String sval = vals[clm+1];				// ２番目(アドレス=1)からカラムデータを取得
				final double val  = ( sval == null || sval.isEmpty() ) ? 0.0d : Double.parseDouble( sval ) ;

				addValue( val , seriesLabels[clm] , category );		// val,row,clm
//				numdata[row][clm] = new Double( val );
				numdata[row][clm] = Double.valueOf( val );			// 6.0.2.4 (2014/10/17) 効率の悪いメソッド
				// Range 求め
				if( isParetoData ) {					// 6.0.2.3 (2014/10/19) パレート図用合計
					sum += val ;
				} else {
					if( val     < minimum ) { minimum = val; }
					if( maximum < val     ) { maximum = val; }
				}
			}

			// 6.0.2.2 (2014/10/03) ColorCategory は、最後のカラム
			if( isColorCategory ) {
				final String colStr = vals[dataSize+1];			// 最後のカラム
				final Color color   = ColorMap.getColorInstance( colStr );	// 6.0.2.1 (2014/09/26) StringUtil → ColorMap
				colorList.add( color );
			}
		}

		// colorList が null でないかどうかで判定。
		if( isColorCategory && colorList != null ) {
			categoryColor = colorList.toArray( new Color[colorList.size()] );
		}

		// 6.0.2.3 (2014/10/19) パレート図は、100分率にする。
//		if( isParetoData ) { maximum = changeParetoData(); }
		if( isParetoData ) {
			changeParetoData( sum );
			minimum = 0.0;
			maximum = 100.0;
		}

		range = new Range( minimum, maximum );
	}

	/**
	 * 指定された行列から、数字オブジェクトを取得します。
	 *
	 * @param	row 	行番号(シリーズ：横持=clm相当)
	 * @param	column	カラム番号(カテゴリ：縦持ち=row相当)
	 *
	 * @return	指定の行列の値
	 */
	@Override
	public Number getValue( final int row, final int column ) {
		// 注意：行列の順序が逆です。
		return numdata[column][row];
	}

	/**
	 * レンジオブジェクトを取得します。(独自メソッド)
	 *
	 * @return	レンジオブジェクト
	 */
	public Range getRange() {
		return range;
	}

	/**
	 * パレート図用のDatasetに値を書き換えます。(独自メソッド)
	 *
	 * 色々と方法はあると思いますが、簡易的に、内部の Number配列を
	 * 積上げ計算して、パレート図用のデータを作成します。
	 * レンジオブジェクト も変更します。
	 *
	 * ※ 注意：親クラスの内部に持っている実データは変更されていないので、
	 * 場合によっては、おかしな動きをするかもしれません。
	 * その場合は、上位にもデータをセットするように変更する必要があります。
	 *
	 * なお、行列の順序が、イメージと異なりますので、注意願います。
	 * (columnは、series , row は、category で、シリーズを積み上げます)
	 *
	 * @og.rev 6.0.2.1 (2014/09/26) 新規追加
	 * @og.rev 6.0.2.2 (2014/10/03) HybsDataset i/f
	 * @og.rev 6.0.2.3 (2014/10/19) パレート図は、100分率にする。
	 *
	 * @param  sum データの合計
	 */
//	private double changeParetoData() {
	private void changeParetoData( final double sum ) {
//		if( numdata == null || numdata.length == 0 || numdata[0].length == 0 ) { return 0.0d; }
		if( numdata == null || numdata.length == 0 || numdata[0].length == 0 || sum == 0.0 ) { return ; }

		final int rowCnt = numdata[0].length ;
		final int clmCnt = numdata.length ;

//		double maximum = Double.NEGATIVE_INFINITY;
		for( int rowNo=0; rowNo<rowCnt; rowNo++ ) {			// 行列が逆。
			double val = 0.0;		// 初期値
			for( int clmNo=0; clmNo<clmCnt; clmNo++ ) {		// 積上げ計算するカラムでループを回す。
				final Number v1Num = numdata[clmNo][rowNo];
				if(v1Num != null) {
					val += v1Num.doubleValue();				// 積上げ計算は、元の値のままにしておきます。
				}
				// データをセットするときに、１００分率にします。
//				numdata[clmNo][rowNo] = new Double(val);
				numdata[clmNo][rowNo] = Double.valueOf( Math.round( val * 1000.0 / sum ) / 10.0 );
	// きちんと計算するなら、BigDecimal で、スケールを指定して四捨五入すべき・・・かも
	//			java.math.BigDecimal bd = new BigDecimal( val * 100.0 / sum );
	//			numdata[clmNo][rowNo] = bd.setScale( 1, java.math.RoundingMode.HALF_UP );
			}
//			if( maximum < val ) { maximum = val; }			// パレート図用の積上げなので、最大値は、最後のデータ
		}

//		return maximum;
	}

	/**
	 * categoryカラー配列を取得します。(独自メソッド)
	 *
	 * このクラスは、一番最後のカラムを、色文字列として処理し、categoryにColorを指定できます。
	 * select文で指定されていなかった場合は、null を返します。
	 *
	 * select category,series1,series2,series3,･･･,color from ･･･
	 *
	 * @og.rev 6.0.2.2 (2014/10/03) 新規追加
	 *
	 * なお、Colorコードは、このクラスで作成しますが、Renderer に与える必要があります。
	 * 通常のRenderer には、categoryにカラーを指定する機能がありませんので、HybsBarRenderer に
	 * setCategoryColor( Color[] ) メソッドを用意します。(正確には、HybsDrawItem インターフェース)
	 * このRenderer で、getItemPaint( int  , int )メソッドをオーバーライドすることで、カテゴリごとの
	 * 色を返します。
	 * この設定を行うと、シリーズは、カテゴリと同一色になります。
	 *
	 * @return	categoryカラー配列(なければ null)
	 */
	public Color[] getCategoryColor() {
		// 6.0.2.5 (2014/10/31) refactoring
//		return categoryColor;
		return ( categoryColor == null ) ? null : categoryColor.clone();
	}

	/**
	 * category の重複をさけて、必要であれば、新しいカテゴリ名を作成します。
	 *
	 * カテゴリが同じ場合、JFreeChartでは、表示されません。これは、同じカテゴリと認識され
	 * 値が上書きされるためです。
	 * この問題は、なかなか気づきにくく、デバッグ等に時間がかかってしまいます。
	 * 重複チェックを行い、警告してもよいのですが、ここでは、新しいカテゴリ名を作成することで
	 * エラーを回避しつつ、とりあえずグラフ表示をするようにします。
	 *
	 * @og.rev 6.0.2.3 (2014/10/10) 新規追加
	 *
	 * @param	category	元のカテゴリ名
	 * @return	新しい元のカテゴリ名
	 */
	private String uniqCategory( final String category ) {
		String newCate = category ;
		int i = 0;
		while( !cateCheck.add( newCate ) ) {	// すでに存在している場合。
			newCate = category + "(" + (i++) + ")" ;
		}

		return newCate ;
	}

	/**
	 * この文字列と指定されたオブジェクトを比較します。
	 *
	 * 親クラスで、equals メソッドが実装されているため、警告がでます。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) findbug対応
	 * @og.rev 5.1.9.0 (2010/08/01) findbug対応
	 *
	 * @param	object	比較するオブジェクト
	 *
	 * @return	Objectが等しい場合は true、そうでない場合は false
	 */
	@Override
	public boolean equals( final Object object ) {
		if( super.equals( object ) ) {
			return hsCode == ((HybsCategoryDataset)object).hsCode;
		}
		return false;
	}

	/**
	 * このオブジェクトのハッシュコードを取得します。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) findbug対応
	 * @og.rev 5.1.9.0 (2010/08/01) findbug対応
	 *
	 * @return	ハッシュコード
	 */
	@Override
	public int hashCode() { return hsCode ; }
}
