package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.io.JsChartData;

/**
 * 設定された値をJsChartDataに設定し、
 * JsChartTagのJsChartDataリストに追加するタグです。
 * 
 * @og.formSample
 * ●形式：&lt;og:jsChartData chartColumn="…" … /&gt;
 * ●body:なし
 * 
 * ●Tag定義：
 * &lt;og:jsChartData
 *  chartColumn         ○ 【TAG】ﾁｬｰﾄのｶﾗﾑ名を指定します。(必須)
 *  label                  【TAG】凡例の値を指定します。
 *  fill                   【TAG】線下を塗りつぶすかどうか[true/false]を指定します。(初期値:false)
 *  tension                【TAG】線の伸張を指定します。0で直線になります。(初期値:0.4)
 *  borderColor            【TAG】線の色を指定します。
 *  borderWidth            【TAG】線の幅を指定します。
 *  backgroundColor        【TAG】ﾃﾞｰﾀの背景色を指定します。
 *  optionAttributes       【TAG】その他ｵﾌﾟｼｮﾝを指定します。
 *  /&gt; 
 * 
 * ●使用例
 * &lt;og:jsChart...&gt;
 *     &lt;og:jsChartData
 *         chartColumn ="CLM1"
 *         label       ="ラベル"
 *         fill        ="true"
 *         tension     ="0"
 *         borderColor ="rbga(150,150,150,0.7)"
 *         borderWidth ="2"
 *     /&gt;
 * &lt;/og:jsChart&gt;
 * 
 * 
 * @og.group 画面表示
 * 
 * @version	5.9.17.2		2017/02/08
 * @author	T.OTA
 * @since	JDK7.0
 * 
 */
public class JsChartDataTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。{@VALUE} */
	private static final String		VERSION				= "5.9.17.2 (2017/02/07)";
	private static final long		serialVersionUID	= -2188726810088630172L;
	private static final String[]	TYPE_BOOLEAN		= new String[] { "true", "false" };
	// 変数宣言
	private String					chartColumn			= null;
	private String					label				= null;
	private String					fill				= "false";
	private String					tension				= "0.4";
	private String					borderColor			= null;
	private String					borderWidth			= null;
	private String					backgroundColor		= null;
	private String					optionAttributes	= null;
	private transient JsChartData	jsData				= new JsChartData();

	/**
	 * Taglibの開始タグが見つかった時に処理する doStartTag() を オーバーライドします。
	 * 
	 * @return 後続処理の指示
	 */
	@Override
	public int doStartTag() {
		return SKIP_BODY; // Body を評価しない
	}

	/**
	 * Taglibの終了タグが見つかった時に処理する doEndTag() を オーバーライドします。
	 * 
	 * @return 後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();

		JsChartTag jsChartTag = (JsChartTag) findAncestorWithClass( this, JsChartTag.class );

		if( jsChartTag == null ) {
			String errMsg = "jsChart タグが見つかりませんでした。";
			throw new HybsSystemException( errMsg );
		}

		jsData.setChartColumn( chartColumn );
		jsData.setLabel( label );
		jsData.setFill( fill );
		jsData.setTension( tension );
		jsData.setBorderColor( borderColor );
		jsData.setBorderWidth( borderWidth );
		jsData.setBackgroundColor( backgroundColor );
		jsData.setOptionAttributes( optionAttributes );

		jsChartTag.addJsChartData( jsData );

		return EVAL_PAGE;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 * 
	 */
	@Override
	protected void release2() {
		super.release2();
		jsData = new JsChartData();
		chartColumn = null;
		label = null;
		fill = "false";
		tension = "0.4";
		borderColor = null;
		borderWidth = null;
		backgroundColor = null;
		optionAttributes = null;
	}

	/**
	 * チャートカラム を設定します。
	 * 
	 * @param val
	 */
	public void setChartColumn( String val ) {
		chartColumn = getRequestParameter( val );
	}

	/**
	 * ラベル を設定します。
	 * 
	 * @param label
	 */
	public void setLabel( String label ) {
		this.label = getRequestParameter( label );
	}

	/**
	 * フィル(線より下の塗りつぶし) を設定します。
	 * 
	 * @param fill
	 */
	public void setFill( String fill ) {
		this.fill = getRequestParameter( fill );

		checkPara( this.fill, TYPE_BOOLEAN, "fill" );
	}

	/**
	 * 伸張 を設定します。
	 * @param tension
	 */
	public void setTension( String tension ) {
		this.tension = getRequestParameter( tension );
	}

	/**
	 * 線の色 を設定します。
	 * 
	 * @param borderColor
	 */
	public void setBorderColor( String borderColor ) {
		this.borderColor = getRequestParameter( borderColor );
	}

	/**
	 * 線の幅 を設定します。
	 * 
	 * @param borderWidth
	 */
	public void setBorderWidth( String borderWidth ) {
		this.borderWidth = getRequestParameter( borderWidth );
	}

	/**
	 * 背景色 を設定します。
	 * 
	 * @param backgroundColor
	 */
	public void setBackgroundColor( String backgroundColor ) {
		this.backgroundColor = getRequestParameter( backgroundColor );
	}

	/**
	 * オプション情報 を設定します。
	 * 
	 * @param optionAttributes
	 */
	public void setOptionAttributes( String optionAttributes ) {
		this.optionAttributes = getRequestParameter( optionAttributes );
	}

	/**
	 * ﾊﾟﾗﾒｰﾀﾁｪｯｸ用ﾒｿｯﾄﾞ
	 * 
	 * @param trg チェック対象
	 * @param list 設定可能なリスト
	 * @param trgStr チェック対象の文字列(ｴﾗｰ表示用)
	 */
	private void checkPara( String trg, String[] list, String trgStr ) {
		if( trg != null && trg.length() > 0 && !check( trg, list ) ) {
			StringBuilder errMsg = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
			errMsg.append( "指定の" ).append( trgStr ).append( "は指定できません。" );
			errMsg.append( HybsSystem.CR );
			errMsg.append( trgStr ).append( "=[" ).append( trg ).append( "]" );
			errMsg.append( HybsSystem.CR );
			for( int i = 0; i < list.length; i++ ) {
				errMsg.append( " | " );
				errMsg.append( list[i] );
			}
			throw new HybsSystemException( errMsg.toString() );
		}
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 * 
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
			.println( "VERSIION", VERSION )
			.println( "chartColumn", chartColumn )
			.println( "label", label )
			.println( "fill", fill )
			.println( "tension", tension )
			.println( "borderColor", borderColor )
			.println( "borderWidth", borderWidth )
			.println( "backgroundColor", backgroundColor )
			.println( "optionAttributes", optionAttributes ).fixForm().toString();
	}
}
