/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBEventColumn;
import org.opengion.hayabusa.db.DBLastSql;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.resource.GUIInfo;
import org.opengion.hayabusa.resource.LabelInterface;
import org.opengion.hayabusa.resource.ResourceFactory;
import org.opengion.hayabusa.resource.ResourceManager;
import org.opengion.hayabusa.resource.UserInfo;

import static org.opengion.fukurou.util.StringUtil.nval;
import org.opengion.fukurou.system.DateSet;										// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.system.BuildNumber;									// 6.4.2.0 (2016/01/29) hayabusa.common.BuildNumber → fukurou.system.BuildNumber に移動
import org.opengion.fukurou.system.HybsConst;									// 6.1.0.0 (2014/12/26)
import org.opengion.fukurou.system.MsgUtil;										// 6.4.3.2 (2016/02/19)
// import org.opengion.fukurou.system.OgRuntimeException;						// 6.9.2.1 (2018/03/12)
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.HybsDateUtil;
import org.opengion.fukurou.util.ToString;										// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.db.DBFunctionName;
import org.opengion.fukurou.db.ApplicationInfo;
import org.opengion.fukurou.db.Transaction;										// 6.3.6.1 (2015/08/28)
import org.opengion.fukurou.db.TransactionReal;									// 6.3.6.1 (2015/08/28)

import java.io.IOException;
import java.io.ObjectInputStream;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.LinkedHashMap;													// 6.2.5.1 (2015/06/12)
import java.util.Set;															// 6.4.3.4 (2016/03/11)
import java.util.Scanner;														// 5.1010.1 (2019/04/05)

import jakarta.servlet.ServletContext;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.http.Cookie;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.servlet.http.HttpSession;
import jakarta.servlet.jsp.JspWriter;
import jakarta.servlet.jsp.tagext.BodyContent;
import jakarta.servlet.jsp.tagext.BodyTagSupport;
import jakarta.servlet.jsp.tagext.TryCatchFinally;

/**
 * TagSupport から継承されたｻﾌﾞｸﾗｽです｡
 *
 * 汎用属性 のsetterﾒｿｯﾄﾞと、Attributes ｵﾌﾞｼﾞｪｸﾄを持っています｡
 * それ以外に、{&#064;XXXX} 変数の対応と、lang属性のﾒｿｯﾄﾞも用意しています｡
 *
 * language 属性は、個々のﾘｿｰｽのﾛｹｰﾙを指定できます｡
 * 通常は、ﾕｰｻﾞｰ情報の lang 属性をﾃﾞﾌｫﾙﾄで使用し、ｾｯﾄされていない場合は、
 * ﾘｸｴｽﾄ情報のﾛｹｰﾙから取得します｡
 *
 * 以下に、このﾒｿｯﾄﾞ内で定義される属性を記述します｡
 *
 *       language           【TAG】ﾀｸﾞ内部で使用する言語ｺｰﾄﾞ[ja/en/zh/…]を指定します
 *       lbl                【TAG】ﾗﾍﾞﾙﾘｿｰｽのﾗﾍﾞﾙIDを指定します
 *       debug              【TAG】ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうか[true/false]を指定します (初期値:false)
 *       caseKey            【TAG】このﾀｸﾞ自体を利用するかどうかの条件ｷｰを指定します (初期値:null)
 *       caseVal            【TAG】このﾀｸﾞ自体を利用するかどうかの条件値を指定します (初期値:null)
 *       caseNN             【TAG】指定の値が、null/ｾﾞﾛ文字列 でない場合(Not Null=NN)は、このﾀｸﾞは使用されます (初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ｾﾞﾛ文字列 の場合は、このﾀｸﾞは使用されます (初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このﾀｸﾞは使用されます (初期値:判定しない)
 *       scope              【TAG】ｷｬｯｼｭする場合のｽｺｰﾌﾟ(request,session)を指定します (初期値:session)
 *
 * 各属性は、{&#064;XXXX} 変数が使用できます｡
 * これは、ServletRequest から、XXXX をｷｰに値を取出し、この変数に割り当てます｡
 * つまり、このXXXXをｷｰにﾘｸｴｽﾄすれば、この変数に値をｾｯﾄすることができます｡
 *
 * http://localhost/query.jsp?KEY1=VLA1&amp;KEY2=VAL2
 *
 * のようなﾘｸｴｽﾄで、{&#064;KEY1} とすれば、VAL1 がｾｯﾄされます｡
 *
 * このﾀｸﾞは、ﾗﾍﾞﾙ部分と入力ﾌｨｰﾙﾄﾞ部分がﾃｰﾌﾞﾙﾀｸﾞの&lt;td&gt;により左右に分割されます｡
 * HTML 表示時は、前後に&lt;tr&gt;ﾀｸﾞで囲って整形できます｡
 *
 * @og.group 画面制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
class CommonTagSupport extends BodyTagSupport implements TryCatchFinally {
	/** このﾌﾟﾛｸﾞﾗﾑのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "8.1.2.1 (2022/03/25)" ;
	private static final long serialVersionUID = 812120220325L ;

	/** ｼｽﾃﾑの改行ｺｰﾄﾞを設定します。*/
	protected static final String CR		= HybsConst.CR;						// 6.1.0.0 (2014/12/26) refactoring
	/** HTMLの改行ｺｰﾄﾞ(&lt;br /&gt; + CR)を設定します。*/
	protected static final String BR		= HybsConst.BR;						// 6.1.0.0 (2014/12/26) refactoring
	/** StringBilderなどの初期値を設定します。	{@value} */
	protected static final int BUFFER_MIDDLE = HybsConst.BUFFER_MIDDLE;			// 6.1.0.0 (2014/12/26) refactoring

	private transient Attributes		attri		= new Attributes();
	private transient ResourceManager	resource	;
	private transient UserInfo			userInfo	;
	private transient GUIInfo			guiInfo		;
	private transient HttpSession		session		;
	private transient ServletRequest	request		;
	/** 6.4.3.1 (2016/02/12) 取得元の HashMap を ConcurrentHashMap に置き換え。 */
	private transient Map<String,String[]>	requestCache;						// 3.5.6.2 (2004/07/05)
	private transient LabelInterface	msglbl		;							// 4.0.0 (2005/01/31)

	private String		language	;
	private boolean		debugFlag	;											// 3.5.5.3 (2004/04/09)
	private boolean		isReqNull	;
	private boolean		quotCheck	;											// 4.0.0 (2005/08/31)
	private String		scope		= "session";								// "request","page","session","application"
	// 3.1.7.0 (2003/05/02) value値の使用可否を指定する、useValue 属性を追加。
	private Long		startTransaction;										// 3.6.0.8 (2004/11/19)
	private int[]		rowNo		;											// 4.0.0 (2005/01/31)
	private boolean		xssCheck	= HybsSystem.sysBool( "USE_XSS_CHECK" );	// 5.0.0.2 (2009/09/15)
	private boolean		useTrans	;											// 5.1.6.0 (2010/05/01)

	private String		caseKey		;											// 5.2.2.0 (2010/11/01) 新規追加
	private String		caseVal		;											// 5.2.2.0 (2010/11/01) 新規追加
	private boolean		caseNN		= true;										// 5.6.7.0 (2013/07/27) 新規追加
	private boolean		caseNull	= true;										// 5.6.8.0 (2013/09/06) 新規追加
	private boolean		caseIf		= true;										// 6.2.6.0 (2015/06/19) 新規追加

	private boolean		isSanitized	;											// 5.7.4.2 (2014/03/20) 新規追加

	/**
	 * Taglibの開始ﾀｸﾞが見つかったときに処理する doStartTag() を ｵｰﾊﾞｰﾗｲﾄﾞします｡
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doStartTag() {
		return SKIP_BODY ;														// Body を評価しない
	//	return( EVAL_BODY_INCLUDE );											// Body ｲﾝｸﾙｰﾄﾞ( extends TagSupport 時)
	//	return EVAL_BODY_BUFFERED ;												// Body を評価する｡( extends BodyTagSupport 時)
	}

	/**
	 * Taglibのﾀｸﾞ本体を処理する doAfterBody() を ｵｰﾊﾞｰﾗｲﾄﾞします｡
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		return SKIP_BODY ;														// Body を評価しない
	//	return( EVAL_BODY_AGAIN );												// Body を再評価( extends TagSupport 時)
	//	return EVAL_BODY_BUFFERED ;												// Body を再評価( extends BodyTagSupport 時)
	}

	/**
	 * Taglibの終了ﾀｸﾞが見つかったときに処理する doEndTag() を ｵｰﾊﾞｰﾗｲﾄﾞします｡
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応｡release2() を doEndTag()で呼ぶ｡
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();															// 4.0.0 (2005/02/28)

		return EVAL_PAGE ;														// ﾍﾟｰｼﾞの残りを評価する｡( extends TagSupport 時)
	//	return(SKIP_PAGE);														// ﾍﾟｰｼﾞの残りの処理を行わない｡
	}

	/**
	 * ﾀｸﾞの処理中(ｾｯﾀｰﾒｿｯﾄﾞを除く)の例外を全て受け取ります｡
	 *
	 * ﾀｸﾞの中のﾎﾞﾃﾞｨ部の評価中、または Tag.doStartTag(), Tag.doEndTag(),
	 * IterationTag.doAfterBody(), BodyTag.doInitBody() のいずれもの
	 * ﾒｿｯﾄﾞの中で、Throwableが投げられたときに呼出されます｡
	 *
	 * このﾒｿｯﾄﾞはｾｯﾀｰﾒｿｯﾄﾞの中でThrowableが起きた場合は呼出されません｡
	 *
	 * @og.rev 3.5.0.0 (2003/09/17) TryCatchFinally ｲﾝﾀｰﾌｪｰｽを適用｡
	 * @og.rev 6.8.5.0 (2018/01/09) ﾀｸﾞﾘﾌﾞで発生したｴﾗｰを、session に登録しておきます｡
	 * @og.rev 6.9.2.1 (2018/03/12) ﾀｸﾞﾘﾌﾞで発生したｴﾗｰを、session に登録する処理を一旦廃止
	 * @og.rev 7.0.6.4 (2019/11/29) HybsSystemException を throw するように変更します｡
	 *
	 * @param	th	このﾀｸﾞを通過してきたThrowableな例外
	 */
	@Override
	public void doCatch( final Throwable th ) throws Throwable {
//		// 6.8.5.0 (2018/01/09) ※ なぜか、うまく使えていません。
//		final Throwable cause = (Throwable)getSessionAttribute( "CommonTagThrowable" );
//
//		if( cause != null ) {
//			th.addSuppressed( cause );
//		}
//
//		setSessionAttribute( "CommonTagThrowable" , th );

//		throw th;

		if( th instanceof HybsSystemException ) {
			throw th;
		}
		else {
			throw new HybsSystemException( th );								// 7.0.6.4 (2019/11/29)
		}
	}

	/**
	 * ﾀｸﾞの処理毎の、doEndTag()の後で呼出されます｡
	 *
	 * Tag,IterationTag,BodyTagを実装した全てのｸﾗｽの doEndTag()の後で呼出されます｡
	 * このﾒｿｯﾄﾞはﾀｸﾞのﾎﾞﾃﾞｨ部や Tag.doStartTag(), Tag.doEndTag(), IterationTag.doAfterBody(),
	 * BodyTag.doInitBody()のすべてのﾒｿｯﾄﾞで例外が発生した後でも呼出されます｡
	 *
	 * このﾒｿｯﾄﾞはｾｯﾀｰﾒｿｯﾄﾞの中でThrowableが起きた場合は呼出されません｡
	 *
	 * このﾒｿｯﾄﾞからは例外を投げるべきではありません｡
	 * このﾒｿｯﾄﾞは呼び出し毎のﾃﾞｰﾀの整合性をとることとﾘｿｰｽ管理の
	 * 動作をさせることを意図しています｡
	 *
	 * @og.rev 3.5.0.0 (2003/09/17) TryCatchFinally ｲﾝﾀｰﾌｪｰｽを適用｡
	 */
	@Override
	public void doFinally() {
		release2();
	}

	/**
	 * ﾀｸﾞﾘﾌﾞｵﾌﾞｼﾞｪｸﾄをﾘﾘｰｽします｡
	 * ｷｬｯｼｭされて再利用されるので、ﾌｨｰﾙﾄﾞの初期設定を行います｡
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) ｶｽﾀﾑﾀｸﾞの release() ﾒｿｯﾄﾞを追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応｡release2() を doEndTag()で呼ぶ｡
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応｡HybsRequestWrapper 廃止｡直接 Mapでｷｬｯｼｭする｡
	 * @og.rev 3.1.3.0 (2003/04/10) ｴﾝｺｰﾄﾞ情報の取得を廃止する｡
	 * @og.rev 3.1.7.0 (2003/05/02) value値の使用可否を指定する、useValue 属性を追加｡
	 * @og.rev 3.5.5.3 (2004/04/09) debugFlag を String ではなく、boolean 型に変更
	 * @og.rev 3.6.0.8 (2004/11/19) startTransaction 属性を追加
	 * @og.rev 3.8.0.2 (2005/07/11) rightNow 属性を追加
	 * @og.rev 5.0.0.2 (2009/09/15) XSS対応
	 * @og.rev 5.1.6.0 (2010/05/01) DBLastSQL周りの実装見直し
	 * @og.rev 5.1.8.0 (2010/07/01) isNullSet 属性廃止にともなう、useValue 属性廃止
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey,caseVal 属性の追加
	 * @og.rev 5.3.2.0 (2011/02/01) paramNames 属性の追加
	 * @og.rev 5.6.7.0 (2013/07/27) caseNN 属性の追加
	 * @og.rev 5.6.8.0 (2013/09/06) caseNull 属性の追加
	 * @og.rev 5.7.4.1 (2014/03/14) rightNow 属性の廃止
	 * @og.rev 5.7.4.1 (2014/03/14) isSanitized 属性の追加
	 * @og.rev 6.2.6.0 (2015/06/19) caseIf 属性の追加
	 */
	protected void release2() {
		language			= null;
		attri				= new Attributes();
		resource			= null;
		debugFlag			= false;											// 3.5.5.3 (2004/04/09)
		userInfo			= null;
		guiInfo				= null;
		session				= null;
		request				= null;
		isReqNull			= false;
		scope				= "session";										// "request","page","session","application"
		requestCache		= null;
		startTransaction	= null;												// 3.6.0.8 (2004/11/19)
		rowNo				= null;												// 4.0.0 (2005/01/31)
		msglbl				= null;												// 4.0.0 (2005/01/31)
		quotCheck			= false;											// 4.0.0 (2005/08/31)
		xssCheck			= HybsSystem.sysBool( "USE_XSS_CHECK" );			// 5.0.0.2 (2009/09/15)
		useTrans			= false;											// 5.1.6.0 (2010/05/01)
		caseKey				= null;												// 5.2.2.0 (2010/11/01)
		caseVal				= null;												// 5.2.2.0 (2010/11/01)
		caseNN				= true;												// 5.6.7.0 (2013/07/27) 新規追加
		caseNull			= true;												// 5.6.8.0 (2013/09/06) 新規追加
		caseIf				= true;												// 6.2.6.0 (2015/06/19) 新規追加
		isSanitized			= false;											// 5.7.4.2 (2014/03/20) 新規追加。一応入れておくが都度、初期化しています。
	}

	/**
	 * 【TAG】ﾀｸﾞ内部で使用する言語ｺｰﾄﾞ[ja/en/zh/…]を指定します｡
	 *
	 * @og.tag
	 * この言語ｺｰﾄﾞに基づいて、表示のﾗﾍﾞﾙをﾘｿｰｽから作成します｡
	 *
	 * @param	lang	言語ｺｰﾄﾞ [ja/en/zh/…]
	 * @see		#getLanguage()
	 */
	public void setLanguage( final String lang ) {
		language = getRequestParameter( lang );
	}

	/**
	 * 言語ｺｰﾄﾞ[ja/en/zh/…]を取得します｡
	 *
	 * 言語ｺｰﾄﾞがｾｯﾄされている場合は、設定値を優先します｡
	 * 設定されていない場合は、ﾛｸﾞｲﾝﾕｰｻﾞｰの持つLANG属性を、それが null の場合は、
	 * 実行環境のﾘｸｴｽﾄの言語を返します｡
	 *
	 * @og.rev 2.1.1.0 (2002/11/08) ｾｯｼｮﾝ情報から取得できない場合に、ｸﾗｲｱﾝﾄの
	 *                      ﾘｸｴｽﾄ情報のﾛｹｰﾙから取得する処理を追加
	 * @og.rev 2.2.0.0 (2002/12/17) ｾｯｼｮﾝ情報から取得するのではなく、ﾕｰｻﾞｰ情報より
	 *                      取得するように変更｡そこにない場合は、ﾘｸｴｽﾄ情報の
	 *                      ﾛｹｰﾙから取得する
	 * @og.rev 6.0.2.5 (2014/10/31) 初期ｴﾗｰ発生時は、ﾕｰｻﾞｰも取得できないので、null でも返す｡
	 * @og.rev 6.4.3.2 (2016/02/19) なにも取得できない場合は、"ja" を返すように変更｡
	 *
	 * @return	言語ｺｰﾄﾞ[ja/en/zh/…]
	 * @see		#setLanguage( String )
	 */
	protected String getLanguage() {
		// 6.0.2.5 (2014/10/31) 初期ｴﾗｰ発生時は、ﾕｰｻﾞｰも取得できないので、null でも返す｡
		try {
			if( language == null ) {
				language = getUser().getLang();
				if( language == null ) {
					language = getRequest().getLocale().getLanguage();
				}
			}
		}
		catch( final RuntimeException ex ) {
			final String errMsg = "言語ｺｰﾄﾞを取得できませんでした。"
						+ ex.getMessage() ;
			System.err.println( errMsg );
		}

		// 6.4.3.2 (2016/02/19) なにも取得できない場合は、"ja" を返すように変更。
		return language == null ? "ja" : language ;
	}

	/**
	 * 【TAG】ﾗﾍﾞﾙﾘｿｰｽのﾗﾍﾞﾙIDを指定します｡
	 *
	 * @og.tag
	 * ﾗﾍﾞﾙを変更するときに、lbl属性を使います｡
	 *
	 * ﾗﾍﾞﾙID は、所定の language に基づく ResourceManager の
	 * getLabelData( id ) を呼び出し、その結果のLabelInterfaceを使用します｡
	 * getMsglbl() で取り出せます｡
	 * ﾗﾍﾞﾙとﾒｯｾｰｼﾞは統一されました｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) label 変数は、生ﾃﾞｰﾀを保管するように変更｡
	 *
	 * @param	lbl	ﾗﾍﾞﾙID
	 * @see		#getMsglbl()
	 */
	public void setLbl( final String lbl ) {
		msglbl = (LabelInterface)getResource().getLabelData( getRequestParameter( lbl ) ) ;		// 4.0.0 (2005/01/31)
	}

	/**
	 * 【TAG】ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうか[true/false]を指定します(初期値:false)｡
	 *
	 * @og.tag
	 * ﾃﾞﾊﾞｯｸﾞ情報を [true:出力する/false:しない]を指定します｡
	 * 出力形式自体は、個々のﾀｸﾞによって異なります｡
	 *
	 * @og.rev 3.5.5.3 (2004/04/09) debugFlag を String ではなく、boolean 型に変更
	 *
	 * @param	flag	ﾃﾞﾊﾞｯｸﾞ出力 [true:する/それ以外:しない]
	 */
	public void setDebug( final String flag ) {
		debugFlag = nval( getRequestParameter( flag ),debugFlag );
	}

	/**
	 * 【TAG】ｷｬｯｼｭする場合のｽｺｰﾌﾟ[request/page/session/application]を指定します(初期値:session)｡
	 *
	 * @og.tag
	 * "request","page","session","application" が指定できます｡
	 * JSPのｽｺｰﾌﾟは｢変数の有効範囲｣を表すもので、ﾌﾚｰﾑﾜｰｸ上では、
	 * 主にﾃｰﾌﾞﾙﾓﾃﾞﾙを管理するｽｺｰﾌﾟを指します｡
	 *
	 * <table class="plain">
	 *   <caption>ﾃｰﾌﾞﾙﾓﾃﾞﾙを管理するｽｺｰﾌﾟ変数の有効範囲</caption>
	 *   <tr><th>ｽｺｰﾌﾟ	</th><th>変数の有効範囲</th><th>説明</th></tr>
	 *   <tr><td>page		</td><td>JSPﾍﾟｰｼﾞ内</td>
	 *       <td>そのJSPﾍﾟｰｼﾞ内のみで有効です｡ﾌﾚｰﾑﾜｰｸ的には、JSPﾍﾟｰｼﾞにまたがる処理が多いため、ほとんど使う機会はありません｡</td></tr>
	 *   <tr><td>request	</td><td>HTTPﾘｸｴｽﾄ</td>
	 *       <td>ﾘｸｴｽﾄの一連の処理期間中に有効な変数で、ﾒﾓﾘに多くの情報を残したくない場合に利用します｡検索系やﾎﾟｯﾌﾟｱｯﾌﾟのJSP画面等に利用します｡</td></tr>
	 *   <tr><td>session	</td><td>HTTPｾｯｼｮﾝ</td>
	 *       <td>初期設定されているｽｺｰﾌﾟで、ﾛｸﾞｲﾝﾕｰｻﾞｰ単位にﾛｸﾞｱｳﾄまで保持されます｡
	 *           内部的には、同じｷｰﾜｰﾄﾞ（tableId）で管理しているため、検索都度、破棄されます｡
	 *           （ｶﾞｰﾍﾞｼﾞｺﾚｸｼｮﾝにて破棄されるのを待ちます｡）</td></tr>
	 *   <tr><td>application</td><td>Webｱﾌﾟﾘｹｰｼｮﾝ</td>
	 *       <td>ﾕｰｻﾞｰ間で共有する場合のｽｺｰﾌﾟになります｡JSP画面の開発では、まず使うことはありません｡</td></tr>
	 * </table>
	 *
	 * @param	scp	ｽｺｰﾌﾟ [request/page/session/application]
	 * @see		#getScope()
	 */
	public void setScope( final String scp ) {
		scope = nval( getRequestParameter( scp ),scope );
	}

	/**
	 * ｷｬｯｼｭする場合のｽｺｰﾌﾟ[request/page/session/application]を返します｡
	 *
	 * "request","page","session","application" があります｡
	 *
	 * @og.rev 3.5.5.8 (2004/05/20) 新規追加
	 *
	 * @return	ｽｺｰﾌﾟ [request/page/session/application]
	 * @see		#setScope( String )
	 */
	public String getScope() {
		return scope ;
	}

	/**
	 * 【TAG】このﾀｸﾞ自体を利用するかどうかの条件ｷｰを指定します(初期値:null)｡
	 *
	 * @og.tag
	 * caseKey.matches( caseVal ) の値が、true の場合は、このﾀｸﾞは使用されます｡
	 * false の場合は、このﾀｸﾞは使用されません｡
	 *
	 * caseKey , caseVal ともに null の場合は、true です｡どちらかが 非null の場合は、
	 * もう片方も 非null で且つcaseKey.matches( caseVal ) が成立する必要があります｡
	 * この属性は、caseKey , caseVal , caseNN , caseNull , caseIf とともに useTag() の判定で使用されます｡
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) 新規追加
	 * @og.rev 5.6.3.3 (2013/04/19) 条件変更のためのｺﾒﾝﾄ修正
	 *
	 * @param	ckey	条件ｷｰ (何も指定されない場合は使用すると判断)
	 * @see		#setCaseVal( String )
	 * @see		#useTag()
	 */
	public void setCaseKey( final String ckey ) {
		caseKey = nval( getRequestParameter( ckey ),caseKey );
	}

	/**
	 * 【TAG】このﾀｸﾞ自体を利用するかどうかの条件値を指定します(初期値:null)｡
	 *
	 * @og.tag
	 * caseKey.matches( caseVal ) の値が true の場合は、このﾀｸﾞは使用されます｡
	 * false の場合は、このﾀｸﾞは使用されません｡
	 *
	 * caseKey , caseVal ともに null の場合は、true です｡どちらかが 非null の場合は、
	 * もう片方も 非null で且つcaseKey.matches( caseVal ) が成立する必要があります｡
	 * この属性は、caseKey , caseVal , caseNN , caseNull , caseIf とともに useTag() の判定で使用されます｡
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) 新規追加
	 * @og.rev 5.6.3.3 (2013/04/19) 条件変更のためのｺﾒﾝﾄ修正
	 *
	 * @param	cval	条件値 (何も指定されない場合は使用すると判断)
	 * @see		#setCaseKey( String )
	 * @see		#useTag()
	 */
	public void setCaseVal( final String cval ) {
		caseVal = nval( getRequestParameter( cval ),caseVal );
	}

	/**
	 * 【TAG】このﾀｸﾞ自体を利用するかどうかの条件として、NotNullかどうか判定します(初期値:判定しない)｡
	 *
	 * @og.tag
	 * この値が、null/ｾﾞﾛ文字列 でない場合(Not Null=NN)はこのﾀｸﾞは使用されます｡
	 * null/ｾﾞﾛ文字列 の場合は、このﾀｸﾞは使用されません｡
	 * 何も指定しない場合は、使用されます｡
	 * caseNull と逆の動きをします｡
	 * {&#064;XXXX} で指定した場合は、値が設定されなければ使用されません｡
	 *
	 * この属性は、caseKey , caseVal , caseNN , caseNull , caseIf とともに useTag() の判定で使用されます｡
	 *
	 * @og.rev 5.6.7.0 (2013/07/27) 新規追加
	 *
	 * @param	cnn	NotNull判定値 (何も指定されない場合は使用すると判断)
	 * @see		#setCaseVal( String )
	 * @see		#useTag()
	 */
	public void setCaseNN( final String cnn ) {
		final String tempNN = nval( getRequestParameter( cnn ),null );

		caseNN = tempNN != null && !tempNN.isEmpty() ;
	}

	/**
	 * 【TAG】このﾀｸﾞ自体を利用するかどうかの条件として、Nullかどうか判定します(初期値:判定しない)｡
	 *
	 * @og.tag
	 * この値が、null/ｾﾞﾛ文字列 の場合はこのﾀｸﾞは使用されます｡
	 * null/ｾﾞﾛ文字列 でない場合は、このﾀｸﾞは使用されません｡
	 * 何も指定しない場合は、使用されます｡
	 * caseNN と逆の動きをします｡
	 * {&#064;XXXX} で指定した場合は、値が設定されていなければ使用されます｡
	 *
	 * この属性は、caseKey , caseVal , caseNN , caseNull , caseIf とともに useTag() の判定で使用されます｡
	 *
	 * @og.rev 5.6.8.0 (2013/09/06) 新規追加
	 *
	 * @param	cnul	Null判定値 (何も指定されない場合は使用すると判断)
	 * @see		#setCaseVal( String )
	 * @see		#useTag()
	 */
	public void setCaseNull( final String cnul ) {
		final String tempNull = nval( getRequestParameter( cnul ),null );

		caseNull = tempNull == null || tempNull.isEmpty() ;
	}

	/**
	 * 【TAG】このﾀｸﾞ自体を利用するかどうかの条件としてtrue/TRUEかどうか判定します(初期値:判定しない)｡
	 *
	 * @og.tag
	 * この値が、true/TRUE文字列 の場合はこのﾀｸﾞは使用されます｡
	 * それ以外の場合は、このﾀｸﾞは使用されません｡
	 * {&#064;XXXX} を指定した場合は、null/ｾﾞﾛ文字列でも使用しないと判定されますので、ご注意ください｡
	 *
	 * 7.0.1.1 (2018/10/22)
	 *  引数の先頭に、'!' を記述した場合、false/FALSE 判定を行います｡
	 *  ﾊﾟｰｽ後の値が、"true" か、"!false" の場合のみ、条件成立となります｡
	 *
	 *  "!{&#064;XXXX}" 指定の場合、null/ｾﾞﾛ文字列の場合は、"!" だけになり、使用されません
	 *  これは、caseIfの反転ではありません｡
	 *
	 * この属性は、caseKey , caseVal , caseNN , caseNull , caseIf とともに useTag() の判定で使用されます｡
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) caseIf 属性の追加
	 * @og.rev 7.0.1.1 (2018/10/22) 属性に、'!' を使用できるようにします｡
	 *
	 * @param	cif	true/TRUE判定 (null/ｾﾞﾛ文字列場合のは使用しないと判定)
	 * @see		#setCaseVal( String )
	 * @see		#useTag()
	 */
	public void setCaseIf( final String cif ) {
		// ※ 引数がnullの場合は、false。  nvl( flag,caseIf ) ではないので注意
//		caseIf = "true".equalsIgnoreCase( getRequestParameter( cif ) );
		final String flag = getRequestParameter( cif );		// null,ゼロ文字列もありうる。
		caseIf = "true".equalsIgnoreCase( flag ) || "!false".equalsIgnoreCase( flag );
	}

	/**
	 * このﾀｸﾞ自体を利用するかどうかの条件判定を行います｡
	 *
	 * caseNN &amp;&amp; caseNull &amp;&amp; caseIf &amp;&amp;
	 *    ( (caseKey == null &amp;&amp; caseVal == null) ||
	 *      (caseKey != null &amp;&amp; caseVal != null &amp;&amp; caseKey.matches( caseVal )))
	 * の結果を返します｡
	 *
	 * これは、ﾀｸﾞ本体に、条件式を登録できる機能です｡必要なﾀｸﾞには、tld ﾌｧｲﾙで、
	 * caseKey 、caseVal 、caseNN 、caseNull属性が使用できるように、設定します｡
	 * 各ﾀｸﾞを、equals ﾀｸﾞで括る方法では、ｿｰｽの見通しが悪くなるため、
	 * ある程度ﾀｸﾞ自身に判定機能を設けることで対応できるようにしました｡
	 * ただ、本来、JSP 側にﾛｼﾞｯｸを持ち込むのはよくないので、利用に関しては、
	 * 慎重にお願いします｡
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) 新規追加
	 * @og.rev 5.6.3.3 (2013/04/19) 条件変更(caseKey と caseVal が 両方とも null の場合のみ true)
	 * @og.rev 5.6.7.0 (2013/07/27) caseNN 属性追加｡先のcaseKey、caseVal 条件と、AND 結合になります｡
	 * @og.rev 5.6.8.0 (2013/09/06) caseNull 属性追加｡先の条件と、AND 結合になります｡
	 * @og.rev 6.2.6.0 (2015/06/19) caseIf 属性の追加
	 *
	 * @return	このﾀｸﾞ自体を利用するかどうか(true:利用する/false:利用しない)
	 * @see		#setCaseVal( String )
	 * @see		#setCaseKey( String )
	 * @see		#setCaseNN( String )
	 * @see		#setCaseIf( String )
	 */
	protected boolean useTag() {
//		return caseNN && caseNull && caseIf &&
//					( (caseKey == null && caseVal == null) ||
//					  (caseKey != null && caseVal != null && caseKey.matches( caseVal ))) ;
		return caseNN && caseNull && caseIf &&									// 6.9.7.0 (2018/05/14) PMD Useless parentheses.
					( caseKey == null && caseVal == null ||
					  caseKey != null && caseVal != null && caseKey.matches( caseVal ) ) ;
	}

	/**
	 * (通常は使いません)ﾀｸﾞで処理される処理がﾒｲﾝとなるﾄﾗﾝｻﾞｸｼｮﾝ処理かどうかを指定します(初期値:false)｡
	 *
	 * この値は、ﾌｧｲﾙﾀﾞｳﾝﾛｰﾄﾞ処理に影響します｡この値がtrueに指定された時にcommitされたDBTableModelが
	 * ﾌｧｲﾙﾀﾞｳﾝﾛｰﾄﾞの対象の表になります｡
	 *
	 * このﾊﾟﾗﾒｰﾀｰは、通常、各ﾀｸﾞにより実装され、ﾕｰｻﾞｰが指定する必要はありません｡
	 * 但し、1つのJSP内でDBTableModelが複数生成される場合に、前に処理したDBTableModelについてﾌｧｲﾙﾀﾞｳﾝﾛｰﾄﾞをさせたい
	 * 場合は、後ろでDBTableModelを生成するﾀｸﾞで、明示的にこの値をfalseに指定することで、ﾌｧｲﾙﾀﾞｳﾝﾛｰﾄﾞ処理の対象から
	 * 除外することができます｡
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @param	flag	ﾒｲﾝﾄﾗﾝｻﾞｸｼｮﾝ処理かどうか [true:ﾒｲﾝ/false:その他]
	 */
	protected void useMainTrans( final boolean flag ) {
		useTrans = flag;
	}

	/**
	 * ﾒｯｾｰｼﾞﾗﾍﾞﾙ(msglbl)を取得します｡
	 *
	 * ﾒｯｾｰｼﾞﾗﾍﾞﾙは、lbl属性で登録された値を、
	 * それぞれのﾘｿｰｽに応じて各言語に変換した結果を格納しています｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) msglbl は、LabelInterface ｵﾌﾞｼﾞｪｸﾄを利用
	 *
	 * @return	ﾒｯｾｰｼﾞﾗﾍﾞﾙ
	 * @see		#setLbl( String )
	 */
	protected String getMsglbl() {
		String rtn = null;

		if( msglbl != null ) { rtn = msglbl.getLabel(); }

		return rtn ;
	}

	/**
	 * ﾒｯｾｰｼﾞﾗﾍﾞﾙ(msglbl)のﾁｯﾌﾟｽ表記を取得します｡
	 *
	 * ﾒｯｾｰｼﾞﾗﾍﾞﾙは、lbl属性で登録された値を、
	 * それぞれのﾘｿｰｽに応じて各言語に変換した結果を格納しています｡
	 * 別途、title属性を指定している場合は、置き換えます｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) msglbl は、LabelInterface ｵﾌﾞｼﾞｪｸﾄを利用
	 *
	 * @return	ﾒｯｾｰｼﾞﾗﾍﾞﾙ
	 */
	protected String getLongLabel() {
		String rtn = null;

		if( msglbl != null ) {
			rtn = msglbl.getLongLabel( get( "title" ) );
		}

		return rtn ;
	}

	/**
	 * ﾒｯｾｰｼﾞﾗﾍﾞﾙ(LabelInterface)を取得します｡
	 *
	 * ﾒｯｾｰｼﾞﾗﾍﾞﾙは、lbl属性で登録された値を、
	 * それぞれのﾘｿｰｽに応じて各言語に変換した結果を格納しています｡
	 *
	 * @return	ﾒｯｾｰｼﾞﾗﾍﾞﾙ
	 */
	protected LabelInterface getLabelInterface() {
		return msglbl ;
	}

	/**
	 * ResourceManager を取得します｡
	 *
	 * ﾍﾟｰｼﾞｽｺｰﾌﾟ にｾｯﾄされた ResourceManager があれば、それを、
	 * なければ、language 属性よりﾃﾞﾌｫﾙﾄ ResourceManager を構築します｡
	 * LOCAL_RES_KEY で管理される ResourceManager は、LocalResourceTag で
	 * 登録されたﾘｿｰｽです｡これは、ﾛｰｶﾙﾘｿｰｽ情報として、使用されます｡
	 *
	 * @return	ResourceManagerｵﾌﾞｼﾞｪｸﾄ
	 */
	protected ResourceManager getResource() {
		if( resource == null ) {
			resource = (ResourceManager)pageContext.getAttribute( HybsSystem.LOCAL_RES_KEY );
			if( resource == null ) {
				resource = ResourceFactory.newInstance( getLanguage() );
			}
		}
		return resource;
	}

	/**
	 * ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうか[true:する/false:しない]を取得します｡
	 *
	 * setDebug( String )で登録します｡
	 * 初期値は、false です｡
	 *
	 * @og.rev 3.5.5.3 (2004/04/09) getDebug() から、ﾒｿｯﾄﾞ名変更
	 *
	 * @return	ﾃﾞﾊﾞｯｸﾞ出力 [true:する/false:しない]
	 */
	protected boolean isDebug() {
		return debugFlag ;
	}

	/**
	 * ﾗﾍﾞﾙ文字列を返します｡
	 *
	 * これは、{&#064;LBL.XXXX %Y} 引数処理をおこなうための、ｻﾎﾟｰﾄﾒｿｯﾄﾞです｡
	 * 引数は、"XXXX %Y" という状態で受け取ります｡(LBL. ははずした形)
	 * ﾗﾍﾞﾙには、通常のﾗﾍﾞﾙ(Label)以外に、Short,Tips,Description,RawShortLabel の情報を持っています｡
	 * {&#064;LBL.XXXX %Y} の Y に、L(ﾗﾍﾞﾙ長),S(ﾗﾍﾞﾙ短),R(生ﾗﾍﾞﾙ短),B(ｺｰﾄﾞ長),C(ｺｰﾄﾞ短),D(概要説明),T(ﾂｰﾙﾁｯﾌﾟ)
	 * を指定することで、それぞれの状態を取得することが可能になります｡
	 * Y を指定しない場合({&#064;LBL.XXXX}) は、'L' が指定された場合と同じ効果があります｡
	 * Y は、先頭文字１文字で判断していますので、{&#064;LBL.XXXX %Short}と記述できます｡
	 * Y 引数には、&#064;で始まるﾘｸｴｽﾄ引数が指定できます｡例えば、{&#064;LBL.XXXX &#064;ZZ}
	 * とすると、ZZ のﾘｸｴｽﾄ引数の値が Y に適用されます｡
	 * ﾗﾍﾞﾙｷｰそのものをﾊﾟﾗﾒｰﾀ化することが出来ます｡
	 * これは、{&#064;LBL.&#064;XXXX}という形式になります｡引数は、先の説明と同じです｡
	 * この場合は、XXXX をｷｰにﾘｸｴｽﾄ引数の値が、ﾗﾍﾞﾙﾘｿｰｽのｷｰになります｡
	 *
	 * @og.rev 4.0.0.0 (2007/10/17) ﾒｯｾｰｼﾞﾘｿｰｽ統合に伴い、{&#064;LBL.XXXX Y}⇒{&#064;LBL.XXXX %Y}
	 * @og.rev 5.4.0.1 (2011/11/01) ﾗﾍﾞﾙ形式('L','S','T','D') に、R(RawShortLabel) を追加
	 * @og.rev 5.5.7.2 (2012/10/09) ﾗﾍﾞﾙ形式('L','S','T','D','R') に、C(CodeData) を追加
	 * @og.rev 7.2.9.0 (2020/10/12) ﾗﾍﾞﾙ形式('L','S','T','D','R','C') 処理を、ﾒｿｯﾄﾞに変更(null時はｷｰを返す)
	 * @og.rev 7.2.9.0 (2020/10/12) getRawShortLabelで、null時は key を返します｡
	 * @og.rev 8.1.2.1 (2022/03/25) ﾗﾍﾞﾙ形式に B(ｺｰﾄﾞ長) 追加
	 *
	 * @param	lbl ﾗﾍﾞﾙのｷｰ
	 *
	 * @return	ﾗﾍﾞﾙ文字列
	 */
	protected String getLabel( final String lbl ) {

		String key = lbl ;
		String val = null;

		final int spc = lbl.indexOf( ' ' );		// " " があるかどうか
		if( spc > 0 ) {
			key = lbl.substring( 0,spc );
			if( key.charAt(0) == '@' ) { key = getRequestValue( key.substring( 1 ) ); }

			// リクエスト引数が指定された場合
			char ch  = lbl.length() > spc+1 ? lbl.toUpperCase( Locale.JAPAN ).charAt( spc+1 ) : ' ';	// Label,Short,Tips,Description
			char ch2 = lbl.length() > spc+2 ? lbl.toUpperCase( Locale.JAPAN ).charAt( spc+2 ) : ' ';	// Label,Short,Tips,Description
			if( ch == '@' ) {
				final String tmp = getRequestValue( lbl.substring( spc+2 ) );
				if( tmp != null && tmp.length() > 0 ) {
					ch  = tmp.toUpperCase( Locale.JAPAN ).charAt(0);
					ch2 = tmp.length() > 1 ? tmp.toUpperCase( Locale.JAPAN ).charAt( 1 ) : ' ';
				}
			}
			// 4.0.0.0 (2007/10/19)
			if( ch == '%' ) {
				switch( ch2 ) {
					case 'L': val = getResource().getLabel( key ); break;
//					case 'S': val = getResource().getLabelData( key ).getShortLabel(); break;
//					case 'T': val = getResource().getLabelData( key ).getLongLabel(); break;
//					case 'D': val = getResource().getLabelData( key ).getDescription(); break;
//					case 'R': val = getResource().getLabelData( key ).getRawShortLabel(); break;	// 5.4.0.1 (2011/11/01)
//					case 'C': val = getResource().getLabelData( key + "." + getRequestValue( key ) ).getShortLabel(); break;	// 5.5.7.2 (2012/10/09)
					// 7.2.9.0 (2020/10/12) ﾗﾍﾞﾙ形式('L','S','T','D','R','C') 処理を、ﾒｿｯﾄﾞに変更(null時はｷｰを返す)
					case 'S': val = getResource().getShortLabel(    key ); break;
//					case 'R': val = getResource().getRawShortLabel( key ); break;		// 5.4.0.1 (2011/11/01)
					case 'R': val = getResource().getRawShortLabel( key,true ); break;	// 7.2.9.0 (2020/10/12)
					case 'B': val = getResource().getLabel( key + "." + getRequestValue( key ) ); break;		// 8.1.2.1 (2022/03/25)
					case 'C': val = getResource().getShortLabel( key + "." + getRequestValue( key ) ); break;	// 5.5.7.2 (2012/10/09)
					case 'D': val = getResource().getDescription(   key ); break;
					case 'T': val = getResource().getLongLabel(     key ); break;
					default : break;
				}
			}
			else if( ch != ' ' ) {
				String[] msgVals = StringUtil.csv2Array( lbl.substring( spc+1 ),' ' );
				for( int i=0; i<msgVals.length; i++ ) {
					// ﾘｸｴｽﾄ文字ﾊﾟﾗﾒｰﾀ時の処理｡その他は、ﾗﾍﾞﾙ文字は処理不要｡
					if( StringUtil.startsChar( msgVals[i] , '@' ) ) {			// 6.4.1.1 (2016/01/16) １文字 String.startsWith
						msgVals[i] = getRequestValue( msgVals[i].substring( 1 ) );
					}
				}
				val = getResource().getLabel( key,msgVals );
			}
		}
		else {
			if( key.charAt(0) == '@' ) { key = getRequestValue( key.substring( 1 ) ); }
		}

		if( val == null ) { val = getResource().getLabel( key ); }
		return val;
	}

	/**
	 * DBColumn ｵﾌﾞｼﾞｪｸﾄを返します｡
	 *
	 * これは、ｷｰを元に DBColumnｵﾌﾞｼﾞｪｸﾄをｶﾗﾑﾘｿｰｽの
	 * 定義ﾌｧｲﾙより取得して、ﾘｿｰｽﾏﾈｰｼﾞｬで管理します｡
	 *
	 * @param	key	ｵﾌﾞｼﾞｪｸﾄのｷｰ
	 * @return	DBColumnｵﾌﾞｼﾞｪｸﾄ
	 */
	protected DBColumn getDBColumn( final String key ) {
		return getResource().makeDBColumn( key ) ;
	}

	/**
	 * 内部の Attributes ｵﾌﾞｼﾞｪｸﾄに、属性値をｾｯﾄします｡
	 *
	 * 同じｷｰの値が登録されていた場合は、置き換えられます｡
	 *
	 * @param	key		ｷｰ
	 * @param	value	属性値
	 * @see		#add( String , String )
	 */
	protected void set( final String key, final String value ) {
		attri.set( key,value );
	}

	/**
	 * 内部の Attributes ｵﾌﾞｼﾞｪｸﾄに、属性値を追加します｡
	 *
	 * ここでは、すでに同じｷｰが登録されている場合は、その値に、
	 * 標準ｾﾊﾟﾚｰﾀ(ｽﾍﾟｰｽ)を追加して、文字列結合します｡
	 * たとえば、class 属性などは、値をｽﾍﾟｰｽで追加する事で、
	 * CSS で処理することができます｡
	 *
	 * @og.rev 4.0.0.0 (2007/05/18) 新規追加
	 *
	 * @param	key		ｷｰ
	 * @param	value	属性値
	 * @see		#add( String , String , String )
	 * @see		#set( String , String )
	 */
	protected void add( final String key, final String value ) {
		attri.add( key,value );
	}

	/**
	 * 内部の Attributes ｵﾌﾞｼﾞｪｸﾄに、属性値を追加します｡
	 *
	 * ここでは、すでに同じｷｰが登録されている場合は、その値に、
	 * 引数のｾﾊﾟﾚｰﾀを追加して、文字列結合します｡
	 *
	 * @og.rev 3.5.0.0 (2003/09/17) 新規追加
	 * @og.rev 3.5.5.9 (2004/06/07) ｾﾊﾟﾚｰﾀ引数付きのﾒｿｯﾄﾞに変更
	 *
	 * @param	key		ｷｰ
	 * @param	value	属性値
	 * @param	sepa	ｾﾊﾟﾚｰﾀ
	 * @see		#add( String , String )
	 */
	protected void add( final String key, final String value, final String sepa ) {
		attri.add( key,value,sepa );
	}

	/**
	 * 内部の Attributes ｵﾌﾞｼﾞｪｸﾄから、属性値を取得します｡
	 *
	 * @param	key	ｷｰ
	 * @return	属性値
	 * @see		#set( String , String )
	 */
	protected String get( final String key ) {
		return attri.get( key );
	}

	/**
	 * 属性ｵﾌﾞｼﾞｪｸﾄの取得｡
	 *
	 * Attributes ｵﾌﾞｼﾞｪｸﾄを取得します｡
	 *
	 * @return	Attributesｵﾌﾞｼﾞｪｸﾄ
	 */
	protected Attributes getAttributes() {
		return attri;
	}

	/**
	 * {&#064;XXXX} 形式の文字列から XXXX をｷｰとして ServletRequest から getParameter で値を取り出します｡
	 *
	 * 他の文字列に混在した {&#064;XXXX} 文字を変換可能です｡
	 * ただし、処理の簡素化のため、上記形式以外は変換いたしません｡
	 * ｴﾗｰ例)× { &#064;XXXX }、{&#064; XXXX }、{&#064;XXXX&#064;yyyy}、{&#064;XXXX{&#064;yyyy}}
	 * また、"{&#064;" を通常の記述で使うことは無いと考え、ｴｽｹｰﾌﾟも用意して
	 * いません｡よって、"{&#064;" のﾊﾟﾀｰﾝが見つかって,"}" で閉じられていない
	 * 場合は,ｴﾗｰとして、HybsSystemException を throw します｡
	 *
	 * @og.rev 3.8.0.4 (2005/08/08) {} の処理方法見直し｡連続処理、単体処理を可能にします｡
	 *
	 * @param	key	ﾘｸｴｽﾄのｷｰ
	 * @return	ﾘｸｴｽﾄの値
	 * @og.rtnNotNull
	 */
	protected String getRequestParameter( final String key ) {
		isReqNull = false;

		if( key == null ) { isReqNull = true; return ""; }
		int index = key.indexOf( "{@" );
		if( index < 0 ) { return key; }

		// 変数が "{@XXXX}" の場合を優先的に検索。
		// これにより多くのﾊﾟﾀｰﾝで、StringTokenizer による
		// 文字列操作を行う必要がなくなります｡
		if( index == 0 &&
			key.indexOf( '}' ) == key.lastIndexOf( '}' ) &&						// 6.4.2.1 (2016/02/05) PMD refactoring. Useless parentheses.
			key.charAt(key.length()-1) == '}' ) {
				return getRequestValue( key.substring( 2,key.length()-1 ) );
		}

		// 3.8.0.4 (2005/08/08) {} の処理方法見直し。連続処理、単体処理を可能にします。
		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE );
		int start = 0;
		while( index >= 0 ) {
			final int end = key.indexOf( '}',index );
			if( end < 0 ) {
				final String errMsg = "{@ と } との対応関係がずれています。" + CR
							+ "key=[" + key + "] : index=" + index ;
				throw new HybsSystemException( errMsg );
			}

			// {@ より前方の文字列を追加
			if( index > 0 ) { rtn.append( key.substring( start,index ) ); }

			// {@XXXX} の XXXX部分を処理
			final String val = getRequestValue( key.substring( index+2,end ) );
			if( val != null ) { rtn.append( val ); }

			start = end+1 ;
			index = key.indexOf( "{@",start );
		}
		rtn.append( key.substring( start ) );

		return rtn.toString();
	}

	/**
	 * {&#064;XXX.YYYY} 形式の文字列から値を取得します｡
	 * 予約語のみ処理をし、それ以外は{&#064;xxx}のままとします｡
	 *
	 * 他の文字列に混在した {&#064;XXXX} 文字を変換可能です｡
	 * ただし、処理の簡素化のため、上記形式以外は変換いたしません｡
	 * ｴﾗｰ例）× { &#064;XXXX }、{&#064; XXXX }、{&#064;XXXX&#064;yyyy}、{&#064;XXXX{&#064;yyyy}}
	 * また、"{&#064;" を通常の記述で使うことは無いと考え、ｴｽｹｰﾌﾟも用意して
	 * いません｡よって、"{&#064;" のﾊﾟﾀｰﾝが見つかって,"}" で閉じられていない
	 * 場合は,ｴﾗｰとして、HybsSystemException を throw します｡
	 *
	 * @og.rev 5.5.4.0 (2012/07/02) 新規作成
	 *
	 * @param	key	ﾘｸｴｽﾄのｷｰ
	 * @return	ﾘｸｴｽﾄの値
	 * @og.rtnNotNull
	 */
	protected String getReservedParameter( final String key ) {
		isReqNull = false;

		if( key == null ) { isReqNull = true; return ""; }
		int index = key.indexOf( "{@" );
		if( index < 0 ) { return key; }

		// 変数が "{@XXXX}" の場合を優先的に検索｡
		// これにより多くのﾊﾟﾀｰﾝで、StringTokenizer による
		// 文字列操作を行う必要がなくなります｡
		if( index == 0 &&
			key.indexOf( '}' ) == key.lastIndexOf( '}' ) &&						// 6.4.2.1 (2016/02/05) PMD refactoring. Useless parentheses.
			key.charAt(key.length()-1) == '}' ) {								// 6.4.2.1 (2016/02/05) PMD refactoring. Useless parentheses.
				return getReservedValue( key.substring( 2,key.length()-1 ) );
		}

		// 3.8.0.4 (2005/08/08) {} の処理方法見直し。連続処理、単体処理を可能にします。
		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE );
		int start = 0;
		while( index >= 0 ) {
			final int end = key.indexOf( '}',index );
			if( end < 0 ) {
				final String errMsg = "{@ と } との対応関係がずれています。" + CR
							+ "key=[" + key + "] : index=" + index ;
				throw new HybsSystemException( errMsg );
			}

			// {@ より前方の文字列を追加
			if( index > 0 ) { rtn.append( key.substring( start,index ) ); }

			// {@XXXX} の XXXX部分を処理
			final String val = getReservedValue( key.substring( index+2,end ) );
			if( val != null ) { rtn.append( val ); }

			start = end+1 ;
			index = key.indexOf( "{@",start );
		}
		rtn.append( key.substring( start ) );

		return rtn.toString();
	}

	/**
	 * {&#064;XXXX} 形式の文字列から XXXX をｷｰとして ServletRequest から getParameterValues で値を取り出します｡
	 *
	 * これは、複数(配列)でﾘｸｴｽﾄを取り出すことが可能です｡
	 * そのため、他の文字列に混在させて変換することができません｡
	 * "{&#064;XXXX}" 形式 からのみの変換となります｡
	 *
	 * @og.rev 3.6.0.0 (2004/09/22) ｷｰがnull のときにnullではなく長さ０の配列を返します｡
	 *
	 * @param	key	ﾘｸｴｽﾄのｷｰ
	 * @return	ﾘｸｴｽﾄの値
	 * @og.rtnNotNull
	 */
	protected String[] getRequestParameterValues( final String key ) {
		if( key == null ) { return new String[0]; }		// 3.6.0.0 (2004/09/22)
		final int index = key.indexOf( "{@" );
		if( index < 0 ) { return StringUtil.csv2Array( key ); }

		if( index == 0 && key.charAt( key.length()-1 ) == '}' ) {
			return getRequestValues( key.substring( 2,key.length()-1 ) );
		}

		final String errMsg = "引数の形式が異なります。 [" + key + "]" ;
		throw new HybsSystemException( errMsg );
	}

	/**
	 * 引数 inStr が、引数 check のSetの中に存在すれば、 true を、存在しなければ、false を返します｡
	 *
	 * check は、 String配列 を、inStr は、null でも構いません｡
	 * ※ 6.3.5.0 (2015/08/08) 大文字小文字の区別廃止｡
	 *
	 * @og.rev 2.1.0.3 (2002/11/08) 文字列配列を引数に取るﾒｿｯﾄﾞを追加
	 * @og.rev 6.3.5.0 (2015/08/08) forﾙｰﾌﾟの変更と、大文字小文字の区別廃止
	 * @og.rev 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます｡
	 *
	 * @param	inStr	ﾁｪｯｸする文字列
	 * @param	chkSet	ﾁｪｯｸ用の文字列Set
	 * @return	存在する true ／ 存在しない false
	 */
	protected boolean check( final String inStr, final Set<String> chkSet ) {
		return inStr != null && chkSet != null && chkSet.contains( inStr );
	}

	/**
	 * ﾕｰｻﾞｰｵﾌﾞｼﾞｪｸﾄが持っている内部情報を取得します｡
	 *
	 * これは、UserInfo#getAttribute( String ) で取得される情報です｡
	 * ﾕｰｻﾞｰﾊﾟﾗﾒｰﾀとは異なります｡
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 getUser() を使用するように変更
	 *
	 * @param	user	ﾕｰｻﾞｰ情報を取り出すｷｰ
	 *
	 * @return	ﾕｰｻﾞｰ情報文字列
	 */
	protected String getUserInfo( final String user ) {
		if( user == null ) { return null; }

		String key = user ;
		if( key.charAt(0) == '@' ) {
			key = getRequestValue( key.substring( 1 ) );
		}

		return getUser().getAttribute( key );
	}

	/**
	 * ﾕｰｻﾞｰｵﾌﾞｼﾞｪｸﾄが持っているEditの内部情報を取得します｡
	 *
	 * これは、UserInfo#getSelectedEdit( String ) で取得される情報です｡
	 * ﾕｰｻﾞｰﾊﾟﾗﾒｰﾀとは異なります｡
	 *
	 * @og.rev 5.8.2.3 (2014/12/27) 新規作成
	 *
	 * @param	guikey 取り出す画面ID
	 *
	 * @return	ﾕｰｻﾞｰ情報文字列
	 */
	protected String getUserEditInfo( final String guikey ) {
		if( guikey == null ) { return null; }

		String key = guikey ;
		if( key.charAt(0) == '@' ) {
			key = getRequestValue( key.substring( 1 ) );
		}

	// * @og.rev 6.4.5.0 (2016/04/08) UserInfo のEditConfig関連機能を、DBEditConfigManagerに移植します。
	//	final DBEditConfigManager dbConfMgr = getUser().getEditConfigManager();		// 6.4.5.0 (2016/04/08)
	//	return dbConfMgr.getSelectedEdit( key );
		return getUser().getSelectedEdit( key );
	}

	/**
	 * ﾕｰｻﾞｰ情報を設定します｡
	 *
	 * 初めての場合は、session 登録項目 の HybsSystem#USERINFO_KEY ｷｰ の値で
	 * 取得します｡
	 * save属性は、GE20(ﾕｰｻﾞｰ定数)に情報を保存するかどうかを指定します｡
	 *
	 * @og.rev 2.1.1.4 (2002/11/25) ﾕｰｻﾞｰ情報をｾｯﾄするﾒｿｯﾄﾞを追加
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 getUser() を使用するように変更
	 * @og.rev 4.3.4.0 (2008/12/01) GE20(ﾕｰｻﾞｰ定数)へ登録するかのﾌﾗｸﾞを追加
	 *
	 * @param	key		ﾕｰｻﾞｰ情報をｾｯﾄするｷｰ
	 * @param	value	ﾕｰｻﾞｰ情報文字列
	 * @param	save	情報保存 [true:保存/false:保存しない]
	 */
	protected void setUserInfo( final String key,final String value, final boolean save ) {
		if( key != null ) {
			getUser().setAttribute( key, value, save );
		}
	}

	/**
	 * ﾕｰｻﾞｰ情報ｵﾌﾞｼﾞｪｸﾄを取得します｡
	 *
	 * 初めての場合は、session 登録項目 の HybsSystem#USERINFO_KEY ｷｰ の値で
	 * 取得します｡
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 新規追加
	 * @og.rev 3.6.0.0 (2004/09/17) private ⇒ protected 化します｡
	 *
	 * @return	ﾕｰｻﾞｰ情報ｵﾌﾞｼﾞｪｸﾄ
	 * @og.rtnNotNull
	 */
	protected UserInfo getUser() {
		if( userInfo == null ) {
			userInfo = (UserInfo)getSessionAttribute( HybsSystem.USERINFO_KEY );
		}
		if( userInfo == null ) {
			final String errMsg = "ﾕｰｻﾞｰｵﾌﾞｼﾞｪｸﾄが存在しません。";
			throw new HybsSystemException( errMsg );
		}
		return userInfo ;
	}

	/**
	 * 画面情報(GUIInfo)を取得します｡
	 *
	 * これは、session 登録項目 の HybsSystem#GUIMAP_KEY ｷｰ の値で
	 * 登録された MAP を取り出し、そこから取得します｡
	 * 画面情報は、ﾛｸﾞｲﾝﾕｰｻﾞｰ毎に個別に持っています｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) GUIInfo が存在しない場合も処理を続けます｡
	 *
	 * @param	gamenId	画面ID
	 * @return	画面情報(GUIInfo)
	 */
	protected GUIInfo getGUIInfo( final String gamenId ) {
		return getUser().getGUIInfo( gamenId );
	}

	/**
	 * 画面情報(GUIInfo)の属性値を取得します｡
	 *
	 * これは、{&#064;GUI.XXXX ID} 引数処理をおこなうための、ｻﾎﾟｰﾄﾒｿｯﾄﾞです｡
	 * 引数は、"XXXX ID" という状態で受け取ります｡(GUI. ははずした形)
	 * XXXXには、画面情報(GUIInfo)の属性ｷｰを指定します｡IDが指定されない場合は、
	 * 実行中の自分自身の画面が指定されたことになります｡
	 * これは、session 登録項目 の HybsSystem#GUIINFO_KEY ｷｰ の値で取得します｡
	 * この値は,画面が呼び出される毎に毎回設定されており、ﾘｸｴｽﾄ毎に
	 * 所得し直す必要があります｡
	 *
	 * ID に、画面IDを指定した場合は、&#064; 指定によるﾘｸｴｽﾄ引数の値を適用できます｡
	 * {&#064;GUI.&#064;XXXX ID} や、{&#064;GUI.XXXX &#064;ID} です｡(同時指定も可能)
	 *
	 * @og.rev 3.6.0.6 (2004/10/22) GUIInfo が存在しない場合も処理を続けます｡
	 * @og.rev 4.0.0.0 (2004/11/30) 画面ID引数や、ﾘｸｴｽﾄ引数の使用を可能にします｡
	 * @og.rev 5.9.32.2 (2018/05/18) ｴﾗｰ表示変更
	 * @og.rev 5.10.12.3 (2019/06/21) ｴﾗｰ表示変更
	 *
	 * @param	attkey	画面情報を取り出すｷｰ
	 * @return	画面情報文字列
	 */
	protected String getGUIInfoAttri( final String attkey ) {
		if( attkey == null ) { return null; }

		String  key = attkey ;
		final GUIInfo gui ;

		final int spc = key.indexOf( ' ' );										// " " があるかどうか
		if( spc > 0 ) {
			key = attkey.substring( 0,spc );
			String id = attkey.substring( spc+1 );
			if( StringUtil.startsChar( id , '@' ) ) {							// 6.2.0.0 (2015/02/27) １文字 String.startsWith
				id = getRequestValue( id.substring( 1 ) );
			}
			gui = getUser().getGUIInfo( id );
		}
		else {
			if( guiInfo == null ) {
				guiInfo = (GUIInfo)getSessionAttribute( HybsSystem.GUIINFO_KEY );
			}
			gui = guiInfo;
		}
//		if( gui == null ) { return "Not Found[" + attkey + "]"; }
//		if( gui == null ) { return "Not Found(" + attkey + ")"; }				// 5.9.32.2 (2018/05/18) []で囲むとﾃｰﾌﾞﾙﾓﾃﾞﾙで予期せぬｴﾗｰになるため変更
		if( gui == null ) { return "NotFound_" + attkey ; }						// 5.10.12.3 (2019/06/21) URLで確実に使える文字のみで構成しておく
		if( StringUtil.startsChar( key , '@' ) ) {								// 6.2.0.0 (2015/02/27) １文字 String.startsWith
			key = getRequestValue( key.substring( 1 ) );
		}

		return gui.getAttribute( key );
	}

	/**
	 * {&#064;NVL.XXX 値} で、XXXが NULLの場合、値を返します｡
	 *
	 * ORACLE等の COALESCE ( expr_list ) 処理に近い結果を返します｡
	 * NVL ( expr1 , expr2 ) は、expr1 が NULL の場合、expr2 を返しますが、
	 * COALESCE は、第一引数が NULLなら、次の引数、それも NULL なら、さらに次と
	 * 引数ﾘｽﾄを順次処理していきます｡
	 * それと同じく、XXX が NULL なら、その次･･と順次評価していき、NULL でない
	 * 値が返ってきたところで、その値を返します｡
	 * これは、{&#064;NVL.XXX &#064;YYY ZZZ ･･･} 形式を処理します｡
	 * これは、ﾊﾟﾗﾒｰﾀ XXX が NULLの場合、次の値を使います｡(&#064;YYY)
	 * &#064;YYY は、YYY ﾊﾟﾗﾒｰﾀの事で、これも NULL の場合は、ZZZ を使います｡
	 * 最後まで NULL の場合は、 ｾﾞﾛ文字列が返されます｡
	 *
	 * @og.rev 5.6.4.0 (2013/05/02) 新規追加
	 * @og.rev 6.9.3.0 (2018/03/26) ﾊﾟﾗﾒｰﾀ処理で、ﾀﾞﾌﾞﾙｸｵｰﾄ内は分解しない
	 * @og.rev 6.9.3.1 (2018/04/02) ﾊﾟﾗﾒｰﾀ処理で、ﾀﾞﾌﾞﾙｸｵｰﾄ内は分解しない･･の処理漏れ追加
	 * @og.rev 7.2.9.1 (2020/10/23) protected → private に変更
	 *
	 * @param	attkey	NVL情報を取り出すﾊﾟﾗﾒｰﾀ
	 *
	 * @return	NULL以外の値が出てくるまで、順次評価した結果
	 */
//	protected String getNVLAttri( final String attkey ) {
	private String getNVLAttri( final String attkey ) {
		if( attkey == null ) { return null; }

		final String[] keys = StringUtil.csv2Array( attkey,' ' );				// ﾀﾞﾌﾞﾙｸｵｰﾄ内は分解しない｡
//		final String[] keys = attkey.split( " " );								// ｽﾍﾟｰｽで、ﾊﾟﾗﾒｰﾀを分解する｡
		String val = getRequestValue( keys[0] );								// 第一ﾊﾟﾗﾒｰﾀは、そのままのｷｰで検索

		// val が null の間は、ﾁｪｯｸする｡
		if( val == null || val.isEmpty() ) {
			for( int i=1; i<keys.length; i++ ) {
				val = keys[i];
				// 先頭が @ の場合は、ﾘｸｴｽﾄ変数のｷｰとして、値を判定
				if( StringUtil.startsChar( val , '@' ) ) {						// 6.2.0.0 (2015/02/27) １文字 String.startsWith
					if( val.indexOf( ' ' ) > 0 ) {								// 6.9.3.1 (2018/04/02) 空白を含む場合は、ﾀﾞﾌﾞﾙｸｵｰﾄ内の処理
						val = getReservedValue( val.substring( 1 ) );			// 6.9.3.0 (2018/03/26) もう一度変数処理
					}
					else {
						val = getRequestValue( val.substring( 1 ) );			// 6.9.3.1 (2018/04/02) 一旦元に戻す｡
					}
//					val = getRequestValue( val.substring( 1 ) );
//					val = getReservedValue( val.substring( 1 ) );				// 6.9.3.0 (2018/03/26) もう一度変数処理
				}
				if( val != null && val.length() > 0 ) { break; }
			}
		}

		if( val == null ) { val = ""; }											// NULL の場合は、ｾﾞﾛ文字列を返す。

		return val;
	}

	/**
	 * {&#064;REP.XXX 置換元 置換先} で、XXX の文字列変換した値を返します｡
	 *
	 * XXX に対して、置換元文字列を置換先文字列で置換します｡
	 * これは、{&#064;REP.XXX &#064;YYY ZZZ} 形式を処理します｡
	 * これは、ﾊﾟﾗﾒｰﾀ XXX が NULLの場合、NULLを返します｡
	 * &#064;YYY は、YYY ﾊﾟﾗﾒｰﾀの事で、これが NULL の場合は、XXX を返します｡
	 *
	 * ※ ｸｫｰﾃｨｼｮﾝ(')を含む処理と、元々が IN 引数に、'AA,BB,CC' などの文字列処理を
	 *    行う( 'AA,'BB,'CC' ) のが目的だったので、引数なしの場合は、上記変換を行う｡
	 *
	 * @og.rev 8.0.0.0 (2021/09/30) REP.XXXX を追加します｡
	 *
	 * @param	attkey	文字列置換ﾊﾟﾗﾒｰﾀ
	 * @return	XXX 値の第一引数と第二引数を置換した値
	 */
	private String getREPAttri( final String attkey ) {
		if( attkey == null ) { return null; }

		final String[] keys = StringUtil.csv2Array( attkey,' ' );				// ﾀﾞﾌﾞﾙｸｵｰﾄ内は分解しない｡
		String val = getRequestValue( keys[0] );								// 第一ﾊﾟﾗﾒｰﾀは、そのままのｷｰで検索

		if( val == null ) { val = ""; }											// NULL の場合は、ｾﾞﾛ文字列を返す｡
		if( keys.length < 3 ) {													// 引数が少ない場合は、"," を"','" に変換する。
			return val.replace( ",","','" );
		}

		// 置換元 文字列の取得
		String key1 = keys[1];
		// 先頭が @ の場合は、ﾘｸｴｽﾄ変数のｷｰとして、値を判定
		if( StringUtil.startsChar( key1 , '@' ) ) {
			key1 = getRequestValue( key1.substring( 1 ) );						// @ を削除して、値の取得
		}
		if( key1 == null || key1.isEmpty() ) {
			return val;															// 置換元が null か空文字列の場合は、置換しない
		}

		// 置換先 文字列の取得
		String key2 = keys[2];
		// 先頭が @ の場合は、ﾘｸｴｽﾄ変数のｷｰとして、値を判定
		if( StringUtil.startsChar( key2 , '@' ) ) {
			key2 = getRequestValue( key2.substring( 1 ) );						// @ を削除して、値の取得
		}
		if( key2 == null )  {
			key2 = "";															// 置換先が null か空文字列の場合は、削除になる。
		}

		return val.replace( key1,key2 );
	}

	/**
	 * {&#064;NN.XXX 値} で、XXXが NOT_NULLの場合、値を返します｡
	 *
	 * 値が存在した場合に、別の値に変更する動作になります｡
	 * 引数が、NULLの場合は、NULLなので、例えばある値が設定されれば、
	 * "true" を返すとか、"checked" を返すなどの処理が考えられます｡
	 * 値をｾｯﾄしなかった場合は、処理を行いません｡
	 *
	 * @og.rev 7.2.9.1 (2020/10/23) 新規追加
	 *
	 * @param	attkey	NOT_NULL情報を取り出すﾊﾟﾗﾒｰﾀ
	 * @return	NOT_NULLの場合、値を返します｡
	 */
	private String getNNAttri( final String attkey ) {
		if( attkey == null ) { return null; }									// 継続処理をさせるため、nullを返しておきます｡

		String val = "" ;														// 不正なﾊﾟﾗﾒｰﾀやｷｰが NULLの場合の初期値

		final int spc = attkey.indexOf( ' ' );									// " " があるかどうか
		if( spc > 0 ) {															// 値ﾊﾟﾗﾒｰﾀが存在するときのみ処理を行う｡
			String key = attkey.substring( 0,spc );
			if( StringUtil.startsChar( key , '@' ) ) {
				key = getRequestValue( key.substring( 1 ) );
			}
			key = getRequestValue( key );

			if( key != null && !key.isEmpty() ) {								// key が NOT NULL の場合のみ、処理を行う。
				// 値を求める。
				val = attkey.substring( spc+1 );
				if( StringUtil.startsChar( id , '@' ) ) {						// 6.2.0.0 (2015/02/27) １文字 String.startsWith
					val = getRequestValue( id.substring( 1 ) );
				}
			}
		}

		return val;
	}

	/**
	 * {&#064;VALS.XXX ｱﾄﾞﾚｽ ｾﾊﾟﾚｰﾀ} で、XXXをｾﾊﾟﾚｰﾀで分解して、ｱﾄﾞﾚｽの値を返します｡
	 *
	 * ｾﾊﾟﾚｰﾀ の初期値は、,(ｶﾝﾏ)です｡
	 * XXX は、変数のｷｰﾜｰﾄﾞになります｡
	 *
	 * @og.rev 7.2.3.1 (2020/04/17) 新規追加
	 * @og.rev 7.2.9.1 (2020/10/23) protected → private に変更
	 *
	 * @param	attkey	VALS情報を取り出すﾊﾟﾗﾒｰﾀ
	 *
	 * @return	指定の変数をｾﾊﾟﾚｰﾀで分割してｱﾄﾞﾚｽ番目の文字列を取り出す｡
	 */
//	protected String getValsAttri( final String attkey ) {
	private String getValsAttri( final String attkey ) {
		if( attkey == null ) { return null; }

		final String[] keys = StringUtil.csv2Array( attkey,' ' );				// ﾀﾞﾌﾞﾙｸｵｰﾄ内は分解しない｡
		String val = getRequestValue( keys[0] );								// 第一ﾊﾟﾗﾒｰﾀは、そのままのｷｰで検索

		if( val == null || val.isEmpty() ) {
			val = "";															// NULL の場合は、ｾﾞﾛ文字列を返す｡
		}
		else {
			final int      adrs = keys.length > 1 ? Integer.parseInt( keys[1] ) : 0 ;	// ｱﾄﾞﾚｽを取り出す
			final String   sep  = keys.length > 2 ? keys[2] : "," ;				// 区切り文字列
			final String[] vals = val.split( sep );

			val = vals[adrs];
		}

		return val;
	}

	/**
	 * {&#064;SUBSTR.XXX 開始ｲﾝﾃﾞｯｸｽ 終了ｲﾝﾃﾞｯｸｽ} で、XXXの部分文字列を取り出します｡
	 *
	 * 終了ｲﾝﾃﾞｯｸｽを指定しない場合は、最後までです｡
	 * 開始ｲﾝﾃﾞｯｸｽ(この値を含む)から、終了ｲﾝﾃﾞｯｸｽ(この値を含まない)を切り出します｡
	 * ｲﾝﾃﾞｯｸｽは、文字数です｡
	 *
	 * @og.rev 7.2.3.1 (2020/04/17) 新規追加
	 * @og.rev 7.2.9.1 (2020/10/23) protected → private に変更
	 *
	 * @param	attkey	SUBSTR情報を取り出すﾊﾟﾗﾒｰﾀ
	 * @return	XXXの部分文字列
	 */
//	protected String getSubstrAttri( final String attkey ) {
	private String getSubstrAttri( final String attkey ) {
		if( attkey == null ) { return null; }

		final String[] keys = StringUtil.csv2Array( attkey,' ' );				// ﾀﾞﾌﾞﾙｸｵｰﾄ内は分解しない｡

		final String val = getAtmarkVas( keys[0],true );							// ｷｰﾜｰﾄﾞをﾘｸｴｽﾄから処理する
		final String ad1 = keys.length > 1 ? getAtmarkVas( keys[1],false ) : "" ;	// ｷｰﾜｰﾄﾞは、@が無ければそのままの値
		final String ad2 = keys.length > 2 ? getAtmarkVas( keys[2],false ) : "" ;	// ｷｰﾜｰﾄﾞは、@が無ければそのままの値

		final int id1 = ad1.isEmpty() ? 0				: Integer.parseInt( ad1 );
		final int id2 = ad2.isEmpty() ? val.length()	: Integer.parseInt( ad2 );

		if( id1 < id2 && val.length() > id1 && val.length() >= id2 ) {
			return val.substring( id1,id2 );
		}

		return val;
	}

	/**
	 * {&#064;LAST.XXX} で、XXXが 最後に使われた値を返します｡
	 *
	 * XXX は、command="NEW" でｾｯﾄされたﾘｸｴｽﾄ値です｡通常、{&#064;MEM.XXX} は
	 * 画面単位に、既存のｷｬｯｼｭから値を取り出しますが、{&#064;LAST.XXX} は、
	 * 画面に関係なく、ﾕｰｻﾞｰ単位に管理しています｡
	 * また、値は、ﾃﾞｰﾀﾍﾞｰｽ(GE20)に保管されますので、次回ﾛｸﾞｲﾝ時にも有効です｡
	 * この処理が呼び出されたときに、ﾘｸｴｽﾄ変数に、XXXX が存在した場合は、そちらを優先に
	 * 使います｡その場合は、command に関係なく、値を設定しておきます｡
	 *
	 * command="NEW"の場合のﾘｸｴｽﾄ変数の値の設定は、RequestCacheTag で行います｡
	 *
	 * ※ ﾃﾞｰﾀﾍﾞｰｽには、画面ｱｸｾｽ情報のｾｰﾌﾞ時に行われます｡
	 * valueﾀｸﾞのsave属性の様に、ﾘｱﾙﾀｲﾑではありませんので、Tomcatが
	 * 異常終了した場合は、ｾｰﾌﾞされません｡
	 *
	 * @og.rev 5.6.8.1 (2013/09/13) 新規追加
	 * @og.rev 7.2.9.1 (2020/10/23) protected → private に変更
	 *
	 * @param	attkey	最後に使われた値をを取り出すﾊﾟﾗﾒｰﾀ
	 * @return	XXXが 最後に使われた値
	 * @see		org.opengion.hayabusa.taglib.RequestCacheTag#commandExec( String ,ServletRequest )
	 * @see		org.opengion.hayabusa.resource.UserInfo#getLastRequestValue( String )
	 */
//	protected String getLASTAttri( final String attkey ) {
	private String getLASTAttri( final String attkey ) {
		if( attkey == null ) { return null; }

		// 最新のﾘｸｴｽﾄ変数をﾁｪｯｸ
		final String[] vals = getRequest().getParameterValues( attkey );

		String val = null;
		if( vals == null ) {
			val = getUser().getLastRequestValue( attkey );						// なければ、取得
		}
		else {
			for( int i=0; i<vals.length; i++ ) {
				val = vals[i];
				if( ! "0".equals( val ) ) { break; }							// ﾁｪｯｸﾎﾞｯｸｽ対応
			}
			getUser().setLastRequestValue( attkey,val );						// あれば、最新の値をｾｯﾄ
		}

		return val ;
	}

	/**
	 * {&#064;SUMR.XXX} で、XXXの 複数ﾘｸｴｽﾄ時の値を合算します｡
	 *
	 * 同一ｷｰのﾘｸｴｽﾄ変数に、複数のﾊﾟﾗﾒｰﾀが
	 * 設定された場合、その値を合計します｡
	 * 数値変換できない場合は、ｶﾝﾏで連結します｡
	 * 通常は、edior="BITBOX" などで、数値の合計を求めるために使われます｡
	 *
	 * @og.rev 6.2.2.4 (2015/04/24) SUM追加
	 * @og.rev 6.2.3.0 (2015/05/01) CSV形式の作成を、String#join( CharSequence , CharSequence... )を使用｡
	 * @og.rev 7.2.0.0 (2020/02/14) SUMはValueTagとﾀﾞﾌﾞるため、SUMR とします｡
	 * @og.rev 7.2.9.1 (2020/10/23) protected → private に変更
	 *
	 * @param	attkey	最後に使われた値をを取り出すﾊﾟﾗﾒｰﾀ
	 * @return	XXXの 複数ﾘｸｴｽﾄ時の値を合算
	 */
//	protected String getSumRequestValue( final String attkey ) {
	private String getSumRequestValue( final String attkey ) {
		if( attkey == null ) { return null; }

		// 最新のﾘｸｴｽﾄ変数をﾁｪｯｸ
		final String[] vals = getRequestValues( attkey );

		String rtn = "";
		if( vals != null && vals.length > 0 ) {
			try {
				int sum = 0;
				for( int i=0; i<vals.length; i++ ) {
					final String val = vals[i];
					if( val != null && !val.isEmpty() ) {
						sum += Integer.parseInt( val );
					}
				}
				rtn = String.valueOf( sum );									// 最後までｴﾗｰがなかった場合｡
			}
			catch( final NumberFormatException ex ) {
				// 数値変換ｴﾗｰ時は、文字列連結します｡
				// 6.2.3.0 (2015/05/01) CSV形式の作成を、String#join( CharSequence , CharSequence... )を使用｡
				rtn = String.join( "," , vals ) ;
			}
		}

		return rtn ;
	}

	/**
	 * {&#064;REQ.XXX} で、XXXの ﾘｸｴｽﾄｵﾌﾞｼﾞｪｸﾄのﾒｿｯﾄﾞの値を取得します｡
	 *
	 * HttpServletRequest のﾒｿｯﾄﾞを実行します｡
	 * それ以外に、ROWCOUNT というｷｰﾜｰﾄﾞで、選択された行数を返します｡
	 *
	 * 通常のﾘｸｴｽﾄの値以外にRestAPIで利用される場合のあるX-HTTP-Method-Overrideと、
	 * POSTﾃﾞｰﾀのBodyの値（JSONを変数名を指定せずに送信する場合がある）についても
	 * 取得できるようにしておきます｡
	 *
	 * <table class="plain">
	 *   <caption>{&#064;REQ.XXX}の説明</caption>
	 *   <tr><th>KEY</th><th>VALUE</th></tr>
	 *   <tr><td>ROWCOUNT           </td><td style="white-space: normal">ﾁｪｯｸされた件数</td></tr>
	 *   <tr><td>RequestURL         </td><td style="white-space: normal">request.getRequestURL()</td></tr>
	 *   <tr><td>AuthType           </td><td style="white-space: normal">request.getAuthType()</td></tr>
	 *   <tr><td>ContextPath        </td><td style="white-space: normal">request.getContextPath()</td></tr>
	 *   <tr><td>Method             </td><td style="white-space: normal">request.getMethod()</td></tr>
	 *   <tr><td>PathInfo           </td><td style="white-space: normal">request.getPathInfo()</td></tr>
	 *   <tr><td>PathTranslated     </td><td style="white-space: normal">request.getPathTranslated()</td></tr>
	 *   <tr><td>QueryString        </td><td style="white-space: normal">request.getQueryString()</td></tr>
	 *   <tr><td>RemoteUser         </td><td style="white-space: normal">request.getRemoteUser()</td></tr>
	 *   <tr><td>RequestURI         </td><td style="white-space: normal">request.getRequestURI()</td></tr>
	 *   <tr><td>ServletPath        </td><td style="white-space: normal">request.getServletPath()</td></tr>
	 *   <tr><td>RemoteAddr         </td><td style="white-space: normal">request.getRemoteAddr()</td></tr>
	 *   <tr><td>RemoteHost         </td><td style="white-space: normal">request.getRemoteHost()</td></tr>
	 *   <tr><td>Scheme             </td><td style="white-space: normal">request.getScheme()</td></tr>
	 *   <tr><td>ServerName         </td><td style="white-space: normal">request.getServerName()</td></tr>
	 *   <tr><td>ServerPort         </td><td style="white-space: normal">request.getServerPort()</td></tr>
	 *   <tr><td>MethodOverride     </td><td style="white-space: normal">X-HTTP-Method-Override</td></tr>
	 *   <tr><td>PostData           </td><td style="white-space: normal">request.getInputStream()</td></tr>
	 *   <tr><td>JSPID              </td><td style="white-space: normal">JSPﾌｧｲﾙ名</td></tr>
	 * </table>
	 *
	 * @og.rev 6.4.7.0 (2016/06/03) REQ追加
	 * @og.rev 5.10.10.0 (2019/03/29) MethodOverride追加
	 * @og.rev 7.0.4.1 (2019/06/10) {&#064;REQ.JSPID}追加
	 * @og.rev 5.10.10.1 (2019/04/05) BodyData追加
	 *
	 * @param	attkey	最後に使われた値を取り出すﾊﾟﾗﾒｰﾀ
	 * @return	XXXに対応したﾘｸｴｽﾄﾒｿｯﾄﾞの実行結果
	 */
	protected String getRequestMethod( final String attkey ) {
		if( attkey == null ) { return null; }

		final HttpServletRequest req = (HttpServletRequest)pageContext.getRequest();

		String rtn = "";

		if(      "ROWCOUNT"			.equalsIgnoreCase( attkey ) ) {
			final String[] vals = req.getParameterValues( HybsSystem.ROW_SEL_KEY );
			rtn = vals == null ? "0" : String.valueOf( vals.length );
		}
		else if( "RequestURL"		.equalsIgnoreCase( attkey ) ) { rtn = req.getRequestURL().toString();			}
		else if( "AuthType"			.equalsIgnoreCase( attkey ) ) { rtn = req.getAuthType();						}
		else if( "ContextPath"		.equalsIgnoreCase( attkey ) ) { rtn = req.getContextPath();						}
		else if( "Method"			.equalsIgnoreCase( attkey ) ) { rtn = req.getMethod();							}
		else if( "PathInfo"			.equalsIgnoreCase( attkey ) ) { rtn = req.getPathInfo();						}
		else if( "PathTranslated"	.equalsIgnoreCase( attkey ) ) { rtn = req.getPathTranslated();					}
		else if( "QueryString"		.equalsIgnoreCase( attkey ) ) { rtn = req.getQueryString();						}
		else if( "RemoteUser"		.equalsIgnoreCase( attkey ) ) { rtn = req.getRemoteUser();						}
		else if( "RequestURI"		.equalsIgnoreCase( attkey ) ) { rtn = req.getRequestURI();						}
		else if( "ServletPath"		.equalsIgnoreCase( attkey ) ) { rtn = req.getServletPath();						}
		else if( "RemoteAddr"		.equalsIgnoreCase( attkey ) ) { rtn = req.getRemoteAddr();						}
		else if( "RemoteHost"		.equalsIgnoreCase( attkey ) ) { rtn = req.getRemoteHost();						}
		else if( "Scheme"			.equalsIgnoreCase( attkey ) ) { rtn = req.getScheme();							}
		else if( "ServerName"		.equalsIgnoreCase( attkey ) ) { rtn = req.getServerName();						}
		else if( "ServerPort"		.equalsIgnoreCase( attkey ) ) { rtn = String.valueOf( req.getServerPort() );	}
		else if( "MethodOverride"	.equalsIgnoreCase( attkey ) ) { rtn = String.valueOf( req.getHeader("X-HTTP-Method-Override") ); }	//5.10.10.0 (2019/03/29)
		else if( "ContentType"		.equalsIgnoreCase( attkey ) ) { rtn = String.valueOf( req.getHeader("Content-Type") ); }			//5.10.10.1 (2019/04/05)
		else if( "PostData"			.equalsIgnoreCase( attkey ) ) { 																	//5.10.10.1 (2019/04/05)
			try( Scanner sc = new Scanner(req.getInputStream(), "UTF-8").useDelimiter("\\A"); ){
				rtn = sc.hasNext() ? sc.next() : "";
			}catch( final IOException ie ) {
				System.out.println(ie);
				rtn="";
			}
		}
		// 7.0.4.1 (2019/06/10) {&#064;REQ.JSPID}追加
		else if( "JSPID"			.equalsIgnoreCase( attkey ) ) {
			rtn = req.getRequestURI();
			final int ad = rtn.lastIndexOf( '/' );
			if( ad >= 0 ) {
				rtn = rtn.substring( ad+1 );
			}
		}
		// ﾕｰｻﾞｰｴｰｼﾞｪﾝﾄを取得します｡8.0.0.1 (2021/10/08)
		else if( "userAgent"	.equalsIgnoreCase( attkey ) ) { rtn = String.valueOf( req.getHeader("user-agent") ); }

		if( rtn == null ) { rtn = ""; }

		return rtn ;
	}

	/**
	 * 予約語に関する情報の文字列を取得します｡
	 *
	 * @og.rev 5.5.4.0 (2012/07/02) 予約語部分のみ分離
	 * @og.rev 5.6.4.0 (2013/05/02) NVL 追加
	 * @og.rev 5.6.8.1 (2013/09/13) LAST 追加
	 * @og.rev 5.8.2.3 (2014/12/27) USEREDIT追加
	 * @og.rev 6.2.2.4 (2015/04/24) SUM追加
	 * @og.rev 6.4.7.0 (2016/06/03) REQ追加
	 * @og.rev 6.5.0.0 (2016/09/30)) VAL追加｡value値とﾘｸｴｽﾄ変数では、ﾘｸｴｽﾄ変数が上位なので、value値を取り出したい場合に使用します｡
	 * @og.rev 6.7.7.0 (2017/03/31) applicationｽｺｰﾌﾟの文字列を取得します｡
	 * @og.rev 6.7.7.2 (2017/04/14) VAL に、&#064; 付きのﾊﾟﾗﾒｰﾀを使えるようにします｡
	 * @og.rev 5.9.26.1 (2017/11/10) JSON追加｡JSON化するのではなく、JSONﾀｲﾌﾟのｴｽｹｰﾌﾟ処理をする｡
	 * @og.rev 7.0.5.0 (2019/09/09) 追加 ENV.で環境変数の値を取得できるようにします｡
	 * @og.rev 7.2.0.0 (2020/02/14) SUMはValueTagとﾀﾞﾌﾞるため、SUMR とします｡
	 * @og.rev 7.2.3.1 (2020/04/17) System.getenv → HybsConst.getenv 変更(ｻｰﾋﾞｽ化対応) , VALS 追加 , SUBSTR 追加
	 * @og.rev 7.2.9.1 (2020/10/23) NN 追加、if処理を少しだけ分割します｡
	 * @og.rev 7.4.4.0 (2021/06/30) openGionV8事前準備(DataRole.java廃止)
	 * @og.rev 8.0.0.0 (2021/09/30) TBL.XXXX を設定します｡(TBL.CLMS,TBL.LBLS,TBL.SLBLS,TBL.COLS,TBL.ROWS)
	 * @og.rev 8.0.0.0 (2021/09/30) REP.XXXX を追加します｡
	 *
	 * @param	key	ｷｰ
	 * @return	ﾘｸｴｽﾄ情報の文字列
	 * @og.rtnNotNull
	 */
	protected String getReservedValue( final String key ) {
		if( key == null ) { isReqNull = true; return ""; }						// 3.4.0.3 (2003/09/10)

		String rtn = null;
		final int adrs = key.indexOf( '.' );

		if( adrs > 0 ) {
			final String subKey = key.substring( adrs+1 );
			if( adrs == 2 ) {
				// 4.0.0.0 (2007/06/12) DB.XXXX は、直接取り出すように変更します。
				if( key.startsWith( "DB." ) ) {
					rtn = (String)getRequestAttribute( key );					// ※ 取り出しは、key で
				}
				// 7.2.9.1 (2020/10/23)  NN 追加
				else if( key.startsWith( "NN." ) ) {
					rtn = getNNAttri( subKey );
				}
			}
			else if( adrs == 3 ) {
				// 3.8.0.2 (2005/07/11) MSG.XXXX で、ﾒｯｾｰｼﾞﾘｿｰｽの値を取得できるように追加｡
				// 3.8.0.2 (2005/07/11) LBL.XXXX で、ﾗﾍﾞﾙﾘｿｰｽの値を取得できるように追加｡
				if( key.startsWith( "LBL." ) ) {
					rtn = getLabel( subKey );
				}
				else if( key.startsWith( "SYS." ) ) {
					rtn = sys( subKey );		// 3.5.6.6 (2004/08/23)
				}
				// 3.4.0.3 (2003/09/10) MEM.XXXX で、REQUEST_CACHE の値を取得できるように修正。
				else if( key.startsWith( "MEM." ) ) {
					// 3.5.4.7 (2004/02/06) getRequestCacheData を使用するように修正
					rtn = getRequestCacheData( subKey );
				}
				// 5.6.4.0 (2013/05/02) NVL 追加
				else if( key.startsWith( "NVL." ) ) {
					rtn = getNVLAttri( subKey );
				}
				// 8.0.0.0 (2021/09/30) REP.XXXX を追加します。
				else if( key.startsWith( "REP." ) ) {
					rtn = getREPAttri( subKey );
				}
				// 6.5.0.0 (2016/09/30)) VAL追加
				else if( key.startsWith( "VAL." ) ) {
					// 6.7.7.2 (2017/04/14) VAL に、&#064; 付きのﾊﾟﾗﾒｰﾀを使えるようにします｡
					if( subKey != null && !subKey.isEmpty() && subKey.charAt(0) == '@' ) {
						final String tmpKey = getRequestValue( subKey.substring( 1 ) );
						rtn = (String)getRequestAttribute( tmpKey );
					}
					else {
						rtn = (String)getRequestAttribute( subKey );			// ※ 取り出しは、subKey で
					}
				}
				// 6.4.7.0 (2016/06/03) REQ追加
				else if( key.startsWith( "REQ." ) ) {
					rtn = getRequestMethod( subKey );
				}
				else if( key.startsWith( "GUI." ) ) {
					rtn = getGUIInfoAttri( subKey );							// 4.0.0 (2005/01/31)
				}
				// 6.7.7.0 (2017/03/31) applicationｽｺｰﾌﾟの文字列を取得します｡
				else if( key.startsWith( "APP." ) ) {
					rtn = String.valueOf( getContextAttribute( subKey ) );
				}
				// 4.3.7.0 (2009/06/01) DB関数名の取得
				else if( key.startsWith( "DBF." ) ) {
					rtn = getDBFunctionName( subKey );
				}
				// 5.5.1.3 (2012/04/09) ｴｽｹｰﾌﾟ変換
				else if( key.startsWith( "ESC." ) ) {
					rtn = StringUtil.htmlFilter( getRequestValue(subKey,false) );
				}
				// 4.4.0.0 (2009/08/02) ﾃﾞｰﾀﾛｰﾙに基づく条件式の取得 7.4.4.0 (2021/06/30) Delete
	//			else if( key.startsWith( "SEC." ) ) {
	//				rtn = getDataCondition( subKey );
	//			}
				// 5.3.9.0 (2011/09/01) URLｴﾝｺｰﾄﾞ変換
				else if( key.startsWith( "URL." ) ) {
					rtn = StringUtil.urlEncode( getRequestValue( subKey ) );
				}
				// 7.0.5.0 (2019/09/09) 追加 ENV.で環境変数の値を取得できるようにします。
				else if( key.startsWith( "ENV." ) ) {
	//				rtn = System.getenv( subKey );
					rtn = HybsConst.getenv( subKey );							// 7.2.3.1 (2020/04/17)
				}
				// 8.0.0.0 (2021/09/30) TBL.XXXX は、直接取り出すように変更します。(TBL.CLMS,TBL.LBLS,TBL.SLBLS,TBL.COLS,TBL.ROWS)
				else if( key.startsWith( "TBL." ) ) {
					rtn = (String)getRequestAttribute( key );					// ※ 取り出しは、key で
				}
			}
			else if( adrs == 4 ) {
				// 3.8.0.2 (2005/07/11) DATE.XXXX で、日付関係の値を取得できるように追加。
				if( key.startsWith( "DATE." ) ) {
					rtn = getDateFormat( subKey );
				}
				else if( key.startsWith( "USER." ) ) {
					rtn = getUserInfo( subKey );
				}
				// 5.6.8.1 (2013/09/13) LAST 追加
				else if( key.startsWith( "LAST." ) ) {
					rtn = getLASTAttri( subKey );
				}
				// 4.3.6.0 (2009/04/01) ﾒｰﾙﾓｼﾞｭｰﾙ用の予約語
				else if( key.startsWith( "MAIL." ) ) {
					rtn = ( String )getSessionAttribute( key );
				}
				// 6.2.2.4 (2015/04/24) SUM追加
				// 7.2.0.0 (2020/02/14) SUMはValueTagとﾀﾞﾌﾞるため、SUMR とします｡
				// 他に、MIN,MAX,HMF,AVG も、ValueTagで使用中なので注意すること｡
	//			else if( key.startsWith( "SUM." ) ) {
				else if( key.startsWith( "SUMR." ) ) {
					rtn = getSumRequestValue( subKey );
				}
				// 5.9.26.1 (2017/11/10) 追加 JSONﾀｲﾌﾟのｴｽｹｰﾌﾟを行う(JSONにするわけではない）
				else if( key.startsWith( "JSON." ) ) {
					rtn = StringUtil.jsonFilter( getRequestValue(subKey) );
				}
				// 7.2.3.1 (2020/04/17) VALS 追加
				else if( key.startsWith( "VALS." ) ) {
					rtn = getValsAttri( subKey );
				}
				// 3.8.0.1 (2005/06/17) NVAR.XXXX で、getUnicodeEscape 変換() を行います｡
				// NVAR. を取り除いた XXXX で再度、ﾘｸｴｽﾄ値を取得し、それを Escape変換します｡
				else if( key.startsWith( "NVAR." ) ) {
					rtn = StringUtil.getUnicodeEscape( getRequestValue( subKey ) );
				}
			}
			else if( adrs > 5 ) {
				// 7.2.3.1 (2020/04/17) SUBSTR 追加
				if( key.startsWith( "SUBSTR." ) ) {
					rtn = getSubstrAttri( subKey );
				}
				else if( key.startsWith( "USEREDIT." ) ) {
					rtn = getUserEditInfo( subKey );							// 5.8.2.3 (2014/12/27)
				}
				else if( key.startsWith( "SESSION." ) ) {						// 3.5.5.3 (2004/04/09)
					rtn = String.valueOf( getSessionAttribute( subKey ) );
				}
			}
			// 7.2.9.1 (2020/10/23) 処理されなかった場合は、そのままｷｰとして使用する｡
			if( rtn == null ) {
				rtn = (String)getRequestAttribute( key );						// ※ 取り出しは、key で
			}
//			// 6.3.5.0 (2015/08/08) CHART.TAG は、直接取り出すように変更します。
//			else { // 4.0.0.0 (2007/11/16)
//				rtn = (String)getRequestAttribute( key );						// ※ 取り出しは、key で
//			}
		}
		// 予約語以外は括弧を付けて書き戻します。
		else{
			rtn = "{@" + key + "}";
		}
		return rtn;
	}

	/**
	 * ﾘｸｴｽﾄ情報の文字列を取得します｡
	 *
	 * @og.rev 5.0.0.2 (2009/09/15) XSS対策
	 *
	 * @param	key	ｷｰ
	 * @return	ﾘｸｴｽﾄ情報の文字列
	 */
	protected String getRequestValue( final String key ) {
		return getRequestValue( key, xssCheck);
	}

	/**
	 * ﾘｸｴｽﾄ情報の文字列を取得します｡
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 ｴﾝｺｰﾄﾞの取得方法変更
	 * @og.rev 3.0.0.0 (2002/12/25) StringUtil#changeString 廃止
	 * @og.rev 3.0.0.0 (2002/12/25) ValueTag追加の為、指定の scope の Attributeより取得
	 * @og.rev 3.1.0.1 (2003/03/26) Valueﾀｸﾞの値と、request情報の値の所得優先順位を、request が優先されるように変更｡
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応｡HybsRequestWrapper 廃止｡直接 Mapでｷｬｯｼｭする｡
	 * @og.rev 3.1.5.0 (2003/04/22) SYS.XXXX で、ｼｽﾃﾑﾊﾟﾗﾒｰﾀ の値を取得できるように修正｡
	 * @og.rev 3.1.7.0 (2003/05/02) ﾘｸｴｽﾄ情報の取得順序を、Request、ｷｬｯｼｭ、Value の順に変更｡
	 * @og.rev 3.1.7.0 (2003/05/02) value値の使用可否を指定する、useValue 属性を追加｡
	 * @og.rev 3.4.0.3 (2003/09/10) MEM.XXXX で、REQUEST_CACHE の値を取得できるように修正｡
	 * @og.rev 3.5.4.7 (2004/02/06) getRequestCacheData を使用するように修正
	 * @og.rev 3.5.5.3 (2004/04/09) {&#064;SESSION.XXXX} で、session.getAttribute( "XXXX" ) の値を取得するように修正
	 * @og.rev 3.5.6.6 (2004/08/23) SYS.XXXX の処理を getSystemParameter( String key ) ﾒｿｯﾄﾞへ移動
	 * @og.rev 3.8.0.1 (2005/06/17) NVAR.XXXX で、getUnicodeEscape 変換() を行います｡
	 * @og.rev 3.8.0.2 (2005/07/11) MSG.XXXX , LBL.XXXX の処理を追加
	 * @og.rev 3.8.0.2 (2005/07/11) ﾁｪｯｸﾎﾞｯｸｽ対応で、重複ﾘｸｴｽﾄに対応させます｡
	 * @og.rev 3.8.8.8 (2007/05/11) 重複ﾘｸｴｽﾄ処理の場所を移動｡ﾘｸｴｽﾄのみ対象とする｡
	 * @og.rev 4.0.0.0 (2005/08/31) quotCheck によるＳＱＬｲﾝｼﾞｪｸｼｮﾝ対策
	 * @og.rev 4.0.0.0 (2005/08/31) getSystemParameter を sys に名称変更
	 * @og.rev 4.0.0.0 (2007/04/02) Valueﾀｸﾞの値と、ｷｬｯｼｭでは、Valueﾀｸﾞの値を優先するように変更
	 * @og.rev 4.0.0.0 (2007/11/16) "."付きのﾊﾟﾗﾒｰﾀのｴﾗｰ処理をなくし、getRequestAttributeで取得する｡
	 * @og.rev 4.3.0.0 (2008/07/04) DB.XXXX は、必ずStringｵﾌﾞｼﾞｪｸﾄとし、String.valueOf しない｡
	 * @og.rev 4.3.6.0 (2009/04/01) ﾒｰﾙﾓｼﾞｭｰﾙ用の予約語MAIL.XXXXの取得対応
	 * @og.rev 4.4.0.0 (2009/08/02) ﾃﾞｰﾀﾛｰﾙ対応(SEC.xxxの取得対応)
	 * @og.rev 5.0.0.2 (2009/09/15) XSS対策用にﾒｿｯﾄﾞにﾌﾗｸﾞを追加
	 * @og.rev 5.1.8.0 (2010/07/01) isNullSet 属性 廃止にともなう、useValue 属性廃止
	 * @og.rev 5.3.9.0 (2011/09/01) URL.XXXX処理を追加
	 * @og.rev 5.5.1.3 (2012/04/09) ESC.XXXX処理を追加
	 * @og.rev 5.5.4.0 (2012/07/01) 予約語の処理を分離
	 * @og.rev 5.7.4.2 (2014/03/20) ｻﾆﾀｲｽﾞ処理は、getSanitizedBodyString() ではなく、ここで行います｡
	 * @og.rev 5.9.25.2 (2017/10/27) xssCheck及びquotCheckのｴﾗｰﾒｯｾｰｼﾞをﾗﾍﾞﾙﾘｿｰｽ化
	 *
	 * @param	key			ｷｰ
	 * @param	xssCheckFlg	XSS対策用[true:行う/false:行わない]
	 *
	 * @return	ﾘｸｴｽﾄ情報の文字列
	 * @og.rtnNotNull
	 */
	protected String getRequestValue( final String key, final boolean xssCheckFlg ) {
		if( key == null ) { isReqNull = true; return ""; }						// 3.4.0.3 (2003/09/10)

		String rtn ;
		final int adrs = key.indexOf( '.' );

		if( adrs > 0 ) {
			rtn = getReservedValue( key );										// 5.5.4.0 (2012/07/02)
		}
		else {
			rtn = getRequest().getParameter( key );

			// 5.7.4.2 (2014/03/20) ｻﾆﾀｲｽﾞ処理は、getSanitizedBodyString() ではなく、ここで行います｡
			// 6.0.0.1 (2014/04/25) These nested if statements could be combined
			if( isSanitized && rtn != null && rtn.indexOf( '[' ) >= 0 ) {
				rtn = rtn.replace( "[", "\\]\\" );
			}

			// 5.0.0.2 (2009/09/15) tagCheck によるthan signﾁｪｯｸ Parameterのみにかけるためこの位置
			if( rtn != null && rtn.length() > 0 && xssCheckFlg && ( rtn.indexOf( '<' ) >= 0 || rtn.indexOf( '>' ) >= 0 ) ) {

				// 5.9.25.2 (2017/10/27)
				getResource();
				final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
					.append( resource.getLabel( "ERR0048.1" ) ).append( CR )
					.append( resource.getLabel( "ERR0048.2",new String[] { key,rtn,getTagName() } ) ).append( CR );

				throw new HybsSystemException( buf.toString() );
			}

			// 3.8.8.8 (2007/05/11) 重複ﾘｸｴｽﾄ処理の場所を移動｡ﾘｸｴｽﾄのみ対象とする｡
			// 3.8.0.2 (2005/07/11) ﾁｪｯｸﾎﾞｯｸｽ対応で、重複ﾘｸｴｽﾄに対応させます｡
			// {@XXXX} で、値が"0"の場合、複数ﾘｸｴｽﾄを確認して、"1"が含まれていれば、"1"とします｡
			if( "0".equals(rtn) ) {
				final boolean backFlag = isReqNull ;
				final String[] vals = getRequestValues( key );
				if( vals != null && vals.length > 1 ) {
					for( int i=0; i<vals.length; i++ ) {
						if( "1".equals( vals[i] ) ) { rtn = "1"; break; }
					}
				}
				isReqNull = backFlag;											// 3.8.8.8 (2007/05/11) getRequestValues での NULLセット解除
			}

			// 3.1.0.1 (2003/03/26) Valueﾀｸﾞの値と、request情報の値の取得優先順位を、
			// request が優先されるように変更｡
			if( ( rtn == null || rtn.isEmpty() ) && requestCache != null ) {
				final String[] str = requestCache.get( key );
				if( str != null && str.length > 0 ) {
					rtn = str[0];
				}
			}
			// 5.1.8.0 (2010/07/01) isNullSet 属性 廃止にともなう、useValue 属性廃止
			if( rtn == null || rtn.isEmpty() ) {
				final Object obj = pageContext.findAttribute( key );
				if( obj != null ) {
					rtn = obj.toString();
				}
			}
		}
		if( rtn == null || rtn.isEmpty() ) {
			isReqNull = true;
			rtn    = "";
		}
		// 4.0.0 (2005/08/31) quotCheck によるSQLｲﾝｼﾞｪｸｼｮﾝ対策
//		else if( quotCheck && rtn.indexOf( '\'' ) >= 0 && !key.startsWith( "SEC." ) ) {	// 6.0.2.5 (2014/10/31) refactoring
		else if( quotCheck && rtn.indexOf( '\'' ) >= 0 && !key.startsWith( "REP." ) ) {	// 7.4.4.0 (2021/06/30) Delete SEC." 廃止

			// 5.9.25.2 (2017/10/27)
			getResource();
			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
				.append( resource.getLabel( "ERR0049.1" ) ).append( CR )
				.append( resource.getLabel( "ERR0049.2",new String[] { key,rtn,getTagName() } ) ).append( CR );

			throw new HybsSystemException( buf.toString() );
		}

		return rtn ;
	}

	/**
	 * ﾘｸｴｽﾄ情報の文字列を取得します｡
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 ｴﾝｺｰﾄﾞの取得方法変更
	 * @og.rev 3.0.0.0 (2002/12/25) StringUtil#changeString 廃止
	 * @og.rev 3.1.8.0 (2003/05/16) RequestCache ﾃﾞｰﾀをﾘｸｴｽﾄ配列情報にも適用する｡
	 * @og.rev 5.3.8.0 (2011/08/01) Attribute等からも値が取得できるようにする｡
	 *
	 * @param	key	ｷｰ
	 * @return	ﾘｸｴｽﾄ情報の文字列
	 */
	protected String[] getRequestValues( final String key ) {
		String[] rtn = getRequest().getParameterValues( key );

		// 3.1.8.0 (2003/05/16) RequestCache ﾃﾞｰﾀをﾘｸｴｽﾄ配列情報にも適用する｡
		if( ( rtn == null || rtn.length == 0 ) && requestCache != null ) {
			rtn =requestCache.get( key );
		}

		// 5.3.8.0 (2011/08/01) Attribute等からも値が取得できるようにする｡
		if( rtn == null || rtn.length == 0 ) {
			final String tmp = getRequestValue( key );
			if( tmp != null && tmp.length() > 0 ) {
				rtn = new String[]{ tmp };
			}
		}

		if( rtn == null || rtn.length == 0 ) { isReqNull = true; }
		return rtn ;
	}

	/**
	 * ﾘｸｴｽﾄ情報の文字列のｷｰ集合を取得します｡
	 *
	 * @og.rev 5.3.2.0 (2011/02/01) ﾊﾟﾗﾒｰﾀｰの外部指定対応
	 *
	 * @return	ﾘｸｴｽﾄ情報の文字列のｷｰ集合
	 */
	protected Enumeration<?> getParameterNames() {								// 4.3.3.6 (2008/11/15) Generics警告対応
		final String[] names = (String[])getRequestAttribute( HybsSystem.PARAM_NAMES_KEY );
		return names == null ? getRequest().getParameterNames() : Collections.enumeration( Arrays.asList( names ) ) ;
	}

	/**
	 * ﾘｸｴｽﾄ情報の文字列のｷｰ集合をｾｯﾄします｡
	 *
	 * @og.rev 5.3.2.0 (2011/02/01) ﾊﾟﾗﾒｰﾀｰの外部指定対応
	 *
	 * @param	names	ﾘｸｴｽﾄ情報の文字列のｷｰ配列(可変長引数)
	 */
	protected void setParameterNames( final String... names ) {
		setRequestAttribute( HybsSystem.PARAM_NAMES_KEY, names );
	}

	/**
	 * ﾘｸｴｽﾄ情報の文字列に NULL が存在していたかどうかを取得します｡
	 *
	 * これは、getRequestParameter( String ) の呼出し毎に設定されます｡
	 * つまり、上記ﾒｿｯﾄﾞの実行直後の値を取り出す必要があります｡
	 * NULL が含まれていた(true)／含まれていなかった｡(false)
	 *
	 * @return	NULLが含まれていた(true)／含まれていなかった｡(false)
	 */
	protected boolean isNull() {
		return isReqNull;
	}

	/**
	 * ｾｯｼｮﾝに登録されているｵﾌﾞｼﾞｪｸﾄを取得します｡
	 *
	 * @param	key	ｷｰ
	 * @return	ｾｯｼｮﾝに登録されているｵﾌﾞｼﾞｪｸﾄ
	 */
	protected Object getSessionAttribute( final String key ) {
		if( session == null ) { session = pageContext.getSession(); }
		return session.getAttribute( key );
	}

	/**
	 * ｾｯｼｮﾝに 指定のｷｰでｵﾌﾞｼﾞｪｸﾄをｾｯﾄします｡
	 *
	 * @param	key		ｷｰ
	 * @param	object	ｾｯｼｮﾝに登録するｵﾌﾞｼﾞｪｸﾄ
	 */
	protected void setSessionAttribute( final String key ,final Object object ) {
		if( session == null ) { session = pageContext.getSession(); }
		session.setAttribute( key,object );
	}

	/**
	 * ｾｯｼｮﾝに指定のｷｰで登録されているｵﾌﾞｼﾞｪｸﾄを 削除します｡
	 *
	 * @param	key	ｷｰ
	 */
	protected void removeSessionAttribute( final String key ) {
		if( session == null ) { session = pageContext.getSession(); }
		session.removeAttribute( key );
	}

	/**
	 * ﾘｸｴｽﾄに登録されているｵﾌﾞｼﾞｪｸﾄを取得します｡
	 *
	 * @param	key	ｷｰ
	 *
	 * @return	ﾘｸｴｽﾄﾝに登録されているｵﾌﾞｼﾞｪｸﾄ
	 */
	protected Object getRequestAttribute( final String key ) {
		return getRequest().getAttribute( key );
	}

	/**
	 * ﾘｸｴｽﾄに 指定のｷｰでｵﾌﾞｼﾞｪｸﾄをｾｯﾄします｡
	 *
	 * @param	key		ｷｰ
	 * @param	object	ﾘｸｴｽﾄに登録するｵﾌﾞｼﾞｪｸﾄ
	 */
	protected void setRequestAttribute( final String key ,final Object object ) {
		getRequest().setAttribute( key,object );
	}

	/**
	 * ﾘｸｴｽﾄに指定のｷｰで登録されているｵﾌﾞｼﾞｪｸﾄを削除します｡
	 *
	 * @param	key	ｷｰ
	 */
	protected void removeRequestAttribute( final String key ) {
		getRequest().removeAttribute( key );
	}

	/**
	 * ｺﾝﾃｷｽﾄ(application)に登録されているｵﾌﾞｼﾞｪｸﾄを取得します｡
	 *
	 * scope属性に、"application" が指定された場合に、実行されます｡
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) scope="application" 指定の追加
	 *
	 * @param	key	ｷｰ
	 * @return	ｺﾝﾃｷｽﾄ(application)に登録されているｵﾌﾞｼﾞｪｸﾄ
	 */
	protected Object getContextAttribute( final String key ) {
		final ServletContext application = pageContext.getServletContext();
		return application.getAttribute( key );
	}

	/**
	 * ｺﾝﾃｷｽﾄ(application)指定のｷｰでｵﾌﾞｼﾞｪｸﾄをｾｯﾄします｡
	 *
	 * scope属性に、"application" が指定された場合に、実行されます｡
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) scope="application" 指定の追加
	 *
	 * @param	key	ｷｰ
	 * @param	object ｺﾝﾃｷｽﾄ(application)に登録するｵﾌﾞｼﾞｪｸﾄ
	 */
	protected void setContextAttribute( final String key ,final Object object ) {
		final ServletContext application = pageContext.getServletContext();
		application.setAttribute( key,object );
	}

	/**
	 * ｺﾝﾃｷｽﾄ(application)指定のｷｰで登録されているｵﾌﾞｼﾞｪｸﾄを 削除します｡
	 *
	 * scope属性に、"application" が指定された場合に、実行されます｡
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) scope="application" 指定の追加
	 *
	 * @param	key	ｷｰ
	 */
	protected void removeContextAttribute( final String key ) {
		final ServletContext application = pageContext.getServletContext();
		application.removeAttribute( key );
	}

	/**
	 * ｱﾌﾟﾘｹｰｼｮﾝｻｰﾊﾞｰのｺﾝﾃｷｽﾄﾊﾟｽのURLを返します｡
	 *
	 * @return	ｺﾝﾃｷｽﾄﾊﾟｽ
	 */
	protected String getContextPath() {
		return ((HttpServletRequest)getRequest()).getContextPath();
	}

	/**
	 * ｽｺｰﾌﾟに応じて登録されているｵﾌﾞｼﾞｪｸﾄを取得します｡
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) scope="application" 指定の追加
	 * @og.rev 6.7.7.0 (2017/03/31) scope="application" つづり間違い訂正
	 *
	 * @param	key	ｷｰ
	 * @return	ｽｺｰﾌﾟに応じて登録されているｵﾌﾞｼﾞｪｸﾄ
	 */
	protected Object getObject( final String key ) {
		if(      "session".equals( scope )		) { return getSessionAttribute( key ); }
		else if( "request".equals( scope )		) { return getRequestAttribute( key ); }
		else if( "application".equals( scope )	) { return getContextAttribute( key ); }		// 6.7.7.0 (2017/03/31)
		else {
			final String errMsg = "このｽｺｰﾌﾟはｻﾎﾟｰﾄされていません。[" + scope + "]";
			throw new IllegalArgumentException( errMsg );
		}
	}

	/**
	 * ｽｺｰﾌﾟに応じて登録されているｵﾌﾞｼﾞｪｸﾄを指定のｷｰでｾｯﾄします｡
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) scope="application" 指定の追加
	 *
	 * @param	key		ｷｰ
	 * @param	object	ﾘｸｴｽﾄに登録するｵﾌﾞｼﾞｪｸﾄ
	 * @see		#setObject( String ,Object ,String )
	 */
	protected void setObject( final String key ,final Object object ) {
		setObject( key,object,scope );
	}

	/**
	 * ｽｺｰﾌﾟに応じて登録されているｵﾌﾞｼﾞｪｸﾄを指定のｷｰでｾｯﾄします｡
	 *
	 * 引数にｽｺｰﾌﾟを指定します｡ｽｺｰﾌﾟが null の場合は、ｵﾘｼﾞﾅﾙの
	 * ｽｺｰﾌﾟを使用します｡
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) 新規追加
	 * @og.rev 6.7.7.0 (2017/03/31) scope="application" つづり間違い訂正
	 *
	 * @param	key		ｷｰ
	 * @param	object	ﾘｸｴｽﾄに登録するｵﾌﾞｼﾞｪｸﾄ
	 * @param	scp		ｽｺｰﾌﾟ
	 * @see		#setObject( String ,Object )
	 */
	protected void setObject( final String key ,final Object object ,final String scp ) {
		final String inScp = (scp == null) ? scope : scp ;

		if(      "session".equals( inScp	) ) { setSessionAttribute( key,object ); }
		else if( "request".equals( inScp	) ) { setRequestAttribute( key,object ); }
		else if( "application".equals( inScp ) ) { setContextAttribute( key,object ); }	// 6.7.7.0 (2017/03/31)
		else {
			final String errMsg = "このｽｺｰﾌﾟはｻﾎﾟｰﾄされていません。[" + inScp + "]";
			throw new IllegalArgumentException( errMsg );
		}
	}

	/**
	 * ｽｺｰﾌﾟに応じて登録されているｵﾌﾞｼﾞｪｸﾄを指定のｷｰで削除します｡
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) scope="application" 指定の追加
	 * @og.rev 6.7.7.0 (2017/03/31) scope="application" つづり間違い訂正
	 *
	 * @param	key	ｷｰ
	 */
	protected void removeObject( final String key ) {
		if( "session".equals( scope ) ) { removeSessionAttribute( key ); }
		else if( "request".equals( scope ) ) { removeRequestAttribute( key ); }
		else if( "application".equals( scope ) ) { removeContextAttribute( key ); }	// 6.7.7.0 (2017/03/31)
		else {
			final String errMsg = "このｽｺｰﾌﾟはｻﾎﾟｰﾄされていません。[" + scope + "]";
			throw new IllegalArgumentException( errMsg );
		}
	}

	/**
	 * ﾘｸｴｽﾄｵﾌﾞｼﾞｪｸﾄを取得します｡
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 ｴﾝｺｰﾄﾞの取得方法変更
	 * @og.rev 2.2.0.0 (2002/12/17) 文字化け対策 setCharacterEncoding が効いていないので削除
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応｡HybsRequestWrapper 廃止｡直接 Mapでｷｬｯｼｭする｡
	 * @og.rev 3.4.0.3 (2003/09/10) 冗長な個所や、無意味な個所を、等価な内容のﾛｼﾞｯｸに置き換える｡
	 * @og.rev 3.5.5.0 (2004/03/12) command=RESET 時にも、ｷｬｯｼｭを取り出すように変更します｡
	 *
	 * @return	ﾘｸｴｽﾄｵﾌﾞｼﾞｪｸﾄ
	 */
	@SuppressWarnings(value={"unchecked"})
	protected ServletRequest getRequest() {
		if( request == null ) {
			request = pageContext.getRequest();
			// ﾘｸｴｽﾄｷｬｯｼｭ機能
			final String cmd =request.getParameter( "command" );
			if( "RENEW".equals( cmd ) || "RESET".equals( cmd ) ) {				// 3.5.5.0
				requestCache = (Map<String,String[]>)getSessionAttribute( HybsSystem.REQ_CACHE_KEY );
			}
		}
		return request;
	}

	/**
	 * BodyContent ｵﾌﾞｼﾞｪｸﾄを取得して、ﾎﾞﾃﾞｨの内容を取得します｡
	 *
	 * 処理としては、getRequestParameter() によるﾊﾟﾗﾒｰﾀ処理も含みます｡
	 * このﾒｿｯﾄﾞは、必ず doAfterBody() から呼び出してください｡それ以外(例えば、
	 * doEndTag()等)では、すでに Body情報が破棄/再利用されている可能性があり、
	 * 正常に動作しなくなる可能性があります｡
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) BodyContent ｵﾌﾞｼﾞｪｸﾄを取得して、ﾎﾞﾃﾞｨの内容を取得する処理を追加
	 * @og.rev 6.3.1.1 (2015/07/10) BodyString,BodyRawStringは、CommonTagSupport で、trim() します｡
	 *
	 * @return	ﾎﾞﾃﾞｨ文字列
	 */
	protected String getBodyString() {
		final BodyContent body = getBodyContent();
		return getRequestParameter( body.getString().trim() );					// 6.3.1.1 (2015/07/10)
	}

	/**
	 * BodyContent ｵﾌﾞｼﾞｪｸﾄを取得して、ﾎﾞﾃﾞｨの内容を取得します｡
	 *
	 * {&#064;XXXX}を変換しない生のBODY文を返します
	 *
	 * @og.rev 4.3.6.0 (2009/04/01) 新規作成
	 * @og.rev 6.3.1.1 (2015/07/10) BodyString,BodyRawStringは、CommonTagSupport で、trim() します｡
	 *
	 * @return	ﾎﾞﾃﾞｨ文字列
	 */
	protected String getBodyRawString() {
		final BodyContent body = getBodyContent();
		return body.getString().trim();											// 6.3.1.1 (2015/07/10)
	}

	/**
	 * BodyContent ｵﾌﾞｼﾞｪｸﾄを取得して、ﾎﾞﾃﾞｨの内容を取得します｡
	 *
	 * {&#064;XXXX}の変換を行いますが、その変換結果に、"["が含まれる場合は、
	 * "\\]\\"に変換して、ﾌｫｰﾏｯﾄ処理されないようにｻﾆﾀｲｽﾞします｡
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 新規作成
	 * @og.rev 5.7.4.2 (2014/03/20) ｻﾆﾀｲｽﾞ処理を、標準の処理で行う｡
	 *
	 * @return	ﾎﾞﾃﾞｨ文字列
	 */
	protected String getSanitizedBodyString() {
		isSanitized = true;
		final String rtn = getBodyString();
		isSanitized = false;													// 一連の処理の中だけ、有効とします。

		return rtn;
	}

	/**
	 * JspWriter を使用した画面出力です。
	 *
	 * @param	msg	画面に出力する文字列
	 */
	protected void jspPrint( final String msg ) {
		if( msg == null ) { return ; }
		try {
			final JspWriter out = pageContext.getOut();
			out.print( msg );
		} catch( final IOException ex ) {
			final String errMsg = "画面出力時の PageContext の取得時にｴﾗｰが発生しました。"
							+ ex.getMessage();									// 5.1.8.0 (2010/07/01) errMsg 修正
			throw new HybsSystemException( errMsg,ex );							// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
	}

	/**
	 * ﾃﾞﾊﾞｯｸﾞ用の文字列を JspWriter を使用して画面に出力します｡
	 * 引数を単に画面に表示するだけです｡
	 *
	 * @og.rev 8.0.2.0 (2021/11/30) 検索実行前に、SQL文字をdebugPrint出来るように修正｡
	 *
	 * @param	msg	表示する文字列
	 */
	protected void debugPrint( final String msg ) {
		if( debugFlag ) {
			try {
				final JspWriter out = pageContext.getOut();
				out.println( "<pre>" );
				out.println( msg );
				out.println( "</pre>" );
			} catch( final IOException ex ) {
				final String errMsg = "ﾃﾞﾊﾞｯｸﾞ画面出力時の PageContext の取得時にｴﾗｰが発生しました。"
							+ ex.getMessage();
				throw new HybsSystemException( errMsg,ex );
			}
		}
	}

	/**
	 * ﾃﾞﾊﾞｯｸﾞ用の文字列を JspWriter を使用して画面に出力します｡
	 * このﾒｿｯﾄﾞは、debugFlag=true の場合のみ動作します｡
	 *
	 * 出力内容は，各ｵﾌﾞｼﾞｪｸﾄの toString() 文字列です｡
	 *
	 * @og.rev 4.0.0.0 (2005/02/28) debugFlag の条件式を追加｡
	 * @og.rev 4.0.0.0 (2005/02/28) ｸｲｯｸﾘﾌｧﾚﾝｽへのﾘﾝｸを追加｡
	 */
	protected void debugPrint() {
		if( debugFlag ) {
			try {
				final JspWriter out = pageContext.getOut();
				out.println( getDocumentLink() );								// 4.0.0 (2005/02/28)
				out.println( "<pre>" );
				out.println( toString() );
				out.println( "</pre>" );
			} catch( final IOException ex ) {
				final String errMsg = "ﾃﾞﾊﾞｯｸﾞ画面出力時の PageContext の取得時にｴﾗｰが発生しました。"
							+ ex.getMessage();									// 5.1.8.0 (2010/07/01) errMsg 修正
				throw new HybsSystemException( errMsg,ex );						// 3.5.5.4 (2004/04/15) 引数の並び順変更
			}
		}
	}

	/**
	 * GAMENID付のﾘｸｴｽﾄｷｬｯｼｭ情報を取り出します｡
	 *
	 * @og.rev 3.5.4.7 (2004/02/06) 新規作成
	 *
	 * @param	key	ﾘｸｴｽﾄｷｬｯｼｭのｷｰ情報
	 * @return	ﾘｸｴｽﾄ情報(存在しない場合は、null)
	 */
	protected String getRequestCacheData( final String key ) {
		String rtn = null;

		final String memKey = HybsSystem.REQ_CACHE_KEY + getGUIInfoAttri( "KEY" );	// 4.0.0 (2005/01/31)
		final Map<?,?> mem = (Map<?,?>)getSessionAttribute( memKey );				// 4.3.3.6 (2008/11/15) Generics警告対応

		if( mem != null ) {
			final String[] vals = (String[])mem.get( key );
			if( vals != null && vals.length > 0 ) {
				rtn = vals[0];
			}
		}
		return rtn ;
	}

	/**
	 * GAMENID付のﾘｸｴｽﾄｷｬｯｼｭ情報を取り出します｡
	 *
	 * @og.rev 3.5.4.7 (2004/02/06) 新規作成
	 *
	 * @param	key		ﾘｸｴｽﾄｷｬｯｼｭのｷｰ情報
	 * @param	value	ﾘｸｴｽﾄｷｬｯｼｭに登録する値
	 */
	protected void setRequestCacheData( final String key,final String value ) {
		final String[] vals = new String[] { value } ;

		final String memKey = HybsSystem.REQ_CACHE_KEY + getGUIInfoAttri( "KEY" );	// 4.0.0 (2005/01/31)
		@SuppressWarnings(value={"unchecked"})
		final Map<String,String[]> mem = (Map<String,String[]>)getSessionAttribute( memKey );
		if( mem != null ) {
			mem.put( key,vals );
		}
	}

	/**
	 * CSV形式引数(CSV引数)を配列に分解して返します｡
	 *
	 * CSV形式引数(CSV引数)で複数指定されたﾘｸｴｽﾄﾊﾟﾗﾒｰﾀを
	 * 文字列配列に分解して、返します｡
	 * 引数は、{&#064;XXXX} 変数も使用できます｡
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します｡
	 * こうしないとﾃﾞｰﾀ自身にｶﾝﾏを持っている場合に分解をﾐｽる為です｡
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 新規作成
	 *
	 * @param	csvKey	引数(CSV形式)
	 * @return	配列に分解されたﾘｸｴｽﾄﾊﾟﾗﾒｰﾀ値
	 */
	protected String[] getCSVParameter( final String csvKey ) {
		final String[] keys = StringUtil.csv2Array( csvKey );					// ﾀﾞﾌﾞﾙｸｵｰﾄ内は分解しない。
		String[] vals = new String[keys.length];
		for( int i=0; i<keys.length; i++ ) {
			vals[i] = getRequestParameter( keys[i] ) ;
		}
		return vals ;
	}

	/**
	 * CSV形式のkeys,vals引数(CSV引数)を配列に分解して返します｡
	 *
	 * CSV形式引数(CSV引数)で複数指定されたﾘｸｴｽﾄﾊﾟﾗﾒｰﾀを
	 * 文字列配列に分解して、返します｡
	 * 引数は、{&#064;XXXX} 変数も使用できます｡
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します｡
	 * こうしないとﾃﾞｰﾀ自身にｶﾝﾏを持っている場合に分解をﾐｽる為です｡
	 *
	 * ※ これは、keys,vals を同時に処理します｡keys を分解後、ｶﾝﾏがあれば、
	 *    vals も 再度、CSV分解します｡
	 *
	 * @og.rev 6.2.5.1 (2015/06/12) CSV形式のkeys,vals引数(CSV引数)対応｡新規作成｡
	 *
	 * @param	csvKey	引数(CSV形式)
	 * @param	csvVal	引数(CSV形式)
	 * @return	Mapにｾｯﾄされたﾘｸｴｽﾄﾊﾟﾗﾒｰﾀ値
	 * @see		#getCSVParameter( String )
	 */
	protected Map<String,String> getCSVParameter( final String csvKey , final String csvVal ) {
		final String[] keys = StringUtil.csv2Array( csvKey );					// ﾀﾞﾌﾞﾙｸｵｰﾄ内は分解しない｡
		final String[] vals = StringUtil.csv2Array( csvVal );					// ﾀﾞﾌﾞﾙｸｵｰﾄ内は分解しない｡

		if( keys.length != vals.length ) {
			final String errMsg = "ｷｰとﾊﾞﾘｭｰの個数が異なります。" + CR
						+ " keys.length=[" + keys.length + "]  vals.length=[" + vals.length + "]" + CR
						+ " keys=" + csvKey  + CR
						+ " vals=" + csvVal ;
			throw new HybsSystemException( errMsg );
		}

		final Map<String,String> kvMap = new LinkedHashMap<>();					// keys,valsの登録順は残しておきます。

		for( int i=0; i<keys.length; i++ ) {
			final String key1 = getRequestParameter( keys[i] ) ;				// ※ rtnNotNull
			String val1 = getRequestParameter( vals[i] ) ;
			if( key1.isEmpty() ) { continue; }									// ｷｰに関しては、何か値が必要｡

			if( key1.contains( "," ) ) {										// ｷｰにｶﾝﾏが含まれるとき
				final String[] keys2 = StringUtil.csv2Array( key1 );
				final String[] vals2 = StringUtil.csv2Array( val1 );

				// keys 分解の個別の個数ﾁｪｯｸ
				if( keys2.length != vals2.length ) {
					final String errMsg = "部分ｷｰと部分ﾊﾞﾘｭｰの個数が異なります。" + CR
								+ " keys2.length=[" + keys2.length + "]  vals2.length=[" + vals2.length + "]" + CR
								+ " orgKey=" + csvKey  + CR
								+ " orgVal=" + csvVal  + CR
								+ " keys2=" + key1 + CR
								+ " vals2=" + val1 ;
					throw new HybsSystemException( errMsg );
				}

				for( int j=0; j<keys2.length; j++ ) {
					if( keys2[j] != null && !keys2[j].isEmpty() && vals2[j] != null ) {
						kvMap.put( keys2[j] , vals2[j] );
					}
				}
			}
			else {
				// val にｶﾝﾏが含まれる場合は、前後に、ﾀﾞﾌﾞﾙｸｵｰﾄを追加する｡
				if( val1.contains( "," ) ) { val1 = '"' + val1 + '"' ; }
				kvMap.put( key1 , val1 );
			}
		}

		return kvMap ;
	}

	/**
	 * ｼｽﾃﾑ変数 {&#064;SYS.XXXX} に対する値の取得を行います｡
	 *
	 * 本来は、ｼｽﾃﾑﾊﾟﾗﾒｰﾀ の値を取得できますが、
	 * ｼｽﾃﾑ的な共有値も取得できるように機能追加しています｡
	 * また、ﾕｰｻﾞｰ個別にｼｽﾃﾑﾊﾟﾗﾒｰﾀを変更できます｡この取得は、
	 * ｼｽﾃﾑﾊﾟﾗﾒｰﾀとして取得します｡(値はﾕｰｻﾞｰ個別値を返します｡)
	 * ここで、引数に、&#064;変数が使用できます｡具体的には、{&#064;SYS.&#064;XXXX}
	 * で、&#064;XXXX をﾘｸｴｽﾄ変数から取得した値を利用します｡
	 * この中で、&#064;GUIID だけが、さらに特殊で、実行中の画面IDを割り当てます｡
	 * この &#064;GUIID は、ここまでの文字列を画面IDに置き換えるとともに、それ以降の
	 * 文字列を、画面IDに連結させます｡
	 * {&#064;SYS.&#064;GUIID_XXXX} ⇒ 画面ID_XXXX 文字列で、ｼｽﾃﾑﾊﾟﾗﾒｰﾀ の値を取得します｡
	 *
	 *  SERVER_NAME     このTomcatが実行しているｻｰﾊﾞｰ名       localhost 等
	 *  SERVER_URL      Portも含むURLｱﾄﾞﾚｽ                    http://localhost:8823/
	 *  CONTEXT_URL     実行しているｺﾝﾃｷｽﾄのURLｱﾄﾞﾚｽ          http://localhost:8823/dbdef2/
	 *  REAL_PATH       / ﾙｰﾄに対応する物理ﾃﾞｨﾚｸﾄﾘ            d:/webapps/dbdef2/ 等
	 *  CONTEXT_NAME    ｺﾝﾃｷｽﾄ名(webapps 直下の仮想ﾌｫﾙﾀﾞ名)   dbdef 等
	 *  COOKIE          ｸｯｷｰ取得
	 *  DATE            YMDH とほぼ同じですが、'yyyy/MM/dd HH:mm:ss' の形式で取得できます｡
	 *  HOSTNAME        IPﾄﾞﾚｽ ｽﾍﾟｰｽ区切りで指定したIPｱﾄﾞﾚｽからﾎｽﾄ名を逆引きします(5.6.6.2 (2013/07/19))
	 *  任意            ｼｽﾃﾑﾊﾟﾗﾒｰﾀ(GE12)の値を取得
	 *    &#064;GUIID_XXXX  既存の画面IDに、_XXXX を追加した文字列
	 *    &#064;XXXX        XXXX でﾘｸｴｽﾄ変数から取得した文字列
	 *    XXXX              XXXX の文字列
	 *  PRINTER         ｻｰﾊﾞｰのﾌﾟﾘﾝﾀｰ一覧(6.2.6.0 (2015/06/19))
	 *
	 * @og.rev 3.5.6.6 (2004/08/23) 新規作成
	 * @og.rev 3.7.0.3 (2005/03/01) ｸｯｷｰ取得機能を追加
	 * @og.rev 4.0.0.0 (2005/11/30) ﾕｰｻﾞｰﾊﾟﾗﾒｰﾀは、ｼｽﾃﾑﾊﾟﾗﾒｰﾀとして取得します｡
	 * @og.rev 5.1.6.0 (2010/05/01) ｼｽﾃﾑﾊﾟﾗﾒｰﾀに、&#064;GUIID という特殊ﾊﾟﾗﾒｰﾀが使用できるように対応します｡
	 * @og.rev 5.6.6.2 (2013/07/19) SYS.HOSTNAMEに対応します｡
	 * @og.rev 6.2.6.0 (2015/06/19) PRINTERに対応します｡
	 * @og.rev 6.4.2.0 (2016/01/29) DateSet.getDate( String ) を利用するように修正します｡
	 * @og.rev 6.9.5.0 (2018/04/23) うまく動いていないようなので、廃止
	 *
	 * @param	key	{&#064;SYS.XXXX}のXXXX部分の文字列(ｷｰ)
	 * @return	ｷｰに対する値｡なければ、null
	 */
	protected String sys( final String key ) {
		final String rtn;

		if( key.startsWith( "COOKIE." ) ) {										// 3.7.0.3 (2005/03/01)
			rtn = getCookie( key.substring( "COOKIE.".length() ) );
		}
		else if( key.startsWith( "DATE" ) ) {
			final int idx = key.indexOf( ' ' );
			if( idx >= 0 ) {
				rtn = DateSet.getDate( key.substring( idx+1 ) );				// 6.4.2.0 (2016/01/29)
			}
			else {
				rtn = HybsSystem.getDate();
			}
		}
		else if( key.startsWith( "HOSTNAME" ) ) {								// 5.6.6.2 (2013/07/19)
			final int idx = key.indexOf( ' ' );
			if( idx >= 0 ) {
				final String key2 = key.substring( idx+1 ) ;
				if( StringUtil.startsChar( key2 , '@' ) ) {						// 6.2.0.0 (2015/02/27) １文字 String.startsWith
					rtn = getHostName( getRequestValue( key2.substring( 1 ) ) );
				}
				else{
					rtn = getHostName( key2 );
				}
			}
			else{
				rtn = getUser().getParameter( key );
			}
		}
//		// 6.9.5.0 (2018/04/23) うまく動いていないようなので、廃止
//		// 5.1.6.0 (2010/05/01) {@SYS.@GUIID_XXXX} ﾊﾟﾗﾒｰﾀ対応
//		else if( key.startsWith( "@GUIID" ) ) {
//			final String key2 = getGUIInfoAttri( "ID" ) +  key.substring( "@GUIID".length() );
//			rtn = getUser().getParameter( key2 );
//		}
		// 6.2.6.0 (2015/06/19) PRINTERに対応します。
		else if( key.startsWith( "PRINTER" ) ) {
			rtn = HybsSystem.getPrinter();
		}
		// 5.1.6.0 (2010/05/01) {@SYS.@XXXX} ﾊﾟﾗﾒｰﾀ対応
		else if( StringUtil.startsChar( key , '@' ) ) {							// 6.2.0.0 (2015/02/27) １文字 String.startsWith
			final String key2 = getRequestValue( key.substring( 1 ) );
			rtn = getUser().getParameter( key2 );
		}
		else {
			rtn = getUser().getParameter( key );
		}

		return rtn ;
	}

	/**
	 * ｼｽﾃﾑﾊﾟﾗﾒｰﾀの値を、boolean 型に変換して返します｡
	 *
	 * 本来は、ｼｽﾃﾑﾊﾟﾗﾒｰﾀ の値を取得できますが、
	 * ｼｽﾃﾑ的な共有値も取得できるように機能追加しています｡
	 * また、ﾕｰｻﾞｰ個別にｼｽﾃﾑﾊﾟﾗﾒｰﾀを変更できます｡この取得は、
	 * ｼｽﾃﾑﾊﾟﾗﾒｰﾀとして取得します｡(値はﾕｰｻﾞｰ個別値を返します｡)
	 *
	 * @og.rev 4.0.0.0 (2005/11/30) 新規追加
	 *
	 * @param	key	ｼｽﾃﾑ設定ｷｰ
	 * @return	ｼｽﾃﾑ設定値(boolean型)
	 */
	protected boolean sysBool( final String key ) {
		return Boolean.parseBoolean( sys( key ) );		// 6.1.0.0 (2014/12/26) refactoring
	}

	/**
	 * ｼｽﾃﾑﾊﾟﾗﾒｰﾀの値を、int 型に変換して返します｡
	 *
	 * 本来は、ｼｽﾃﾑﾊﾟﾗﾒｰﾀ の値を取得できますが、
	 * ｼｽﾃﾑ的な共有値も取得できるように機能追加しています｡
	 * また、ﾕｰｻﾞｰ個別にｼｽﾃﾑﾊﾟﾗﾒｰﾀを変更できます｡この取得は、
	 * ｼｽﾃﾑﾊﾟﾗﾒｰﾀとして取得します｡(値はﾕｰｻﾞｰ個別値を返します｡)
	 *
	 * ※ ｼｽﾃﾑﾊﾟﾗﾒｰﾀの値が数字でない場合、HybsSystemException が throw されます｡
	 * ※ ｷｰの値が nullの場合、HybsSystemException が throw されます｡
	 *
	 * @og.rev 4.0.0.0 (2005/11/30) 新規追加
	 *
	 * @param	key	ｼｽﾃﾑ設定ｷｰ
	 * @return	ｼｽﾃﾑ設定値(int型)
	 */
	protected int sysInt( final String key ) {
		String tmp = null;
		int rtn ;
		try {
			tmp = sys( key );
			rtn = Integer.parseInt( tmp );
		}
		catch( final NumberFormatException ex ) {
			final String errMsg = "ｼｽﾃﾑﾊﾟﾗﾒｰﾀの値が数字ではありません。" + CR
					+ "  Resource key=[" + key + "] val=[" + tmp + "]"  ;
			throw new HybsSystemException( errMsg,ex );							// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
		catch( final IllegalArgumentException ex ) {
			final String errMsg = "ｷｰの値が null です。key=[" + key + "] val=[" + tmp + "]";
			throw new HybsSystemException( errMsg,ex );							// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}

		return rtn;
	}

	/**
	 * Transactionｵﾌﾞｼﾞｪｸﾄを取得します｡
	 * これは、自身のﾀｸﾞの親ﾀｸﾞ（囲われているﾀｸﾞ）から、TransactionTag を
	 * 見つけて、すでに、Transactionｵﾌﾞｼﾞｪｸﾄが作成済みなら、そのｵﾌﾞｼﾞｪｸﾄを
	 * そうでないなら、新規に作成して返します｡
	 *
	 * Transactionｵﾌﾞｼﾞｪｸﾄは、AutoCloseableｲﾝﾀﾌｪｰｽを実装しているため、
	 * try-with-resources構築を使用することが可能です｡
	 *
	 * @og.rev 6.3.6.1 (2015/08/28) Transactionｵﾌﾞｼﾞｪｸﾄの取得方法変更｡
	 *
	 * @return	Transactionｵﾌﾞｼﾞｪｸﾄ
	 */
	protected Transaction getTransaction() {
		final TransactionTag tranTag = (TransactionTag)findAncestorWithClass( this,TransactionTag.class );

		return tranTag == null  ? new TransactionReal( getApplicationInfo() )
								: tranTag.getTranObj();
	}

	/**
	 * session に、処理開始時刻を設定します｡
	 * これは、DBTableModel を登録する場合に、一連の処理が連続であるかどうかを
	 * 判断する時に使用します｡
	 * 処理が一連でない(start 時のﾀｲﾑｽﾀﾝﾌﾟが書き換えられている)場合は、
	 * DBTableModel の登録処理を行いません｡
	 * なお、判断処理を行うのは、scope が session の場合のみです｡
	 * 判定は、commitTableObject( String ,DBTableModel ) で行います｡
	 *
	 * @og.rev 3.6.0.8 (2004/11/19) 新規追加
	 * @og.rev 4.3.0.0 (2008/07/04) fileUD 対応｡
	 * @og.rev 5.1.6.0 (2010/05/01) DBLastSqlの処理は、DBTableModelが新規作成された処理でのみ行う｡
	 *
	 * @param	tableId	ｷｰ
	 * @see		#commitTableObject( String ,DBTableModel )
	 */
	protected void startQueryTransaction( final String tableId ) {
		if( "session".equals( scope ) ) {
			startTransaction = Long.valueOf( System.currentTimeMillis() );
			setSessionAttribute( tableId+"_TRANSACTION", startTransaction );
		}

		// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し
		if( useTrans ) {
			// 4.3.0.0 (2008/07/04) fileUD 対応
			removeSessionAttribute( HybsSystem.DB_LAST_SQL_KEY );				// 無条件削除
		}
	}

	/**
	 * ｽｺｰﾌﾟに応じて登録されている DBTableModel を指定のｷｰでｾｯﾄします｡
	 * これは、startQueryTransaction( String ) でｾｯｼｮﾝに登録した処理開始時刻と、
	 * このｵﾌﾞｼﾞｪｸﾄ自身が持っている(ｾｯｼｮﾝに登録した開始時刻そのもの)を
	 * 比較し、異なる場合は、DBTableModel の登録を行いません｡
	 * これにより、検索処理の開始順にしか登録しないようなﾛｼﾞｯｸを入れています｡
	 * 検索処理時間が掛かるSQLを実行した場合、先に検索した結果があとから登録される
	 * ｹｰｽがあるためです｡
	 * また、判断処理を行うのは、scope が session の場合のみです｡
	 *
	 * @og.rev 3.6.0.8 (2004/11/19) 新規追加
	 * @og.rev 3.8.1.1 (2005/11/21) ExcelOut の整合性を取る為の仕掛け
	 * @og.rev 4.3.0.0 (2008/07/04) fileUD 対応｡
	 * @og.rev 5.1.6.0 (2010/05/01) DBLastSqlの処理は、DBTableModelが新規作成された処理でのみ行う｡
	 * @og.rev 8.0.0.0 (2021/09/30) TBL.XXXX を設定します｡(TBL.CLMS,TBL.LBLS,TBL.SLBLS,TBL.COLS,TBL.ROWS)
	 *
	 * @param	tableId	ｷｰ
	 * @param	table	登録するDBTableModelｵﾌﾞｼﾞｪｸﾄ
	 * @return	正常に登録(true) / 一連でないため登録していない(false)
	 * @see		#startQueryTransaction( String )
	 */
	protected boolean commitTableObject( final String tableId ,final DBTableModel table ) {
		// 登録しないケースをピックアップします。
		if( "session".equals( scope ) ) {
			final String key = tableId+"_TRANSACTION";
			final Long endTime = (Long)getSessionAttribute( key );
			removeSessionAttribute( key );
			if( endTime == null ||
				startTransaction == null ||
				endTime.compareTo( startTransaction ) != 0 ) {
					final String msg = "CommonTagSupport Query処理が割り込まれました。DBTableModel は登録しません。"
								+ "[" + getUser().getUserID() + "],"
								+ "[" + getGUIInfoAttri( "KEY" ) + "]"			// 4.0.0 (2005/01/31)
								+ "[" + startTransaction + "]"					// 4.0.0 (2005/01/31)
								+ "[" + endTime + "]";							// 4.0.0 (2005/01/31)
					System.out.println( msg );
					return false;
			}
			// 3.8.1.1 (2005/11/21) ExcelOut の整合性を取る為の仕掛け
			if( table != null && HybsSystem.TBL_MDL_KEY.equals( tableId ) ) {
				final String consisKey = table.getConsistencyKey();
				setSessionAttribute( HybsSystem.TBL_MDL_CONKEY,consisKey );
			}
		}

		// 4.3.0.0 (2008/07/04) fileUD 対応
		// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し
		if( useTrans && table != null ) {
			final String guikey = getGUIInfoAttri( "KEY" );
			final DBLastSql lastSql = new DBLastSql( scope,guikey,table.isOverflow(),tableId );
			setSessionAttribute( HybsSystem.DB_LAST_SQL_KEY,lastSql );
		}

		setObject( tableId,table );

		// 8.0.0.0 (2021/09/30) TBL.XXXX を設定します。(TBL.CLMS,TBL.LBLS,TBL.SLBLS,TBL.COLS,TBL.ROWS)
		if( table != null ) {
			final StringBuilder clms = new StringBuilder( BUFFER_MIDDLE );
			final StringBuilder lbls = new StringBuilder( BUFFER_MIDDLE );
			final StringBuilder slbls = new StringBuilder( BUFFER_MIDDLE );
			for( final DBColumn clm : table.getDBColumns() ) {
				clms.append( clm.getName() ).append( ',' );									// ｶﾗﾑ名
				lbls.append( StringUtil.tagCut( clm.getLabel() ) ).append( ',' );			// ﾗﾍﾞﾙ名
				slbls.append( StringUtil.tagCut( clm.getShortLabel() ) ).append( ',' );		// ﾗﾍﾞﾙ名(名前(短))
			}

			// 最後のｶﾝﾏを除外します。
			clms.setLength( clms.length()-1 );
			lbls.setLength( lbls.length()-1 );
			slbls.setLength( slbls.length()-1 );

			setRequestAttribute( "TBL.CLMS" , clms.toString() );
			setRequestAttribute( "TBL.LBLS" , lbls.toString() );
			setRequestAttribute( "TBL.SLBLS", slbls.toString() );
			setRequestAttribute( "TBL.COLS" , String.valueOf( table.getColumnCount() ) );	// ｶﾗﾑの数
			setRequestAttribute( "TBL.ROWS" , String.valueOf( table.getRowCount() ) );		// 行数(={@DB.COUNT})
		}

		return true;
	}

	/**
	 * 表示ﾃﾞｰﾀの HybsSystem.ROW_SEL_KEY を元に、選ばれた 行番号の
	 * 配列を返します｡
	 * 配列情報は、行番号でｿｰﾄされて返されます｡
	 * なにも選ばれていない場合は、ｻｲｽﾞ０の配列を返します｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 * @og.rev 6.3.9.1 (2015/11/27) 内部処理が、長さが0の配列を返すように変更されたので、その対応｡
	 *
	 * @return	(選ばれていない場合は、ｻｲｽﾞ０の配列を返す)
	 * @og.rtnNotNull
	 */
	protected int[] getParameterRows() {
		if( rowNo != null ) { return rowNo; }

		// 6.3.9.1 (2015/11/27) org.opengion.hayabusa.servlet.MultipartRequest#getIntParameters(String) が、
		// 存在しない場合、長さが0の配列を返すことにしたので、同じﾛｼﾞｯｸに変更しておきます｡
		rowNo = (int[])getRequestAttribute( HybsSystem.ROW_SEL_KEY );
		if( rowNo != null && rowNo.length > 0 ) { return rowNo; }

		final String[] selected = getRequestValues( HybsSystem.ROW_SEL_KEY ) ;

		// 6.3.9.1 (2015/11/27) ﾗﾑﾀﾞ式で書き直します｡
		return selected == null || selected.length == 0
					? new int[0]
					: Arrays.stream( selected )
							.filter( str -> str != null && !str.isEmpty() )
							.mapToInt( Integer::parseInt )
							.sorted()
							.toArray();
	}

	/**
	 * 表示ﾃﾞｰﾀの HybsSystem.ROW_SEL_KEY に対して、選ばれた 行番号の
	 * 配列を設定します｡
	 * ここで設定した選択配列は、getParameterRows() ﾒｿｯﾄﾞで取得する場合、優先されます｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @param	rowNo	行番号配列(可変長引数)
	 */
	protected void setParameterRows( final int... rowNo ) {
		setRequestAttribute(  HybsSystem.ROW_SEL_KEY , rowNo );
	}

	/**
	 * 指定のｸｯｷｰをｾｯﾄします｡
	 * これは、従来，各Taglibﾊﾟｯｹｰｼﾞで使用していました、ErrorMessage ｵﾌﾞｼﾞｪｸﾄを、
	 * HTMLﾃｰﾌﾞﾙ形式で表示する為に、DBUtilｸﾗｽや、ViewFormｸﾗｽなど、複数のｸﾗｽを
	 * 複雑に組み合わせて使用していましたが、一つの static ﾒｿｯﾄﾞにまとめたものです｡
	 *
	 * @og.rev 3.7.0.3 (2005/03/01) 新規登録
	 *
	 * @param	key		ｸｯｷｰのｷｰ
	 * @param	value	ｸｯｷｰの設定値
	 * @param	maxage	最長存続期間を秒単位で設定 (負の値は Cookie を保存しない、 0 なら Cookie を削除する)
	 */
	protected void setCookie( final String key,final String value,final int maxage ) {
		final HttpServletResponse res = (HttpServletResponse)pageContext.getResponse();
		final Cookie ck = new Cookie( key, value );
		ck.setMaxAge( maxage );													// 有効秒
		res.addCookie( ck );
	}

	/**
	 * 指定のｸｯｷｰを取得します｡
	 * 見つからない場合は、null を返します｡
	 *
	 * @og.rev 3.7.0.3 (2005/03/01) 新規登録
	 *
	 * @param	key	ｸｯｷｰのｷｰ
	 * @return	ｸｯｷｰの設定値
	 */
	protected String getCookie( final String key ) {
		final HttpServletRequest req = (HttpServletRequest)pageContext.getRequest();
		final Cookie[] cks = req.getCookies();

		String val = null;
		for( int i=0; i<cks.length; i++ ) {
			final Cookie ck = cks[i];
			if( ck.getName().equals( key ) ) {
				val = ck.getValue();
				break;
			}
		}
		return val ;
	}

	/**
	 * ﾘｸｴｽﾄ情報の ｼﾝｸﾞﾙｸｫｰﾄ(') 存在ﾁｪｯｸを実施するかどうか[true/false]を設定します
	 *		(初期値:USE_SQL_INJECTION_CHECK[={@og.value SystemData#USE_SQL_INJECTION_CHECK}])｡
	 *
	 * SQLｲﾝｼﾞｪｸｼｮﾝ対策の一つとして、暫定的ではありますが、SQLのﾊﾟﾗﾒｰﾀに
	 * 渡す文字列にｼﾝｸﾞﾙｸｫｰﾄ(') を許さない設定にすれば、ある程度は防止できます｡
	 * 数字ﾀｲﾌﾟの引数には、 or 5=5 などのｼﾝｸﾞﾙｸｫｰﾄを使用しないｺｰﾄﾞを埋めても、
	 * 数字ﾁｪｯｸで検出可能です｡文字ﾀｲﾌﾟの場合は、必ず (')をはずして、
	 * ' or 'A' like 'A のような形式になる為、(')ﾁｪｯｸだけでも有効です｡
	 * (') が含まれていたｴﾗｰにする(true)／かﾉｰﾁｪｯｸか(false)を指定します｡
	 * (初期値:ｼｽﾃﾑ定数のUSE_SQL_INJECTION_CHECK[={@og.value SystemData#USE_SQL_INJECTION_CHECK}])｡
	 *
	 * @og.rev 4.0.0.0 (2005/08/31) 新規追加
	 *
	 * @param	flag	ｼﾝｸﾞﾙｸｫｰﾄﾁｪｯｸ  [true:する/:falseしない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_SQL_INJECTION_CHECK
	 */
	protected void useQuotCheck( final boolean flag ) {
		quotCheck = flag;
	}

	/**
	 * ﾘｸｴｽﾄ情報の HTMLTag開始/終了文字(&gt;&lt;) 存在ﾁｪｯｸを実施するかどうか[true/false]を設定します
	 *    (初期値:USE_XSS_CHECK[={@og.value SystemData#USE_XSS_CHECK}])｡
	 *
	 * ｸﾛｽｻｲﾄｽｸﾘﾌﾟﾃｨﾝｸﾞ(XSS)対策の一環としてless/greater than signについてのﾁｪｯｸを行います｡
	 * (&gt;&lt;) が含まれていたｴﾗｰにする(true)／かﾉｰﾁｪｯｸか(false)を指定します｡
	 * 現在の実装としてはﾘｸｴｽﾄﾊﾟﾗﾒｰﾀのみﾁｪｯｸして、attributesに対しては行いません｡
	 * (初期値:ｼｽﾃﾑ定数のUSE_XSS_CHECK[={@og.value SystemData#USE_XSS_CHECK}])｡
	 *
	 * @og.rev 5.0.0.2 (2009/09/15) 新規追加
	 *
	 * @param	flag	XSSﾁｪｯｸ [true:する/false:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_XSS_CHECK
	 */
	protected void useXssCheck( final boolean flag ) {
		xssCheck = flag;
	}

	/**
	 * 日付関係の情報を簡易的に取り出す処理を行います｡
	 *
	 * これは、{&#064;DATE.XXXX AA BB CC} 引数処理をおこなうための、ｻﾎﾟｰﾄﾒｿｯﾄﾞです｡
	 * XXXX は結果のﾌｫｰﾏｯﾄ、AA が基準時刻で省略した場合は、現在時刻が利用されます｡
	 * BB 引数は、日付についての加減算処理を行うためのｺﾏﾝﾄﾞです｡
	 * CC 引数は、BB引数のｺﾏﾝﾄﾞに付属するﾊﾟﾗﾒｰﾀです｡加減算処理の数値を指定できます｡
	 * AA,BB,CC 引数については、先頭に、@ を付ける事で、ﾘｸｴｽﾄ変数が使用できます｡
	 *
	 * 引数は、"XXXX AA BB CC" という状態で受け取ります｡(DATE. ははずした形)
	 * 第一引数(key) "XXXX" は、日付処理を行うﾌｫｰﾏｯﾄの予約語になっています｡
	 * ･Y4     ：4文字の年ﾃﾞｰﾀ(yyyy)を扱います｡
	 * ･YMD    ：8文字の4-2-2年月日ﾃﾞｰﾀ(yyyyMMdd)を扱います｡
	 * ･Y2MD   ：6文字の2-2-2年月日ﾃﾞｰﾀ(yyMMdd)を扱います｡
	 * ･YM     ：6文字の4-2年月ﾃﾞｰﾀ(yyyyMM)を扱います｡
	 * ･HMS    ：6文字の2-2-2時分秒ﾃﾞｰﾀ(HHmmss)を扱います｡
	 * ･HM     ：4文字の2-2時分ﾃﾞｰﾀ(HHmm)を扱います｡6.7.4.1 (2017/02/17)
	 * ･YMDHMS ：1４文字の4-2-2-2-2-2年月日時分秒ﾃﾞｰﾀ(yyyyMMddHHmmss)を扱います｡
	 * ･EEE    ：曜日をﾃﾞﾌｫﾙﾄﾛｹｰﾙで表示します｡
	 *
	 * F付きは、ﾌｫｰﾏｯﾄされた日付を返します｡
	 * ･YMDF   ：10文字の日付表現(yyyy/MM/dd)を扱います｡
	 * ･Y2MDF  ：8文字の日付表現(yy/MM/dd)を扱います｡
	 * ･YMF    ：7文字の日付表現(yyyy/MM)を扱います｡
	 * ･HMSF   ：8文字の時刻表現(HH:mm:ss)を扱います｡
	 * ･HMF    ：5文字の時刻表現(HH:mm)を扱います｡6.7.4.1 (2017/02/17)
	 * ･YMDHMSF：19文字の日付表現(yyyy/MM/dd HH:mm:ss)を扱います｡
	 * ･MDF    ：5文字の月日表現(MM/dd)を扱います｡
	 * ･MDEF   ：5文字＋曜日の月日表現(MM/dd(EEE))を扱います｡
	 * ･MDHMF  ：11文字の月日時分表現(MM/dd HH:mm)を扱います｡ (7.0.0.1 (2018/10/09) 追加)
	 * ･MD2F   ：漢字の月日表現(MM月dd日)を扱います｡(5.5.5.2 追加)
	 * ･HM2F   ：漢字の時分表現(HH時mm分)を扱います｡(7.0.0.1 (2018/10/09) 追加)
	 * ･MDHM2F ：漢字の月日時分表現(MM月dd日 HH時mm分)を扱います｡(7.0.0.1 (2018/10/09) 追加)
	 * ･GYMDF  ：和暦の年月日表現(GGGGyyyy年MM月dd日)を扱います｡
	 * ･G2YMDF ：和暦の日付表現(Gyyyy/MM/dd)を扱います｡
	 * ･GYMF   ：和暦の年月表現(GGGGyyyy年MM月)を扱います｡
	 * ･GYF    ：和暦の年表現(GGGGyyyy)を扱います｡
	 *
	 * ･DIFF   ：日付の差分を求めます｡(7.0.1.1 (2018/10/22) 追加)
	 *               AA - BB を求め、CCの数値で単位を指定します｡
	 *
	 * なお、上記以外のﾌｫｰﾏｯﾄを指定する場合は、XXXX部分に直接記述できます｡(5.5.5.2 追加)
	 * ただし、ｽﾍﾟｰｽで分解するため、ﾌｫｰﾏｯﾄにｽﾍﾟｰｽを含む場合は、ﾀﾞﾌﾞﾙｺｰﾃｰｼｮﾝ等で
	 * くくる等の処理が必要です｡基本的には、自由ﾌｫｰﾏｯﾄは、ｴﾗｰﾁｪｯｸがない為、使わないでください｡
	 *
	 * 第二引数 AA は、基準となる日付を、yyyyMMdd形式で指定します｡nullの場合は、現在日時を使用します｡
	 * 指定できる日付は、yyyyMMdd形式を推奨しますが、'/' , '-' , ' ' , ':' を削除して使います｡
	 * 6桁の場合は、yyyyMM + 01 とし、8ｹﾀの場合は、yyyyMMdd とし、14ｹﾀ以上の場合は、前半14文字を
	 * yyyyMMddHHmmss として処理します｡それ以外の桁数の場合は、ｴﾗｰになります｡
	 * たとえば、"2012/09/05 16:52:36" のようなﾌｫｰﾏｯﾄﾃﾞｰﾀの場合、'/' , '-' , ' ' , ':' を削除して
	 * "20120905165236" に変換後、日付ｵﾌﾞｼﾞｪｸﾄに変換されます｡
	 *
	 * AA には、数字で始まる(20050701など)実日付と&#064;で始まるﾊﾟﾗﾒｰﾀが使用できます｡
	 * &#064;AA と記述することで、{&#064;AA}で指定する場合と同様のﾘｸｴｽﾄﾊﾟﾗﾒｰﾀが使用できます｡
	 * このﾊﾟﾗﾒｰﾀの値の解析結果が、null の場合は、現在時刻が使用されます｡
	 * 数字以外の場合は、省略されたと判断して、ｺﾏﾝﾄﾞ（BB引数）として使用されます｡
	 *
	 * BB 引数は、日付についての加減算処理を行います｡
	 * 省略すると、なにも加減算処理を行いません｡
	 * この引数もﾊﾟﾗﾒｰﾀ(&#064;BB)指定で、ﾘｸｴｽﾄﾊﾟﾗﾒｰﾀが使用できます｡
	 * 加減算処理のﾊﾟﾗﾒｰﾀが使用できるのは、"H" , "D" , "M" の１文字ﾊﾟﾗﾒｰﾀの場合のみです｡
	 * それ以外のｺﾏﾝﾄﾞで、加減算処理する場合は、独立した CC 引数 を使用してください｡
	 * ･SY :当年の最初の日付にｾｯﾄします｡(当年1月1日)｡CC引数は、-N:N年前、0:当年(=SY)、N:N年後 6.9.2.1 (2018/03/12)
	 * ･SD :当月の最初の日付にｾｯﾄします｡(当月1日)｡CC引数は、-N:N月前、0:当月(=SD)、N:N月後、-1:BSD と同じ、1:ASD と同じ
	 * ･SW :日付処理の週初め(月曜日)にｾｯﾄします｡日付は当日より前に移動します｡CC引数は、-N:N週前、0:今週(=SW)、N:N週後
	 * ･SH :指定の最初の時にｾｯﾄします｡(分秒を0000にする)｡CC引数は、時の倍数(4と指定すれば、4時間単位に前に戻る) 6.7.4.1 (2017/02/17)
	 * ･SM :指定の最初の分にｾｯﾄします｡(秒を00にする)｡CC引数は、分の倍数(15と指定すれば、15分単位に前に戻る) 6.7.4.1 (2017/02/17)
	 * ･SS :指定の最初の秒にｾｯﾄします｡CC引数は、秒の倍数(15と指定すれば、15秒単位に前に戻る) 6.7.4.1 (2017/02/17)
	 * ･EY :当年の最後の日付にｾｯﾄします｡(当年年末)｡CC引数は、-N:N年前、0:当年(=EY)、N:N年後 6.9.2.1 (2018/03/12)
	 * ･ED :当月の最後の日付にｾｯﾄします｡(当月月末)｡CC引数は、-N:N月前、0:当月(=ED)、N:N月後、-1:BED と同じ、1:AED と同じ
	 * ･EW :日付処理の週末(日曜日)にｾｯﾄします｡日付は当日より後ろに移動します｡CC引数は、-N:N週前、0:今週(=EW)、N:N週後
	 * ･EH :指定の次の時にｾｯﾄします｡(分秒を0000にした次の時)｡CC引数は、時の倍数(4と指定すれば、4時間単位に前に進む) 6.7.4.1 (2017/02/17)
	 * ･EM :指定の次の分にｾｯﾄします｡(秒を00にした次の分)｡CC引数は、分の倍数(15と指定すれば、15分単位に前に進む) 6.7.4.1 (2017/02/17)
	 * ･ES :指定の次の秒にｾｯﾄします｡CC引数は、秒の倍数(15と指定すれば、15秒単位に前に進む) 6.7.4.1 (2017/02/17)
	 * ･M1 ～ MXXX :月を指定の分だけ進めます｡M1なら翌月、M6 なら半年後
	 * ･D1 ～ DXXX :日を指定の分だけ進めます｡D1なら翌日、D200 なら200日後
	 * ･H1 ～ HXXX :時を指定の分だけ進めます｡H1なら1時間後、H24 なら24時間後(5.5.5.6 (2012/08/31) 追加)
	 * ･MI  :分を指定の分だけ進めます｡第四引数(intC) で、時間を指定します｡(6.8.4.1 (2017/12/18) 追加)
	 * ･YMD :CC 引数のYMD表記の日付を加減算します｡6.8.4.1 (2017/12/18) 追加
	 * ･HM  :CC 引数のHM表記の時刻を加減算します｡6.8.4.1 (2017/12/18) 追加
	 * ･NO  :AA 引数がnullの場合、現在時刻ではなく空文字列にします｡  7.0.1.3 (2018/11/12) 追加
	 * ･(有閑)BSD :先月の最初の日付にｾｯﾄします｡(先月1日)(5.5.5.2 追加)｡SD -1 と同等
	 * ･(有閑)BED :先月の最後の日付にｾｯﾄします｡(先月月末)(5.5.5.2 追加)｡ED -1 と同等
	 * ･(有閑)ASD :翌月の最初の日付にｾｯﾄします｡(翌月1日)(5.5.5.2 追加)｡SD 1  と同等
	 * ･(有閑)AED :翌月の最後の日付にｾｯﾄします｡(翌月月末)(5.5.5.2 追加)｡ED 1  と同等
	 *
	 * 7.0.1.1 (2018/10/22)
	 *   DATE.DIFF の場合、BB 引数は、日付ﾃﾞｰﾀになります｡AA-BB の関係です｡
	 *
	 * CC 引数は、特別な処理で、BB 引数に対して、加算、減算のための数字を指定できます｡(5.7.4.1 (2014/03/14) 追加)
	 * 従来は、BB 引数が、"H" , "D" , "M" の 1文字ﾊﾟﾗﾒｰﾀの場合のみ利用可能でした｡
	 * これは、"H15" と指定するのと、"H" "15" と指定するのと同じ意味になります｡
	 * 異なるのは、CC 引数も、(&#064;CC)指定で、ﾘｸｴｽﾄﾊﾟﾗﾒｰﾀが使用できます｡
	 * 従来は、文字列として結合された状態でしか、BB 引数を渡せませんでしたが、この、CC 引数の
	 * 追加で、日付の加減算を、ﾊﾟﾗﾒｰﾀ指定できるようになります｡
	 * 数字以外の文字が指定されたり、ﾊﾟﾗﾒｰﾀの解析結果が NULL の場合には、BB引数自体も無視されます｡
	 * 注意点は、各 BB 引数に応じて、数字の意味が異なるという事です｡
	 *
	 * HXXX,DXXX,MXXX 形式に、CC 引数を付けた場合は、XXX にさらに加算されます｡
	 * prmB に、数字を使用した場合、(ｺﾏﾝﾄﾞでない場合)にも、CC 引数は、加算されます｡
	 *
	 * 7.0.1.1 (2018/10/22)
	 *   DATE.DIFF の場合、CC 引数は、差分の単位を指定するｷｰﾜｰﾄﾞになります｡AA-BB の結果を、
	 *   1:年 2:月 3:日 4:時 5:分 6:秒 に換算 して返します｡端数は切り捨てで整数で返します｡
	 *
	 * @og.rev 3.8.0.2 (2005/07/11) 新規追加
	 * @og.rev 5.4.0.1 (2011/11/01) 日付処理の機能追加(BB 引数に、ﾘｸｴｽﾄﾊﾟﾗﾒｰﾀ対応)
	 * @og.rev 5.5.0.2 (2012/03/09) 和暦対応
	 * @og.rev 5.5.5.2 (2012/08/18) XXXXﾌｫｰﾏｯﾄ追加、自由ﾌｫｰﾏｯﾄ対応、BB引数追加、/,-削除機能追加、SM,EM廃止
	 * @og.rev 5.5.5.6 (2012/08/31) H1 ～ HXXX 追加｡時間の加算を指定できる｡
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します｡
	 * @og.rev 5.5.8.2 (2012/11/09) prmA の判定に、null と ｾﾞﾛ文字列を判定する｡
	 * @og.rev 5.6.0.1 (2013/01/11) 5.5.7.2でyyyyMMddしか取っていないため、HHmmssを追加します
	 * @og.rev 5.7.4.1 (2014/03/14) AA 引数の@解析後のｺﾏﾝﾄﾞ判定方法を、８ｹﾀ以下から先頭が数字以外に変更します｡
	 * @og.rev 5.7.4.1 (2014/03/14) CC 引数を、"H" , "D" , "M" 以外でも使用できるように拡張します｡
	 *
	 * @param	value	ﾊﾟﾗﾒｰﾀ
	 * @return	ﾒｯｾｰｼﾞ情報
	 * @see		org.opengion.fukurou.util.HybsDateUtil#getDateFormat( String,String,String,int )
	 */
	protected String getDateFormat( final String value ) {
		// {@DATE.XXXX AA BB CC} を分割
		final String[] vals = StringUtil.csv2Array( value,' ' );		// ダブルクオート内は保持される。

		final String key = vals[0] ;

		// 5.7.4.1 (2014/03/14) 初期化時に、vals を設定しておきます。
		String prmA = vals.length >= 2 ? vals[1] : null ;
		String prmB = vals.length >= 3 ? vals[2] : null ;
		String prmC = vals.length >= 4 ? vals[vals.length-1] : null ;		// 互換性。最後の値が、CC引数

		if( StringUtil.startsChar( prmA , '@' ) ) {							// 6.2.0.0 (2015/02/27) １文字 String.startsWith
			prmA = getRequestValue( prmA.substring(1) );
		}

		if( StringUtil.startsChar( prmB , '@' ) ) {							// 6.2.0.0 (2015/02/27) １文字 String.startsWith
			prmB = getRequestValue( prmB.substring(1) );
		}

		if( StringUtil.startsChar( prmC , '@' ) ) {							// 6.2.0.0 (2015/02/27) １文字 String.startsWith
			prmC = getRequestValue( prmC.substring(1) );
		}

		// 5.7.4.1 (2014/03/14) AA 引数の@解析後のｺﾏﾝﾄﾞ判定方法を、８ｹﾀ以下から先頭が数字以外に変更します｡
		if( prmA != null && prmA.length() > 0 ) {
			final char chA = prmA.charAt(0);
			// 先頭が、数字以外の場合は、ｺﾏﾝﾄﾞなので、一つずつずらす｡
			if( chA < '0' || chA > '9' ) {
				prmC = prmB;
				prmB = prmA;
				prmA = null;
			}
		}

		// 5.7.4.1 (2014/03/14) CC 引数を、"H" , "D" , "M" 以外でも使用できるように拡張します。
		int intC = 0;
		if( prmC != null && prmC.length() > 0 ) {
			try {
				intC = Integer.parseInt( prmC );
			}
			catch( final NumberFormatException ex ) {
				final String errMsg = "CC引数が数字ではありません。value=[" + value + "]"
								+ ex.getMessage() ;
				System.err.println( errMsg );
			}
		}

		// prmA が null か、isEmpty() の場合は、現在時刻が使用される。
		return HybsDateUtil.getDateFormat( key,prmA,prmB,intC );				// 5.7.4.1 (2014/03/14) CC 引数を拡張します。
	}

	/**
	 * debug や ｴﾗｰ時に参考にする、ｸｲｯｸﾘﾌｧﾚﾝｽへのﾘﾝｸを作成します｡
	 * ﾘﾝｸを行うﾀｸﾞの名称は、getTagName() ﾒｿｯﾄﾞより取得します｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 * @og.rev 4.2.1.0 (2008/04/11) URLを相対ﾊﾟｽに変更
	 * @og.rev 6.2.1.0 (2015/03/13) ｸｲｯｸﾘﾌｧﾚﾝｽへのﾘﾝｸが相対ﾊﾟｽでは、common/gamen 以下から使えない｡
	 * @og.rev 6.3.1.1 (2015/07/10) ｸｲｯｸﾘﾌｧﾚﾝｽは、documents.html ではなく、quickReference.html です｡
	 * @og.rev 7.0.1.0 (2018/10/15) XHTML → HTML5 対応(空要素の、"／＞" 止めを、"＞" に変更します)｡
	 *
	 * @return	ｸｲｯｸﾘﾌｧﾚﾝｽへのﾘﾝｸ
	 * @og.rtnNotNull
	 * @see		#getTagName()
	 */
	protected String getDocumentLink() {
		final String name = getTagName();

	//	try {
	//		Field fld = getClass().getDeclaredField( "VERSION" ) ;
	//		version = (String)fld.get( null );
	//	}
	//	catch( final Exception ex ) {
	//		version = ex.toString();
	//	}

		// org.opengion.hayabusa.taglib.AbcdTag というｸﾗｽ名より、abcd を取り出す｡
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
//			.append( "<br /><a href=\"" )
			.append( "<br><a href=\"" )											// 7.0.1.0 (2018/10/15)
			.append( HybsSystem.sys( "JSP" ) )									// 6.2.1.0 (2015/03/13) 追加
			.append( "/common/quickReference.html#VAL_" )						// 6.3.1.1 (2015/07/10) ﾘﾝｸ先間違い
			.append( name )
			.append( "\" target=\"_brank\" >ｸｲｯｸﾘﾌｧﾚﾝｽ[" )
			.append( name )
			.append( "] " )
			.append( BuildNumber.ENGINE_INFO )									// 6.3.1.1 (2015/07/10) ENGINE_INFO の追加
		//	.append( version )
			.append( "</a>" ).append( BR );

		return buf.toString();
	}

	/**
	 * ﾀｸﾞの名称を返します｡
	 * これは、debug や ｴﾗｰ時に参考にする、ｸｲｯｸﾘﾌｧﾚﾝｽへのﾘﾝｸを作成する場合に
	 * 使用します｡
	 * 通常は、org.opengion.hayabusa.taglib.AbcdTag という自分自身のｸﾗｽ名より、
	 * abcd の部分を取り出し、返します｡
	 * ｸﾗｽ名とﾀｸﾞ名が、上記変換ﾙｰﾙと異なる場合は、このﾒｿｯﾄﾞを
	 * 使用して、直接 abcd の部分に相当する文字列を返すようにしてください｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @return	ﾀｸﾞの名称
	 * @og.rtnNotNull
	 * @see	#getDocumentLink()
	 */
	protected String getTagName() {
		final String name = getClass().getName();
		final int adrs = name.lastIndexOf('.');

		// org.opengion.hayabusa.taglib.AbcdTag というクラス名より、abcd を取り出す。
		return name.substring( adrs+1,adrs+2 ).toLowerCase(Locale.JAPAN)
				+ name.substring( adrs+2,name.length()-3 ) ;
	}

	/**
	 * ﾘｸｴｽﾄに対して、画面遷移なしﾓｰﾄﾞを有効にします[true/false]｡
	 * この情報は画面IDをｷｰにｾｯｼｮﾝに保存されるため、
	 * 各ﾀｸﾞで共有することができます｡
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 新規作成
	 * @og.rev 4.3.8.0 (2009/08/01) ﾘｸｴｽﾄからｾｯｼｮﾝに変更(名称も変更)
	 * @og.rev 5.1.3.0 (2010/02/01) noTransition、ajaxSubmitのｺﾝﾄﾛｰﾙは、requestで行う｡
	 *
	 * @param	flag	画面遷移なしﾓｰﾄﾞ [true:有効/false:無効]
	 */
	protected void setNoTransitionRequest( final boolean flag ) {
		setRequestAttribute( HybsSystem.NO_TRANSITION_MODE_KEY, String.valueOf( flag ));
	}

	/**
	 * ﾘｸｴｽﾄで画面遷移なしﾓｰﾄﾞが有効になっているかを返します｡
	 * この情報はｾｯｼｮﾝから画面IDをｷｰに取得します｡
	 * ｾｯｼｮﾝに情報が設定されていない(#setNoTransition()が呼ばれていない)場合は、
	 * falseを返します｡
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 新規作成
	 * @og.rev 4.3.8.0 (2009/08/01) ﾘｸｴｽﾄからｾｯｼｮﾝに変更
	 * @og.rev 5.1.3.0 (2010/02/01) noTransition、ajaxSubmitのｺﾝﾄﾛｰﾙは、requestで行う｡
	 *
	 * @return	画面遷移なしﾓｰﾄﾞが有効
	 */
	protected boolean isNoTransitionRequest() {
		return nval( (String)getRequestAttribute( HybsSystem.NO_TRANSITION_MODE_KEY ), false );
	}

	/**
	 * ﾘｸｴｽﾄに対して、AjaxSubmitﾓｰﾄﾞを使用するかどうか指定します[true/false]｡
	 * この情報は画面IDをｷｰにｾｯｼｮﾝに保存されるため、
	 * 各ﾀｸﾞで共有することができます｡
	 *
	 * @og.rev 4.3.8.0 (2009/08/01) 新規作成
	 * @og.rev 5.1.3.0 (2010/02/01) noTransition、ajaxSubmitのｺﾝﾄﾛｰﾙは、requestで行う｡
	 *
	 * @param	flag	AjaxSubmitﾓｰﾄﾞ [true:使用する/false:使用しない]
	 */
	protected void setAjaxSubmitRequest( final boolean flag ) {
		setRequestAttribute( HybsSystem.USE_AJAX_SUBMIT_KEY, String.valueOf( flag ));
	}

	/**
	 * ﾘｸｴｽﾄでAjaxSubmitﾓｰﾄﾞが有効になっているかを返します｡
	 * この情報はｾｯｼｮﾝから画面IDをｷｰに取得します｡
	 * ｾｯｼｮﾝに情報が設定されていない(#setNoTransition()が呼ばれていない)場合は、
	 * falseを返します｡
	 *
	 * @og.rev 4.3.8.0 (2009/08/01) 新規作成
	 * @og.rev 5.1.3.0 (2010/02/01) noTransition、ajaxSubmitのｺﾝﾄﾛｰﾙは、requestで行う｡
	 *
	 * @return	AjaxSubmitﾓｰﾄﾞが有効
	 */
	protected boolean isAjaxSubmitRequest() {
		return nval( (String)getRequestAttribute( HybsSystem.USE_AJAX_SUBMIT_KEY ), false );
	}

	/**
	 * ｼﾘｱﾗｲｽﾞ用のｶｽﾀﾑｼﾘｱﾗｲｽﾞ読み込みﾒｿｯﾄﾞ
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なﾌｨｰﾙﾄﾞのみ設定します｡
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @serialData 一部のｵﾌﾞｼﾞｪｸﾄは、ｼﾘｱﾗｲｽﾞされません｡
	 *
	 * @param	strm	ObjectInputStreamｵﾌﾞｼﾞｪｸﾄ
	 * @see #release2()
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
		attri = new Attributes();
	}

	/**
	 * ｱｸｾｽﾛｸﾞ取得の為,ApplicationInfoｵﾌﾞｼﾞｪｸﾄを返します｡
	 * 見つからない場合は、null が返ります｡(暫定対応)
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) 新規追加
	 *
	 * @return	ｱｸｾｽﾛｸﾞ取得の為の管理ｵﾌﾞｼﾞｪｸﾄ
	 */
	protected ApplicationInfo getApplicationInfo() {
		final String gamenId = getGUIInfoAttri( "KEY" );
		final String jspId   = (String)getSessionAttribute( "JSPID" );

		return getUser().getApplicationInfo( gamenId,jspId );
	}

	/**
	 * ｲﾍﾞﾝﾄｶﾗﾑの実行に必要なｶﾗﾑ情報をﾏｯﾌﾟに登録します｡
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 動的ﾌﾟﾙﾀﾞｳﾝ実装見直し
	 * @og.rev 6.3.3.0 (2015/07/25) eventValue 追加
	 *
	 * @param	col	DBｶﾗﾑ
	 */
	protected void addEventColumn( final DBColumn col ) {
		addEventColumn( col.getName(), col.getEventColumn(), col.getEventValue(), col.getEventURL()
						, col.getRenderer(), col.getEditor(), col.getRawRendParam(), col.getRawEditParam() );
	}

	/**
	 * ｲﾍﾞﾝﾄｶﾗﾑの実行に必要なｶﾗﾑ情報をﾏｯﾌﾟに登録します｡
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 動的ﾌﾟﾙﾀﾞｳﾝ実装見直し
	 * @og.rev 6.3.3.0 (2015/07/25) eventValue 追加
	 *
	 * @param	name		ｶﾗﾑ名
	 * @param	evCol		ｲﾍﾞﾝﾄｶﾗﾑ名
	 * @param	evVal		子ｶﾗﾑの値SQL
	 * @param	url			ｲﾍﾞﾝﾄURL
	 * @param	renderer	ｶﾗﾑのﾚﾝﾃﾞﾗｰ
	 * @param	editor		ｶﾗﾑのｴﾃﾞｨﾀｰ
	 * @param	rendParam	ｶﾗﾑの表示ﾊﾟﾗﾒｰﾀｰ
	 * @param	editParam	ｶﾗﾑの編集ﾊﾟﾗﾒｰﾀｰ
	 */
	protected void addEventColumn( final String name, final String evCol , final String evVal, final String url,
						final String renderer, final String editor, final String rendParam, final String editParam ) {
		if( evCol == null || evCol.isEmpty() ){ return; }

		String key = (String)getRequestAttribute( HybsSystem.USE_EVENT_COLUMN_KEY );
		if( key == null || key.isEmpty() ) {
			key = HybsSystem.USE_EVENT_COLUMN_KEY + System.currentTimeMillis();
			setRequestAttribute( HybsSystem.USE_EVENT_COLUMN_KEY, key );
		}

		@SuppressWarnings(value={"unchecked"})
		Map<String, DBEventColumn> evColMap = (Map<String, DBEventColumn>)( getSessionAttribute( key ) );
		if( evColMap == null ){
			evColMap = new HashMap<>();
		}
		if( evColMap.get( name ) == null ) {
			evColMap.put( name, new DBEventColumn( name, evCol, evVal, url, renderer, editor, rendParam, editParam ) );	// 6.3.3.0 (2015/07/25)
		}
		setSessionAttribute( key, evColMap );
	}

	/**
	 * 各ﾃﾞｰﾀﾍﾞｰｽに対応するﾌｧﾝｸｼｮﾝ名を返します｡
	 *
	 * @og.rev 4.3.7.0 (2009/06/01) 新規作成
	 * @og.rev 5.1.4.0 (2010/03/01) ﾃﾞｰﾀﾍﾞｰｽ名 でなく、DBID 名で検索するようにします｡
	 *
	 * @param	key	ﾌｧﾝｸｼｮﾝ名(定義文字)
	 * @return	実ﾌｧﾝｸｼｮﾝ名
	 */
	protected String getDBFunctionName( final String key ) {
		final int idx = key.indexOf( ' ' );
		String func = null;
		String dbid = null;
		if( idx >= 0 ) {
			func = key.substring( 0, idx );
			dbid = key.substring( idx+1, key.length() );
			if( StringUtil.startsChar( dbid , '@' ) ) { dbid = getRequestValue( dbid.substring( 1 ) ); }	// 6.2.0.0 (2015/02/27) １文字 String.startsWith
		}
		else {
			func = key;
		}
		// 5.1.4.0 (2010/03/01) データベース名 でなく、DBID 名で検索する。
		return DBFunctionName.getFunctionName( func, dbid );
	}

	// /**
	// * ﾃﾞｰﾀﾛｰﾙの設定に基づき、ｷｰ(ｶﾗﾑ名)に対応する条件式を返します｡
	// *
	// * @og.rev 4.4.0.0 (2009/08/02) 新規作成
	// * @og.rev 7.4.4.0 (2021/06/30) openGionV8事前準備(DataRole.java廃止)
	// *
	// * @param		key	ｶﾗﾑ名
	// * @return	条件式
	// */
	// 7.4.4.0 (2021/06/30) Delete
	// protected String getDataCondition( final String key ) {
	// 	return getUser().getDataCondition( key );
	// }

	/**
	 * 与えたIPｱﾄﾞﾚｽからﾎｽﾄ名を取得して返します｡
	 * 取得できなかった場合は空文字を返します｡
	 * IPｱﾄﾞﾚｽが解決できない場合や、ｾｷｭﾘﾃｨﾏﾈｰｼﾞｬｰで
	 * 許可されていない場合はｴﾗｰを返します｡
	 *
	 * @og.rev 5.6.6.2 (2013/07/19) 新規作成
	 *
	 * @param	ip	IPｱﾄﾞﾚｽ
	 * @return	ﾎｽﾄ名(取得できない場合は空文字)
	 */
	protected String getHostName( final String ip ) {
		String hostname = "";
		try{
			hostname =  InetAddress.getByName(ip).getHostName();
		}
		catch( final UnknownHostException ex ){
			final String errMsg = "IPｱﾄﾞﾚｽが解決できません。"
				+ ex.getMessage();
			throw new HybsSystemException( errMsg,ex );
		}
		catch( final SecurityException ex ){
			final String errMsg = "この操作はｾｷｭﾘﾃｨ上許可されていません。"
				+ ex.getMessage();
			throw new HybsSystemException( errMsg,ex );
		}
		return hostname;
	}

	/**
	 * 指定のURLの文字列から、最適なURLを作成します｡
	 *
	 * 引数の url が、'/' で始まる場合は、ｺﾝﾃｷｽﾄ以下の相対ﾊﾟｽと判断します｡
	 * つまり、"/gf" + url で表されることになります｡
	 * ただし、経過処置として、先頭が、ｺﾝﾃｷｽﾄを含む文字列の場合は、そのまま返します｡
	 *
	 * それ以外は、そのまま返します｡
	 * ただし、引数が、null やｾﾞﾛ文字列の時は、ｾﾞﾛ文字列を返します｡
	 *
	 * @og.rev 6.3.2.0 (2015/07/10) 指定のURLの文字列から、最適なURLを作成します｡
	 *
	 * @param	url	指定のURLの文字列から、最適なURLを作成します｡
	 * @return	ｺﾝﾃｷｽﾄに応じた URL を返します｡
	 * @og.rtnNotNull
	 */
	protected static String makeUrl( final String url ) {
		if( url == null || url.isEmpty() ) { return ""; }

		if( url.charAt(0) == '/' ) {
			final String CPATH = '/' + HybsSystem.sys( "CONTEXT_NAME" );		// CPATH = "/gf"
			if( !url.startsWith( CPATH ) ) {
				return CPATH + url ;
			}
		}

		return url ;
	}

	/**
	 * "org.opengion.hayabusa.message" の、Locale.JAPANESE ﾘｿｰｽから取得するﾒｯｾｰｼﾞを文字列で返します｡
	 *
	 * id と引数を受け取り、ResourceBundle と、MessageFormat.format で加工した
	 * 文字列を返します｡
	 * baseName は、F_BS_NM で、Locale に、Locale.JAPANESE を指定したﾒｯｾｰｼﾞを作成します｡
	 * hayabusaﾊﾟｯｹｰｼﾞと、pluginﾊﾟｯｹｰｼﾞで主に使われる、ﾒｯｾｰｼﾞの取得方法です｡
	 *
	 * @og.rev 6.4.3.2 (2016/02/19) 新規追加
	 *
	 * @param	id		ﾘｿｰｽのｷｰとなるID｡
	 * @param	args	ﾘｿｰｽを、MessageFormat.format で加工する場合の引数｡
	 * @return	MessageFormat.formatで加工された文字列
	 * @see		MsgUtil#H_BS_NM
	 */
	protected String getMsg( final String id , final Object... args ) {
		return MsgUtil.getMsg( MsgUtil.H_BS_NM , new Locale( getLanguage() ) , id , args );
	}

//	/**
//	 * ﾀｸﾞﾘﾌﾞで発生したThrowableを session から取り出します｡
//	 *
//	 * とりだした Throwable は、remove しています｡
//	 *
//	 * @og.rev 6.8.5.0 (2018/01/09) ﾀｸﾞﾘﾌﾞで発生したｴﾗｰを、session に登録しておきます｡
//	 * @og.rev 6.9.2.1 (2018/03/12) ﾀｸﾞﾘﾌﾞで発生したｴﾗｰを、session に登録する処理を、一旦廃止
//	 *
//	 * @return	元のThrowableｵﾌﾞｼﾞｪｸﾄ
//	 */
//	protected Throwable getCommonTagThrowable() {
//		final Throwable th  = (Throwable)getSessionAttribute( "CommonTagThrowable" );
//
//		removeSessionAttribute( "CommonTagThrowable" );
//
//		return th;
//	}

	/**
	 * &#064;があればﾘｸｴｽﾄから、なければ useReq に応じて、値を返します｡
	 *
	 * &#064;があるかどうか判断して、なければkeyそのもので、getRequestValue した値を返します｡
	 * あれば、まず、ｽﾍﾟｰｽが存在しているかどうか判定して、あれば、&#064;を取り除いて、
	 * getReservedValue を、なければ、 &#064;を取り除いて、getRequestValue した値を返します｡
	 * ｷｰが null の場合は、そのまま、null を返しますが、getRequestValue等した結果が
	 * null かｾﾞﾛ文字列の場合は、defvalを返します｡
	 *
	 * @og.rev 7.2.3.1 (2020/04/17) 新規追加
	 *
	 * @param	key		&#064;付き／なしのﾘｸｴｽﾄ変数のｷｰ
	 * @param	useReq	&#064;が存在しない場合 [true:ﾘｸｴｽﾄから/false:ｷｰを返す]
	 * @return	&#064;があればﾘｸｴｽﾄから、なければ useReq に応じて、値を返す
	 */
	private String getAtmarkVas( final String key , final boolean useReq ) {
		if( key == null ) { return null; }

		final String val ;

		// 先頭が @ の場合は、ﾘｸｴｽﾄ変数のｷｰとして、値を判定
		if( StringUtil.startsChar( key , '@' ) ) {								// １文字 String.startsWith
			if( key.indexOf( ' ' ) > 0 ) {
				val = getReservedValue( key.substring( 1 ) );					// もう一度変数処理
			}
			else {
				val = getRequestValue( key.substring( 1 ) );
			}
		}
		else {
			val = useReq ? getRequestValue( key ) : key ;						// @ がない場合は、useReq で、判定する。
		}

		return val;
	}

	/**
	 * このｵﾌﾞｼﾞｪｸﾄの文字列表現を返します｡
	 * 基本的にﾃﾞﾊﾞｯｸﾞ目的に使用します｡
	 *
	 * @return	このｸﾗｽの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
