/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import java.util.concurrent.ConcurrentMap;							// 6.4.3.3 (2016/03/04)
import java.util.Map;												// 7.4.0.1 (2021/04/16)
import java.util.List;												// 8.2.1.0 (2022/07/15)

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.db.Transaction;
import org.opengion.hayabusa.resource.ResourceManager;
import org.opengion.hayabusa.html.ViewMarker;						// 8.0.0.0 (2021/09/30)

/**
 * TableFilter は､TableFilterTag で使用される DBTableModelの処理用のｲﾝﾀｰﾌｪｰｽです｡
 * 適切な実装(ｻﾌﾞｸﾗｽ)を TableFilterTag に与えることで，DBTableModelの処理を適宜
 * 切り替えることが出来ます｡
 *
 * @og.rev 3.8.5.0 (2006/03/20) 新規作成
 * @og.rev 5.5.2.6 (2012/05/25) ｲﾝﾀｰﾌｪｰｽにgetterﾒｿｯﾄﾞ追加
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public interface TableFilter {
	/**
	 * DBTableModelの処理を実行します｡
	 * 処理自体は, 各ｻﾌﾞｸﾗｽの実装に依存します｡
	 *
	 * @return	DBTableModelｵﾌﾞｼﾞｪｸﾄ
	 */
	DBTableModel execute() ;

	/**
	 * DBTableModel をｾｯﾄします｡
	 *
	 * @param	table DBTableModelｵﾌﾞｼﾞｪｸﾄ
	 */
	void setDBTableModel( DBTableModel table ) ;

	/**
	 * DBTableModel を取得します｡
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) ｲﾝﾀｰﾌｪｰｽにgetterﾒｿｯﾄﾞ追加
	 *
	 * @return	内部のDBTableModel
	 */
	DBTableModel getDBTableModel() ;

	/**
	 * ﾃﾞｰﾀ処理の方法(A:追加 C:更新 D:削除)を指定します｡
	 *
	 * 通常は､DBTableModel に自動設定されている modifyType を元に､ﾃﾞｰﾀ処理方法を
	 * 選別します｡(A:追加 C:更新 D:削除)
	 * この場合､行単位で modifyType の値を取得して判別する必要がありますが､一般には
	 * 処理対象は､全件おなじ modifyType である可能性が高いです｡
	 * また､selectedAll などで強制的に全件処理対象とする場合は､modifyType に値が
	 * 設定さていません｡その様な場合に外部より modifyType を指定します｡
	 * 初期値は､自動判定 です｡
	 *
	 * @param  type ﾃﾞｰﾀ処理の方法(A:追加 C:更新 D:削除)
	 */
	void setModifyType( String type ) ;

	/**
	 * ﾃﾞｰﾀ処理の方法(A:追加 C:更新 D:削除)を取得します｡
	 *
	 * 初期値は､自動判定 です｡
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) ｲﾝﾀｰﾌｪｰｽにgetterﾒｿｯﾄﾞ追加
	 *
	 * @return  ﾃﾞｰﾀ処理の方法(A:追加 C:更新 D:削除)
	 */
	String getModifyType() ;

	/**
	 * ｷｰと値のﾍﾟｱの変数配列を受け取ります｡
	 *
	 * ここでは､この方式以外に､ﾊﾟﾗﾒｰﾀｰMapを受け取る方法もあります｡
	 *
	 * keys,vals とﾊﾟﾗﾒｰﾀｰMapを同時に指定した場合は､両方とも有効です｡
	 * ただし､ｷｰが重複した場合は､不定と考えてください｡
	 *
	 * @param   keys ｷｰ配列
	 * @param   vals 値配列
	 */
	void setKeysVals( String[] keys , String[] vals ) ;

	/**
	 * 選択された行番号の配列をｾｯﾄします｡
	 *
	 * 表示ﾃﾞｰﾀの HybsSystem.ROW_SEL_KEY を元に､選ばれた 行を
	 * 処理の対象とします｡
	 *
	 * @param   rowNo 行番号の配列(可変長引数)
	 */
	void setParameterRows( int... rowNo ) ;

	/**
	 * 選択された行番号の配列を取得します｡
	 *
	 * 表示ﾃﾞｰﾀの HybsSystem.ROW_SEL_KEY を元に､選ばれた 行を
	 * 処理の対象とします｡
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) ｲﾝﾀｰﾌｪｰｽにgetterﾒｿｯﾄﾞ追加
	 *
	 * @return  行番号の配列
	 */
	int[] getParameterRows() ;

	/**
	 * ｱｸｾｽﾛｸﾞ取得の為,Transactionｵﾌﾞｼﾞｪｸﾄを設定します｡
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応(新規追加)
	 *
	 * @param   tran Transactionｵﾌﾞｼﾞｪｸﾄ
	 */
	void setTransaction( Transaction tran ) ;

	/**
	 * ｱｸｾｽﾛｸﾞ取得の為,Transactionｵﾌﾞｼﾞｪｸﾄを取得します｡
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応(新規追加)
	 * @og.rev 5.5.2.6 (2012/05/25) ｲﾝﾀｰﾌｪｰｽにgetterﾒｿｯﾄﾞ追加
	 *
	 * @return	Transactionｵﾌﾞｼﾞｪｸﾄ
	 */
	Transaction getTransaction() ;

	/**
	 * DBIDを指定します｡
	 *
	 * @og.rev 4.2.4.0 (2008/06/23) 新規追加
	 *
	 * @param dbid 接続先ID
	 */
	void setDbid( String dbid ) ;

	/**
	 * DBIDを取得します｡
	 *
	 * @og.rev 4.2.4.0 (2008/06/23) 新規追加
	 * @og.rev 5.5.2.6 (2012/05/25) ｲﾝﾀｰﾌｪｰｽにgetterﾒｿｯﾄﾞ追加
	 *
	 * @return DBID(接続先ID)
	 */
	String getDbid() ;

	/**
	 * ﾎﾞﾃﾞｨｰ部分のSQLを指定します｡
	 *
	 * @og.rev 4.2.4.0 (2008/06/23) 新規追加
	 *
	 * @param sql ﾎﾞﾃﾞｨｰ部分のSQL
	 */
	void setSql( String sql ) ;

	/**
	 * ﾎﾞﾃﾞｨｰ部分のSQLを取得します｡
	 *
	 * @og.rev 4.2.4.0 (2008/06/23) 新規追加
	 * @og.rev 5.5.2.6 (2012/05/25) ｲﾝﾀｰﾌｪｰｽにgetterﾒｿｯﾄﾞ追加
	 *
	 * @return ﾎﾞﾃﾞｨｰ部分のSQL
	 */
	String getSql() ;

	/**
	 * ﾊﾟﾗﾒｰﾀｰMapを指定します｡
	 *
	 * keys,vals と ﾊﾟﾗﾒｰﾀｰMapを同時に指定した場合は､両方とも有効です｡
	 * ただし､ｷｰが重複した場合は､不定と考えてください｡
	 *
	 * @og.rev 5.6.5.2 (2013/06/21) 新規追加
	 * @og.rev 6.4.3.3 (2016/03/04) ConcurrentHashMap を受け取ることを明確にするため､I/FをConcurrentMapに変更します｡
	 *
	 * @param paramMap ﾊﾟﾗﾒｰﾀｰMap
	 */
	void setParamMap( ConcurrentMap<String,String> paramMap ) ;

	/**
	 * ﾘｿｰｽｵﾌﾞｼﾞｪｸﾄを指定します｡
	 *
	 * @og.rev 4.3.7.4 (2009/07/01) 新規追加
	 *
	 * @param resource ﾘｿｰｽｵﾌﾞｼﾞｪｸﾄ
	 */
	void setResource( ResourceManager resource ) ;

	/**
	 * ﾘｿｰｽｵﾌﾞｼﾞｪｸﾄを取得します｡
	 *
	 * @og.rev 4.3.7.4 (2009/07/01) 新規追加
	 * @og.rev 5.5.2.6 (2012/05/25) ｲﾝﾀｰﾌｪｰｽにgetterﾒｿｯﾄﾞ追加
	 *
	 * @return ﾘｿｰｽｵﾌﾞｼﾞｪｸﾄ
	 */
	ResourceManager getResource() ;

	/**
	 * viewMarkerｵﾌﾞｼﾞｪｸﾄを指定します｡
	 *
	 * @og.rev 8.0.0.0 (2021/09/30) viewMarker追加
	 *
	 * @param marker viewMarkerｵﾌﾞｼﾞｪｸﾄ
	 */
	void setViewMarker( final ViewMarker marker );

	/**
	 * viewMarkerｵﾌﾞｼﾞｪｸﾄを取得します｡
	 *
	 * @og.rev 8.0.0.0 (2021/09/30) viewMarker追加
	 *
	 * @return viewMarkerｵﾌﾞｼﾞｪｸﾄ
	 */
	ViewMarker getViewMarker();

	/**
	 * 値を返すためのMapを返します｡
	 *
	 * Mapそのものを返しますので､中身の書き換えは行わないでください｡
	 *
	 * @og.rev 7.4.0.1 (2021/04/16) 値を返すための変数
	 *
	 * @return Mapｵﾌﾞｼﾞｪｸﾄ
	 */
	Map<String,String> getReturnMap() ;

	/**
	 * TableFilter_MAPCLM 専用のMapを管理するListを設定します。
	 *
	 * @og.rev 8.2.1.0 (2022/07/15) TableFilter_MAPCLM 専用のMapを管理するList
	 *
	 * @param	valList Mapｵﾌﾞｼﾞｪｸﾄのﾘｽﾄ
	 */
	void setMapList( final List<Map<String,String>> valList );

	/**
	 * ﾃﾞﾊﾞｯｸﾞ情報を表示するかどうか[true/false]を指定します｡
	 * true でﾃﾞﾊﾞｯｸﾞ情報を表示します｡
	 *
	 * @param   flag 表示するかどうか [true:表示する/false:表示しない]
	 */
	void setDebug( boolean flag ) ;

	/**
	 * ﾃﾞﾊﾞｯｸﾞ情報を表示するかどうか[true/false]を取得します｡
	 * true でﾃﾞﾊﾞｯｸﾞ情報を表示します｡
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) ｲﾝﾀｰﾌｪｰｽにgetterﾒｿｯﾄﾞ追加
	 *
	 * @return  ﾃﾞﾊﾞｯｸﾞ情報 [true:表示する/false:表示しない]
	 */
	boolean isDebug() ;

	/**
	 * ｴﾗｰｺｰﾄﾞ を取得します｡
	 * ｴﾗｰｺｰﾄﾞ は､ErrorMessage ｸﾗｽで規定されているｺｰﾄﾞです｡
	 *
	 * @return   ｴﾗｰｺｰﾄﾞ
	 */
	int getErrorCode() ;

	/**
	 * ｴﾗｰﾒｯｾｰｼﾞｵﾌﾞｼﾞｪｸﾄ を取得します｡
	 *
	 * @return   ｴﾗｰﾒｯｾｰｼﾞｵﾌﾞｼﾞｪｸﾄ
	 */
	ErrorMessage getErrorMessage() ;
}
