/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import static org.opengion.fukurou.util.StringUtil.nval;

import java.util.concurrent.atomic.AtomicInteger;	// 5.5.2.6 (2012/05/25) findbugs対応

/**
 * 表示のON/OFF制御が出来るフィールドセットを作成するタグです。
 *
 * フィールドセットは関連するフォームの部品やラベルをグループ化する要素で、
 * 表示のON/OFF制御が出来ます。
 * BODY 部分にるフォーム部品などのタグを記述すれば、そのままタイトル付きのグループ化
 * された状態を作成できます。
 * useDisplayHide="false" で通常の fieldset と同じ機能になります。通常、useDisplayHide="true"
 * にすることで、タイトル部(一般のlegendタグ)をクリックすると、表示がON/OFFします。
 * useDisplayHide の初期値は、true(表示 ON/OFF機能を使用する)です。
 * 表示機能が使用できる状態(useDisplayHide="true")では、さらに、初期表示を行うかどうかを
 * 指定できます。これは、defaultNone="true" とすると初期表示は "style=display:none;" に
 * 設定され(つまり、表示されない状態)、defaultNone="false" とすると初期表示されます。
 * defaultNone の初期値は、true(表示されない状態)です。
 * このタグには、通常、第一要素としてBODY部に記述する legendタグ は設定不要です。
 * このタグの lbl 属性に、リソース情報を記述することで、直接 legendタグ を
 * 生成しています。
 *
 * @og.formSample
 * ●形式：&lt;og:fieldset
 *                    lbl="…"                      ラベルリソースのキー
 *                  [ useDisplayHide="true" ]       表示 ON/OFF機能を使用する(true)かどうか
 *                  [ useDisplayHide="true" ]       初期値を表示ON(false)にするかOFF(true)にするか
 *         &gt;
 *             &lt;input …" /&gt;
 *             &lt;input …" /&gt;
 *         &lt;/og:fieldset&gt;
 * ●body：あり(EVAL_BODY_INCLUDE:BODYをインクルードし、{&#064;XXXX} は解析しません)
 *
 * ●Tag定義：
 *   &lt;og:fieldset
 *       lbl                【TAG】ラベルリソースのラベルIDを指定します
 *       useDisplayHide     【TAG】表示 ON/OFF機能を使用するかどうか[true/false]を指定します(初期値:true)
 *       defaultNone        【TAG】表示ON/OFF機能を使用する場合の初期値を、隠し(none)にするかどうか[true/false]を指定します(初期値:true)
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null) 5.7.7.2 (2014/06/20)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null) 5.7.7.2 (2014/06/20)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない) 5.7.7.2 (2014/06/20)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない) 5.7.7.2 (2014/06/20)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:fieldset&gt;
 *
 * ●使用例
 *    例１:通常の状態。表示ON/OFF機能を使用し、初期値は表示OFF 状態
 *     &lt;og:fieldset lbl="INSERT_GEA11" &gt;
 *         &lt;og:submit  value="insertGEA11" lbl="COPY" command="COPY" /&gt;
 *         &lt;og:column  name="SYSTEM_ID" useRequestValue="false" must="true" td="no" /&gt;
 *     &lt;/og:fieldset&gt;
 *
 *    例２:通常の状態。表示ON/OFF機能を使用し、初期値は表示ON 状態
 *     &lt;og:fieldset lbl="GEM0001" defaultNone="false" &gt;
 *         &lt;og:column  name="SYSTEM_ID" useRequestValue="false" must="true" td="no" /&gt;
 *     &lt;/og:fieldset&gt;
 *
 * @og.rev 4.0.0.0 (2005/11/30) 新規作成
 * @og.group 画面部品
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class FieldsetTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.0.1.0 (2018/10/15)" ;
	private static final long serialVersionUID = 701020181015L ;

	private static AtomicInteger uniqID = new AtomicInteger(1);		// 5.5.2.6 (2012/05/25) findbugs対応

	private boolean useDisplayHide	= true;		// 表示 ON/OFF機能を使用する(true)かどうか
	private boolean defaultNone		= true;		// 初期値を表示ON(false)にするかOFF(true)にするか

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public FieldsetTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 3.8.5.2 (2006/05/31) 初期値:defaultNone を hidden で出力しておく。
	 * @og.rev 5.5.2.6 (2012/05/25) findbugs対応。staticフィールドへの書き込みに、AtomicInteger を利用します。
	 * @og.rev 5.7.7.2 (2014/06/20) caseKey,caseVal,caseNN,caseNull 属性を追加
	 * @og.rev 7.0.1.0 (2018/10/15) XHTML → HTML5 対応(空要素の、"／＞" 止めを、"＞" に変更します)。
	 *
	 * @return	後続処理の指示( EVAL_BODY_INCLUDE )
	 */
	@Override
	public int doStartTag() {
		// 5.7.7.2 (2014/06/20) caseKey,caseVal,caseNN,caseNull 属性を追加
		if( !useTag() ) { return SKIP_BODY ; }

		final String id = String.valueOf( uniqID.getAndIncrement() );	// 5.5.2.6 (2012/05/25) findbugs対応

		String hideFunc = "";
		String onoffMsg = "";
		String divBody  = "";
		String hiddenVal = "";

		if( useDisplayHide ) {
			hideFunc = " onClick=\"hide( \'FS" + id + "\' );\"" ;
			divBody  = "<div id=\"FS" + id + "A\" style=\"display:" ;
//			hiddenVal = "<input type=\"hidden\" name=\"FS" + id + "C\" value=\"" + defaultNone + "\" />" ;
			hiddenVal = "<input type=\"hidden\" name=\"FS" + id + "C\" value=\"" + defaultNone + "\" >" ;		// 7.0.1.0 (2018/10/15)
			if( defaultNone ) {
				onoffMsg = "<span id=\"FS" + id + "B\" >+ </span>";
				divBody += "none;\" >" ;
			}
			else {
				onoffMsg = "<span id=\"FS" + id + "B\" >- </span>";
				divBody += "inline;\" >" ;
			}
		}

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
			.append( "<fieldset style=\"display:inline;\">" ).append( CR )
			.append( "<legend" ).append( hideFunc ).append( '>' )		// 6.0.2.5 (2014/10/31) char を append する。
			.append( "<strong>" ).append( onoffMsg )
			.append( getLongLabel() )
			.append( "</strong></legend>" ).append( CR )
			.append( divBody )
			.append( CR )
			.append( hiddenVal )
			.append( CR );

		jspPrint( buf.toString() );
		return EVAL_BODY_INCLUDE ;	// Body インクルード( extends TagSupport 時)
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 5.7.7.2 (2014/06/20) caseKey,caseVal,caseNN,caseNull 属性を追加
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		// 5.7.7.2 (2014/06/20) caseKey,caseVal,caseNN,caseNull 属性を追加
		if( useTag() ) {
			if( useDisplayHide ) {
				jspPrint( "</div></fieldset>" + CR );
			}
			else {
				jspPrint( "</fieldset>" + CR );
			}
		}

		return EVAL_PAGE ;		// ページの残りを評価する。
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		useDisplayHide	= true;
		defaultNone		= true;
	}

	/**
	 * 【TAG】表示 ON/OFF機能を使用するかどうか[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * 表示 ON/OFF機能を使用する場合は、true / 通常の fieldset を使用する場合は、
	 * false を指定します。
	 * 初期値は、true(ON/OFF機能を使用する)です。
	 *
	 * @param	flag 表示ON/OFF機能 [true:使用する/false:使用しない]
	 * @see		#setDefaultNone( String )
	 */
	public void setUseDisplayHide( final String flag ) {
		useDisplayHide = nval( getRequestParameter( flag ),useDisplayHide ) ;
	}

	/**
	 * 【TAG】表示ON/OFF機能を使用する場合の初期値を、隠し(none)にするかどうか[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * 表示 ON/OFF機能を使用する場合にのみ設定値は有効に機能します。
	 * 隠し(none)にする場合は、true を、表示にする場合は、false をセットします。
	 * 初期値は、true(隠し(none)にする)です。
	 *
	 * @param	flag 初期隠し [true:隠し(none)にする/false:表示にする]
	 * @see		#setUseDisplayHide( String )
	 */
	public void setDefaultNone( final String flag ) {
		defaultNone = nval( getRequestParameter( flag ),defaultNone ) ;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION						)
				.println( "key"				,getLabelInterface().getKey()	)
				.println( "msglbl"			,getMsglbl()					)
				.println( "useDisplayHide"	,useDisplayHide					)
				.println( "defaultNone"		,defaultNone					)
				.println( "Other..."		,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
