/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.db.ConnectionFactory;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.hayabusa.resource.UserInfo;
import org.opengion.hayabusa.resource.ResourceFactory;
import org.opengion.hayabusa.resource.CalendarFactory;
import org.opengion.hayabusa.db.SelectionFactory;

import static org.opengion.fukurou.util.StringUtil.nval ;
import static org.opengion.fukurou.system.HybsConst.BR;		// 6.1.0.0 (2014/12/26) refactoring

/**
 * キャッシュしている各種データを初期化(クリア)します。
 *
 * データベースリソースを使用している場合は、キャッシュをクリアすることで、再度
 * 読み込みなおしますので、データベースの値を変更した後には、クリアが必要です。
 *
 * クリアする対象は、connection,resource,gui,query,calendar があります。
 * all という キーワードは、下記のすべてのリソースキャッシュをクリアします。
 * <table class="plain">
 *   <caption>初期化(クリア)対象のキーワード</caption>
 *   <tr><th>対象        </th><th>説明</th></tr>
 *   <tr><td>connection  </td><td>コネクションプーリングのコネクションを開放します。</td></tr>
 *   <tr><td>resource    </td><td>ユーザー、システムを除くリソースキャッシュをクリアします。</td></tr>
 *   <tr><td>gui         </td><td>GUI情報のキャッシュをクリアします。</td></tr>
 *   <tr><td>calendar    </td><td>事業所カレンダを含むカレンダオブジェクトキャッシュをクリアします。</td></tr>
 * </table>
 *
 * @og.formSample
 * ●形式：&lt;og:engine connectionClear="[true|false]" ･･･ /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:engine
 *       allClear           【TAG】すべてのキャッシュデータをクリアします(初期値:false)
 *       connectionClear    【TAG】データベースコネクションプールのコネクションを削除(開放)します(初期値:false)
 *       resourceClear      【TAG】リソースデータをクリアします(初期値:false)
 *       guiClear           【TAG】GUI情報のキャッシュをクリアします(初期値:false)
 *       userClear          【TAG】User情報のパラメータデータをクリアします(初期値:false)
 *       calendarClear      【TAG】カレンダオブジェクトのキャッシュをクリアします(初期値:false)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *     &lt;og:engine
 *         connectionClear = "true"  初期値:false
 *         resourceClear   = "true"  初期値:false
 *         guiClear        = "true"  初期値:false
 *         userClear       = "true"  初期値:false
 *         calendarClear   = "true"  初期値:false
 *     /&gt;
 *
 *     &lt;og:engine
 *         allClear = "true"
 *     /&gt;
 *
 * @og.rev 3.6.0.0 (2004/09/17) 新規作成
 * @og.group ログイン制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class EngineTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;
	private static final long serialVersionUID = 642020160129L ;

	private static final Object LOCK = new Object();		// 6.4.1.1 (2016/01/16) lock → LOCK  refactoring

	private boolean allClear		;
	private boolean connectionClear	;
	private boolean resourceClear	;
	private boolean guiClear		;
	private boolean userClear		;
	private boolean calendarClear	;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public EngineTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

		synchronized( LOCK ) {
			if( connectionClear ) {
				ConnectionFactory.realClose();
				buf.append( "Connection Factory Real Closed" ).append( BR );
				System.out.println( "Connection Factory Real Closed" );
			}

			if( calendarClear ) {
				CalendarFactory.clear();		// 3.6.0.0 (2004/09/17)
				buf.append( "Calendar Factory Cleared" ).append( BR );
				System.out.println( "Calendar Factory Cleared" );
			}

			if( guiClear ) {
				ResourceFactory.guiClear();
				final UserInfo userInfo = getUser();
				userInfo.clear();
				getResource().makeGUIInfos( userInfo );
				buf.append( "GUIInfo Data Cleared" ).append( BR );
				System.out.println( "GUIInfo Data Cleared" );
			}

			if( userClear ) {
				final UserInfo userInfo = getUser();
				userInfo.clear();
				buf.append( "UserInfo Parameter Data Cleared" ).append( BR );
				System.out.println( "UserInfo Parameter Data Cleared" );
			}

			if( resourceClear ) {
				ResourceFactory.clear();
				SelectionFactory.clear();				// 4.0.0 (2006/11/17)
				buf.append( "Resource Factory Cleared" ).append( BR );
				System.out.println( "Resource Factory Cleared" );
			}
		}

		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		allClear		= false;
		connectionClear	= false;
		resourceClear	= false;
		guiClear		= false;
		userClear		= false;
		calendarClear	= false;
	}

	/**
	 * 【TAG】すべてのキャッシュデータをクリアします(初期値:false)。
	 *
	 * @og.tag
	 * connectionClear,resourceClear,guiClear,calendarClearの
	 * すべてを、true に設定することと同じです。
	 *
	 * @param   flag キャッシュクリア [true:すべて/false:個別]
	 */
	public void setAllClear( final String flag ) {
		allClear = nval( getRequestParameter( flag ),allClear );

		connectionClear	= connectionClear	|| allClear ;
		resourceClear	= resourceClear		|| allClear ;
		guiClear		= guiClear			|| allClear ;
		userClear		= userClear			|| allClear ;
		calendarClear	= calendarClear		|| allClear ;
	}

	/**
	 * 【TAG】データベースコネクションプールのコネクションを削除(開放)します(初期値:false)。
	 *
	 * @og.tag データベース操作がおかしい場合は、クリアしてみてください。
	 *
	 * @param   flag コネクション削除可否 [true:する/false:しない]
	 */
	public void setConnectionClear( final String flag ) {
		connectionClear = nval( getRequestParameter( flag ),connectionClear );
	}

	/**
	 * 【TAG】リソースデータをクリアします(初期値:false)。
	 *
	 * @og.tag
	 * ユーザーリソースとシステムリソースを除くリソースキャッシュ
	 * データをクリアします。
	 *
	 * @param   flag リソースデータのクリア可否 [true:する/false:しない]
	 */
	public void setResourceClear( final String flag ) {
		resourceClear = nval( getRequestParameter( flag ),resourceClear );
	}

	/**
	 * 【TAG】User情報のパラメータデータをクリアします(初期値:false)。
	 *
	 * @og.tag
	 * User情報のパラメータデータに追加/更新/削除が発生したときは、
	 * キャッシュをクリアする必要があります。
	 * User情報のパラメータデータは、{&#064;USER.XXXX}でアクセスできる値ではなく、
	 * システムパラメータ {&#064;SYS.XXXX}としてアクセスできるユーザー個別情報のことです。
	 *
	 * @param   flag パラメータデータのクリア可否 [true:する/false:しない]
	 */
	public void setUserClear( final String flag ) {
		userClear = nval( getRequestParameter( flag ),userClear );
	}

	/**
	 * 【TAG】GUI情報のキャッシュをクリアします(初期値:false)。
	 *
	 * @og.tag
	 * GUI情報の追加/更新/削除が発生したときは、キャッシュをクリアする必要があります。
	 *
	 * @param   flag GUI情報キャッシュのクリア可否 [true:する/false:しない]
	 */
	public void setGuiClear( final String flag ) {
		guiClear = nval( getRequestParameter( flag ),guiClear );
	}

	/**
	 * 【TAG】カレンダオブジェクトのキャッシュをクリアします(初期値:false)。
	 *
	 * @og.tag
	 * カレンダオブジェクトは、事業所カレンダ(例：GE13)毎にオブジェクトを
	 * 作成します。キャッシュに持っているため、データベースを更新するたびに、
	 * キャッシュをクリアしなおす必要があります。
	 *
	 * @param   flag カレンダのキャッシュのクリア可否 [true:する/false:しない]
	 */
	public void setCalendarClear( final String flag ) {
		calendarClear = nval( getRequestParameter( flag ),calendarClear );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "allClear"		,allClear		)
				.println( "connectionClear"	,connectionClear)
				.println( "resourceClear"	,resourceClear	)
				.println( "guiClear"		,guiClear		)
				.println( "userClear"		,userClear		)
				.println( "calendarClear"	,calendarClear	)
				.println( "Other..."		,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
