/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import java.io.File;															// 8.1.2.0 (2022/03/10)
import java.io.IOException;
import java.util.List;
import java.util.ArrayList;
import java.util.Locale;														// 6.0.2.4 (2014/10/17)

import jakarta.servlet.http.HttpServletRequest;

// import org.opengion.fukurou.model.FileOperation;								// 8.0.1.0 (2021/10/29)
import org.opengion.fukurou.security.HybsCryptography;							// 8.1.2.0 (2022/03/10)
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ToString;										// 6.1.1.0 (2015/01/17)
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.hayabusa.servlet.MultipartRequest;
import org.opengion.hayabusa.servlet.UploadedFile;

import static org.opengion.fukurou.util.StringUtil.nval;

/**
 * ｸﾗｲｱﾝﾄのﾌｧｲﾙをｻｰﾊﾞｰにｱｯﾌﾟﾛｰﾄﾞするﾀｸﾞです。
 *
 * ｱｯﾌﾟﾛｰﾄﾞ後の属性は､DBTableModel に格納することにより､通常のﾃﾞｰﾀと
 * 同様の取り出し方が可能です。
 * また､通常のﾌｧｲﾙｱｯﾌﾟﾛｰﾄﾞ時の､form で使用する､enctype="multipart/form-data"
 * を指定した場合の､他のﾘｸｴｽﾄ情報も､{&#064;XXXX} 変数で取り出すことが可能です。
 *
 * この upload ﾀｸﾞでは､ｱｯﾌﾟﾛｰﾄﾞ後に､指定のﾌｧｲﾙ名に変更する機能があります。
 * file 登録ﾀﾞｲｱﾛｸﾞで指定した name に､"_NEW" という名称を付けたﾘｸｴｽﾄ値を
 * ﾌｧｲﾙのｱｯﾌﾟﾛｰﾄﾞと同時に送信することで､この名前にﾌｧｲﾙを付け替えます。
 * また､ｱｯﾌﾟﾛｰﾄﾞ後のﾌｧｲﾙ名は､name 指定の名称で､取り出せます。
 * ｸﾗｲｱﾝﾄから登録したｵﾘｼﾞﾅﾙのﾌｧｲﾙ名は､name に､"_ORG" という名称
 * で取り出すことが可能です。
 *
 * 通常では､これらのﾊﾟﾗﾒｰﾀを､RequestAttribute 変数にｾｯﾄしますので､{&#064;XXXX}で
 * 取り出すことが可能になります。さらに､"KEY","VALUE","ISFILE" のｶﾗﾑを持った､
 * DBTableModel にｾｯﾄします。
 *
 * 新機能として､columns を指定する事で､columns のｶﾗﾑを持つ DBTableModel にｾｯﾄします。
 * その場合は､ｶﾗﾑ名_01 ～ ｶﾗﾑ名_99 のように､ｱﾝﾀﾞｰﾊﾞｰで列ﾃﾞｰﾀとなるｷｰを定義してください。
 * ｱﾝﾀﾞｰﾊﾞｰがない場合は､ｶﾗﾑだけ作成されます。ｶﾗﾑ名と同じﾘｸｴｽﾄがあれば､
 * すべてのﾚｺｰﾄﾞに同じ値がｾｯﾄされます。
 *
 * 新ﾌｧｲﾙ名に拡張子が設定されていない場合は､ｵﾘｼﾞﾅﾙﾌｧｲﾙ名の拡張子をｾｯﾄします。
 *
 * HTML5 の type="file" 時の multiple 属性（ｱｯﾌﾟﾛｰﾄﾞﾌｧｲﾙの複数選択機能）に対応します。(5.7.1.1 (2013/12/13))
 * その場合は､新しいﾌｧｲﾙ名への変更はできません。ｵﾘｼﾞﾅﾙのﾌｧｲﾙ名でｱｯﾌﾟﾛｰﾄﾞされます
 *
 * 6.0.2.4 (2014/10/17)
 * ① name + _NEW と同じ考え方で､_PFX(接頭辞) , _SFX(接尾辞) 機能を追加します。
 * ② 上記機能は､_NEW との併用可能。_NEW がない場合は､ｵﾘｼﾞﾅﾙ(ｱｯﾌﾟﾛｰﾄﾞされた元のﾌｧｲﾙ名)に付与
 * ③ useBackup="rename" で､すでに同名のﾌｧｲﾙが存在した場合に､"_001" のような文字列を追加したﾌｧｲﾙにﾘﾈｰﾑします。
 *    Windowsの " - ｺﾋﾟｰ (2)" に近いですが､桁数を抑えるのと､useBackup="true" と異なり､過去の同一ﾌｧｲﾙ名は
 *    そのまま､有効になります。同一ﾌｧｲﾙが同一ﾌｫﾙﾀﾞに存在する場合のみ連番が付与されます。
 * ④ name + _NEW の指定に､ﾌｫﾙﾀﾞを含めることが可能。ただし､接続文字列は､"/" とする。
 *    同時にｱｯﾌﾟﾛｰﾄﾞを行う場合に､個別にﾌｫﾙﾀﾞを振り分けたり､_PFX(接頭辞) に､ﾌｫﾙﾀﾞを指定
 *    することも可能です。
 * ⑤ 最大転送ｻｲｽﾞ(Byte)を 10M → 30M に変更
 *
 * 8.1.2.0 (2022/03/10)
 * name + _NEW と同じ考え方で､_HASH(ﾌｧｲﾙのﾊｯｼｭ値) 機能を追加します。
 * ｼｽﾃﾑ定数のFILE_HASH_CODEで計算されたﾊｯｼｭ値を DBTableModel に格納します。
 *
 * @og.formSample
 * ●形式：&lt;og:upload fileURL="…" maxPostSize="…" /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *	 &lt;og:upload
 *		 fileURL			【TAG】ﾌｧｲﾙをｱｯﾌﾟﾛｰﾄﾞするﾃﾞｨﾚｸﾄﾘを指定します (初期値:FILE_URL[=filetemp/])
 *		 filename			【TAG】(通常は使いません)ﾌｧｲﾙを作成するときのﾌｧｲﾙ名をｾｯﾄします(初期値:null)
 *		 maxPostSize		【TAG】最大転送ｻｲｽﾞ(Byte)を指定します(初期値:31457280=30M) 0,またはﾏｲﾅｽで無制限です。
 *		 scope				【TAG】ｷｬｯｼｭする場合のｽｺｰﾌﾟ[request/page/session/application]を指定します(初期値:session)
 *		 tableId			【TAG】(通常は使いません)sessionから所得する DBTableModelｵﾌﾞｼﾞｪｸﾄの ID
 *		 columns			【TAG】DBTableModel作成時に､指定のｶﾗﾑの"_01"～"_99"の添え字をﾚｺｰﾄﾞとして作成します。
 *		 useBackup			【TAG】すでに同名のﾌｧｲﾙが存在した場合に､ﾊﾞｯｸｱｯﾌﾟ処理するかどうか[true/false/rename]を指定します(初期値:null=false)
 *									useBackup="rename" で､"_001" のような文字列を追加します。
 * ×	 storage			【廃止】保存先ｽﾄﾚｰｼﾞﾀｲﾌﾟを指定します   5.10.9.0 (2019/03/01) ADD
 * ×	 bucket 			【廃止】保存先ﾊﾞｹｯﾄ名を指定します 5.10.9.0 (2019/03/01) ADD
 *		 useLocal			【TAG】ｼｽﾃﾑ定数でｸﾗｳﾄﾞ設定されていても､ｸﾗｳﾄﾞ環境を使用しない場合､trueを指定します(初期値:false) 8.0.1.0 (2021/10/29)
 *		 language			【TAG】ﾀｸﾞ内部で使用する言語ｺｰﾄﾞ[ja/en/zh/…]を指定します
 *		 debug				【TAG】ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *	 /&gt;
 *
 * ●使用例 :
 * 【query.jsp】
 *		 &lt;form method="POST" action="result.jsp" enctype="multipart/form-data" target="RESULT"&gt;
 *		 &lt;table&gt;
 *		 &lt;tr&gt;&lt;og:input type="text" name="submitter" value="{&#064;USER.JNAME}" size="20" lbl="MSG0014" /&gt;&lt;/tr&gt;
 *		 &lt;tr&gt;
 *			 &lt;og:input type="file"	name="file_01"		size="30" lbl="MSG0015" /&gt;
 *			 &lt;og:input				name="file_01_NEW"	size="10" lbl="FILENAME" /&gt;
 *		 &lt;/tr&gt;&lt;tr&gt;
 *			 &lt;og:input type="file"	name="file_02"		size="30" lbl="MSG0015" /&gt;
 *			 &lt;og:input				name="file_02_NEW"	size="10" lbl="FILENAME" /&gt;
 *		 &lt;/tr&gt;&lt;tr&gt;
 *			 &lt;og:input type="file"	name="file_03"		size="30" lbl="MSG0015" /&gt;
 *			 &lt;og:input				name="file_03_NEW"	size="10" lbl="FILENAME" /&gt;
 *		 &lt;/tr&gt;&lt;tr&gt;
 *			 &lt;og:column name="writable"	value="false"	/&gt;
 *		 &lt;/tr&gt;
 *		 &lt;/table&gt;
 *
 * 【result.jsp】
 *		 &lt;og:upload
 *			 fileURL	= "{&#064;USER.ID}"
 *		 /&gt;
 *		 &lt;br /&gt;
 *		 &lt;og:message lbl="MSG0003" comment="ﾌｧｲﾙの登録が完了しました。" /&gt;
 *
 *		 &lt;og:view
 *			 command		= "NEW"
 *			 viewFormType	= "HTMLTable"
 *			 writable		= "{&#064;writable}"
 *		 /&gt;
 *
 *		 &lt;table&gt;
 *		 &lt;tr&gt;&lt;og:input name="submitter"	value="{&#064;submitter}"	/&gt;&lt;/tr&gt;
 *		 &lt;tr&gt;&lt;og:input name="writable"		value="{&#064;writable}"	/&gt;&lt;/tr&gt;
 *		 &lt;tr&gt;&lt;og:input name="directory"	value="{&#064;directory}"	/&gt;&lt;/tr&gt;
 *		 &lt;tr&gt;&lt;og:input name="file_01"		value="{&#064;file_01}"		/&gt;&lt;/tr&gt;
 *		 &lt;tr&gt;&lt;og:input name="file_01_NEW"	value="{&#064;file_01_NEW}"	/&gt;&lt;/tr&gt;
 *		 &lt;tr&gt;&lt;og:input name="file_01_ORG"	value="{&#064;file_01_ORG}"	/&gt;&lt;/tr&gt;
 *		 &lt;tr&gt;&lt;og:input name="file_02"		value="{&#064;file_02}"		/&gt;&lt;/tr&gt;
 *		 &lt;tr&gt;&lt;og:input name="file_02_NEW"	value="{&#064;file_02_NEW}"	/&gt;&lt;/tr&gt;
 *		 &lt;tr&gt;&lt;og:input name="file_02_ORG"	value="{&#064;file_02_ORG}"	/&gt;&lt;/tr&gt;
 *		 &lt;tr&gt;&lt;og:input name="file_03"		value="{&#064;file_03}"		/&gt;&lt;/tr&gt;
 *		 &lt;tr&gt;&lt;og:input name="file_03_NEW"	value="{&#064;file_03_NEW}"	/&gt;&lt;/tr&gt;
 *		 &lt;tr&gt;&lt;og:input name="file_03_ORG"	value="{&#064;file_03_ORG}"	/&gt;&lt;/tr&gt;
 *		 &lt;/table&gt;
 *
 * 【result.jsp】
 *		 &lt;og:upload
 *			 fileURL	 = "{&#064;USER.ID}"
 *			 columns	 = "submitter,file"
 *		 /&gt;
 *		 &lt;br /&gt;
 *		 &lt;og:message lbl="MSG0003" comment="ﾌｧｲﾙの登録が完了しました。" /&gt;
 *
 *		 &lt;og:view
 *			 command	  = "NEW"
 *			 viewFormType = "HTMLTable"
 *			 writable	  = "{&#064;writable}"
 *		 /&gt;
 *
 * @og.group ﾌｧｲﾙ入力
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class FileUploadTag extends CommonTagSupport {
	/** このﾌﾟﾛｸﾞﾗﾑのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "8.2.0.2 (2022/06/24)" ;
	private static final long serialVersionUID = 820220220624L ;

	/** 5.7.6.3 (2014/05/23) ｱｯﾌﾟﾛｰﾄﾞﾌｧｲﾙのCSVｾｯﾄのｷｰ */
	public static final String UPLOAD_FILES = "UPLOAD_FILES" ;

	// 3.5.2.0 (2003/10/20) ｶﾗﾑ名に､ISFILEを追加。
	private static final String[] NAMES = { "KEY","VALUE","ISFILE" };			// 6.4.1.1 (2016/01/16) names → NAMES  refactoring
	private static final String   ENCODE = "UTF-8";								// 3.5.2.0 (2003/10/20) 廃止
	private static final String   HASH_CODE = HybsSystem.sys( "FILE_HASH_CODE" );// 8.1.2.0 (2022/03/10)

	private String	fileURL		= HybsSystem.sys( "FILE_URL" );
	private String  filename	;												// 3.5.4.2 (2003/12/15)

	private int		maxPostSize = 30*1024*1024;									// 6.0.2.4 (2014/10/17) 最大ﾌｧｲﾙ容量  30MB
	private String	tableId		= HybsSystem.TBL_MDL_KEY ;

	// 5.6.5.2 (2013/06/21) DBTableModel作成時に､指定のｶﾗﾑの"_01"～"_99"の添え字をﾚｺｰﾄﾞとして作成します。
	private String		columns		;
	// 5.6.5.3 (2013/06/28) すでに同名のﾌｧｲﾙが存在した場合に､ﾊﾞｯｸｱｯﾌﾟ処理(renameTo)するかどうか[true/false]を指定します(初期値:false)
	// 6.0.2.4 (2014/10/17) useBackup は､true/false/rename が指定可能
	private String	useBackup	;
//	private String	storage		;		// 5.10.9.0 (2019/03/01) ADD 8.0.1.0 (2021/10/29) DEL
//	private String	bucket		;		// 5.10.9.0 (2019/03/01) ADD 8.0.1.0 (2021/10/29) DEL
	private boolean useLocal	;		// 8.0.1.0 (2021/10/29) ｸﾗｳﾄﾞ設定を使用しない場合は､true

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰ
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public FileUploadTag() { super(); }		// これも､自動的に呼ばれるが､空のﾒｿｯﾄﾞを作成すると警告されるので､明示的にしておきます。

	/**
	 * Taglibの終了ﾀｸﾞが見つかったときに処理する doEndTag() を ｵｰﾊﾞｰﾗｲﾄﾞします。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 ｴﾝｺｰﾄﾞの取得方法変更
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.1.3.0 (2003/04/10) UTF-8 決め打ちのｴﾝｺｰﾄﾞ情報を取得する。
	 * @og.rev 3.5.2.0 (2003/10/20) scope 属性を追加。
	 * @og.rev 3.5.4.2 (2003/12/15) ﾌｧｲﾙ名を指定できるようにします。
	 * @og.rev 3.6.0.8 (2004/11/19) DBTableModel をｾｰﾌﾞする時に､ﾄﾗﾝｻﾞｸｼｮﾝﾁｪｯｸを行います。
	 * @og.rev 3.7.1.1 (2005/05/23) ﾌｫﾙﾀﾞがない場合は､複数階層分のﾌｫﾙﾀﾞを自動で作成します。
	 * @og.rev 3.8.1.3A (2006/01/30) 新ﾌｧｲﾙ名にｵﾘｼﾞﾅﾙﾌｧｲﾙ名の拡張子をｾｯﾄします
	 * @og.rev 5.3.7.0 (2011/07/01) ｴﾗｰﾒｯｾｰｼﾞ内容変更
	 * @og.rev 5.6.5.2 (2013/06/21) columns 属性の追加
	 * @og.rev 5.6.5.3 (2013/06/28) useBackup 属性の追加
	 * @og.rev 5.8.8.0 (2015/06/05) ｴﾗｰ文の日本語化
	 * @og.rev 5.9.25.0 (2017/10/06) ｸﾗｳﾄﾞｽﾄﾚｰｼﾞの利用追加対応
	 * @og.rev 6.9.0.1 (2018/02/05) ﾌｧｲﾙをｾｰﾌﾞするﾃﾞｨﾚｸﾄﾘは､必要な場合のみ､作成します。
	 * @og.rev 8.0.1.0 (2021/10/29) storageType , bucketName 削除
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		startQueryTransaction( tableId );		// 3.6.0.8 (2004/11/19)
		final HttpServletRequest request = (HttpServletRequest)getRequest();

//		if( useLocal ) {	// 8.0.1.0 (2021/10/29) ｸﾗｳﾄﾞ設定を使用しない場合は､true
//			bucket = FileOperation.LOCAL ;
//		}

		try {
			final String directory = HybsSystem.url2dir( fileURL );
			// 6.9.0.1 (2018/02/05) ﾌｧｲﾙをｾｰﾌﾞするﾃﾞｨﾚｸﾄﾘは､必要な場合のみ､作成します。
//			final File dir = new File(directory);
//			if( ! dir.exists() && ! dir.mkdirs() ) {
//				// ERR0043:ﾃﾞｨﾚｸﾄﾘの作成に失敗しました。
//				final String errMsg = getResource().getLabel( "ERR0043" ) + "[" + directory + "]"; // 5.8.8.0 (2015/06/05)
//				throw new HybsSystemException( errMsg );
//			}

			// 3.8.1.3A (2006/01/30) 新ﾌｧｲﾙ名にｵﾘｼﾞﾅﾙﾌｧｲﾙ名の拡張子をｾｯﾄします
			// 5.6.5.3 (2013/06/28) useBackup 属性の追加
			// 5.9.25.0 (2017/10/06) fileURL 属性の追加
			// 5.10.9.0 (2019/03/01) storage, bucket を追加。
//			final MultipartRequest multi = new MultipartRequest( request,directory,maxPostSize,ENCODE,filename,useBackup,fileURL );
			// 8.0.1.0 (2021/10/29) storageType , bucketName 削除
//			final MultipartRequest multi = new MultipartRequest( request,directory,maxPostSize,ENCODE,filename,useBackup,fileURL,storage,bucket );
			final MultipartRequest multi = new MultipartRequest( request,directory,maxPostSize,ENCODE,filename,useBackup,fileURL,useLocal );
//			// 5.6.5.2 (2013/06/21) columns 属性の追加
//			DBTableModel table = null;
//			if( columns == null ) {						// 5.6.5.2 (2013/06/21) columns 属性の追加
//				table = makeDBTable( multi );
//			}
//			else {
//				table = makeDBTableFromClms( multi );
//			}

			// 6.9.0.1 (2018/02/05) ついでに修正
			final DBTableModel table = columns == null ? makeDBTable( multi ) : makeDBTableFromClms( multi );

			// 3.5.2.0 (2003/10/20) scope 属性を追加。
			// 3.6.0.8 (2004/11/19) ﾄﾗﾝｻﾞｸｼｮﾝﾁｪｯｸを行います。
			if( ! commitTableObject( tableId, table ) ) {
				// ERR0041:検索処理中に割り込みの検索要求がありました。処理されません。
				jspPrint( "FileUploadTag " + getResource().getLabel( "ERR0041" ) ); // 5.8.8.0 (2015/06/05)
				return SKIP_PAGE;
			}
		}
		// 6.9.0.1 (2018/02/05) ｾｰﾌﾞﾃﾞｨﾚｸﾄﾘ に関係するｴﾗｰを､IllegalArgumentException で返します。(無理から)
		catch( final IllegalArgumentException ex ) {
				// ERR0043:ﾃﾞｨﾚｸﾄﾘの作成に失敗しました。
				final String errMsg = getResource().getLabel( "ERR0043" ) + "[" + fileURL + "]"	// 5.8.8.0 (2015/06/05)
								+ CR + ex.getMessage() ;
				throw new HybsSystemException( errMsg , ex );
		}
		catch( final IOException ex ) {
			// ERR0044:ﾌｧｲﾙ登録ｴﾗｰ！
			final String errMsg = getResource().getLabel( "ERR0044" ) // 5.8.8.0 (2015/06/05)
				+ ex.getMessage() + CR	// 5.3.7.0 (2011/07/01) errMsg 修正
				+ "(" + toString() + CR
				+ "FileURL=" + fileURL + ")";
			throw new HybsSystemException( errMsg , ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}

		return EVAL_PAGE ;
	}

	/**
	 * ﾀｸﾞﾘﾌﾞｵﾌﾞｼﾞｪｸﾄをﾘﾘｰｽします。
	 * ｷｬｯｼｭされて再利用されるので､ﾌｨｰﾙﾄﾞの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) ｶｽﾀﾑﾀｸﾞの release() ﾒｿｯﾄﾞを､追加
	 * @og.rev 3.0.1.1 (2003/03/06) columns を廃止
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.4.2 (2003/12/15) ﾌｧｲﾙ名を指定できるようにします。
	 * @og.rev 5.6.5.2 (2013/06/21) columns 属性の追加
	 * @og.rev 5.6.5.2 (2013/06/21) useBackup 属性の追加
	 * @og.rev 6.0.2.4 (2014/10/17) useBackup 修正､最大ﾌｧｲﾙ容量 変更
	 * @og.rev 8.0.1.0 (2021/10/29) useLocal 属性を追加。
	 * @og.rev 8.0.1.0 (2021/10/29) storageType , bucketName 削除
	 */
	@Override
	protected void release2() {
		super.release2();
		fileURL			= HybsSystem.sys( "FILE_URL" );
		maxPostSize		= 30*1024*1024;				// 6.0.2.4 (2014/10/17) 最大ﾌｧｲﾙ容量  30MB
		tableId 		= HybsSystem.TBL_MDL_KEY ;
		filename		= null;			// 3.5.4.2 (2003/12/15)
		columns			= null;			// 5.6.5.2 (2013/06/21)
		useBackup		= null;			// 6.0.2.4 (2014/10/17)
//		storage			= null;			// 5.10.9.0 (2019/03/01) ADD
//		bucket			= null;			// 5.10.9.0 (2019/03/01) ADD
		useLocal		= false;		// 8.0.1.0 (2021/10/29) ｸﾗｳﾄﾞ設定を使用しない場合は､true
	}

	/**
	 * ﾌｧｲﾙｱｯﾌﾟﾛｰﾄﾞの実行結果を DBTableModel に記述します。
	 *
	 * ここでは､"KEY","VALUE","ISFILE" のｶﾗﾑに対して､値を設定していきます。
	 * 同時に､RequestAttribute 変数に､これらの値をｾｯﾄすることで､
	 * {&#064;XXXX} で値が取り出せる様にしています。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 ｴﾝｺｰﾄﾞの取得方法変更
	 * @og.rev 3.0.1.1 (2003/03/06) request 情報から{&#064;XXXX} で値が取り出せる様に修正。
	 * @og.rev 3.5.2.0 (2003/10/20) ｶﾗﾑ名(KEY,VALUE)に ISFILE を追加
	 * @og.rev 3.5.6.5 (2004/08/09) MultipartRequest 変更に伴なう修正(Enum変更､元ﾌｧｲﾙ名取得)
	 * @og.rev 3.5.6.6 (2004/08/23) 上記変更時のﾊﾞｸﾞ修正。
	 * @og.rev 3.5.6.6 (2004/08/23) 元ﾌｧｲﾙのｷｰを､XXXX_ORG にします。
	 * @og.rev 4.0.0.0 (2007/10/12) ﾃｰﾌﾞﾙﾓﾃﾞﾙの登録方法を変更
	 * @og.rev 5.3.2.0 (2011/02/01) ﾁｪｯｸ行のﾊﾟﾗﾒｰﾀｰはint配列側に変換して復元する。ﾊﾟﾗﾒｰﾀｰ名を保存する。
	 * @og.rev 5.4.4.2 (2012/02/03) CommonTagSupportと同様のﾁｪｯｸﾎﾞｯｸｽ判定を行う
	 * @og.rev 5.7.1.1 (2013/12/13) HTML5 ﾌｧｲﾙｱｯﾌﾟﾛｰﾄﾞの複数選択（multiple）対応
	 * @og.rev 5.7.1.2 (2013/12/20) 5.7.1.2 (2013/12/20) zip対応で､UploadedFile のﾒｿｯﾄﾞ変更
	 * @og.rev 5.7.3.0 (2014/02/07) zip対応の修正で､取得ﾌｧｲﾙ名が異なっていた。
	 * @og.rev 5.7.6.3 (2014/05/23) ｱｯﾌﾟﾛｰﾄﾞﾌｧｲﾙのCSVｾｯﾄ
	 * @og.rev 6.2.3.0 (2015/05/01) ｱｯﾌﾟﾛｰﾄﾞﾌｧｲﾙを､UPLOAD_FILESｷｰに､Listｵﾌﾞｼﾞｪｸﾄに格納します。
	 * @og.rev 5.9.25.0 (2017/10/06) ｸﾗｳﾄﾞ対応
	 * @og.rev 8.2.0.2 (2022/06/24) must と mustAny の複数選択対応
	 *
	 * @param	 multi	   MultipartRequestｵﾌﾞｼﾞｪｸﾄ
	 *
	 * @return	 ﾃｰﾌﾞﾙﾓﾃﾞﾙ
	 */
	private DBTableModel makeDBTable( final MultipartRequest multi ) {

		final DBTableModel table = DBTableModelUtil.newDBTable();

		table.init( NAMES.length );

		for( int i=0; i<NAMES.length; i++ ) {
			final DBColumn dbColumn = getDBColumn( NAMES[i] );
			table.setDBColumn( i,dbColumn );
		}

		String[] values ;		// 4.0.0.0 (2007/10/12)
		final List<String> prmNames = new ArrayList<>();

		// 5.7.1.1 (2013/12/13) HTML5 ﾌｧｲﾙｱｯﾌﾟﾛｰﾄﾞの複数選択（multiple）対応
		// 6.2.3.0 (2015/05/01) ｱｯﾌﾟﾛｰﾄﾞﾌｧｲﾙを､UPLOAD_FILESｷｰに､Listｵﾌﾞｼﾞｪｸﾄに格納します。
		final List<String> list = new ArrayList<>();
		final UploadedFile[] files = multi.getUploadedFile();
		for( int i=0; i<files.length; i++ ) {
			final String name = files[i].getName();					// multiple対応では､ｷｰがかぶることがある。
			String val  = files[i].getUploadFile();				// 5.9.25.0 (2017/10/06) MODIFY getUploadFileで取得される値をFileからStringに変更対応

			// 6.2.3.0 (2015/05/01) ｱｯﾌﾟﾛｰﾄﾞﾌｧｲﾙを､UPLOAD_FILESｷｰに､Listｵﾌﾞｼﾞｪｸﾄに格納します。
			list.add( val );

			// "KEY","VALUE","ISFILE" の順にﾃﾞｰﾀを作成します。
			values = new String[] { name, val, "1" };
			table.addColumnValues( values );
			setRequestAttribute( name,val );
			prmNames.add( name );				// 5.7.1.1 (2013/12/13) List に設定する。

			final String orgName = name + "_ORG" ;
			val  = files[i].getOriginalFileName();	// 注意：取得は､送信名

			// "KEY","VALUE","ISFILE" の順にﾃﾞｰﾀを作成します。
			values = new String[] { orgName, val, "2" };
			table.addColumnValues( values );
			setRequestAttribute( orgName,val );
		}

		// 6.2.3.0 (2015/05/01) ｱｯﾌﾟﾛｰﾄﾞﾌｧｲﾙを､UPLOAD_FILESｷｰに､Listｵﾌﾞｼﾞｪｸﾄに格納します。
		setRequestAttribute( UPLOAD_FILES,list );

		// "KEY","VALUE","ISFILE" の順にﾃﾞｰﾀを作成します。
		values = new String[] { "directory", fileURL, "0" };
		table.addColumnValues( values );
		setRequestAttribute( "directory",fileURL );

		final String[] params = multi.getParameterNames();
		for( int i=0; i<params.length; i++ ) {
			final String name = params[i];
			// 5.3.2.0 (2011/02/01) ﾁｪｯｸ行のﾊﾟﾗﾒｰﾀｰはint配列側に変換
			if( HybsSystem.ROW_SEL_KEY.equals( name ) ) {
				setRequestAttribute( name,multi.getIntParameters(name) );
			}
			// 8.2.0.2 (2022/06/24) must と mustAny の複数選択対応
			else if( name.startsWith( HybsSystem.MUST_KEY ) ) {
				setRequestAttribute( name,multi.getParameters(name) );		// MUST_KEY+"must" or MUST_KEY+"mustAny"
			}
			else {
				// 5.6.5.2 (2013/06/21) ﾁｪｯｸﾎﾞｯｸｽ配列の値取得を考慮した､MultipartRequest のﾊﾟﾗﾒｰﾀ値取得
				final String val = getParamVal( name,multi );

				values = new String[] { name, val, "0" };
				table.addColumnValues( values );
				setRequestAttribute( name,val );
				prmNames.add( name );				// 5.7.1.1 (2013/12/13) List に設定する。
			}
		}

		// 5.3.2.0 (2011/02/01) ﾊﾟﾗﾒｰﾀｰ名を保存する。
		// 5.7.1.1 (2013/12/13) List に設定する。
		setParameterNames( prmNames.toArray( new String[prmNames.size()] ) );

		return table ;
	}

	/**
	 * ﾌｧｲﾙｱｯﾌﾟﾛｰﾄﾞの実行結果を 横持の DBTableModel に記述します。
	 *
	 * この処理は､columns 属性を設定した場合のみとします。
	 *
	 * DBTableModel作成時に､指定のｶﾗﾑの"_01"～"_99"の添え字をﾚｺｰﾄﾞとして作成します。
	 * 現状は､"KEY","VALUE","ISFILE" のｶﾗﾑに､ﾃﾞｰﾀを縦持ちで作成しています。
 	 * これを､横持で作成しますが､ｶﾗﾑの末尾に､"_01"～"_99" までの添え字を
	 * 持つ場合は､これをﾚｺｰﾄﾞと認識させます。
	 * 添え字がない場合は､ｶﾗﾑだけ作成されます。ｶﾗﾑ名と同じﾘｸｴｽﾄがあれば､
	 * すべてのﾚｺｰﾄﾞに同じ値がｾｯﾄされます。
	 *
	 * @og.rev 5.6.5.2 (2013/06/21) 新規作成
	 * @og.rev 5.6.6.1 (2013/07/12) 添え字がない場合の処理の見直し
	 * @og.rev 5.7.1.2 (2013/12/20) zip対応で､UploadedFile のﾒｿｯﾄﾞ変更
	 * @og.rev 5.7.3.0 (2014/02/07) zip対応の修正で､取得ﾌｧｲﾙ名が異なっていた。
	 * @og.rev 5.7.6.3 (2014/05/23) ｱｯﾌﾟﾛｰﾄﾞﾌｧｲﾙのCSVｾｯﾄ
	 * @og.rev 6.0.2.4 (2014/10/17) _PFX , _SFX 対応
	 * @og.rev 6.2.3.0 (2015/05/01) ｱｯﾌﾟﾛｰﾄﾞﾌｧｲﾙを､UPLOAD_FILESｷｰに､Listｵﾌﾞｼﾞｪｸﾄに格納します。
	 * @og.rev 5.9.25.0 (2017/10/06) ｸﾗｳﾄﾞ対応
	 * @og.rev 8.1.2.0 (2022/03/10) _HASH(ﾌｧｲﾙのﾊｯｼｭ値) 対応
	 * @og.rev 8.2.0.2 (2022/06/24) must と mustAny の複数選択対応
	 *
	 * @param	 multi	   MultipartRequestｵﾌﾞｼﾞｪｸﾄ
	 *
	 * @return	 ﾃｰﾌﾞﾙﾓﾃﾞﾙ
	 */
	private DBTableModel makeDBTableFromClms( final MultipartRequest multi ) {

		final DBTableModel table = DBTableModelUtil.newDBTable();

		final String[] clmNames = columns.split( "," );

		table.init( clmNames.length );

		// 値配列（１行分）
		String[] rowVal = new String[clmNames.length];

		// 5.7.1.1 (2013/12/13) HTML5 ﾌｧｲﾙｱｯﾌﾟﾛｰﾄﾞの複数選択（multiple）対応
		final UploadedFile[] files = multi.getUploadedFile();

		final List<String> prmNames = new ArrayList<>();

		for( int i=0; i<clmNames.length; i++ ) {
			final String clm = clmNames[i] ;
			final DBColumn dbColumn = getDBColumn( clm );
			table.setDBColumn( i,dbColumn );

			// 先に､ｶﾗﾑ名と一致するﾊﾟﾗﾒｰﾀを初期値としてｾｯﾄしておきます。
			String val = getParamVal( clm,multi );

			// 5.6.6.1 (2013/07/12) ﾌｧｲﾙ名も同様に､あれば初期値ｾｯﾄしておきます。
			if( val == null ) {
				// 5.7.1.1 (2013/12/13) HTML5 ﾌｧｲﾙｱｯﾌﾟﾛｰﾄﾞの複数選択（multiple）対応
				for( int j=0; j<files.length; j++ ) {
					final String nm = files[j].getName();
					if( clm.equalsIgnoreCase( nm ) ) {
						val = files[i].getUploadFile();				// 5.9.25.0 (2017/10/06) getUploadFileの型をFileからStringに変更対応
						break;										// 5.7.6.3 (2014/05/23) たぶん有ったほうが良い。
					}
				}
			}
			// 5.7.1.1 (2013/12/13) getFilesystemName() の中に､newFile が null の場合は､original を返す処理がある。
			rowVal[i] = ( val == null ) ? "" : val ;
		}

		// 6.2.3.0 (2015/05/01) ｱｯﾌﾟﾛｰﾄﾞﾌｧｲﾙを､UPLOAD_FILESｷｰに､Listｵﾌﾞｼﾞｪｸﾄに格納します。
		final List<String> list = new ArrayList<>();

		for( int i=0; i<files.length; i++ ) {
			final String name = files[i].getName();
			// 5.6.6.1 (2013/07/12) 添え字がない場合の処理の見直し。先にﾚｺｰﾄﾞを作成
			String[] values = new String[clmNames.length];
			System.arraycopy( rowVal,0,values,0,values.length );		// 行にｾｯﾄするに当たり､rowVal を values にｺﾋﾟｰしておく。

			// ﾌｧｲﾙ名を Attribute で使えるようにｾｯﾄしておく。
			final String fval = files[i].getUploadFile();				// 5.9.25.0 (2017/10/06) getUploadで取得される値をFileからStringに変更
			setRequestAttribute( name,fval );
			prmNames.add( name );										// 5.7.1.1 (2013/12/13) List に設定する。

			// 6.2.3.0 (2015/05/01) ｱｯﾌﾟﾛｰﾄﾞﾌｧｲﾙを､UPLOAD_FILESｷｰに､Listｵﾌﾞｼﾞｪｸﾄに格納します。
			list.add( fval );

			final String orgName = name + "_ORG" ;
			final String oval  = files[i].getOriginalFileName();	// 注意：取得は､送信名
			setRequestAttribute( orgName,oval );

			// ﾌｧｲﾙのｷｰを元に､添え字を検索します。
			final int adrs = name.lastIndexOf( '_' );		// 添え字は､'_' で区切られます。
			// 5.6.6.1 (2013/07/12) 添え字がない場合の処理の見直し。後続処理を行う。
			if( adrs > 0 ) {
				final String fnm = name.substring( 0,adrs );	// ﾌｧｲﾙ名(分割後)
				final String sub = name.substring( adrs );	// 添え字(ｱﾝﾀﾞｰﾊﾞｰ含む)

				// ｶﾗﾑ名で検索しながら､ﾚｺｰﾄﾞ単位になるようにｾｯﾄしていきます。
				for( int j=0; j<clmNames.length; j++ ) {
					final String clm = clmNames[j] ;
					String nm  = null;
					String val = null;

					if( fnm.equalsIgnoreCase( clm ) ) {		// ﾌｧｲﾙ名ｶﾗﾑ(_NEWﾌｧｲﾙ名も､この値にすでに変わっている）
						val = fval;
					}
					else if( ( fnm + "_ORG" ).equalsIgnoreCase( clm ) ) {		// 元ﾌｧｲﾙ名ｶﾗﾑ
						val  = oval;
					}
					else if( ( fnm + "_NEW" ).equalsIgnoreCase( clm ) ) {		// 新ﾌｧｲﾙ名ｶﾗﾑ
						nm   = name + "_NEW" ;
						val  = multi.getParameter( nm );
					}
					// 6.0.2.4 (2014/10/17) _PFX , _SFX 対応
					else if( ( fnm + "_PFX" ).equalsIgnoreCase( clm ) ) {		// ﾌﾟﾚﾌｨｯｸｽｶﾗﾑ
						nm   = name + "_PFX" ;
						val  = multi.getParameter( nm );
					}
					else if( ( fnm + "_SFX" ).equalsIgnoreCase( clm ) ) {		// ｻﾌｨｯｸｽｶﾗﾑ
						nm   = name + "_SFX" ;
						val  = multi.getParameter( nm );
					}
					else if( ( fnm + "_HASH" ).equalsIgnoreCase( clm ) ) {		// ﾊｯｼｭｺｰﾄﾞ 8.1.2.0 (2022/03/10)
						nm   = name + "_HASH" ;
						final File localFile = files[i].getFile( useLocal );
						val  = HybsCryptography.getHash( HASH_CODE, localFile );
					}
					else {
						nm  = clmNames[j] + sub;					// sub は､ｱﾝﾀﾞｰﾊﾞｰ含む添え字
						// 5.6.5.2 (2013/06/21) ﾁｪｯｸﾎﾞｯｸｽ配列の値取得を考慮した､MultipartRequest のﾊﾟﾗﾒｰﾀ値取得
						val = getParamVal( nm,multi );
					}
					if( val != null ) { values[j] = val ; }		// val が null でないときのみｾｯﾄします。
				}
			}
			table.addColumnValues( values );
		}
		// 6.2.3.0 (2015/05/01) ｱｯﾌﾟﾛｰﾄﾞﾌｧｲﾙを､UPLOAD_FILESｷｰに､Listｵﾌﾞｼﾞｪｸﾄに格納します。
		setRequestAttribute( UPLOAD_FILES,list );

		// Attribute で使えるようにｾｯﾄしておく。
		setRequestAttribute( "directory",fileURL );

		// Attribute で使えるようにｾｯﾄしておく。
		final String[] params = multi.getParameterNames();
		for( int i=0; i<params.length; i++ ) {
			final String name = params[i];
			// 5.3.2.0 (2011/02/01) ﾁｪｯｸ行のﾊﾟﾗﾒｰﾀｰはint配列側に変換
			if( HybsSystem.ROW_SEL_KEY.equals( name ) ) {
				setRequestAttribute( name,multi.getIntParameters(name) );
			}
			// 8.2.0.2 (2022/06/24) must と mustAny の複数選択対応
			else if( name.startsWith( HybsSystem.MUST_KEY ) ) {
				setRequestAttribute( name,multi.getParameters(name) );		// MUST_KEY+"must" or MUST_KEY+"mustAny"
			}
			else {
				// 5.6.5.2 (2013/06/21) ﾁｪｯｸﾎﾞｯｸｽ配列の値取得を考慮した､MultipartRequest のﾊﾟﾗﾒｰﾀ値取得
				final String val = getParamVal( name,multi );
				setRequestAttribute( name,val );
				prmNames.add( name );				// 5.7.1.1 (2013/12/13) List に設定する。
			}
		}

		// 5.3.2.0 (2011/02/01) ﾊﾟﾗﾒｰﾀｰ名を保存する。
		// 5.7.1.1 (2013/12/13) List に設定する。
		setParameterNames( prmNames.toArray( new String[prmNames.size()] ) );

		return table ;
	}

	/**
	 * ﾁｪｯｸﾎﾞｯｸｽ配列の値取得を考慮した､MultipartRequest のﾊﾟﾗﾒｰﾀ値取得
	 *
	 * ここでは､､MultipartRequest のﾊﾟﾗﾒｰﾀ値を取得します。
	 * 値の取得で､ﾁｪｯｸﾎﾞｯｸｽ配列の場合は､取得した値が､"0" の場合のみ
	 * 配列でﾊﾟﾗﾒｰﾀを取得し直し､"1" がないかどうか再度検索します。
 	 * ﾁｪｯｸﾎﾞｯｸｽでは､ﾁｪｯｸ時の "1" と､ﾁｪｯｸされなかった場合の､
	 * hidden の "0" の両方の値が配列としてﾘｸｴｽﾄされるｹｰｽがある為です。
	 *
	 * @og.rev 5.6.5.2 (2013/06/21) 新規作成
	 * @og.rev 6.3.9.1 (2015/11/27) MultipartRequest#getParameters(String)は､null ではなく長さが0の配列を返す。
	 *
	 * @param	 key	   MultipartRequestのﾊﾟﾗﾒｰﾀ取得のためのｷｰ
	 * @param	 multi	   MultipartRequestｵﾌﾞｼﾞｪｸﾄ
	 *
	 * @return	 ﾁｪｯｸﾎﾞｯｸｽ配列を考慮したﾊﾟﾗﾒｰﾀ値
	 */
	private String getParamVal( final String key , final MultipartRequest multi ) {
		String val = multi.getParameter( key );

		if( "0".equals(val) ){ // 5.4.4.2 ﾁｪｯｸﾎﾞｯｸｽ配列対応。
			final String[] vals = multi.getParameters( key );
			for( final String tmp : vals ) {
				if( "1".equals( tmp ) ) {
					val = "1";
					break;
				}
			}
		}
		return val ;
	}

	/**
	 * 【TAG】ﾌｧｲﾙをｱｯﾌﾟﾛｰﾄﾞするﾃﾞｨﾚｸﾄﾘを指定します
	 *		(初期値:FILE_URL[={@og.value SystemData#FILE_URL}])。
	 *
	 * @og.tag
	 * この属性で指定されるﾃﾞｨﾚｸﾄﾘに､ｱｯﾌﾟﾛｰﾄﾞされたﾌｧｲﾙをｾｰﾌﾞします。
	 * 指定方法は､通常の fileURL 属性と同様に､先頭が､'/' (UNIX) または､２文字目が､
	 * ":" (Windows)の場合は､指定のURLそのままのﾃﾞｨﾚｸﾄﾘに､そうでない場合は､
	 * fileURL = "{&#064;USER.ID}" と指定すると､FILE_URL 属性で指定のﾌｫﾙﾀﾞの下に､
	 * さらに､各個人ID別のﾌｫﾙﾀﾞを作成して､そこにｾｰﾌﾞします。
	 * (初期値:ｼｽﾃﾑ定数のFILE_URL[={@og.value SystemData#FILE_URL}])。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) StringUtil.urlAppend ﾒｿｯﾄﾞの利用
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたﾃﾞｨﾚｸﾄﾘ名の最後が"\"or"/"で終わっていない場合に､"/"を付加する。
	 * @og.rev 5.7.1.1 (2013/12/13) ﾘｸｴｽﾄ変数が使えないｴﾗｰを表示する。
	 * @og.rev 5.7.4.3 (2014/03/28) ﾘｸｴｽﾄ変数が使えない事のﾁｪｯｸを行う。
	 * @og.rev 6.4.2.1 (2016/02/05) URLの最後に､"/" を追加する処理を廃止。
	 *
	 * @param	url ﾌｧｲﾙURL
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_URL
	 */
	public void setFileURL( final String url ) {
		final String furl = nval( getRequestParameter( url ),null );
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		if( furl == null ) {
			chckReqParam( url,"fileURL" );		// 5.7.4.3 (2014/03/28) ﾘｸｴｽﾄ変数が使えない事のﾁｪｯｸ
		}
		else {
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】最大転送ｻｲｽﾞ(Byte)を指定します(初期値:31457280)。
	 *
	 * @og.tag
	 * 最大転送ｻｲｽﾞを指定します。初期値は､30*1024*1024 = 30MB です。
	 * 指定は､Byte 単位で指定します。
	 * 0,またはﾏｲﾅｽを指定することで､制限ﾁｪｯｸを外す（＝無制限）事ができます。
	 *
	 * @og.rev 3.0.1.1 (2003/03/06) maxPostSize の設定ﾊﾞｸﾞ修正。
	 * @og.rev 5.6.5.3 (2013/06/28) ｺﾒﾝﾄ追加（0,またはﾏｲﾅｽで無制限）
	 * @og.rev 5.7.4.3 (2014/03/28) ﾘｸｴｽﾄ変数が使えない事のﾁｪｯｸを行う。
	 * @og.rev 6.0.2.4 (2014/10/17) 最大ﾌｧｲﾙ容量 10M → 30M 変更(JavaDocの修正)
	 *
	 * @param	maxPS 最大転送ｻｲｽﾞ
	 */
	public void setMaxPostSize( final String maxPS ) {
		maxPostSize = nval( getRequestParameter( maxPS ),maxPostSize );

		chckReqParam( maxPS,"maxPostSize" );		// 5.7.4.3 (2014/03/28) ﾘｸｴｽﾄ変数が使えない事のﾁｪｯｸ
	}

	/**
	 * 【TAG】(通常は使いません)結果のDBTableModelを､sessionに登録するときのｷｰを指定します
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @og.tag
	 * 検索結果より､DBTableModelｵﾌﾞｼﾞｪｸﾄを作成します。これを､下流のviewﾀｸﾞ等に
	 * 渡す場合に､通常は､session を利用します。その場合の登録ｷｰです。
	 * query ﾀｸﾞを同時に実行して､結果を求める場合､同一ﾒﾓﾘに配置される為､
	 * この tableId 属性を利用して､ﾒﾓﾘ空間を分けます。
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @og.rev 5.7.4.3 (2014/03/28) ﾘｸｴｽﾄ変数が使えない事のﾁｪｯｸを行う。
	 *
	 * @param	id ﾃｰﾌﾞﾙID (sessionに登録する時のID)
	 */
	public void setTableId( final String id ) {
		tableId = nval( getRequestParameter( id ),tableId );

		chckReqParam( id,"tableId" );		// 5.7.4.3 (2014/03/28) ﾘｸｴｽﾄ変数が使えない事のﾁｪｯｸ
	}

	/**
	 * 【TAG】(通常は使いません)ﾌｧｲﾙを作成するときのﾌｧｲﾙ名をｾｯﾄします(初期値:null)。
	 *
	 * @og.tag
	 * ﾌｧｲﾙを作成するときのﾌｧｲﾙ名をｾｯﾄします。
	 * これは､複数同時にｱｯﾌﾟﾛｰﾄﾞﾌｧｲﾙ名を変更する時に使用できません。
	 * 通常､ｱｯﾌﾟﾛｰﾄﾞされたﾌｧｲﾙ名を指定する場合､ｱｯﾌﾟﾛｰﾄﾞするinput ﾀｸﾞの
	 * name 属性に指定する名称 ＋ "_NEW" というﾘｸｴｽﾄ値を同時に送信すれば､
	 * 内部的に関連付けて､ﾌｧｲﾙ名を更新します。
	 * その場合､ｸﾗｲｱﾝﾄより指定したﾌｧｲﾙ名は､name属性＋"_ORG" という
	 * ﾘｸｴｽﾄ値として取得することが可能になります。
	 * name属性 には､最終的に設定されたﾌｧｲﾙ名がｾｯﾄされています。
	 * いずれの値も､{&#064;name属性＋"_ORG"} や､{&#064;name属性＋"_NEW"}として､
	 * ｱｯﾌﾟﾛｰﾄﾞのｵﾘｼﾞﾅﾙと変更後のﾌｧｲﾙ名を取得することが出来ます。
	 *
	 * 5.7.1.2 (2013/12/20) zip 対応
	 * filename 属性に､".zip" の拡張子のﾌｧｲﾙ名を指定した場合は､ｱｯﾌﾟﾛｰﾄﾞされた一連のﾌｧｲﾙを
	 * ZIP圧縮します。これは､ｱｯﾌﾟﾛｰﾄﾞ後の処理になります。
	 * ZIP圧縮のｵﾘｼﾞﾅﾙﾌｧｲﾙは､そのまま残ります。
	 * なお､ZIPﾌｧｲﾙは､useBackup属性を true に設定しても､無関係に､上書きされます。
	 *
	 * 5.7.4.3 (2014/03/28) filename 属性のﾘｸｴｽﾄ変数対応
	 * filename 属性のみ､{&#064;XXXX} のﾘｸｴｽﾄ変数が使えるようにします。
	 * 他のﾊﾟﾗﾒｰﾀでは使えません。
	 * これは､multipart/form-data のﾘｸｴｽﾄでは､ﾊﾟｰﾄの分解処理をしないと､ﾘｸｴｽﾄ変数が
	 * 拾えない為､ﾘｸｴｽﾄ変数は､この､upload ﾀｸﾞ以降でのみ利用可能でした。
	 * zip対応と関連付けて､filename 属性のみ､利用できるように､MultipartRequest 側の処理に組み込みます。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) ﾌｧｲﾙ名を指定できるようにします。
	 * @og.rev 5.7.1.1 (2013/12/13) ﾘｸｴｽﾄ変数が使えないｴﾗｰを表示する。
	 * @og.rev 5.7.4.3 (2014/03/28) ﾘｸｴｽﾄ変数を使えるようにします。
	 *
	 * @param   fname ﾌｧｲﾙ名
	 */
	public void setFilename( final String fname ) {
		filename = nval( getReservedParameter( fname ),null );		// 予約語のみ処理をします。
	}

	/**
	 * 【TAG】DBTableModel作成時に､指定のｶﾗﾑの"_01"～"_99"の添え字をﾚｺｰﾄﾞとして作成します。
	 *
	 * @og.tag
	 * 現状は､"KEY","VALUE","ISFILE" のｶﾗﾑに､ﾃﾞｰﾀを縦持ちで作成しています。
	 * これを､横持で作成しますが､ｶﾗﾑの末尾に､"_01"～"_99" までの添え字を
	 * 持つ場合は､これをﾚｺｰﾄﾞと認識させます。
	 * ｱﾝﾀﾞｰﾊﾞｰがない場合は､ｶﾗﾑだけ作成されます。ｶﾗﾑ名と同じﾘｸｴｽﾄがあれば､
	 * すべてのﾚｺｰﾄﾞに同じ値がｾｯﾄされます。
	 * この処理は､columns 属性を設定した場合のみとします。
	 *
	 * @og.rev 5.6.5.2 (2013/06/21) 新規作成
	 * @og.rev 5.7.4.3 (2014/03/28) ﾘｸｴｽﾄ変数が使えない事のﾁｪｯｸを行う。
	 *
	 * @param   clms DBTableModel作成時のｶﾗﾑ列(CSV形式)
	 */
	public void setColumns( final String clms ) {
		columns = nval( getRequestParameter( clms ),columns );

		chckReqParam( clms,"columns" );		// 5.7.4.3 (2014/03/28) ﾘｸｴｽﾄ変数が使えない事のﾁｪｯｸ
	}

	/**
	 * 【TAG】すでに同名のﾌｧｲﾙが存在した場合に､ﾊﾞｯｸｱｯﾌﾟ処理するかどうか[true/false/rename]を指定します(初期値:null=false)。
	 *
	 * @og.tag
	 * ﾌｧｲﾙｱｯﾌﾟﾛｰﾄﾞ時に､ｱｯﾌﾟﾛｰﾄﾞ先に､同名のﾌｧｲﾙが存在した場合は､既存機能は､そのまま
	 * 置き換えていましたが､簡易ﾊﾞｰｼﾞｮﾝｱｯﾌﾟ機能として､useBackup="true" を指定すると､既存のﾌｧｲﾙを
	 * ﾘﾈｰﾑして､ﾊﾞｯｸｱｯﾌﾟﾌｧｲﾙを作成します。
	 * ﾊﾞｯｸｱｯﾌﾟﾌｧｲﾙは､ｱｯﾌﾟﾛｰﾄﾞﾌｫﾙﾀﾞを基準として､_backup/ﾌｧｲﾙ名.拡張子_処理時刻のlong値.拡張子 になります。
	 * ｵﾘｼﾞﾅﾙのﾌｧｲﾙ名（拡張子付）を残したまま､"_処理時刻のlong値" を追加し､さらに､ｵﾘｼﾞﾅﾙの拡張子を追加します。
	 * ﾊﾞｯｸｱｯﾌﾟﾌｧｲﾙの形式は指定できません。
	 *
	 * 初期値は､互換性を持たせるため､null(=false) です。
	 *
	 * 6.0.2.4 (2014/10/17)
	 * useBackup="rename" で､すでに同名のﾌｧｲﾙが存在した場合に､"_001" のような文字列を追加したﾌｧｲﾙにﾘﾈｰﾑします。
	 * Windowsの " - ｺﾋﾟｰ (2)" に近いですが､桁数を抑えるのと､useBackup="true" と異なり､過去の同一ﾌｧｲﾙ名は
	 * そのまま､有効になります。同一ﾌｧｲﾙが同一ﾌｫﾙﾀﾞに存在する場合のみ連番が付与されます。
	 *
	 * @og.rev 5.6.5.3 (2013/06/28) 新規作成
	 * @og.rev 5.7.4.3 (2014/03/28) ﾘｸｴｽﾄ変数が使えない事のﾁｪｯｸを行う。
	 * @og.rev 6.0.2.4 (2014/10/17) true/false 以外に､rename も有効とする。
	 *
	 * @param   flag ﾊﾞｯｸｱｯﾌﾟ処理可否 [true:する/false:しない]
	 */
	public void setUseBackup( final String flag ) {
		useBackup = nval( getRequestParameter( flag ),useBackup );

		// 6.0.2.4 (2014/10/17) ﾊﾟﾗﾒｰﾀﾁｪｯｸをしておきます。
		if( useBackup != null && "true/false/rename".indexOf( useBackup.toLowerCase(Locale.JAPAN) ) < 0 ) {
			final String errMsg = "useBackup 属性は､[true/false/rename] から､指定してください。"
								+ " useBackup=[" + useBackup + "]" + CR ;
			throw new HybsSystemException( errMsg );
		}

		chckReqParam( flag,"useBackup" );		// 5.7.4.3 (2014/03/28) ﾘｸｴｽﾄ変数が使えない事のﾁｪｯｸ
	}

	/**
	 * ﾘｸｴｽﾄ変数が使えない事のﾁｪｯｸを行います。
	 *
	 * upload では､enctype="multipart/form-data" のため､{&#064;XXXX}形式のﾊﾟﾗﾒｰﾀが使えません。
	 * 5.7.4.3 (2014/03/28) から､filename のみ利用可能としたことで､同様に利用できると
	 * 勘違いするｹｰｽに対応する為､すべてのﾊﾟﾗﾒｰﾀについてﾁｪｯｸを行います。
	 * ここでは､getRequestParameter( String ) の実行後､すぐに､isNull() 判定を行う事で､
	 * ﾘｸｴｽﾄ変数の存在ﾁｪｯｸを行う事にしています。
	 *
	 * @og.rev 5.7.4.3 (2014/03/28) ﾘｸｴｽﾄ変数が使えない事のﾁｪｯｸを行う。
	 *
	 * @param   org 引数のｵﾘｼﾞﾅﾙ値
	 * @param   key ｴﾗｰの発生した変数名
	 * @throws	HybsSystemException	ﾊﾟﾗﾒｰﾀが使用されていた場合
	 */
	private void chckReqParam( final String org,final String key ) {
		if( isNull() && org != null && org.contains( "{@" ) ) {
			final String errMsg = "upload では､enctype=\"multipart/form-data\" のため､{@XXXX}形式のﾊﾟﾗﾒｰﾀが使えません。"
								+ CR
								+ "    " + key + "=[" + org + "]" ;
			throw new HybsSystemException( errMsg );		// ﾘｸｴｽﾄ変数が使えないｴﾗｰ
		}
	}

	/**
	 * ﾀｸﾞの名称を､返します。
	 * 自分自身のｸﾗｽ名より､自動的に取り出せないため､このﾒｿｯﾄﾞをｵｰﾊﾞｰﾗｲﾄﾞします。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @return  ﾀｸﾞの名称
	 * @og.rtnNotNull
	 */
	@Override
	protected String getTagName() {
		return "upload" ;
	}

//	/**
//	 * 【TAG】保存先ｽﾄﾚｰｼﾞﾀｲﾌﾟを設定します。
//	 *
//	 * @og.tag
//	 * ﾌｧｲﾙを読み取り元の､ｽﾄﾚｰｼﾞﾀｲﾌﾟを設定します。
//	 * 未設定の場合は､ｼｽﾃﾑﾘｿｰｽの｢CLOUD_TARGET｣が参照されます。
//	 * 自身のｻｰﾊﾞを指定する場合は､｢default｣を設定してください。
//	 * 強制的に､ﾛｰｶﾙﾌｧｲﾙを指定する場合は､空文字列ではなく､"LOCAL" を指定します。
//	 *
//	 * @og.rev 5.10.9.0 (2019/03/01) 新規追加
//	 * @og.rev 8.0.1.0 (2021/10/29) storageType , bucketName 削除
//	 *
//	 * @param type 保存先ｽﾄﾚｰｼﾞﾀｲﾌﾟ
//	 */
//	public void setStorageType( final String type ) {
//		storage  = nval( getRequestParameter(type), storage );
//	}

//	/**
//	 * 【TAG】保存先ﾊﾞｹｯﾄ名を設定します。
//	 *
//	 * @og.tag
//	 * ﾌｧｲﾙを読み取り元の､ﾊﾞｹｯﾄ名を指定します。
//	 * ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ利用時のみ有効です。
//	 * 未設定の場合は､ｼｽﾃﾑﾘｿｰｽの｢CLOUD_BUCKET｣が参照されます。
//	 * 強制的に､ﾛｰｶﾙﾌｧｲﾙを指定する場合は､空文字列ではなく､"LOCAL" を指定します。
//	 *
//	 * @og.rev 5.10.9.0 (2019/03/01) 新規追加
//	 * @og.rev 8.0.1.0 (2021/10/29) storageType , bucketName 削除
//	 *
//	 * @param bkt 保存先ﾊﾞｹｯﾄ名
//	 */
//	public void setBucket( final String bkt )  {
//		bucket = nval( getRequestParameter(bkt), bucket );
//	}

	/**
	 * 【TAG】ｼｽﾃﾑ定数でｸﾗｳﾄﾞ設定されていても､ｸﾗｳﾄﾞ環境を使用しない場合､trueを指定します(初期値:false)。
	 *
	 * @og.tag
	 * ｸﾗｳﾄﾞ設定は､ｼｽﾃﾑ定数の『CLOUD_TARGET』と『CLOUD_BUCKET』の設定で自動的に使用しますが､
	 * どうしてもﾛｰｶﾙでのみ使いたい場合は､この属性を true に設定します。
	 * 標準はfalse:設定どおりとなります。
	 *
	 * true/false以外を指定した場合はfalse扱いとします。
	 *
	 * @og.rev 8.0.1.0 (2021/10/29) useLocal 属性を追加。
	 *
	 * @param flag ﾛｰｶﾙ環境のみ [true:ﾛｰｶﾙのみ/false:設定どおり]
	 */
	public void setUseLocal( final String flag ) {
		useLocal = nval( getRequestParameter( flag ),useLocal );
	}

	/**
	 * このｵﾌﾞｼﾞｪｸﾄの文字列表現を返します。
	 * 基本的にﾃﾞﾊﾞｯｸﾞ目的に使用します。
	 *
	 * @return このｸﾗｽの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "names"		,NAMES		)
				.println( "ENCODE"		,ENCODE		)
				.println( "fileURL"		,fileURL	)
				.println( "filename"	,filename	)
				.println( "maxPostSize"	,maxPostSize)
				.println( "tableId"		,tableId	)
				.println( "columns"		,columns	)				// 5.6.5.2 (2013/06/21)
				.println( "useBackup"	,useBackup	)				// 5.6.5.3 (2013/06/28)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}

/*
 * ※ ｺﾒﾝﾄが多すぎて、GF23 登録ｴﾗｰになったため、削除します。
 *
 * 5.7.1.2 (2013/12/20) zip 対応
 * filename 属性に､".zip" の拡張子のﾌｧｲﾙ名を指定した場合は､ｱｯﾌﾟﾛｰﾄﾞされた一連のﾌｧｲﾙを
 * ZIP圧縮します。これは､ｱｯﾌﾟﾛｰﾄﾞ後の処理になります。
 * ZIP圧縮のｵﾘｼﾞﾅﾙﾌｧｲﾙは､そのまま残ります。
 * なお､ZIPﾌｧｲﾙは､useBackup属性を true に設定しても､無関係に､上書きされます。
 *
 * 5.7.4.3 (2014/03/28) filename 属性のﾘｸｴｽﾄ変数対応
 * filename 属性のみ､{&#064;XXXX} のﾘｸｴｽﾄ変数が使えるようにします。
 * 他のﾊﾟﾗﾒｰﾀでは使えません。
 * これは､multipart/form-data のﾘｸｴｽﾄでは､ﾊﾟｰﾄの分解処理をしないと､ﾘｸｴｽﾄ変数が
 * 拾えない為､ﾘｸｴｽﾄ変数は､この､upload ﾀｸﾞ以降でのみ利用可能でした。
 * zip対応と関連付けて､filename 属性のみ､利用できるように､MultipartRequest 側の処理に組み込みます。
 *
 * 5.7.6.3 (2014/05/23) ｱｯﾌﾟﾛｰﾄﾞﾌｧｲﾙのCSVｾｯﾄ
 * 個々に指定したｱｯﾌﾟﾛｰﾄﾞﾌｧｲﾙ名は､XXX_NEW や XXX_ORG で取得できますが､
 * HTML5 の multiple 属性使用時や､ｱｯﾌﾟﾛｰﾄﾞされたﾌｧｲﾙを一連で処理したい場合に
 * ﾌｧｲﾙ名を､CSV形式で取り出せるようにします。
 * ｷｰは､共通で､UPLOAD_FILES とします。
 *
 * 5.9.25.0 (2017/10/06)
 * ｸﾗｳﾄﾞ上のPaaSでｵﾌﾞｼﾞｪｸﾄｽﾄﾚｰｼﾞを利用する際は以下のｼｽﾃﾑﾘｿｰｽを設定してください。
 * CLOUD_TARGET,CLOUD_BUCKET
 * plugin/cloud内のｸﾗｽを利用してﾌｧｲﾙｱｯﾌﾟﾛｰﾄﾞ(FileUploadﾀｸﾞ)､ﾀﾞｳﾝﾛｰﾄﾞ(FileDownloadｻｰﾌﾞﾚｯﾄ)をAPI経由で行います。
 * ﾌﾟﾗｸﾞｲﾝが利用するjarﾌｧｲﾙの配置は必要です。
 */
