/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import java.util.Map;
import java.util.Set;											// 6.4.3.4 (2016/03/11)

import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.util.ArraySet;						// 6.4.3.4 (2016/03/11)
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBColumnConfig;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.resource.CodeData;
import org.opengion.hayabusa.resource.LabelData;
import static org.opengion.fukurou.util.StringUtil.nval;

/**
 * 検索結果のｶﾗﾑに対して様々な属性を付加するﾀｸﾞです。
 *
 * このﾀｸﾞでは、ｶﾗﾑﾘｿｰｽの属性を再設定できます。検索結果(DBTableModel)
 * には、ｶﾗﾑｵﾌﾞｼﾞｪｸﾄ(DBColumn)を持っています。このｵﾌﾞｼﾞｪｸﾄは、
 * ｶﾗﾑﾘｿｰｽで設定した値を持っています。その個々の値を再設定することで、
 * ｶﾗﾑｵﾌﾞｼﾞｪｸﾄの属性をその都度設定しなおすことができます。
 * 最もよく使用するのは、must 属性の設定と、DBMENU ｴﾃﾞｨﾀｰの設定です。
 * must属性は、予め必須項目として指定することが出来ない為、画面毎に再設定が
 * 必要です。ただし、ここでのmust属性は、ｶﾗﾑの表示色を変更するだけで、
 * 実際のnullﾁｪｯｸは、columnCheck ﾀｸﾞのnullCheck属性で指定します。
 * DBMENU ｴﾃﾞｨﾀｰは、ﾃﾞｰﾀﾍﾞｰｽの属性から選択ﾘｽﾄを作成する機能ですが、
 * これも、ﾏｽﾀ登録時は、ﾃｷｽﾄﾌｨｰﾙﾄﾞで入力し、使用時は、選択するｹｰｽ
 * の様に、画面毎に異なる形式で使用したい場合があるためです。
 * このﾀｸﾞでは、ｶﾗﾑｵﾌﾞｼﾞｪｸﾄの属性以外に、HTMLの属性を付加する事も
 * できます。
 *
 *    6.7.6.0 (2017/03/17) [strictCheck属性]は、ｶﾗﾑIDの存在ﾁｪｯｸを行うかどうかを指定します(初期値:true)
 *      true    ｶﾗﾑIDがDBTableModel に存在しない場合は、ｴﾗｰになる。
 *      false   ｶﾗﾑIDがDBTableModel に存在しない場合は、無視する。
 *
 * @og.formSample
 * ●形式：&lt;og:ColumnEditor column=&quot;･･･&quot; ･･･ &gt; ･･･ &lt;/og:ColumnEditor &gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:columnEditor
 *       column             【TAG】ｶﾗﾑ名をｾｯﾄします
 *       columns            【TAG】ｶﾗﾑ名をCSV形式で複数ｾｯﾄします
 *       label              【TAG】ﾗﾍﾞﾙﾘｿｰｽに定義されたﾗﾍﾞﾙIDをｾｯﾄします
 *       className          【TAG】ｶﾗﾑのｸﾗｽ名(VARCHAR2,NUMBER など)をｾｯﾄします。既存値:設定値対応
 *       size               【HTML】ｶﾗﾑの入力枠ｻｲｽﾞをｾｯﾄします(大きさだけで入力文字制限ではありません)
 *       viewSize           【TAG】ｶﾗﾑの表示ｻｲｽﾞ(横幅)を文字数でｾｯﾄします。
 *       maxlength          【HTML】ｶﾗﾑの入力文字制限数をｾｯﾄします
 *       writable           【TAG】ｶﾗﾑが書き込み可能かどうか[true/false]をｾｯﾄします(初期値:null)
 *       renderer           【TAG】ｶﾗﾑの表示用ﾚﾝﾃﾞﾗｰ({@og.doc03Link renderer Renderer_****})をｾｯﾄします。既存値:設定値対応
 *       editor             【TAG】ｶﾗﾑの編集用ｴﾃﾞｨﾀｰ({@og.doc03Link editor Editor_****})をｾｯﾄします。既存値:設定値対応
 *       dbType             【TAG】DBﾀｲﾌﾟ({@og.doc03Link dbType DBType_****})をｾｯﾄします。既存値:設定値対応
 *       codeName           【TAG】ｺｰﾄﾞｾﾚｸﾄのﾕﾆｰｸｷｰ(ｺｰﾄﾞ名称)をｾｯﾄします
 *       codeList           【TAG】ｺｰﾄﾞｾﾚｸﾄのｻﾌﾞｾｯﾄとなるｺｰﾄﾞをCSV形式で指定します
 *       codeGroup          【TAG】ｺｰﾄﾞｾﾚｸﾄのｻﾌﾞｾｯﾄとなるｸﾞﾙｰﾌﾟを指定します
 *       codeKeyVal         【TAG】新しいｺｰﾄﾞｾﾚｸﾄを作成する為の｢ｷｰ:ﾗﾍﾞﾙ ｷｰ:ﾗﾍﾞﾙ｣の文字列を指定します
 *       defaultVal         【TAG】初期値をｾｯﾄします
 *       param              【TAG】ﾊﾟﾗﾒｰﾀをｾｯﾄします
 *       roles              【TAG】ﾛｰﾙをｾｯﾄします
 *       must               【TAG】必須入力を表す色に変えるかどうか[true/false]を指定します(初期値:false)
 *       mustAny            【TAG】選択必須入力(どれかひとつ必須)を表す色[true/mustAny/その他]を指定します(初期値:無指定)
 *       optionAttributes   【TAG】JavaScript などの HTML基本ﾀｸﾞ以外の属性を、そのままﾀｸﾞとして使用します
 *       addNoValue         【TAG】ｾﾚｸﾄﾒﾆｭｰの値なしのOptionの有無[true/false]を指定します(初期値:falseと同等)
 *       addKeyLabel        【TAG】ｾﾚｸﾄﾒﾆｭｰの場合、ｷｰ:ﾗﾍﾞﾙ形式で表示するかどうか[true/false/null]を指定します(初期値:null)
 *       writeKeyLabel      【TAG】ｾﾚｸﾄﾒﾆｭｰの場合、ｷｰとﾗﾍﾞﾙを分けて出力するかどうか[true/false]を指定します(初期値:false)
 *       useLabelMap        【TAG】SQL文を指定してﾗﾍﾞﾙ情報を動的に読み込むかどうか[true/false]指定します(初期値:false)
 *       useSLabel          【TAG】MENU,DBMENUでSLABELを利用するかどうか[auto/true/false]を指定します(初期値:null)
 *       noDisplayVal       【TAG】指定の値を非表示にします(ﾚﾝﾃﾞﾗｰのみ有効)(初期値:null)
 *       scope              【TAG】ｷｬｯｼｭする場合のｽｺｰﾌﾟ[request/page/session/application]を指定します(初期値:session)
 *       tableId            【TAG】(通常は使いません)sessionから所得する DBTableModelｵﾌﾞｼﾞｪｸﾄの ID(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])
 *       dbid               【TAG】(通常は使いません)ﾃﾞｰﾀﾍﾞｰｽへ接続する場合のID(初期値:null)
 *       caseKey            【TAG】このﾀｸﾞ自体を利用するかどうかの条件ｷｰを指定します(初期値:null)
 *       caseVal            【TAG】このﾀｸﾞ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ｾﾞﾛ文字列 でない場合(Not Null=NN)は、このﾀｸﾞは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ｾﾞﾛ文字列 の場合は、このﾀｸﾞは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このﾀｸﾞは使用されます(初期値:判定しない)
 *       lang               【HTML】要素の内容と他の属性値の言語(lang,xml:lang)を指定します
 *       style              【HTML】この要素に対して適用させるｽﾀｲﾙｼｰﾄ(style)を設定します
 *       clazz              【HTML】要素に対して class 属性を設定します
 *       readonly           【HTML】その部品に対して変更が出来ないように(readonly)指定します(ｻｰﾊﾞｰに送信される) 8.4.3.0 (2023/03/31)
 *       disabled           【HTML】その部品に対して、選択や変更が出来ないように(disabled)指定します(ｻｰﾊﾞｰに送信されない) 8.4.3.0 (2023/03/31)
 *       onClick            【HTML】JavaScriptのｲﾍﾞﾝﾄ onClick を設定します(例:onClick="renew('query.jsp','QUERY');")
 *       onChange           【HTML】JavaScriptのｲﾍﾞﾝﾄ onChange を設定します(例:onChange="renew('query.jsp','QUERY');")
 *       onBlur             【HTML】JavaScriptのｲﾍﾞﾝﾄ onBlur を設定します(例:onBlur="this.value=value.toUpperCase();")
 *       onFocus            【HTML】JavaScriptのｲﾍﾞﾝﾄ onFocus を設定します
 *       ondblClick         【HTML】JavaScriptのｲﾍﾞﾝﾄ ondblClick を設定します
 *       onMouseDown        【HTML】JavaScriptのｲﾍﾞﾝﾄ onMouseDown を設定します
 *       onMouseUp          【HTML】JavaScriptのｲﾍﾞﾝﾄ onMouseUp を設定します
 *       onMouseMove        【HTML】JavaScriptのｲﾍﾞﾝﾄ onMouseMove を設定します
 *       onMouseOut         【HTML】JavaScriptのｲﾍﾞﾝﾄ onMouseOut を設定します
 *       onMouseOver        【HTML】JavaScriptのｲﾍﾞﾝﾄ onMouseOver を設定します
 *       onSelect           【HTML】JavaScriptのｲﾍﾞﾝﾄ onSelect を設定します
 *       onKeydown          【HTML】JavaScriptのｲﾍﾞﾝﾄ onKeydown を設定します
 *       onKeypress         【HTML】JavaScriptのｲﾍﾞﾝﾄ onKeypress を設定します
 *       onKeyup            【HTML】JavaScriptのｲﾍﾞﾝﾄ onKeyup を設定します
 *       autocomplete       【HTML5】入力候補を提示して入力内容を自動補完する[on/off](初期値はon)
 *       autofocus          【HTML5】指定した入力欄にｶｰｿﾙが当たって自動的にﾌｫｰｶｽされます
 *       pattern            【HTML5】正規表現で入力値のﾊﾟﾀｰﾝを指定します
 *       placeholder        【HTML5】入力欄に初期表示する内容を指定します
 *       required           【HTML5】入力必須を指定します
 *       list               【HTML5】ﾕｰｻﾞｰに入力候補として提案するﾃﾞｰﾀﾘｽﾄの要素のid属性の値を指定します
 *       min                【HTML5】入力欄で入力できる最小値を指定します
 *       max                【HTML5】入力欄で入力できる最大値を指定します
 *       step               【HTML5】入力欄で刻むｽﾃｯﾌﾟ値を指定する
 *       type               【HTML5】HTML5追加分[text/search/tel/url/email/datetime/date/month/week/time/datetime-local/number/range/color]
 *       eventColumn        【TAG】ｲﾍﾞﾝﾄｶﾗﾑ(親ｶﾗﾑ)のｾｯﾄ
 *       eventValue         【TAG】ｲﾍﾞﾝﾄValueのSQL分
 *       eventURL           【TAG】eventColumn利用時に呼ばれるJSPのURL
 *       eventCallback      【TAG】eventColumn実行後のcallbak関数指定を指定します
 *       noResource         【TAG】ResourceManager を使用しないかどうか[true/false]を指定します(初期値:false)
 *       strictCheck        【TAG】(通常は使いません)ｶﾗﾑIDの存在ﾁｪｯｸを行うかどうか[true/false]を指定します(初期値:true)
 *       stringOutput       【TAG】ﾚﾝﾃﾞﾗ利用のExcel出力時にｾﾙﾀｲﾌﾟを文字列にするかどうか(初期値:true)
 *       useRealTimeCheck   【TAG】(通常は使いません)ﾘｱﾙﾀｲﾑﾁｪｯｸを行うかどうか(初期値:true:行う) 5.9.32.2 (2018/05/18)
 *       useDateFeed        【TAG】日付送り戻し機能を有効にするかどうか[true:有効/false:無効]指定します (初期値:true) 8.1.2.3 (2022/05/20)
 *       debug              【TAG】ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:columnEditor&gt;
 *
 * ●使用例：
 *     ･必須表示の場合
 *         &lt;og:columnEditor column=&quot;NOSYN&quot; must=&quot;true&quot; /&gt;
 *
 *     ･DBからのｾﾚｸﾄﾒﾆｭｰの場合
 *         &lt;og:columnEditor column=&quot;CDBK&quot; editor=&quot;DBMENU&quot;&gt;
 *                 select CDBK,CDBK||&#39;:&#39;||NMBK from DB02 order by 1
 *         &lt;/og:columnEditor&gt;
 *
 *     ･親子関係のﾌﾟﾙﾀﾞｳﾝを作る場合
 *       ({&#064;xxx.value}で同じ行のxxxｶﾗﾑで選択した値を取得します)
 *         &lt;og:columnEditor column=&quot;CDBK&quot; editor=&quot;DBMENU&quot;
 *              eventColumn="OYA"&gt;
 *                 select CDBK,CDBK||&#39;:&#39;||NMBK from DB02
 *                   where CLM = {&#064;OYA.value} order by 1
 *         &lt;/og:columnEditor&gt;
 *
 *
 * @og.rev 3.5.1.0 (2003/10/03) 親ｸﾗｽをCommonTagSupportからHTMLTagSupportに変更。
 * @og.group 画面表示
 *
 * @version	4.0
 * @author	Kazuhiko Hasegawa
 * @since	JDK5.0,
 */
public class ColumnEditorTag extends HTMLTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "8.1.2.3 (2022/05/20)" ;
	private static final long serialVersionUID = 812320220520L ;

	// 5.7.1.0 (2013/12/06) HTML5 で新たに追加された、type を追加
	// 6.3.4.0 (2015/08/01) String から String配列 に置き換え

	// 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	private static final Set<String> TYPE_SET = new ArraySet<>( "text","search","tel","url","email","datetime","date","month","week","time","datetime-local","number","range","color" );

	private static final String DEFAULT_ADD_NO_VALUE= "true";	// 3.7.0.5 (2005/04/11)

	private transient	LabelData				labelData	;

	/** 6.4.3.1 (2016/02/12) 取得元が ConcurrentHashMap で返す。 */
	private transient	Map<String,LabelData>	labelMap	;					// 6.3.9.0 (2015/11/06) transient 追加

	private String		tableId			= HybsSystem.TBL_MDL_KEY;
	private String[]	columns			;										// 3.7.0.5 (2005/04/11)
	private boolean		allColumns		;										// 3.8.8.5 (2007/03/09)
	private String		className		;
	private String		fieldSize		;										// 6.2.0.0 (2015/02/27)
	private String		viewSize		;										// 6.2.0.0 (2015/02/27)
	private String		maxlength		;										// 3.5.5.5 (2004/04/23)
	private String		writable		;
	private String		renderer		;
	private String		editor			;
	private String		dbType			;
	private String		codeName		;										// 3.5.5.7 (2004/05/10)
	/** Codeﾘｿｰｽのｻﾌﾞｾｯﾄ */
	private String		codeList		;										// 5.1.9.0 (2010/08/01)
	/** Codeﾘｿｰｽのｻﾌﾞｾｯﾄとなるｸﾞﾙｰﾌﾟ */
	private String		codeGroup		;										// 5.1.9.0 (2010/08/01)
	/** 新しいｺｰﾄﾞｾﾚｸﾄを作成する為の KEY:VAL 文字列 */
	private String		codeKeyVal		;										// 6.3.1.1 (2015/07/10)
	private String		defaultVal		;
	private String		parameter		;
	private String		dbid			;
	private String		addNoValue		;										// 3.5.5.9 (2004/06/07)
	/** ｷｰ:ﾗﾍﾞﾙ形式で表示するかどうか */
	private String		addKeyLabel		;										// 6.0.4.0 (2014/11/28)
	private boolean		writeKeyLabel	;										// 6.2.3.0 (2015/05/01)

	/** 親ｶﾗﾑの指定 */
	private String		eventColumn		;										// 4.3.6.0 (2009/04/01)
	/** ｲﾍﾞﾝﾄｶﾗﾑの初期値 */
	private String		eventValue		;										// 6.3.4.0 (2015/08/01)
	/** 生ﾊﾟﾗﾒｰﾀ */
	private String		rawParameter	;										// 4.3.6.0 (2009/04/01)
	/** eventColumnで飛ばすURL */
	private String		eventURL		;										// 4.3.6.0 (2009/04/01)

	/** ﾗﾍﾞﾙﾃﾞｰﾀの動的取得に対応 */
	private boolean		useLabelMap		;										// 4.3.4.0 (2008/12/01)
	/** MENU,DBMENUでSLABELを利用するかどうか */
	private String		useSLabel		;										// 5.6.2.3 (2013/03/22)
	/** 非表示文字の設定 */
	private String		noDisplayVal	;										// 5.6.2.3 (2013/03/22)

	// 5.7.3.0 (2014/02/07) ResourceManager を使用しないかどうか[true/false]を指定
	private boolean		noResource		;
	/** ｶﾗﾑIDの存在ﾁｪｯｸを行うかどうか */
	private boolean		strictCheck		= true;									// 6.7.6.0 (2017/03/17)

	private boolean		stringOutput	= HybsSystem.sysBool( "USE_STRING_EXCEL_OUTPUT" );	// 5.7.6.3 (2014/05/23)
	/** 日付送り戻し機能の有効/無効 */
	private boolean		useDateFeed		= true;									// 8.1.2.3 (2022/05/20)

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰ
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public ColumnEditorTag() { super(); }		// これも、自動的に呼ばれるが、空のﾒｿｯﾄﾞを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始ﾀｸﾞが見つかったときに処理する doStartTag() を ｵｰﾊﾞｰﾗｲﾄﾞします。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) DBColumnConfig#getColumnConfig() ﾒｿｯﾄﾞ廃止
	 * @og.rev 3.1.1.0 (2003/03/28) ﾎﾞﾃﾞｨの内容を取得する処理を、CommonTagSupport で行う。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。ﾎﾞﾃﾞｨが存在する場合のみﾎﾞﾃﾞｨが呼ばれる対応。
	 * @og.rev 3.5.1.0 (2003/10/03) 親ｸﾗｽをCommonTagSupportからHTMLTagSupportに変更。
	 * @og.rev 3.5.5.5 (2004/04/23) 入力欄の大きさを指定する viewSize 属性を追加。
	 * @og.rev 3.5.5.7 (2004/05/10) CodeSelection を、codeName を元に構築できるように設定します。
	 * @og.rev 3.5.5.9 (2004/06/07) addNoValue が、設定された場合のみ、config に設定します
	 * @og.rev 3.7.0.5 (2005/04/11) 複数ｶﾗﾑを同時設定できる columns 属性を追加
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		// 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
		if( useTag() ) {
			if( "DBMENU".equalsIgnoreCase(editor) ) {
				className = "VARCHAR2";
				renderer  = "DBMENU";
				editor    = "DBMENU";

				if( addNoValue == null ) {
					addNoValue = DEFAULT_ADD_NO_VALUE ;
				}
			}

			return EVAL_BODY_BUFFERED ;
		}
		return SKIP_BODY ;				// Body を評価しない
	}

	/**
	 * Taglibのﾀｸﾞ本体を処理する doAfterBody() を ｵｰﾊﾞｰﾗｲﾄﾞします。
	 *
	 * @og.rev 2.3.1.3 (2003/02/08) ﾗﾍﾞﾙ、ｻｲｽﾞのみでも変更できるように修正
	 * @og.rev 3.0.0.1 (2003/02/14) DBMENU作成時に、ﾗﾍﾞﾙの最後のｽﾍﾟｰｽを除去する。
	 * @og.rev 3.1.0.0 (2003/03/20) DBColumnConfig#getColumnConfig() ﾒｿｯﾄﾞ廃止
	 * @og.rev 3.1.1.0 (2003/03/28) ﾎﾞﾃﾞｨの内容を取得する処理を、CommonTagSupport で行う。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。ﾎﾞﾃﾞｨが存在する場合のみﾎﾞﾃﾞｨが呼ばれる対応。
	 * @og.rev 3.5.0.0 (2003/09/17) ｷｰ、ﾗﾍﾞﾙともに、""(ﾀﾞﾌﾞﾙｺｰﾃｰｼｮﾝ)で囲うように変更。
	 * @og.rev 3.5.5.7 (2004/05/10) このﾀｸﾞでSQLを発行するのではなく、DBMENU の DBColumnを作成する。
	 * @og.rev 3.5.5.9 (2004/06/07) addNoValue が null の場合は、初期値を設定します。
	 * @og.rev 3.7.0.5 (2005/04/11) 複数ｶﾗﾑを同時設定できる columns 属性を追加
	 * @og.rev 3.8.6.3 (2006/11/30) SQL 文の前後のｽﾍﾟｰｽを取り除きます。
	 * @og.rev 4.3.4.0 (2008/12/01) ﾗﾍﾞﾙﾃﾞｰﾀの動的取得に対応します。
	 * @og.rev 4.3.6.0 (2009/04/01) ｲﾍﾞﾝﾄｶﾗﾑ対応
	 * @og.rev 5.1.7.0 (2010/06/01) 動的ﾌﾟﾙﾀﾞｳﾝ実装見直し
	 * @og.rev 5.5.4.0 (2012/07/02) 予約語対応
	 * @og.rev 6.3.1.1 (2015/07/10) BodyString,BodyRawStringは、CommonTagSupport で、trim() します。
	 * @og.rev 6.4.0.5 (2016/01/09) useLabelMap="true" 時のSQL文の実行は、dbid を使用して行う。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		final String sql = getBodyString();
		if( sql != null && sql.length() > 0 ) {
			if( useLabelMap )	{
				// 6.4.0.5 (2016/01/09) useLabelMap="true" 時のSQL文の実行は、dbid を使用して行う。
				labelMap = getResource().getLabelMap( sql.trim() , dbid );		// 4.3.4.0 (2008/12/01)
			}
			else				{
				if( eventColumn != null && eventColumn.length() >0 ){			// 4.3.6.0 (2009/04/01)
					rawParameter = getReservedParameter( getBodyRawString() );	// 6.3.1.1 (2015/07/10)
				}
				parameter = sql.trim() ;
			}
		}

		return SKIP_BODY ;
	}

	/**
	 * Taglibの終了ﾀｸﾞが見つかったときに処理する doEndTag() を ｵｰﾊﾞｰﾗｲﾄﾞします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。ﾎﾞﾃﾞｨが存在する場合のみﾎﾞﾃﾞｨが呼ばれる対応。
	 * @og.rev 3.7.0.5 (2005/04/11) 複数ｶﾗﾑを同時設定できる columns 属性を追加
	 * @og.rev 3.8.8.5 (2007/03/09) 全ｶﾗﾑ が指定されたことにする allColumns 属性を追加
	 * @og.rev 4.1.2.1 (2008/03/13) must , mustAny 属性を自動化します。
	 * @og.rev 4.3.4.0 (2008/12/01) ｶﾗﾑ指定のﾜｲﾙﾄﾞｶｰﾄﾞ対応
	 * @og.rev 5.2.1.0 (2010/10/01) must , mustAny 属性の処理を、HTMLTagSupport に移します。
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
	 * @og.rev 6.7.6.0 (2017/03/17) strictCheck 追加。
	 * @og.rev 6.8.1.4 (2017/08/25) must="false"で、config.setEditorAttributes で、属性を追加するようになった為の取り消しの対応。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		// 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
		if( !useTag() ) { return EVAL_PAGE; }

		final DBTableModel table = (DBTableModel)getObject( tableId );
		if( table == null ) { return EVAL_PAGE; }

		if( columns != null ) {
			// 3.8.8.5 (2007/03/09) 全ｶﾗﾑ が指定
			final String mustType = getMustType();	// 5.2.1.0 (2010/10/01)
			if( allColumns ) {
				final int len = table.getColumnCount();
				for( int i=0; i<len; i++ ) {

					// 4.3.4.0 (2008/12/01)
					final DBColumn dbColumn = table.getDBColumn( i );
					for( int j=0; j<columns.length; j++ ) {
						final String prefix = columns[j].replace( "*", "" );
						if( dbColumn.getName().indexOf( prefix ) == 0 ) { // 前方一致のみ対応
							// 6.8.1.4 (2017/08/25) must="false"で、config.setEditorAttributes で、属性を追加するようになった為の取り消しの対応。
							final DBColumnConfig config = getModifyConfig( dbColumn );

							// 4.1.2.1 (2008/03/13)
							if( mustType != null ) {
								table.addMustType( i,mustType );
								// 6.8.1.4 (2017/08/25) must="false" で、must の取り消し
								if( "false".equalsIgnoreCase( mustType ) ) {
									config.removeEditorAttributes( "class" , "must" );
								}
							}

							table.setDBColumn( i,new DBColumn( config ) );

							break;
						}
					}
				}
			}
			else {
				for( int i=0; i<columns.length; i++ ) {
					final int clmNo = table.getColumnNo( columns[i],strictCheck );		// 6.7.6.0 (2017/03/17)
					if( clmNo < 0 ) { continue; }										// 6.7.6.0 (2017/03/17)

					final DBColumn dbColumn = table.getDBColumn( clmNo );

					final DBColumnConfig config = getModifyConfig( dbColumn );
					// 6.8.1.4 (2017/08/25) must="false"で、config.setEditorAttributes で、属性を追加するようになった為の取り消しの対応。

					// 4.1.2.1 (2008/03/13)
					if( mustType != null ) {
						table.addMustType( clmNo,mustType );
						// 6.8.1.4 (2017/08/25) must="false" で、must の取り消し
						if( "false".equalsIgnoreCase( mustType ) ) {
							config.removeEditorAttributes( "class" , "must" );
						}
					}

					table.setDBColumn( clmNo,new DBColumn( config ) );
				}
			}
		}

		return EVAL_PAGE ;
	}

	/**
	 * ﾀｸﾞ文字列を作成します。
	 *
	 * ColumnEditor としては、HTMLTagSupport を継承しているため、
	 * このﾒｿｯﾄﾞを定義する必要がありますが、内部的には使用していません。
	 * 各種属性は、入力する必要がある為、継承しているだけです。
	 * このｸﾗｽでは、このﾒｿｯﾄﾞは、使われておりません。
	 *
	 * @og.rev 3.5.1.0 (2003/10/03) 新規追加
	 *
	 * @return	ﾀｸﾞ文字列(常に null を返す)
	 */
	@Override
	protected String makeTag() {
		return null;
	}

	/**
	 * ﾀｸﾞﾘﾌﾞｵﾌﾞｼﾞｪｸﾄをﾘﾘｰｽします。
	 *
	 * ｷｬｯｼｭされて再利用されるので、ﾌｨｰﾙﾄﾞの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) ｶｽﾀﾑﾀｸﾞの release() ﾒｿｯﾄﾞを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.1.0 (2003/10/03) 親ｸﾗｽをCommonTagSupportからHTMLTagSupportに変更。
	 * @og.rev 3.5.5.5 (2004/04/23) 入力欄の大きさを指定する viewSize 属性を追加。
	 * @og.rev 3.5.5.7 (2004/05/10) codeName 属性を追加。
	 * @og.rev 3.5.5.9 (2004/06/07) addNoValue 属性を修正。
	 * @og.rev 3.7.0.5 (2005/04/11) 複数ｶﾗﾑを同時設定できる columns 属性を追加
	 * @og.rev 3.8.8.5 (2007/03/09) 全ｶﾗﾑ が指定されたことにする allColumns 属性を追加
	 * @og.rev 4.1.2.1 (2008/03/13) must , mustAny 属性を自動化します。
	 * @og.rev 4.3.4.0 (2008/12/01) ﾗﾍﾞﾙﾃﾞｰﾀの動的取得に対応します。
	 * @og.rev 5.1.9.0 (2010/08/01) codeList , codeGroup 属性を追加。(Codeﾘｿｰｽのｻﾌﾞｾｯﾄ)
	 * @og.rev 5.2.1.0 (2010/10/01) must , mustAny 属性の処理を、HTMLTagSupport に移します。
	 * @og.rev 5.6.2.3 (2013/03/22) useSLabel 属性を追加。(MENU,DBMENUでSLABELを利用するかどうか)
	 * @og.rev 5.6.2.3 (2013/03/22) 非表示文字の設定。noDisplayVal 属性を追加。
	 * @og.rev 5.7.3.0 (2014/02/07) noResource 属性を追加。ResourceManager を使用しないかどうか
	 * @og.rev 5.7.6.3 (2014/05/23) stringOutput追加
	 * @og.rev 6.0.4.0 (2014/11/28) addKeyLabel 属性を追加
	 * @og.rev 6.2.0.0 (2015/02/27) viewSize 属性の追加
	 * @og.rev 6.2.3.0 (2015/05/01) ｷｰ,ﾗﾍﾞﾙで分離して出力するかどうか。writeKeyLabel 属性を追加
	 * @og.rev 6.3.1.1 (2015/07/10) codeKeyVal 属性を追加
	 * @og.rev 6.3.4.0 (2015/08/01) eventValue 追加
	 * @og.rev 6.7.6.0 (2017/03/17) strictCheck 追加
	 * @og.rev 8.1.2.3 (2022/05/20) 日付送り戻し不具合対応(useDateFeed 属性追加)
	 */
	@Override
	protected void release2() {
		super.release2();
		tableId			= HybsSystem.TBL_MDL_KEY;
		columns			= null;													// 3.7.0.5 (2005/04/11)
		allColumns		= false;												// 3.8.8.5 (2007/03/09)
		labelData		= null;													// 4.0.0 (2005/01/31)
		className		= null;
		fieldSize		= null;													// 6.2.0.0 (2015/02/27)
		viewSize		= null;													// 6.2.0.0 (2015/02/27)
		maxlength		= null;													// 3.5.5.5 (2004/04/23)
		writable		= null;
		renderer		= null;
		editor			= null;
		dbType			= null;
		codeName		= null;													// 3.5.5.7 (2004/05/10)
		codeList		= null;													// 5.1.9.0 (2010/08/01) 新規追加(Codeﾘｿｰｽのｻﾌﾞｾｯﾄ)
		codeGroup		= null;													// 5.1.9.0 (2010/08/01) 新規追加(Codeﾘｿｰｽのｻﾌﾞｾｯﾄ)
		codeKeyVal		= null;													// 6.3.1.1 (2015/07/10) 新規追加(新しいｺｰﾄﾞｾﾚｸﾄを作成する為の KEY:VAL 文字列)
		defaultVal		= null;
		parameter		= null;
		dbid			= null;
		addNoValue		= null;													// 3.5.5.9 (2004/06/07)
		addKeyLabel		= null;													// 6.0.4.0 (2014/11/28) ｷｰ:ﾗﾍﾞﾙ形式で表示するかどうか
		writeKeyLabel	= false;												// 6.2.3.0 (2015/05/01) ｷｰ,ﾗﾍﾞﾙで分離して出力するかどうか
		useLabelMap		= false;												// 4.3.4.0 (2008/12/01)
		useSLabel		= null;													// 5.6.2.3 (2013/03/22) MENU,DBMENUでSLABELを利用するかどうか
		noDisplayVal	= null;													// 5.6.2.3 (2013/03/22) 非表示文字の設定
		labelMap		= null;													// 4.3.4.0 (2008/12/01)
		eventColumn		= null;													// 4.3.6.0 (2009/04/01)
		eventValue		= null;													// 6.3.4.0 (2015/08/01)
		rawParameter	= null;													// 4.3.6.0 (2009/04/01)
		eventURL		= null;													// 4.3.6.0 (2009/04/01)
		noResource		= false;												// 5.7.3.0 (2014/02/07) noResource 属性を追加。
		strictCheck		= true;													// 6.7.6.0 (2017/03/17)
		stringOutput	= HybsSystem.sysBool( "USE_STRING_EXCEL_OUTPUT" );		// 5.7.6.3 (2014/05/23)
		useDateFeed		= true;													// 8.1.2.3 (2022/05/20)
	}

	/**
	 * ｶｽﾀﾏｲｽﾞされた DBColumnConfig を作成します。
	 *
	 * DBColumnConfig は、DBColumn より取得します。この DBColumnConfig に、
	 * 属性で指定された値をｾｯﾄすることで、ｶｽﾀﾏｲｽﾞを行います。
	 *
	 * @og.rev 3.7.0.5 (2005/04/11) 新規追加
	 * @og.rev 4.1.2.1 (2008/03/13) must , mustAny 属性を自動化します。
	 * @og.rev 4.3.4.0 (2008/12/01) ﾗﾍﾞﾙﾃﾞｰﾀの動的取得に対応します。
	 * @og.rev 4.3.6.0 (2009/04/01) evnetColumn対応
	 * @og.rev 5.1.8.0 (2010/07/01) ﾒｿｯﾄﾞ名変更(setDefValue ⇒ setDefault)
	 * @og.rev 5.1.9.0 (2010/08/01) codeList ,codeGroup 属性を追加(Codeﾘｿｰｽのｻﾌﾞｾｯﾄ)
	 * @og.rev 5.6.2.3 (2013/03/22) useSLabel 属性を追加。(MENU,DBMENUでSLABELを利用するかどうか)
	 * @og.rev 5.6.2.3 (2013/03/22) 非表示文字の設定。noDisplayVal 属性を追加。
	 * @og.rev 5.7.3.0 (2014/02/07) noResource 属性を追加。ResourceManager を使用しないかどうか
	 * @og.rev 5.7.6.2 (2014/05/16) IEのHTML5機能が無効の場合の処理
	 * @og.rev 5.7.6.3 (2014/05/23) stringOutput追加
	 * @og.rev 6.0.4.0 (2014/11/28) addKeyLabel 属性を追加
	 * @og.rev 6.2.0.0 (2015/02/27) viewSize 属性の追加
	 * @og.rev 6.2.3.0 (2015/05/01) maxlength 変数は、0 の場合は、無制限になります。
	 * @og.rev 6.3.1.0 (2015/06/28) 個人ﾘｿｰｽのUSE_ADD_KEY_LABELで初期値設定
	 * @og.rev 6.3.1.1 (2015/07/10) codeKeyVal 属性を追加
	 * @og.rev 6.3.4.0 (2015/08/01) eventValue 追加
	 * @og.rev 5.9.18.1 (2017/03/24) DATALIST_R用のIEのHTML5機能が無効の場合の処理を追加
	 *
	 * @param	dbColumn	ｶﾗﾑｵﾌﾞｼﾞｪｸﾄ
	 *
	 * @return	ｶｽﾀﾏｲｽﾞされた DBColumnConfig
	 */
	private DBColumnConfig getModifyConfig( final DBColumn dbColumn ) {
		// 5.7.3.0 (2014/02/07) noResource 属性を追加
		if( noResource ) {
			labelData = new LabelData( dbColumn.getName() );
			renderer  = "LABEL";
		}

		final DBColumnConfig config = dbColumn.getConfig();

		// 5.7.6.2 (2014/05/16) IEのHTML5機能が無効の場合の処理
		final String ieHTML5 = (String)getSessionAttribute( HybsSystem.IE_HTML5_KEY );
		if( "FALSE".equalsIgnoreCase( ieHTML5 ) ) {
			if( "DATALIST".equalsIgnoreCase( editor ) ||
				"DATALIST".equalsIgnoreCase( config.getEditor() ) ) {
					editor = "INDBMENU";
			}
			// 5.9.18.1 (2017/03/24) IEのHTML5機能が無効の場合の処理(DATALIST_R用)
			if( "DATALIST_R".equalsIgnoreCase( editor ) ||
				"DATALIST_R".equalsIgnoreCase( config.getEditor() )){
					editor = "INMENU";
			}
		}

		// 6.3.1.0 (2015/06/28) 個人ﾘｿｰｽのUSE_ADD_KEY_LABELで初期値設定
		if( addKeyLabel	 == null ) { addKeyLabel = nval( sys( "USE_ADD_KEY_LABEL" ) , null ); }		// 4:個人設定可

		final String lang = get( "lang" );	// 4.0.0 (2005/01/31) super ｸﾗｽで受けます。

		if( lang		!= null ) { config.setLang(		 lang		); }
		if( labelData	!= null ) { config.setLabelData( labelData	); }		// 4.0.0 (2005/01/31)
		if( className	!= null ) { config.setClassName( className	); }
		if( fieldSize	!= null ) { config.setFieldSize( fieldSize	); }		// 6.2.0.0 (2015/02/27) ﾒｿｯﾄﾞ名変更
		if( viewSize	!= null ) { config.setViewLength( viewSize	); }		// 6.2.0.0 (2015/02/27) 新規追加
		if( maxlength	!= null ) { config.setMaxlength( maxlength	); }		// 3.5.5.5 (2004/04/23)
		if( writable	!= null ) { config.setWritable( writable	); }
		if( renderer	!= null ) { config.setRenderer( renderer	); }
		if( editor		!= null ) { config.setEditor( editor 		); }
		if( dbType		!= null ) { config.setDbType( dbType 		); }
		if( defaultVal	!= null ) { config.setDefault( defaultVal	); }		// 5.1.8.0 (2010/07/01)
		if( parameter	!= null ) { config.setParameter( parameter	); }
		if( dbid		!= null ) { config.setDbid( dbid			); }
		if( eventColumn	!= null ) { config.setEventColumn( eventColumn ); }		// 4.3.6.0 (2009/04/01)
		if( eventValue	!= null ) { config.setEventValue(  eventValue  ); }		// 6.3.4.0 (2015/08/01)
		if( eventURL	!= null ) { config.setEventURL( eventURL	   ); }		// 4.3.6.0 (2009/04/01)
		if( rawParameter != null ){ config.setRawParameter( rawParameter ); }	// 4.3.6.0 (2009/04/01)
		if( useSLabel	 != null ) { config.setUseSLabel( useSLabel  ); }		// 5.6.2.3 (2013/03/22)
		if( noDisplayVal != null ) { config.setNoDisplayVal( noDisplayVal ); }	// 5.6.2.3 (2013/03/22)
		if( addKeyLabel	 != null ) { config.setAddKeyLabel( addKeyLabel	); }	// 6.0.4.0 (2014/11/28)

		if( writeKeyLabel ) { config.setWriteKeyLabel( writeKeyLabel ); }		// 6.2.3.0 (2015/05/01)
		if( !useDateFeed )  { config.setUseDateFeed( useDateFeed ); }			// 8.1.2.3 (2022/05/20)

		// 4.0.0 (2005/11/30) ﾛｰﾙを指定
		final String roles = get( "roles" );
		if( roles != null ) { config.setRoles( roles ); }

		// 4.0.0 (2005/01/31)
		if( codeName != null ) {
			final CodeData codeData = getResource().getCodeData( codeName ) ;
			config.setCodeData( codeData );
		}

		// codeName で指定のﾘｿｰｽでも、ｻﾌﾞｾｯﾄが作成できます。
		// 5.1.9.0 (2010/08/01) codeList 属性を追加(Codeﾘｿｰｽのｻﾌﾞｾｯﾄ)
		if( codeList != null ) {
			final CodeData codeData = config.getCodeData() ;
			if( codeData != null ) {
				config.setCodeData( codeData.subsetList( codeList ) );
			}
		}

		// 5.1.9.0 (2010/08/01) codeGroup 属性を追加(Codeﾘｿｰｽのｻﾌﾞｾｯﾄ)
		if( codeGroup != null ) {
			final CodeData codeData = config.getCodeData() ;
			if( codeData != null && codeData.useGroup() ) {
				config.setCodeData( codeData.subsetGroup( codeGroup ) );
			}
		}

		// 6.3.1.1 (2015/07/10) 新規追加(新しいｺｰﾄﾞｾﾚｸﾄを作成する為の KEY:VAL 文字列)
		if( codeKeyVal != null ) {
			config.setParameter( codeKeyVal );
			config.setRenderer( "KVMENU" );
			config.setEditor(   "KVMENU" );
		}

		// 3.5.5.9 (2004/06/07)
		// 注意：addNoValue は、互換性の関係で、値が指定された場合のみ、
		// config に設定します。
		// editor="DBMENU" が、指定された場合は、addNoValue が指定されていない場合のみ、
		// 従来との互換性の関係より、このｸﾗｽのﾃﾞﾌｫﾙﾄ(true)を、config に設定します。
		if( addNoValue != null ) {
			// Boolean.parseBoolean は、引数がnullの場合は、falseを返すが、値が設定されている時のみｾｯﾄするので、nullﾁｪｯｸしている。
			config.setAddNoValue( Boolean.parseBoolean( addNoValue ) );	// 6.1.0.0 (2014/12/26) refactoring
		}

		// 3.5.1.0 (2003/10/03) 親ｸﾗｽをCommonTagSupportからHTMLTagSupportに変更。
		config.setEditorAttributes( getAttributes() );

		// 4.3.4.0 (2008/12/01) ﾗﾍﾞﾙﾃﾞｰﾀの動的取得に対応します。
		if( useLabelMap && labelMap != null ) {
			final LabelData ldata = labelMap.get( dbColumn.getName() );
			if( ldata != null ) {
				config.setLabelData( ldata );
			}
		}

		// 5.7.6.3 (2014/05/23)
		config.setStringOutput( stringOutput );

		return config ;
	}

	/**
	 * 【TAG】ｶﾗﾑ名をｾｯﾄします。
	 *
	 * @og.tag
	 * ｶﾗﾑ名をｾｯﾄします。
	 * columns と同時にはｾｯﾄできません。
	 * 値に"*"を指定することで、全ｶﾗﾑを対象とすることができます。
	 * また、"PN*,CDK*"等のように前方一致でのﾜｲﾙﾄﾞｶｰﾄﾞ指定も可能です。
	 *
	 * @og.rev 3.7.0.5 (2005/04/11) 複数ｶﾗﾑを同時設定できる columns 属性を追加
	 * @og.rev 3.8.8.5 (2007/03/09) 全ｶﾗﾑ が指定されたことにする allColumns 属性を追加
	 *
	 * @param	clm	ｶﾗﾑ名
	 */
	public void setColumn( final String clm ) {
		if( columns != null ) {
			final String errMsg = "columns との同時セットは出来ません。: " + CR
						+ "column=[" + clm + "] , "
						+ "columns=[" + StringUtil.array2csv( columns ) + "]";
			throw new HybsSystemException( errMsg );
		}
		final String column = getRequestParameter( clm );
		if( column != null && column.length() > 0 ) {
			columns = new String[] { column };
			allColumns = column.indexOf( '*' ) >= 0 ;	// 3.8.8.5 (2007/03/09)
		}
	}

	/**
	 * 【TAG】ｶﾗﾑ名をCSV形式で複数ｾｯﾄします。
	 *
	 * @og.tag
	 * ｶﾗﾑ名をｾｯﾄします。
	 * column と同時にはｾｯﾄできません。
	 * 分解方法は、通常のﾊﾟﾗﾒｰﾀ取得後に、CSV分解します。
	 * 値に"*"を指定することで、全ｶﾗﾑを対象とすることができます。
	 * また、"PN*,CDK*"等のように前方一致でのﾜｲﾙﾄﾞｶｰﾄﾞ指定も可能です。
	 *
	 * @og.rev 3.7.0.5 (2005/04/11) 複数ｶﾗﾑを同時設定できる columns 属性を追加
	 * @og.rev 3.8.8.5 (2007/03/09) 通常のﾊﾟﾗﾒｰﾀ取得後に、CSV分解に戻します。
	 * @og.rev 3.8.8.5 (2007/03/09) 全ｶﾗﾑ が指定されたことにする allColumns 属性を追加
	 *
	 * @param	clms	ｶﾗﾑ名(CSV形式)
	 */
	public void setColumns( final String clms ) {
		if( columns != null ) {
			final String errMsg = "column との同時セットは出来ません。: " + CR
						+ "column=[" + StringUtil.array2csv( columns ) + "] , "
						+ "columns=[" + clms + "]" ;
			throw new HybsSystemException( errMsg );
		}
		final String column = nval( getRequestParameter( clms ),null );
		if( column != null ) {
			allColumns = column.indexOf( '*' ) >= 0 ;		// 3.8.8.5 (2007/03/09)
			columns = StringUtil.csv2Array( column );
			if( columns.length == 0 ) { columns = null; }
		}
	}

	/**
	 * 【TAG】ﾗﾍﾞﾙﾘｿｰｽに定義されたﾗﾍﾞﾙIDをｾｯﾄします。
	 *
	 * @og.tag
	 * ﾗﾍﾞﾙをｾｯﾄします。
	 *
	 * @og.rev 2.3.1.2 (2003/01/28) ﾗﾍﾞﾙﾘｿｰｽからﾗﾍﾞﾙ名を取得するように修正
	 * @og.rev 4.0.0.0 (2005/01/31) ｷｰより、ﾗﾍﾞﾙﾃﾞｰﾀｵﾌﾞｼﾞｪｸﾄを取得します。
	 *
	 * @param	lbl	ﾗﾍﾞﾙ
	 */
	public void setLabel( final String lbl ) {
		labelData = getResource().getLabelData( getRequestParameter( lbl ) ) ;	// 4.0.0 (2005/01/31)
	}

	/**
	 * 【TAG】ｶﾗﾑのｸﾗｽ名(VARCHAR2,NUMBER など)をｾｯﾄします。
	 *
	 * @og.tag
	 * ｶﾗﾑのｸﾗｽ名をｾｯﾄします。
	 * "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @og.rev 6.4.1.1 (2016/01/16) "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @param	cls	ｸﾗｽ
	 */
	public void setClassName ( final String cls ) {
		className = nval( getRequestParameter( cls ),className );
	}

	/**
	 * 【HTML】最大入力文字数を指定します(入力可能文字数で、ﾊﾞｲﾄ数ではありません)。
	 *
	 * @og.tag
	 * ｶﾗﾑの入力文字制限数をｾｯﾄします。
	 *
	 * @og.rev 6.2.3.0 (2015/05/01) maxlength 変数は、0 の場合は、無制限になります。
	 *
	 * @param	len	入力文字制限数
	 */
	public void setMaxlength( final String len ) {
		maxlength = nval( getRequestParameter( len ),maxlength );

		// 6.2.3.0 (2015/05/01) maxlength 変数は、0 の場合は、無制限になります。
		if( "0".equals( maxlength ) ) { set( "NO_MAXLEN", "true" ); }
	}

	/**
	 * 【TAG】ｶﾗﾑの入力枠ｻｲｽﾞ(横幅)を文字数でｾｯﾄします(入力文字制限ではありません)。
	 *
	 * @og.tag
	 * ｶﾗﾑのｻｲｽﾞをｾｯﾄします。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) 入力欄の大きさを指定する viewSize 属性を追加。
	 *
	 * @param	siz	ｻｲｽﾞ
	 */
	public void setSize( final String siz ) {
		fieldSize = nval( getRequestParameter( siz ),null );		// 6.2.0.0 (2015/02/27)
	}

	/**
	 * 【TAG】ｶﾗﾑの表示ｻｲｽﾞ(横幅)を文字数でｾｯﾄします。
	 *
	 * @og.tag
	 * 入力枠ではなく、表示する時の文字数制限に使われます。
	 * maxLength は、DB上の桁数(ﾊﾞｲﾄ数)ですが、viewSize は、文字数です。
	 * SLABEL の簡易ﾊﾞｲﾄ数計算(全角2Byte)でもなく、文字数計算になります。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) viewSize 属性の追加
	 *
	 * @param	len	表示ｻｲｽﾞ(横幅)
	 */
	public void setViewSize( final String len ) {
		viewSize = nval( getRequestParameter( len ),null );
	}

	/**
	 * 【TAG】ｶﾗﾑが書き込み可能かどうか[true:可能/false:不可]を設定します(初期値:null)。
	 *
	 * @og.tag
	 * ｶﾗﾑの書き込み可能を設定します。
	 * 初期値は、null:無指定 です。
	 *
	 * @param	flag	書き込み許可 [true:許可/false:不可]
	 */
	public void setWritable ( final String flag ) {
		writable = nval( getRequestParameter( flag ),writable );
	}

	/**
	 * 【TAG】ｶﾗﾑの表示用ﾚﾝﾃﾞﾗｰ({@og.doc03Link renderer Renderer_****})をｾｯﾄします。
	 *
	 * @og.tag
	 * これは、plugin.column 以下の Renderer_**** ｸﾗｽの **** を与えます。
	 * これらは、CellRenderer ｲﾝﾀｰﾌｪｰｽを継承したｻﾌﾞｸﾗｽです。
	 * 属性ｸﾗｽ定義の {@link org.opengion.hayabusa.db.CellRenderer Renderer} を参照願います。
	 * {@og.doc03Link renderer Renderer_**** ｸﾗｽ}
	 * "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @og.rev 6.4.1.1 (2016/01/16) "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @param	rdr	表示用ﾚﾝﾃﾞﾗｰ (Renderer_**** の ****)
	 * @see		org.opengion.hayabusa.db.CellRenderer
	 */
	public void setRenderer ( final String rdr ) {
		renderer = nval( getRequestParameter( rdr ),renderer );
	}

	/**
	 * 【TAG】ｶﾗﾑの編集用ｴﾃﾞｨﾀｰ({@og.doc03Link editor Editor_****})をｾｯﾄします。
	 *
	 * @og.tag
	 * これは、plugin.column 以下の Editor_**** ｸﾗｽの **** を
	 * 与えます。これらは、CellEditor ｲﾝﾀｰﾌｪｰｽを継承したｻﾌﾞｸﾗｽです。
	 * 属性ｸﾗｽ定義の {@link org.opengion.hayabusa.db.CellEditor Editor} を参照願います。
	 * {@og.doc03Link editor Editor_**** ｸﾗｽ}
	 * "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @og.rev 6.4.1.1 (2016/01/16) "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @param	ed	編集用ｴﾃﾞｨﾀｰ (Editor_**** の ****)
	 * @see		org.opengion.hayabusa.db.CellEditor
	 */
	public void setEditor ( final String ed ) {
		editor = nval( getRequestParameter( ed ),editor );
	}

	/**
	 * 【TAG】DBﾀｲﾌﾟ({@og.doc03Link dbType DBType_****})をｾｯﾄします。
	 *
	 * @og.tag
	 * これは、org.opengion.hayabusa.db.column 以下の DBType_**** ｸﾗｽの **** を
	 * 与えます。これらは、DBType ｲﾝﾀｰﾌｪｰｽを継承したｻﾌﾞｸﾗｽです。
	 * 属性ｸﾗｽ定義の {@link org.opengion.hayabusa.db.DBType DBType} を参照願います。
	 * {@og.doc03Link dbType DBType_**** ｸﾗｽ}
	 * "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @og.rev 6.4.1.1 (2016/01/16) "既存値:設定値" という指定を行えば、既存値を持っている場合のみ、設定値に置き換えます。
	 *
	 * @param	dt	ﾊﾟﾗﾒｰﾀ
	 * @see		org.opengion.hayabusa.db.DBType DBType_**** の ****
	 */
	public void setDbType( final String dt ) {
		dbType = nval( getRequestParameter( dt ),dbType );
	}

	/**
	 * 【TAG】ｺｰﾄﾞｾﾚｸﾄのﾕﾆｰｸｷｰ(ｺｰﾄﾞ名称)をｾｯﾄします。
	 *
	 * @og.tag
	 * このｷｰを元に、CodeSelectionｵﾌﾞｼﾞｪｸﾄを構築します。
	 *
	 * @og.rev 3.5.5.7 (2004/05/10) 新規作成
	 *
	 * @param	name	ﾒﾆｭｰのﾕﾆｰｸｷｰ(ｺｰﾄﾞ名称)
	 */
	public void setCodeName( final String name ) {
		codeName = nval( getRequestParameter( name ),codeName );
	}

	/**
	 * 【TAG】ｺｰﾄﾞｾﾚｸﾄのｻﾌﾞｾｯﾄとなるｺｰﾄﾞ(ｷｰ)をCSV形式で指定します。
	 *
	 * @og.tag
	 * この値に限定された、ｺｰﾄﾞﾘｿｰｽを、既存の、CodeData の
	 * ｻﾌﾞｾｯﾄとして作成します。
	 * codeName で指定されたﾘｿｰｽでも、ｻﾌﾞｾｯﾄを作成する事ができます。
	 * ここで指定するのは、ｺｰﾄﾞ(ｷｰ)だけです。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規作成
	 *
	 * @param	list	ｺｰﾄﾞｾﾚｸﾄのｻﾌﾞｾｯﾄとなるｺｰﾄﾞ(ｷｰ)
	 */
	public void setCodeList( final String list ) {
		codeList = nval( getRequestParameter( list ),codeList );
	}

	/**
	 * 【TAG】ｺｰﾄﾞｾﾚｸﾄのｻﾌﾞｾｯﾄとなるｸﾞﾙｰﾌﾟを指定します。
	 *
	 * @og.tag
	 * この値に限定された、ｺｰﾄﾞﾘｿｰｽを、既存の、CodeData の
	 * ｻﾌﾞｾｯﾄとして作成します。
	 * codeName で指定されたﾘｿｰｽでも、ｻﾌﾞｾｯﾄを作成する事ができます。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規作成
	 *
	 * @param	group	ｺｰﾄﾞｾﾚｸﾄのｻﾌﾞｾｯﾄとなるｸﾞﾙｰﾌﾟ
	 */
	public void setCodeGroup( final String group ) {
		codeGroup = nval( getRequestParameter( group ),codeGroup );
	}

	/**
	 * 【TAG】新しいｺｰﾄﾞｾﾚｸﾄを作成する為の｢ｷｰ:ﾗﾍﾞﾙ ｷｰ:ﾗﾍﾞﾙ｣の文字列を指定します。
	 *
	 * @og.tag
	 * 既存の、CodeData には存在しない、新しいｺｰﾄﾞﾘｿｰｽを作成する為の、文字列を指定します。
	 * 文字列は、｢ｷｰ:ﾗﾍﾞﾙ ｷｰ:ﾗﾍﾞﾙ｣形式で、ｽﾍﾟｰｽで分解後、":" でｷｰとﾗﾍﾞﾙに分離します。
	 * ｽﾍﾟｰｽ分解後の文字列に、":" が含まれていない場合は、ｷｰをﾗﾍﾞﾙとして扱います。
	 * また、ﾗﾍﾞﾙ部分は、ﾗﾍﾞﾙﾘｿｰｽを使用して、変換を行います。
	 * 内部的には、CodeData を作成しません。DBColumnｵﾌﾞｼﾞｪｸﾄ内で、直接、Selection_KEYVAL を生成します。
	 * codeName 、codeList、codeGroup などが指定された場合は、そちらが優先されます。
	 * ｢ｷｰ:ﾗﾍﾞﾙ ｷｰ:ﾗﾍﾞﾙ｣で、ﾗﾍﾞﾙ にｽﾍﾟｰｽを含ませる場合は、ﾀﾞﾌﾞﾙｸｫｰﾃｰｼｮﾝで囲ってください。
	 * ｢"ｷｰ:ﾗﾍﾞﾙ" "ｷｰ:ﾗﾍﾞﾙ"｣という感じです。
	 *
	 * ※ editor,renderer を、"KVMENU" にして、param に、｢ｷｰ:ﾗﾍﾞﾙ ｷｰ:ﾗﾍﾞﾙ｣
	 *    の文字列を指定する方法の、簡易版です。
	 *
	 * @og.rev 6.3.1.1 (2015/07/10) codeKeyVal 属性を追加
	 *
	 * @param	keyval	KEY:VAL文字列
	 */
	public void setCodeKeyVal( final String keyval ) {
		codeKeyVal = nval( getRequestParameter( keyval ),codeKeyVal );
	}

	/**
	 * 【TAG】ﾊﾟﾗﾒｰﾀをｾｯﾄします。
	 *
	 * @og.tag
	 * ここで指定したﾊﾟﾗﾒｰﾀは、renderer、editor、dbType すべてのﾊﾟﾗﾒｰﾀに
	 * ｾｯﾄされます。
	 * ﾊﾟﾗﾒｰﾀをｾｯﾄします。
	 *
	 * @og.rev 5.6.4.3 (2013/05/24) 新規追加
	 *
	 * @param	prm	ﾊﾟﾗﾒｰﾀ
	 */
	public void setParam( final String prm ) {
		parameter = nval( getRequestParameter( prm ),parameter );
		rawParameter = nval( prm, parameter );
	}

	/**
	 * 【TAG】value属性に値がｾｯﾄされていないとき使用する、初期値を指定します。
	 *
	 * @og.tag
	 * value属性に値がｾｯﾄされていないときに、この初期値を使用します。
	 *
	 * 8.4.3.0 (2023/03/31) defaultVal の設定値の取り扱い(変更なし)
	 *   {&#064;XXXX} は、ﾘｸｴｽﾄ変数 ⇒ valueﾀｸﾞｾｯﾄ値 を確認
	 *     値がなければ、null となる。DBColumnの初期値は、""(空文字列)にｾｯﾄされる。
	 *   通常の固定値は、そのまま使用されるが、""(空文字列)の場合は、
	 *     null となる。DBColumnの初期値は、""(空文字列)にｾｯﾄされる。
	 *   defaultVal属性を使用しない場合は、
	 *     null のままで、DBColumnの初期値は、""(空文字列)にｾｯﾄされる。
	 *
	 * @param	dv	初期値
	 */
	public void setDefaultVal( final String dv ) {
		defaultVal = nval( getRequestParameter( dv ),defaultVal );
	}

	/**
	 * 【TAG】ｾﾚｸﾄﾒﾆｭｰの値なしのOptionの有無[true/false]を指定します(初期値:falseと同等)。
	 *
	 * @og.tag
	 * ｾﾚｸﾄﾒﾆｭｰの場合、値なしのOptionをSelectに含めるかどうかを指定します。
	 * ここでは、検索結果の登録画面に使用されるｹｰｽが多いため、なにも指定されない
	 * 場合は、なにもｾｯﾄしません。(Coderesourceの指定どおり)
	 * 動作としては、初期値は、含めない("false") と同等です。
	 *
	 * &lt;og:columnEditor column="CDBK" addNoValue="false" /&gt;
	 *
	 * @param	flag	値なしのOptionの有無 [true:含める/それ以外:含めない]
	 */
	public void setAddNoValue( final String flag ) {
		addNoValue = nval( getRequestParameter( flag ),addNoValue );
	}

	/**
	 * 【TAG】ｾﾚｸﾄﾒﾆｭｰの場合、ｷｰ:ﾗﾍﾞﾙ形式で表示するかどうか[true/false/null]を指定します(初期値:null)。
	 *
	 * @og.tag
	 * ｶﾗﾑがﾌﾟﾙﾀﾞｳﾝﾒﾆｭｰ形式の場合、ｷｰ:ﾗﾍﾞﾙ形式で表示するか、ﾗﾍﾞﾙだけで表示するかを
	 * 選択できます。
	 * true を指定すると、｢強制的に｣ｷｰ:ﾗﾍﾞﾙ形式で表示します。
	 * false の場合は、｢強制的に｣ﾗﾍﾞﾙのみで表示されます。
	 * 初期値の null の場合、ｺｰﾄﾞﾘｿｰｽや、SQL文に準拠した指定通りの形式で表示します。
	 *
	 * &lt;og:column name="CDC" addKeyLabel="true" /&gt;
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) ｷｰ:ﾗﾍﾞﾙ形式で表示するかどうか。新規追加
	 *
	 * @param	flag	ｷｰ:ﾗﾍﾞﾙ形式表示 [true:ｷｰ:ﾗﾍﾞﾙ形式/false:ﾗﾍﾞﾙのみ/null:指定通り]
	 */
	public void setAddKeyLabel( final String flag ) {
		addKeyLabel = nval( getRequestParameter( flag ),addKeyLabel );
	}

	/**
	 * 【TAG】ｾﾚｸﾄﾒﾆｭｰの場合、ｷｰとﾗﾍﾞﾙを分けて出力するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * ｶﾗﾑがﾌﾟﾙﾀﾞｳﾝﾒﾆｭｰ形式の場合、ﾌｧｲﾙ出力時に、ｷｰとﾗﾍﾞﾙを分離して出力する事ができます。
	 * この属性と、addKeyLabel とは、微妙に関連していて、addKeyLabel="true" のみの場合は、
	 * ｷｰ:ﾗﾍﾞﾙ形式で、出力されますが、writeKeyLabel="true" も指定されている場合は、こちらが優先されます。
	 * なお、この指定が有効なのは、ｾﾚｸﾄﾒﾆｭｰのｶﾗﾑのみです。
	 * 初期値は、false です。
	 *
	 * @og.rev 6.2.3.0 (2015/05/01) ｷｰ／ﾗﾍﾞﾙで分離して出力するかどうか。
	 *
	 * @param	flag	ｷｰ／ﾗﾍﾞﾙ分離出力するかどうか [true:分離出力/false:通常]
	 */
	public void setWriteKeyLabel( final String flag ) {
		writeKeyLabel = nval( getRequestParameter( flag ),writeKeyLabel );
	}

	/**
	 * 【TAG】SQL文を指定してﾗﾍﾞﾙ情報を動的に読み込むかどうか[true/false]指定します(初期値:false)。
	 *
	 * @og.tag
	 * trueを指定すると、body部分に記述されたSQL文よりﾗﾍﾞﾙ情報を生成して、
	 * 動的にﾗﾍﾞﾙ表示を切り替えます。
	 *
	 * ﾜｲﾙﾄﾞｶｰﾄﾞを使用した、ｶﾗﾑの複数指定と組み合わせることで、外部から設定
	 * されたﾗﾍﾞﾙ情報で一括で変更することができます。
	 *
	 * 発行するｸｴﾘでは、第1ｶﾗﾑにﾗﾍﾞﾙｷｰを、第2ｶﾗﾑにﾗﾍﾞﾙ名称を設定します。
	 * 第3ｶﾗﾑが存在する場合は、名称(短)として使用されます。(必須ではありません)
	 *
	 * ここで作成されたﾗﾍﾞﾙ情報は、ｶﾗﾑｴﾃﾞｨﾀｰを適用したDBTableModelのみで
	 * 有効であり、内部的にもｷｬｯｼｭされません。
	 *
	 * 初期値は、false(動的読込しない)です。
	 *
	 * @og.rev 4.3.4.0 (2008/12/01) 新規作成
	 *
	 * @param	flag	動的読込 [true:する/false:しない]
	 */
	public void setUseLabelMap( final String flag ) {
		useLabelMap = nval( getRequestParameter( flag ),useLabelMap );
	}

	/**
	 * 【TAG】MENU,DBMENUでSLABELを利用するかどうか[auto/true/false]を指定します(初期値:null)。
	 *
	 * @og.tag
	 * 通常はｾﾚｸﾄﾒﾆｭｰは一覧とそれ以外で短縮ﾗﾍﾞﾙの利用が自動で切り替わります。
	 * 強制的にSLABELを利用させたい場合にこの属性をtrueにｾｯﾄして下さい。
	 * auto/true/false以外を指定した場合はfalse扱いとします。
	 * ここでの初期値は、値の変更を行わないということで、null です。
	 *
	 * @og.rev 5.6.2.3 (2013/03/22) columnﾀｸﾞに実装されているので移植
	 *
	 * @param	prm	SLABEL利用 [auto:自動/true:利用する/false:利用しない]
	 */
	public void setUseSLabel( final String prm ) {
		useSLabel = nval( getRequestParameter( prm ),useSLabel );
		if( useSLabel != null &&
			!"auto".equalsIgnoreCase( useSLabel ) &&
			!"true".equalsIgnoreCase( useSLabel ) &&
			!"false".equalsIgnoreCase( useSLabel ) ) {
				final String errMsg = "useSLabel は、[auto:自動/true:利用する/false:利用しない] から選んでください。"
								+ "useSLabel=[" + prm + "]" + CR ;
				throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】非表示文字列を指定します(初期値:null)。
	 *
	 * @og.tag
	 * これは、例えば、数字型の場合は、0 がﾃﾞﾌｫﾙﾄなどの場合、
	 * あえて表示したくないなどのｹｰｽに使います。
	 * そのような状況が設定されていない場合は、null が返されます。
	 * 初期値は、null です。
	 *
	 * @og.rev 5.6.2.3 (2013/03/22) 新規追加
	 *
	 * @param	prm	非表示文字列 [auto:自動/true:利用する/false:利用しない]
	 */
	public void setNoDisplayVal( final String prm ) {
		noDisplayVal = nval( getRequestParameter( prm ),noDisplayVal );
	}

	/**
	 * 【TAG】(通常は使いません)結果のDBTableModelを、sessionに登録するときのｷｰを指定します
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @og.tag
	 * 検索結果より、DBTableModelｵﾌﾞｼﾞｪｸﾄを作成します。これを、下流のviewﾀｸﾞ等に
	 * 渡す場合に、通常は、session を利用します。その場合の登録ｷｰです。
	 * query ﾀｸﾞを同時に実行して、結果を求める場合、同一ﾒﾓﾘに配置される為、
	 * この tableId 属性を利用して、ﾒﾓﾘ空間を分けます。
	 * (初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @param	id	ﾃｰﾌﾞﾙID (sessionに登録する時のID)
	 */
	public void setTableId( final String id ) {
		tableId = nval( getRequestParameter( id ),tableId );
	}

	/**
	 * 【TAG】ﾃﾞｰﾀﾍﾞｰｽへ接続する場合のID(初期値:null)。
	 *
	 * @og.tag
	 * ﾃﾞｰﾀﾍﾞｰｽ接続を使用するｴﾃﾞｨﾀｰ、ﾚﾝﾃﾞﾗｰを使用する場合に使用する接続先IDを指定します。
	 * ｶﾗﾑｵﾌﾞｼﾞｪｸﾄは、query処理ではなく、view処理時に適用されるため、ﾃﾞｰﾀﾍﾞｰｽ
	 * 接続先を指定する必要があります。
	 * 初期値は、null(DEFAULT 接続) です。
	 *
	 * @og.rev 4.0.0.0 (2006/04/02) 新規追加
	 *
	 * @param	id	接続先ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}

	/**
	 * 【TAG】ｲﾍﾞﾝﾄｶﾗﾑ(親ｶﾗﾑ)を設定します。
	 *
	 * @og.tag
	 * 親子関係を持った項目の動的な入れ替えを行う場合に利用します。
	 * このｶﾗﾑを変化させるﾄﾘｶﾞとなるｶﾗﾑの名前をｾｯﾄします。
	 *
	 * ※詳細はcolumnTagのeventURL属性の説明をご覧下さい。
	 *
	 * なお、columnEditorで、Viewに対して、SQL文の定義(置き換え定義)をする場合でも、
	 * 他ｶﾗﾑの値の参照は、columnﾀｸﾞの場合と同様に、{&#064;XXXX}または$nの変数により行います。
	 * この場合、これらの変数は、"自身と同一行のｶﾗﾑ"を意味します。
	 *
	 * @og.rev 4.3.6.0 (2009/04/01)
	 * @og.rev 5.1.7.0 (2010/06/01) ｺﾒﾝﾄ大幅修正
	 *
	 * @param	col	親ｶﾗﾑ
	 */
	public void setEventColumn( final String col ) {
		eventColumn = nval( getRequestParameter( col ), eventColumn );
	}

	/**
	 * 【TAG】eventColumn使用時の値を、SQL文から求めるためのqueryをｾｯﾄします。
	 *
	 * @og.tag
	 * eventColumn利用時は{&#064;XXXX}はﾘｸｴｽﾄﾊﾟﾗﾒｰﾀではなく
	 * xxxｶﾗﾑの実際の値が利用されます。
	 *
	 * @og.rev 6.3.4.0 (2015/08/01) eventValue 追加
	 * @param	val	ﾊﾟﾗﾒｰﾀ
	 */
	public void setEventValue( final String val ) {
		eventValue = nval( getReservedParameter( val ), eventValue );	// ここでは{@XXXX}変換をしない
	}

	/**
	 * 【TAG】ｲﾍﾞﾝﾄｶﾗﾑ指定時に呼び出すURLを指定します。
	 *
	 * @og.tag
	 * ｲﾍﾞﾝﾄｶﾗﾑ指定時に部品を作成するJSPを指定します。
	 * 初期値はｼｽﾃﾑﾘｿｰｽのEVENT_COLUMN_URLです。
	 * (例：eventURL="makeColumn_custom.jsp")
	 *
	 * ※詳細はcolumnTagのeventURL属性の説明をご覧下さい。
	 *
	 * @og.rev 4.3.6.0 (2009/04/01)
	 *
	 * @param	url	JSPのURL
	 */
	public void setEventURL( final String url ) {
		eventURL = nval( getRequestParameter( url ), eventURL );
	}

	/**
	 * 【TAG】eventColumn実行後のcallbak関数を指定します。
	 *
	 * @og.tag
	 * eventColumnの動作終了後に実行するCallback関数の指定を行います。
	 * 関数名のみを指定して下さい。
	 * 　ex) sampleCallback()を実行する場合はeventCallback="sampleCallback"
	 * 実行する関数の第一引数には自動的に次の要素が入った配列が渡されます
	 * 　[0] 変更された子ｶﾗﾑのID属性値
	 * 　[1] evnetColumn実行後の子ｶﾗﾑの値
	 * 　[2] eventColumn実行前の子ｶﾗﾑの値
	 *
	 * この属性は、optionAttributesへの設定と同様の動作を行います。
	 *
	 * @og.rev 5.5.4.0 (2012/07/02) 新規追加
	 * @og.rev 6.9.8.0 (2018/05/28) ﾊﾟﾗﾒｰﾀ引数が、未設定の場合の考慮不足
	 *
	 * @param	callback	Callback関数
	 */
	public void setEventCallback( final String callback ) {
		// optionAttributes扱いで登録します。
		final String clbk = getRequestParameter( callback );
		if( StringUtil.isNotNull( clbk ) ){
			add( "optionAttributes", "eventCallback='" + clbk + "'" );
		}

//		// optionAttributes扱いで登録します。
//		if( callback != null && callback.length() > 0 ){
//			add( "optionAttributes", "eventCallback='"+getRequestParameter( callback )+"'" );
//		}
	}

	/**
	 * 【HTML5】表示形式を指定します(初期値:text)。
	 *
	 * @og.tag
	 * ColumnTagは、ｶﾗﾑﾘｿｰｽで作成される為、ﾀｲﾌﾟは固定です。
	 * しかし、HTML5 で追加された形式には、有用なﾀｲﾌﾟも存在する為、
	 * それらへの書き換えをできるようにします。
	 * よって、ここでは、従来型のﾀｲﾌﾟへの書き換えはできません。
	 * (textだけは例外で戻せるようにします)。
	 *
	 * [search/tel/url/email/datetime/date/month/week/time/datetime-local/number/range/color]
	 * が、設定できます。
	 *
	 * HTML5 の機能を有効にするには、ﾈｲﾃｨﾌﾞﾓｰﾄﾞで動作させる必要があります。
	 *   ① USE_IE7_HEADER = "false" に設定する。
	 *   ② USE_HTML5_HEADER = "true" に設定する。
	 *   ③ IEのﾂｰﾙ⇒互換表示設定で、互換表示に追加したWebｻｲﾄから削除する。
	 *   ④ 同上の設定で、ｲﾝﾄﾗｻｲﾄを互換表示で表示するのﾁｪｯｸを外す。
	 * 必要があります。
	 *
	 * &lt;og:input type="text" /&gt; ﾃｷｽﾄﾎﾞｯｸｽを表示
	 *
	 * HTML5 で追加されたﾀｲﾌﾟ
	 * &lt;og:input type="search"			/&gt; 検索ﾃｷｽﾄの入力欄を作成する
	 * &lt;og:input type="tel"				/&gt; 電話番号の入力欄を作成する
	 * &lt;og:input type="url"				/&gt; URLの入力欄を作成する
	 * &lt;og:input type="email"			/&gt; ﾒｰﾙｱﾄﾞﾚｽの入力欄を作成する
	 * &lt;og:input type="datetime"			/&gt; UTC(協定世界時)による日時の入力欄を作成する
	 * &lt;og:input type="date"				/&gt; 日付の入力欄を作成する
	 * &lt;og:input type="month"			/&gt; 月の入力欄を作成する
	 * &lt;og:input type="week"				/&gt; 週の入力欄を作成する
	 * &lt;og:input type="time"				/&gt; 時間の入力欄を作成する
	 * &lt;og:input type="datetime-local"	/&gt; UTC(協定世界時)によらないﾛｰｶﾙ日時の入力欄を作成する
	 * &lt;og:input type="number"			/&gt; 数値の入力欄を作成する
	 * &lt;og:input type="range"			/&gt; ﾚﾝｼﾞの入力欄を作成する
	 * &lt;og:input type="color"			/&gt; 色の入力欄を作成する
	 *
	 * @og.rev 5.7.1.0 (2013/12/06) HTML5 対応(新規追加)
	 * @og.rev 6.3.4.0 (2015/08/01) Arrays.toString から String.join に置き換え。
	 * @og.rev 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	 *
	 * @param	type	HTML5用 [text/search/tel/url/email/datetime/date/month/week/time/datetime-local/number/range/color]
	 */
	public void setType( final String type ) {
		final String tp = getRequestParameter( type );

		if( check( tp , TYPE_SET ) ) {
			set( "type",tp );
		}
		else {
			final String errMsg = "type 属性は、下記の中から選択してください。type=["
						 + tp + "] in [" + String.join( ", " , TYPE_SET ) + "]";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】ResourceManager を使用しないかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * trueを指定すると、使用しない設定になります。
	 * これは、ﾃﾞｰﾀﾍﾞｰｽから検索した値そのものが返される ｶﾗﾑｵﾌﾞｼﾞｪｸﾄを使用することになります。
	 * ｢noResource｣なので、意味が反転しているのをご注意ください。
	 *
	 * false(初期値)は、ResourceManager を使用します。
	 *
	 * @og.rev 5.7.3.0 (2014/02/07) 新規作成
	 *
	 * @param	flag	ResourceManagerの不使用 [true:不使用/false:使用]
	 */
	public void setNoResource( final String flag ) {
		noResource = nval( getRequestParameter( flag ),noResource );
	}

	/**
	 * 【TAG】(通常は使いません)ｶﾗﾑIDの存在ﾁｪｯｸを行うかどうか[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * true の場合、ｶﾗﾑIDがDBTableModel に存在しない場合は、ｴﾗｰになります。
	 * false の場合、ｶﾗﾑIDがDBTableModel に存在しない場合は、無視します。
	 * これは、検索条件によって、設定されるｶﾗﾑが異なる場合でも、後続処理を
	 * 正常に動作させたい場合に、使用します。
	 * 初期値は true (ﾁｪｯｸを行う) です。
	 *
	 * @og.rev 6.7.6.0 (2017/03/17) strictCheck 追加。
	 *
	 * @param	check	存在ﾁｪｯｸ [true:行う/false:行わない]
	 */
	public void setStrictCheck( final String check ) {
		strictCheck = nval( getRequestParameter( check ),strictCheck );
	}

	/**
	 * 【TAG】ﾚﾝﾃﾞﾗ利用時のExcel出力でｾﾙﾀｲﾌﾟを文字固定にするか[true/false]を指定します(初期値:ｼｽﾃﾑ定数のUSE_STRING_EXCEL_OUTPUT)。
	 *
	 * @og.tag
	 * falseを指定すると、dbTypeに従ったｾﾙﾀｲﾌﾟとなります。
	 * TableWriter内部でdouble変換をするためｴﾗｰに注意してください。
	 * (初期値:ｼｽﾃﾑ定数のUSE_STRING_EXCEL_OUTPUT[={@og.value SystemData#USE_STRING_EXCEL_OUTPUT}])。
	 *
	 * @og.rev 5.7.6.3 (2014/05/23) 新規作成
	 *
	 * @param	flag	ｾﾙﾀｲﾌﾟの文字固定にするか [true:固定する/false:固定しない]
	 */
	public void setStringOutput( final String flag ) {
		stringOutput = nval( getRequestParameter( flag ),stringOutput );
	}

	/**
	 * 【TAG】ﾘｱﾙﾀｲﾑﾁｪｯｸを行うかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * ｶﾗﾑ単位でﾘｱﾙﾀｲﾑﾁｪｯｸを行うかどうかを設定をします。
	 * ﾀｸﾞに独自属性としてrealTimeCheck="true/false"を追記します。
	 * falseを設定した場合にﾁｪｯｸが行われなくなります。
	 * 初期値はﾘｱﾙﾀｲﾑﾁｪｯｸを行う(true)です。
	 *
	 * この属性は、optionAttributesへの設定と同様の動作を行います。
	 * 引数の文字列の、true/false 判定を行っていません。そのままの文字列が設定されます。
	 * JavaScript 側では、false 判定のみ行っているので、不正な文字列の場合は、
	 * 初期値(true:ﾘｱﾙﾀｲﾑﾁｪｯｸを行う)が適用されます。
	 * これは、Ver5 とﾛｼﾞｯｸを合わせておくための処置です。
	 *
	 * @og.rev 5.9.32.2 (2018/05/18) 新規追加
	 * @og.rev 6.9.8.0 (2018/05/28) Ver5 とﾛｼﾞｯｸを合わせます。
	 *
	 * @param	flag	ﾘｱﾙﾀｲﾑﾁｪｯｸを行うかどうか [true:行う/false:行わない]
	 */
	public void setUseRealTimeCheck( final String flag ) {
		// optionAttributes扱いで登録します。
		final String rtcf = getRequestParameter( flag );

		if( StringUtil.isNotNull( rtcf ) ){
			add( "optionAttributes", "realTimeChk=\"" + rtcf + "\"" );			// 連結時のｽﾍﾟｰｽは、Attributes ｸﾗｽで処理
		}

//		final boolean useRtc = nval( getRequestParameter( flag ) , true );
//
//		if( !useRtc ) {		// true 以外の場合のみ、optionAttributes に属性を追加します。
//			add( "optionAttributes", "realTimeChk=\"false\"" );			// 連結時のｽﾍﾟｰｽは、Attributes ｸﾗｽで処理
//		}
	}

	/**
	 * 【TAG】日付送り戻し機能を有効にするかどうか[true:有効/false:無効]指定します(初期値:true)。
	 *
	 * @og.tag
	 * 編集種別が年月(YM)、年月日(YMD)、年月日(YMD2)のとき、日付送り戻し機能の
	 * 有効(true)/無効(false)を指定します。
	 * ｲﾍﾞﾝﾄｶﾗﾑ(親ｶﾗﾑ)を設定し、param属性に定義したSQL文を実行させたい場合には、
	 * false を指定します。
	 *
	 * @og.rev 8.1.2.3 (2022/05/20) 日付送り戻し不具合対応(useDateFeed 属性追加)
	 *
	 * @param	flag	日付送り戻し機能有無 [true:有効/false:無効]
	 */
	public void setUseDateFeed( final String flag ) {
		useDateFeed = nval( getRequestParameter( flag ),useDateFeed );
	}

	/**
	 * このｵﾌﾞｼﾞｪｸﾄの文字列表現を返します。
	 * 基本的にﾃﾞﾊﾞｯｸﾞ目的に使用します。
	 *
	 * @return	このｸﾗｽの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "tableId"		,tableId	)
				.println( "columns"		,columns	)
				.println( "allColumns"	,allColumns	)
				.println( "className"	,className	)
				.println( "size"		,fieldSize	)			// 6.2.0.0 (2015/02/27)
				.println( "viewSize"	,viewSize	)			// 6.2.0.0 (2015/02/27)
				.println( "maxlength"	,maxlength	)
				.println( "writable"	,writable	)
				.println( "renderer"	,renderer	)
				.println( "editor"		,editor		)
				.println( "codeName"	,codeName	)
				.println( "dbType"		,dbType		)
				.println( "defaultVal"	,defaultVal	)
				.println( "parameter"	,parameter	)
				.println( "dbid"		,dbid		)
				.println( "addNoValue"	,addNoValue	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
