/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;									// 6.9.0.0 (2018/01/31)
import java.io.Writer;
import java.util.Map;

import org.opengion.fukurou.system.LogWriter;
import org.opengion.fukurou.system.Closer;
import org.opengion.fukurou.system.HybsConst;					// 6.4.5.2 (2016/05/06)
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.util.ArraySet;						// 5.10.10.0 (2019/03/29)
// import org.opengion.fukurou.util.SOAPConnect;				// 6.9.0.0 (2018/01/31) SOAPConnect 廃止に伴う修正
import org.opengion.fukurou.util.StringUtil;
// import org.opengion.fukurou.util.URLConnect;					// 6.9.0.0 (2018/01/31) URLConnect 廃止
import org.opengion.fukurou.util.HttpConnect;					// 6.9.0.0 (2018/01/31) 新規追加
// import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.xml.XML2TableParser;
import org.opengion.fukurou.xml.XSLT;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.hayabusa.io.HybsFileOperationFactory;		// 8.0.0.1 (2021/10/08)
// import org.opengion.fukurou.model.FileOperation;				// 8.0.0.1 (2021/10/08)

import static org.opengion.fukurou.util.StringUtil.nval;

import java.util.Set;											// 5.10.10.0 (2019/03/29)

/**
 * 指定のURLに接続します｡
 *
 * ｴﾝｼﾞﾝでは､URL にｱｸｾｽすることで､ﾃﾞｰﾓﾝを起動したり､
 * ｺﾏﾝﾄﾞを実行(adminﾒﾆｭｰ)することが出来ます｡
 * もちろん､検索条件を指定して､結果を取得することも可能です｡
 * このﾀｸﾞでは､URLにｱｸｾｽして､ｺﾏﾝﾄﾞを実行したり結果を取得できます｡
 * さらに､ﾌｧｲﾙを POST で転送したり､処理結果を XSLT変換したり出来ます｡
 *
 * @og.formSample
 * ●形式：
 *     &lt;og:urlConnect
 *         url           = "http://･･･ "    必須
 *         proxyHost     = "proxy.opengion.org"
 *         proxyPort     = "8080"
 *         timeout       = "1"
 *         keys          = "command,SYSTEM_ID"
 *         vals          = "NEW,GE"
 *         useSystemUser = "true/false"     初期値:true
 *         authUserPass  = "admin:******"   初期値:admin:******
 *         display       = "false/true"     初期値:false
 *         xslFile       = "filter.xsl"
 *         saveFile      = "outdata.xml"
 *    ×   soapNameSpace = "MyWebService"	6.9.0.0 (2018/01/31) 廃止
 *    ×   soapMethodName= "test"			6.9.0.0 (2018/01/31) 廃止
 *         tableId       = "DEFAULT"
 *         rowKey        = "item"
 *         colKeys       = "person_id,person_name"
 *         rtnKeys       = "version,summary"
 *         encode        = "UTF-8"
 *     /&gt;
 *
 * url           : 接続するURLを指定します｡必須属性です｡
 * proxyHost     : proxy が存在する場合は､そのﾎｽﾄ名(例：proxy.opengion.org)
 * proxyPort     : proxy が存在する場合は､そのﾎﾟｰﾄ番号(例：8080)
 * timeout       : 通信ﾘﾝｸのｵｰﾌﾟﾝ時に､指定された秒単位のﾀｲﾑ･ｱｳﾄ値を使用(例：1)
 * keys,vals     : URLの指定時に､ﾊﾟﾗﾒｰﾀ(引数)を追加します｡URLに含めても構いません｡
 *               : SOAPによる呼び出しの場合の詳細については､keysの属性定義を参照して下さい｡
 * postKey       : POST を使って､postFile属性のﾌｧｲﾙ内容を送信する時のｷｰです｡
 * postFile      : POST を使って､postFile属性のﾌｧｲﾙ内容を送信します｡
 *                 postFile を指定せず､postKey のみ指定して､BODY部に何か書き込めば､
 *                 そのBODY部の文字列を POSTの内容として送信します｡
 * authUserPass  : Basic認証を使用する場合の接続ﾕｰｻﾞｰ：ﾊﾟｽﾜｰﾄﾞを指定します｡
 *                 接続時のﾕｰｻﾞｰとﾊﾟｽﾜｰﾄﾞを､USER:PASSWD 形式 で指定します｡
 *                 useSystemUser="false" で何も指定しない場合は､Basic認証を使用しません｡
 * useSystemUser : Basic認証の接続ﾕｰｻﾞｰ：ﾊﾟｽﾜｰﾄﾞに､ｼｽﾃﾑﾕｰｻﾞｰを使用
 *                 するかどうかを指定します(初期値:true)｡
 *                 true の場合は､SYSTEM:***** を使用します｡
 * xslFile       : 接続先ﾃﾞｰﾀを取得し､そのﾃﾞｰﾀを XSLT変換する場合のXSLﾌｧｲﾙを指定します｡
 * display       : 接続した結果のﾚｽﾎﾟﾝｽを画面に表示するかどうかを指定します(初期値:false)｡
 *                 ｴﾝｼﾞﾝの場合､ｺﾏﾝﾄﾞを投げるだけであれば､結果を取得する必要は
 *                 ありません｡ｲﾒｰｼﾞ的には､取得ﾃﾞｰﾀが､このﾀｸﾞの位置に置き換わります｡
 *                 xslFile が指定されている場合､XSLT変換してｾｰﾌﾞします｡
 * saveFile      : 接続先ﾃﾞｰﾀを取得した結果を､ﾌｧｲﾙ出力します｡
 *                 display="true" と､saveFile を併用することはできません｡
 *                 xslFile が指定されている場合､XSLT変換してｾｰﾌﾞします｡
 * X soapNameSpace : 廃止：SOAPによるWebｻｰﾋﾞｽの呼び出しで､ﾒｿｯﾄﾞ名及びﾊﾟﾗﾒｰﾀｰの名前空間を指定します｡
 *                 この名前空間は､通常WSDLﾌｧｲﾙのdescriptionsﾀｸﾞのtargetNamespace属性の値により
 *                 定義されます｡  6.9.0.0 (2018/01/31) 廃止
 * X soapMethodName: 廃止：SOAPによるWebｻｰﾋﾞｽの呼び出しで､ﾒｿｯﾄﾞ名を指定します｡
 *                 WSDLﾌｧｲﾙで定義されるoperationﾀｸﾞのname属性の値に相当します｡ 6.9.0.0 (2018/01/31) 廃止
 * tableId       : 結果のXMLﾌｧｲﾙをDBTableModelに変換した際に､登録するTableIdを指定します｡
 * rowKey        : XMLをDBTableModelに変換する際の､行を表すﾀｸﾞｷｰを指定します｡
 * colKeys       : XMLをDBTableModelに変換する際の､項目を表すﾀｸﾞｷｰの一覧を指定します｡
 *                 ｷｰにPARENT_TAG､PARENT_FULL_TAGを指定することで､rowKeyで指定されたﾀｸﾞの
 *                 直近の親ﾀｸﾞ､及びﾌﾙの親ﾀｸﾞ名(親ﾀｸﾞの階層を"&gt;[ﾀｸﾞA]&gt;[ﾀｸﾞB]&gt;[ﾀｸﾞC]&gt;"で表現)を
 *                 取得することができます｡
 * rtnKeys       : XMLのﾀｸﾞｷｰを指定して値を取り出します｡取り出した値は､{&#064;XX}形式で処理することが可能です｡
 * encode        : ﾃﾞｰﾀの入出力を行うｴﾝｺｰﾄﾞを指定します｡
 * authJson      : JSONｺｰﾄﾞで認証する場合に使用します｡8.0.0.0 (2021/08/31)
 * authURL       : JSONｺｰﾄﾞで認証するURLを指定します｡8.0.0.0 (2021/08/31)
 * reqJson       : JSONｺｰﾄﾞでﾘｸｴｽﾄするﾊﾟﾗﾒｰﾀを指定します｡8.0.0.0 (2021/09/30)
 *
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し､{&#064;XXXX} を解析します)
 *         POSTﾃﾞｰﾀを記述します｡
 *
 * ●Tag定義：
 *   &lt;og:urlConnect
 *       url              ○【TAG】ｱｸｾｽする ＵＲＬ を指定します(必須)(必須)｡
 *       proxyHost          【TAG】ﾌﾟﾛｷｼ経由で接続する場合の､ﾌﾟﾛｷｼﾎｽﾄ名を指定します
 *       proxyPort          【TAG】ﾌﾟﾛｷｼ経由で接続する場合の､ﾌﾟﾛｷｼﾎﾟｰﾄ番号を指定します
 *       timeout            【TAG】通信ﾘﾝｸのｵｰﾌﾟﾝ時に､指定された秒単位のﾀｲﾑ･ｱｳﾄ値を使用
 *                                  (初期値:URL_CONNECT_TIMEOUT[={@og.value SystemData#URL_CONNECT_TIMEOUT}])｡)
 *       keys               【TAG】ｱｸｾｽﾊﾟﾗﾒｰﾀｷｰをCSV形式で複数指定します
 *       vals               【TAG】keys属性に対応する値をCSV形式で複数指定します
 *       useSystemUser      【TAG】Basic認証で接続するﾕｰｻﾞｰにSYSTEMﾕｰｻﾞｰを使用するかどうか[true/false]を指定します(初期値:true)
 *       authUserPass       【TAG】Basic認証を使用して接続する場合のﾕｰｻﾞｰ:ﾊﾟｽﾜｰﾄﾞを指定します(初期値:null)
 *       display            【TAG】接続の結果を表示するかどうかを指定します(初期値:false)
 *       xslFile            【TAG】接続の結果を表示する場合にXSLT変換する場合のﾌｧｲﾙを指定します
 *       saveFile           【TAG】接続の結果をﾌｧｲﾙに保存します
 *       postKey            【TAG】POST を使って､postFile属性のﾌｧｲﾙ内容を送信する時のｷｰを指定します
 *       postFile           【TAG】POST を使って､postFile属性のﾌｧｲﾙ内容を送信します
 *       method             【TAG】送信ﾒｿｯﾄﾞ[GET/POST/PUT/DELETE/SOAP]を指定します(初期値:GET)
 *       errNeglect         【TAG】(通常は使いません) 接続ｴﾗｰを無視する場合にtrueとします(初期値false)
 *       tableId            【TAG】(通常は使いません)結果のDBTableModelを､sessionに登録するときのｷｰを指定します
 *       scope              【TAG】ｷｬｯｼｭする場合のｽｺｰﾌﾟ[request/page/session/application]を指定します(初期値:session)
 *       rowKey             【TAG】結果のXMLをDBTableModelに変換する際に､行のｷｰとなるﾀｸﾞ名を指定します
 *       colKeys            【TAG】結果のXMLをDBTableModelに変換する際に､項目のｷｰとなるﾀｸﾞ名の一覧を指定します
 *       rtnKeys            【TAG】結果のXMLを変換する際に､ﾊﾟﾗﾒｰﾀ(Attributes)として取り込むｷｰの一覧を指定します
 *       encode             【TAG】ﾃﾞｰﾀの入出力のｴﾝｺｰﾄﾞを指定します(初期値:UTF-8)
 *       mainTrans          【TAG】(通常は使いません)ﾀｸﾞで処理される処理がﾒｲﾝとなるﾄﾗﾝｻﾞｸｼｮﾝ処理かどうかを指定します(初期値:false)
 *       propKeys           【TAG】ｺﾈｸｼｮﾝのrequestParameterに渡すｷｰをCSV形式で複数指定します｡ 5.10.16.0 (2019/10/04)
 *       propVals           【TAG】ｺﾈｸｼｮﾝのrequestParameterに渡す値をCSV形式で複数指定します｡ 5.10.16.0 (2019/10/04)
 *       postDataOnly       【TAG】trueの場合はｷ-=値から､値のみ送信に変更します｡(jsonを送信する場合等に利用) 5.10.19.0 (2019/12/27)
 *       authJson           【TAG】JSONｺｰﾄﾞで認証する場合に使用します｡8.0.0.0 (2021/08/31)
 *       authURL            【TAG】JSONｺｰﾄﾞで認証するURLを指定します｡8.0.0.0 (2021/08/31)
 *       reqJson            【TAG】JSONｺｰﾄﾞでﾘｸｴｽﾄするﾊﾟﾗﾒｰﾀを指定します｡8.0.0.0 (2021/09/30)
 *       useLocal           【TAG】ｼｽﾃﾑ定数でｸﾗｳﾄﾞ設定されていても､ｸﾗｳﾄﾞ環境を使用しない場合､trueを指定します(初期値:false) 8.0.1.0 (2021/10/29)
 *       caseKey            【TAG】このﾀｸﾞ自体を利用するかどうかの条件ｷｰを指定します(初期値:null) 5.7.7.2 (2014/06/20)
 *       caseVal            【TAG】このﾀｸﾞ自体を利用するかどうかの条件値を指定します(初期値:null) 5.7.7.2 (2014/06/20)
 *       caseNN             【TAG】指定の値が､null/ｾﾞﾛ文字列 でない場合(Not Null=NN)は､このﾀｸﾞは使用されます(初期値:判定しない) 5.7.7.2 (2014/06/20)
 *       caseNull           【TAG】指定の値が､null/ｾﾞﾛ文字列 の場合は､このﾀｸﾞは使用されます(初期値:判定しない) 5.7.7.2 (2014/06/20)
 *       caseIf             【TAG】指定の値が､true/TRUE文字列の場合は､このﾀｸﾞは使用されます(初期値:判定しない)
 *       debug              【TAG】ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:urlConnect&gt;
 *
 * 【廃止】6.9.0.0 (2018/01/31) 物理削除
 *   //  soapNameSpace      【廃止】SOAPによるWebｻｰﾋﾞｽの呼び出しで､ﾒｿｯﾄﾞ名及びﾊﾟﾗﾒｰﾀｰの名前空間を指定します ( 6.9.0.0 (2018/01/31) SOAPConnect 廃止に伴う修正)
 *   //  soapMethodName     【廃止】SOAPによるWebｻｰﾋﾞｽの呼び出しで､ﾒｿｯﾄﾞ名を指定します ( 6.9.0.0 (2018/01/31) SOAPConnect 廃止に伴う修正)
 *   //  storageType        【廃止】保存先ｽﾄﾚｰｼﾞﾀｲﾌﾟを指定します 5.10.9.0 (2019/03/01) 8.0.1.0 (2021/10/29) del
 *   //  bucketName         【廃止】保存先ﾊﾞｹｯﾄ名を指定します 5.10.9.0 (2019/03/01) 8.0.1.0 (2021/10/29) del
 *
 * ●例：
 * ｱﾄﾞﾐﾝ関連
 * http://localhost:8823/gf/jsp/admin?COMMAND=infomation     [状況表示]
 * http://localhost:8823/gf/jsp/admin?COMMAND=close          [ﾌﾟｰﾙ削除]
 * http://localhost:8823/gf/jsp/admin?COMMAND=loginUser      [ﾛｸﾞｲﾝﾕｰｻﾞｰ]
 * http://localhost:8823/gf/jsp/admin?COMMAND=ｼｽﾃﾑﾊﾟﾗﾒｰﾀ [ｼｽﾃﾑﾊﾟﾗﾒｰﾀ]
 *
 * 帳票ﾃﾞｰﾓﾝ
 * http://localhost:8823/gf/jsp/REP08/result.jsp?cmd=SET&amp;period=5000&amp;command=NEW&amp;timerTask=org.opengion.hayabusa.report.ReportDaemon&amp;name=ReportDaemon  ﾃﾞｰﾓﾝ起動
 * http://localhost:8823/gf/jsp/REP08/result.jsp?cmd=CANCEL&amp;period=5000&amp;command=NEW&amp;timerTask=org.opengion.hayabusa.report.ReportDaemon&amp;name=ReportDaemon  ﾃﾞｰﾓﾝ停止
 *
 *Tomcat Manager 画面
 * http://localhost:8823/manager/reload?path=/ver4 ｱﾌﾟﾘｹｰｼｮﾝを再ﾛｰﾄﾞ
 *
 * ●その他:
 * DB.ERR_CODEに､実行結果のHTTPｽﾃｰﾀｽｺｰﾄﾞを設定します｡
 *
 * @og.rev 3.6.0.0 (2004/09/17) 新規作成
 * @og.rev 4.1.0.0 (2007/12/22) POSTﾒｿｯﾄﾞで複数ｷｰを登録できるように属性追加
 * @og.rev 5.1.5.0 (2010/04/01) SOAP対応
 * @og.rev 6.9.0.0 (2018/01/31) SOAPConnect 廃止に伴う修正
 * @og.rev 8.0.0.0 (2021/08/31) httpclient5 対応のHttpConnectを使用
 * @og.group その他部品
 *
 * @version  8.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK11.0,
 */
public class URLConnectTag extends CommonTagSupport {
	/** このﾌﾟﾛｸﾞﾗﾑのVERSION文字列を設定します｡	{@value} */
	private static final String VERSION = "8.0.1.0 (2021/10/29)" ;
	private static final long serialVersionUID = 801020211029L ;

//	private static final String DEFAULT_USER = "SYSTEM:MANAGER" ;
	private static final String DEFAULT_USER = "admin:admin" ;			// 6.9.0.1 (2018/02/05) この際､変更しておきます｡

//	private static final String[] POST_LIST = new String[] { "POST","PUT","DELETE" };		// 5.10.10.0 (2019/03/29)
	private static final Set<String> POST_SET = new ArraySet<>( "POST","PUT","DELETE" );	// 5.10.10.0 (2019/03/29)

	private transient DBTableModel table;			// 5.1.5.0 (2010/04/01) DBTableModel出力対応

	private String		urlStr			;
	private String[]	keys			;
	private String[]	vals			;
	private String		xslFile			;
	private String		saveFile		;
	private String		postKey			;
	private String		postData		;			// postFile ﾌｧｲﾙか､BODY部
	private int			timeout			= HybsSystem.sysInt( "URL_CONNECT_TIMEOUT" );	// 6.2.5.0 (2015/06/05) 新規追加
	private boolean 	useSystemUser	= true;
	private String		authUserPass	;
	private boolean 	display			;
	private String		proxyHost		;			// 4.0.0 (2007/07/25)
	private int 		proxyPort		= -1;		// 4.0.0 (2007/07/25)
	private String		method			= "GET";	// 4.1.0.0 (2007/12/22) POSTorGETorSOAP
	private boolean		errNglctFlag	;			// 4.1.1.0 (2008/01/22) ｴﾗｰ無視ﾌﾗｸﾞ
//	private String		soapNameSpace	;			// 5.1.5.0 (2010/04/01) SOAP対応	// 6.9.0.0 (2018/01/31) SOAPConnect 廃止に伴う修正
//	private String		soapMethodName	;			// 5.1.5.0 (2010/04/01) SOAP対応	// 6.9.0.0 (2018/01/31) SOAPConnect 廃止に伴う修正
	private String		tableId			= HybsSystem.TBL_MDL_KEY; // 5.1.5.0 (2010/04/01) DBTableModel出力対応
	private String		rowKey			;			// 5.1.5.0 (2010/04/01) DBTableModel出力対応
	private String[]	colKeys			;			// 5.1.5.0 (2010/04/01) DBTableModel出力対応
	private String[]	rtnKeys			;			// 5.1.5.0 (2010/04/01) DBTableModel出力対応
	private String		encode			= "UTF-8";	// 5.1.6.0 (2010/05/01) ｴﾝｺｰﾄﾞ指定対応

	private boolean		isTableOut		;			// 5.1.5.0 (2010/04/01) DBTableModel出力対応
	private boolean		isOutParse		;			// 5.1.5.0 (2010/04/01) DBTableModel出力対応
	private boolean		isMainTrans		= true;		// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し

//	private String		storageType		;			// 5.10.9.0 (2019/03/01) ADD 8.0.1.0 (2021/10/29) del
//	private String		bucketName		;			// 5.10.9.0 (2019/03/01) ADD 8.0.1.0 (2021/10/29) del
	private String		contentType		;			// 5.10.10.2 (2019/04/12)

	private String		propKeys		;			// 5.10.16.0 (2019/10/04)
	private String		propVals		;			// 5.10.16.0 (2019/10/04)
	private boolean		postDataOnly	;			// 5.10.19.0 (2019/12/27) postDataOnly対応 ････ Ver7では未対応
	private String		authJson		;			// 8.0.0.0 (2021/08/31)
	private String		authURL			;			// 8.0.0.0 (2021/08/31)
	private String		reqJson			;			// 8.0.0.0 (2021/09/30)
	private boolean		useLocal		;			// 8.0.1.0 (2021/10/29) ｸﾗｳﾄﾞ設定を使用しない場合は､true

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰ
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public URLConnectTag() { super(); }		// これも､自動的に呼ばれるが､空のﾒｿｯﾄﾞを作成すると警告されるので､明示的にしておきます｡

	/**
	 * Taglibの開始ﾀｸﾞが見つかったときに処理する doStartTag() を ｵｰﾊﾞｰﾗｲﾄﾞします｡
	 *
	 * @og.rev 5.1.5.0 (2010/04/01) SOAP･DBTableModel出力対応
	 * @og.rev 5.1.6.0 (2010/05/01) DBLastSqlの処理は､DBTableModelが新規作成された処理でのみ行う｡
	 * @og.rev 5.7.7.2 (2014/06/20) caseKey,caseVal,caseNN,caseNull 属性を追加
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doStartTag() {
		// 5.7.7.2 (2014/06/20) caseKey,caseVal,caseNN,caseNull 属性を追加
		if( !useTag() ) { return SKIP_BODY ; }

		// 5.1.5.0 (2010/04/01) DBTableModel出力対応
		if( rowKey != null || colKeys != null || rtnKeys != null ) {
			isOutParse = true;
			if( rowKey != null || colKeys != null ) {
				isTableOut = true;
				useMainTrans( isMainTrans );			// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し
				startQueryTransaction( tableId );
			}
		}

		// 5.1.5.0 (2010/04/01) SOAPの場合は､postKeyが指定されない｡
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		return postData == null ? EVAL_BODY_BUFFERED : SKIP_BODY ;
	}

	/**
	 * Taglibのﾀｸﾞ本体を処理する doAfterBody() を ｵｰﾊﾞｰﾗｲﾄﾞします｡
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		postData = getBodyString();

		return SKIP_BODY ;
	}

	/**
	 * Taglibの終了ﾀｸﾞが見つかったときに処理する doEndTag() を ｵｰﾊﾞｰﾗｲﾄﾞします｡
	 *
	 * @og.rev 4.0.1.0 (2007/12/12) PostKeys,PostVals処理を追加
	 * @og.rev 5.1.5.0 (2010/04/01) SOAP･DBTableModel出力対応
	 * @og.rev 5.2.0.0 (2010/09/01) ｴﾗｰ処理でNullPointerExceptionが発生するﾊﾞｸﾞを修正
	 * @og.rev 5.7.7.2 (2014/06/20) caseKey,caseVal,caseNN,caseNull 属性を追加
	 * @og.rev 6.3.8.0 (2015/09/11) FileUtil#getPrintWriter( OutputStream,String ) を使用｡
	 * @og.rev 6.9.0.0 (2018/01/31) URLConnect 廃止､HttpConnect に置き換えます｡
	 * @og.rev 6.9.0.1 (2018/02/05) IOException を､Throwable に変更して､すべてのｴﾗｰをｷｬｯﾁするようにします｡
	 * @og.rev 5.10.19.0 (2019/12/27) 結果ｺｰﾄﾞをDB.ERR_CODEに設定
	 * @og.rev 8.0.0.2 (2021/10/15) ﾛｰｶﾙﾌｧｲﾙとｸﾗｳﾄﾞﾌｧｲﾙ間の移動
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		// 5.7.7.2 (2014/06/20) caseKey,caseVal,caseNN,caseNull 属性を追加
		if( !useTag() ) { return EVAL_PAGE ; }

//		URLConnect conn = null;
		HttpConnect conn = null;						// 6.9.0.0 (2018/01/31) URLConnect 廃止､HttpConnect に置き換えます｡
		ByteArrayOutputStream pipeOut = null;
		Writer outWriter = null;						// 6.3.8.0 (2015/09/11) rty の外に移動
		try {
			conn = connect();

			// 5.10.19.0 (2019/12/27) ADD
			setRequestAttribute( "DB.ERR_CODE", String.valueOf( conn.getCode() ) );

			// 出力先が､画面かﾌｧｲﾙかを判断します｡
			// 5.1.5.0 (2010/04/01) DBTableModel出力対応
			if( isOutParse ) {
				pipeOut = new ByteArrayOutputStream();
					// 6.3.8.0 (2015/09/11) FileUtil#getPrintWriter( OutputStream,String ) を使用｡
					outWriter = FileUtil.getPrintWriter( pipeOut, "UTF-8" );		// 6.3.8.0 (2015/09/11)
			}
			else if( display ) {
				outWriter = FileUtil.getNonFlushPrintWriter( pageContext.getOut() ) ;	// JspWriter の取得
			}
			else if( saveFile != null ) {
				outWriter = FileUtil.getPrintWriter( new File( saveFile ),"UTF-8" );
			}

			final String rtnData = conn.readData();		// 6.9.0.1 (2018/02/05)

			// 出力先が存在する場合｡
			if( outWriter != null ) {
				// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
				if( xslFile == null ) {
					outWriter.write( rtnData );
				}
				else {
					final XSLT xslt = new XSLT();
					xslt.setXslFile( xslFile );
					xslt.setOutFile( outWriter );
//					xslt.transform( conn.getReader() );
					xslt.transform( new StringReader( rtnData ) );		// 6.9.0.0 (2018/01/31) ﾘｰﾀﾞｰが取り出せないので､文字列から作成します｡
					xslt.close();
				}

				// 8.0.0.2 (2021/10/15) ﾛｰｶﾙﾌｧｲﾙとｸﾗｳﾄﾞﾌｧｲﾙ間の移動
				// 5.10.9.0 (2019/03/01) ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ指定の場合は､ｱｯﾌﾟﾛｰﾄﾞする｡
				if( saveFile != null) { // 5.10.10.0 (2019/03/29)
					HybsFileOperationFactory.local2cloud( useLocal,() -> new File(saveFile) );

//					final FileOperation cloudFile = HybsFileOperationFactory.create( saveFile );
//					if( cloudFile.isCloud() ) {
//						final File localFile = new File(saveFile);
//						FileUtil.copy( localFile, cloudFile );
//						localFile.delete();
//					}
				}
			}
		}
//		catch( final IOException ex ) {
		catch( final Throwable th ) {							// 6.9.0.1 (2018/02/05)
			final String errMsg = "ﾃﾞｰﾀ取得中にｴﾗｰが発生しました｡" 	+ CR
						+ " url=[" + urlStr + "]"							+ CR
						// 5.2.0.0 (2010/09/01) ｴﾗｰ処理でNullPointerExceptionが発生するﾊﾞｸﾞを修正
						+ " message=[" + ( conn == null ? "NO_CONNECTION" : conn.getMessage() ) + "]" + CR
						+ " Exception=[" + th.getMessage() + "]" ;
			if( errNglctFlag ) { // 4.1.1.0 (2008/01/22) ｴﾗｰを無視(標準ｴﾗｰ出力のみ)
				LogWriter.log( errMsg );
			}
			else { // 通常は無視しない
				throw new HybsSystemException( errMsg,th );
			}
		}
		finally {
			Closer.ioClose( outWriter );						// 6.3.8.0 (2015/09/11) finally に移動
//			if( conn != null ) { conn.disconnect(); }
		}

		// 5.1.5.0 (2010/04/01) DBTableModel出力対応
		if( isOutParse ) {
			// 6.3.8.0 (2015/09/11) AutoCloseableを使用したtry-with-resources構築に対応
			try( InputStream strm = new BufferedInputStream( new ByteArrayInputStream( pipeOut.toByteArray() ) ) ) {
				parse( strm );
			}
			// 6.3.8.0 (2015/09/11) 今まで不要だった､IOException が発生｡AutoCloseable のせい？｡
			catch( final IOException ex ) {		// catch は､close() されてから呼ばれます｡
				final String errMsg = "parse 処理中でｴﾗｰが発生しました｡"	+ CR
							+ "\t close() ｴﾗｰです｡"							+ CR
							+ "\t " + ex.getMessage()							+ CR ;
				System.err.println( errMsg );
			}

			if( isTableOut && table != null && !commitTableObject( tableId, table ) ) {
				jspPrint( "URLConnectTag Query処理が割り込まれました｡DBTableModel は登録しません｡" );
				return SKIP_PAGE ;
			}
		}

		return EVAL_PAGE ;
	}

	/**
	 * URLに対して接続を行います｡
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) ｴﾝｺｰﾄﾞ指定対応
	 * @og.rev 6.2.5.0 (2015/06/05) timeout属性追加
	 * @og.rev 6.9.0.0 (2018/01/31) SOAPConnect 廃止に伴う修正
	 * @og.rev 6.9.0.0 (2018/01/31) URLConnect 廃止､HttpConnect に置き換えます｡
	 * @og.rev 5.10.10.0 (2019/03/19) PUT.DELETE対応
	 * @og.rev 5.10.10.2 (2019/04/12) contentType
	 * @og.rev 5.10.16.0 (2019/10/04) propKeys,propVals指定追加
	 * @og.rev 5.10.19.0 (2019/12/27) postDataOnly対応 ････ Ver7では未対応
	 * @og.rev 8.0.0.0 (2021/08/31) authJson,authURL 追加
	 *
	 * @return 接続ｵﾌﾞｼﾞｪｸﾄ
	 * @throws IOException 入出力ｴﾗｰが発生したとき
	 */
//	private URLConnect connect() throws IOException {
	private HttpConnect connect() throws IOException {
		if( useSystemUser ) { authUserPass = DEFAULT_USER; }

		// 5.1.5.0 (2010/04/01) SOAP対応
		// SOAPの場合､PostDataは､SOAPConnectﾀｸﾞの中で生成します｡
//		URLConnect conn = null;
		HttpConnect conn = null;									// 6.9.0.0 (2018/01/31)
		// 6.9.0.0 (2018/01/31) SOAPConnect 廃止に伴う修正
//		if( "SOAP".equalsIgnoreCase( method ) ) {
//			if( soapNameSpace == null || soapNameSpace.isEmpty()
//					|| soapMethodName == null || soapMethodName.isEmpty() ) {
//				final String errMsg = "SOAP接続の場合､soapNameSpace及びsoapMethodNameは必ず指定して下さい｡";
//				throw new HybsSystemException( errMsg );
//			}
//
//			if( postData != null && postData.length() > 0 ) {
//				conn = new SOAPConnect( urlStr,authUserPass, soapNameSpace, soapMethodName, postData );
//			}
//			else {
//				conn = new SOAPConnect( urlStr,authUserPass, soapNameSpace, soapMethodName, keys, vals );
//			}
//		}
//		else {
			conn = new HttpConnect( urlStr,authUserPass );		// HttpConnect は､GET でも後付で引数を渡せます｡

			conn.setDebug( isDebug() );

			if( keys != null ) {
				for( int i=0; i<keys.length; i++ ) {
					conn.addRequestProperty( keys[i] , vals[i] );
				}
			}

//			if( "POST".equalsIgnoreCase( method ) ) { conn.usePost( true ); }		// 6.9.0.1 (2018/02/05)
			if( check( method,POST_SET ) ) { conn.usePost( true ); }				// 5.10.10.0 (2019/03/29)

//			String urlEnc = XHTMLTag.urlEncode( keys,vals );
			if( postKey != null ) { // 4.1.0.0 (2007/12/22)
//				method = "POST";
//				urlEnc = urlEnc + "&" + postKey + "=" + postData;	// &連結
				conn.usePost( true );		// 6.9.0.1 (2018/02/05)
				conn.addRequestProperty( postKey , postData );
				if( check( method,POST_SET ) && !"POST".equalsIgnoreCase( method ) ) {
					conn.addHeaderProperty( "X-HTTP-Method-Override" , method );		// 5.10.10.0 (2019/03/29)
				}
			}

//			if( ! "POST".equals( method ) ) { // 4.1.0.0 (2007/12/22)
//				// String urlEnc = XHTMLTag.urlEncode( keys,vals );
//				urlStr = XHTMLTag.addUrlEncode( urlStr,urlEnc );
//			}
//			conn = new URLConnect( urlStr,authUserPass );

			// if( postKey != null ) {
			//	conn.setPostData( postKey,postData );
			// }
//			if( "POST".equals( method ) && keys != null && vals != null ) { // 4.1.0.0 (2007/12/22)
//				conn.setPostData( urlEnc );
//			}
//		}

		// 4.0.0 (2007/07/25) ﾌﾟﾛｷｼの設定追加
		if( proxyHost != null ) {
			conn.setProxy( proxyHost,proxyPort );
		}

		// 5.1.6.0 (2010/05/01) ｴﾝｺｰﾄﾞ指定対応
		if( encode != null && encode.length() > 0 ) {
			conn.setCharset( encode );
		}

		// 6.2.5.0 (2015/06/05) timeout属性追加
		if( timeout >= 0 ) {
			conn.setTimeout( timeout );
		}

		// 5.10.10.2 (2019/04/12)
		if( contentType != null && contentType.length() > 0 ){
			conn.addRequestProperty( "Content-Type", contentType );
//			conn.setConentType( contentType );
		}

		// 5.10.16.0 (2019/10/04)
		if( propKeys != null && propVals != null ) {
			conn.setRequestProperty( propKeys,propVals );
		}

		// 8.0.0.0 (2021/08/31) authJson,authURL 追加
		if( authJson != null && authURL != null ) {
			conn.setAuthJson( authJson,authURL );
		}

		// 8.0.0.0 (2021/09/30) reqJson 追加
		if( reqJson != null ) {
			conn.setReqJson( reqJson );
		}

//		conn.connect();

		return conn;
	}

	/**
	 * 出力ﾃﾞｰﾀをﾊﾟｰｽし､DBTableModel及び属性ﾊﾟﾗﾒｰﾀｰに分解します｡
	 * 現時点では､XMLﾃﾞｰﾀのみﾊﾟｰｽすることが可能です｡
	 *
	 * @og.rev 6.4.3.4 (2016/03/11) forﾙｰﾌﾟを､forEach ﾒｿｯﾄﾞに置き換えます｡
	 *
	 * @param input ｲﾝﾌﾟｯﾄｽﾄﾘｰﾑ
	 */
	private void parse( final InputStream input ) {
		final XML2TableParser parser = new XML2TableParser( input );
		parser.setTableCols( rowKey, colKeys );
		parser.setReturnCols( rtnKeys );
		parser.parse();

		// DBTableModelを生成します｡
		if( isTableOut ) {
			table = DBTableModelUtil.makeDBTable( parser.getCols(), parser.getData(), getResource() );
		}

		// 戻り値を取得し､Attributeに登録します｡
		if( rtnKeys != null ) {
			final Map<String,String> rtn = parser.getRtn();
			// 6.4.3.4 (2016/03/11) forﾙｰﾌﾟを､forEach ﾒｿｯﾄﾞに置き換えます｡
			rtn.forEach( (k,v) -> setRequestAttribute( k,v ) );
		}
	}

	/**
	 * ﾀｸﾞﾘﾌﾞｵﾌﾞｼﾞｪｸﾄをﾘﾘｰｽします｡
	 * ｷｬｯｼｭされて再利用されるので､ﾌｨｰﾙﾄﾞの初期設定を行います｡
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) DBLastSqlの処理は､DBTableModelが新規作成された処理でのみ行う｡
	 * @og.rev 6.2.5.0 (2015/06/05) timeout属性追加
	 * @og.rev 6.9.0.0 (2018/01/31) SOAPConnect 廃止に伴う修正
	 * @og.rev 5.10.9.0 (2019/03/01) storageType,bucketNameの初期を追加
	 * @og.rev 5.10.10.2 (2019/04/12) contentTypeを追加
	 * @og.rev 5.10.16.0 (2019/10/04) propKeys,propVals指定追加
	 * @og.rev 5.10.19.0 (2019/12/27) postDataOnly対応 ････ Ver7では未対応
	 * @og.rev 8.0.0.0 (2021/08/31) authJson,authURL 追加
	 * @og.rev 8.0.1.0 (2021/10/29) useLocal 属性を追加｡storageType , bucketName 削除
	 */
	@Override
	protected void release2() {
		super.release2();
		urlStr			= null;
		proxyHost		= null;		// 4.0.0 (2007/07/25)
		proxyPort		= -1;		// 4.0.0 (2007/07/25)
		keys			= null;
		vals			= null;
		xslFile			= null;
		saveFile		= null;
		postKey			= null;
		postData		= null;
		timeout			= HybsSystem.sysInt( "URL_CONNECT_TIMEOUT" );	// 6.2.5.0 (2015/06/05) 新規追加
		useSystemUser	= true;
		authUserPass	= null;
		display			= false;
		method			= "GET";	// 4.1.0.0 (2007/12/22)
		errNglctFlag	= false;	// 4.1.1.0 (2008/01/22)
//		soapNameSpace	= null;		// 5.1.5.0 (2010/04/01) SOAP対応	// 6.9.0.0 (2018/01/31) SOAPConnect 廃止に伴う修正
//		soapMethodName	= null;		// 5.1.5.0 (2010/04/01) SOAP対応	// 6.9.0.0 (2018/01/31) SOAPConnect 廃止に伴う修正
		table			= null;		// 5.1.5.0 (2010/04/01) DBTableModel出力対応
		tableId			= HybsSystem.TBL_MDL_KEY; // 5.1.5.0 (2010/04/01) DBTableModel出力対応
		rowKey			= null;		// 5.1.5.0 (2010/04/01) DBTableModel出力対応
		colKeys			= null;		// 5.1.5.0 (2010/04/01) DBTableModel出力対応
		rtnKeys			= null;		// 5.1.5.0 (2010/04/01) DBTableModel出力対応
		isTableOut		= false;	// 5.1.5.0 (2010/04/01) DBTableModel出力対応
		isOutParse		= false;	// 5.1.5.0 (2010/04/01) DBTableModel出力対応
		encode			= "UTF-8";	// 5.1.6.0 (2010/05/01) ｴﾝｺｰﾄﾞ指定対応
		isMainTrans		= true;		// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し
//		storageType		= null;		// 5.10.9.0 (2019/03/01) ADD
//		bucketName		= null;		// 5.10.9.0 (2019/03/01) ADD
		contentType		= null;		// 5.10.10.2 (2019/04/12)
		propKeys		= null;		// 5.10.16.0 (2019/10/04)
		propVals		= null; 	// 5.10.16.0 (2019/10/04)
		postDataOnly	= false;	// 5.10.19.0 (2019/12/27)
		authJson		= null;		// 8.0.0.0 (2021/08/31)
		authURL			= null;		// 8.0.0.0 (2021/08/31)
		reqJson			= null;		// 8.0.0.0 (2021/09/30)
		useLocal		= false;	// 8.0.1.0 (2021/10/29) ｸﾗｳﾄﾞ設定を使用しない場合は､true
	}

	/**
	 * 【TAG】ｱｸｾｽする接続先ＵＲＬを指定します｡
	 *
	 * @og.tag
	 * 接続するＵＲＬを指定します｡(例：http:// ･･････)
	 * ?以降のﾊﾟﾗﾒｰﾀが含まれていても構いません｡
	 * このURL に､keys,vals で指定されたﾊﾟﾗﾒｰﾀも追加されます｡
	 *
	 * @param	url	接続先
	 */
	public void setUrl( final String url ) {
		urlStr = nval( getRequestParameter( url ),urlStr );
	}

	/**
	 * 【TAG】ﾌﾟﾛｷｼ経由で接続する場合の､ﾌﾟﾛｷｼﾎｽﾄ名を指定します｡
	 *
	 * @og.tag
	 * 接続先が､ﾌﾟﾛｷｼ経由の場合､ﾌﾟﾛｷｼのﾎｽﾄ名を指定します｡
	 * 例：proxy.opengion.org
	 *
	 * @param	host	ﾌﾟﾛｷｼﾎｽﾄ名
	 */
	public void setProxyHost( final String host ) {
		proxyHost = nval( getRequestParameter( host ),proxyHost );
		useSystemUser = false;	// ﾌﾟﾛｷｼ接続時は､ｼｽﾃﾑﾕｰｻﾞｰは使えません｡
	}

	/**
	 * 【TAG】ﾌﾟﾛｷｼ経由で接続する場合の､ﾌﾟﾛｷｼﾎﾟｰﾄ番号を指定します｡
	 *
	 * @og.tag
	 * 接続先が､ﾌﾟﾛｷｼ経由の場合､ﾌﾟﾛｷｼのﾎﾟｰﾄ番号を指定します｡
	 * 例：8080
	 *
	 * @param	port	ﾌﾟﾛｷｼﾎﾟｰﾄ番号
	 */
	public void setProxyPort( final String port ) {
		proxyPort = nval( getRequestParameter( port ),proxyPort );
	}

	/**
	 * 【TAG】ｱｸｾｽﾊﾟﾗﾒｰﾀｷｰをCSV形式で複数指定します｡
	 *
	 * @og.tag
	 * ｱｸｾｽする ＵＲＬに追加するﾊﾟﾗﾒｰﾀのｷｰを指定します｡
	 * CSV形式で複数指定できます｡
	 * vals 属性には､ｷｰに対応する値を､設定してください｡
	 * 例:<b>keys="command,SYSTEM_ID"</b> vals="NEW,GE"
	 * 分解方法は､CSV変数を先に分解してから､getRequestParameter で値を取得します｡
	 * こうしないとﾃﾞｰﾀ自身にｶﾝﾏを持っている場合に分解をﾐｽる為です｡
	 *
	 * @og.rev 6.9.0.0 (2018/01/31) SOAPConnect 廃止に伴う修正
//	 * [SOAP対応]
//	 * SOAPによるWebｻｰﾋﾞｽの呼び出しの場合､keys,valsに指定された値より､env:Envelopﾀｸﾞを
//	 * rootﾀｸﾞとするXMLﾃﾞｰﾀを生成します｡
//	 * (BODY部分に直接XMLﾃﾞｰﾀを出力することも可能です｡)
//	 * この際､項目名に'&gt;'を含めることで､階層的なXMLﾃﾞｰﾀを表現することができます｡
//	 * 例)
//	 *   [属性定義]
//	 *   keys="param0&gt;AAA,param0&gt;BBB,param1&gt;CCC,DDD"
//	 *   vals="v1,v2,v3,v4"
//	 *   [XMLﾃﾞｰﾀ(※ﾃﾞｰﾀ部のみ)]
//	 *   &lt;param0&gt;
//	 *     &lt;AAA&gt;v1&lt;/AAA&gt;
//	 *     &lt;BBB&gt;v2&lt;/BBB&gt;
//	 *   &lt;/param0&gt;
//	 *   &lt;param1&gt;
//	 *     &lt;CCC&gt;v3&lt;/CCC&gt;
//	 *   &lt;/param1&gt;
//	 *   &lt;DDD&gt;v4&lt;/DDD&gt;
//	 * 項目の値を"null"とすることで､XMLで言うところの｢xsi:nil=\"true\"｣のﾃﾞｰﾀを表現すること
//	 * もできます｡
//	 * また､ｷｰ名の先頭を'&#064;'にすることで､項目名に名前空間のPREFIXを付加することができます｡
//	 * 一般的には､JavaやRubyで実装されたWebｻｰﾋﾞｽを呼び出しする場合は､必要ありませんが､
//	 * .NETで実装されたWebｻｰﾋﾞｽを呼び出しする場合は､各項目にPREFIXを付与しないと､正しく
/	 * ﾊﾟﾗﾒｰﾀｰを渡すことができません｡
//	 * ※現時点では､keysの階層定義は､2階層まで対応しています｡
//	 *   3階層以上のXML構造を定義する場合は､postFile属性によるﾌｧｲﾙ指定又は､Body部分で直接
//	 *   XMLﾃﾞｰﾀを記述して下さい｡
	 *
	 * @param	key ﾘﾝｸ先に渡すｷｰ (CSV形式)
	 * @see		#setVals( String )
	 */
	public void setKeys( final String key ) {
		keys = getCSVParameter( key );
	}

	/**
	 * 【TAG】keys属性に対応する値をCSV形式で複数指定します｡
	 *
	 * @og.tag
	 * ｷｰに設定した値を､CSV形式で複数して出来ます｡
	 * 指定順序は､ｷｰと同じにしておいて下さい｡
	 * 例:<b>keys="command,SYSTEM_ID"</b> vals="NEW,GE"
	 * 分解方法は､CSV変数を先に分解してから､getRequestParameter で値を取得します｡
	 * こうしないとﾃﾞｰﾀ自身にｶﾝﾏを持っている場合に分解をﾐｽる為です｡
	 *
	 * @param	val 設定値 keys属性に対応する値(CSV形式)
	 * @see		#setKeys( String )
	 */
	public void setVals( final String val ) {
		vals = getCSVParameter( val );
	}

	/**
	 * 【TAG】送信ﾒｿｯﾄﾞ[GET/POST]を指定します(初期値:GET)｡
//	 * 【TAG】送信ﾒｿｯﾄﾞ[GET/POST/SOAP]を指定します(初期値:GET)｡
	 *
	 * @og.tag
	 * URLConnectTagのﾒｿｯﾄﾞの初期設定はGETです｡
	 * ここで"POST"(大文字)を指定するとkyes,valsの値ｾｯﾄをPOSTで送信します｡
	 * (postKeyが設定されている場合はこの値に関係なくPOSTです)
	 *
	 * @og.rev 4.1.0.0 (2007/12/22) 新規作成
	 * @og.rev 6.9.0.0 (2018/01/31) SOAPConnect 廃止に伴う修正
	 *
	 * @param	post_get	送信ﾒｿｯﾄﾞ [GET/POST]
//	 * @param	post_get	送信ﾒｿｯﾄﾞ [GET/POST/SOAP]
	 */
	public void setMethod ( final String post_get ) {
		method = nval( getRequestParameter( post_get ), method );
	}

	/**
	 * 【TAG】Basic認証で接続するﾕｰｻﾞｰにSYSTEMﾕｰｻﾞｰを使用するかどうか[true/false]を指定します(初期値:true)｡
	 *
	 * @og.tag
	 * useSystemUser="true"(初期値) の場合､URL接続時のｺﾈｸｼｮﾝに､Basic認証を
	 * 使用しますが､その時のﾕｰｻﾞｰにｼｽﾃﾑﾕｰｻﾞｰ(SYSTEM)を使用します｡
	 * useSystemUser="false"の場合は､authUserPass で指定したﾕｰｻﾞｰ：ﾊﾟｽﾜｰﾄﾞを
	 * 使用します｡authUserPass で､何も指定されなかった場合は､Basic認証を使用しません｡
	 * 初期値は､true(SYSTEMﾕｰｻﾞｰ認証する) です｡
	 *
	 * @param   flag SYSTEMﾕｰｻﾞｰ認証 [true:SYSTEMﾕｰｻﾞｰ認証する/false:この接続のﾕｰｻﾞｰで認証する]
	 * @see #setAuthUserPass( String )
	 */
	public void setUseSystemUser( final String flag ) {
		useSystemUser = nval( getRequestParameter( flag ),useSystemUser );
	}

	/**
	 * 【TAG】Basic認証を使用して接続する場合のﾕｰｻﾞｰ:ﾊﾟｽﾜｰﾄﾞを指定します(初期値:null)｡
	 *
	 * @og.tag
	 * 接続時のﾕｰｻﾞｰとﾊﾟｽﾜｰﾄﾞを､USER:PASSWD 形式で指定します｡
	 * useSystemUser="false"の場合は､ここで指定したﾕｰｻﾞｰとﾊﾟｽﾜｰﾄﾞを使用します｡
	 * その場合に､何も指定しない場合は､Basic認証を使用しません｡
	 *
	 * @param	userPass	ﾕｰｻﾞｰとﾊﾟｽﾜｰﾄﾞ (USER:PASSWD形式)
	 * @see #setUseSystemUser( String )
	 */
	public void setAuthUserPass( final String userPass ) {
		authUserPass = nval( getRequestParameter( userPass ),authUserPass );
	}

	/**
	 * 【TAG】接続の結果を表示する場合にXSLT変換する場合のﾌｧｲﾙを指定します｡
	 *
	 * @og.tag
	 *
	 * 接続先のﾃﾞｰﾀが､XML形式の場合､そのままでは､画面出力できない場合が
	 * あります｡通常は､HTML形式に変換しますが､その変換に､ XSL ﾌｧｲﾙを
	 * 指定することが可能です｡
	 * display="true" の場合や､saveFile を指定した場合に､適用されます｡
	 *
	 * @param	file	XSLTﾌｧｲﾙ
	 * @see #setSaveFile( String )
	 * @see #setDisplay( String )
	 */
	public void setXslFile( final String file ) {
		xslFile = HybsSystem.url2dir( nval( getRequestParameter( file ),xslFile ) );
	}

	/**
	 * 【TAG】接続の結果を表示するかどうかを指定します(初期値:false)｡
	 *
	 * @og.tag
	 * true で､接続結果を表示します｡ false では､何も表示しません(初期値:false)
	 * 接続結果を表示する使い方より､admin 画面に接続して､ｷｬｯｼｭｸﾘｱするような
	 * 使い方が多いと考え､初期値は､false になっています｡
	 * xslFile が指定されている場合､XSLT変換して画面表示します｡
	 * display="true" と､saveFile を併用することはできません｡
	 *
	 * @param	flag	結果表示 [true:する/false:しない]
	 * @see #setSaveFile( String )
	 * @see #setXslFile( String )
	 */
	public void setDisplay( final String flag ) {
		display = nval( getRequestParameter( flag ),display );

		if( display && saveFile != null ) {
			final String errMsg = "display=\"true\" と､saveFile を併用することはできません｡";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】接続の結果をﾌｧｲﾙに保存します｡
	 *
	 * @og.tag
	 * 接続先のﾃﾞｰﾀを受け取って､ﾌｧｲﾙに保存します｡その場合､
	 * xslFile が指定されている場合､XSLT変換してｾｰﾌﾞします｡
	 * display="true" と､saveFile を併用することはできません｡
	 *
	 * @param	file	保存先ﾌｧｲﾙ
	 * @see #setXslFile( String )
	 * @see #setDisplay( String )
	 */
	public void setSaveFile( final String file ) {
		saveFile = HybsSystem.url2dir( nval( getRequestParameter( file ),saveFile ) );

		if( display ) {
			final String errMsg = "display=\"true\" と､saveFile を併用することはできません｡";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】POST を使って､postFile属性のﾌｧｲﾙ内容を送信する時のｷｰを指定します｡
	 *
	 * @og.tag
	 * 接続先にﾊﾟﾗﾒｰﾀ(引数)を投げる場合に､POST を使用できます｡
	 * そのときの ｷｰをここで指定します｡
	 * POSTするﾃﾞｰﾀは､postFileで指定されたﾌｧｲﾙか､BODY部に記述された文字列です｡
	 *
	 * @param	key	ﾌｧｲﾙ内容送信ｷｰ
	 * @see  #setPostFile( String )
	 */
	public void setPostKey( final String key ) {
		postKey = nval( getRequestParameter( key ),postKey );
	}

	/**
	 * 【TAG】接続ﾀｲﾑｱｳﾄ時間を(秒)で指定します
	 *		(初期値:URL_CONNECT_TIMEOUT[={@og.value SystemData#URL_CONNECT_TIMEOUT}])｡
	 *
	 * @og.tag
	 * 実際には､java.net.URLConnection#setConnectTimeout(int) に 1000倍して設定されます｡
	 * 0 は､無限のﾀｲﾑｱｳﾄ､ﾏｲﾅｽは､設定しません｡(つまりJavaの初期値のまま)
	 * (初期値:ｼｽﾃﾑ定数のURL_CONNECT_TIMEOUT[={@og.value SystemData#URL_CONNECT_TIMEOUT}])｡
	 *
	 * @og.rev 6.2.5.0 (2015/06/05) timeout属性追加
	 *
	 * @param	tout	ﾀｲﾑｱｳﾄ時間(秒) (ｾﾞﾛは､無制限)
	 * @see		org.opengion.fukurou.util.HttpConnect#setTimeout(int)
	 * @see		java.net.URLConnection#setConnectTimeout(int)
	 */
	public void setTimeout( final String tout ) {
		timeout = nval( getRequestParameter( tout ),timeout );
	}

	/**
	 * 【TAG】POST を使って､postFile属性のﾌｧｲﾙ内容を送信します｡
	 *
	 * @og.tag
	 * 接続先にﾊﾟﾗﾒｰﾀ(引数)を投げる場合に､POST を使用できます｡
	 * そのときの 送信ﾃﾞｰﾀのﾌｧｲﾙをここで指定します｡
	 * postKey のみ指定されて､postFile が指定されない場合は､BODY部を送信します｡
//	 * SOAPによる呼び出しの場合は､ここ(BODY部での定義を含む)で､送信するXMLﾃﾞｰﾀを
//	 * 定義することができます｡
	 *
	 * @og.rev 6.4.5.1 (2016/04/28) FileStringのｺﾝｽﾄﾗｸﾀｰ変更
	 * @og.rev 6.4.5.2 (2016/05/06) fukurou.util.FileString から､fukurou.util.FileUtil に移動｡
	 * @og.rev 6.9.0.0 (2018/01/31) SOAPConnect 廃止に伴う修正
	 *
	 * @param	file	送信ﾌｧｲﾙ
	 * @see  #setPostKey( String )
	 */
	public void setPostFile( final String file ) {
		final String postFile = nval( getRequestParameter( file ),null );

		if( postFile != null ) {
			// 6.4.5.1 (2016/04/28) FileStringのｺﾝｽﾄﾗｸﾀｰ変更
			postData = FileUtil.getValue( HybsSystem.url2dir( postFile ) , HybsConst.UTF_8 );
		}
	}

	/**
	 * 【TAG】(通常は使いません) 接続ｴﾗｰを無視する場合にtrueとします(初期値false)｡
	 *
	 * @og.tag
	 * trueにするとConnectで発生したｴﾗｰを投げずに処理を続行します｡
	 * (標準ｴﾗｰ出力にｴﾗｰ内容は出力されます)
	 * 接続ｴﾗｰが発生しても処理を中断したくない場合に設定します｡
	 *
	 * @og.rev 4.1.1.0 (2008/01/22) 新規追加
	 *
	 * @param	flag	ｴﾗｰを無視するか [true:する/false:しない]
	 */
	public void setErrNeglect( final String flag ) {
		errNglctFlag = nval( getRequestParameter( flag ), errNglctFlag );
	}

//	/**
//	 * 【TAG】SOAPによるWebｻｰﾋﾞｽの呼び出しで､ﾒｿｯﾄﾞ名及びﾊﾟﾗﾒｰﾀｰの名前空間を指定します｡
//	 *
//	 * @og.tag
//	 * SOAPによるWebｻｰﾋﾞｽの呼び出しで､ﾒｿｯﾄﾞ名及びﾊﾟﾗﾒｰﾀｰの名前空間を指定します｡
//	 * この名前空間は､通常WSDLﾌｧｲﾙのdescriptionsﾀｸﾞのtargetNamespace属性の値により
//	 * 定義されます｡
//	 *
//	 * @og.rev 5.1.5.0 (2010/04/01) 新規追加
//	 * @og.rev 6.9.0.0 (2018/01/31) SOAPConnect 廃止に伴う修正
//	 *
//	 * @param	ns	名前空間
//	 */
//	public void setSoapNameSpace( final String ns ) {
//		soapNameSpace = nval( getRequestParameter( ns ), soapNameSpace );
//	}
//
//	/**
//	 * 【TAG】SOAPによるWebｻｰﾋﾞｽの呼び出しで､ﾒｿｯﾄﾞ名を指定します｡
//	 *
//	 * @og.tag
//	 * SOAPによるWebｻｰﾋﾞｽの呼び出しで､ﾒｿｯﾄﾞ名を指定します｡
//	 * WSDLﾌｧｲﾙで定義されるoperationﾀｸﾞのname属性の値に相当します｡
//	 *
//	 * @og.rev 5.1.5.0 (2010/04/01) 新規追加
//	 * @og.rev 6.9.0.0 (2018/01/31) SOAPConnect 廃止に伴う修正
//	 *
//	 * @param	method	ﾒｿｯﾄﾞ名
//	 */
//	public void setSoapMethodName( final String method ) {
//		soapMethodName = nval( getRequestParameter( method ), soapMethodName );
//	}

	/**
	 * 【TAG】(通常は使いません)結果のDBTableModelを､sessionに登録するときのｷｰを指定します
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])｡
	 *
	 * @og.tag
	 * 検索結果より､DBTableModelｵﾌﾞｼﾞｪｸﾄを作成します｡これを､下流のviewﾀｸﾞ等に
	 * 渡す場合に､通常は､session を利用します｡その場合の登録ｷｰです｡
	 * query ﾀｸﾞを同時に実行して､結果を求める場合､同一ﾒﾓﾘに配置される為､
	 * この tableId 属性を利用して､ﾒﾓﾘ空間を分けます｡
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])｡
	 *
	 * @param	id ﾃｰﾌﾞﾙID (sessionに登録する時のID)
	 */
	public void setTableId( final String id ) {
		tableId   = nval( getRequestParameter( id ),tableId );
	}

	/**
	 * 【TAG】結果のXMLをDBTableModelに変換する際に､行のｷｰとなるﾀｸﾞ名を指定します｡
	 *
	 * @og.tag
	 * 結果のXMLを表形式に変換する際に､行のｷｰとなるﾀｸﾞ名を指定します｡
	 * このｷｰのﾀｸﾞを基点として､colKeysで指定されたﾀｸﾞの値を各項目の値として取り込みます｡
	 * (詳細は､colKeysのﾀｸﾞﾘﾌﾞ属性ﾏﾆｭｱﾙを参照して下さい｡)
	 *
	 * @param	key 行のｷｰとなるﾀｸﾞ名
	 * @see #setColKeys( String )
	 */
	public void setRowKey( final String key ) {
		rowKey = nval( getRequestParameter( key ),rowKey );
	}

	/**
	 * 【TAG】結果のXMLをDBTableModelに変換する際に､項目のｷｰとなるﾀｸﾞ名の一覧を指定します｡
	 *
	 * @og.tag
	 * 結果のXMLをDBTableModelに変換する際に､項目のｷｰとなるﾀｸﾞ名の一覧を指定します｡
	 * rowKeyで行を､colKeysで項目を表現し､rowKeyのﾀｸﾞで囲われた固まりを1つの行とみなします｡
	 * このため､colKeysに指定されたｷｰのﾀｸﾞでも､rowKeyの外にある場合は､取り込み対象となりません｡
	 *
	 * また､ｷｰにPARENT_TAG､PARENT_FULL_TAGを指定することで､rowKeyで指定されたﾀｸﾞの
	 * 直近の親ﾀｸﾞ､及びﾌﾙの親ﾀｸﾞ名(親ﾀｸﾞの階層を"&gt;[ﾀｸﾞA]&gt;[ﾀｸﾞB]&gt;[ﾀｸﾞC]&gt;"で表現)を
	 * 取得することができます｡
	 *
	 * @param	keys 項目ｷｰ ﾀｸﾞ名の一覧(CSV形式)
	 */
	public void setColKeys( final String keys ) {
		colKeys = StringUtil.csv2Array( getRequestParameter( keys ) );
	}

	/**
	 * 【TAG】結果のXMLを変換する際に､ﾊﾟﾗﾒｰﾀ(Attributes)として取り込むｷｰの一覧を指定します｡
	 *
	 * @og.tag
	 * 結果のXMLを変換する際に､ﾊﾟﾗﾒｰﾀ(Attributes)として取り込むｷｰの一覧を指定します｡
	 * ここで指定されたｷｰのﾀｸﾞの値を取り出し､{&#064;XX}形式(Attributes)として処理できるようにします｡
	 *
	 * @param	keys 戻り値ｷｰ ﾊﾟﾗﾒｰﾀｰ(Attributes)のｷｰとなるﾀｸﾞ名の一覧(CSV形式)
	 */
	public void setRtnKeys( final String keys ) {
		rtnKeys = StringUtil.csv2Array( getRequestParameter( keys ) );
	}

	/**
	 * 【TAG】ﾃﾞｰﾀの入出力のｴﾝｺｰﾄﾞを指定します(初期値:UTF-8)｡
	 *
	 * @og.tag
	 * ﾃﾞｰﾀの入出力のｴﾝｺｰﾄﾞを指定します｡
	 * 初期値は､"UTF-8"です｡
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 * @param	enc	ｴﾝｺｰﾄﾞ
	 */
	public void setEncode( final String enc ) {
		encode = nval( getRequestParameter( enc ), encode );
	}

	/**
	 * ﾀｸﾞの名称を､返します｡
	 * 自分自身のｸﾗｽ名より､自動的に取り出せないため､このﾒｿｯﾄﾞをｵｰﾊﾞｰﾗｲﾄﾞします｡
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @return  ﾀｸﾞの名称
	 * @og.rtnNotNull
	 */
	@Override
	protected String getTagName() {
		return "urlConnect" ;
	}

	/**
	 * 【TAG】(通常は使いません)ﾀｸﾞで処理される処理がﾒｲﾝとなるﾄﾗﾝｻﾞｸｼｮﾝ処理かどうかを指定します(初期値:true)｡
	 *
	 * @og.tag
	 * この値は､ﾌｧｲﾙﾀﾞｳﾝﾛｰﾄﾞ処理に影響します｡この値がtrueに指定された時にcommitされたDBTableModelが
	 * ﾌｧｲﾙﾀﾞｳﾝﾛｰﾄﾞの対象の表になります｡
	 *
	 * このﾊﾟﾗﾒｰﾀｰは､通常､各ﾀｸﾞにより実装され､ﾕｰｻﾞｰが指定する必要はありません｡
	 * 但し､1つのJSP内でDBTableModelが複数生成される場合に､前に処理したDBTableModelについてﾌｧｲﾙﾀﾞｳﾝﾛｰﾄﾞをさせたい
	 * 場合は､後ろでDBTableModelを生成するﾀｸﾞで､明示的にこの値をfalseに指定することで､ﾌｧｲﾙﾀﾞｳﾝﾛｰﾄﾞ処理の対象から
	 * 除外することができます｡
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @param  flag ﾒｲﾝﾄﾗﾝｻﾞｸｼｮﾝかどうか [true:ﾒｲﾝ/false:その他]
	 */
	public void setMainTrans( final String flag ) {
		isMainTrans = nval( getRequestParameter( flag ),isMainTrans );
	}

//	/**
//	 * 【TAG】保存先ｽﾄﾚｰｼﾞﾀｲﾌﾟを設定します｡
//	 *
//	 * @og.tag
//	 * ﾌｧｲﾙを読み取り元の､ｽﾄﾚｰｼﾞﾀｲﾌﾟを設定します｡
//	 * 未設定の場合は､ｼｽﾃﾑﾘｿｰｽの｢CLOUD_TARGET｣が参照されます｡
//	 * 自身のｻｰﾊﾞを指定する場合は､｢default｣を設定してください｡
//	 *
//	 * @og.rev 5.10.9.0 (2019/03/01) 新規追加
//	 * @og.rev 8.0.1.0 (2021/10/29) 削除
//	 *
//	 * @param storage 保存先ｽﾄﾚｰｼﾞﾀｲﾌﾟ
//	 */
//	public void setStorageType( final String storage ) {
//		storageType = nval( getRequestParameter( storage ), storageType );
//	}

//	/**
//	 * 【TAG】保存先ﾊﾞｹｯﾄ名を設定します｡
//	 *
//	 * @og.tag
//	 * ﾌｧｲﾙを読み取り元の､ﾊﾞｹｯﾄ名を指定します｡
//	 * ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ利用時のみ有効です｡
//	 * 未設定の場合は､ｼｽﾃﾑﾘｿｰｽの｢CLOUD_BUKET｣が参照されます｡
//	 *
//	 * @og.rev 5.10.9.0 (2019/03/01) 新規追加
//	 * @og.rev 8.0.1.0 (2021/10/29) 削除
//	 *
//	 * @param bucket 保存先ﾊﾞｹｯﾄ名
//	 */
//	public void setBucketName( final String bucket ) {
//		bucketName = nval( getRequestParameter( bucket ), bucketName );
//	}

	/**
	 * 【TAG】送信ﾍｯﾀﾞのｺﾝﾃﾝﾄﾀｲﾌﾟを指定します｡
	 *
	 * @og.tag
	 * 通常は指定不要です｡
	 * API等でapplication/json等を指定された場合に設定してください｡
	 *
	 * @og.rev 5.10.10.2 (2019/04/12) 追加
	 *
	 * @param ctype ｺﾝﾃﾝﾄﾀｲﾌﾟ
	 */
	public void setContentType( final String ctype ) {
		contentType = nval( getRequestParameter( ctype ), contentType );
	}

	/**
	 * 【TAG】ｺﾈｸｼｮﾝのrequestParameterに渡すｷｰをCSV形式で複数指定します｡
	 *
	 * @og.tag
	 * 分解方法は､CSV変数を先に分解してから､getRequestParameter で値を取得します｡
	 *
	 * @og.rev 5.10.16.0 (2019/10/04) propKeys,propVals指定追加
	 *
	 * @param	key ｷｰ(CSV形式)
	 */
	public void setPropKeys( final String key ) {
		propKeys = nval( getRequestParameter( key ), propKeys );
	}

	/**
	 * 【TAG】ｺﾈｸｼｮﾝのrequestParameterに渡す値をCSV形式で複数指定します｡
	 *
	 * @og.tag
	 * 分解方法は､CSV変数を先に分解してから､getRequestParameter で値を取得します｡
	 *
	 * @og.rev 5.10.16.0 (2019/10/04) propKeys,propVals指定追加
	 *
	 * @param	val 値(CSV形式) 新規追加
	 */
	public void setPropVals( final String val ) {
		propVals = nval( getRequestParameter( val ), propVals );
	}

	/**
	 * 【TAG】postDataのみ送信ﾌﾗｸﾞ｡
	 *
	 * @og.tag
	 * trueを設定時､keyなしでpostDataをそのまま送信します｡
	 * 主にapplication/json形式で送信時に利用します｡
	 *
	 * @og.rev 5.10.19.0 (2019/12/27) postDataOnly対応 ････ Ver7では未対応
	 * @param flg postDataのみ送信ﾌﾗｸﾞ
	 */
	public void setPostDataOnly( final String flg ) {
		postDataOnly = nval( getRequestParameter( flg ), postDataOnly);
	}

	/**
	 * 【TAG】JSONｺｰﾄﾞで認証する場合に使用します｡
	 *
	 * @og.tag
	 * JSONｺｰﾄﾞで認証する場合に使用します｡
	 *
	 * @og.rev 8.0.0.0 (2021/08/31) authJson,authURL 追加
	 *
	 * @param	json JSONｺｰﾄﾞの文字列
	 */
	public void setAuthJson( final String json ) {
		authJson = nval( getRequestParameter( json ), authJson );
	}

	/**
	 * 【TAG】JSONｺｰﾄﾞで認証するURLを指定します｡
	 *
	 * @og.tag
	 * JSONｺｰﾄﾞで認証するURLを指定します｡
	 *
	 * @og.rev 8.0.0.0 (2021/08/31) authJson,authURL 追加
	 *
	 * @param	url JSONｺｰﾄﾞで認証するURL
	 */
	public void setAuthURL( final String url ) {
		authURL = nval( getRequestParameter( url ), authURL );
	}

	/**
	 * 【TAG】JSONｺｰﾄﾞでﾘｸｴｽﾄするﾊﾟﾗﾒｰﾀを指定します｡
	 *
	 * @og.tag
	 * JSONｺｰﾄﾞでﾘｸｴｽﾄするﾊﾟﾗﾒｰﾀを指定します｡
	 *
	 * @og.rev 8.0.0.0 (2021/09/30) reqJson 追加
	 *
	 * @param	json JSONｺｰﾄﾞの文字列
	 */
	public void setReqJson( final String json ) {
		reqJson = nval( getRequestParameter( json ), reqJson );
	}

	/**
	 * 【TAG】ｼｽﾃﾑ定数でｸﾗｳﾄﾞ設定されていても､ｸﾗｳﾄﾞ環境を使用しない場合､trueを指定します(初期値:false)｡
	 *
	 * @og.tag
	 * ｸﾗｳﾄﾞ設定は､ｼｽﾃﾑ定数の『CLOUD_TARGET』と『CLOUD_BUCKET』の設定で自動的に使用しますが､
	 * どうしてもﾛｰｶﾙでのみ使いたい場合は､この属性を true に設定します｡
	 * 標準はfalse:設定どおりとなります｡
	 *
	 * true/false以外を指定した場合はfalse扱いとします｡
	 *
	 * @og.rev 8.0.1.0 (2021/10/29) useLocal 属性を追加｡
	 *
	 * @param flag ﾛｰｶﾙ環境のみ [true:ﾛｰｶﾙのみ/false:設定どおり]
	 */
	public void setUseLocal( final String flag ) {
		useLocal = nval( getRequestParameter( flag ),useLocal );
	}

	/**
	 * このｵﾌﾞｼﾞｪｸﾄの文字列表現を返します｡
	 * 基本的にﾃﾞﾊﾞｯｸﾞ目的に使用します｡
	 *
	 * @og.rev 5.10.16.0 (2019/10/04) propKeys,propVals指定追加
	 *
	 * @return このｸﾗｽの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "method"			,method			)
				.println( "urlStr"			,urlStr			)
				.println( "keys"			,StringUtil.array2csv( keys ) )
				.println( "vals"			,StringUtil.array2csv( vals ) )
				.println( "proxyHost"		,proxyHost		)
				.println( "proxyPort"		,proxyPort		)
				.println( "timeout"			,timeout		)		// 6.2.5.0 (2015/06/05) 新規追加
				.println( "useSystemUser"	,useSystemUser	)
				.println( "authUserPass"	,authUserPass	)
				.println( "display"			,display		)
				.println( "postKey"			,postKey		)
				.println( "postData"		,postData		)
				.println( "xslFile"			,xslFile		)
				.println( "saveFile"		,saveFile		)
				.println( "errNglctFlag"	,errNglctFlag	)
//				.println( "soapNameSpace"	,soapNameSpace	)		// 6.9.0.0 (2018/01/31) SOAPConnect 廃止に伴う修正
//				.println( "soapMethodName"	,soapMethodName	)		// 6.9.0.0 (2018/01/31) SOAPConnect 廃止に伴う修正
				.println( "tableId"			,tableId		)
				.println( "rowKey"			,rowKey			)
				.println( "colKeys"			,StringUtil.array2csv( colKeys ) )
				.println( "rtnKeys"			,StringUtil.array2csv( rtnKeys ) )
//				.println( "storageType"		,storageType	)		// 5.10.10.2 (2019/04/12) 追加
//				.println( "bucketName"		,bucketName		)		// 5.10.10.2 (2019/04/12) 追加
				.println( "contentType"		,contentType	)		// 5.10.10.2 (2019/04/12) 追加
				.println( "propKeys"		,propKeys		)		// 5.10.16.0 (2019/10/04) 追加
				.println( "propVals"		,propVals		)		// 5.10.16.0 (2019/10/04) 追加
				.println( "Other..."		,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
