/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.model;

import java.util.regex.Pattern ;

/**
 * Native Type(int型やlong型等)を管理するEnum型です。
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author   Hiroki Nakamura
 * @since    JDK5.0,
 */
public enum NativeType {

	/** NATIVEの型 [int] の識別コード (INT) */
	INT		,

	/** NATIVEの型 [long] の識別コード (LONG) */
	LONG	,

	/** NATIVEの型 [double] の識別コード (DOUBLE) */
	DOUBLE	,

	/** NATIVEの型 [String] の識別コード (STRING) */
	STRING	,

	/** NATIVEの型 [Calendar] の識別コード (CALENDAR) */
	CALENDAR ;

	private static final Pattern P_INT		= Pattern.compile("-?[0-9]{1,9}");
	private static final Pattern P_LONG		= Pattern.compile("-?[0-9]+");
	private static final Pattern P_DOUBLE	= Pattern.compile("-?[0-9]+\\.?[0-9]+");
	private static final Pattern P_CALENDAR	= Pattern.compile("[0-9]{4}/[01]{1}[0-9]{1}/[0-3]{1}[0-9]{1}");

	/**
	 * 指定の文字列が、どの、NativeType なのか、判定します。
	 *
	 * 判定ロジックは、以下の様にしています。
	 * <pre>
	 *   INT      = Pattern.compile("-?[0-9]{1,9}");
	 *   LONG     = Pattern.compile("-?[0-9]+");
	 *   DOUBLE   = Pattern.compile("-?[0-9]+\\.?[0-9]+");
	 *   CALENDAR = Pattern.compile("[0-9]{4}/[01]{1}[0-9]{1}/[0-3]{1}[0-9]{1}");
	 * </pre>
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) 新規追加
	 * @og.rev 6.3.9.1 (2015/11/27) メソッドの出口は、最後の１か所にすべきです(PMD)。
	 *
	 * @param	str	判定する文字列
	 *
	 * @return	判定結果
	 * @og.rtnNotNull
	 */
	public static NativeType getType( final String str ) {

		final NativeType type ;

		if(                          str == null     ) { type = STRING   ; }
		else if( P_INT.matcher(      str ).matches() ) { type = INT      ; }
		else if( P_LONG.matcher(     str ).matches() ) { type = LONG     ; }
		else if( P_DOUBLE.matcher(   str ).matches() ) { type = DOUBLE   ; }
		else if( P_CALENDAR.matcher( str ).matches() ) { type = CALENDAR ; }
		else                                           { type = STRING   ; }

		return type;
	}
}
