/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.system;								// 6.4.2.0 (2016/01/29) package変更 fukurou.util → fukurou.system

import java.nio.charset.Charset;									// 5.5.2.6 (2012/05/25)

import java.net.NetworkInterface;									// 8.0.0.0 (2021/07/01)
import java.util.Enumeration;										// 8.0.0.0 (2021/07/01)
import java.net.InetAddress;

/**
 * 共通的に使用される固定値を集約したクラスです。
 *
 * 全変数は、public static final 宣言されています。
 *
 * @og.rev 6.1.0.0 (2014/12/26) 固定値を集約したクラス対応
 * @og.rev 6.4.2.0 (2016/01/29) package変更 fukurou.util → fukurou.system
 *
 * @og.group その他
 *
 * @version  6.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK8.0,
 */
public final class HybsConst {

	/** 8.0.0.0 (2021/07/01) 実行しているサーバーの名称 */
	public static final String HOST_NAME ;
	/** 8.0.0.0 (2021/07/01) 実行しているサーバーのIPアドレス */
	public static final String HOST_ADRS ;

	/**
	 * ホスト名と、IPアドレスを取得
	 *
	 * Java VM が実行しているホスト名と、IPアドレスを取得します。
	 * InetAddress.getLocalHost().getHostName() で求められる値は、Virtual ｱﾄﾞﾚｽなどの
	 * 複数考えられる為、出来るだけ直接設定されているIPｱﾄﾞﾚｽに近い値を取得するようにします。
	 * でも、完全には特定できないと思われます。
	 *
	 * @og.rev 7.3.1.1 (2021/02/25) ホスト名と、IPアドレスを取得
	 * @og.rev 8.0.0.0 (2021/07/01) fukurou.system.HybsConst に移植
	 */
	static {
		String dmnHost = "Unknown" ;
		String dmnAdrs = "Unknown" ;
		try {
			boolean isNext = true;
			final Enumeration<NetworkInterface> enuIfs = NetworkInterface.getNetworkInterfaces();
			while( isNext && enuIfs.hasMoreElements() ) {
				final NetworkInterface ni = enuIfs.nextElement();

				final String displayName = ni.getDisplayName();
				if( displayName.contains("Virtual") ) { continue; }

				final Enumeration<InetAddress> enuIP = ni.getInetAddresses();
				while( isNext && enuIP.hasMoreElements() ) {
					final InetAddress adrs = enuIP.nextElement();

					if( adrs.isLinkLocalAddress() || adrs.isLoopbackAddress() ) { continue; }

					dmnHost = adrs.getHostName() ;			// adrs.getCanonicalHostName() はとりあえず使わないでおく。
					dmnAdrs = adrs.getHostAddress() ;
					isNext = false;
//					break;									// Avoid using a branching statement as the last in a loop.
				}
			}
		}
		catch( final Throwable th ) {
			System.err.println( "HOST_NAME and HOST_ADRS Unknown!" );
		}
		HOST_NAME = dmnHost;
		HOST_ADRS = dmnAdrs;
	}

	/** バッファの初期容量を設定する固定値(通常より若干多い目)。  {@value} */
	public static final int BUFFER_SMALL  = 100;

	/** バッファの初期容量を設定する固定値(通常より多い目)。  {@value} */
	public static final int BUFFER_MIDDLE = 200;

	/** バッファの初期容量を設定する固定値(通常より大幅に多い目)。  {@value} */
	public static final int BUFFER_LARGE  = 500;

	/** システム依存の改行記号(String)。	*/
	public static final String CR = System.getProperty("line.separator");

	/** HTMLでの改行記号( &lt;br /&gt; )。	*/
	// 7.0.1.0 (2018/10/15) XHTML → HTML5 対応(空要素の、"／＞" 止めを、"＞" に変更します)。
//	public static final String BR = "<br />" + CR ;
	public static final String BR = "<br>" + CR ;

	/** システム依存のファイルセパレーター文字(char)。	*/
	public static final char FS = System.getProperty("file.separator").charAt(0);

	/** タブ文字(char)。	*/
	public static final char TAB = '\t';

	/**
	 * プラットフォーム依存のデフォルトの Charset です。
	 * プラットフォーム依存性を考慮する場合、エンコード指定で作成しておく事をお勧めします。
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) findbugs対応
	 * @og.rev 6.4.2.0 (2016/01/29) fukurou.util.StringUtil → fukurou.system.HybsConst に変更
	 */
	public static final Charset DEFAULT_CHARSET = Charset.defaultCharset() ;

	/**
	 * ファイル等を読み取る場合のデフォルトエンコードを指定します。
	 * 通常は、UTF-8 にしておきます。
	 *
	 * @og.rev 6.4.5.1 (2016/04/28) ファイル等を読み取る場合のデフォルトエンコードを指定します。
	 */
	public static final String UTF_8 = "UTF-8" ;

	/**
	 * データ検索時のフェッチサイズ  {@value}
	 *
	 * ORACLEのデフォルトは、10です。
	 * 最適なフェッチ・サイズは、予想される結果サイズの半分または4分の1です。結果セットのサイズかどうかわからず、
	 * フェッチ・サイズの設定を大きくしすぎたり小さくしすぎたりすると、パフォーマンスが低下することに注意してください。
	 * openGionでは、検索時の最大件数を、1000にしていますが、オーバー判定のための＋１件、余分に取得しています。
	 * よって、1/4の250件に、プラスして、251件を、初期値にしておきます。
	 *
	 * @og.rev 6.9.4.1 (2018/04/09) 共通の固定値として登録
	 */
	public static final int DB_FETCH_SIZE = 251 ;

	/**
	 * データ登録時のバッチサイズ  {@value}
	 *
	 * JDBC 2.0仕様では、これをバッチ更新(batch updates)と呼びます。
	 * 標準のバッチ更新は、手動の明示的なモデルです。バッチ値は設定されていません。
	 * 手動で操作をバッチに追加し、明示的にバッチ処理のタイミングを選択します。
	 * openGionでは、検索時のページ表示件数を、100にしていますので、初期値を、 100件にしておきます。
	 *
	 * @og.rev 6.9.4.1 (2018/04/09) 共通の固定値として登録
	 */
	public static final int DB_BATCH_SIZE = 100 ;

	/**
	 * プロパティ―をみて、なければ環境変数から値を取得します。
	 *
	 * 名称は、getenv ですが、処理としては、
	 * return System.getProperty( key , System.getenv( key ) );
	 * です。
	 *
	 * System.getProperty は、java起動時に、-Dxxx=yyy と渡す変数で、key=xxx , 値=yyy を受け取ります。
	 * 変数が未定義の場合は、System.getenv で、環境変数の値を取得します。
	 *
	 * @og.rev 7.2.3.1 (2020/04/17) プロパティ―をみて、なければ環境変数から取る(サービス化対応)
	 *
	 * @param key パラメータキー
	 * @return プロパティ―をみて、なければ環境変数から値を取得
	 */
	public static String getenv( final String key ) {
		return System.getProperty( key , System.getenv( key ) );
	}

	/**
	 * プロパティ―をみて、なければ環境変数から値を取得します。
	 *
	 * 名称は、getenv ですが、処理としては、
	 * System.getProperty( key , System.getenv( key ) ); で受け取った値が、
	 * null か、ゼロ文字列の場合は、defval で指定した文字列を返します。
	 *
	 * System.getProperty は、java起動時に、-Dxxx=yyy と渡す変数で、key=xxx , 値=yyy を受け取ります。
	 * 変数が未定義の場合は、System.getenv で、環境変数の値を取得します。
	 *
	 * @og.rev 7.2.3.1 (2020/04/17) プロパティ―をみて、なければ環境変数から取る(サービス化対応)
	 *
	 * @param key パラメータキー
	 * @param defval	null か、ゼロ文字列の場合の初期値
	 * @return プロパティ―をみて、なければ環境変数から値を取得
	 */
	public static String getenv( final String key , final String defval ) {
		final String rtn = System.getProperty( key , System.getenv( key ) );

		return rtn == null || rtn.isEmpty() ? defval : rtn ;
	}

	/**
	 * デフォルトコンストラクターをprivateにして、
	 * オブジェクトの生成をさせないようにする。
	 */
	private HybsConst() {}
}
