/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.common;

import org.opengion.fukurou.system.OgRuntimeException ;							// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.system.LogWriter;
import org.opengion.fukurou.system.DateSet;										// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.system.MsgUtil;										// 6.4.3.2 (2016/02/19)
// import org.opengion.fukurou.system.HybsConst;								// 5.10.9.0 (2019/03/01)
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.HybsEntry;
import static org.opengion.fukurou.system.HybsConst.CR;							// 6.1.0.0 (2014/12/26) refactoring
import static org.opengion.fukurou.system.HybsConst.FS;							// 6.1.0.0 (2014/12/26) refactoring
import static org.opengion.fukurou.system.HybsConst.BUFFER_MIDDLE;				// 6.2.6.0 (2015/06/19)

import java.util.Set;
import java.util.TreeSet;														// 6.4.3.3 (2016/03/04)

import java.util.Map;
import java.util.concurrent.ConcurrentMap;										// 6.4.3.3 (2016/03/04)
import java.util.concurrent.ConcurrentHashMap;									// 6.4.3.1 (2016/02/12) refactoring
import java.util.concurrent.ConcurrentSkipListMap;								// 6.4.3.3 (2016/03/04)
import java.util.Locale;														// 6.4.3.2 (2016/02/19)
import java.io.File;															// 6.2.4.1 (2015/05/22) REAL_PATH 対応
// 8.0.0.0 (2021/07/01) fukurou.system.HybsConst に移植
// import java.net.InetAddress;													//
// import java.net.UnknownHostException;										// 7.0.6.4 (2019/11/29) ｴﾗｰ全てを捉える
// import java.net.NetworkInterface;											// 7.3.1.1 (2021/02/25)
// import java.util.Enumeration;												// 7.3.1.1 (2021/02/25)

import javax.print.PrintServiceLookup;											// 6.2.6.0 (2015/06/19)
import javax.print.PrintService;												// 6.2.6.0 (2015/06/19)

/**
 * 共通的に、使用されるｼｽﾃﾑ関連ﾒｿｯﾄﾞを集約したｸﾗｽです。
 *
 * 全変数は、public static final 宣言されています。
 * ﾒｿｯﾄﾞは、日付取得、ｴﾗｰﾒｯｾｰｼﾞ処理 を行っています。
 *
 * @og.group 初期化
 *
 * @version	4.0
 * @author	Kazuhiko Hasegawa
 * @since	JDK5.0,
 */
public final class HybsSystem {

	/** ｼｽﾃﾑ設定DBのｷｰと値を、Map に読み込みます。 */
	// 4.0.0 (2005/01/31) ｼｽﾃﾑﾊﾟﾗﾒｰﾀの管理ｵﾌﾞｼﾞｪｸﾄ
	/** 6.4.3.1 (2016/02/12) ConcurrentHashMap で同期処理を行います。 */
	private static final ConcurrentMap<String,String> SYS_MAP = new ConcurrentHashMap<>();		// 6.4.3.3 (2016/03/04)

	// 4.3.1.0 (2008/09/03) ﾘｸｴｽﾄでのみ生成できる設定値のｷｬｯｼｭ
	/** 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。 */
	private static final ConcurrentMap<String, String> REQ_CACHE = new ConcurrentHashMap<>();	// 6.4.9.1 (2016/08/05) reqCache → REQ_CACHE 変更

	/** URL の /(ﾙｰﾄ)に対応する物理ﾃﾞｨﾚｸﾄﾘ */
	private static String  realPath	;						// 4.0.0 (2005/01/31)

	/** ﾘｸｴｽﾄ関係ﾊﾟﾗﾒｰﾀの初期設定ﾌﾗｸﾞ */
	private static boolean paramSetFlag	;					// 4.0.0 (2005/11/30)

	/**
	 * ﾃｰﾌﾞﾙ表示時のﾁｪｯｸﾎﾞｯｸｽのname属性  {@value}
	 * ﾃｰﾌﾞﾙ表示時にﾁｪｯｸﾎﾞｯｸｽを表示しておき、どの行が選ばれたかを
	 * 指定する場合に、その name 属性に指定するｷｰの値です。
	 * 受け取る側は、このｷｰで、request 情報より選ばれた行番号を知ることができます。
	 */
	public static final String ROW_SEL_KEY = "h_rowSel";

	/**
	 * ﾃｰﾌﾞﾙ表示のﾁｪｯｸﾎﾞｯｸｽを特定する id の 名称( id は、この名称＋行番号) {@value}
	 */
	public static final String ROW_ID_KEY = "cb";	// 3.6.0.0 (2004/09/17)

	/** session ｵﾌﾞｼﾞｪｸﾄに関連付ける DBTableModel ｲﾝｽﾀﾝｽのﾃﾞﾌｫﾙﾄｷｰ {@value} */
	public static final String TBL_MDL_KEY = "h_tblmdl" ;

	// 3.8.1.1 (2005/11/21) ExcelOut の整合性を取る為の仕掛け
	/** session ｵﾌﾞｼﾞｪｸﾄに関連付ける ExcelOut の整合性ﾁｪｯｸ用のｷｰ */
	public static final String TBL_MDL_CONKEY = "h_tblmdlCon" ;

	/** session ｵﾌﾞｼﾞｪｸﾄに関連付ける Viewform ｲﾝｽﾀﾝｽのﾃﾞﾌｫﾙﾄｷｰ {@value} */
	public static final String VIEWFORM_KEY = "h_vform" ;

	/** session ｵﾌﾞｼﾞｪｸﾄに関連付ける ViewLink ｲﾝｽﾀﾝｽのﾃﾞﾌｫﾙﾄｷｰ {@value} */
	public static final String VIEWLINK_KEY = "h_vlink" ;

	/** session ｵﾌﾞｼﾞｪｸﾄに関連付ける ViewMark ｲﾝｽﾀﾝｽのﾃﾞﾌｫﾙﾄｷｰ {@value} */
	public static final String VIEW_MARK_KEY = "h_vmark" ;

	// 3.8.6.1 (2006/10/20) 編集時にｶﾗﾑを置き換える 機能に対応
	/** session ｵﾌﾞｼﾞｪｸﾄに関連付ける ViewMark:editor ｲﾝｽﾀﾝｽのﾃﾞﾌｫﾙﾄｷｰ */
	public static final String EDIT_MARK_KEY = "h_emark" ;

	/** session ｵﾌﾞｼﾞｪｸﾄに関連付ける UserInfo ｲﾝｽﾀﾝｽのｷｰ {@value} */
	public static final String USERINFO_KEY = "h_uInfo" ;

	/** session ｵﾌﾞｼﾞｪｸﾄに関連付ける GUIInfo ｲﾝｽﾀﾝｽのｷｰ {@value} */
	public static final String GUIINFO_KEY = "h_gInfo" ;

	/** session ｵﾌﾞｼﾞｪｸﾄに関連付ける ﾘｸｴｽﾄ情報ｷｬｯｼｭのｷｰ {@value} */
	public static final String REQ_CACHE_KEY = "h_req" ;

	/** session ｵﾌﾞｼﾞｪｸﾄに関連付ける ｴﾗｰﾒｯｾｰｼﾞ情報ｷｬｯｼｭのｷｰ {@value} */
	public static final String ERR_MSG_KEY = "h_errMsg" ;

	/** 6.4.4.1 (2016/03/18) application ｵﾌﾞｼﾞｪｸﾄに関連付ける 共通ﾒｯｾｰｼﾞ情報のｷｰ {@value} */
	public static final String COMMON_MSG_KEY = "h_cmnMsg" ;

	// 2.3.1.0 (2002/12/20) 管理者が全てのﾕｰｻﾞｰにﾒｯｾｰｼﾞを同報できるﾀｸﾞを追加
	// /** session 管理者がｽﾃｰﾀｽﾊﾞｰに設定するｵﾌﾞｼﾞｪｸﾄのｷｰ {@value} */
	// 7.4.4.0 (2021/06/30) openGionV8事前準備(未使用の為廃止)
	// public static final String ST_BAR_OBJ_KEY = "h_stBar" ;

	// 3.0.0.0 (2002/12/25) SceduleTag 追加の伴う、Timerｵﾌﾞｼﾞｪｸﾄの管理用ｷｰ追加
	/** Timerｵﾌﾞｼﾞｪｸﾄの管理用ｷｰ {@value} */
	public static final String SCEDULE_KEY = "h_timer" ;

	// 3.0.0.0 (2002/12/25) SceduleTag 追加の伴う、Timerｵﾌﾞｼﾞｪｸﾄの管理用ｷｰ追加
	/** TimerTaskｵﾌﾞｼﾞｪｸﾄの管理用ｷｰ {@value} */
	public static final String SCEDULE_TASK_KEY = "h_timer_task" ;

	// 3.1.2.0 (2003/04/07) ｿｰｽｺｰﾄﾞ中の固定値を、定義ﾌｧｲﾙを使用するように変更する。
	/** Forward時の転送先ﾍﾟｰｼﾞ名をｾｯｼｮﾝに登録するときのｷｰ {@value} */
	public static final String FORWARD_PAGE_KEY = "h_fpg" ;

	// 3.1.2.0 (2003/04/07) ｿｰｽｺｰﾄﾞ中の固定値を、定義ﾌｧｲﾙを使用するように変更する。
	/** Forward時の転送元ﾍﾟｰｼﾞ名をｾｯｼｮﾝに登録するときのｷｰ {@value} */
	public static final String REVIEW_PAGE_KEY = "h_rpg" ;

	// 3.1.8.0 (2003/05/16) BACK_GAMENID のｷｬｯｼｭ方法を、通常のﾘｸｴｽﾄ情報と分離して、管理する。
	/** BACK_GAMENID情報をｾｯｼｮﾝに登録するときのｷｰ {@value} */
	public static final String BACK_GAMENID_KEY = "h_bgid" ;

	// 4.0.1.0 (2007/12/17) BACK_ADDRESS のｷｬｯｼｭ方法を、通常のﾘｸｴｽﾄ情報と分離して、管理する。
	/** BACK_GAMENID情報をｾｯｼｮﾝに登録するときのｷｰ {@value} */
	public static final String BACK_ADDRESS_KEY = "h_bgadrs" ;

	// 3.7.0.3 (2005/03/01) 戻るﾘﾝｸで戻る行番号を指定します。(BACK_GAMENIDとｾｯﾄ)
	/** BACK_ROW情報をｾｯｼｮﾝに登録するときのｷｰ {@value} */
	public static final String BACK_ROW_KEY = "h_brow" ;

	// 3.1.9.0 (2003/05/16) ShellTag で、Shell ｵﾌﾞｼﾞｪｸﾄをｷｰﾌﾟする為のｷｰを作成。
	/** Shell ｵﾌﾞｼﾞｪｸﾄをｾｯｼｮﾝに登録するときのｷｰ {@value} */
	public static final String SHELL_KEEP_KEY = "h_shKeepKey" ;

	// 3.5.0.0 (2003/09/17) ﾕｰｻﾞｰ変更時の、旧ﾕｰｻﾞｰIDのをｷｰﾌﾟする為のｷｰを作成。
	/** oldUserId 文字列をｾｯｼｮﾝに登録するときのｷｰ {@value} */
	public static final String OLD_USERID_KEY = "h_oldID" ;

	// 3.5.4.7 (2004/02/06) DBTableModelのｿｰﾄ時のｶﾗﾑ名の受け渡しｷｰ。
	/** sortColumns 文字列をﾘｸｴｽﾄ情報より取得するときのｷｰ {@value} */
	public static final String SORT_COLUMNS = "h_sortClmns" ;

	// 3.5.4.7 (2004/02/06) DBTableModelのｿｰﾄ時のｿｰﾄ方向の受け渡しｷｰ。
	/** sortAscending をﾘｸｴｽﾄ情報より取得するときのｷｰ {@value} */
	public static final String SORT_ASCENDING = "h_sortAsc" ;

	// 3.5.5.0 (2004/03/12) ｷｰ＋行番号の連結文字列(例：__)を設定します。
	/** ｶﾗﾑID(連結文字列)行番号の連結文字列を定義 {@value} */
	public static final String JOINT_STRING = "__" ;

	// 3.5.5.5 (2004/04/23) DBTableModel の整合性ﾊﾟﾗﾒｰﾀを hidden で出力します。
	/** DBTableModel の整合性ﾊﾟﾗﾒｰﾀをﾘｸｴｽﾄ情報より取得するときのｷｰ {@value} */
	public static final String CONSISTENCY_KEY = "h_consisKey" ;

	// 3.5.5.5 (2004/04/23) 転送不要ﾊﾟﾗﾒｰﾀの先頭に付けるｷｰﾜｰﾄﾞです。
	/** SubmitTag で出力されるﾎﾞﾀﾝﾊﾟﾗﾒｰﾀは、次ﾍﾟｰｼﾞへ転送しません。 {@value} */
	public static final String NO_XFER_KEY = "hX_" ;

	// 3.8.0.0 (2005/06/07) 同一ｾｯｼｮﾝでのﾏﾙﾁ起動対策用ｷｰﾜｰﾄﾞです。
	/** 同一ｾｯｼｮﾝでのﾏﾙﾁ起動された場合に、前に開かれた画面からのｱｸｾｽを禁止します。 */
	public static final String MULTI_SESSION_CHECK = "MSC" ;

	/** 3.8.9.1 (2005/07/29) JFreeChart ｵﾌﾞｼﾞｪｸﾄのｷｬｯｼｭ {@value} */
	public static final String JF_CHART_CACHE_KEY = "h_chartKey" ;

	// 3.8.0.8 (2005/10/03) GET時のURL に使用可能な文字数制限
	/** URL に使用可能な文字数は最大 2,083 文字
	 * http://support.microsoft.com/default.aspx?scid=kb;ja;208427
	 */
	public static final int MAX_GET_LENGTH = 2083 ;

	/** 4.0.0 (2005/01/31) GUIのﾛｰﾙを区切る、区切り文字 {@value} */
	public static final char GUI_DELIMITER = '|' ;

	/** 4.0.0 (2005/01/31) ﾛｰｶﾙﾘｿｰｽのｵﾌﾞｼﾞｪｸﾄ保存時のｷｰ {@value} */
	public static final String LOCAL_RES_KEY = "h_lclRes" ;

	/** 4.0.0 (2005/02/28) Tomcat起動時ｴﾗｰが発生した場合のｴﾗｰｷｰ {@value} */
	public static final String LOCAL_CONTX_ERR_KEY = "h_lclConErr" ;

	/** 4.0.0 (2005/08/31) ｸﾞﾙｰﾌﾟﾒﾆｭｰのｷｬｯｼｭｷｰ {@value} */
	public static final String GUI_GR_MENU_CACHE_KEY = "h_grpCache" ;

	// /** 4.0.0 (2005/09/30) 個人別お気に入りﾒﾆｭｰの GE20 登録ｷｰﾜｰﾄﾞ	{@value} */
	// 7.4.4.0 (2021/06/30) openGionV8事前準備(未使用の為廃止)
	// public static final String GUI_FAV_MENU_KEY = "GUI_FAV_MENU_KEY" ;

	// /** 4.0.0 (2005/06/04) DBTableWriter の columns 属性のｾｯｼｮﾝ登録ｷｰﾜｰﾄﾞ {@value} */
	// 7.4.4.0 (2021/06/30) openGionV8事前準備(未使用の為廃止)
	// public static final String TBL_WRITER_CLMS = "TBL_WRITER_CLMS" ;

	/** 4.1.2.1 (2008/03/13) must 属性をrequest変数にｾｯﾄする場合のｷｰ {@value} */
	public static final String MUST_KEY = "h_must_" ;

	/** 6.2.0.0 (2015/02/27) aimai 属性をrequest変数にｾｯﾄする場合のｷｰ {@value} */
	public static final String AIMAI_KEY = "h_aimai_" ;

	/** 4.3.0.0 (2008/07/04) 最後に Queryを実行した情報の保存時のｷｰ {@value} */
	public static final String DB_LAST_SQL_KEY = "h_lastSql" ;

	/** 4.3.3.0 (2008/10/01) ﾍﾟｰｼﾞ内で画面遷移なしﾓｰﾄﾞを指定しようする場合のｷｰ {@value} */
	public static final String NO_TRANSITION_MODE_KEY = "h_noTrnsKey" ;

	/** 4.3.6.0 (2009/04/01) ｲﾍﾞﾝﾄｶﾗﾑを利用したかどうかのﾁｪｯｸｷｰ {@value} */
	public static final String USE_EVENT_COLUMN_KEY = "h_eventColumn" ;

	/** 5.1.7.0 (2010/06/01) ｲﾍﾞﾝﾄｶﾗﾑをﾗｯﾌﾟするspanﾀｸﾞのｸﾗｽ {@value} */
	public static final String EVENT_COLUMN_CLASS = "h_evElm";

	/** 5.1.7.0 (2010/06/01) ｲﾍﾞﾝﾄｶﾗﾑをﾗｯﾌﾟするspanﾀｸﾞで、子ｶﾗﾑの書き込み可/不可を設定する属性名 {@value} */
	public static final String EVENT_COLUMN_WRITABLE = "h_evWrtbl";

	/** 5.1.7.0 (2010/06/01) ｲﾍﾞﾝﾄｶﾗﾑをﾗｯﾌﾟするspanﾀｸﾞで、子ｶﾗﾑのIDを設定する属性名 {@value} */
	public static final String EVENT_COLUMN_ID = "h_evCid";

	/** 5.1.7.0 (2010/06/01) ｲﾍﾞﾝﾄｶﾗﾑをﾗｯﾌﾟするspanﾀｸﾞで、子ｶﾗﾑの初期値を設定する属性名 {@value} */
	public static final String EVENT_COLUMN_INITVAL = "h_evVal";

	/** 4.3.7.0 (2009/06/01) linkﾀｸﾞのurlCheckｷｰを出力する際のﾊﾟﾗﾒｰﾀｰｷｰ {@value} */
	public static final String URL_CHECK_KEY = "h_urlCheck" ;

	/** 4.3.7.5 (2009/07/08) ﾍﾟｰｼﾞ内でajaxSubmitを利用する場合のｷｰ {@value} */
	public static final String USE_AJAX_SUBMIT_KEY = "h_ajaxSubmit" ;

	/** 5.1.6.0 (2010/05/01) Viewでｿｰﾄ済みかどうかを判定する際のｷｰ {@value} */
	public static final String VIEW_SORT_KEY = "h_viewSrt";

	// /** 5.1.9.0 (2010/08/01) ﾄﾗﾝｻﾞｸｼｮﾝを実現する際のｷｰ {@value} */
	// 7.4.4.0 (2021/06/30) openGionV8事前準備(未使用の為廃止)
	// public static final String TRANSACTION_KEY = "h_tran";

	/** 5.2.2.0 (2010/11/01) URLの引数部をﾊｯｼｭする際のﾘｸｴｽﾄｷｰ {@value} */
	public static final String URL_HASH_REQ_KEY = "h_r";

	/** 5.3.2.0 (2011/02/01) ﾘｸｴｽﾄﾊﾟﾗﾒｰﾀｰのｷｰ一覧を保存する際のｷｰ {@value} */
	public static final String PARAM_NAMES_KEY = "h_prmNms";

	/** 5.3.6.0 (2011/06/01) ｸｴﾘを呼び出し時のﾌｫｰｶｽを制御するためのｷｰ {@value} */
	public static final String QUERY_FOCUS_KEY = "h_qryFcs";

	/** 5.5.9.1 (2012/12/07) ｾｯｼｮﾝ作成時に、ｾｯｼｮﾝIDを保存しておく、規定のｷｰ {@value} */
	public static final String SESSION_KEY = "h_session_key";

//	/** 5.7.6.2 (2014/05/16) ｾｯｼｮﾝ作成時に、ｾｯｼｮﾝIDを保存しておく、規定のｷｰ {@value} */
//	public static final String IE_HTML5_KEY = "h_html5_key";					// 8.5.2.0 (2023/07/14) Delete

	/** 6.4.5.2 (2016/05/06) UPLOADから呼ばれたかどうかを判定するための、規定のｷｰ {@value} */
	public static final String USE_UPLOAD_KEY = "h_UPLOAD";

//	// 8.0.0.1 (2021/10/08) USE_STR_CONV_KEY 廃止
//	/** 6.8.4.2 (2017/12/25) ｴﾝｺｰﾄﾞ変換対応対応のｷｰ(fileDownloadｻｰﾌﾞﾚｯﾄでｴﾝｺｰﾄﾞをON/OFF指定に利用) {@value} */
//	public static final String USE_STR_CONV_KEY = "useStringConvert";

	// /**
	// * 帳票ｴﾝｺｰﾃﾞｨﾝｸﾞ {@value}
	// * 設定されていない場合は、UTF-8 に設定されます。
	// */
	// 7.4.4.0 (2021/06/30) openGionV8事前準備(未使用の為廃止)
	// public static final String REPORT_ENCODE = "UTF-8" ;

//	/** 7.0.6.4 (2019/11/29) ｺﾝﾃｷｽﾄがﾛｰﾄﾞされた時刻(yyyyMMddHHmmss) */
//	8.3.0.0 (2022/08/01) SYS_MAP に登録する。
//	public static final String START_TIME = DateSet.getDate( "yyyyMMddHHmmss" );

	/** 8.0.0.0 (2021/07/01) 実行しているｻｰﾊﾞｰの名称 */
	public static final String HOST_NAME = org.opengion.fukurou.system.HybsConst.HOST_NAME;
	/** 8.0.0.0 (2021/07/01) 実行しているｻｰﾊﾞｰのIPｱﾄﾞﾚｽ */
	public static final String HOST_ADRS = org.opengion.fukurou.system.HybsConst.HOST_ADRS;

//	static {
//		String dmnHost ;
//		String dmnAdrs ;
//		try {
//			final InetAddress address = InetAddress.getLocalHost();
//			dmnHost = address.getHostName() ;
//			dmnAdrs = address.getHostAddress() ;
//		}
////		catch( final UnknownHostException ex ) {
//		catch( final Throwable th ) {						// 7.0.6.4 (2019/11/29) ｴﾗｰ全てを捉える
//			dmnHost = "Unknown";
//			dmnAdrs = "Unknown";
//		}
//		HOST_NAME = dmnHost;
//		HOST_ADRS = dmnAdrs;
//	}

//	/**
//	 * ﾎｽﾄ名と、IPｱﾄﾞﾚｽを取得
//	 *
//	 * Java VM が実行しているﾎｽﾄ名と、IPｱﾄﾞﾚｽを取得します。
//	 * InetAddress.getLocalHost().getHostName() で求められる値は、Virtual ｱﾄﾞﾚｽなどの
//	 * 複数考えられる為、出来るだけ直接設定されているIPｱﾄﾞﾚｽに近い値を取得するようにします。
//	 * でも、完全には特定できないと思われます。
//	 *
//	 * @og.rev 7.3.1.1 (2021/02/25) ﾎｽﾄ名と、IPｱﾄﾞﾚｽを取得
//	 * @og.rev 8.0.0.0 (2021/07/01) fukurou.system.HybsConst に移植
//	 */
//	static {
//		String dmnHost = "Unknown" ;
//		String dmnAdrs = "Unknown" ;
//		try {
//			boolean isNext = true;
//			final Enumeration<NetworkInterface> enuIfs = NetworkInterface.getNetworkInterfaces();
//			while( isNext && enuIfs.hasMoreElements() ) {
//				final NetworkInterface ni = enuIfs.nextElement();
//
//				final String displayName = ni.getDisplayName();
//				if( displayName.contains("Virtual") ) { continue; }
//
//				final Enumeration<InetAddress> enuIP = ni.getInetAddresses();
//				while( isNext && enuIP.hasMoreElements() ) {
//					final InetAddress adrs = enuIP.nextElement();
//
//					if( adrs.isLinkLocalAddress() || adrs.isLoopbackAddress() ) { continue; }
//
//					dmnHost = adrs.getHostName() ;			// adrs.getCanonicalHostName() はとりあえず使わないでおく。
//					dmnAdrs = adrs.getHostAddress() ;
//					isNext = false;
//					break;
//				}
//			}
//		}
//		catch( final Throwable th ) {
//			System.err.println( "HOST_NAME and HOST_ADRS Unknown!" );
//		}
//		HOST_NAME = dmnHost;
//		HOST_ADRS = dmnAdrs;
//	}

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰをprivateにして、
	 * ｵﾌﾞｼﾞｪｸﾄの生成をさせないようにする。
	 *
	 */
	private HybsSystem() {}

	/**
	 * 内部ｼｽﾃﾑﾘｿｰｽﾏｯﾌﾟｵﾌﾞｼﾞｪｸﾄを設定します。
	 *
	 * @og.rev 3.5.3.1 (2003/10/31) 新規作成
	 * @og.rev 4.0.0.0 (2005/01/31) Ver4 のｼｽﾃﾑﾊﾟﾗﾒｰﾀ情報の取得処理を追加します。
	 * @og.rev 4.3.1.0 (2008/09/03) SERVER_NAME,SERVER_URL,CONTEXT_URLをｷｬｯｼｭから復元
	 * @og.rev 4.3.1.0 (2008/09/03) SERVER_NAME,SERVER_URL,CONTEXT_URLをｷｬｯｼｭから復元
	 * @og.rev 6.2.4.1 (2015/05/22) REAL_PATH 対応。realPath は、setInitialData ではなく、setRealPathﾒｿｯﾄﾞで設定する。
	 * @og.rev 6.4.3.1 (2016/02/12) ConcurrentHashMap で同期処理を行います。
	 * @og.rev 6.4.3.3 (2016/03/04) SYS_MAP を、ConcurrentHashMap で同期するため、not null制限ﾁｪｯｸを追加します。
	 * @og.rev 6.4.9.1 (2016/08/05) reqCache → REQ_CACHE 変更
	 *
	 * @param	param	ｼｽﾃﾑﾘｿｰｽﾏｯﾌﾟ
	 */
	public static void setInitialData( final Map<String,String> param ) {
		// key,val の not null 制限が解除できないので、とりあえず、ConcurrentHashMap で対応。
		SYS_MAP.clear();									// 初期化した上で、全部ｺﾋﾟｰします。
		if( param != null ) {
			// 6.4.3.3 (2016/03/04) SYS_MAP を、ConcurrentHashMap で同期するため、not null制限ﾁｪｯｸを追加します。
			param.forEach( (k,v) -> { if( k != null && v != null ) { SYS_MAP.put( k,v ); } } );
		}
		// 4.3.1.0 (2008/09/03) SERVER_NAME,SERVER_URL,CONTEXT_URLをｷｬｯｼｭから復元
		if( paramSetFlag ) {
			SYS_MAP.putAll( REQ_CACHE );					// 6.4.9.1 (2016/08/05)
		}

		// 8.3.0.0 (2022/08/01) SYS 変数で受け取れるように、変更
		SYS_MAP.put( "START_TIME",DateSet.getDate( "yyyyMMddHHmmss" ) );

		System.out.println( "    [" + SYS_MAP.get( "CONTEXT_NAME" ) + "] のシステムパラメータが、[" + SYS_MAP.size() + "] 件設定されました。" );
	}

	/**
	 * 内部ｼｽﾃﾑﾊﾟﾗﾒｰﾀ に、realPath を設定します。
	 *
	 * その際、Tomcatのﾊﾞｰｼﾞｮﾝ(過去の一部ﾊﾞｰｼﾞｮﾝ)で、最後のﾌｧｲﾙｾﾊﾟﾚｰﾀが
	 * 付かないｹｰｽがありましたので、付ける方向で、統一します。
	 * このﾒｿｯﾄﾞは、SystemParameter#loadInitialParameter( ServletContext , String )からのみ
	 * 呼ばれることを想定しています。
	 *
	 * @og.rev 6.2.4.1 (2015/05/22) REAL_PATH 対応。realPath は、setInitialData ではなく、setRealPathﾒｿｯﾄﾞで設定する。
	 *
	 * @param	path	context.getRealPath( "" ) を引数で渡してきます。
	 */
	/* default */ static void setRealPath( final String path ) {
		if( path != null && path.endsWith( File.separator ) ) {
			realPath = path ;										// 後ろにｾﾊﾟﾚｰﾀが付いていれば、そのまま。
		}
		else {
			realPath = path + File.separator ;						// 付いていなければ、付ける。
		}
	}

	/**
	 * 内部ｼｽﾃﾑﾊﾟﾗﾒｰﾀ に、realPath を取得します。
	 *
	 * その際、Tomcatのﾊﾞｰｼﾞｮﾝ(過去の一部ﾊﾞｰｼﾞｮﾝ)で、最後のﾌｧｲﾙｾﾊﾟﾚｰﾀが
	 * 付かないｹｰｽがありましたので、付ける方向で、統一します。
	 *
	 * @og.rev 6.2.4.1 (2015/05/22) REAL_PATH 対応。realPath は、setInitialData ではなく、setRealPathﾒｿｯﾄﾞで設定する。
	 *
	 * @return	REAL_PATH	ｺﾝﾃｷｽﾄのﾄｯﾌﾟﾊﾟｽ
	 */
	public static String getRealPath() {
		return realPath ;
	}

	/**
	 * ｼｽﾃﾑﾊﾟﾗﾒｰﾀ情報を取得します。
	 * Ver3 では、ﾌﾟﾛﾊﾟﾃｨｰﾌｧｲﾙでﾘｿｰｽを提供している為、
	 * 立ち上げﾀｲﾐﾝｸﾞとの関係で始めて使用する場合にﾃﾞｰﾀを取得します。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規作成
	 * @og.rev 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。
	 *
	 * @param	key	ｼｽﾃﾑﾊﾟﾗﾒｰﾀのｷｰ(not null)
	 * @return	ｼｽﾃﾑﾊﾟﾗﾒｰﾀの値
	 */
	private static String getParam( final String key ) {
			return SYS_MAP.get( key );
	}

	/**
	 * ｼｽﾃﾑﾊﾟﾗﾒｰﾀの追加情報ｾｯﾄが終了しているかどうかを取得します。
	 * 設定処理を行うかどうかの判定に使用します。
	 *
	 * @og.rev 4.0.0.0 (2005/11/30) 新規作成
	 *
	 * @return	true:設定完了 / false:未完了(設定可能)
	 * @see #setRequestParam( String,String,int,String )
	 */
	public static boolean isParamSet() {
		return paramSetFlag;
	}

	/**
	 * ｼｽﾃﾑﾊﾟﾗﾒｰﾀ情報を追加設定します。
	 * これは、SERVER_NAME、SERVER_URL、CONTEXT_URL などのﾘｸｴｽﾄｵﾌﾞｼﾞｪｸﾄから
	 * しか取得できない共有情報を、登録する場合に使用します。
	 * これらの値は、taglib/JspInitTag ﾀｸﾞでｾｯﾄされます。
	 * ｾｯﾄする前に、isParamSet() で設定済みかどうか判断してください。
	 * このﾒｿｯﾄﾞは、isParamSet() の値に関係なく、ｾｯﾄします。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規作成
	 * @og.rev 4.3.1.0 (2008/09/03) SERVER_NAME,SERVER_URL,CONTEXT_URLのｷｬｯｼｭ処理追加
	 * @og.rev 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。
	 * @og.rev 6.4.9.1 (2016/08/05) reqCache → REQ_CACHE 変更
	 *
	 * @param	serverName	ｻｰﾊﾞｰ名
	 * @param	scheme		ｽｷｰﾏ
	 * @param	port		ﾎﾟｰﾄ番号
	 * @param	path		ｺﾝﾃｷｽﾄﾊﾟｽ
	 * @see	#isParamSet()
	 */
	public static void setRequestParam( final String serverName,final String scheme,
										final int port,final String path ) {

			final String portStr = ( port == 80 ) ? "" : ":" + port ;
			final String serverURL = scheme + "://" + serverName + portStr ;
			final String contextURL = serverURL + path + "/" ;

			SYS_MAP.put( "SERVER_NAME"	, serverName );
			SYS_MAP.put( "SERVER_URL"	, serverURL + "/" );
			SYS_MAP.put( "CONTEXT_URL"	, contextURL );

			// 4.3.1.0 (2008/09/03) SERVER_NAME,SERVER_URL,CONTEXT_URLをｷｬｯｼｭして置く
			// 6.4.9.1 (2016/08/05) reqCache → REQ_CACHE 変更
			REQ_CACHE.put( "SERVER_NAME"	, serverName );
			REQ_CACHE.put( "SERVER_URL"		, serverURL + "/" );
			REQ_CACHE.put( "CONTEXT_URL"	, contextURL );
			paramSetFlag = true;
	}

	/**
	 * 現在日付、時刻を所得します。
	 * 出力ﾌｫｰﾏｯﾄは、ｼｽﾃﾑﾊﾟﾗﾒｰﾀ の SYS_TIME の
	 * 設定を適用します。( "yyyy/MM/dd HH:mm:ss" )
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) getSystemResource() ⇒ sysParam に変更
	 * @og.rev 6.4.2.0 (2016/01/29) DateSet.getDate( String ) を利用するように修正します。
	 *
	 * @return	現在日付、時刻 ( 例 2001/04/17 15:48:22 )
	 */
	public static String getDate() {
		return DateSet.getDate( getParam( "SYS_TIME" ) );
	}

	/**
	 * 指定時刻をﾃﾞﾌｫﾙﾄのﾌｫｰﾏｯﾄで、所得します。
	 * ﾌｫｰﾏｯﾄの指定方法は、java.text.SimpleDateFormat の指定方法と同一です。
	 * 出力ﾌｫｰﾏｯﾄは、ｼｽﾃﾑﾊﾟﾗﾒｰﾀ の SYS_TIME の
	 * 設定を適用します。( "yyyy/MM/dd HH:mm:ss" )
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) getSystemResource() ⇒ sysParam に変更
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 *
	 * @param	time	指定のｶﾚﾝﾄﾀｲﾑ
	 *
	 * @return	現在日付、時刻( 例 2001/04/17 15:48:22 )
	 */
	public static String getDate( final long time ) {
		final String form = getParam( "SYS_TIME" );
		return DateSet.getDate( time , form );
	}

	/**
	 * ｼｽﾃﾑ設定ﾌｧｲﾙのﾃﾞｰﾀを所得します。
	 * ｷｰが存在しない場合は、null を返します。
	 * ｷｰがnullの場合は、HybsSystemException をｽﾛｰします。
	 *
	 * @og.rev 6.7.4.1 (2017/02/17) ｷｰが存在しなかった場合のﾛｸﾞ出力を出すかどうか(true:出す)
	 *
	 * @param	key	ｼｽﾃﾑ設定 ｷｰ
	 * @return	ｼｽﾃﾑ設定値
	 * @og.rtnNotNull
	 * @throws	HybsSystemException ｷｰが null の場合。
	 */
	public static String sys( final String key ) throws HybsSystemException {
		return sys( key , true );
	}

	/**
	 * ｼｽﾃﾑ設定ﾌｧｲﾙのﾃﾞｰﾀを所得します。
	 * ｷｰが存在しない場合は、null を返します。
	 * ｷｰがnullの場合は、HybsSystemException をｽﾛｰします。
	 *
	 * @og.rev 3.5.5.0 (2004/03/12) {&#064;SYS.XXXX} で使用できる変数を増やします。
	 * @og.rev 3.5.5.3 (2004/04/09) {&#064;SYS.XXXX} に、DATE というｷｰを追加します。
	 * @og.rev 4.0.0.0 (2005/01/31) getSystemResource() ⇒ sysParam に変更
	 * @og.rev 6.7.4.1 (2017/02/17) ｷｰが存在しなかった場合のﾛｸﾞ出力を出すかどうか(true:出す)
	 *
	 * @param	key			ｼｽﾃﾑ設定 ｷｰ
	 * @param	useNoKeyLog	ｼｽﾃﾑﾘｿｰｽｷｰが存在しない場合に、ﾛｸﾞ出力するかどうか [true:出力する/false:出力しない]
	 * @return	ｼｽﾃﾑ設定値
	 * @og.rtnNotNull
	 * @throws	HybsSystemException ｷｰが null の場合。
	 */
	public static String sys( final String key , final boolean useNoKeyLog ) throws HybsSystemException {
		if( key == null ) {
			final String errMsg = "キーが null です。";
			throw new HybsSystemException( errMsg );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}

		if( "DATE".equals( key ) ) { return getDate() ; }	// 3.5.5.3 (2004/04/09)

		final String rtn = getParam( key );
		if( rtn == null && ! key.startsWith( "h_" ) && useNoKeyLog ) {
			final String errMsg = CR + "システムリソースキーが存在しません。key = " + key;
			LogWriter.log( errMsg );
		}
		return rtn;
	}

	/**
	 * ｼｽﾃﾑ設定ﾌｧｲﾙから、複数のﾃﾞｰﾀを所得します。
	 * ｷｰが、"ABC" とすれば、それで始まるﾃﾞｰﾀ( ABCD, ABC_01 など)
	 * を配列で取得できます。
	 * ﾘﾀｰﾝ配列は、ｿｰﾄされています。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) getSystemResource() ⇒ sysParam に変更
	 * @og.rev 6.3.9.0 (2015/11/06) Map.keySet() ではなく、Map.entrySet() を使う様に変更。
	 * @og.rev 6.4.3.3 (2016/03/04) HybsSystem.newInstance(String,String) への置き換え。
	 *
	 * @param	key	ｼｽﾃﾑ設定 ｷｰ(の先頭の部分文字列)
	 * @return	ｼｽﾃﾑ設定配列 HybsEntry
	 */
	public static HybsEntry[] sysEntry( final String key ) {
		final Set<HybsEntry> entSet = new TreeSet<>();

		// 6.4.3.3 (2016/03/04) SYS_MAP を、ConcurrentHashMap で同期するため、not null制限ﾁｪｯｸを追加します。
		SYS_MAP.forEach( (k,v) -> { if( k.startsWith( key ) ) { entSet.add( new HybsEntry( k,v ) ); } } );

		return entSet.toArray( new HybsEntry[entSet.size()] );
	}

	/**
	 * ｼｽﾃﾑ設定ﾌｧｲﾙのﾃﾞｰﾀを所得します。
	 * 特別に、int 型にﾃﾞｰﾀを変換して渡します。
	 * 変換できない場合は、NumberFormatException がｽﾛｰされます。
	 *
	 * @og.rev 3.5.3.1 (2003/10/31) ｴﾗｰ時に、判るようにﾒｯｾｰｼﾞを出します。
	 * @og.rev 4.0.0.0 (2005/01/31) getSystemResource() ⇒ sysParam に変更
	 *
	 * @param	key	ｼｽﾃﾑ設定 ｷｰ
	 * @return	ｼｽﾃﾑ設定値
	 */
	public static int sysInt( final String key ) {
		if( key == null ) {
			final String errMsg = "キーが null です。";
			throw new HybsSystemException( errMsg );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}

		final String tmp = getParam( key );
		int rtn ;
		try {
			rtn = Integer.parseInt( tmp );
		}
		catch( final NumberFormatException ex2 ) {
			final String errMsg = "システムリソース の値が数字ではありません。" + CR
					+ "  Resource key=[" + key + "] val=[" + tmp + "]"  ;
			throw new HybsSystemException( errMsg,ex2 );	// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}

		return rtn;
	}

	/**
	 * ｼｽﾃﾑ設定ﾌｧｲﾙのﾃﾞｰﾀを所得します。
	 * 特別に、boolean 型にﾃﾞｰﾀを変換して渡します。
	 * 変換できない場合は、NumberFormatException がｽﾛｰされます。
	 *
	 * @og.rev 3.5.6.0 (2004/06/18) Booleanのｺﾝｽﾄﾗｸﾀの代わりに、Boolean.valueOf(...) を使用します。
	 *
	 * @param	key	ｼｽﾃﾑ設定 ｷｰ
	 * @return	ｼｽﾃﾑ設定値
	 */
	public static boolean sysBool( final String key ) {
		return Boolean.parseBoolean( HybsSystem.sys( key ) );					// 6.1.0.0 (2014/12/26) refactoring
	}

	/**
	 * ｼｽﾃﾑ依存のURL文字列から、実際のﾃﾞｨﾚｸﾄﾘ名を所得します。
	 * UNIX と Windows を URL ﾊﾟｽ名から判断して、実ﾃﾞｨﾚｸﾄﾘを返します。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' (UNIX) または、2文字目が、
	 * ":" (Windows)の場合は、指定のURLそのままのﾃﾞｨﾚｸﾄﾘに、そうでない場合は、
	 * ｼｽﾃﾑﾊﾟﾗﾒｰﾀ の FILE_URL 属性で指定のﾌｫﾙﾀﾞの下に、作成されます。
	 * 相対ﾊﾟｽ、または、URL の場合は、頭に / を入れないでください。UNIX の
	 * 絶対ﾊﾟｽと判断されます。
	 * また、絶対ﾊﾟｽの最後には、/ を入れておいてください。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) StringUtil.urlAppend ﾒｿｯﾄﾞの利用
	 * @og.rev 6.4.2.1 (2016/02/05) 引数をすべて、StringUtil.urlAppend していきます。
	 * @og.rev 5.10.9.0 (2019/03/01) ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ対応を追加。
	 * @og.rev 7.0.2.2 (2019/03/25) StringBuilder を使わなくします。
	 *
	 * @param	urls	URL文字列(可変長配列)
	 * @return	ﾃﾞｨﾚｸﾄﾘ文字列
	 */
	public static String url2dir( final String... urls ) {
//		return StringUtil.urlAppend( realPath,urls ).replace( '/',FS );
//		// 5.10.9.0 (2019/03/01) ADD
//		final StringBuilder sb = new StringBuilder( BUFFER_MIDDLE );

		// \を/に変換する処理を追加。
		final String path = StringUtil.urlAppend( realPath,urls ).replace('\\','/').replace( '/',FS );

		// linux環境かつ、2文字目が「:」Windows環境のﾄﾞﾗｲﾌﾞ指定の場合は、頭に「/」を付与して、絶対ﾊﾟｽとして扱う。
		return FS == '/' && path.charAt(1) == ':' ? "/" + path : path ;

		// linux環境かつ、2文字目が「:」Windows環境のﾄﾞﾗｲﾌﾞ指定の場合は、頭に「/」を付与して、絶対ﾊﾟｽとして扱う。
//		if(FS == '/' && path.charAt(1) == ':') {
//			sb.append("/").append(path);
//		}else {
//			sb.append(path);
//		}
//
//		return sb.toString();
	}

	/**
	 * 絶対ﾊﾟｽと判断されるものは変換せず、それ以外を絶対ﾊﾟｽに変換します。
	 *
	 * @og.rev 5.10.9.2 (2019/03/15) ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ対応を追加。
	 * @og.rev 7.0.2.2 (2019/03/25) StringBuilder を使わなくします。
	 * @og.rev 5.10.21.1 (2020/04/21) \\から開始するﾃﾞｨﾚｸﾄﾘについても無変換の対象とします。
	 * @og.rev 7.2.6.2 (2020/07/29) 引数がｾﾞﾛ文字列の場合の処理追加。
	 *
	 * @param	url	URL文字列
	 * @return	ﾃﾞｨﾚｸﾄﾘ文字列
	 */
	public static String url2absPath( final String url ) {
		if( url == null || url.isEmpty() ) { return ""; }						// 7.2.6.2 (2020/07/29)

		final String absPath ;

		// 2文字目が「:」の場合はそのままWindowsの絶対ﾊﾟｽとして扱う。
//		if( url.charAt(1) == ':' ) {
		if( url.charAt(1) == ':' || url.startsWith( "\\\\" )) {					// 5.10.21.1
			absPath = url;
		}
		else {
			// それ以外は絶対ﾊﾟｽに変換する
			final String path = StringUtil.urlAppend( realPath,url ).replace('\\','/').replace( '/',FS );
//			absPath = new File( path ).getAbsolutePath().toString();			// spotbugs:String.toString() を呼び出すのは冗長です。
			absPath = new File( path ).getAbsolutePath();
		}

		return absPath;

//		// 5.10.9.0 (2019/03/01) ADD
//		final StringBuilder sb = new StringBuilder(BUFFER_MIDDLE);
//
//		// 2文字目が「:」の場合はそのままWindowsの絶対ﾊﾟｽとして扱う。
//		if( url.charAt(1) == ':' ) {
//			sb.append(url);
//		}
//		else {
//			//　それ以外は絶対ﾊﾟｽに変換する
//			final String path = StringUtil.urlAppend( realPath,url ).replace('\\','/').replace( '/',FS );
//			sb.append( new File( path ).getAbsolutePath().toString() );
//		}
//
//		return sb.toString();
	}

	/**
	 * ｼｽﾃﾑﾘｿｰｽ情報を取得します。
	 * 形式は、Map です。
	 * 実質的には、ｷｰでｿｰﾃｨﾝｸﾞされた、ConcurrentSkipListMap を返します。
	 *
	 * @og.rev 6.4.3.3 (2016/03/04) 配列返しではなく、Map を返します。
	 *
	 * @return	ｼｽﾃﾑﾘｿｰｽ情報:ｷｰ([0])、値([1])の2次元配列
	 */
	public static ConcurrentMap<String,String> getSystemResourceMap() {
		return new ConcurrentSkipListMap<String,String>( SYS_MAP );
	}

	/**
	 * ｼｽﾃﾑのｺﾝﾃｷｽﾄ名を取得します。
	 * ｺﾝﾃｷｽﾄ名は、Tomcat の servr.xml で指定する仮想ﾌｫﾙﾀﾞ名です。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @return	ｺﾝﾃｷｽﾄ名(仮想ﾌｫﾙﾀﾞ名)
	 */
	public static String getContextName() {
		return sys( "CONTEXT_NAME" );	// 4.0.0 (2005/01/31)
	}

	/**
	 * ｻｰﾊﾞｰのﾌﾟﾘﾝﾀｰ一覧のｽﾍﾟｰｽ区切り文字列を返します。
	 * ﾌﾟﾘﾝﾀ名にｽﾍﾟｰｽが含まれる場合は、前後にﾀﾞﾌﾞﾙｸｵｰﾄを付けます。
	 * ここで求めた値は、"SYS.PRINTER" で、取得可能です。(最初の取得時に内部ｷｬｯｼｭされます。)
	 * SystemData#OMIT_PRINTER_NAMES に登録されている一覧名と先頭一致するﾌﾟﾘﾝﾀは除外します。
	 * 例えば、Windows等で、初めから登録されている Fax,Microsoft XPS などは、不要なのでOMITします。
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) ｻｰﾊﾞｰのﾌﾟﾘﾝﾀｰ一覧のｽﾍﾟｰｽ区切り文字列対応。
	 * @og.rev 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。
	 * @og.rev 7.2.4.0 (2020/05/11) bufに何もｾｯﾄされなかった場合、ﾏｲﾅｽｴﾗｰになった。
	 *
	 * @return	ｻｰﾊﾞｰのﾌﾟﾘﾝﾀｰ一覧のｽﾍﾟｰｽ区切り文字列
	 */
	public static String getPrinter() {
		String rtn = getParam( "PRINTER" );
		if( rtn == null ) {
			final String[] omitNames = HybsSystem.sys( "OMIT_PRINTER_NAMES" ).split( "," ) ;

			final PrintService[] service = PrintServiceLookup.lookupPrintServices( null,null );

			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE ) ;
			for( int i=0; i<service.length; i++ ) {
				final String name = service[i].getName() ;
				boolean isAdd = true;						// true:追加する/false:追加しない
				for( int j=0; j<omitNames.length; j++ ) {
					if( name.startsWith( omitNames[j] ) ) { isAdd = false; break; }
				}
				if( isAdd ) {
					if( name.contains( " " ) ) {
						buf.append( '"' ).append( name ).append( "\" " );
					}
					else {
						buf.append( name ).append( ' ' );
					}
				}
			}
			if( buf.length() > 0 ) {						// 7.2.4.0 (2020/05/11)
				buf.setLength( buf.length()-1 );			// 最後のｽﾍﾟｰｽを削除
			}

			rtn = buf.toString();
			SYS_MAP.put( "PRINTER",rtn );
		}
		return rtn ;
	}

	/**
	 * HybsSystem.sys から識別idとなるｸﾗｽ名を取得して、ｵﾌﾞｼﾞｪｸﾄを作成します。
	 * 簡易的なﾒｿｯﾄﾞです。
	 *
	 * @og.rev 6.4.3.3 (2016/03/04) 新規作成。
	 *
	 * @param	<T>		ｷｬｽﾄされて生成される新しいｵﾌﾞｼﾞｪｸﾄ
	 * @param	sysKey	ｸﾗｽのｷｰとなるID(共通ｷｰ)
	 * @param	cls		作成するｸﾗｽの個別ｷｰ
	 *
	 * @return	Tｵﾌﾞｼﾞｪｸﾄ
	 * @throws	RuntimeException 何らかのｴﾗｰが発生した場合
	 */
	@SuppressWarnings(value={"unchecked"})
	public static <T> T newInstance( final String sysKey,final String cls ) {
		final String clsStr = HybsSystem.sys( sysKey + cls );

		// NullPointerException が発生するので、事前にﾁｪｯｸします。
		if( clsStr == null ) {
			final String errMsg = sysKey + " クラスが見つかりません。class=" + sysKey + cls;
			throw new OgRuntimeException( errMsg );
		}

		return (T) newInstance( clsStr );
	}

	/**
	 * 標準のｸﾗｽﾛｰﾀﾞを使って、識別id に応じた ｵﾌﾞｼﾞｪｸﾄを作成します。
	 * 作成するには、ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰが必要です。
	 *
	 * @og.rev 6.4.3.3 (2016/03/04) ﾘﾌﾚｸｼｮﾝ系の見直しと、例外の共通ｸﾗｽに置き換えます。
	 * @og.rev 6.8.2.3 (2017/11/10) java9対応(cls.newInstance() → cls.getDeclaredConstructor().newInstance())
	 *
	 * @param	<T>	ｷｬｽﾄされて生成される新しいｵﾌﾞｼﾞｪｸﾄ
	 * @param	cls	作成するｸﾗｽのﾌﾙﾈｰﾑ(識別id)
	 *
	 * @return	Tｵﾌﾞｼﾞｪｸﾄ
	 * @throws	RuntimeException 何らかのｴﾗｰが発生した場合
	 */
	@SuppressWarnings(value={"unchecked"})
	public static <T> T newInstance( final String cls ) {
		try {
			return (T) Class.forName( cls ).getDeclaredConstructor().newInstance();	// 6.8.2.3 (2017/11/10)
		}
		catch( final ReflectiveOperationException ex ) {
			final String errMsg = "Class.forName( String ).newInstance() 処理に失敗しました class=[" + cls + "]" + CR
						+ ex.getMessage() ;
			throw new OgRuntimeException( errMsg,ex );
		}
	}

	/**
	 * "org.opengion.hayabusa.message" の Locale.JAPANESE ﾘｿｰｽから取得するﾒｯｾｰｼﾞを文字列で返します。
	 *
	 * id と引数を受け取り、ResourceBundle と、MessageFormat.format で加工した
	 * 文字列を返します。
	 * baseName は、F_BS_NM で、Locale に、Locale.JAPANESE を指定したﾒｯｾｰｼﾞを作成します。
	 * hayabusaﾊﾟｯｹｰｼﾞと、pluginﾊﾟｯｹｰｼﾞで主に使われる、ﾒｯｾｰｼﾞの取得方法です。
	 *
	 * @og.rev 6.4.3.2 (2016/02/19) 新規追加
	 *
	 * @param	id		ﾘｿｰｽのｷｰとなるID。
	 * @param	args	ﾘｿｰｽを、MessageFormat.format で加工する場合の引数。
	 * @return	MessageFormat.formatで加工された文字列
	 * @see		MsgUtil#H_BS_NM
	 */
	public static String getMsg( final String id , final Object... args ) {
		return MsgUtil.getMsg( MsgUtil.H_BS_NM , Locale.JAPANESE , id , args );
	}

	/**
	 * 入力文字列に、{&#064;SYS},{&#064;ENV} の文字列変換を行います。
	 *
	 *   {&#064;SYS.XXX}  ｼｽﾃﾑﾘｿｰｽMapを元に、文字列を変換します。
	 *   {&#064;ENV.XXX}  環境変数(Javaｵﾌﾟｼｮﾝ)の文字列を変換します。(java -Dkey=value ｵﾌﾟｼｮﾝ)
	 *
	 * @og.rev 6.9.0.1 (2018/02/05) 新規追加
	 *
	 * @param	inMsg	入力文字列(nullの場合、そのまま返します)
	 * @return  変換後文字列
	 */
	public static String changeParam( final String inMsg ) {
		if( inMsg == null ) { return inMsg; }

		String message = inMsg;

		// {@SYS.XXXX} 変数の置換処理
		int adrs = message.indexOf( "{@SYS." ) ;
		while( adrs >= 0 ) {
			final int end = message.indexOf( '}',adrs ) ;
			final String key = message.substring( adrs+6,end );			// 6 = "{@SYS.".length のこと。
			final String oldData = "{@SYS." + key + "}" ;
			final String newData = SYS_MAP.getOrDefault( key , "" );	// SYS_MAP は、ConcurrentMap なので、null は、返らない。
			message = StringUtil.replace( message,oldData,newData );
			adrs = message.indexOf( "{@SYS.",adrs ) ;
		}
		// {@ENV.XXXX} 変数の置換処理
		adrs = message.indexOf( "{@ENV." ) ;
		while( adrs >= 0 ) {
			final int end = message.indexOf( '}',adrs ) ;
			final String key = message.substring( adrs+6,end );			// 6 = "{@ENV.".length のこと。
			final String oldData = "{@ENV." + key + "}" ;
			final String newData = System.getProperty( key,"" );
			message = StringUtil.replace( message,oldData,newData );
			adrs = message.indexOf( "{@ENV.",adrs ) ;
		}

		return message;
	}
}
