/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval;

import java.io.File;
import java.util.Locale;
import java.util.concurrent.ConcurrentMap;						// 6.4.3.3 (2016/03/04)
import java.util.Arrays;
import java.util.Map;											// 6.9.9.0 (2018/08/20)

import org.opengion.fukurou.business.ArrayTableModel;
import org.opengion.fukurou.model.DataModel;					// 6.7.9.1 (2017/05/19) ArrayTableModel を、DataModel I/F に変更
import org.opengion.fukurou.business.BizLogicHelper;
import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.util.ErrMsg;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.HybsLoader;
import org.opengion.fukurou.util.HybsLoaderConfig;
import org.opengion.fukurou.util.HybsLoaderFactory;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;

/**
 * 業務ﾛｼﾞｯｸを呼び出すためのﾀｸﾞです。
 *
 * logics属性に呼び出す業務ﾛｼﾞｯｸのｸﾗｽ名を記述します。
 * このﾀｸﾞでは、複数の業務ﾛｼﾞｯｸを1度に呼び出すことができ、
 * DB接続のcommit,rollbackは一番最後に、1度のみ実行されます。
 * 各業務ﾛｼﾞｯｸは、記述した順番に呼び出されます。
 *
 * 業務ﾛｼﾞｯｸは、{@link org.opengion.fukurou.business.AbstractBizLogic AbstractBizLogic}を
 * 継承したｻﾌﾞｸﾗｽである必要があります。
 * 標準ｻﾌﾞｸﾗｽとして、BizLogic_ENTRYとBizLogic_TABLE を用意しています。
 * BizLogic_ENTRY は、ﾊﾟﾗﾒｰﾀｰのみを使用する業務ﾛｼﾞｯｸです。
 * BizLogic_TABLE は、配列型ﾃｰﾌﾞﾙﾓﾃﾞﾙをﾒｲﾝｶｰｿﾙとした業務ﾛｼﾞｯｸです。
 *   配列型ﾃｰﾌﾞﾙﾓﾃﾞﾙでは、初期処理、ﾙｰﾌﾟ処理、後処理とﾒｿｯﾄﾞがｺｰﾙされます。
 *
 *  fstchk()          変更区分に関わらず   処理を始める前に呼び出し
 *  first()           変更区分に関わらず   最初の行でのみ呼び出し
 *    useLoop == true
 *      befall( int row ) 変更区分に関わらず  各行について呼び出し(insert,modify,deleteの前に呼び出し)
 *      insert( int row ) 変更区分が"A"の場合 各行について呼び出し
 *      modify( int row ) 変更区分が"C"の場合 各行について呼び出し
 *      delete( int row ) 変更区分が"D"の場合 各行について呼び出し
 *      allrow( int row ) 変更区分に関わらず  各行について呼び出し(insert,modify,deleteの後に呼び出し)
 *  last()            変更区分に関わらず   最後の行でのみ呼び出し
 *
 * 業務ﾛｼﾞｯｸｸﾗｽについては、ﾎｯﾄﾃﾞﾌﾟﾛｲ機能により、動的ｺﾝﾊﾟｲﾙ、ｸﾗｽﾛｰﾄﾞが
 * 行われます。
 *
 * 業務ﾛｼﾞｯｸのｿｰｽﾃﾞｨﾚｸﾄﾘは、ｼｽﾃﾑﾘｿｰｽの BIZLOGIC_SRC_PATH で定義されます。
 * また、同様にｸﾗｽﾃﾞｨﾚｸﾄﾘは、ｼｽﾃﾑﾘｿｰｽの BIZLOGIC_CLASS_PATH で定義されます。
 * さらに、ｼｽﾃﾑﾘｿｰｽの BIZLOGIC_HOTDEPLOY を false に設定することで、動的ｺﾝﾊﾟｲﾙ
 * 、ｸﾗｽﾛｰﾄﾞを行わなくすることもできます。
 * この場合、予めｺﾝﾊﾟｲﾙされたｸﾗｽを、初回呼び出し時に1回のみﾛｰﾄﾞされます。
 *
 * SystemData の USE_SQL_INJECTION_CHECK が true か、quotCheck 属性が true の場合は、
 * ＳＱＬｲﾝｼﾞｪｸｼｮﾝ対策用のｼﾝｸﾞﾙｸｫｰﾄﾁｪｯｸを行います。ﾘｸｴｽﾄ引数に
 * ｼﾝｸﾞﾙｸｫｰﾄ(')が含まれると、ｴﾗｰになります。
 * 同様にUSE_XSS_CHECKがtrueか、xssCheck属性がtrueの場合は、
 * ｸﾛｽｻｲﾄｽｽｸﾘﾌﾟﾃｨﾝｸﾞ(XSS)対策のためless/greater than signのﾁｪｯｸを行います。
 *
 * ※ このﾀｸﾞは、Transaction ﾀｸﾞの対象です。
 *
 * @og.formSample
 * ●形式：
 *       ･&lt;og:bizLog
 *             logics       = "業務ﾛｼﾞｯｸのｸﾗｽ名"
 *             command      = "ENTRY"
 *             scope        = "session"
 *             dbid         = "DEFAULT"
 *             tableId      = "DEFAULT"
 *             selectedAll  = "false"
 *             modifyType   = "A"
 *             keys         = "SYSTEM_ID"
 *             vals         = "{&#064;SYSTEM_ID}"
 *        /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:bizLogic
 *       logics           ○【TAG】実行する業務ﾛｼﾞｯｸ名を指定します (必須)
 *       command            【TAG】ｺﾏﾝﾄﾞをｾｯﾄします (初期値:ENTRY)
 *       scope              【TAG】ｷｬｯｼｭする場合のｽｺｰﾌﾟ[request/page/session/application]を指定します (初期値:session)
 *       dbid               【TAG】(通常は使いません)Queryｵﾌﾞｼﾞｪｸﾄを作成する時のDB接続IDを指定します (初期値:null)
 *       tableId            【TAG】(通常は使いません)結果のDBTableModelを、sessionに登録するときのｷｰを指定します
 *       selectedAll        【TAG】ﾃﾞｰﾀを全件選択済みとして処理するかどうか[true/false]を指定します (初期値:false)
 *       modifyType         【TAG】DB検索時の ﾓﾃﾞｨﾌｧｲﾀｲﾌﾟを指定します[A:追加/C:更新/D:削除]
 *       keys               【TAG】ﾘﾝｸ先に渡すｷｰをCSV形式で複数指定します
 *       vals               【TAG】ﾘﾝｸ先に渡す値をCSV形式で複数指定します
 *       stopError          【TAG】処理ｴﾗｰの時に処理を中止するかどうか[true/false]を設定します (初期値:true)
 *       dispError          【TAG】ｴﾗｰ時にﾒｯｾｰｼﾞを表示するか[true/false]を設定します。通常はstopErrorと併用 (初期値:true)
 *       quotCheck          【TAG】ﾘｸｴｽﾄ情報の ｼﾝｸﾞﾙｸｫｰﾄ(') 存在ﾁｪｯｸを実施するかどうか[true/false]を設定します (初期値:USE_SQL_INJECTION_CHECK[=true])
 *       xssCheck           【TAG】ﾘｸｴｽﾄ情報の HTMLTag開始/終了文字(&gt;&lt;) 存在ﾁｪｯｸを実施するかどうか[true/false]を設定します (初期値:USE_XSS_CHECK[=true])
 *       multi              【TAG】vals属性でﾊﾟﾗﾒｰﾀｰを取得する際、複数件存在する場合に、値を連結するかどうかを指定します (初期値:false)
 *       debug              【TAG】ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうか[true/false]を指定します (初期値:false)
 *   /&gt;
 *
 * ●使用例
 *     &lt;!-- 業務ﾛｼﾞｯｸの呼び出しを行います --&gt;
 *     &lt;og:bizLogic logics="org.opengion.logic.gf9110.BizLogic_0001" keys="SYSTEM_ID" vals="{&#064;MEM.SYSTEM_ID}" /&gt;
 *
 * @og.rev 5.1.1.0 (2009/12/01) 新規作成
 * @og.group 業務ﾛｼﾞｯｸ
 *
 * @version	5.0
 * @author	Hiroki Nakamura
 * @since	JDK1.6,
 */
public class BizLogicTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "8.3.1.0 (2022/10/14)" ;
	private static final long serialVersionUID = 831020221014L ;

	/** command 引数に渡す事の出来る ｺﾏﾝﾄﾞ ｴﾝﾄﾘｰ {@value} */
	public static final String CMD_ENTRY		= "ENTRY";						// 5.1.9.0 (2010/08/01)
	private static final String ERR_MSG_ID		= HybsSystem.ERR_MSG_KEY;

	private				String		command		= CMD_ENTRY;
	private				String[]	logics		;
	private				String		dbid		;
	private transient	DBTableModel table		;
	private				String		tableId		= HybsSystem.TBL_MDL_KEY;
	private				boolean		selectedAll	;
	private				String		modifyType	;
	private				String[]	keys		;
	private				String[]	vals		;

	private				boolean		stopError	= true;
	private				boolean		quotCheck	= HybsSystem.sysBool( "USE_SQL_INJECTION_CHECK" );	// 4.0.0 (2005/08/31)
	private				boolean		xssCheck	= HybsSystem.sysBool( "USE_XSS_CHECK" );			// 5.0.0.2 (2009/09/15)

	private transient	ErrorMessage errMessage ;
	private				int		errCode			= ErrorMessage.OK;
	private				int		executeCount	= -1;
	private	transient	DataModel<String>	arrTable;							// 6.7.9.1 (2017/05/19) ArrayTableModel を、DataModel I/F に変更
	private	transient	HybsLoader			loader	;							// 6.3.9.0 (2015/11/06) transient 追加

	// 8.0.2.0 (2021/11/30) static定義するので、Tomcat再起動が必要
	private static final String SRC_PATH		= HybsSystem.sys( "REAL_PATH" ) + HybsSystem.sys( "BIZLOGIC_SRC_PATH" );
	private static final String CLS_PATH		= HybsSystem.sys( "REAL_PATH" ) + HybsSystem.sys( "BIZLOGIC_CLASS_PATH" );
	private static final String WEB_LIB			= HybsSystem.sys( "REAL_PATH" ) + "WEB-INF" + File.separator + "lib";
	private static final String WEB_CLS			= HybsSystem.sys( "REAL_PATH" ) + "WEB-INF" + File.separator + "classes";

//	private				String	srcDir			= HybsSystem.sys( "REAL_PATH" ) + HybsSystem.sys( "BIZLOGIC_SRC_PATH" );
	private				String	srcDir			= SRC_PATH;						// 8.0.2.0 (2021/11/30)
//	private				String	classDir		= HybsSystem.sys( "REAL_PATH" ) + HybsSystem.sys( "BIZLOGIC_CLASS_PATH" );
	private				String	classDir		= CLS_PATH;						// 8.0.2.0 (2021/11/30)
	private				boolean	isAutoCompile	= HybsSystem.sysBool( "BIZLOGIC_AUTO_COMPILE" );
	private				boolean	isHotDeploy		= HybsSystem.sysBool( "BIZLOGIC_HOT_DEPLOY" );
	private				boolean	isMulti			;								// 5.1.8.0 (2010/07/01) 追加

	// 5.9.26.1 (2017/11/10) 実行ｴﾗｰの際に、ｴﾗｰを画面に出力するかどうか。
	private				boolean dispError		= true;

	private static final String	CLASS_PATH;

	// HotDeploy機能を使用する場合に、Javaｸﾗｽをｺﾝﾊﾟｲﾙするためのｸﾗｽﾊﾟｽを設定します。
	// 対象となるクラスパスは、WEB-INF/classes 及び WEB-INF/lib/*.jar です。

	static {
		final StringBuilder sb = new StringBuilder( BUFFER_MIDDLE )
			.append( '.' ).append( File.pathSeparatorChar );
//		final File lib = new File( HybsSystem.sys( "REAL_PATH" ) + "WEB-INF" + File.separator + "lib" );
		final File lib = new File( WEB_LIB );									// 8.0.2.0 (2021/11/30)
		final File[] libFiles = lib.listFiles();
		// 6.3.9.0 (2015/11/06) null になっている可能性がある(findbugs)
		if( libFiles != null ) {
			for( final File file : libFiles ) {
				// 5.1.1.2 (2009/12/10) File.pathSeparatorCharを使用
				// 5.1.8.0 (2010/07/01) libの検索パスの不具合対応
				sb.append( file.getAbsolutePath() ).append( File.pathSeparatorChar );
			}
		}
//		sb.append( HybsSystem.sys( "REAL_PATH" ) + "WEB-INF" + File.separator + "classes" )
		sb.append( WEB_CLS )													// 8.0.2.0 (2021/11/30)
			.append( File.pathSeparatorChar )
			// 5.1.8.0 (2010/07/01) bizの下も検索パスに追加
//			.append( HybsSystem.sys( "REAL_PATH" ) + HybsSystem.sys( "BIZLOGIC_CLASS_PATH" ) ).append( File.pathSeparatorChar );
			.append( CLS_PATH ).append( File.pathSeparatorChar );				// 8.0.2.0 (2021/11/30)

		CLASS_PATH = sb.toString();
	}

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰ
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public BizLogicTag() { super(); }		// これも、自動的に呼ばれるが、空のﾒｿｯﾄﾞを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始ﾀｸﾞが見つかったときに処理する doStartTag() を ｵｰﾊﾞｰﾗｲﾄﾞします。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) command=ENTRY以外ではDBTableModelの処理を行わない。
	 * @og.rev 6.4.8.1 (2016/07/02) xssCheckを、doStartTag に移動
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doStartTag() {
		useXssCheck( xssCheck );												// 6.4.8.1 (2016/07/02)

		// 5.3.4.0 (2011/04/01)
		if( CMD_ENTRY.equals( command ) ) {
			startQueryTransaction( tableId );
		}

		return SKIP_BODY ;
	}

	/**
	 * Taglibの終了ﾀｸﾞが見つかったときに処理する doEndTag() を ｵｰﾊﾞｰﾗｲﾄﾞします。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) isMulti対応
	 * @og.rev 5.3.4.0 (2011/04/01) command=ENTRY以外ではDBTableModelの処理を行わない。
	 * @og.rev 6.4.8.1 (2016/07/02) xssCheckを、doStartTag に移動
	 * @og.rev 5.9.26.1 (2017/11/10) dispErrorの処理追加
	 * @og.rev 8.3.1.0 (2022/10/14) 正常終了時、過去のｴﾗｰﾒｯｾｰｼﾞが消えない不具合対応
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();

		// 5.3.4.0 (2011/04/01)
		useQuotCheck( quotCheck );

		makeVals();
		execute();

		final String err = TaglibUtil.makeHTMLErrorTable( errMessage,getResource() );
		if( err != null && err.length() > 0 ) {
			setSessionAttribute( ERR_MSG_ID,errMessage );
		}
		// 8.3.1.0 (2022/10/14) Add
		else {
			removeSessionAttribute( ERR_MSG_ID );
		}

		if( table != null && ! commitTableObject( tableId, table ) ) {
			jspPrint( "BizLoicTag 処理が割り込まれました。DBTableModel は登録しません。" );
			return SKIP_PAGE;
		}

		// 5.9.26.1 (2017/11/10) ｴﾗｰﾒｯｾｰｼﾞをﾘｸｴｽﾄ変数で持つようにしておく
		setRequestAttribute( "DB.ERR_MSG", err );
		// 5.9.26.1 (2017/11/10) dispErrorで表示をｺﾝﾄﾛｰﾙ
		if( dispError ) {
			jspPrint( err );
		}

		if( errCode >= ErrorMessage.NG && stopError ) {
			return SKIP_PAGE;
		}
		else {
			return EVAL_PAGE;
		}
	}

	/**
	 * ﾀｸﾞﾘﾌﾞｵﾌﾞｼﾞｪｸﾄをﾘﾘｰｽします。
	 * ｷｬｯｼｭされて再利用されるので、ﾌｨｰﾙﾄﾞの初期設定を行います。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) isMultiを追加
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応
	 * @og.rev 5.9.26.1 (2017/11/10) dispError追加
	 * @og.rev 8.0.2.0 (2021/11/30) srcDir,classDir static定義定数を使用
	 */
	@Override
	protected void release2() {
		super.release2();
		command			= CMD_ENTRY;
		logics			= null;
		dbid			= null;
		table 			= null;
		tableId			= HybsSystem.TBL_MDL_KEY;
		selectedAll		= false;
		modifyType		= null;
		keys 			= null;
		vals 			= null;
		stopError		= true;
		quotCheck		= HybsSystem.sysBool( "USE_SQL_INJECTION_CHECK" );
		xssCheck		= HybsSystem.sysBool( "USE_XSS_CHECK" );
		errMessage		= null;
		errCode			= ErrorMessage.OK;
		executeCount	= -1;
		arrTable		= null;
		loader			= null;
//		srcDir			= HybsSystem.sys( "REAL_PATH" ) + HybsSystem.sys( "BIZLOGIC_SRC_PATH" );
		srcDir			= SRC_PATH;												// 8.0.2.0 (2021/11/30)
//		classDir		= HybsSystem.sys( "REAL_PATH" ) + HybsSystem.sys( "BIZLOGIC_CLASS_PATH" );
		classDir		= CLS_PATH;												// 8.0.2.0 (2021/11/30)
		isAutoCompile	= HybsSystem.sysBool( "BIZLOGIC_AUTO_COMPILE" );
		isHotDeploy		= HybsSystem.sysBool( "BIZLOGIC_HOT_DEPLOY" );
		isMulti			= false;												// 5.1.8.0 (2010/07/01) 追加
		dispError		= true;													// 5.9.26.1 (2017/11/10) 追加
	}

	/**
	 * 業務ﾛｼﾞｯｸを実行します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応
	 * @og.rev 5.3.4.0 (2011/04/01) command=ENTRY以外ではDBTableModelの処理を行わない。
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更 、Transaction対応で、close処理を入れる。
	 * @og.rev 5.6.0.3 (2012/01/24) arrTable に変更された値を、table に書き戻す処理を追加
	 * @og.rev 6.3.6.1 (2015/08/28) Transaction でAutoCloseableを使用したtry-with-resources構築に対応。
	 * @og.rev 6.3.6.1 (2015/08/28) Transaction でAutoCloseableを使用したtry-with-resources構築に対応。
	 * @og.rev 6.4.3.3 (2016/03/04) Map#forEach で対応する。
	 * @og.rev 6.7.9.1 (2017/05/19) ArrayTableModel を、DataModel I/F に変更
	 */
	private void execute() {

		// 5.3.4.0 (2011/04/01)
		if( CMD_ENTRY.equals( command ) ) {
			table = (DBTableModel)getObject( tableId );
		}

		final int[] rowNos;
		if( table != null ) {
			rowNos = getParameterRows();
			String[][] tblVals = new String[rowNos.length][table.getColumnCount()];
			String[] modTypes = new String[rowNos.length];
			for( int i=0; i<rowNos.length; i++ ) {
				tblVals[i]  = table.getValues( rowNos[i] );
				modTypes[i] = table.getModifyType( rowNos[i] );
			}
			arrTable = new ArrayTableModel( table.getNames(), tblVals, modTypes );
		}
		else {
			return;
		}

		// 5.1.9.0 (2010/08/01) Transaction 対応
		// 5.3.7.0 (2011/07/01) Transaction対応で、close処理を入れる。
	//		conn = ConnectionFactory.connection( dbid, null );
		// 6.3.6.1 (2015/08/28) Transaction でAutoCloseableを使用したtry-with-resources構築に対応。
		try( Transaction tran = getTransaction() ) {
			errMessage = new ErrorMessage();
			loader = HybsLoaderFactory.getLoader(
						new HybsLoaderConfig( srcDir, classDir, isAutoCompile, isHotDeploy, CLASS_PATH )
					);

			boolean rtn = false;
			for( int i=0; i<logics.length; i++ ) {
	//			BizLogic logic = (BizLogic)loader.newInstance( logics[i] );
	//			rtn = call( logic );
	//			rtn = call( logics[i] );
				rtn = call( logics[i] , tran );									// 5.1.9.0 (2010/08/01) Transaction 対応
				if( !rtn ) { break; }
			}

			// 5.6.0.3 (2012/01/24) arrTable に変更された値を、table に書き戻す処理
			// 6.9.7.0 (2018/05/14) PMD These nested if statements could be combined
//			if( arrTable != null ) {
//				// 6.4.3.3 (2016/03/04) 変更が無い場合は、nulllではなく、空のConcurrentMapを返しましす。
//				// 6.7.9.1 (2017/05/19) ArrayTableModel を、DataModel I/F に変更
//				if( arrTable instanceof ArrayTableModel ) {
			// 6.9.8.0 (2018/05/28) FindBugs:常に true を返す instanceof
//			if( arrTable != null && arrTable instanceof ArrayTableModel ) {
			if( arrTable != null ) {
				// 6.4.3.3 (2016/03/04) 変更が無い場合は、nulllではなく、空のConcurrentMapを返しましす。
				// 6.7.9.1 (2017/05/19) ArrayTableModel を、DataModel I/F に変更
					final ConcurrentMap<Integer,String[]> valMap = ((ArrayTableModel)arrTable).getModifyVals();
					valMap.forEach( (seq,vals) -> table.setValues( vals , rowNos[seq] ) );
//				}

				// 8.2.1.1 (2022/07/19) DELETE だけ特別に処置する(論理削除で)
				for( int i=0; i<rowNos.length; i++ ) {
					if( DataModel.DELETE_TYPE.equals( arrTable.getModifyType(i) ) ) {
						// 論理削除…後で物理削除している。
						table.rowDelete( rowNos[i] );
					}
				}
			}

			executeCount = rowNos.length;
			errCode = errMessage.getKekka();
			setRequestAttribute( "DB.COUNT"   , String.valueOf( executeCount ) );
			setRequestAttribute( "DB.ERR_CODE", String.valueOf( errCode ) );

			if( errCode < ErrorMessage.NG ) {
	//			Closer.commit( conn );
				tran.commit();													// 5.1.9.0 (2010/08/01) Transaction 対応

	//			if( table != null && rowNos.length > 0 ) {
					for( int j=rowNos.length-1; j>=0; j-- ) {
						final int row = rowNos[j];
						if( DBTableModel.DELETE_TYPE.equals( table.getModifyType( row ) ) ) {
							table.removeValue( row );
						}
						else {
							table.resetModify( row );
						}
					}
	//			}
			}
			else {
	//			Closer.rollback( conn );
				tran.rollback();												// 5.1.9.0 (2010/08/01) Transaction 対応
			}
	//		ConnectionFactory.close( conn, dbid );
		}

		// ｴﾗｰﾒｯｾｰｼﾞの行番号を元の選択行に戻します。
		final ErrMsg[] errs = errMessage.toArray();
		final ErrorMessage errMsgTmp = new ErrorMessage();
		for( int i=0; i<errs.length; i++ ) {
			if( table != null && rowNos.length > 0 ) {
				errMsgTmp.addMessage( errs[i].copy( rowNos[errs[i].getNo()] + 1 ) );
			}
			else {
				errMsgTmp.addMessage( errs[i].copy( errs[i].getNo() + 1 ) );
			}
		}
		errMessage = errMsgTmp;
	}

	/**
	 * 業務ﾛｼﾞｯｸをCallします。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応
	 * @og.rev 5.1.9.0 (2010/08/01) DBIDをｾｯﾄ、ConnectonではなくTransactionを渡すように変更
	 * @og.rev 5.6.7.0 (2013/07/27) Exception を throw するとき、一旦、errMsg 変数にｾｯﾄします。
	 * @og.rev 6.7.9.1 (2017/05/19) RETURN を返す変数が、logicName ＋ .RETURN にします。
	 * @og.rev 6.9.9.0 (2018/08/20) 戻り値を返せるようにします。
	 *
	 * @param	logicName	業務ﾛｼﾞｯｸ名
	 * @param	tran		ﾗﾝｻﾞｸｼｮﾝｵﾌﾞｼﾞｪｸﾄ
	 *
	 * @return	業務ﾛｼﾞｯｸの呼び出しが成功したかどうか
	 */
	private boolean call( final String logicName , final Transaction tran ) {
		final BizLogicHelper logicHp = new BizLogicHelper( logicName, loader );

		if( logicHp.isRequireTable() ) {
			if( arrTable == null ) {
				// 5.6.7.0 (2013/07/27) Exception を throw するとき、一旦、errMsg 変数にｾｯﾄします。
				final String errMsg = "TableModelが存在しません。logic=[" + logicName + "]" ;
				throw new HybsSystemException( errMsg );
			}
			else if( arrTable.getRowCount() == 0 ) { return true; } // 0件のときは呼び出ししない
			else { logicHp.setTable( arrTable ); }
		}
		logicHp.setTransaction( tran );
		logicHp.setDbid( dbid );												// 5.1.9.0 (2010/08/01) DBIDをセット
		logicHp.setKeys( keys );
		logicHp.setVals( vals );
		logicHp.setUserId( getUser().getUserID() );
		logicHp.setParentPgId( getGUIInfoAttri( "KEY" ) );
		logicHp.setLoader( loader );
		if( isDebug() ) { logicHp.setDebug(); }

		boolean rtn = false;
		try {
			rtn = logicHp.exec();
		}
		catch( final Throwable th ) {
			tran.rollback();		// 5.1.9.0 (2010/08/01) Transaction 対応
			// 5.6.7.0 (2013/07/27) Exception を throw するとき、一旦、errMsg 変数にｾｯﾄします。
			final String errMsg = "業務ﾛｼﾞｯｸの処理中にｴﾗｰが発生しました。" + th.getMessage() ;
			throw new HybsSystemException( errMsg,th );
		}
		errMessage.append( logicHp.getErrMsg() );

		// 6.7.9.1 (2017/05/19) RETURN を返す変数が、logicName ＋ .RETURN にします。
		// 現状は、BizLogicHelper.RETURN なので、本来は、分けて使いたかったはずなので。
		setRequestAttribute( logicName + ".RETURN"   , logicHp.getReturn() );
		setRequestAttribute( "RETURN", logicHp.getReturn() );

		// 6.9.9.0 (2018/08/20) 戻り値を返せるようにします。
		final Map<String,String> rtnMap = logicHp.getReturnMap();
		if( rtnMap != null ) {
			rtnMap.forEach( (k,v) -> setRequestAttribute( k,v ) );
		}

		if( isDebug() ) { jspPrint( logicHp.getDebugMsg() ); }

		return rtn;
	}

	/**
	 * Valsの配列を生成します。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) 新規作成-
	 * @og.rev 5.6.7.0 (2013/07/27) Exception を throw するとき、一旦、errMsg 変数にｾｯﾄします。
	 */
	private void makeVals() {
		if( keys != null && keys.length > 0 ) {
			final boolean isSetVal = vals != null && vals.length > 0 ;
			if( isSetVal ) {
				if( keys.length != vals.length ) {
					// 5.6.7.0 (2013/07/27) Exception を throw するとき、一旦、errMsg 変数にｾｯﾄします。
					final String errMsg = "keysとValsの個数が異なります。"  + CR
								+ "  keys=" + Arrays.toString( keys ) + CR
								+ "  vals=" + Arrays.toString( vals ) ;
					throw new HybsSystemException( errMsg );
				}
			}
			else {
				vals = new String[keys.length];
			}
			for( int i=0; i<keys.length; i++ ) {
				if( isSetVal ) {
					if( isMulti )	{ vals[i] = StringUtil.array2csv( getRequestParameterValues( vals[i] ) );}
					else			{ vals[i] = getRequestParameter( vals[i] ); }
				}
				else {
					if( isMulti )	{ vals[i] = StringUtil.array2csv( getRequestValues( keys[i] ) );}
					else			{ vals[i] = getRequestValue( keys[i] ); }
				}
			}
		}
	}

	/**
	 * 【TAG】ｺﾏﾝﾄﾞをｾｯﾄします(初期値:ENTRY)。
	 *
	 * @og.tag
	 * command=ENTRY以外ではDBTableModelの処理を行いません。
	 * ｺﾏﾝﾄﾞは、HTMLから[get/post]指定されますので、CMD_xxx で設定される
	 * ﾌｨｰﾙﾄﾞ定数値のいづれかを、指定できます。
	 *
	 * @param	cmd	ｺﾏﾝﾄﾞ (public static final 宣言されている文字列)
	 * @see		<a href="../../../../constant-values.html#org.opengion.hayabusa.taglib.BizLogicTag.CMD_ENTRY">ｺﾏﾝﾄﾞ定数</a>
	 */
	public void setCommand( final String cmd ) {
		final String cmd2 = getRequestParameter( cmd );
		if( cmd2 != null && cmd2.length() >= 0 ) { command = cmd2.toUpperCase(Locale.JAPAN); }
	}

	/**
	 * 【TAG】実行する業務ﾛｼﾞｯｸ名を指定します。
	 *
	 * @og.tag
	 * 実行する業務ﾛｼﾞｯｸ名を指定します。業務ﾛｼﾞｯｸ名は、ｸﾗｽ名を指定します。
	 * ｸﾗｽ名については、ｸﾗｽ自身の名称のみを指定することができます。
	 * (ﾊﾟｯｹｰｼﾞ名を含めた完全な形のｸﾗｽ名を指定することもできます)
	 * また、CSV形式で、複数指定することもできます。
	 * この場合、指定した順番に処理されます。
	 *
	 * @param	lgs	業務ﾛｼﾞｯｸ名
	 */
	public void setLogics( final String lgs ) {
		logics = getCSVParameter( lgs );
	}

	/**
	 * 【TAG】(通常は使いません)Queryｵﾌﾞｼﾞｪｸﾄを作成する時のDB接続IDを指定します(初期値:null)。
	 *
	 * @og.tag
	 * Queryｵﾌﾞｼﾞｪｸﾄを作成する時のDB接続IDを指定します。
	 * これは、ｼｽﾃﾑﾘｿｰｽで、DEFAULT_DB_URL 等で指定している ﾃﾞｰﾀﾍﾞｰｽ接続先
	 * 情報に、XX_DB_URL を定義することで、 dbid="XX" とすると、この 接続先を使用して
	 * ﾃﾞｰﾀﾍﾞｰｽにｱｸｾｽできます。
	 *
	 * @param	id	ﾃﾞｰﾀﾍﾞｰｽ接続ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}

	/**
	 * 【TAG】(通常は使いません)結果のDBTableModelを、sessionに登録するときのｷｰを指定します
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @og.tag
	 * 検索結果より、DBTableModelｵﾌﾞｼﾞｪｸﾄを作成します。これを、下流のviewﾀｸﾞ等に
	 * 渡す場合に、通常は、session を利用します。その場合の登録ｷｰです。
	 * query ﾀｸﾞを同時に実行して、結果を求める場合、同一ﾒﾓﾘに配置される為、
	 * この tableId 属性を利用して、ﾒﾓﾘ空間を分けます。
	 * (初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @param	id	ﾃｰﾌﾞﾙID (sessionに登録する時のID)
	 */
	public void setTableId( final String id ) {
		tableId = nval( getRequestParameter( id ),tableId );
	}

	/**
	 * 【TAG】ﾃﾞｰﾀを全件選択済みとして処理するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 全てのﾃﾞｰﾀを選択済みﾃﾞｰﾀとして扱って処理します。
	 * 全件処理する場合に、(true/false)を指定します。
	 * 初期値は false です。
	 *
	 * @param	all	ﾃﾞｰﾀを全件選択済み [true:全件選択済み/false:通常]
	 */
	public void setSelectedAll( final String all ) {
		selectedAll = nval( getRequestParameter( all ),selectedAll );
	}

	/**
	 * 【TAG】DB検索時の ﾓﾃﾞｨﾌｧｲﾀｲﾌﾟを指定します[A:追加/C:更新/D:削除]。
	 *
	 * @og.tag
	 * DB検索時に、そのﾃﾞｰﾀをA(追加)、C(更新)、D(削除)のﾓﾃﾞｨﾌｧｲﾀｲﾌﾟを
	 * つけた状態にします。
	 * その状態で、そのまま、update する事が可能になります。
	 *
	 * @param	type	ﾓﾃﾞｨﾌｧｲﾀｲﾌﾟ [A:追加/C:更新/D:削除]
	 */
	public void setModifyType( final String type ) {
		modifyType = nval( getRequestParameter( type ),modifyType );
	}

	/**
	 * 【TAG】ﾘﾝｸ先に渡すｷｰをCSV形式で複数指定します。
	 *
	 * @og.tag
	 * ﾘﾝｸ先に渡すｷｰを指定します。
	 * Keysだけを指定して、Valsを指定しない場合、Keysで指定された項目名に対応するﾊﾟﾗﾒｰﾀｰを取得し、
	 * Valsとして使用します。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) CommonTagSupport#getCSVParameter を使用
	 *
	 * @param	key	ﾘﾝｸ先に渡すｷｰ(CSV形式)
	 */
	public void setKeys( final String key ) {
		keys = getCSVParameter( getRequestParameter( key ) );
	}

	/**
	 * 【TAG】ﾘﾝｸ先に渡す値をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * ﾘﾝｸ先に渡す値を指定します。
	 * Keysだけを指定して、Valsを指定しない場合、Keysで指定された項目名に対応するﾊﾟﾗﾒｰﾀｰを取得し、
	 * Valsとして使用します。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとﾃﾞｰﾀ自身にｶﾝﾏを持っている場合に分解をﾐｽる為です。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) CommonTagSupport#getCSVParameter を使用
	 * @og.rev 5.1.8.0 (2010/07/01) isMuitl対応
	 *
	 * @param	val	ﾘﾝｸ先に渡す値(CSV形式)
	 */
	public void setVals( final String val ) {
		vals = StringUtil.csv2Array( val );
	}

	/**
	 * 【TAG】処理ｴﾗｰの時に処理を中止するかどうか[true/false]を設定します(初期値:true)。
	 *
	 * @og.tag
	 * false(中止しない)に設定する場合、後続処理では、{&#064;DB.ERR_CODE}の値により、
	 * PLSQL/SQLの異常/正常終了によって分岐処理は可能となります。
	 * 初期値は、true(中止する)です。
	 *
	 * @param	flag	処理の中止 [true:中止する/false:中止しない]
	 */
	public void setStopError( final String flag ) {
		stopError = nval( getRequestParameter( flag ),stopError );
	}

	/**
	 * 【TAG】PLSQL/SQL処理ｴﾗｰの時にｴﾗｰを画面表示するか[true/false]を設定します(初期値:true)。
	 *
	 * @og.tag
	 * false(表示しない)に設定する場合、後続処理では、{&#064;DB.ERR_MSG}の値により、
	 * 本来表示されるはずだったﾒｯｾｰｼﾞを取得可能です。
	 * stopErrorと併用して、JSON形式でｴﾗｰを返す場合等に利用します。
	 * 初期値は、true(表示する)です。
	 *
	 * @og.rev 5.9.26.1 (2017/11/10) 新規追加
	 *
	 * @param	flag	[true:表示する/false:表示しない]
	 */
	public void setDispError( final String flag ) {
		dispError = nval( getRequestParameter( flag ),dispError );
	}

	/**
	 * 【TAG】ﾘｸｴｽﾄ情報の ｼﾝｸﾞﾙｸｫｰﾄ(') 存在ﾁｪｯｸを実施するかどうか[true/false]を設定します
	 *		(初期値:USE_SQL_INJECTION_CHECK[={@og.value SystemData#USE_SQL_INJECTION_CHECK}])。
	 *
	 * @og.tag
	 * ＳＱＬｲﾝｼﾞｪｸｼｮﾝ対策の一つとして、暫定的ではありますが、SQLのﾊﾟﾗﾒｰﾀに
	 * 渡す文字列にｼﾝｸﾞﾙｸｫｰﾄ(') を許さない設定にすれば、ある程度は防止できます。
	 * 数字ﾀｲﾌﾟの引数には、 or 5=5 などのｼﾝｸﾞﾙｸｫｰﾄを使用しないｺｰﾄﾞを埋めても、
	 * 数字ﾁｪｯｸで検出可能です。文字ﾀｲﾌﾟの場合は、必ず (')をはずして、
	 * ' or 'A' like 'A のような形式になる為、(')ﾁｪｯｸだけでも有効です。
	 * (') が含まれていたｴﾗｰにする(true)／かﾉｰﾁｪｯｸか(false)を指定します。
	 * (初期値:ｼｽﾃﾑ定数のUSE_SQL_INJECTION_CHECK[={@og.value SystemData#USE_SQL_INJECTION_CHECK}])。
	 *
	 * @param	flag	ｸｫｰﾄﾁｪｯｸ [true:する/それ以外:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_SQL_INJECTION_CHECK
	 */
	public void setQuotCheck( final String flag ) {
		quotCheck = nval( getRequestParameter( flag ),quotCheck );
	}

	/**
	 * 【TAG】ﾘｸｴｽﾄ情報の HTMLTag開始/終了文字(&gt;&lt;) 存在ﾁｪｯｸを実施するかどうか[true/false]を設定します
	 *		(初期値:USE_XSS_CHECK[={@og.value SystemData#USE_XSS_CHECK}])。
	 *
	 * @og.tag
	 * ｸﾛｽｻｲﾄｽｸﾘﾌﾟﾃｨﾝｸﾞ(XSS)対策の一環としてless/greater than signについてのﾁｪｯｸを行います。
	 * (&gt;&lt;) が含まれていたｴﾗｰにする(true)／かﾉｰﾁｪｯｸか(false)を指定します。
	 * (初期値:ｼｽﾃﾑ定数のUSE_XSS_CHECK[={@og.value SystemData#USE_XSS_CHECK}])。
	 *
	 * @param	flag	XSSﾁｪｯｸする [true:ﾁｪｯｸする/false:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_XSS_CHECK
	 */
	public void setXssCheck( final String flag ) {
		xssCheck = nval( getRequestParameter( flag ),xssCheck );
	}

	/**
	 * 【TAG】vals属性でﾊﾟﾗﾒｰﾀｰを取得する際、複数件存在する場合に、値を連結するかどうかを指定します(初期値:false)。
	 *
	 * @og.tag
	 * この属性がtrueに指定された場合、ﾊﾟﾗﾒｰﾀｰが複数存在する場合に、ｶﾝﾏで連結します。
	 * 初期値は、false(連結しない)です。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) 新規作成
	 *
	 * @param	flag	値連結 [true:する/false:しない]
	 */
	public void setMulti( final String flag ) {
		isMulti = nval( getRequestParameter( flag ),isMulti );
	}

	/**
	 * 表示ﾃﾞｰﾀの HybsSystem.ROW_SEL_KEY を元に、選ばれた 行を処理の対象とします。
	 *
	 * @return	選択行の配列
	 */
	@Override
	protected int[] getParameterRows() {
		final int[] rowNo ;
		if( selectedAll ) {
			final int rowCnt = table.getRowCount();
			rowNo = new int[ rowCnt ];
			for( int i=0; i<rowCnt; i++ ) {
				rowNo[i] = i;
			}
		} else {
			rowNo = super.getParameterRows();
		}
		return rowNo ;
	}

	/**
	 * このｵﾌﾞｼﾞｪｸﾄの文字列表現を返します。
	 * 基本的にﾃﾞﾊﾞｯｸﾞ目的に使用します。
	 *
	 * @return このｸﾗｽの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "command"			,command		)
				.println( "logics"			,logics			)
				.println( "dbid"			,dbid			)
				.println( "tableId"			,tableId		)
				.println( "selectedAll"		,selectedAll	)
				.println( "modifyType"		,modifyType		)
				.println( "keys"			,keys			)
				.println( "vals"			,vals			)
				.println( "stopError"		,stopError		)
				.println( "quotCheck"		,quotCheck		)
				.println( "xssCheck"		,xssCheck		)
				.println( "executeCount"	,executeCount	)
				.println( "errCode"			,errCode		)
				.println( "Other..."		,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
