/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBColumnConfig;
import org.opengion.hayabusa.resource.CodeData;
import org.opengion.hayabusa.resource.RoleMode;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.util.ArraySet;						// 6.4.3.4 (2016/03/11)
import static org.opengion.fukurou.util.StringUtil.nval;

import java.util.Set;											// 6.4.3.4 (2016/03/11)

/**
 * ｶﾗﾑﾘｿｰｽに基づいた、入力ﾌｨｰﾙﾄﾞなどを作成するHTML拡張ﾀｸﾞです。
 *
 * name 属性に ｶﾗﾑﾘｿｰｽのｷｰを与えることにより、指定した表示形式にて画面に表示します。
 * column ﾀｸﾞは、ﾘｿｰｽの設定が反映されるため、ｱﾌﾟﾘｹｰｼｮﾝを修正せずに、
 * 機能拡張することが可能です。たとえば、ﾃｷｽﾄﾌｨｰﾙﾄﾞ入力に設定していたｶﾗﾑを、
 * ﾌﾟﾙﾀﾞｳﾝﾒﾆｭｰに即座に変更可能です。
 * ｶﾗﾑ定義でﾛｰﾙが指定されている場合、ﾛｸﾞｲﾝﾕｰｻﾞｰでｱｸｾｽ許可がない場合は
 * ｶﾗﾑは表示されません。
 * ｶﾗﾑﾘｿｰｽに存在しない場合や特殊用途に使用する場合は、inputﾀｸﾞを使用してください。
 *
 * @og.formSample
 * ●形式：&lt;og:column name=… value=… /&gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:column
 *       name                 ○【TAG】ｶﾗﾑの名前を指定します(必須)。
 *       value                  【TAG】設定値を指定します
 *       defaultVal             【TAG】value属性に値がｾｯﾄされていないときに、初期値をｾｯﾄします
 *       td                     【TAG】ﾃｰﾌﾞﾙ形式の &lt;td&gt; ﾀｸﾞを使用するかどうか[yes/no/false]を指定します (初期値:yes)
 *       addNoValue             【TAG】ｾﾚｸﾄﾒﾆｭｰの場合、値なしのOptionを含めるかどうか[true/false]を指定します (初期値:true)
 *       addKeyLabel            【TAG】ｾﾚｸﾄﾒﾆｭｰの場合、ｷｰ:ﾗﾍﾞﾙ形式で表示するかどうか[true/false/null]を指定します (初期値:null)
 *       lbl                    【TAG】ﾗﾍﾞﾙﾘｿｰｽのﾗﾍﾞﾙIDを指定します
 *       must                   【TAG】必須入力を表す色に変えるかどうか[true/false]を指定します (初期値:false)
 *       mustAny                【TAG】選択必須入力(どれかひとつ必須)を表す色[true/mustAny/その他]を指定します (初期値:無指定)
 *       useMustHidden          【TAG】必須の自動ﾁｪｯｸ用Hiddenを出力するかどうか[true/false]を指定します (初期値:true)
 *       aimai                  【TAG】曖昧検索可能ﾌｨｰﾙﾄﾞとして、曖昧検索方法を指定します (初期値:null)
 *       writable               【TAG】書き込みﾀｲﾌﾟかどうか[true/false]を指定します (初期値:true)
 *       colspan                【HTML】ﾌｨｰﾙﾄﾞ部分の colspan を指定します
 *       rowspan                【HTML】ﾗﾍﾞﾙ、ﾌｨｰﾙﾄﾞ共に rowspan を指定します
 *       id                     【HTML】要素に対して固有の名前(id)をつける場合に設定します
 *       size                   【HTML】ｶﾗﾑの入力枠ｻｲｽﾞ(横幅)を文字数でｾｯﾄします(入力文字制限ではありません)
 *       viewSize               【TAG】ｶﾗﾑの表示ｻｲｽﾞ(横幅)を文字数でｾｯﾄします。
 *       maxlength              【HTML】最大入力文字数を指定します(入力可能文字数で、ﾊﾞｲﾄ数ではありません)
 *       renderer               【TAG】ｶﾗﾑの表示用ﾚﾝﾃﾞﾗｰ({@og.doc03Link renderer Renderer_****})をｾｯﾄします
 *       editor                 【TAG】ｶﾗﾑの編集用ｴﾃﾞｨﾀｰ({@og.doc03Link editor Editor_****})をｾｯﾄします
 *       dbType                 【TAG】DBﾀｲﾌﾟ({@og.doc03Link dbType DBType_****})をｾｯﾄします
 *       optionAttributes       【TAG】JavaScript などの HTML基本ﾀｸﾞ以外の属性を、そのままﾀｸﾞとして使用します
 *  //   useDefaultPropertieVal 【廃止】ｶﾗﾑﾘｿｰｽの初期値を使うかどうか[true/false]を指定します (初期値:false) 8.4.3.0 (2023/03/31) Delete
 *       nextForm               【TAG】入力ｶｰｿﾙを、指定されたname属性のﾌｨｰﾙﾄﾞへ自動的に移動させます
 *       useRequestValue        【TAG】value が null の場合に、name 属性から Request情報 を適用するかどうかを指定します (初期値:true)
 *       codeName               【TAG】ｺｰﾄﾞｾﾚｸﾄのﾕﾆｰｸｷｰ(ｺｰﾄﾞ名称)をｾｯﾄします
 *       codeList               【TAG】ｺｰﾄﾞｾﾚｸﾄのｻﾌﾞｾｯﾄとなるｺｰﾄﾞをCSV形式で指定します
 *       codeGroup              【TAG】ｺｰﾄﾞｾﾚｸﾄのｻﾌﾞｾｯﾄとなるｸﾞﾙｰﾌﾟを指定します
 *       codeKeyVal             【TAG】新しいｺｰﾄﾞｾﾚｸﾄを作成する為の｢ｷｰ:ﾗﾍﾞﾙ ｷｰ:ﾗﾍﾞﾙ｣の文字列を指定します
 *       useLblBody             【TAG】BODYに記述した内容で、ﾗﾍﾞﾙを作成します 7.4.2.0 (2021/04/30)
 *       clazz                  【HTML】要素に対して class 属性を設定します
 *       caseKey                【TAG】このﾀｸﾞ自体を利用するかどうかの条件ｷｰを指定します (初期値:null)
 *       caseVal                【TAG】このﾀｸﾞ自体を利用するかどうかの条件値を指定します (初期値:null)
 *       caseNN                 【TAG】指定の値が、null/ｾﾞﾛ文字列 でない場合(Not Null=NN)は、このﾀｸﾞは使用されます (初期値:判定しない)
 *       caseNull               【TAG】指定の値が、null/ｾﾞﾛ文字列 の場合は、このﾀｸﾞは使用されます (初期値:判定しない)
 *       caseIf                 【TAG】指定の値が、true/TRUE文字列の場合は、このﾀｸﾞは使用されます (初期値:判定しない)
 *       lang                   【HTML】要素の内容と他の属性値の言語(lang,xml:lang)を指定します
 *       dir                    【HTML】文字表記の方向(dir)を指定します
 *       title                  【HTML】要素に対する補足的情報(title)を設定します
 *       style                  【HTML】この要素に対して適用させるｽﾀｲﾙｼｰﾄ(style)を設定します
 *       readonly               【HTML】その部品に対して変更が出来ないように(readonly)指定します(ｻｰﾊﾞｰに送信される)
 *       disabled               【HTML】その部品に対して、選択や変更が出来ないように(disabled)指定します(ｻｰﾊﾞｰに送信されない)
 *       tabindex               【HTML】ﾀﾌﾞの移動順(tabindex)を指定します(0 ～ 32767)
 *       accesskey              【HTML】ｱｸｾｽｷｰ(alt+ｷｰで直接指定)を割り当てます
 *       language               【HTML】ﾀｸﾞ内部で使用する言語ｺｰﾄﾞ[ja/en/zh/…]を指定します
 *       onClick                【HTML】JavaScriptのｲﾍﾞﾝﾄ onClick を設定します(例:onClick="renew('query.jsp','QUERY');")
 *       onChange               【HTML】JavaScriptのｲﾍﾞﾝﾄ onChange を設定します(例:onChange="renew('query.jsp','QUERY');")
 *       onBlur                 【HTML】JavaScriptのｲﾍﾞﾝﾄ onBlur を設定します(例:onBlur="this.value=value.toUpperCase();")
 *       onFocus                【HTML】JavaScriptのｲﾍﾞﾝﾄ onFocus を設定します
 *       ondblClick             【HTML】JavaScriptのｲﾍﾞﾝﾄ ondblClick を設定します
 *       onMouseDown            【HTML】JavaScriptのｲﾍﾞﾝﾄ onMouseDown を設定します
 *       onMouseUp              【HTML】JavaScriptのｲﾍﾞﾝﾄ onMouseUp を設定します
 *       onMouseMove            【HTML】JavaScriptのｲﾍﾞﾝﾄ onMouseMove を設定します
 *       onMouseOut             【HTML】JavaScriptのｲﾍﾞﾝﾄ onMouseOut を設定します
 *       onMouseOver            【HTML】JavaScriptのｲﾍﾞﾝﾄ onMouseOver を設定します
 *       onSelect               【HTML】JavaScriptのｲﾍﾞﾝﾄ onSelect を設定します
 *       onKeydown              【HTML】JavaScriptのｲﾍﾞﾝﾄ onKeydown を設定します
 *       onKeypress             【HTML】JavaScriptのｲﾍﾞﾝﾄ onKeypress を設定します
 *       onKeyup                【HTML】JavaScriptのｲﾍﾞﾝﾄ onKeyup を設定します
 *       autocomplete           【HTML5】入力候補を提示して入力内容を自動補完する[on/off] (初期値はon)
 *       autofocus              【HTML5】指定した入力欄にｶｰｿﾙが当たって自動的にﾌｫｰｶｽされます
 *       pattern                【HTML5】正規表現で入力値のﾊﾟﾀｰﾝを指定します
 *       placeholder            【HTML5】入力欄に初期表示する内容を指定します
 *       required               【HTML5】入力必須を指定します
 *       list                   【HTML5】ﾕｰｻﾞｰに入力候補として提案するﾃﾞｰﾀﾘｽﾄの要素のid属性の値を指定します
 *       min                    【HTML5】入力欄で入力できる最小値を指定します
 *       max                    【HTML5】入力欄で入力できる最大値を指定します
 *       step                   【HTML5】入力欄で刻むｽﾃｯﾌﾟ値を指定する
 *       type                   【HTML5】HTML5追加分[text/search/tel/url/email/datetime/date/month/week/time/datetime-local/number/range/color]
 *       roles                  【TAG】ﾛｰﾙをｾｯﾄします
 *       eventColumn            【TAG】ｲﾍﾞﾝﾄｶﾗﾑ(親ｶﾗﾑ)ｾｯﾄ
 *       eventValue             【TAG】ｲﾍﾞﾝﾄValueのSQL分
 *       eventURL               【TAG】ｲﾍﾞﾝﾄURLのｾｯﾄ
 *       eventCallback          【TAG】eventColumn実行後のcallbak関数指定
 *       param                  【TAG】ﾊﾟﾗﾒｰﾀをｾｯﾄします(eventColumn でDBMENUの場合は、select文を記述します)
 *       dbid                   【TAG】(通常は使いません)ﾃﾞｰﾀﾍﾞｰｽへ接続する場合のID (初期値:null)
 *       xssCheck               【TAG】ﾘｸｴｽﾄ情報の HTMLTag開始/終了文字(&gt;&lt;) 存在ﾁｪｯｸを実施するかどうか[true/false]を設定します (初期値:USE_XSS_CHECK[=true])
 *       useSLabel              【TAG】MENU,DBMENUでSLABELを利用するかどうか[auto/true/false]を指定します (初期値:auto)
 *       help                   【TAG】このｶﾗﾑに対するﾍﾙﾌﾟﾌｧｲﾙを指定します (初期値:null)
 *       useRealTimeCheck       【TAG】(通常は使いません)ﾘｱﾙﾀｲﾑﾁｪｯｸを行うかどうか (初期値:true:行う) 5.9.32.2 (2018/05/18)
 *       useDateFeed            【TAG】日付送り戻し機能を有効にするかどうか[true:有効/false:無効]指定します (初期値:true) 8.1.2.3 (2022/05/20)
 *       useHidden              【TAG】隠しﾌｨｰﾙﾄﾞを作成するかどうか[true/false]指定します (初期値:false) 8.3.1.0 (2022/10/14)
 *       debug                  【TAG】ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうか[true/false]を指定します (初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:column&gt;
 *
 * ●使用例
 *    &lt;og:column name="PN" /&gt;                                 編集種別がTexｔの場合は、ﾃｷｽﾄﾌｨｰﾙﾄﾞが表示されます。
 *                                                                   通常はNAME属性を指定するだけです。
 *    &lt;og:column name="CDC" /&gt;                                編集種別がMenuの場合は、ﾒﾆｭｰが表示されます。
 *    &lt;og:column name="PN" value="syokichi" /&gt;                初期値をｾｯﾄするときはvalue属性を使います。
 *    &lt;og:column name="PN" lbl="PNL" /&gt;                       ﾗﾍﾞﾙを変更するときはlbl属性を使います。
 *    &lt;og:column name="CDC" /&gt;                                ｶﾗﾑﾘｿｰｽからｾﾚｸﾄﾒﾆｭｰも自動作成されます。
 *    &lt;og:column name="CDC" addNoValue="false" /&gt;             NULL値を消すときはaddNoValue属性を使います。
 *    &lt;og:column name="CDC" addNoValue="false" must="true" /&gt; 属性は、複数組み合わせて使えます。
 *    &lt;og:column name="PN" must="true" /&gt;                     色を変更するときはmust属性を使います。
 *    &lt;og:column name="PN" td="false" /&gt;                      自動作成されるtdﾀｸﾞやﾗﾍﾞﾙを外すときはtd属性を使います。
 *    &lt;og:column name="PN" td="no" /&gt;                         自動作成されるtdﾀｸﾞを外し、ﾗﾍﾞﾙとﾌｨｰﾙﾄﾞは残します。
 *    &lt;og:column name="PN" writable="false" /&gt;                非入力にするときはwritable属性を使います。
 *    &lt;og:column name="PN" size="20" /&gt;                       強制的にSIZEの変更をするときはsize属性を使います。
 *    &lt;og:column name="A"  nextForm="B" /&gt;                    最大桁数入力後、ﾌｫｰﾑ B にﾌｫｰｶｽが移動します。
 *    &lt;og:column name="B" /&gt;
 *
 *    BODY 部分に記述した値は、ｶﾗﾑﾌｨｰﾙﾄﾞの直後にｾｯﾄされます。
 *    &lt;og:column name="PN" &gt;
 *      &lt;og:button type="button" onclick="popup(･･･);return false;"&gt;
 *          &lt;og:message lbl="POPUP" comment="▼" /&gt;
 *      &lt;/og:button&gt;
 *    &lt;/og:column&gt;
 *
 *    eventColumnを利用すると、親ｶﾗﾑの変更に従って子ｶﾗﾑの内容を変える事が可能です。
 *    ･親子関係のﾌﾟﾙﾀﾞｳﾝを作成する場合
 *    &lt;og:column name="OYA" /&gt;
 *    &lt;og:column name="KO" eventColumn="OYA" param="SELECT ... WHERE COL = '{&#064;OYA}'" /&gt;
 *
 *    ･親子関係の入力欄(親のﾃﾞｰﾀで子をDBから検索)を作成する場合
 *    &lt;og:column name="OYA" /&gt;
 *    &lt;og:column name="KO" eventColumn="OYA" param="SELECT A AS VALUE WHERE COL = {&#064;OYA}" /&gt;
 *
 *    ･親子関係の要素をｶｽﾀﾑ化されたJSPを利用して作成する場合
 *    変更するｶﾗﾑはJSPで記述されており、指定する事が可能です。
 *    &lt;og:column name="OYA" /&gt;
 *    &lt;og:column name="KO" eventColumn="OYA" param="{&#064;OYA}" eventURL="custom_makeColumn.jsp" /&gt;
 *
 * @og.group 画面部品
 *
 * @version	4.0
 * @author	Kazuhiko Hasegawa
 * @since	JDK5.0,
 */
public class ColumnTag extends HTMLTagSupport {
	/** このﾌﾟﾛｸﾞﾗﾑのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "8.5.2.0 (2023/07/14)" ;
	private static final long serialVersionUID = 852020230714L ;

	// 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	private static final Set<String> TYPE_SET = new ArraySet<>( "text","search","tel","url","email","datetime","date","month","week","time","datetime-local","number","range","color" );
	private static final Set<String> TD_SET   = new ArraySet<>( "yes" , "no" , "false" );

	// 8.4.3.0 (2023/03/31) ｼｽﾃﾑ定数 初期値を 使う:true / 使わない:false
	private static final boolean USE_DEFVAL = HybsSystem.sysBool( "USE_DEFAULT_PROPERTIE_VALUE" );

	/** HTML_LABEL_SEPARATOR を boolean 変数として取得します。 */
	private final String CLM = HybsSystem.sysBool( "HTML_LABEL_SEPARATOR" ) ? ":" : "" ;	// 4.0.0 (2005/01/31) 

	/** yes:&lt;td&gt; ﾀｸﾞを使用 / false:ﾌｨｰﾙﾄﾞのみ / no:ﾗﾍﾞﾙとﾌｨｰﾙﾄﾞ */
	private String	tdFlag		= "yes";
	/** 値なしのOptionをSelectに含めるかどうか */
	private boolean	addNoValue	= true;
	/** ｷｰ:ﾗﾍﾞﾙ形式で表示するかどうか */
	private String	addKeyLabel	;												// 6.0.4.0 (2014/11/28)
	/** ｶﾗﾑの名前 */
	private String	name		;
	/** 設定値 */
	private String	value		;
	/** 書き込み/ﾗﾍﾞﾙを指定する */
	private boolean	writable	= true;											// true で書き込み
	/** value がｾｯﾄされていないときの初期値 */
	private String	defaultVal	= "";
	// 2.1.0.0 (2002/10/11) useDefaultPropertieVal 属性を追加。
	// 5.2.3.0 (2010/12/01) useDefaultPropertieVal 属性の初期値をｼｽﾃﾑ定数で定義
	// properties ﾌｧｲﾙの初期値を 使う:true / 使わない:false
//	private boolean	useDefVal	= HybsSystem.sysBool( "USE_DEFAULT_PROPERTIE_VALUE" );	// 8.4.3.0 (2023/03/31) Delete

	/** colspan 属性 */
	private String	colspan		= "";											// 3.0.0.0 (2002/12/25)
	/** rowspan 属性 */
	private String	rowspan		= "";											// 6.2.0.0 (2015/02/27)
	/** nextForm 属性 */
	private String	nextForm	;												// 3.5.4.2 (2003/12/15)

	/** value が null の場合に、ﾘｸｴｽﾄ情報を設定します(初期値:true)。 */
	private boolean useRequestValue	= true;										// 3.5.4.9 (2004/02/25)

	// 3.5.6.0 (2004/06/18) Attributes 経由でなく、直接 DBColumnConfig に設定します。
	/** 入力枠ｻｲｽﾞ(横幅) */
	private String	fieldSize	;												// 6.2.0.0 (2015/02/27)
	/** 表示ｻｲｽﾞ(横幅) */
	private String	viewSize	;												// 6.2.0.0 (2015/02/27)
	/** 最大入力文字数 */
	private String	maxlength	;

	/** BODY 部分に記述した値は、ｶﾗﾑﾌｨｰﾙﾄﾞの直後にｾｯﾄされます。 */
	private String	body		;												// 3.8.0.4 (2005/08/08)
	/** ｺｰﾄﾞｾﾚｸﾄのﾕﾆｰｸｷｰ */
	private String	codeName	;												// 3.8.0.6 (2005/09/12)
	/** Codeﾘｿｰｽのｻﾌﾞｾｯﾄ */
	private String	codeList	;												// 5.1.9.0 (2010/08/01)
	/** Codeﾘｿｰｽのｻﾌﾞｾｯﾄとなるｸﾞﾙｰﾌﾟ */
	private String	codeGroup	;												// 5.1.9.0 (2010/08/01)
	/** 新しいｺｰﾄﾞｾﾚｸﾄを作成する為の KEY:VAL 文字列 */
	private String	codeKeyVal	;												// 5.6.6.0 (2013/07/05)
	/** BODYに記述した内容で、ﾗﾍﾞﾙを作成します */
	private boolean	useLblBody	;												// 7.4.2.0 (2021/04/30)

	// 4.0.0 (2007/05/30) ｶﾗﾑ状態を動的に変更します。
	/** ｶﾗﾑの表示用ﾚﾝﾃﾞﾗｰ */
	private String	renderer	;
	/** ｶﾗﾑの編集用ｴﾃﾞｨﾀｰ */
	private String	editor		;
	/** ｶﾗﾑのDBﾀｲﾌﾟ */
	private String	dbType		;

	/** 親ｶﾗﾑの指定 */
	private String	eventColumn	;												// 4.3.6.0 (2009/04/01)
	/** ｲﾍﾞﾝﾄValueのSQL分 */
	private String	eventValue	;												// 6.3.3.0 (2015/07/25)
	/** eventColumnで飛ばすURL */
	private String	eventURL	;												// 4.3.6.0 (2009/04/01)
	/** ﾊﾟﾗﾒｰﾀを変更可能にする */
	private String	param		;												// 4.3.6.0 (2009/04/01)
	/** ﾃﾞｰﾀﾍﾞｰｽへ接続する場合のID */
	private String	dbid		;												// 5.7.6.2 (2014/05/16)
	/** ｶﾗﾑに対するﾍﾙﾌﾟﾌｧｲﾙ */
	private String	help		;												// 6.3.2.0 (2015/07/10)

	/** XSS対策 */
	private boolean	xssCheck	= HybsSystem.sysBool( "USE_XSS_CHECK" );		// 5.1.7.0 (2010/06/01)

	/** 短縮ﾗﾍﾞﾙの利用有無 */
	private String	useSLabel	= "auto";										// 5.5.1.0 (2012/04/03)
	/** 日付送り戻し機能の有効／無効 */
	private boolean	useDateFeed	= true;											// 8.1.2.3 (2022/05/20)
	/** 隠しﾌｨｰﾙﾄﾞの作成有無 */
	private boolean	useHidden	;												// 8.3.1.0 (2022/10/14)

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰ
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public ColumnTag() { super(); }		// これも、自動的に呼ばれるが、空のﾒｿｯﾄﾞを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始ﾀｸﾞが見つかったときに処理する doStartTag() を ｵｰﾊﾞｰﾗｲﾄﾞします。
	 *
	 * @og.rev 3.8.0.4 (2005/08/08) 新規追加(BODY部の評価)
	 * @og.rev 5.1.7.0 (2010/06/01) command=RENEWの場合はdefaultValを無視する。
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
	 * @og.rev 6.3.3.0 (2015/07/25) eventValue 追加。
	 * @og.rev 6.3.4.0 (2015/08/01) eventValue のSQL処理は、jsp/common/eventColumnMaker.jsp で行う。
	 * @og.rev 6.4.8.1 (2016/07/02) xssCheckを、doStartTag に移動
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		// 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
		if( useTag() ) {
			// 5.9.10.0 (2016/07/01) doAfterBodyから移動
			useXssCheck( xssCheck );

			// makeTag() より移動
			if( value == null && useRequestValue ) {							// 3.5.4.9 (2004/02/25)
				// 5.1.7.0 (2010/06/01) command=RENEWの場合はdefaultValを無視する。
				final String cmd = pageContext.getRequest().getParameter( "command" );
				if( "RENEW".equals( cmd ) ) {
					value = getRequestValue( name );
				}
				else {
					value = nval( getRequestValue( name ),defaultVal );
				}
			}

			return EVAL_BODY_BUFFERED ;	// Body を評価する。( extends BodyTagSupport 時)
		}
		return SKIP_BODY ;				// Body を評価しない
	}

	/**
	 * Taglibのﾀｸﾞ本体を処理する doAfterBody() を ｵｰﾊﾞｰﾗｲﾄﾞします。
	 *
	 * @og.rev 3.8.0.4 (2005/08/08) 新規追加(BODY部の評価)
	 * @og.rev 5.1.7.0 (2010/06/01) XSS解除対応
	 * @og.rev 6.4.8.1 (2016/07/02) xssCheckを、doStartTag に移動
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {

		body = getBodyString();
		return SKIP_BODY ;
	}

	/**
	 * ﾀｸﾞﾘﾌﾞｵﾌﾞｼﾞｪｸﾄをﾘﾘｰｽします。
	 * ｷｬｯｼｭされて再利用されるので、ﾌｨｰﾙﾄﾞの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) ｶｽﾀﾑﾀｸﾞの release() ﾒｿｯﾄﾞを、追加
	 * @og.rev 2.0.0.8 (2002/10/09) yes/no/false で指定するように変更
	 * @og.rev 3.0.0.0 (2002/12/25) colspan 属性を追加。
	 * @og.rev 3.0.1.4 (2003/03/17) colspan 属性の制御方法を変更
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.4.2 (2003/12/15) nextForm 属性を追加。
	 * @og.rev 3.5.4.9 (2004/02/25) useRequestValue 属性を追加。
	 * @og.rev 3.5.6.0 (2004/06/18) Attributes 経由でなく、直接 DBColumnConfig に設定します。
	 * @og.rev 3.8.0.4 (2005/08/08) body , useBody 属性を追加。
	 * @og.rev 3.8.0.6 (2005/09/12) codeName 属性を追加。
	 * @og.rev 4.1.2.1 (2008/03/13) must , mustAny 属性を自動化します。
	 * @og.rev 5.1.7.0 (2010/06/01) XSS解除対応
	 * @og.rev 5.1.9.0 (2010/08/01) codeList , codeGroup属性を追加。(Codeﾘｿｰｽのｻﾌﾞｾｯﾄとなるｸﾞﾙｰﾌﾟ)
	 * @og.rev 5.2.1.0 (2010/10/01) must , mustAny 属性の処理を、HTMLTagSupport に移します
	 * @og.rev 5.2.3.0 (2010/12/01) useDefaultPropertieVal 属性の初期値をｼｽﾃﾑ定数で定義
	 * @og.rev 5.6.6.0 (2013/07/05) codeKeyVal 属性を追加
	 * @og.rev 5.7.6.2 (2014/05/16) dbid 属性を追加
	 * @og.rev 6.0.4.0 (2014/11/28) addKeyLabel 属性を追加
	 * @og.rev 6.2.0.0 (2015/02/27) rowspan 属性を追加
	 * @og.rev 6.2.0.0 (2015/02/27) viewSize 属性の追加
	 * @og.rev 6.3.2.0 (2015/07/10) help 属性の追加
	 * @og.rev 6.3.3.0 (2015/07/25) eventValue 追加
	 * @og.rev 7.4.2.0 (2021/04/30) useLblBodyをtrueにすると、BODYに記述した内容で、ﾗﾍﾞﾙを作成します
	 * @og.rev 8.1.2.3 (2022/05/20) 日付送り戻し不具合対応(useDateFeed 属性追加)
	 * @og.rev 8.3.1.0 (2022/10/14) 隠しﾌｨｰﾙﾄﾞの作成対応(useHidden 属性追加)
	 * @og.rev 8.4.3.0 (2023/03/31) useDefaultPropertieVal 属性廃止
	 */
	@Override
	protected void release2() {
		super.release2();
		tdFlag			= "yes";												// table形式の <td> ﾀｸﾞを使用するかどうか
		addNoValue		= true;													// 値なしのOptionをSelectに含めるかどうか
		addKeyLabel		= null;													// 6.0.4.0 (2014/11/28) ｷｰ:ﾗﾍﾞﾙ形式で表示するかどうか
		name			= null;
		value			= null;
		writable		= true;													// 書き込み/ﾗﾍﾞﾙを指定する。(true で書き込み)
		defaultVal		= "";													// value がｾｯﾄされていないときの初期値
//		useDefVal		= HybsSystem.sysBool( "USE_DEFAULT_PROPERTIE_VALUE" );	// 5.2.3.0 (2010/12/01) 8.4.3.0 (2023/03/31) Delete
		colspan			= "";
		rowspan			= "";													// 6.2.0.0 (2015/02/27) rowspan 属性を追加
		nextForm		= null;													// 3.5.4.2 (2003/12/15)
		useRequestValue	= true;													// 3.5.4.9 (2004/02/25)
		fieldSize		= null;													// 6.2.0.0 (2015/02/27)
		viewSize		= null;													// 6.2.0.0 (2015/02/27)
		maxlength		= null;													// 3.5.6.0 (2004/06/18)
		body			= null;													// 3.8.0.4 (2005/08/08)
		codeName		= null;													// 3.8.0.6 (2005/09/12)
		codeList		= null;													// 5.1.9.0 (2010/08/01) 新規追加(Codeﾘｿｰｽのｻﾌﾞｾｯﾄ)
		codeGroup		= null;													// 5.1.9.0 (2010/08/01) 新規追加(Codeﾘｿｰｽのｻﾌﾞｾｯﾄとなるｸﾞﾙｰﾌﾟ)
		codeKeyVal		= null;													// 5.6.6.0 (2013/07/05) 新規追加(新しいｺｰﾄﾞｾﾚｸﾄを作成する為の KEY:VAL 文字列)
		useLblBody		= false;												// 7.4.2.0 (2021/04/30) BODYに記述した内容で、ﾗﾍﾞﾙを作成します
		renderer		= null;
		editor			= null;
		dbType			= null;
		eventColumn		= null;													// 4.3.6.0 (2009/04/01)
		eventValue		= null;													// 6.3.3.0 (2015/07/25)
		eventURL		= null;													// 4.3.6.0 (2009/04/01)
		param			= null;													// 4.3.6.0 (2009/04/01)
		dbid			= null;													// 5.7.6.2 (2014/05/16)
		help			= null;													// 6.3.2.0 (2015/07/10)
		xssCheck		= HybsSystem.sysBool( "USE_XSS_CHECK" );				// 5.1.7.0 (2010/06/01) XSS解除対応
		useSLabel		= "auto";												// 5.5.1.0 (2012/04/03)
		useDateFeed		= true;													// 8.1.2.3 (2022/05/20)
		useHidden		= false;												// 8.3.1.0 (2022/10/14)
	}

	/**
	 * 検索条件の入力用のﾃｷｽﾄﾌｨｰﾙﾄﾞを作成します。
	 *
	 * ﾃｰﾌﾞﾙﾀｸﾞによりﾌｨｰﾙﾄﾞは分割されます。
	 * 使用時は、ﾃｰﾌﾞﾙﾀｸﾞを前後に使用して下さい。
	 *
	 * @og.rev 2.0.0.8 (2002/10/09) yes/no/false で指定するように変更
	 * @og.rev 2.0.1.0 (2002/10/10) ﾗﾍﾞﾙとﾌｨｰﾙﾄﾞのｾﾊﾟﾚｰﾀｰとして、ｺﾛﾝ(：)を使用するかどうかを指定できる
	 * @og.rev 3.0.0.0 (2002/12/25) colspan 属性を追加。
	 * @og.rev 3.0.1.4 (2003/03/17) colspan 属性の制御方法を変更
	 * @og.rev 3.5.4.9 (2004/02/25) value が null の時の 値の設定方法
	 * @og.rev 3.8.0.4 (2005/08/08) body 属性を追加。
	 * @og.rev 3.8.7.0 (2006/12/15) ｶﾗﾑﾘｿｰｽで指定したｶﾗﾑﾗﾍﾞﾙを反映させる。
	 * @og.rev 4.0.0.0 (2005/11/30) ｶﾗﾑﾛｰﾙを考慮
	 * @og.rev 4.1.2.1 (2008/03/13) must , mustAny 属性を自動化します。
	 * @og.rev 4.3.7.1 (2009/06/08) id=labelのclass化
	 * @og.rev 5.2.1.0 (2010/10/01) must , mustAny 属性の処理を、HTMLTagSupport に移します。
	 * @og.rev 5.6.2.2 (2013/03/15) 自動must処理の出力位置を変更
	 * @og.rev 6.2.0.0 (2015/02/27) rowspan 属性を追加
	 * @og.rev 6.2.0.0 (2015/02/27) aimai 属性を追加。
	 * @og.rev 6.2.4.0 (2015/05/15) aimai 属性は、BODYのincludeより前に追加。
	 * @og.rev 6.3.2.0 (2015/07/10) help 属性の追加
	 * @og.rev 7.4.2.0 (2021/04/30) useLblBodyをtrueにすると、BODYに記述した内容で、ﾗﾍﾞﾙを作成します
	 * @og.rev 8.3.1.0 (2022/10/14) 隠しﾌｨｰﾙﾄﾞの作成対応(useHidden 属性追加)
	 * @og.rev 8.4.3.0 (2023/03/31) defaultVal の設定値の取り扱い(変更あり)
	 *
	 * @return	入力用のﾃｷｽﾄﾌｨｰﾙﾄﾞﾀｸﾞ
	 * @og.rtnNotNull
	 */
	@Override
	protected String makeTag() {
		// 3.8.7.0 (2006/12/15) ｶﾗﾑﾘｿｰｽで指定したｶﾗﾑﾗﾍﾞﾙを反映させる。
		final DBColumn dbColumn = getDBColumn( name );
		String label = getLongLabel() ;
		if( label == null ) { label = dbColumn.getLongLabel() ; }

		final String tag = getInsertTag( dbColumn,value );

		// 4.0.0 (2005/11/30) ｶﾗﾑﾛｰﾙを考慮
		// ｱｸｾｽ不可の場合は、ｾﾞﾛ文字列を返す。
		if( tag == null ) { return ""; }

		// 3.8.0.4 (2005/08/08) BODY を使用する場合は、ｶﾗﾑﾌｨｰﾙﾄﾞの直後に挿入する。
		// 6.2.4.0 (2015/05/15) aimai 属性は、BODYのincludeより前に追加。
		if( body == null ) { body = ""; }
		// 7.4.2.0 (2021/04/30) useLblBodyをtrueにすると、BODYに記述した内容で、ﾗﾍﾞﾙを作成します
		else if( useLblBody ) { label = body; body = ""; }

		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE );
		if( "yes".equals( tdFlag ) ) {
			// 4.3.7.1 (2009/06/08) id=labelのclass化
			// rtn.append( "<td id=\"label\">" );
			rtn.append( "<td class=\"label\"" )									// 6.2.0.0 (2015/02/27)
				.append( rowspan ).append( '>' )								// 6.2.0.0 (2015/02/27)
				.append( label )
				.append( CLM )
				.append( "</td><td" )
				.append( colspan )
				.append( rowspan )												// 6.2.0.0 (2015/02/27)
				.append( '>' )													// 6.0.2.5 (2014/10/31) char を append する。
				.append( tag )
				.append( makeHelpLink( help ) )									// 6.3.2.0 (2015/07/10) help 属性を追加
				.append( makeAimaiPicker( name ) )								// 6.2.0.0 (2015/02/27) aimai 属性を追加
				.append( makeMustHidden( name ) )								// 6.2.0.0 (2015/02/27) , 5.6.2.2 (2013/03/15)
				.append( body )													// 6.2.4.0 (2015/05/15)
				.append( "</td>" );
		}
		else if( "no".equals( tdFlag ) ) {
			// 4.3.7.1 (2009/06/08) id=labelのclass化
			rtn.append( "<span class=\"label\">" )
				.append( label )
				.append( CLM )
				.append( "</span>" )
				.append( tag )
				.append( makeHelpLink( help ) )									// 6.3.2.0 (2015/07/10) help 属性を追加
				.append( makeAimaiPicker( name ) )								// 6.2.0.0 (2015/02/27) aimai 属性を追加
				.append( makeMustHidden( name ) )								// 6.2.0.0 (2015/02/27) , 5.6.2.2 (2013/03/15)
				.append( body );												// 6.2.4.0 (2015/05/15)
		}
		// "false".equals( tdFlag ) の条件
		else {
			rtn.append( tag )
				.append( makeHelpLink( help ) )									// 6.3.2.0 (2015/07/10) help 属性を追加
				.append( makeAimaiPicker( name ) )								// 6.2.0.0 (2015/02/27) aimai 属性を追加
				.append( makeMustHidden( name ) )								// 6.2.0.0 (2015/02/27) , 5.6.2.2 (2013/03/15)
				.append( body );												// 6.2.4.0 (2015/05/15)
		}

		// 8.3.1.0 (2022/10/14) 隠しﾌｨｰﾙﾄﾞの作成対応
		if( useHidden ) {
			final String disabled = get( "disabled" );
			if( !writable || "disabled".equalsIgnoreCase( disabled ) ) {
				// 8.4.3.0 (2023/03/31) ｶﾗﾑﾘｿｰｽの初期値を使わない時に、"null"を設定すると、ｾﾞﾛ文字列と置き換えます。(暫定対応)
				if( "null".equalsIgnoreCase( value ) ) { value = ""; }
				rtn.append( XHTMLTag.hidden( name,value ) );
			}
			else {
				final String errMsg = "useHidden は、writableが false 又は、disabledが disabled のときしか使えません。" + CR
										+ "\t name=[" + name + "] , writable=[" + writable + "] , disabled=[" + disabled + "]";
				throw new HybsSystemException( errMsg );
			}
		}

		return rtn.toString() ;
	}

	/**
	 * ｶﾗﾑがselectﾀｸﾞの場合に、値なしのｵﾌﾟｼｮﾝ文字列を挿入します。
	 *
	 * addNoValue 値が True の場合のみ、挿入します。
	 *
	 * @og.rev 2.1.0.0 (2002/10/11) useDefaultPropertieVal 属性を追加。
	 * @og.rev 3.5.4.2 (2003/12/15) 入力ｶｰｿﾙを自動的に次のﾌｨｰﾙﾄﾞへ移動する機能を追加する。
	 * @og.rev 3.5.5.5 (2004/04/23) DBColumn の size と maxlength の 意味を変更
	 * @og.rev 3.5.5.7 (2004/05/10) DBColumn.getOption( String ) 廃止
	 * @og.rev 3.5.6.0 (2004/06/18) Attributes 経由でなく、直接 DBColumnConfig に設定します。
	 * @og.rev 3.8.0.6 (2005/09/12) codeName 属性を追加。
	 * @og.rev 3.8.7.0 (2006/12/15) ｶﾗﾑﾘｿｰｽで指定したｶﾗﾑﾗﾍﾞﾙを反映させる。
	 * @og.rev 4.0.0.0 (2005/01/31) getFieldSize ﾒｿｯﾄﾞを XHTMLTag から DBColumn へ移動
	 * @og.rev 4.0.0.0 (2005/11/30) ｶﾗﾑﾛｰﾙを考慮。ｱｸｾｽ不可の場合は、null を返す。
	 * @og.rev 4.3.0.0 (2008/07/04) ﾛｰﾙﾓｰﾄﾞﾏﾙﾁ対応
	 * @og.rev 4.3.7.2 (2009/06/15) 不要なIDが出力されるﾊﾞｸﾞを修正
	 * @og.rev 5.1.7.0 (2010/06/01) 動的ﾌﾟﾙﾀﾞｳﾝ実装見直し
	 * @og.rev 5.1.7.0 (2010/06/01) DBColumnの{&#064;XXXX}変数対応
	 * @og.rev 5.1.9.0 (2010/08/01) codeList ,codeGroup 属性を追加(Codeﾘｿｰｽのｻﾌﾞｾｯﾄとなるｸﾞﾙｰﾌﾟ)
	 * @og.rev 5.5.4.0 (2012/07/02) eventColumnの{&#064;USER.XXXX}対応
	 * @og.rev 5.6.6.0 (2013/07/05) codeKeyVal 属性を追加
	 * @og.rev 5.7.6.2 (2014/05/16) dbid 属性を追加、IEのHTML5機能が無効の場合の処理
	 * @og.rev 5.7.6.2 (2014/05/16) IEのHTML5機能が無効の場合、list 属性は削除します。
	 * @og.rev 5.7.7.1 (2014/06/13) codeName 設定時に、renderer,editor が null の場合のみ、MENUに設定
	 * @og.rev 6.0.4.0 (2014/11/28) addKeyLabel 属性を追加
	 * @og.rev 6.2.0.0 (2015/02/27) viewSize 属性の追加
	 * @og.rev 6.2.3.0 (2015/05/01) maxlength 変数は、0 の場合は、無制限になります。
	 * @og.rev 6.3.1.0 (2015/06/28) 個人ﾘｿｰｽのUSE_ADD_KEY_LABELで初期値設定
	 * @og.rev 6.3.3.0 (2015/07/25) eventValue 追加
	 * @og.rev 5.9.18.1 (2017/03/24) editorにDATALIST_Rの追加対応
	 * @og.rev 6.7.7.2 (2017/04/14) (注意点3)現時点では、INMENU,INDBMENUについては、ｲﾍﾞﾝﾄｶﾗﾑが正常に動作しません。
	 * @og.rev 8.4.3.0 (2023/03/31) useDefaultPropertieVal 属性廃止
	 * @og.rev 8.4.3.0 (2023/03/31) defaultVal の設定値の取り扱い(変更あり)
	 * @og.rev 8.5.2.0 (2023/07/14) 一部の機能廃止による修正(問合・ﾄﾗﾌﾞﾙ 0200010980)
	 *
	 * @param	clm		DBColumnｶﾗﾑｵﾌﾞｼﾞｪｸﾄ
	 * @param	value	設定値
	 *
	 * @return	入力用のｶﾗﾑﾀｸﾞ
	 */
	private String getInsertTag( final DBColumn clm, final String value ) {

		// 4.3.0.0 (2008/07/04) ﾛｰﾙﾓｰﾄﾞﾏﾙﾁ対応
		final String roles = get( "roles" );
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		final RoleMode roleMode = roles == null
									? clm.getRoleMode() : RoleMode.newInstance( roles );

		final byte byteMode = getUser().getAccessBitMode( roleMode ) ;

		if( !RoleMode.isAccess( byteMode ) ) { return null; }					// ｱｸｾｽ不可(rw不可)

		// 書き込み権限のﾁｪｯｸ
	//	writable = writable && RoleMode.isWrite( byteMode );

		// 3.5.4.2 (2003/12/15) 入力ｶｰｿﾙを自動的に次のﾌｨｰﾙﾄﾞへ移動する機能
		if( nextForm != null ) {
			// 6.2.3.0 (2015/05/01) maxlength 変数は使用しません。
			final String onKeyup = "nextForm(this,'" + nextForm + "'," + clm.getTotalSize() + ");" ;
			set( "onKeyup",onKeyup );
		}

		// 3.5.6.0 (2004/06/18) 直接 DBColumnConfig に設定します。
		if( fieldSize == null && maxlength != null ) {
			// 4.0.0 (2005/01/31) getFieldSize ﾒｿｯﾄﾞを XHTMLTag から DBColumn へ移動
			final int fsize = clm.getFieldSize( Integer.parseInt( maxlength ) );
			fieldSize = String.valueOf( fsize );
		}

		final DBColumnConfig config = clm.getConfig();

//		// 5.7.6.2 (2014/05/16) IEのHTML5機能が無効の場合の処理 8.5.2.0 (2023/07/14) Delete
//		final String ieHTML5 = (String)getSessionAttribute( HybsSystem.IE_HTML5_KEY );
//		if( "FALSE".equalsIgnoreCase( ieHTML5 ) ) {
//			if( "DATALIST".equalsIgnoreCase( editor ) ||
//				"DATALIST".equalsIgnoreCase( config.getEditor() ) ) {
//					editor = "INDBMENU";
//					// 5.7.6.2 (2014/05/16) list 属性は削除します。
//					set( "list",null );
//			}
//			// 5.9.18.1 (2017/03/24) IEのHTML5機能が無効の場合の処理(DATALIST_R用)
//			if( "DATALIST_R".equalsIgnoreCase( editor ) ||
//				"DATALIST_R".equalsIgnoreCase( config.getEditor() )){
//					editor = "INMENU";
//					set( "list",null );
//			}
//		}

		if( maxlength	!= null ) { config.setMaxlength( maxlength	); }
		if( fieldSize	!= null ) { config.setFieldSize( fieldSize	); }		// 6.2.0.0 (2015/02/27) ﾒｿｯﾄﾞ名変更
		if( viewSize	!= null ) { config.setViewLength(viewSize	); }		// 6.2.0.0 (2015/02/27) 新規追加
		if( renderer	!= null ) { config.setRenderer(	 renderer	); }
		if( editor		!= null ) { config.setEditor(	 editor		); }
		if( dbType		!= null ) { config.setDbType(	 dbType		); }
		if( eventColumn	!= null ) { config.setEventColumn( eventColumn ); }		// 4.3.6.0 (2009/04/01)
		if( eventValue	!= null ) { config.setEventValue(  eventValue  ); }		// 6.3.3.0 (2015/07/25)

		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		if( param		== null ) {
			if( clm.isNeedsParamParse() ){
				config.setRendererParam( getRequestParameter( config.getRendererParam() ) );
				config.setEditorParam(   getRequestParameter( config.getEditorParam() ) );
			}
		}
		else {
			config.setParameter(    getRequestParameter( param ) );				// ここで{@XXXX}変換
			config.setRawParameter( getReservedParameter(param ) );				// 5.5.4.0 (2012/07/02) 予約語対応
		}

		if( dbid		!= null ) { config.setDbid(		dbid		); }		// 5.7.6.2 (2014/05/16)
		if( eventURL	!= null ) { config.setEventURL(	eventURL	); }		// 4.3.6.0 (2009/04/01)
		if( useSLabel	!= null ) { config.setUseSLabel( useSLabel	); }		// 5.5.1.0 (2012/04/03)
		if( !useDateFeed ) { config.setUseDateFeed( useDateFeed ); }			// 8.1.2.3 (2022/05/20)

		// 6.3.1.0 (2015/06/28) 個人ﾘｿｰｽのUSE_ADD_KEY_LABELで初期値設定
		if( addKeyLabel	== null ) { addKeyLabel = nval( sys( "USE_ADD_KEY_LABEL" ) , null ); }	// 4:個人設定可
		if( addKeyLabel	!= null ) { config.setAddKeyLabel( addKeyLabel	); }	// 6.0.4.0 (2014/11/28)

		// 4.0.0 (2005/01/31)
		if( codeName != null ) {
			final CodeData codeData = getResource().getCodeData( codeName ) ;
			config.setCodeData( codeData );
			// 5.7.7.1 (2014/06/13) codeName 設定時に、renderer,editor が null の場合のみ、MENUに設定。
			if( renderer == null ) { config.setRenderer( "MENU" ); }
			if( editor   == null ) { config.setEditor(   "MENU" ); }
		}

		// 5.6.6.0 (2013/07/05) CodeData ｵﾌﾞｼﾞｪｸﾄの取得を表に移動
		// codeName で指定のﾘｿｰｽでも、ｻﾌﾞｾｯﾄが作成できます。
		final CodeData codeData = config.getCodeData() ;
		if( codeData != null ) {
			// 5.1.9.0 (2010/08/01) codeList 属性を追加(Codeﾘｿｰｽのｻﾌﾞｾｯﾄ)
			if( codeList != null ) {
				config.setCodeData( codeData.subsetList( codeList ) );
			}

			// 5.1.9.0 (2010/08/01) codeGroup 属性を追加(Codeﾘｿｰｽのｻﾌﾞｾｯﾄとなるｸﾞﾙｰﾌﾟ)
			if( codeGroup != null ) {
				config.setCodeData( codeData.subsetGroup( codeGroup ) );
			}

			// 5.1.9.0 (2010/08/01) RoleMode 制御を追加(Codeﾘｿｰｽのｻﾌﾞｾｯﾄ)
			if( codeData.useRoleMode() ) {
				config.setCodeData( codeData.subsetRole( getUser().getRoleMode() ) );
			}
		}

		// 5.6.6.0 (2013/07/05) codeKeyVal 新しいｺｰﾄﾞｾﾚｸﾄを作成する。
		// 6.0.4.0 (2014/11/28) KVMENUｴﾃﾞｨﾀｰ で構築する。
		if( codeKeyVal != null ) {
			config.setParameter( codeKeyVal );
			config.setRenderer( "KVMENU" );
			config.setEditor(   "KVMENU" );
		}

		final Attributes attri = getAttributes();
		config.setEditorAttributes( attri );
		final DBColumn column = new DBColumn( config );

		// 6.1.0.0 (2014/12/26) refactoring
//		final String val = useDefVal && ( value == null || value.isEmpty() ) ? column.getDefault() : value ;
		// 8.4.3.0 (2023/03/31) ｶﾗﾑﾘｿｰｽの初期値を使わない時に、"null"を設定すると、ｾﾞﾛ文字列と置き換えます。(暫定対応)
		String val = USE_DEFVAL && ( value == null || value.isEmpty() ) ? column.getDefault() : value ;
		if( "null".equalsIgnoreCase( val ) ) { val = ""; }

		String rtn = null;
		if( writable ) {
			final String editTag = column.getEditorValue( val );
			final StringBuilder tag = new StringBuilder( BUFFER_MIDDLE )
					.append( editTag );
			if( addNoValue && editTag.startsWith( "<select" ) ) {
				final int offset = editTag.indexOf( '>' );
				tag.insert( offset+1,"<option value=\"\" ></option>" );
			}
			rtn = tag.toString();
		}
		else {
			if( attri == null || attri.size() == 0 ) {
				rtn = column.getRendererValue( val );
			}
			else {
				attri.set( "body" , column.getRendererValue( val ) );
				rtn = XHTMLTag.span( attri );
			}
		}

		// 4.3.6.0 (2009/04/01) eventColumn対応 生SQL文のｾｯｼｮﾝ保存
		// 5.1.7.0 (2010/06/01) 動的ﾌﾟﾙﾀﾞｳﾝ実装見直し
		final String evCol = column.getEventColumn();
		if( evCol != null && evCol.length() > 0 ){
//			// 6.7.7.2 (2017/04/14) (注意点3)現時点では、INMENU,INDBMENUについては、ｲﾍﾞﾝﾄｶﾗﾑが正常に動作しません。8.5.2.0 (2023/07/14) Delete
//			if( "INMENU".equals( config.getEditor() ) || "INDBMENU".equals( config.getEditor() ) ) {
//				final String errMsg = "(注意点3)現時点では、INMENU,INDBMENUについては、ｲﾍﾞﾝﾄｶﾗﾑが正常に動作しません。" + CR
//										+ "\t name=[" + name + "] , eventColumn=[" + evCol + "] , editor=[" + config.getEditor() + "]";
//				throw new HybsSystemException( errMsg );
//			}
			addEventColumn( column );
			rtn = column.getEventColumnTag( rtn, val, writable );
		}

		return rtn;
	}

	/**
	 * 【TAG】ｶﾗﾑの名前を返します。
	 *
	 * @og.tag
	 *  &lt;og:column name="PN" /&gt;
	 *
	 * @og.rev 4.2.4.0 (2008/07/01)
	 *
	 * @return	ｶﾗﾑの名前
	 */
	protected String getName() {
		return name;
	}

	/**
	 * 【TAG】ｶﾗﾑの名前を指定します。
	 *
	 * @og.tag
	 *  &lt;og:column name="PN" /&gt;
	 *
	 * @og.rev 3.8.0.9 (2005/10/17) RequestParameter変数({&#064;XXXX})が使用できるように修正
	 *
	 * @param	name	名前
	 */
	public void setName( final String name ) {
		this.name = nval( getRequestParameter( name ),this.name );
	}

	/**
	 * 【TAG】設定値を指定します。
	 *
	 * @og.tag
	 *  &lt;og:column name="PN" value="syokichi" /&gt;
	 *
	 * @param	val	設定値
	 */
	public void setValue( final String val ) {
		value = nval( getRequestParameter( val ),value );
	}

	/**
	 * 【TAG】value属性に値がｾｯﾄされていないとき使用する、初期値を指定します。
	 *
	 * @og.tag
	 * value属性に値がｾｯﾄされていないときに、この初期値を使用します。
	 *
	 * なお、commandがRENEWの場合は、このdefaultValは適用されません。
	 * これは、defaultValの値が埋め込まれた項目が存在する画面に戻った際に、
	 * defaultValの項目の値がnullとなっていた場合に、defaultValの値が復元されると、
	 * 検索結果との不一致が発生するためです。
	 *
	 * 8.4.3.0 (2023/03/31) defaultVal の設定値の取り扱い(変更あり)
	 *   {&#064;XXXX} は、ﾘｸｴｽﾄ変数 ⇒ valueﾀｸﾞｾｯﾄ値 を確認
	 *     値がなければ、""(空文字列) となる。
	 *   通常の固定値は、そのまま使用される(""(空文字列)も有効
	 *   defaultVal属性を使用しない場合は、""(空文字列) となる。
	 *
	 *   ｼｽﾃﾑ定数 USE_DEFAULT_PROPERTIE_VALUE が true の場合は、ｶﾗﾑﾘｿｰｽの初期値が
	 *   使用されます。ｶﾗﾑﾘｿｰｽの初期値より、defaultVal が優先されます。
	 *   ｶﾗﾑﾘｿｰｽの初期値を使わない時に、"null"を設定すると、ｾﾞﾛ文字列と置き換えます。
	 *
	 * @param	dv	初期値
	 */
	public void setDefaultVal( final String dv ) {
		defaultVal = nval( getRequestParameter( dv ),defaultVal );
	}

//	/**
//	 * 【廃止】ｶﾗﾑﾘｿｰｽの初期値を使うかどうか[true/false]を指定します(初期値:false)。
//	 *
//	 * @og.tag
//	 * 初期値がｾｯﾄされていないときに指定します。
//	 * 使う場合はtrue、使わない場合は falseをｾｯﾄします。
//	 * この初期値を、ｼｽﾃﾑ定数の USE_DEFAULT_PROPERTIE_VALUE で設定可能とします。
//	 *
//	 * @og.rev 2.1.0.0 (2002/10/11) useDefaultPropertieVal 属性を追加。
//	 * @og.rev 5.2.3.0 (2010/12/01) 初期値をｼｽﾃﾑ定数で定義(USE_DEFAULT_PROPERTIE_VALUE)
//	 * @og.rev 8.4.3.0 (2023/03/31) useDefaultPropertieVal 属性廃止
//	 *
//	 * @param	dv	ﾌｧｲﾙの初期値 [true:使う/false:使わない]
//	 */
//	public void setUseDefaultPropertieVal( final String dv ) {
//		useDefVal = nval( getRequestParameter( dv ),useDefVal );
//	}

	/**
	 * 【TAG】ﾃｰﾌﾞﾙ形式の &lt;td&gt; ﾀｸﾞを使用するかどうか[yes/no/false]を指定します(初期値:yes)。
	 *
	 * @og.tag
	 * yes(初期値) とすると、ﾗﾍﾞﾙとﾌｨｰﾙﾄﾞをﾃｰﾌﾞﾙのtdﾀｸﾞでそれぞれ記述します。
	 * no とすると、tdﾀｸﾞを使用せず、ﾗﾍﾞﾙとﾌｨｰﾙﾄﾞを表示します。
	 * false は、ﾗﾍﾞﾙを出さずに、ﾌｨｰﾙﾄﾞのみを表示します。
	 *
	 * &lt;td&gt;&lt;og:column name="PN" td="no" /&gt; &lt;/td&gt;
	 *
	 * @og.rev 2.0.0.8 (2002/10/09) yes/no/false で指定するように変更
	 * @og.rev 2.1.2.3 (2002/12/02) errMsg の文字列の修正漏れを訂正
	 * @og.rev 6.3.4.0 (2015/08/01) Arrays.toString から String.join に置き換え。
	 * @og.rev 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	 *
	 * @param	flag	TDﾀｸﾞ使用 [yes:ﾀｸﾞを使用/no:ﾗﾍﾞﾙとﾌｨｰﾙﾄﾞ/false:ﾌｨｰﾙﾄﾞのみ]
	 */
	public void setTd( final String flag ) {
		final String td = nval( getRequestParameter( flag ),tdFlag );

		if( check( td , TD_SET ) ) {
			tdFlag = td;
		}
		else {
			final String errMsg = "Error in SelectTag [td]: "
						 + td + "  in [" + String.join( ", " , TD_SET ) + "] only used.";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】ｾﾚｸﾄﾒﾆｭｰの場合、値なしのOptionを含めるかどうか[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * ｶﾗﾑがﾌﾟﾙﾀﾞｳﾝﾒﾆｭｰ形式の場合、選択肢には通常ﾃﾞｰﾀﾍﾞｰｽ登録できる候補が
	 * 並びます。しかし、検索時には、候補を絞りたくない(指定しない)場合があり、
	 * その様な場合には、値なしのOptionを含める(true)ことで、対応可能です。
	 * 初期値:true(値なしのOptionを含める)なので、検索画面に利用する場合は、そのまま使用できます。
	 *
	 * &lt;og:column name="CDC" addNoValue="false" /&gt;
	 *
	 * @param	flag	値なしのOptionの有無 [true:含める/それ以外:含めない]
	 */
	public void setAddNoValue( final String flag ) {
		addNoValue = nval( getRequestParameter( flag ),addNoValue );
	}

	/**
	 * 【TAG】ｾﾚｸﾄﾒﾆｭｰの場合、ｷｰ:ﾗﾍﾞﾙ形式で表示するかどうか[true/false/null]を指定します(初期値:null)。
	 *
	 * @og.tag
	 * ｶﾗﾑがﾌﾟﾙﾀﾞｳﾝﾒﾆｭｰ形式の場合、ｷｰ:ﾗﾍﾞﾙ形式で表示するか、ﾗﾍﾞﾙだけで表示するかを
	 * 選択できます。
	 * true を指定すると、｢強制的に｣ｷｰ:ﾗﾍﾞﾙ形式で表示します。
	 * false の場合は、｢強制的に｣ﾗﾍﾞﾙのみで表示されます。
	 * 初期値の null の場合、ｺｰﾄﾞﾘｿｰｽや、SQL文に準拠した指定通りの形式で表示します。
	 *
	 * &lt;og:column name="CDC" addKeyLabel="true" /&gt;
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) ｷｰ:ﾗﾍﾞﾙ形式で表示するかどうか。新規追加
	 *
	 * @param	flag	ｷｰ:ﾗﾍﾞﾙ形式表示 [true:ｷｰ:ﾗﾍﾞﾙ形式/false:ﾗﾍﾞﾙのみ/null:指定通り]
	 */
	public void setAddKeyLabel( final String flag ) {
		addKeyLabel = nval( getRequestParameter( flag ),addKeyLabel );
	}

	/**
	 * 【TAG】書き込みﾀｲﾌﾟかどうか[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * 初期値は、書き込みﾀｲﾌﾟ("true") です。
	 * &lt;og:column name="PN" writable="false" /&gt;
	 *
	 * @param	flag	書き込み許可 [true:許可/false:不可]
	 */
	public void setWritable( final String flag ) {
		writable = nval( getRequestParameter( flag ),writable );
	}

	/**
	 * 【HTML】最大入力文字数を指定します(入力可能文字数で、ﾊﾞｲﾄ数ではありません)。
	 *
	 * @og.tag
	 *  指定しない場合は、ｶﾗﾑ定義に準拠します。
	 *  また、値を 0 に設定した場合、無制限になります。
	 *  &lt;og:column name="PN" maxlength="20" /&gt;
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 最大入力制限数に応じたｻｲｽﾞ自動生成を入れます。
	 * @og.rev 3.5.4.5 (2004/01/23) CSSで処理する場合のｸﾗｽ属性は、maxlength が ５ 以下の場合とする。
	 * @og.rev 3.5.6.0 (2004/06/18) Attributes 経由でなく、直接 DBColumnConfig に設定します。
	 * @og.rev 6.2.0.0 (2015/02/27) 小さなｶﾗﾑのｻｲｽﾞ指定は、すでに廃止 ("S0" + 桁数(1～5))
	 * @og.rev 6.2.3.0 (2015/05/01) maxlength 変数は、0 の場合は、無制限になります。
	 *
	 * @param	maxlen	最大入力文字数
	 */
	public void setMaxlength( final String maxlen ) {
		maxlength = nval( getRequestParameter( maxlen ),null );

		// 6.2.3.0 (2015/05/01) maxlength 変数は、0 の場合は、無制限になります。
		if( "0".equals( maxlength ) ) { set( "NO_MAXLEN", "true" ); }
	}

	/**
	 * 【TAG】ｶﾗﾑの入力枠ｻｲｽﾞ(横幅)を文字数でｾｯﾄします(入力文字制限ではありません)。
	 *
	 * @og.tag
	 *  &lt;og:column name="PN" size="20" /&gt;
	 *
	 * @og.rev 3.5.6.0 (2004/06/18) Attributes 経由でなく、直接 DBColumnConfig に設定します。
	 *
	 * @param	size	入力枠ｻｲｽﾞ(横幅)
	 */
	public void setSize( final String size ) {
		fieldSize = nval( getRequestParameter( size ),null );
	}

	/**
	 * 【TAG】ｶﾗﾑの表示ｻｲｽﾞ(横幅)を文字数でｾｯﾄします。
	 *
	 * @og.tag
	 * 入力枠ではなく、表示する時の文字数制限に使われます。
	 * maxLength は、DB上の桁数(ﾊﾞｲﾄ数)ですが、viewSize は、文字数です。
	 * SLABEL の簡易ﾊﾞｲﾄ数計算(全角2Byte)でもなく、文字数計算になります。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) viewSize 属性の追加
	 *
	 * @param	len	表示ｻｲｽﾞ(横幅)
	 */
	public void setViewSize( final String len ) {
		viewSize = nval( getRequestParameter( len ),null );
	}

	/**
	 * 【TAG】ﾌｨｰﾙﾄﾞ部分の colspan を指定します。
	 *
	 * @og.tag
	 * 通常は、ﾗﾍﾞﾙとﾌｨｰﾙﾄﾞが、別のtdで囲われます。しかし、場合によっては、
	 * ﾌｨｰﾙﾄﾞ部の長いｶﾗﾑがある場合、上下のｶﾗﾑとの位置関係がおかしくなり、
	 * ﾗﾍﾞﾙとﾌｨｰﾙﾄﾞが離れて表示されるｹｰｽがあります。
	 * この場合、長いﾌｨｰﾙﾄﾞ部を持つｶﾗﾑに、colspan 属性を指定すると、
	 * その分のｶﾗﾑを割り当てることにより、上下のｶﾗﾑとの位置関係を
	 * 調節することが可能になります。
	 * 通常は 3 を指定して、tdが、ﾌｨｰﾙﾄﾞ、ﾗﾍﾞﾙ、ﾌｨｰﾙﾄﾞの
	 * 組み合わせ部分に、長いﾌｨｰﾙﾄﾞが適用されるように使用します。
	 *
	 * &lt;og:column name="PN" colspan="3" /&gt;
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) colspan 属性を追加。
	 * @og.rev 3.0.1.4 (2003/03/17) colspan 属性の制御方法を変更
	 *
	 * @param	sp	ﾌｨｰﾙﾄﾞ部分のcolspan
	 */
	public void setColspan( final String sp ) {
		colspan = nval( getRequestParameter( sp ),colspan );
		if( colspan.length() > 0 ) {
			colspan = " colspan=\"" + colspan + "\" ";
		}
	}

	/**
	 * 【TAG】ﾗﾍﾞﾙ、ﾌｨｰﾙﾄﾞ共に rowspan を指定します。
	 *
	 * @og.tag
	 * 通常は、ﾗﾍﾞﾙとﾌｨｰﾙﾄﾞが、別のtdで囲われます。rowspan は、
	 * そのどちらの TDﾌｨｰﾙﾄﾞにも、設定します。
	 * 初期値は、rowspan を付けないです。
	 *
	 * &lt;og:column name="PN" rowspan="3" /&gt;
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) rowspan 属性を追加
	 *
	 * @param	sp	ﾗﾍﾞﾙ、ﾌｨｰﾙﾄﾞ部分のrowspan
	 */
	public void setRowspan( final String sp ) {
		rowspan = nval( getRequestParameter( sp ),rowspan );
		if( rowspan.length() > 0 ) {
			rowspan = " rowspan=\"" + rowspan + "\" ";
		}
	}

	/**
	 * 【TAG】入力ｶｰｿﾙを、指定されたname属性のﾌｨｰﾙﾄﾞへ自動的に移動させます。
	 *
	 * @og.tag
	 * JavaScript の onKeyup ｲﾍﾞﾝﾄを利用して、最大桁数まで入力すると、
	 * 次のﾌｨｰﾙﾄﾞに、自動的にｶｰｿﾙが飛ぶように設定します。
	 * 引数は、次に飛ばすﾌｨｰﾙﾄﾞの name 属性です。
	 * 実行するには、JavaScript の nextForm 関数が必要です。(default.js で指定)
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @param	name	次にﾌｫｰｶｽを移すﾀｸﾞのname属性
	 */
	public void setNextForm( final String name ) {
		nextForm = nval( getRequestParameter( name ),nextForm );
	}

	/**
	 * 【TAG】value が null の場合に、name 属性から Request情報 を適用するかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * name 属性に一致するﾘｸｴｽﾄ情報がある場合、value が null の場合や、
	 * 引数を設定していない場合は、この設定値に応じて、ﾘｸｴｽﾄ情報の値を
	 * 設定します(初期値:true なので、通常はこの動きです)
	 * ごくまれに、ENTRY 系の削除を行う場合、ﾘｸｴｽﾄ情報が残り、削除されずに
	 * 再表示(DB的には削除済み)されるｹｰｽでは、使用しない(false)設定が可能です。
	 *
	 *    useRequestValue |      true(初期値)     |   false
	 *  ------------------+------------------------+------------
	 *  value             |Ａ Ａ Ａ Ａ ＿ ＿ ＿ ＿ |＿ ＿ ＿ ＿
	 *  name(Request情報) |Ｂ Ｂ ＿ ＿ Ｂ Ｂ ＿ ＿ |Ｂ Ｂ ＿ ＿
	 *  defaultVal        |Ｃ ＿ Ｃ ＿ Ｃ ＿ Ｃ ＿ |Ｃ ＿ Ｃ ＿
	 *  ==================+========================+============
	 *  設定値            |Ａ Ａ Ａ Ａ Ｂ Ｂ Ｃ ＿ |Ｃ ＿ Ｃ ＿
	 *
	 * @og.rev 3.5.4.9 (2004/02/25) 新規追加
	 *
	 * @param	flag	Request情報適用 [true:する/false:しない]
	 */
	public void setUseRequestValue( final String flag ) {
		useRequestValue = nval( getRequestParameter( flag ),useRequestValue );
	}

	/**
	 * 【TAG】ｺｰﾄﾞｾﾚｸﾄのﾕﾆｰｸｷｰ(ｺｰﾄﾞ名称)をｾｯﾄします。
	 *
	 * @og.tag
	 * このｷｰを元に、CodeData ｵﾌﾞｼﾞｪｸﾄを構築します。
	 *
	 * @og.rev 3.8.0.6 (2005/09/12) 新規作成
	 *
	 * @param	name	ﾒﾆｭｰのﾕﾆｰｸｷｰ
	 */
	public void setCodeName( final String name ) {
		codeName = nval( getRequestParameter( name ),codeName );
	}

	/**
	 * 【TAG】ｺｰﾄﾞｾﾚｸﾄのｻﾌﾞｾｯﾄとなるｺｰﾄﾞ(ｷｰ)をCSV形式で指定します。
	 *
	 * @og.tag
	 * この値に限定された、ｺｰﾄﾞﾘｿｰｽを、既存の、CodeData の
	 * ｻﾌﾞｾｯﾄとして作成します。
	 * codeName で指定されたﾘｿｰｽでも、ｻﾌﾞｾｯﾄを作成する事ができます。
	 * ここで指定するのは、ｺｰﾄﾞ(ｷｰ)だけです。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規作成
	 *
	 * @param	list	ｺｰﾄﾞｾﾚｸﾄのｻﾌﾞｾｯﾄとなるｺｰﾄﾞ(ｷｰ)
	 */
	public void setCodeList( final String list ) {
		codeList = nval( getRequestParameter( list ),codeList );
	}

	/**
	 * 【TAG】ｺｰﾄﾞｾﾚｸﾄのｻﾌﾞｾｯﾄとなるｸﾞﾙｰﾌﾟを指定します。
	 *
	 * @og.tag
	 * この値に限定された、ｺｰﾄﾞﾘｿｰｽを、既存の、CodeData の
	 * ｻﾌﾞｾｯﾄとして作成します。
	 * codeName で指定されたﾘｿｰｽでも、ｻﾌﾞｾｯﾄを作成する事ができます。
	 * ｸﾞﾙｰﾌﾟとは、optiongroup の事ではなく、ｺｰﾄﾞﾘｿｰｽに関連付けられた
	 * ｸﾞﾙｰﾌﾟを指定する事で、簡易的にｻﾌﾞｾｯﾄを作成します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規作成
	 *
	 * @param	group	ｺｰﾄﾞｾﾚｸﾄのｻﾌﾞｾｯﾄとなるｸﾞﾙｰﾌﾟ
	 */
	public void setCodeGroup( final String group ) {
		codeGroup = nval( getRequestParameter( group ),codeGroup );
	}

	/**
	 * 【TAG】新しいｺｰﾄﾞｾﾚｸﾄを作成する為の｢ｷｰ:ﾗﾍﾞﾙ ｷｰ:ﾗﾍﾞﾙ｣の文字列を指定します。
	 *
	 * @og.tag
	 * 既存の、CodeData には存在しない、新しいｺｰﾄﾞﾘｿｰｽを作成する為の、文字列を指定します。
	 * 文字列は、｢ｷｰ:ﾗﾍﾞﾙ ｷｰ:ﾗﾍﾞﾙ｣形式で、ｽﾍﾟｰｽで分解後、":" でｷｰとﾗﾍﾞﾙに分離します。
	 * ｽﾍﾟｰｽ分解後の文字列に、":" が含まれていない場合は、ｷｰをﾗﾍﾞﾙとして扱います。
	 * また、ﾗﾍﾞﾙ部分は、ﾗﾍﾞﾙﾘｿｰｽを使用して、変換を行います。
	 * 内部的には、CodeData を作成しません。DBColumnｵﾌﾞｼﾞｪｸﾄ内で、直接、Selection_KEYVAL を生成します。
	 * codeName 、codeList、codeGroup などが指定された場合は、そちらが優先されます。
	 * ｢ｷｰ:ﾗﾍﾞﾙ ｷｰ:ﾗﾍﾞﾙ｣で、ﾗﾍﾞﾙ にｽﾍﾟｰｽを含ませる場合は、ﾀﾞﾌﾞﾙｸｫｰﾃｰｼｮﾝで囲ってください。
	 * ｢"ｷｰ:ﾗﾍﾞﾙ" "ｷｰ:ﾗﾍﾞﾙ"｣という感じです。
	 *
	 * ※ 6.0.4.0 (2014/11/28) editor,renderer を、"KVMENU" にして、param に、｢ｷｰ:ﾗﾍﾞﾙ ｷｰ:ﾗﾍﾞﾙ｣
	 *    の文字列を指定してください。近い将来、この属性は、廃止します。
	 * ※ 6.2.6.0 (2015/06/19) 廃止を取りやめます。上記、KVMENU設定と、param でもできますが、それの
	 *    簡易版として、残します。（若干の仕様強化を図ります）
	 *
	 * @og.rev 5.6.6.0 (2013/07/05) 新規作成
	 * @og.rev 6.0.4.0 (2014/11/28) editor,renderer を、"KVMENU" にして、param を使用してください。
	 * @og.rev 6.2.6.0 (2015/06/19) 廃止の取りやめ。なんとなく便利なので残します。
	 *
	 * @param	keyval	KEY:VAL文字列
	 */
	public void setCodeKeyVal( final String keyval ) {
		codeKeyVal = nval( getRequestParameter( keyval ),codeKeyVal );
	}

	/**
	 * 【TAG】BODYに記述した内容で、ﾗﾍﾞﾙを作成します。
	 *
	 * @og.tag
	 * 通常のﾗﾍﾞﾙの代わりに、BODY部に記述した内容を、ﾗﾍﾞﾙとして使用します。
	 * BODY が 未指定の場合は、useLblBody="true" でも、元のﾗﾍﾞﾙを使います。
	 *
	 * @og.rev 7.4.2.0 (2021/04/30) useLblBodyをtrueにすると、BODYに記述した内容で、ﾗﾍﾞﾙを作成します
	 *
	 * @param	flag	KEY:VAL文字列
	 */
	public void setUseLblBody( final String flag ) {
		useLblBody = nval( getRequestParameter( flag ),useLblBody );
	}

	/**
	 * 【TAG】ｶﾗﾑの表示用ﾚﾝﾃﾞﾗｰ({@og.doc03Link renderer Renderer_****})をｾｯﾄします。
	 *
	 * @og.tag
	 * これは、plugin.column 以下の Renderer_**** ｸﾗｽの **** を
	 * 与えます。これらは、CellRenderer ｲﾝﾀｰﾌｪｰｽを継承したｻﾌﾞｸﾗｽです。
	 * 属性ｸﾗｽ定義の {@link org.opengion.hayabusa.db.CellRenderer Renderer} を参照願います。
	 * {@og.doc03Link renderer Renderer_**** ｸﾗｽ}
	 *
	 * @param	rdr	表示用ﾚﾝﾃﾞﾗｰ (Renderer_**** の ****)
	 * @see		org.opengion.hayabusa.db.CellRenderer
	 */
	public void setRenderer ( final String rdr ) {
		renderer = nval( getRequestParameter( rdr ),renderer );
	}

	/**
	 * 【TAG】ｶﾗﾑの編集用ｴﾃﾞｨﾀｰ({@og.doc03Link editor Editor_****})をｾｯﾄします。
	 *
	 * @og.tag
	 * これは、plugin.column 以下の Editor_**** ｸﾗｽの **** を
	 * 与えます。これらは、CellEditor ｲﾝﾀｰﾌｪｰｽを継承したｻﾌﾞｸﾗｽです。
	 * 属性ｸﾗｽ定義の {@link org.opengion.hayabusa.db.CellEditor Editor} を参照願います。
	 * {@og.doc03Link editor Editor_**** ｸﾗｽ}
	 *
	 * @param	ed	編集用ｴﾃﾞｨﾀｰ (Editor_**** の ****)
	 * @see		org.opengion.hayabusa.db.CellEditor
	 */
	public void setEditor ( final String ed ) {
		editor = nval( getRequestParameter( ed ),editor );
	}

	/**
	 * 【TAG】DBﾀｲﾌﾟ({@og.doc03Link dbType DBType_****})をｾｯﾄします。
	 *
	 * @og.tag
	 * これは、org.opengion.hayabusa.db.column 以下の DBType_**** ｸﾗｽの **** を
	 * 与えます。これらは、DBType ｲﾝﾀｰﾌｪｰｽを継承したｻﾌﾞｸﾗｽです。
	 * 属性ｸﾗｽ定義の {@link org.opengion.hayabusa.db.DBType DBType} を参照願います。
	 * {@og.doc03Link dbType DBType_**** ｸﾗｽ}
	 *
	 * @param	dt	ﾊﾟﾗﾒｰﾀ
	 * @see		org.opengion.hayabusa.db.DBType  DBType_**** の ****
	 */
	public void setDbType( final String dt ) {
		dbType = nval( getRequestParameter( dt ),dbType );
	}

	/**
	 * 【TAG】ｲﾍﾞﾝﾄｶﾗﾑ(親ｶﾗﾑ)を設定します。
	 *
	 * @og.tag
	 * 親子関係を持った項目の動的な入れ替えを行う場合に利用します。
	 * このｶﾗﾑを変化させるﾄﾘｶﾞとなるｶﾗﾑの名前をｾｯﾄします。
	 *
	 * この属性がｾｯﾄされた場合、親ｶﾗﾑが変更都度子ｶﾗﾑ(自身のｶﾗﾑ)が
	 * 再表示されます。
	 * 標準で再表示時に受け継がれる属性は以下の通りです。
	 * (1)editor・renderer
	 * (2)writable
	 * (3)addNoValue
	 * (4)class(must・mustAnyや、DBTypeを含みます)
	 * (5)style(optionAttributes等でｾｯﾄしたもの)
	 * (6)size
	 * (7)maxlength
	 * これら以外に特殊な設定を行ったｶﾗﾑに関しては、eventURL属性でｶｽﾀﾑの
	 * 置き換え用JSPを作成し、個別に対応して下さい。
	 *
	 * また、置き換えの定義は、param属性で定義します。
	 * param属性に定義されたSQL文の結果により、各種ｴﾃﾞｨﾀｰ、ﾚﾝﾃﾞﾗｰによって、以下の
	 * ように置き換えが行われます。
	 *
	 * (1)DBMENUの場合
	 *   paramに指定されたSQL文により動的にﾌﾟﾙﾀﾞｳﾝを置き変えます。
	 *   SQL文の{&#064;XXXX}の記述により、その時点での親ｶﾗﾑの値を参照することができます。
	 *
	 *		例) SELECT A.CODE,B.LNAME FROM GEA04 A,GEA08 B
	 *			WHERE A.CLM||'.'||A.CODE = B.CLM
	 *			AND A.SYSTEM_ID = B.SYSTEM_ID
	 *			AND A.CLM like 'KBSAKU%'
	 *			AND B.LANG like '{&#064;LANG}' ← name=LANGを持つ項目のその時点での値に変換される
	 *			AND B.SYSTEM_ID = '**'
	 *
	 * (2)DBMENU以外の場合
	 *   paramに指定されたSQL文により値(value値)を置き換えたｶﾗﾑを作成し、置き換えます。
	 *   DBMENUの場合と同様、SQL文の{&#064;XXXX}の記述により、その時点での親ｶﾗﾑの値を参照することができます。
	 *
	 *		例) SELECT LABEL_NAME FROM GF41
	 *			WHERE SYSTEM_ID='{&#064;SYS.SYSTEM_ID}' AND CLM='{&#064;CLM}' AND LANG='{&#064;LANG}'
	 *
	 * なお、いずれの場合も、{&#064;XXXX}の部分については、$1,$2...$9の形式でも記述することができます。
	 * この場合、$1は1番目のeventColumnのｶﾗﾑ、$2は2番目のeventColumnのｶﾗﾑと指します。
	 *
	 * (注意点1)ｲﾍﾞﾝﾄｶﾗﾑによるｲﾍﾞﾝﾄは、画面表示時に1度実行されます。
	 *          但し、eventColumn属性で、ｶﾗﾑ名の先頭に'_'をつけた場合、
	 *          または親ｶﾗﾑの値がNULLの場合は実行されません。
	 *
	 * (注意点2)SQL文などのﾊﾟﾗﾒｰﾀｰ中に記述された{&#064;XXXX}の値の変換はJavaScriptにより行われるため、
	 *          値が取得できる項目は、"画面に書き込み可能状態で表示されているｶﾗﾑ"に限ります。
	 *          ﾃｰﾌﾞﾙﾓﾃﾞﾙ内の変数や表示のみの値については、ここでは利用することはできません。
	 *          {&#064;USER.XXXX}等の予約語に関しては、5.5.4.0以降では処理されます。
	 *
	 * <del>(注意点3)現時点では、INMENU,INDBMENUについては、入れ替え後のJavaScriptが正常に動作しないため、
	 *          ｲﾍﾞﾝﾄｶﾗﾑが正常に動作しません。</del>
	 *
	 * (注意点4) 6.8.2.4 (2017/11/20) ｲﾍﾞﾝﾄｶﾗﾑの、変数($1,$2...$9)は、先頭から９つまでしか使えません。
	 *          10個目以降の変数は無視されますので、必要であれば、9個までの位置に、ｶﾗﾑを記述してください。
	 *
	 * (注意点5) 7.3.0.0 (2021/01/06) ｲﾍﾞﾝﾄｶﾗﾑを定義すると、子ｶﾗﾑの onChange ｲﾍﾞﾝﾄが発火されます。
	 *          これは、(注意点1)のｶﾗﾑ名の先頭に'_'をつけた場合でも同様です。
	 *          子ｶﾗﾑにonChangeを記述する場合は、ご注意ください。
	 *
	 * @og.rev 4.3.6.0 (2009/04/01)
	 * @og.rev 5.1.7.0 (2010/06/01) ｺﾒﾝﾄ大幅修正
	 * @param	col	親ｶﾗﾑ
	 */
	public void setEventColumn( final String col ) {
		eventColumn = nval( getRequestParameter( col ), eventColumn );
	}

	/**
	 * 【TAG】eventColumn使用時の値を、SQL文から求めるためのqueryをｾｯﾄします。
	 *
	 * @og.tag
	 * eventColumn利用時は{&#064;XXXX}はﾘｸｴｽﾄﾊﾟﾗﾒｰﾀではなく
	 * xxxｶﾗﾑの実際の値が利用されます。
	 *
	 * @og.rev 6.3.3.0 (2015/07/25) eventValue 追加
	 * @param	val	ﾊﾟﾗﾒｰﾀ
	 */
	public void setEventValue( final String val ) {
		eventValue = nval( getReservedParameter( val ), eventValue );	// ここでは{@XXXX}変換をしない
	}

	/**
	 * 【TAG】ｲﾍﾞﾝﾄｶﾗﾑ指定時に呼び出すURLを指定します。
	 *
	 * @og.tag
	 * 初期値はｼｽﾃﾑ定数のEVENT_COLUMN_URLです。
	 * ｲﾍﾞﾝﾄｶﾗﾑ指定時に部品を作成するJSPを指定します。
	 * (設定例：eventURL="makeColumn_custom.jsp")
	 *
	 * 置き換え対象のｶﾗﾑに特殊な設定がされており、標準のｶﾗﾑ置き換えでは対応できない場合に、
	 * このURLを指定することで、個別に置き換え方法を定義する必要があります。
	 *
	 * なお、個別に置き換え方法を定義したJSPを作成する場合は、標準(jsp/common/eventColumnMaker.jsp)を
	 * 参考にして作成して下さい。
	 *
	 * eventColumnから利用するJSPで利用可能なﾘｸｴｽﾄ変数
	 * {&#064;EC_CNAME}		子ｶﾗﾑ名(一覧表示の場合でも行番号__nは付加されません)
	 * {&#064;EC_RENDERER}	子ｶﾗﾑのﾚﾝﾃﾞﾗｰ
	 * {&#064;EC_EDITOR}	子ｶﾗﾑのｴﾃﾞｨﾀｰ
	 * {&#064;EC_SQL}		子ｶﾗﾑのﾌﾟﾙﾀﾞｳﾝ、または値を出力するためのSQL文({&#064;XXXX}をｶﾗﾑのvalue値で置き換えたもの)
	 * {&#064;EC_VALSQL}	子ｶﾗﾑの値を出力するためのSQL文({&#064;XXXX}をｶﾗﾑのvalue値で置き換えたもの)
	 * {&#064;EC_WRITABLE}	子ｶﾗﾑが編集可能などうか
	 * {&#064;EC_CVALUE}	子ｶﾗﾑの現在の値
	 * {&#064;EC_NOVALUE}	子ｶﾗﾑの最上部Option値が空ならtrue。その他false(addNoValue属性値と同じです)
	 * {&#064;EC_CLASS}		子ｶﾗﾑのCLASS属性
	 * {&#064;EC_SIZE}		子ｶﾗﾑのSIZE属性
	 * {&#064;EC_MAXLENGTH}	子ｶﾗﾑのMAXLENGTH属性
	 * {&#064;EC_OPTATTRS}	子ｶﾗﾑのその他OPTION属性
	 * {&#064;EC_PNAMEE}	親ｶﾗﾑ名(一覧表示の場合でも行番号__nは付加されません)
	 * {&#064;EC_PVALUE}	親ｶﾗﾑの現在の値
	 *
	 * @og.rev 4.3.6.0 (2009/04/01)
	 * @og.rev 5.1.7.0 (2010/06/01) ｺﾒﾝﾄ大幅修正
	 * @param	url	ｲﾍﾞﾝﾄURL
	 */
	public void setEventURL( final String url ) {
		eventURL = nval( getRequestParameter( url ), eventURL );
	}

	/**
	 * 【TAG】ﾊﾟﾗﾒｰﾀをｾｯﾄします。
	 *
	 * @og.tag
	 * EditorParam等のﾊﾟﾗﾒｰﾀを一括でｾｯﾄします。
	 * eventColumn利用時は{&#064;XXXX}はﾘｸｴｽﾄﾊﾟﾗﾒｰﾀではなく
	 * xxxｶﾗﾑの実際の値が利用されます。
	 *
	 * @og.rev 4.3.6.0 (2009/04/01)
	 * @param	prm	ﾊﾟﾗﾒｰﾀ
	 */
	public void setParam( final String prm ) {
		param = nval( prm, param ); // ここでは{@XXXX}変換をしない
	}

	/**
	 * 【TAG】ﾃﾞｰﾀﾍﾞｰｽへ接続する場合のID(初期値:null)。
	 *
	 * @og.tag
	 * ﾃﾞｰﾀﾍﾞｰｽ接続を使用するｴﾃﾞｨﾀｰ、ﾚﾝﾃﾞﾗｰを使用する場合に使用する接続先IDを指定します。
	 * ｶﾗﾑｵﾌﾞｼﾞｪｸﾄは、query処理ではなく、view処理時に適用されるため、ﾃﾞｰﾀﾍﾞｰｽ
	 * 接続先を指定する必要があります。
	 * 初期値は、null(DEFAULT 接続) です。
	 *
	 * @og.rev 5.7.6.2 (2014/05/16) 新規追加
	 *
	 * @param	id	接続先ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}

	/**
	 * 【TAG】ﾘｸｴｽﾄ情報の HTMLTag開始/終了文字(&gt;&lt;) 存在ﾁｪｯｸを実施するかどうか[true/false]を設定します
	 *		(初期値:USE_XSS_CHECK[={@og.value SystemData#USE_XSS_CHECK}])。
	 *
	 * @og.tag
	 * ｸﾛｽｻｲﾄｽｸﾘﾌﾟﾃｨﾝｸﾞ(XSS)対策の一環としてless/greater than signについてのﾁｪｯｸを行います。
	 * (&gt;&lt;) が含まれていたｴﾗｰにする(true)／かﾉｰﾁｪｯｸか(false)を指定します。
	 * (初期値:ｼｽﾃﾑ定数のUSE_XSS_CHECK[={@og.value SystemData#USE_XSS_CHECK}])。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 新規追加
	 *
	 * @param	flag	XSSﾁｪｯｸ [true:する/false:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_XSS_CHECK
	 */
	public void setXssCheck( final String flag ) {
		xssCheck = nval( getRequestParameter( flag ),xssCheck );
	}

	/**
	 * 【TAG】MENU,DBMENUでSLABELを利用するかどうか[auto/true/false]を指定します(初期値:auto)。
	 *
	 * @og.tag
	 * 通常はｾﾚｸﾄﾒﾆｭｰは一覧とそれ以外で短縮ﾗﾍﾞﾙの利用が自動で切り替わります。
	 * 強制的にSLABELを利用させたい場合にこの属性をtrueにｾｯﾄして下さい。
	 * 標準はauto(自動)です。
	 * auto/true/false以外を指定した場合はfalse扱いとします。
	 *
	 * @og.rev 5.5.1.0 (2012/04/03)
	 *
	 * @param	prm	SLABEL利用 [auto:自動/true:利用する/false:利用しない]
	 */
	public void setUseSLabel( final String prm ) {
		useSLabel = nval( getRequestParameter( prm ),useSLabel );
	}

	/**
	 * 【TAG】eventColumn実行後のcallbak関数を指定します。
	 *
	 * @og.tag
	 * eventColumnの動作終了後に実行するCallback関数の指定を行います。
	 * 関数名のみを指定して下さい。
	 *  ex) sampleCallback()を実行する場合はeventCallback="sampleCallback"
	 * 実行する関数の第一引数には自動的に次の要素が入った配列が渡されます
	 *  [0] 変更された子ｶﾗﾑのID属性値
	 *  [1] evnetColumn実行後の子ｶﾗﾑの値
	 *  [2] eventColumn実行前の子ｶﾗﾑの値
	 *
	 * この属性は、optionAttributesへの設定と同様の動作を行います。
	 *
	 * @og.rev 5.5.4.0 (2012/07/02) 新規追加
	 * @og.rev 6.9.8.0 (2018/05/28) ﾊﾟﾗﾒｰﾀ引数が、未設定の場合の考慮不足
	 *
	 * @param	callback	callbak関数
	 */
	public void setEventCallback( final String callback ) {
		// optionAttributes扱いで登録します。
		final String clbk = getRequestParameter( callback );
		if( StringUtil.isNotNull( clbk ) ){
			add( "optionAttributes", "eventCallback='" + clbk + "'" );
		}

//		// optionAttributes扱いで登録します。
//		if( callback != null && callback.length() > 0 ){
//			add( "optionAttributes", "eventCallback='"+getRequestParameter( callback )+"'" );
//		}
	}

	/**
	 * 【HTML5】表示形式を指定します(初期値:text)。
	 *
	 * @og.tag
	 * ColumnTagは、ｶﾗﾑﾘｿｰｽで作成される為、ﾀｲﾌﾟは固定です。
	 * しかし、HTML5 で追加された形式には、有用なﾀｲﾌﾟも存在する為、
	 * それらへの書き換えをできるようにします。
	 * よって、ここでは、従来型のﾀｲﾌﾟへの書き換えはできません。
	 * (textだけは例外で戻せるようにします)。
	 *
	 * [search/tel/url/email/datetime/date/month/week/time/datetime-local/number/range/color]
	 * が、設定できます。
	 *
	 * HTML5 の機能を有効にするには、ﾈｲﾃｨﾌﾞﾓｰﾄﾞで動作させる必要があります。
//	 *   ① USE_IE7_HEADER = "false" に設定する。   6.9.5.0 (2018/04/23) 廃止(false固定)
//	 *   ② USE_HTML5_HEADER = "true" に設定する。  6.9.5.0 (2018/04/23) 廃止(true固定)
	 *   ③ IEのﾂｰﾙ⇒互換表示設定で、互換表示に追加したWebｻｲﾄから削除する。
	 *   ④ 同上の設定で、ｲﾝﾄﾗｻｲﾄを互換表示で表示するのﾁｪｯｸを外す。
	 * 必要があります。
	 *
	 * &lt;og:input type="text"     /&gt; ﾃｷｽﾄﾎﾞｯｸｽを表示
	 *
	 * HTML5 で追加されたﾀｲﾌﾟ
	 * &lt;og:input type="search"			/&gt; 検索ﾃｷｽﾄの入力欄を作成する
	 * &lt;og:input type="tel"				/&gt; 電話番号の入力欄を作成する
	 * &lt;og:input type="url"				/&gt; URLの入力欄を作成する
	 * &lt;og:input type="email"			/&gt; ﾒｰﾙｱﾄﾞﾚｽの入力欄を作成する
	 * &lt;og:input type="datetime"			/&gt; UTC（協定世界時）による日時の入力欄を作成する
	 * &lt;og:input type="date"				/&gt; 日付の入力欄を作成する
	 * &lt;og:input type="month"			/&gt; 月の入力欄を作成する
	 * &lt;og:input type="week"				/&gt; 週の入力欄を作成する
	 * &lt;og:input type="time"				/&gt; 時間の入力欄を作成する
	 * &lt;og:input type="datetime-local"	/&gt; UTC（協定世界時）によらないﾛｰｶﾙ日時の入力欄を作成する
	 * &lt;og:input type="number"			/&gt; 数値の入力欄を作成する
	 * &lt;og:input type="range"			/&gt; ﾚﾝｼﾞの入力欄を作成する
	 * &lt;og:input type="color"			/&gt; 色の入力欄を作成する
	 *
	 * @og.rev 5.7.1.0 (2013/12/06) HTML5 対応(新規追加)
	 * @og.rev 6.3.4.0 (2015/08/01) Arrays.toString から String.join に置き換え。
	 * @og.rev 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	 *
	 * @param	type	HTML5用 [text/search/tel/url/email/datetime/date/month/week/time/datetime-local/number/range/color]
	 */
	public void setType( final String type ) {
		final String tp = getRequestParameter( type );

		if( check( tp , TYPE_SET ) ) {
			set( "type",tp );
		}
		else {
			final String errMsg = "type 属性は、下記の中から選択してください。type=["
						 + tp + "] in [" + String.join( ", " , TYPE_SET ) + "]";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】このｶﾗﾑに対するﾍﾙﾌﾟﾌｧｲﾙを指定します(初期値:null)。
	 *
	 * @og.tag
	 * 引数の url は、通常 そのｶﾗﾑに関する詳細な説明を行っているﾌｧｲﾙになります。
	 * 簡易的な説明なら、ｶﾗﾑﾘｿｰｽの概要説明に入れれば、自動的に
	 * title属性に入り、ﾏｳｽｵｰﾊﾞｰでﾁﾙﾄ表示されますが、それより少し
	 * 複雑な説明を、html ﾌｧｲﾙや、PDFﾌｧｲﾙで行う事を想定しています。
	 * ﾃﾞﾌｫﾙﾄｱｲｺﾝは、DEFAULT_CLMHELP_ICON です。
	 * "http" や "/" , "../" などで始まる場合は、そのままのURLを使用します。
	 * (初期値:DEFAULT_CLMHELP_ICON[={@og.value SystemData#DEFAULT_CLMHELP_ICON}])。
	 *
	 * @og.rev 6.3.2.0 (2015/07/10) help 属性の追加
	 *
	 * @param	url	helpﾌｧｲﾙのURL(通常は、helpﾌｫﾙﾀﾞ以下のﾌｧｲﾙ名)
	 */
	public void setHelp( final String url ) {
		help = nval( getRequestParameter( url ),help );
	}

	/**
	 * 【TAG】ﾘｱﾙﾀｲﾑﾁｪｯｸを行うかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * ｶﾗﾑ単位でﾘｱﾙﾀｲﾑﾁｪｯｸを行うかどうかを設定をします。
	 * ﾀｸﾞに独自属性としてrealTimeCheck="true/false"を追記します。
	 * falseを設定した場合にﾁｪｯｸが行われなくなります。
	 * 初期値はﾘｱﾙﾀｲﾑﾁｪｯｸを行う（true）です。
	 *
	 * この属性は、optionAttributesへの設定と同様の動作を行います。
	 * 引数の文字列の、true/false 判定を行っていません。そのままの文字列が設定されます。
	 * JavaScript 側では、false 判定のみ行っているので、不正な文字列の場合は、
	 * 初期値(true:ﾘｱﾙﾀｲﾑﾁｪｯｸを行う)が適用されます。
	 * これは、Ver5 とﾛｼﾞｯｸを合わせておくための処置です。
	 *
	 * @og.rev 5.9.32.2 (2018/05/18) 新規追加
	 * @og.rev 6.9.8.0 (2018/05/28) Ver5 とﾛｼﾞｯｸを合わせます。
	 *
	 * @param	flag	ﾘｱﾙﾀｲﾑﾁｪｯｸを行うかどうか [true:行う/false:行わない]
	 */
	public void setUseRealTimeCheck( final String flag ) {
		// optionAttributes扱いで登録します。
		final String rtcf = getRequestParameter( flag );

		if( StringUtil.isNotNull( rtcf ) ){
			add( "optionAttributes", "realTimeChk=\"" + rtcf + "\"" );			// 連結時のスペースは、Attributes クラスで処理
		}

//		// optionAttributes扱いで登録します。
//		final boolean useRtc = nval( getRequestParameter( flag ) , true );
//
//		if( !useRtc ) {		// true 以外の場合のみ、optionAttributes に属性を追加します。
//			add( "optionAttributes", "realTimeChk=\"false\"" );			// 連結時のスペースは、Attributes クラスで処理
//		}
	}

	/**
	 * 【TAG】日付送り戻し機能を有効にするかどうか[true:有効/false:無効]指定します(初期値:true)。
	 *
	 * @og.tag
	 * 編集種別が年月(YM)、年月日(YMD)、年月日(YMD2)のとき、日付送り戻し機能の
	 * 有効(true)/無効(false)を指定します。
	 * ｲﾍﾞﾝﾄｶﾗﾑ(親ｶﾗﾑ)を設定し、param属性に定義したSQL文を実行させたい場合には、
	 * false を指定します。
	 *
	 * @og.rev 8.1.2.3 (2022/05/20) 日付送り戻し不具合対応(useDateFeed 属性追加)
	 *
	 * @param	flag	日付送り戻し機能有無 [true:有効/false:無効]
	 */
	public void setUseDateFeed( final String flag ) {
		useDateFeed = nval( getRequestParameter( flag ),useDateFeed );
	}

	/**
	 * 【TAG】隠しﾌｨｰﾙﾄﾞを作成するかどうか[true/false]指定します (初期値:false)。
	 *
	 * @og.tag
	 * 書き込み不可(writable="false" 又は、disabled)のとき、ﾘｸｴｽﾄ変数を使用したい場合に
	 * true を指定します。
	 * 初期値は、false です。
	 *
	 * @og.rev 8.3.1.0 (2022/10/14) 隠しﾌｨｰﾙﾄﾞの作成対応(useHidden 属性追加)
	 *
	 * @param	flag	隠しﾌｨｰﾙﾄﾞを作成 [true:する/false:しない]
	 */
	public void setUseHidden( final String flag ) {
		useHidden = nval( getRequestParameter( flag ),useHidden );
	}

	/**
	 * ﾍﾙﾌﾟﾌｧｲﾙ へのﾘﾝｸを生成します。
	 *
	 * @og.rev 6.3.2.0 (2015/07/10) help 属性の追加。
	 * @og.rev 6.4.2.0 (2016/01/29) alt属性にtitle属性を追記。
	 * @og.rev 7.0.1.0 (2018/10/15) XHTML → HTML5 対応(空要素の、"／＞" 止めを、"＞" に変更します)。
	 * @og.rev 7.3.2.3 (2021/04/09) ｼｽﾃﾑ定数のICON_DIRを使用します。
	 *
	 * @param	url		helpﾌｧｲﾙのURL(通常は、helpﾌｫﾙﾀﾞ以下のﾌｧｲﾙ名)
	 *
	 * @return	helpﾌｧｲﾙのための aﾀｸﾞ
	 * @see		#setHelp( String )
	 * @og.rtnNotNull
	 */
	private String makeHelpLink( final String url ) {
		if( url != null ) {
			// 7.3.2.3 (2021/04/09) ｼｽﾃﾑ定数のICON_DIRを使用します。
//			final String JSP	= HybsSystem.sys( "JSP" );
			final String icon	= HybsSystem.sys( "JSP_ICON" ) + "/" + HybsSystem.sys( "DEFAULT_CLMHELP_ICON" );

//			final String img = "<img src=\"" + JSP + icon + "\" alt=\"Help\" title=\"Help\" />";	// 6.4.2.0 (2016/01/29)
			final String img = "<img src=\"" + icon + "\" alt=\"Help\" title=\"Help\" >";			// 7.3.2.3 (2021/04/09)

			return XHTMLTag.link(
					new Attributes()
						.set( "href"	, makeUrl( url ) )
						.set( "body"	, img )
						.set( "target"	, "_blank" )
						.set( "class"	, "clmHelplink" )
			);
		}
		return "" ;
	}

	/**
	 * このｵﾌﾞｼﾞｪｸﾄの文字列表現を返します。
	 * 基本的にﾃﾞﾊﾞｯｸﾞ目的に使用します。
	 *
	 * @return このｸﾗｽの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "tdFlag"			,tdFlag			)
				.println( "addNoValue"		,addNoValue		)
				.println( "name"			,name			)
				.println( "value"			,value			)
				.println( "writable"		,writable		)
				.println( "defaultVal"		,defaultVal		)
//				.println( "useDefVal"		,useDefVal		)					// 8.4.3.0 (2023/03/31) Delete
				.println( "colspan"			,colspan		)
				.println( "rowspan"			,rowspan		)
				.println( "nextForm"		,nextForm		)
				.println( "useRequestValue"	,useRequestValue)
				.println( "viewSize"		,viewSize		)
				.println( "fieldSize"		,fieldSize		)					// 6.2.0.0 (2015/02/27)
				.println( "maxlength"		,maxlength		)
				.println( "codeName"		,codeName		)
				.println( "param"			,param			)					// 8.2.1.0 (2022/07/15)
				.println( "body"			,StringUtil.htmlFilter( body ) )
				.println( "Other..."		,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
