/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.db.ResultSetValue;						// 6.0.4.0 (2014/11/28)
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.system.Closer ;
import org.opengion.fukurou.util.ToString;							// 6.1.1.0 (2015/01/17)

import static org.opengion.fukurou.util.StringUtil.nval ;
import static org.opengion.fukurou.system.HybsConst.BR;				// 6.1.0.0 (2014/12/26) refactoring
import static org.opengion.fukurou.system.HybsConst.DB_FETCH_SIZE;	// 6.9.4.1 (2018/04/09)

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.resource.GUIInfo;
import org.opengion.hayabusa.resource.ResourceManager;
import org.opengion.hayabusa.db.DBErrMsg;
import org.opengion.hayabusa.io.HybsFileOperationFactory;			// 8.0.0.1 (2021/10/08)
// import org.opengion.fukurou.model.FileOperation;				// 8.0.0.1 (2021/10/08)

import java.sql.Connection;
import java.sql.Statement;
import java.sql.CallableStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.sql.Array;								// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のﾄﾞﾗｲﾊﾞ) 対応｡oracle.sql.ARRAY の置き換え
import oracle.jdbc.OracleConnection;				// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のﾄﾞﾗｲﾊﾞ) 対応
import oracle.jdbc.OracleTypes;						// CURSOR が残る
import oracle.jdbc.OracleCallableStatement;			// CURSOR が残る

import java.io.File;
import java.io.PrintWriter;
import java.io.FileOutputStream;
import java.io.BufferedOutputStream;				// 6.0.4.0 (2014/11/28)

import java.io.IOException;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;			// 6.0.4.0 (2014/11/28)
import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream;	// 6.0.4.0 (2014/11/28)

import java.util.Map;

/**
 * SELECT文を直接実行して､指定のﾌｧｲﾙに出力するﾀｸﾞです｡
 *
 * 中間の､ﾃﾞｰﾀ(DBTableModel)を作成しないため､余計なﾒﾓﾘを取らず､
 * 高速にﾃﾞｰﾀを抜き出すことが可能です｡
 * 一方､抜き出すﾃﾞｰﾀは生ﾃﾞｰﾀのため､ﾃﾞｰﾀの再利用等､ｼｽﾃﾑ的な
 * 使用を想定しています｡
 * JDBCErrMsg 形式のPL/SQL をｺｰﾙして､その検索結果(ｶｰｿﾙ)を抜く事もできます｡
 *
 * ※ このﾀｸﾞは､Transaction ﾀｸﾞの対象です｡
 *
 * @og.formSample
 * ●形式：&lt;og:directWriteTable filename="[･･･]" ･･･ &gt;SELECT * FROM ZYXX &lt;/og:directWriteTable &gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し､{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:directWriteTable
 *       fileURL            【TAG】保存先ﾃﾞｨﾚｸﾄﾘ名を指定します (初期値:FILE_URL[=filetemp/])
 *       filename           【TAG】ﾌｧｲﾙを作成するときのﾌｧｲﾙ名をｾｯﾄします(初期値:ｼｽﾃﾑﾊﾟﾗﾒｰﾀのFILE_FILENAME)
 *       encode             【TAG】ﾌｧｲﾙを作成するときのﾌｧｲﾙｴﾝｺｰﾃﾞｨﾝｸﾞ名をｾｯﾄします (初期値:FILE_ENCODE[=UnicodeLittle])
 *       fileAppend         【TAG】追加ﾓｰﾄﾞで書き込むかどうか[true/false]を指定します(初期値:false[通常ﾓｰﾄﾞ])
 *       zip                【TAG】結果をﾌｧｲﾙに出力するときに､ZIPで圧縮するかどうか[true/false]を指定します(初期値:false)
 *       zipFilename        【TAG】ZIPﾌｧｲﾙを作成するときのZIPﾌｧｲﾙ名をｾｯﾄします(初期値:filename + ".zip")
 *       separator          【TAG】可変長ﾌｧｲﾙを作成するときの項目区切り文字をｾｯﾄします (初期値:TAB_SEPARATOR)
 *       useHeader          【TAG】ﾍｯﾀﾞｰを書き込むかどうか[true/false]を指定します(初期値:true)
 *       useQuote           【TAG】ﾃﾞｰﾀをﾀﾞﾌﾞﾙｸｵｰﾄで囲うかどうか指定します(初期値:false)
 *       useQuoteEscape     【TAG】ﾃﾞｰﾀ中にﾀﾞﾌﾞﾙｸｵｰﾄ文字が含まれる場合､ｴｽｹｰﾌﾟするかどうか指定します(初期値:true)
 *       useReturnQuote     【TAG】ﾃﾞｰﾀ中に改行ｺｰﾄﾞが含まれる場合､ﾀﾞﾌﾞﾙｸｵｰﾄで囲うかどうか指定します(初期値:true)
 *       replaceFrom        【TAG】置換元文字を指定｡一文字単位で置換します(初期値:null 置換なし)｡
 *       replaceTo          【TAG】置換先文字を指定｡一文字単位で置換します｡
 *       displayMsg         【TAG】検索結果を画面上に表示するﾒｯｾｰｼﾞﾘｿｰｽIDを指定します(初期値:VIEW_DISPLAY_MSG[=])
 *       notfoundMsg        【TAG】検索結果がｾﾞﾛ件の場合に表示するﾒｯｾｰｼﾞﾘｿｰｽIDを指定します(初期値:MSG0077[対象ﾃﾞｰﾀはありませんでした])
 *       fetchSize          【TAG】(通常は使いません)ﾃﾞｰﾀのﾌｪｯﾁｻｲｽﾞを指定します(初期値:DB_FETCH_SIZE[={@og.value org.opengion.fukurou.system.HybsConst#DB_FETCH_SIZE}])
 *       names              【TAG】PL/SQLを利用する場合の引数にｾｯﾄすべき ﾃﾞｰﾀの名称をCSV形式で複数指定します
 *       queryType          【TAG】Query を発行する為のｸﾗｽID(JDBC,JDBCErrMsg)を指定します({@og.doc03Link queryType 初期値:JDBC})
 *       dbid               【TAG】(通常は使いません)検索時のDB接続IDを指定します(初期値:DEFAULT)
 *       useNumber          【TAG】行番号を出力するかどうか(初期値:true)
 *       quotCheck          【TAG】ﾘｸｴｽﾄ情報の ｼﾝｸﾞﾙｸｫｰﾄ(') 存在ﾁｪｯｸを実施するかどうか[true/false]を設定します(初期値:USE_SQL_INJECTION_CHECK)
 *       xssCheck           【TAG】ﾘｸｴｽﾄ情報の HTMLTag開始/終了文字(&gt;&lt;) 存在ﾁｪｯｸを実施するかどうか[true/false]を設定します (初期値:USE_XSS_CHECK[=true])
 *       useTimeView        【TAG】処理時間を表示する TimeView を表示するかどうかを指定します
 *                                      (初期値:VIEW_USE_TIMEBAR[={@og.value SystemData#VIEW_USE_TIMEBAR}])｡
 *       useSLabel          【TAG】7.0.7.0 (2019/12/13) ｴﾗｰﾒｯｾｰｼﾞにSLABELを利用するかどうか[true/false]を指定します(初期値:false)
 *       useLocal           【TAG】ｼｽﾃﾑ定数でｸﾗｳﾄﾞ設定されていても､ｸﾗｳﾄﾞ環境を使用しない場合､trueを指定します(初期値:false) 8.0.1.0 (2021/10/29)
 *       mapObjKey          【TAG】7.0.7.1 (2019/12/24) valueﾀｸﾞのaction=MAPOBJ を使用したﾗﾍﾞﾙ変換を行う場合の､MAPOBJｷｰを指定します｡
 *       caseKey            【TAG】このﾀｸﾞ自体を利用するかどうかの条件ｷｰを指定します(初期値:null)
 *       caseVal            【TAG】このﾀｸﾞ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が､null/ｾﾞﾛ文字列 でない場合(Not Null=NN)は､このﾀｸﾞは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が､null/ｾﾞﾛ文字列 の場合は､このﾀｸﾞは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が､true/TRUE文字列の場合は､このﾀｸﾞは使用されます(初期値:判定しない)
 *       debug              【TAG】ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:directWriteTable&gt;
 *
 * ●使用例
 *     &lt;og:directWriteTable
 *         dbid        = "ORCL"               接続ﾃﾞｰﾀﾍﾞｰｽID(初期値:DEFAULT)
 *         separator   = ","                  ﾌｧｲﾙの区切り文字(初期値:ﾀﾌﾞ)
 *         fileURL     = "{&#064;USER.ID}"    保存先ﾃﾞｨﾚｸﾄﾘ名
 *         filename    = "{&#064;filename}"   保存ﾌｧｲﾙ名
 *         encode      = "UnicodeLittle"      保存ﾌｧｲﾙｴﾝｺｰﾄﾞ名
 *         useHeader   = "true"               保存ﾌｧｲﾙにﾍｯﾀﾞｰを出力するかどうか
 *         useQuote    = "false"              ﾃﾞｰﾀをﾀﾞﾌﾞﾙｸｵｰﾄで囲うかどうか
 *         useQuoteEscape = "true"            ﾀﾞﾌﾞﾙｸｵｰﾄ文字が含まれる場合､ｴｽｹｰﾌﾟするかどうか
 *         useReturnQuote = "true"            改行ｺｰﾄﾞが含まれる場合､ﾀﾞﾌﾞﾙｸｵｰﾄで囲うかどうか
 *         replaceFrom = "',&quot;*%|"        置換元文字を指定｡一文字単位で置換します｡
 *         replaceTo   = "’，”＊％｜"       置換先文字を指定｡一文字単位で置換します｡
 *         zip         = "true"               ZIPﾌｧｲﾙに圧縮するかどうか
 *         zipFilename = "Sample.zip"         ZIPﾌｧｲﾙのﾌｧｲﾙ名
 *         fileAppend  = "true"               ﾌｧｲﾙを追加ﾓｰﾄﾞで登録するかどうか
 *         displayMsg  = "MSG0033"            実行後の表示ﾒｯｾｰｼﾞ
 *         fetchSize   = "200"                DB検索する場合のﾌｪｯﾁするｻｲｽﾞ
 *     &gt;
 *         SELECT * FROM ZYXX
 *     &lt;/og:directWriteTable &gt;
 *
 *     &lt;og:directWriteTable
 *         fileURL     = "{&#064;USER.ID}"    保存先ﾃﾞｨﾚｸﾄﾘ名
 *         filename    = "{&#064;filename}"   保存ﾌｧｲﾙ名
 *         names       = "AAA,BBB,CCC,･･･"    指定のｷｰに対応するﾘｸｴｽﾄ値を ARG_ARRAY にｾｯﾄします｡
 *         queryType   = "JDBCErrMsg"         JDBCErrMsg 形式のPL/SQL をｺｰﾙします｡
 *     &gt;
 *        { call PL/SQL(?,?,?,? ) }
 *     &lt;/og:directWriteTable &gt;
 *
 * @og.rev 3.5.6.0 (2004/06/18) 新規作成
 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLｺｰﾙ(JDBCErrMsg 形式)の実行を追加
 * @og.group ﾌｧｲﾙ出力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DirectWriteTableTag extends CommonTagSupport {
	/** このﾌﾟﾛｸﾞﾗﾑのVERSION文字列を設定します｡	{@value} */
	private static final String VERSION = "8.0.1.0 (2021/10/29)" ;
	private static final long serialVersionUID = 801020211029L ;

	private static final String TAB_SEPARATOR	= "\t" ;
	private static final String ERR_MSG_ID		= HybsSystem.ERR_MSG_KEY;		// 6.4.1.1 (2016/01/16) errMsgId → ERR_MSG_ID  refactoring

	private final int DB_MAX_QUERY_TIMEOUT		= HybsSystem.sysInt( "DB_MAX_QUERY_TIMEOUT" ) ;
	private static final String ARG_ARRAY		= "ARG_ARRAY" ;
	private static final String ERR_MSG			= "ERR_MSG" ;
	private static final String ERR_MSG_ARRAY	= "ERR_MSG_ARRAY" ;

//	/** 6.9.3.0 (2018/03/26) ﾃﾞｰﾀ検索時のﾌｪｯﾁｻｲｽﾞ */
//	private static final int DB_FETCH_SIZE		= HybsSystem.sysInt( "DB_FETCH_SIZE" ) ;

	// 4.0.0.0 (2007/10/10) dbid の初期値を､"DEFAULT" から null に変更
	private String	dbid		;
	private String  separator	= TAB_SEPARATOR;	// 項目区切り文字
	private boolean useHeader	= true;				// ﾍｯﾀﾞｰの使用可否
	private boolean useQuote	;					// 6.0.3.0 (2014/11/13) ﾀﾞﾌﾞﾙｸｵｰﾄで囲うかどうか
	private boolean useQuoteEscape	= true;			// 6.0.3.0 (2014/11/13) ﾃﾞｰﾀ中にﾀﾞﾌﾞﾙｸｵｰﾄ文字が含まれる場合､ｴｽｹｰﾌﾟするかどうか
	private boolean useReturnQuote	= true;			// 6.0.3.0 (2014/11/13) ﾃﾞｰﾀ中に改行ｺｰﾄﾞが含まれる場合､ﾀﾞﾌﾞﾙｸｵｰﾄで囲うかどうか
	private String  fileURL		= HybsSystem.sys( "FILE_URL" );
	private String  filename	= HybsSystem.sys( "FILE_FILENAME" );	// ﾌｧｲﾙ名
	private String	sql			;
	private String  encode		= HybsSystem.sys( "FILE_ENCODE"   );	// ﾌｧｲﾙｴﾝｺｰﾃﾞｨﾝｸﾞ  "DEFAULT","JISAutoDetect" ,"JIS", "EUC_JP", "MS932", "SJIS" , "Windows-31J" , "Shift_JIS"
	private boolean fileAppend	;					// ﾌｧｲﾙをAPPENDﾓｰﾄﾞで出力するか
	private boolean zip			;					// ﾌｧｲﾙをZIPするか
	private String  zipFilename	;					// ZIPﾌｧｲﾙ名
	private String	displayMsg	= HybsSystem.sys( "VIEW_DISPLAY_MSG" );
	private String	notfoundMsg	= "MSG0077";		// 対象ﾃﾞｰﾀはありませんでした｡
	private long	dyStart		;					// 実行時間測定用のDIV要素を出力します｡
	private boolean	useTimeView	= HybsSystem.sysBool( "VIEW_USE_TIMEBAR" );		// 6.3.6.0 (2015/08/16)
	private int		fetchSize	= DB_FETCH_SIZE ;	// ﾌｪｯﾁする行数(初期値:1001)	6.9.3.0 (2018/03/26) 初期値を100→HybsConst.DB_FETCH_SIZE に変更
	private boolean	useNumber	= true;				// 5.5.7.1(2012/10/05) 行番号出力

	private String  replaceFrom	;					// 6.0.3.0 (2014/11/13) 置換元文字を指定
	private String  replaceTo	;					// 6.0.3.0 (2014/11/13) 置換先文字を指定

	private boolean	quotCheck	= HybsSystem.sysBool( "USE_SQL_INJECTION_CHECK" );	// 6.2.2.0 (2015/03/27)
	private boolean	xssCheck	= HybsSystem.sysBool( "USE_XSS_CHECK" );			// 6.2.2.0 (2015/03/27)

	// 3.6.1.0 (2005/01/05) PL/SQLｺｰﾙ(JDBCErrMsg 形式)への対応
	private boolean	queryType	= true;				// ﾉｰﾏﾙは､true/ JDBCErrMsg の時は､false
	private String	names		;					// 指定のﾘｸｴｽﾄ変数を､ARG_ARRAY にｾｯﾄします｡
	private int		errCode		= ErrorMessage.OK;
	private transient ErrorMessage errMessage	;
	private boolean	useSLabel	;					// 7.0.7.0 (2019/12/13) ｴﾗｰﾒｯｾｰｼﾞにSLABELを利用するかどうか[true/false]を指定します(初期値:false)
	private boolean useLocal	;					// 8.0.1.0 (2021/10/29) ｸﾗｳﾄﾞ設定を使用しない場合は､true
	private String	mapObjKey	;					// 7.0.7.1 (2019/12/24) valueﾀｸﾞのaction=MAPOBJ を使用したﾗﾍﾞﾙ変換を行う場合の､MAPOBJｷｰを指定します｡

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰ
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public DirectWriteTableTag() { super(); }		// これも､自動的に呼ばれるが､空のﾒｿｯﾄﾞを作成すると警告されるので､明示的にしておきます｡

	/**
	 * Taglibの開始ﾀｸﾞが見つかったときに処理する doStartTag() を ｵｰﾊﾞｰﾗｲﾄﾞします｡
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 置換元文字,置換先文字のﾁｪｯｸ
	 * @og.rev 6.3.4.0 (2015/08/01) caseKey,caseVal,caseNN,caseNull,caseIf 属性対応
	 * @og.rev 6.4.8.1 (2016/07/02) xssCheckを､doStartTag に移動
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		// 反転注意､if ﾛｼﾞｯｸ統合
		if( useTag() ) {
			dyStart = System.currentTimeMillis();		// 時間測定用

			useXssCheck( xssCheck );			// 6.4.8.1 (2016/07/02)

			// 6.0.3.0 (2014/11/13) 置換元文字,置換先文字を指定
			if( ( replaceFrom != null || replaceTo != null ) &&
				( replaceFrom == null || replaceTo == null || replaceFrom.length() != replaceTo.length() ) ) {
					final String errMsg = "置換元文字と置換先文字の文字数が異なります｡" + CR
								+ " replaceFrom=[" + replaceFrom + "] , replaceTo=[" + replaceTo + "]"
								+ CR ;
					throw new HybsSystemException( errMsg );
			}

			return EVAL_BODY_BUFFERED ;	// Body を評価する｡( extends BodyTagSupport 時)
		}
		else {
			return SKIP_BODY;
		}
	}

	/**
	 * Taglibのﾀｸﾞ本体を処理する doAfterBody() を ｵｰﾊﾞｰﾗｲﾄﾞします｡
	 *
	 * @og.rev 3.8.6.3 (2006/11/30) SQL 文の前後のｽﾍﾟｰｽを取り除きます｡
	 * @og.rev 6.2.2.0 (2015/03/27) XSSﾁｪｯｸ,ｸｫｰﾄﾁｪｯｸ をｻﾎﾟｰﾄします｡
	 * @og.rev 6.4.8.1 (2016/07/02) xssCheckを､doStartTag に移動
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		// 6.2.2.0 (2015/03/27) XSSﾁｪｯｸ,ｸｫｰﾄﾁｪｯｸ をｻﾎﾟｰﾄします｡
		useQuotCheck( quotCheck );

		sql = getBodyString();
		if( sql == null || sql.isEmpty() ) {
			final String errMsg = "BODY 部の検索用 Select文は､必須です｡";
			throw new HybsSystemException( errMsg );
		}
		sql = sql.trim();
		return SKIP_BODY ;				// Body を評価しない
	}

	/**
	 * Taglibの終了ﾀｸﾞが見つかったときに処理する doEndTag() を ｵｰﾊﾞｰﾗｲﾄﾞします｡
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLｺｰﾙ(JDBCErrMsg 形式)への対応
	 * @og.rev 4.0.0.0 (2007/10/18) ﾒｯｾｰｼﾞﾘｿｰｽ統合( getResource().getMessage ⇒ getResource().getLabel )
	 * @og.rev 6.0.4.0 (2014/11/28) Zip処理を､ZipOutputStream → ZipArchiveOutputStream に変更
	 * @og.rev 6.3.4.0 (2015/08/01) caseKey,caseVal,caseNN,caseNull,caseIf 属性対応
	 * @og.rev 6.3.8.0 (2015/09/11) FileUtil#getPrintWriter( OutputStream,String ) を使用｡
	 * @og.rev 7.0.7.0 (2019/12/13) useSLabel 属性を追加｡
	 * @og.rev 8.0.0.2 (2021/10/15) ﾛｰｶﾙﾌｧｲﾙとｸﾗｳﾄﾞﾌｧｲﾙ間の移動
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		if( !useTag() ) { return EVAL_PAGE ; }			// 6.3.4.0 (2015/08/01)

		PrintWriter pw = null;
		final int executeCount;
		final File localFile = makeLocalFile();			// 8.0.0.2 (2021/10/15)
		try {
			if( zip ) {
//				// 8.0.0.2 (2021/10/15) zip も考慮して makeLocalFile() で対応する｡
//				final String directory = HybsSystem.url2dir( fileURL );

//				if( zipFilename == null ) { zipFilename = filename + ".zip"; }
//				final File zipFile = new File( directory,zipFilename );		// 8.0.0.1 (2021/10/08)
				ZipArchiveOutputStream gzip = null;			// 6.0.4.0 (2014/11/28)
				try {
					// 6.0.4.0 (2014/11/28) Zip処理を､ZipOutputStream → ZipArchiveOutputStream に変更
					gzip = new ZipArchiveOutputStream(
								new BufferedOutputStream (
									new FileOutputStream (
//										new File( directory,zipFilename ))));	// 6.0.4.0 (2014/11/28)
										localFile )));							// 8.0.0.2 (2021/10/15)
					gzip.setEncoding( "Windows-31J" );
					gzip.putArchiveEntry( new ZipArchiveEntry( filename ) );
					// 6.0.4.0 (2014/11/28) ﾌｧｲﾙのencode を指定できるようにする｡
					// 6.3.8.0 (2015/09/11) FileUtil#getPrintWriter( OutputStream,String ) を使用｡
					pw = FileUtil.getPrintWriter( gzip,encode );		// 6.3.8.0 (2015/09/11)
					executeCount = create( pw ) ;

					pw.flush();
					gzip.closeArchiveEntry();				// 6.0.4.0 (2014/11/28)
					gzip.finish() ;
				}
				finally {
					Closer.ioClose( gzip );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
				}

//				// 8.0.0.2 (2021/10/15) ﾛｰｶﾙﾌｧｲﾙとｸﾗｳﾄﾞﾌｧｲﾙ間の移動
//				// 5.10.9.0 (2019/03/01) ADD ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ利用
//				HybsFileOperationFactory.local2cloud( () -> zipFile );
//				final FileOperation cloudFile = HybsFileOperationFactory.create( directory,zipFilename );
//				if( cloudFile.isCloud() ) {
//					FileUtil.copy( zipFile, cloudFile );
//					zipFile.delete();
//				}
			}
			else {
//				pw = getPrintWriter();
				pw = getPrintWriter( localFile );		// 8.0.0.2 (2021/10/15)
				executeCount = create( pw );
			}
		} catch( final IOException ex ) {
			final String errMsg = "Error in DirectWriteTableTag: " + toString();
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		} finally {
			Closer.ioClose( pw );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
		}

		// 8.0.0.2 (2021/10/15) ﾛｰｶﾙﾌｧｲﾙから､ｸﾗｳﾄﾞに戻す｡
		HybsFileOperationFactory.local2cloud( useLocal,() -> localFile );

		// 3.6.1.0 (2005/01/05) 検索結果の件数を､"DB.COUNT" ｷｰでﾘｸｴｽﾄにｾｯﾄする｡
		setRequestAttribute( "DB.COUNT"   , String.valueOf( executeCount ) );
		setRequestAttribute( "DB.ERR_CODE", String.valueOf( errCode ) );

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

		// 実行件数の表示
		if( executeCount > 0 && displayMsg != null && displayMsg.length() > 0 ) {
			buf.append( executeCount )
				.append( getResource().getLabel( displayMsg ) )
				.append( BR );
		}
		else if( executeCount == 0 && notfoundMsg != null && notfoundMsg.length() > 0 ) {
			buf.append( getResource().getLabel( notfoundMsg ) )
				.append( BR );
		}

		// 3.6.1.0 (2005/01/05) TaglibUtil.makeHTMLErrorTable ﾒｿｯﾄﾞを利用
//		final String err = TaglibUtil.makeHTMLErrorTable( errMessage,getResource() );
		final String err = TaglibUtil.makeHTMLErrorTable( errMessage,getResource(),useSLabel );		// 7.0.7.0 (2019/12/13)
		if( err != null && err.length() > 0 ) {
			buf.append( err );
			setSessionAttribute( ERR_MSG_ID,errMessage );
		}
		else {
			removeSessionAttribute( ERR_MSG_ID );
		}

		jspPrint( buf.toString() );

		// 3.6.1.0 (2005/01/05) 警告時に停止していましたが､継続処理させます｡
		int rtnCode = EVAL_PAGE;
		if( errCode >= ErrorMessage.NG )  { 	// 異常
			rtnCode = SKIP_PAGE;
		}

		// 4.0.0 (2005/01/31) ｾｷｭﾘﾃｨﾁｪｯｸ(ﾃﾞｰﾀｱｸｾｽ件数登録)
		final long dyTime = System.currentTimeMillis()-dyStart;
		final GUIInfo guiInfo = (GUIInfo)getSessionAttribute( HybsSystem.GUIINFO_KEY );
		if( guiInfo != null ) { guiInfo.addReadCount( executeCount,dyTime,sql ); }

		if( useTimeView ) {		// 6.3.6.0 (2015/08/16)
			// 時間測定用の DIV 要素を出力
			jspPrint( "<div id=\"queryTime\" value=\"" + (dyTime) + "\"></div>" );	// 3.5.6.3 (2004/07/12)
		}
		return rtnCode ;
	}

	/**
	 * ﾀｸﾞﾘﾌﾞｵﾌﾞｼﾞｪｸﾄをﾘﾘｰｽします｡
	 * ｷｬｯｼｭされて再利用されるので､ﾌｨｰﾙﾄﾞの初期設定を行います｡
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLｺｰﾙ(JDBCErrMsg 形式)への対応
	 * @og.rev 4.0.0.0 (2007/10/10) dbid の初期値を､"DEFAULT" から null に変更
	 * @og.rev 5.5.7.1 (2012/10/05) useNumber追加
	 * @og.rev 6.0.3.0 (2014/11/13) useHeader,useQuote,useQuoteEscape,useReturnQuote,replaceFrom,replaceTo追加
	 * @og.rev 6.2.2.0 (2015/03/27) XSSﾁｪｯｸ,ｸｫｰﾄﾁｪｯｸ をｻﾎﾟｰﾄします｡
	 * @og.rev 6.9.3.0 (2018/03/26) fetchSizeの初期値を100→HybsConst.DB_FETCH_SIZE に変更
	 * @og.rev 7.0.7.0 (2019/12/13) useSLabel 属性を追加｡
	 * @og.rev 7.0.7.1 (2019/12/24) mapObjKey 属性を追加｡
	 * @og.rev 8.0.1.0 (2021/10/29) useLocal 属性を追加｡
	 */
	@Override
	protected void release2() {
		super.release2();
		separator	= TAB_SEPARATOR;	// 項目区切り文字
		fileURL		= HybsSystem.sys( "FILE_URL" );
		filename	= HybsSystem.sys( "FILE_FILENAME" );	// ﾌｧｲﾙ名
		sql			= null;
		encode		= HybsSystem.sys( "FILE_ENCODE" );		// ﾌｧｲﾙｴﾝｺｰﾃﾞｨﾝｸﾞ  "DEFAULT","JISAutoDetect" ,"JIS", "EUC_JP", "MS932", "SJIS" , "Windows-31J" , "Shift_JIS"
		fileAppend	= false;			// ﾌｧｲﾙをAPPENDﾓｰﾄﾞで出力するか
		zip			= false;			// ﾌｧｲﾙをZIPするか
		zipFilename	= null;				// ZIPﾌｧｲﾙ名
		displayMsg	= HybsSystem.sys( "VIEW_DISPLAY_MSG" );
		notfoundMsg	= "MSG0077";		// 対象ﾃﾞｰﾀはありませんでした｡
		dbid		= null;
		fetchSize	= DB_FETCH_SIZE ;	// ﾌｪｯﾁする行数(初期値:0 参考にしない)		6.9.3.0 (2018/03/26) 初期値を100→→HybsConst.DB_FETCH_SIZE に変更
		useTimeView	= HybsSystem.sysBool( "VIEW_USE_TIMEBAR" );	// 6.3.6.0 (2015/08/16)
		queryType	= true;				// ﾉｰﾏﾙは､true/ JDBCErrMsg の時は､false
		names		= null;				// 指定のﾘｸｴｽﾄ変数を､ARG_ARRAY にｾｯﾄします｡
		errCode		= ErrorMessage.OK;
		errMessage	= null;
		useNumber	= true;				// 5.5.7.1 (2012/10/05)
		useHeader	= true;				// ﾍｯﾀﾞｰの使用可否  … 6.0.3.0 (2014/11/13) 追加
		useQuote	= false;			// 6.0.3.0 (2014/11/13) ﾀﾞﾌﾞﾙｸｵｰﾄで囲うかどうか
		useQuoteEscape	= true;			// 6.0.3.0 (2014/11/13) ﾃﾞｰﾀ中にﾀﾞﾌﾞﾙｸｵｰﾄ文字が含まれる場合､ｴｽｹｰﾌﾟするかどうか
		useReturnQuote	= true;			// 6.0.3.0 (2014/11/13) ﾃﾞｰﾀ中に改行ｺｰﾄﾞが含まれる場合､ﾀﾞﾌﾞﾙｸｵｰﾄで囲うかどうか
		replaceFrom	= null;				// 6.0.3.0 (2014/11/13) 置換元文字を指定
		replaceTo	= null;				// 6.0.3.0 (2014/11/13) 置換先文字を指定
		quotCheck	= HybsSystem.sysBool( "USE_SQL_INJECTION_CHECK" );	// 6.2.2.0 (2015/03/27)
		xssCheck	= HybsSystem.sysBool( "USE_XSS_CHECK" );			// 6.2.2.0 (2015/03/27)
		useSLabel	= false;			// 7.0.7.0 (2019/12/13) ｴﾗｰﾒｯｾｰｼﾞにSLABELを利用するかどうか[true/false]を指定します(初期値:false)
		useLocal	= false;			// 8.0.1.0 (2021/10/29) ｸﾗｳﾄﾞ設定を使用しない場合は､true
		mapObjKey	= null;				// 7.0.7.1 (2019/12/24) valueﾀｸﾞのaction=MAPOBJ を使用したﾗﾍﾞﾙ変換を行う場合の､MAPOBJｷｰを指定します｡
	}

	/**
	 * 実ｵﾌﾞｼﾞｪｸﾄを生成して、OutputStream に書き込みます｡
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLｺｰﾙ(JDBCErrMsg 形式)への対応
	 * @og.rev 3.8.6.0 (2006/09/29) ﾍｯﾀﾞｰにﾗﾍﾞﾙを出力するように修正
	 * @og.rev 3.8.7.0 (2006/12/15) ｱｸｾｽﾛｸﾞ取得の為、ApplicationInfoｵﾌﾞｼﾞｪｸﾄを設定
	 * @og.rev 4.3.4.3 (2008/12/22) (Oracle11gDriver対応)PL/SQLｺｰﾙの場合に､"ｸﾛｰｽﾞされた文です｡"のｴﾗｰが発生する問題に対応
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応
	 * @og.rev 5.2.2.0 (2010/11/01) 改行を含む場合は､ﾀﾞﾌﾞﾙｸｵｰﾄを強制的に前後に追加する｡
	 * @og.rev 5.2.2.0 (2010/11/01) ﾀﾞﾌﾞﾙｸｵｰﾄを含む場合は､その直前にﾀﾞﾌﾞﾙｸｵｰﾄを強制的に追加する｡
	 * @og.rev 5.3.0.0 (2010/12/01) executeCall ﾒｿｯﾄﾞの引数見直し
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更
	 * @og.rev 5.5.7.1 (2012/10/05) useNumberの追加
	 * @og.rev 6.0.3.0 (2014/11/13) useQuote,useQuoteEscape,useReturnQuote,replaceFrom,replaceToの追加
	 * @og.rev 6.0.3.0 (2014/11/13) ﾍｯﾀﾞｰとﾗﾍﾞﾙを､指定の separator で出力するように変更します｡
	 * @og.rev 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更｡
	 * @og.rev 6.2.0.0 (2015/02/27) ﾃﾞｰﾀ出力の先頭ｶﾝﾏの判定処理変更
	 * @og.rev 6.3.6.1 (2015/08/28) Transaction でAutoCloseableを使用したtry-with-resources構築に対応｡
	 * @og.rev 7.0.7.1 (2019/12/24) mapObjKey 属性を追加｡
	 *
	 * @param   outPW 出力先のPrintWriterｵﾌﾞｼﾞｪｸﾄ
	 *
	 * @return	検索件数
	 */
	@SuppressWarnings(value={"unchecked"})
	private int create( final PrintWriter outPW )  {
		final int executeCount;
		Statement stmt = null;
		CallableStatement callStmt = null; // 4.3.4.3 (2008/12/22)
		ResultSet resultSet = null ;

		// 6.3.6.1 (2015/08/28) Transaction でAutoCloseableを使用したtry-with-resources構築に対応｡
		try( Transaction tran = getTransaction() ) {
			final Connection conn = tran.getConnection( dbid );				// 5.1.9.0 (2010/08/01) Transaction 対応
			// 3.6.1.0 (2005/01/05)
			if( queryType ) {		// JDBC 通常の SELECT 文
				stmt = conn.createStatement();
				if( fetchSize > 0 ) { stmt.setFetchSize( fetchSize ); }
				resultSet = stmt.executeQuery( sql );
			}
			else {					// PL/SQL Call 文
				String[] values = null;
				if( names != null ) {
					final String[] nameArray = StringUtil.csv2Array( names );
					values = getRequest( nameArray );
				}
				callStmt  = conn.prepareCall( sql );
				resultSet = executeCall( conn,callStmt,values );		// 5.3.0.0 (2010/12/01)
			}
			if( resultSet == null ) { return 0; }

			// 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更｡
			final ResultSetValue rsv = new ResultSetValue( resultSet );
			final int numberOfColumns =  rsv.getColumnCount();

			// ﾍｯﾀﾞｰ部の出力
			if( useHeader && numberOfColumns > 0 ) {
				final StringBuilder headName  = new StringBuilder( BUFFER_MIDDLE );
				final StringBuilder headLabel = new StringBuilder( BUFFER_MIDDLE );
				if( useNumber ){						// 6.0.3.0 (2014/11/13) ﾍｯﾀﾞｰ部の useNumber 対応漏れ
					headName.append(  "#Name" );
					headLabel.append( "#Label" );
				}

				Map<String,String> mapObj = null;		// 7.0.7.1 (2019/12/24) mapObjKey 属性を追加｡
				if( mapObjKey != null ) {
					mapObj = (Map<String,String>)getRequestAttribute( mapObjKey );
				}

				final ResourceManager resource = getResource();
				// 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更｡
				for( int clmNo=0; clmNo<numberOfColumns; clmNo++ ) {
					final String clm = rsv.getColumnName(clmNo);
					if( clmNo > 0 || useNumber ){						// 5.5.7.1 (2012/10/05)
						//この場合だけｾﾊﾟﾚｰﾀ出力する｡
						headName.append( separator );					// 6.0.3.0 (2014/11/13)
						headLabel.append( separator );					// 6.0.3.0 (2014/11/13)
					}
					headName.append( clm );								// 6.0.3.0 (2014/11/13)
//					headLabel.append( resource.getLabel( clm ) );		// 6.0.3.0 (2014/11/13)
					// 7.0.7.1 (2019/12/24) mapObjKey 属性を追加｡
					String lbl = resource.getLabel( clm );
					if( mapObj != null ) {
						lbl = mapObj.getOrDefault( clm,lbl );			// 7.0.7.1 (2019/12/24) Mapの値を優先して､null なら ﾘｿｰｽ使用
					}
					headLabel.append( lbl );							// 6.0.3.0 (2014/11/13)
				}
				outPW.println( headName.toString() );
				outPW.println( headLabel.toString() );
			}

			int rowNo = 0;
			// 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更｡
			while( rsv.next() ) {
				// 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更｡
				if( useNumber ){					// 5.5.7.1 (2012/10/05)
					if( useQuote ) { outPW.print( "\"" + rowNo + "\"" ); }	// 行番号
					else {			 outPW.print( rowNo ); }
				}
				for( int clmNo=0; clmNo<numberOfColumns; clmNo++ ) {
					// 6.0.2.5 (2014/10/31) refactoring:Avoid empty if statements 警告の対応
					if( clmNo > 0 || useNumber ){					// 6.2.0.0 (2015/02/27)
						//この場合だけｾﾊﾟﾚｰﾀ出力する｡
						outPW.print( separator );
					}
					String sval = replace( rsv.getValue(clmNo) );				// 禁則文字の置換処理
					if( sval != null && sval.length() > 0 ) {
						// 6.0.3.0 (2014/11/13) ﾃﾞｰﾀ中にﾀﾞﾌﾞﾙｸｵｰﾄ文字が含まれる場合､ｴｽｹｰﾌﾟするかどうか
						if( useQuoteEscape && sval.indexOf( '"' ) >= 0 ) { sval = sval.replaceAll( "\"" ,"\"\"" ) ; }
						// 6.0.3.0 (2014/11/13) ﾃﾞｰﾀ中に改行ｺｰﾄﾞが含まれる場合､ﾀﾞﾌﾞﾙｸｵｰﾄで囲うかどうか
						if( useQuote || useReturnQuote && sval.indexOf( CR ) >= 0 ) {
							sval = "\"" + sval + "\"" ;
						}
					}
					else {
						sval = useQuote ? "\"\"" : "" ;
					}
					outPW.print( sval );
				}
				outPW.println();
				rowNo++ ;
			}
			executeCount = rowNo ;
			tran.commit();				// 6.3.6.1 (2015/08/28)
		}
		catch( final SQLException ex ) {		// catch は､close() されてから呼ばれます｡
			final String errMsg = "ﾃﾞｰﾀﾍﾞｰｽ処理を実行できませんでした｡"
						 + CR + '[' + sql + ']' + CR
						 + "err=[" + ex.getSQLState() + ']'
						 + ex.getMessage();
			throw new HybsSystemException( errMsg,ex );
		}
		finally {						// finally は､close() されてから呼ばれます｡
			Closer.resultClose( resultSet );
			Closer.stmtClose( stmt );
			Closer.stmtClose( callStmt );	// 4.3.4.3 (2008/12/22)
		}

		return executeCount ;
	}

	/**
	 * ﾛｰｶﾙﾌｧｲﾙを作成します｡
	 *
	 * ﾃﾞｨﾚｸﾄﾘの存在ﾁｪｯｸとなければ作成します｡
	 * zipﾌｧｲﾙの指定がされて､zipFilenameが指定されていない場合は､filename + ".zip" とします｡
	 *
	 * @og.rev 8.0.0.2 (2021/10/15) ﾛｰｶﾙﾌｧｲﾙとｸﾗｳﾄﾞﾌｧｲﾙ間の移動
	 *
	 * @return	ﾛｰｶﾙﾌｧｲﾙｵﾌﾞｼﾞｪｸﾄ
	 */
	private File makeLocalFile() {
		if( filename == null ) {
			final String errMsg = "ﾌｧｲﾙ名がｾｯﾄされていません｡";
			throw new HybsSystemException( errMsg );
		}
		final String directory = HybsSystem.url2dir( fileURL );			// こちらは､ﾛｰｶﾙ変数でよい｡

		// 5.6.1.0 (2013/02/01)
		final File dir = new File(directory);
		if( ! dir.exists() && ! dir.mkdirs() ) {
			final String errMsg = "ﾃﾞｨﾚｸﾄﾘの作成に失敗しました｡[" + directory + "]";
			throw new HybsSystemException( errMsg );
		}

		final String newFilename ;
		if( zip ) {
			if( zipFilename == null ) { newFilename = filename + ".zip"; }
			else {						newFilename = zipFilename ; }
		}
		else {
			newFilename = filename;
		}

		return new File( directory,newFilename );
	}

	/**
	 * replaceFrom,replaceTo に基づく禁則文字の置換処理を行います｡
	 *
	 * replaceFrom の１文字づつを､対応するreplaceToの１文字づつに変換します｡
	 * replaceFrom と replaceTo の文字数は同じでなければなりません｡
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規追加
	 *
	 * @param   str 置換する文字列
	 *
	 * @return	置換後の文字列
	 */
	private String replace( final String str ) {
		String rtn = str;
		if( rtn != null && replaceFrom != null && replaceTo != null ) {
			for( int i=0; i<replaceTo.length(); i++ ) {
				rtn = rtn.replace( replaceFrom.charAt(i) , replaceTo.charAt(i) );		// charの置き換えは､全件
			}
		}
		return rtn ;
	}

	/**
	 * 引数配列付のｸｴﾘｰを実行します｡
	 * 処理自体は、#execute() と同様に､各ｻﾌﾞｸﾗｽの実装に依存します｡
	 * これは､CallableStatement を用いて､ﾃﾞｰﾀﾍﾞｰｽ検索処理を行います｡
	 * {call TYPE3B01.TYPE3B01(?,?,?,?)} で､４番目の引数には､
	 * names で指定したﾘｸｴｽﾄ情報が､ARG_ARRAY 配列に順次ｾｯﾄされます｡
	 * 使用する場合は､一旦わかり易い変数に受けて利用してください｡
	 * 呼び出す PL/SQL では､検索系PL/SQL です｡
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLｺｰﾙ(JDBCErrMsg 形式)への対応
	 * @og.rev 4.3.4.3 (2008/12/22) (Oracle11gDriver対応)PL/SQLｺｰﾙの場合に､"ｸﾛｰｽﾞされた文です｡"のｴﾗｰが発生する問題に対応
	 * @og.rev 5.3.0.0 (2010/12/01) executeCall ﾒｿｯﾄﾞの引数見直し
	 * @og.rev 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のﾄﾞﾗｲﾊﾞ)対応
	 *
	 * @param	conn		ｺﾈｸｼｮﾝ
	 * @param   callStmt	ｺｰﾗﾌﾞﾙｽﾃｰﾄﾒﾝﾄ
	 * @param   args		ｵﾌﾞｼﾞｪｸﾄの引数配列(可変長引数)
	 *
	 * @return	結果ｵﾌﾞｼﾞｪｸﾄ
	 */
	private ResultSet executeCall( final Connection conn,final CallableStatement callStmt,final String... args ) throws SQLException {
		callStmt.setQueryTimeout( DB_MAX_QUERY_TIMEOUT );
		if( fetchSize > 0 ) { callStmt.setFetchSize( fetchSize ); }
		final Map<String,Class<?>> map = conn.getTypeMap();
		try {
			map.put( ERR_MSG,Class.forName( "org.opengion.hayabusa.db.DBErrMsg" ) );
		}
		catch( final ClassNotFoundException ex ) {
			final String errMsg = "org.opengion.hayabusa.db.DBErrMsg ｸﾗｽが見つかりません｡" + CR
					+ ex.getMessage();			// // 5.1.8.0 (2010/07/01) errMsg 修正
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}

		// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のﾄﾞﾗｲﾊﾞ)対応 http://docs.oracle.com/cd/E28389_01/web.1111/b60995/thirdparty.htm
		final Array newArray = ((OracleConnection)conn).createOracleArray( ARG_ARRAY, StringUtil.rTrims( args ));		// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のﾄﾞﾗｲﾊﾞ)対応

		callStmt.registerOutParameter(1, Types.INTEGER);
		callStmt.registerOutParameter(2, Types.ARRAY,ERR_MSG_ARRAY);		// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のﾄﾞﾗｲﾊﾞ)対応
		callStmt.registerOutParameter(3, OracleTypes.CURSOR);
		callStmt.setArray( 4,newArray );									// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のﾄﾞﾗｲﾊﾞ)対応

		callStmt.execute();

		errCode = callStmt.getInt(1);

		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		ResultSet resultSet = null;
		if( errCode < ErrorMessage.NG ) {		// 異常以外の場合
			resultSet = ((OracleCallableStatement)callStmt).getCursor(3);
		}
		if( errCode > ErrorMessage.OK ) {		// 正常以外の場合
			final Array rtn3 = callStmt.getArray(2);								// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のﾄﾞﾗｲﾊﾞ)対応
			final Object[] rtnval3 = (Object[])rtn3.getArray();
			errMessage = new ErrorMessage( "Query_JDBCErrMsg Error!!" );
			for( int i=0; i<rtnval3.length; i++ ) {
				final DBErrMsg er = (DBErrMsg)rtnval3[i];
				if( er == null ) { break; }
				errMessage.addMessage( er.getErrMsg() );
			}
		}
		return resultSet;
	}

	/**
	 * PrintWriter を取得します｡
	 *
	 * ここでは､一般的なﾌｧｲﾙ出力を考慮した PrintWriter を作成します｡
	 *
	 * @og.rev 3.7.1.1 (2005/05/23) ﾌｫﾙﾀﾞがない場合は､複数階層分のﾌｫﾙﾀﾞを自動で作成します｡
	 * @og.rev 3.8.0.0 (2005/06/07) FileUtil#getPrintWriter を利用｡
	 * @og.rev 5.6.1.0 (2013/02/01) 3.7.1.1のｺﾒﾝﾄに入っているが対応されていないのでﾌｫﾙﾀﾞ作成追加
	 * @og.rev 8.0.0.2 (2021/10/15) ﾛｰｶﾙﾌｧｲﾙとｸﾗｳﾄﾞﾌｧｲﾙ間の移動
	 *
	 * @param	 localFile 出力先Fileｵﾌﾞｼﾞｪｸﾄ
	 * @return	 出力用PrintWriterｵﾌﾞｼﾞｪｸﾄ
	 * @og.rtnNotNull
	 */
//	private PrintWriter getPrintWriter() {
	private PrintWriter getPrintWriter( final File localFile ) {
//		// 8.0.0.2 (2021/10/15) ﾃﾞｨﾚｸﾄﾘﾁｪｯｸ､作成も考慮して makeLocalFile() で対応する｡
//		if( filename == null ) {
//			final String errMsg = "ﾌｧｲﾙ名がｾｯﾄされていません｡";
//			throw new HybsSystemException( errMsg );
//		}
//		final String directory = HybsSystem.url2dir( fileURL );

//		// 5.6.1.0 (2013/02/01)
//		final File dir = new File(directory);
//		if( ! dir.exists() && ! dir.mkdirs() ) {
//			final String errMsg = "ﾃﾞｨﾚｸﾄﾘの作成に失敗しました｡[" + directory + "]";
//			throw new HybsSystemException( errMsg );
//		}

		// ※ 注意 StringUtil.urlAppend を組み込んでいる意図が不明｡一旦削除していますが､注意
		// 3.8.0.0 (2005/06/07) FileUtil#getPrintWriter を利用｡
	//	out = FileUtil.getPrintWriter( StringUtil.urlAppend( directory,filename ),fileAppend,encode);

		// 8.0.0.2 (2021/10/15) ﾛｰｶﾙﾌｧｲﾙとｸﾗｳﾄﾞﾌｧｲﾙ間の移動
		// PrintWriterはﾛｰｶﾙなので､appendする場合は､一旦､cloudからﾌｧｲﾙを持ってくる必要がある｡
//		final File localFile = new File(directory, filename);
		// 5.10.9.0 (2019/03/01) ADD ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ利用､fileAppend､かつﾌｧｲﾙが存在する場合はﾀﾞｳﾝﾛｰﾄﾞする｡
		if( fileAppend && localFile.isFile() ) {
			HybsFileOperationFactory.cloud2local( useLocal,() -> localFile );
		}
//		final FileOperation cloudFile = HybsFileOperationFactory.create( directory, filename );
//		if( cloudFile.isCloud() && fileAppend && cloudFile.isFile() ) {
//			localFile.delete();
//			FileUtil.copy( cloudFile, localFile );
//		}

		// 処理を簡素化します｡
//		return FileUtil.getPrintWriter( new File( directory,filename ),encode,fileAppend );
		return FileUtil.getPrintWriter( localFile,encode,fileAppend );
	}

	/**
	 * 名称配列を元に､ﾘｸｴｽﾄ情報のﾃﾞｰﾀを取得します｡
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLｺｰﾙ(JDBCErrMsg 形式)への対応
	 *
	 * @param	nameArray	ｷｰとなる名称の配列(可変長引数)
	 *
	 * @return	そのﾘｸｴｽﾄ情報
	 * @og.rtnNotNull
	 */
	private String[] getRequest( final String... nameArray ) {
		String[] rtn = new String[nameArray.length];

		for( int i=0; i<rtn.length; i++ ) {
			rtn[i] = getRequestValue( nameArray[i] );
		}

		return rtn;
	}

	/**
	 * 【TAG】(通常は使いません)検索時のDB接続IDを指定します(初期値:DEFAULT)｡
	 *
	 * @og.tag
	 *   検索時のDB接続IDを指定します｡初期値は､DEFAULT です｡
	 *
	 * @param	id DB接続ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}

	/**
	 * 【TAG】可変長ﾌｧｲﾙを作成するときの項目区切り文字(ｾﾊﾟﾚｰﾀ)をｾｯﾄします(初期値:TAB_SEPARATOR)｡
	 *
	 * @og.tag 可変長ﾌｧｲﾙを作成するときの項目区切り文字をｾｯﾄします｡
	 * (初期値:ﾛｰｶﾙ定義のTAB_SEPARATOR)｡
	 *
	 * @param   sep ｾﾊﾟﾚｰﾀ
	 * @see		#TAB_SEPARATOR
	 */
	public void setSeparator( final String sep ) {
		separator = nval( getRequestParameter( sep ),TAB_SEPARATOR );
	}

	/**
	 * 【TAG】保存先ﾃﾞｨﾚｸﾄﾘ名を指定します
	 *		(初期値:FILE_URL[={@og.value SystemData#FILE_URL}])｡
	 *
	 * @og.tag
	 * この属性で指定されるﾃﾞｨﾚｸﾄﾘに､ﾌｧｲﾙをｾｰﾌﾞします｡
	 * 指定方法は､通常の fileURL 属性と同様に､先頭が､'/' (UNIX) または､２文字目が､
	 * ":" (Windows)の場合は､指定のURLそのままのﾃﾞｨﾚｸﾄﾘに､そうでない場合は､
	 * fileURL = "{&#064;USER.ID}" と指定すると､FILE_URL 属性で指定のﾌｫﾙﾀﾞの下に､
	 * さらに､各個人ID別のﾌｫﾙﾀﾞを作成して､そこにｾｰﾌﾞします｡
	 * (初期値:ｼｽﾃﾑ定数のFILE_URL[={@og.value SystemData#FILE_URL}])｡
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 内部処理を､makeFileURL に移動｡
	 * @og.rev 4.0.0.0 (2005/01/31) StringUtil.urlAppend ﾒｿｯﾄﾞの利用
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたﾃﾞｨﾚｸﾄﾘ名の最後が"\"or"/"で終わっていない場合に､"/"を付加する｡
	 * @og.rev 6.4.2.1 (2016/02/05) URLの最後に､"/" を追加する処理を廃止｡
	 *
	 * @param	url 保存先ﾃﾞｨﾚｸﾄﾘ名
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_URL
	 */
	public void setFileURL( final String url ) {
		final String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】ﾌｧｲﾙを作成するときのﾌｧｲﾙ名をｾｯﾄします(初期値:ｼｽﾃﾑﾊﾟﾗﾒｰﾀのFILE_FILENAME)｡
	 *
	 * @og.tag ﾌｧｲﾙを作成するときのﾌｧｲﾙ名をｾｯﾄします｡
	 *
	 * @param   fname ﾌｧｲﾙ名
	 */
	public void setFilename( final String fname ) {
		filename = nval( getRequestParameter( fname ),filename );
	}

	/**
	 * 【TAG】ﾌｧｲﾙを作成するときのﾌｧｲﾙｴﾝｺｰﾃﾞｨﾝｸﾞ名をｾｯﾄします
	 *		(初期値:FILE_ENCODE[={@og.value SystemData#FILE_ENCODE}])｡
	 *
	 * @og.tag
	 * "DEFAULT","JISAutoDetect" ,"JIS", "EUC_JP", "MS932", "SJIS" , "Windows-31J" , "Shift_JIS"
	 * (初期値:ｼｽﾃﾑ定数のFILE_ENCODE[={@og.value SystemData#FILE_ENCODE}])｡
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 ｴﾝｺｰﾄﾞの取得方法変更
	 * @og.rev 3.1.3.0 (2003/04/10) FILE_ENCODE から､ｴﾝｺｰﾄﾞ情報を取得する｡
	 *
	 * @param   enc ﾌｧｲﾙｴﾝｺｰﾃﾞｨﾝｸﾞ名
	 * @see     <a href="http://www.iana.org/assignments/character-sets" target="_blank" >IANA Charset Registry</a>
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_ENCODE
	 */
	public void setEncode( final String enc ) {
		encode = nval( getRequestParameter( enc ),encode );
	}

	/**
	 * 【TAG】ﾍｯﾀﾞｰを書き込むかどうか[true/false]を指定します(初期値:true)｡
	 *
	 * @og.tag
	 *  #Name ････ ﾍｯﾀﾞｰの書き込みを指定します｡
	 * 通常は､書き込み(true)にしておき､使用側でｺﾒﾝﾄと解釈するように
	 * 処理を行うべきです｡ｺﾒﾝﾄのため､append ﾓｰﾄﾞで途中に現れても
	 * 無視できます｡また､ｴﾝｼﾞﾝ標準でﾃﾞｰﾀを取り込む場合に､ﾃﾞｰﾀの配置が
	 * 変更されても取り込みﾌﾟﾛｸﾞﾗﾑはそのまま使用できます｡
	 * 初期値は､true(書き込む)です｡
	 *
	 * @param   flag ﾍｯﾀﾞｰ有無 [true:書き込む/false:書き込まない]
	 */
	public void setUseHeader( final String flag ) {
		useHeader = nval( getRequestParameter( flag ),useHeader );
	}

	/**
	 * 【TAG】ﾃﾞｰﾀをﾀﾞﾌﾞﾙｸｵｰﾄで囲うかどうか指定します(初期値:false)｡
	 *
	 * @og.tag
	 * ﾃﾞｰﾀを出力する場合､ﾀﾞﾌﾞﾙｸｵｰﾄで囲うかどうか指定します｡
	 * 主に､区切り文字(separator)を､ﾀﾌﾞではなく､ｶﾝﾏを使う場合に､使用します｡
	 * なお､ﾍｯﾀﾞｰ部は､この指定に関わらず､ﾀﾞﾌﾞﾙｸｵｰﾄで囲いません｡
	 * 初期値は､false(囲わない)です｡
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規追加
	 *
	 * @param   flag ﾀﾞﾌﾞﾙｸｵｰﾄ使用 [true:書き込む/false:書き込まない]
	 */
	public void setUseQuote( final String flag ) {
		useQuote = nval( getRequestParameter( flag ),useQuote );
	}

	/**
	 * 【TAG】ﾃﾞｰﾀ中にﾀﾞﾌﾞﾙｸｵｰﾄ文字が含まれる場合､ｴｽｹｰﾌﾟするかどうか指定します(初期値:true)｡
	 *
	 * @og.tag
	 * ﾃﾞｰﾀ中にﾀﾞﾌﾞﾙｸｵｰﾄ文字が含まれる場合､ｴｽｹｰﾌﾟするかどうか指定します｡
	 * ここでいうｴｽｹｰﾌﾟとは､ﾀﾞﾌﾞﾙｸｵｰﾄ文字を重ねる処理を指します｡
	 * 初期値は､互換性の関係で､true(処理する)です｡
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規追加
	 *
	 * @param   flag ﾀﾞﾌﾞﾙｸｵｰﾄｴｽｹｰﾌﾟ有無 [true:する/false:しない]
	 */
	public void setUseQuoteEscape( final String flag ) {
		useQuoteEscape = nval( getRequestParameter( flag ),useQuoteEscape );
	}

	/**
	 * 【TAG】ﾃﾞｰﾀ中に改行ｺｰﾄﾞが含まれる場合､ﾀﾞﾌﾞﾙｸｵｰﾄで囲うかどうか指定します(初期値:true)｡
	 *
	 * @og.tag
	 * ﾃﾞｰﾀ中に改行ｺｰﾄﾞが含まれたﾃｷｽﾄの場合､EXCELで開くと､改行されてしまう｡
	 * その場合､ﾀﾞﾌﾞﾙｸｵｰﾄで囲うと､ｾﾙに設定してくれます｡
	 * この処理は､useQuote="true" にすると､無条件に､ﾃﾞｰﾀは囲われます｡
	 * 初期値は､互換性の関係で､true(処理する)です｡
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規追加
	 *
	 * @param   flag 改行ｺｰﾄﾞ処理 [true:する/false:しない]
	 * @see		#setUseQuote( String )
	 */
	public void setUseReturnQuote( final String flag ) {
		useReturnQuote = nval( getRequestParameter( flag ),useReturnQuote );
	}

	/**
	 * 【TAG】一文字単位で置換する置換元文字を指定します(初期値:null 置換なし)｡
	 *
	 * @og.tag
	 * ﾃﾞｰﾀ出力時に､禁則文字を､置き換える元の文字を指定します｡
	 * ここでは､一文字単位で､置換しますので､禁則文字は､連続の文字列の
	 * 形で､指定します｡
	 * なお､ﾍｯﾀﾞｰ部は､この指定に関わらず､ﾀﾞﾌﾞﾙｸｵｰﾄで囲いません｡
	 * 初期値は､null の場合は､何も変換しません｡
	 * 文字数は､replaceTo と同じでなければなりません｡
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規追加
	 * @og.rev 6.2.2.0 (2015/03/27) \n,\r,\t をｻﾎﾟｰﾄします｡
	 *
	 * @param   str 置換元文字
	 */
	public void setReplaceFrom( final String str ) {
		replaceFrom = changeRNT( nval( getRequestParameter( str ),replaceFrom ) );
	}

	/**
	 * 【TAG】一文字単位で置換する置換先文字を指定します｡
	 *
	 * @og.tag
	 * ﾃﾞｰﾀ出力時に､禁則文字を､置き換える先の文字を指定します｡
	 * ここでは､一文字単位で､置換しますので､禁則文字は､連続の文字列の
	 * 形で､指定します｡(例えば､全角文字にするとか)
	 * 初期値は､null の場合は､何も変換しません｡
	 * 文字数は､replaceFrom と同じでなければなりません｡
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規追加
	 * @og.rev 6.2.2.0 (2015/03/27) \n,\r,\t をｻﾎﾟｰﾄします｡
	 *
	 * @param   str 置換先文字
	 */
	public void setReplaceTo( final String str ) {
		replaceTo = changeRNT( nval( getRequestParameter( str ),replaceTo ) );
	}

	/**
	 * replaceFrom,replaceTo で､\n,\r,\t をｻﾎﾟｰﾄします｡
	 *
	 * ﾃﾞｰﾀ置換で､改行､復帰､ﾀﾌﾞを､指定する場合､２文字必要です｡
	 * ここでは､\n,\r,\t が指定された場合､ｷｬﾗｸﾀｺｰﾄﾞに置き換えます｡
	 *
	 * @og.rev 6.2.2.0 (2015/03/27) \n,\r,\t をｻﾎﾟｰﾄします｡
	 *
	 * @param	str 置換先文字
	 * @return	置換先文字
	 */
	private String changeRNT( final String str ) {
		String rtn = str ;
		if( rtn != null && !rtn.isEmpty() ) {
			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
			for( int i=0; i<rtn.length(); i++ ) {
				char ch = rtn.charAt(i) ;
				if( ch == '\\' ) {
					final char ch2 = rtn.charAt(++i) ;		// ￥ の次の文字(n,r,tのみｻﾎﾟｰﾄ)
					switch( ch2 ) {
						case 'n' : ch = '\n'; break;
						case 'r' : ch = '\r'; break;
						case 't' : ch = '\t'; break;
						default  :
							final String errMsg = getClass().getName()
											 + "の置換文字列で､｢\\" + ch2 + "｣は､ｻﾎﾟｰﾄされていません｡";
							System.err.println( errMsg );
							break;
					}
				}
				buf.append( ch );
			}
			rtn = buf.toString();
		}
		return rtn ;
	}

	/**
	 * 【TAG】追加ﾓｰﾄﾞで書き込むかどうか[true/false]を指定します(初期値:false[新規ﾓｰﾄﾞ])｡
	 *
	 * @og.tag
	 * ﾌｧｲﾙを書き込む場合､追加ﾓｰﾄﾞで書き込むかどうかをｾｯﾄします｡
	 * 新規ﾓｰﾄﾞ(true)の場合､既存のﾌｧｲﾙが存在し､かつ書き込み許可があれば､
	 * 上書きで新規に作成します｡
	 * 初期値は､false(新規ﾓｰﾄﾞ)です｡
	 *
	 * @param   flag 追加ﾓｰﾄﾞ [true:追加ﾓｰﾄﾞ/false:新規ﾓｰﾄﾞ]
	 */
	public void setFileAppend( final String flag ) {
		fileAppend = nval( getRequestParameter( flag ),fileAppend );
	}

	/**
	 * 【TAG】結果をﾌｧｲﾙに出力するときに､ZIPで圧縮するかどうか[true/false]を指定します(初期値:false)｡
	 *
	 * @og.tag
	 * 大量に抜き出す場合､そのまま､ｻｰﾊﾞｰから取り出すだけでも大変です｡
	 * zip 属性を､true にすると､GZIP で圧縮したﾌｧｲﾙを作成します｡
	 * 初期値は､false(圧縮しない)です｡
	 *
	 * @param  flag ZIP圧縮 [true:する/それ以外:しない]
	 * @see    #setZipFilename( String )
	 */
	public void setZip( final String flag ) {
		zip = nval( getRequestParameter( flag ),zip );
	}

	/**
	 * 【TAG】ZIPﾌｧｲﾙを作成するときのZIPﾌｧｲﾙ名をｾｯﾄします(初期値:filename + ".zip")｡
	 *
	 * @og.tag
	 * zip 属性に､true を指定した場合に､ZIPﾌｧｲﾙ化します｡その場合のﾌｧｲﾙ名を指定します｡
	 * なにも指定しない場合は､filename + ".zip" になります｡
	 *
	 * @param   zipFile ZIPﾌｧｲﾙ名
	 * @see #setZip( String )
	 */
	public void setZipFilename( final String zipFile ) {
		zipFilename = nval( getRequestParameter( zipFile ),zipFilename );
	}

	/**
	 * 【TAG】検索結果を画面上に表示するﾒｯｾｰｼﾞﾘｿｰｽIDを指定します
	 *		(初期値:VIEW_DISPLAY_MSG[={@og.value SystemData#VIEW_DISPLAY_MSG}])｡
	 *
	 * @og.tag
	 * ここでは､検索結果の件数や登録された件数をまず出力し､
	 * その次に､ここで指定したﾒｯｾｰｼﾞをﾘｿｰｽから取得して
	 * 表示します｡
	 * 件数を表示させる場合は､displayMsg = "MSG0033"[　件検索しました] をｾｯﾄしてください｡
	 * 表示させたくない場合は、displayMsg = "" をｾｯﾄしてください｡
	 * (初期値:ｼｽﾃﾑ定数のVIEW_DISPLAY_MSG[={@og.value SystemData#VIEW_DISPLAY_MSG}])｡
	 *
	 * @param	id 結果表示ﾒｯｾｰｼﾞID
	 */
	public void setDisplayMsg( final String id ) {
		final String ids = getRequestParameter( id );
		if( ids != null ) { displayMsg = ids; }
	}

	/**
	 * 【TAG】検索結果がｾﾞﾛ件の場合に表示するﾒｯｾｰｼﾞﾘｿｰｽIDを指定します(初期値:MSG0077[対象ﾃﾞｰﾀはありませんでした])｡
	 *
	 * @og.tag
	 * ここでは､検索結果がｾﾞﾛ件の場合のみ､特別なﾒｯｾｰｼﾞを表示させます｡
	 * 従来は､displayMsg と兼用で､『0　件検索しました』という表示でしたが､
	 * displayMsg の初期表示は､OFF になりましたので､ｾﾞﾛ件の場合のみ別に表示させます｡
	 * 表示させたくない場合は、notfoundMsg = "" をｾｯﾄしてください｡
	 * 初期値は､MSG0077[対象ﾃﾞｰﾀはありませんでした]です｡
	 *
	 * @param	id ｾﾞﾛ件時表示ﾒｯｾｰｼﾞID
	 */
	public void setNotfoundMsg( final String id ) {
		final String ids = getRequestParameter( id );
		if( ids != null ) { notfoundMsg = ids; }
	}

	/**
	 * 【TAG】(通常は使いません)ﾃﾞｰﾀのﾌｪｯﾁｻｲｽﾞを指定します
	 *		(初期値:DB_FETCH_SIZE[={@og.value org.opengion.fukurou.system.HybsConst#DB_FETCH_SIZE}])｡
	 *
	 * @og.tag
	 * より多くの行が必要なときに､ﾃﾞｰﾀﾍﾞｰｽから取り出す必要がある行数に
	 * ついてのﾋﾝﾄを JDBC ﾄﾞﾗｲﾊﾞに提供します｡
	 * 指定された行数は､この Statement を使って作成された結果ｾｯﾄにだけ影響します｡
	 * 指定された値が 0 の場合､ﾋﾝﾄは無視されます｡
	 * (初期値:ｼｽﾃﾑ定数のDB_FETCH_SIZE[={@og.value org.opengion.fukurou.system.HybsConst#DB_FETCH_SIZE}])｡
	 *
	 * @param	size ﾌｪｯﾁ行数
	 */
	public void setFetchSize( final String size ) {
		fetchSize = nval( getRequestParameter( size ),fetchSize );
	}

	/**
	 * 【TAG】PL/SQLを利用する場合の引数にｾｯﾄすべき ﾃﾞｰﾀの名称をCSV形式で複数指定します｡
	 *
	 * @og.tag
	 * 複数ある場合は､CSV形式で渡します｡
	 * PL/SQL を使用しない場合は､無視されます｡
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLｺｰﾙ(JDBCErrMsg 形式)への対応
	 *
	 * @param	nm 引数の名称 (CSV形式)
	 */
	public void setNames( final String nm ) {
		names = nval( getRequestParameter( nm ),names );
	}

	/**
	 * 【TAG】Query を発行する為のｸﾗｽID(JDBC,JDBCErrMsg)を指定します({@og.doc03Link queryType 初期値:JDBC})｡
	 *
	 * @og.tag
	 * ｽﾄｱﾄﾞﾌﾟﾛｼｰｼﾞｬ等を実行する場合に､queryType="JDBCErrMsg" を
	 * 指定する必要があります｡(それ以外の指定は､初期値の JDBC になります｡)
	 * 初期値は､"JDBC" です｡
	 * {@og.doc03Link queryType Query_**** ｸﾗｽ}
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLｺｰﾙ(JDBCErrMsg 形式)への対応
	 *
	 * @param	id Query発行ｸﾗｽID
	 */
	public void setQueryType( final String id ) {
		// 内部的には､JDBCErrMsg:false / それ以外:true で管理しています｡
		queryType = ! "JDBCErrMsg".equalsIgnoreCase( getRequestParameter( id ) );
	}

	/**
	 * 【TAG】ﾌｧｲﾙ出力時に､行番号情報を､出力する/しない[true/false]を指定します(初期値:true)｡
	 *
	 * @og.tag
	 * 通常のﾌｫｰﾏｯﾄでは､各行の先頭に行番号を出力します｡
	 * これは､#NAME 属性を使用する場合には､必ず出力する必要があります｡
	 * (#NAME 属性は､読み取り時にあれば､自動的にｶﾗﾑに割り当てられます｡)
	 * この､先頭の行番号が不要な場合(つまり､他のｼｽﾃﾑへのﾃﾞｰﾀ出力等)
	 * の為に出力する場合に､false を設定することで､行番号列を出力しない
	 * ようにできます｡
	 * 初期値は､true(出力する) です｡
	 *
	 * @og.rev 5.5.7.1 (2012/10/05) 新規追加
	 * @param  flag 行番号出力 [true:する/それ以外:しない]
	 */
	public void setUseNumber( final String flag ) {
		useNumber = nval( getRequestParameter( flag ),useNumber );
	}

	/**
	 * 【TAG】ﾘｸｴｽﾄ情報の ｼﾝｸﾞﾙｸｫｰﾄ(') 存在ﾁｪｯｸを実施するかどうか[true/false]を設定します
	 *		(初期値:USE_SQL_INJECTION_CHECK[={@og.value SystemData#USE_SQL_INJECTION_CHECK}])｡
	 *
	 * @og.tag
	 * ＳＱＬｲﾝｼﾞｪｸｼｮﾝ対策の一つとして､暫定的ではありますが､SQLのﾊﾟﾗﾒｰﾀに
	 * 渡す文字列にｼﾝｸﾞﾙｸｫｰﾄ(') を許さない設定にすれば､ある程度は防止できます｡
	 * 数字ﾀｲﾌﾟの引数には､ or 5=5 などのｼﾝｸﾞﾙｸｫｰﾄを使用しないｺｰﾄﾞを埋めても､
	 * 数字ﾁｪｯｸで検出可能です｡文字ﾀｲﾌﾟの場合は､必ず (')をはずして､
	 * ' or 'A' like 'A のような形式になる為､(')ﾁｪｯｸだけでも有効です｡
	 * (') が含まれていたｴﾗｰにする(true)／かﾉｰﾁｪｯｸか(false)を指定します｡
	 * 初期値は､SystemData#USE_SQL_INJECTION_CHECK です｡
	 *
	 * @og.rev 6.2.2.0 (2015/03/27) XSSﾁｪｯｸ,ｸｫｰﾄﾁｪｯｸ をｻﾎﾟｰﾄします｡
	 *
	 * @param   flag ｸｫｰﾄﾁｪｯｸ [true:する/それ以外:しない]
	 */
	public void setQuotCheck( final String flag ) {
		quotCheck = nval( getRequestParameter( flag ),quotCheck );
	}

	/**
	 * 【TAG】ﾘｸｴｽﾄ情報の HTMLTag開始/終了文字(&gt;&lt;) 存在ﾁｪｯｸを実施するかどうか[true/false]を設定します
	 *		(初期値:USE_XSS_CHECK[={@og.value SystemData#USE_XSS_CHECK}])｡
	 *
	 * @og.tag
	 * ｸﾛｽｻｲﾄｽｸﾘﾌﾟﾃｨﾝｸﾞ(XSS)対策の一環としてless/greater than signについてのﾁｪｯｸを行います｡
	 * (&gt;&lt;) が含まれていたｴﾗｰにする(true)／かﾉｰﾁｪｯｸか(false)を指定します｡
	 * (初期値:ｼｽﾃﾑ定数のUSE_XSS_CHECK[={@og.value SystemData#USE_XSS_CHECK}])
	 *
	 * @og.rev 6.2.2.0 (2015/03/27) XSSﾁｪｯｸ,ｸｫｰﾄﾁｪｯｸ をｻﾎﾟｰﾄします｡
	 *
	 * @param	flag	XSSﾁｪｯｸ [true:する/false:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_XSS_CHECK
	 */
	public void setXssCheck( final String flag ) {
		xssCheck = nval( getRequestParameter( flag ),xssCheck );
	}

	/**
	 * 【TAG】処理時間を表示する TimeView を表示するかどうか[true:する/false:しない]を指定します
	 *		(初期値:VIEW_USE_TIMEBAR[={@og.value SystemData#VIEW_USE_TIMEBAR}])｡
	 *
	 * @og.tag
	 * true に設定すると､処理時間を表示するﾊﾞｰｲﾒｰｼﾞが表示されます｡
	 * これは､DB検索､APｻｰﾊﾞｰ処理､画面表示の各処理時間をﾊﾞｰｲﾒｰｼﾞで
	 * 表示させる機能です｡処理時間の目安になります｡
	 * (初期値:VIEW_USE_TIMEBAR[={@og.value SystemData#VIEW_USE_TIMEBAR}])｡
	 *
	 * @og.rev 6.3.6.0 (2015/08/16) useTimeView の初期値を､VIEW_USE_TIMEBAR にする｡
	 *
	 * @param	flag	処理時間を表示 [true:する/false:しない]
	 */
	public void setUseTimeView( final String flag ) {
		useTimeView = nval( getRequestParameter( flag ),useTimeView );
	}

	/**
	 * 【TAG】ｴﾗｰﾒｯｾｰｼﾞにSLABELを利用するかどうか[true/false]を指定します(初期値:false)｡
	 *
	 * @og.tag
	 * 通常のｴﾗｰﾒｯｾｰｼﾞは､ﾗﾍﾞﾙ（長）が使われますが､これをﾗﾍﾞﾙ（短）を使いたい場合に､true にｾｯﾄします｡
	 * ここでのﾗﾍﾞﾙ（短）は､ﾀｸﾞ修飾なしの､ﾗﾍﾞﾙ（短）です｡
	 * 標準はfalse:利用しない=ﾗﾍﾞﾙ（長）です｡
	 * true/false以外を指定した場合はfalse扱いとします｡
	 *
	 * ﾗﾍﾞﾙﾘｿｰｽの概要説明があれば表示しますが､useSLabel="true" 時は､概要説明を表示しません｡
	 *
	 * @og.rev 7.0.7.0 (2019/12/13) 新規追加
	 *
	 * @param prm SLABEL利用 [true:利用する/false:利用しない]
	 */
	public void setUseSLabel( final String prm ) {
		useSLabel = nval( getRequestParameter( prm ),useSLabel );
	}

	/**
	 * 【TAG】ｼｽﾃﾑ定数でｸﾗｳﾄﾞ設定されていても､ｸﾗｳﾄﾞ環境を使用しない場合､trueを指定します(初期値:false)｡
	 *
	 * @og.tag
	 * ｸﾗｳﾄﾞ設定は､ｼｽﾃﾑ定数の『CLOUD_TARGET』と『CLOUD_BUCKET』の設定で自動的に使用しますが､
	 * どうしてもﾛｰｶﾙでのみ使いたい場合は､この属性を true に設定します｡
	 * 標準はfalse:設定どおりとなります｡
	 *
	 * true/false以外を指定した場合はfalse扱いとします｡
	 *
	 * @og.rev 8.0.1.0 (2021/10/29) useLocal 属性を追加｡
	 *
	 * @param flag ﾛｰｶﾙ環境のみ [true:ﾛｰｶﾙのみ/false:設定どおり]
	 */
	public void setUseLocal( final String flag ) {
		useLocal = nval( getRequestParameter( flag ),useLocal );
	}

	/**
	 * 【TAG】valueﾀｸﾞのaction=MAPOBJ を使用したﾗﾍﾞﾙ変換を行う場合の､MAPOBJｷｰを指定します｡
	 *
	 * @og.tag
	 *
	 * DirectWriteTableTagでは､SQL文から直接ﾌｧｲﾙを作成するため､headLabel はresource から
	 * 取得します｡
	 * 通常の DBTableModel を作成する場合には､ColumnEditorTag でuseLabelMapとしてSQL文でｷｰとﾗﾍﾞﾙの
	 * Mapを作成してﾗﾍﾞﾙの書き換えが行えますが､それと同等の事を行うためには､一旦､valueﾀｸﾞで､
	 * command="SQL" action=MAPOBJ で､ｷｰとﾗﾍﾞﾙのMapを作成して､ﾗﾍﾞﾙとして使用します｡
	 * なお､Mapｵﾌﾞｼﾞｪｸﾄは､scope="request" でのみやり取り可能です｡
	 *
	 * @og.rev 7.0.7.1 (2019/12/24) mapObjKey 属性を追加｡
	 *
	 * @param key valueﾀｸﾞで作成したMAPOBJを取り出すときのｷｰ
	 */
	public void setMapObjKey( final String key ) {
		mapObjKey = nval( getRequestParameter( key ),mapObjKey );
	}

	/**
	 * このｵﾌﾞｼﾞｪｸﾄの文字列表現を返します｡
	 * 基本的にﾃﾞﾊﾞｯｸﾞ目的に使用します｡
	 *
	 * @return このｸﾗｽの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "dbid"		,dbid		)
				.println( "separator"	,separator	)
				.println( "useHeader"	,useHeader	)
				.println( "fileURL"		,fileURL	)
				.println( "filename"	,filename	)
				.println( "sql"			,sql		)
				.println( "encode"		,encode		)
				.println( "fileAppend"	,fileAppend	)
				.println( "zip"			,zip		)
				.println( "zipFilename"	,zipFilename)
				.println( "displayMsg"	,displayMsg	)
				.println( "fetchSize"	,fetchSize	)
				.println( "queryType"	,queryType	)
				.println( "names"		,names		)
				.println( "errCode"		,errCode	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
