/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval;

import java.io.File;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.report2.DocConverter_OOO;
// import org.opengion.hayabusa.io.HybsFileOperationFactory;	// 8.0.0.1 (2021/10/08) , 8.0.1.0 (2021/10/29) 廃止
// import org.opengion.fukurou.model.FileOperation;				// 8.0.0.1 (2021/10/08)
// import org.opengion.fukurou.util.FileUtil;					// 8.0.0.1 (2021/10/08)

/**
 * ﾄﾞｷｭﾒﾝﾄの変換・ﾏｰｼﾞを行うﾀｸﾞです｡
 *
 * 変換を行うことのできる入出力のﾌｫｰﾏｯﾄ以下の通りです｡
 *
 * [対応ﾌｫｰﾏｯﾄ]
 *  入力[Calc(ods)   ,Excel(xls)     ] ⇒ 出力[Calc(ods)   ,Excel(xls,xlsx)      ,PDF]
 *  入力[Writer(odt) ,Word(doc)      ] ⇒ 出力[Writer(odt) ,Word(doc,docx)       ,PDF]
 *  入力[Impress(odp),PowerPoint(ppt)] ⇒ 出力[Impress(odp),PowerPoint(ppt,pptx) ,PDF]
 *
 * ※ xlsx,docx,pptx は､MS 2007形式の為､LibreOffice のみ利用できます｡
 *
 * 入出力のﾌｫｰﾏｯﾄは､入出力ﾌｧｲﾙの拡張子により自動判別されます｡
 * また､入出力ﾌｧｲﾙ名が同じ場合は何も処理されません｡
 *
 * 入力ﾌｧｲﾙを､CSV形式で複数指定することで､複数の入力ﾌｧｲﾙをﾏｰｼﾞして
 * 出力することもできます｡
 *
 * ※2 現状は､ﾌｧｲﾙのﾏｰｼﾞは､入力ﾌｧｲﾙがExcelまたはCalcの場合のみ対応しています｡
 * ※1 この機能を利用するには､OpenOfficeが正しく設定されている必要があります｡
 *
 * @og.formSample
 * ●形式：&lt;og:docConvert fileURL="…" inputFile="…" outputFile="…" /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:docConvert
 *       fileURL            【TAG】操作するﾌｧｲﾙのﾃﾞｨﾚｸﾄﾘを指定します (初期値:FILE_URL[=filetemp/])
 *       inputFile        ○【TAG】入力ﾌｧｲﾙ名を指定します(必須)｡
 *       outputFile       ○【TAG】出力ﾌｧｲﾙ名を指定します(必須)｡
 *       delInput           【TAG】ﾄﾞｷｭﾒﾝﾄ変換後､元のﾌｧｲﾙを削除するかどうかを指定します(初期値:false[=削除しない])
 *       debug              【TAG】ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *    ･Calc(ods)ﾌｧｲﾙをPDFに変換
 *        &lt;og:docConvert inputFile="temp.ods" outputFile="out.pdf" /&gt;
 *
 *    ･Excel(xls)ﾌｧｲﾙを連結
 *        &lt;og:docConvert inputFile="temp1.xls,temp2.xls" outputFile="out.xls" /&gt;
 *
 * @og.group その他部品
 *
 * @version  4.0
 * @author	 Hiroki Nakamura
 * @since    JDK5.0,
 */
public class DocConvertTag extends CommonTagSupport {
	/** このﾌﾟﾛｸﾞﾗﾑのVERSION文字列を設定します｡	{@value} */
	private static final String VERSION = "8.0.1.0 (2021/10/29)" ;
	private static final long serialVersionUID = 801020211029L ;

	private String	fileURL 	= HybsSystem.sys( "FILE_URL" );
	private String[]inputFile	;
	private String	outputFile	;
	private boolean	delInput	;

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰ
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public DocConvertTag() { super(); }		// これも､自動的に呼ばれるが､空のﾒｿｯﾄﾞを作成すると警告されるので､明示的にしておきます｡

	/**
	 * Taglibの開始ﾀｸﾞが見つかったときに処理する doStartTag() を ｵｰﾊﾞｰﾗｲﾄﾞします｡
	 *
	 * @return	後続処理の指示( SKIP_BODY )
	 */
	@Override
	public int doStartTag() {
		return SKIP_BODY ;
	}

	/**
	 * Taglibの終了ﾀｸﾞが見つかったときに処理する doEndTag() を ｵｰﾊﾞｰﾗｲﾄﾞします｡
	 *
	 * @og.rev 8.0.0.2 (2021/10/15) ﾛｰｶﾙﾌｧｲﾙとｸﾗｳﾄﾞﾌｧｲﾙ間の移動
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();

		// 出力ﾌｧｲﾙで拡張子だけが指定された場合は､入力ﾌｧｲﾙ名を付加する
		if( outputFile.indexOf( '.' ) < 0 ) {
			final int inSufIdx = inputFile[0].lastIndexOf( '.' );
			if( inSufIdx >= 0 ) {
				outputFile = inputFile[0].substring( 0, inSufIdx ) + "." + outputFile;
			}
		}

		final String directory = HybsSystem.url2dir( fileURL );
		String[] input = new String[inputFile.length];
		for( int i=0; i<input.length; i++ ) {
			input[i] = StringUtil.urlAppend( directory,inputFile[i] );
		}
		final String output = StringUtil.urlAppend( directory,outputFile );

//		// 8.0.1.0 (2021/10/29) 廃止
//		// 8.0.0.2 (2021/10/15) ﾛｰｶﾙﾌｧｲﾙとｸﾗｳﾄﾞﾌｧｲﾙ間の移動
//		if( !useLocal && HybsFileOperationFactory.useCloud() ) {
//			// 5.10.9.0 (2019/03/01) ADD ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ指定の場合､ﾌｧｲﾙをﾛｰｶﾙにﾀﾞｳﾝﾛｰﾄﾞします｡
//			for( final String in: input ) {
//				HybsFileOperationFactory.cloud2local( false,() -> new File( in ) );		// false:ｼｽﾃﾑ定義のｸﾗｳﾄﾞ設定に従う｡
//			}
//		}
//		final FileOperation inFile = HybsFileOperationFactory.create( input[0] );
//		// 5.10.9.0 (2019/03/01) ADD ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ指定の場合､ﾌｧｲﾙをﾛｰｶﾙにﾀﾞｳﾝﾛｰﾄﾞします｡
//		if( inFile.isCloud() ) {
//			for( final String in: input ) {
//				final File fromFile = HybsFileOperationFactory.create( in );
//				final File toFile = new File( in );
//				toFile.mkdirs();			// toFileのﾃﾞｨﾚｸﾄﾘを作成する必要あり？
//				FileUtil.copy( fromFile, toFile );
//			}
//		}

		// ﾄﾞｷｭﾒﾝﾄ変換
//		DocConverter_OOO.convert( input, output );
		DocConverter_OOO.convert( input, output, true );		//  8.0.3.0 (2021/12/17) ﾒｿｯﾄﾞ削除に伴う処置

//		// 8.0.1.0 (2021/10/29) 廃止
//		// 8.0.0.1 (2021/10/08) cloud対応
//		// 5.10.9.0 (2019/03/01) ADD ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ指定の場合､ﾌｧｲﾙをｸﾗｳﾄﾞｽﾄﾚｰｼﾞにｱｯﾌﾟﾛｰﾄﾞします｡
//		// 8.0.0.2 (2021/10/15) ﾛｰｶﾙﾌｧｲﾙとｸﾗｳﾄﾞﾌｧｲﾙ間の移動
//		HybsFileOperationFactory.local2cloud( useLocal,() -> new File( output ) );
//	//	final FileOperation outFile = HybsFileOperationFactory.create( output );
//	//	if( outFile.isCloud() ) {
//	//		final File localFile = new File( output );
//	//		FileUtil.copy( localFile, outFile );
//	//		localFile.delete();
//	//	}

		if( delInput ) {
			for( int i=0; i<input.length; i++ ) {
				if( !( new File( input[i] ) ).delete() ) {
					System.err.println( "入力ﾌｧｲﾙの削除に失敗しました｡file=[" + input[i] + "]" );
				}
			}
		}

		return EVAL_PAGE ;
	}

	/**
	 * ﾀｸﾞﾘﾌﾞｵﾌﾞｼﾞｪｸﾄをﾘﾘｰｽします｡
	 * ｷｬｯｼｭされて再利用されるので､ﾌｨｰﾙﾄﾞの初期設定を行います｡
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		fileURL 	= HybsSystem.sys( "FILE_URL" );
		inputFile	= null;
		outputFile	= null;
		delInput	= false;
	}

	/**
	 * 【TAG】操作するﾌｧｲﾙのﾃﾞｨﾚｸﾄﾘを指定します
	 *		(初期値:FILE_URL[={@og.value SystemData#FILE_URL}])｡
	 *
	 * @og.tag
	 * この属性で指定されるﾃﾞｨﾚｸﾄﾘのﾌｧｲﾙを操作します｡
	 * 指定方法は､通常の fileURL 属性と同様に､先頭が､'/' (UNIX) または､２文字目が､
	 * ":" (Windows)の場合は､指定のURLそのままのﾃﾞｨﾚｸﾄﾘに､そうでない場合は､
	 * (初期値:ｼｽﾃﾑ定数のFILE_URL[={@og.value SystemData#FILE_URL}])｡
	 *
	 * @og.rev 6.4.2.1 (2016/02/05) URLの最後に､"/" を追加する処理を廃止｡
	 *
	 * @param	url ﾌｧｲﾙURL
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_URL
	 */
	public void setFileURL( final String url ) {
		final String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】入力ﾌｧｲﾙ名を指定します｡
	 *
	 * @og.tag
	 * 入力ﾌｧｲﾙ名を指定します｡
	 *
	 * 入力ﾌｧｲﾙは､CSV形式で複数指定することができます｡
	 * この場合､複数の入力ﾌｧｲﾙをﾏｰｼﾞして出力を行います｡
	 *
	 * ※現状は､ﾌｧｲﾙのﾏｰｼﾞは､入力ﾌｧｲﾙがExcelまたはCalcの場合のみ対応しています｡
	 *   また､ﾏｰｼﾞを行う場合､印刷範囲など､ﾄﾞｷｭﾒﾝﾄに関連する情報は､1つ目のﾌｧｲﾙの
	 *   情報が使用されます｡
	 *
	 * @param	fname 入力ﾌｧｲﾙ名
	 */
	public void setInputFile( final String fname ) {
		inputFile = getCSVParameter( fname );
	}

	/**
	 * 【TAG】出力ﾌｧｲﾙ名を指定します｡
	 *
	 * @og.tag
	 * 出力ﾌｧｲﾙ名を指定します｡
	 * 出力ﾌｧｲﾙには､拡張子のみ(xls,ods等)を指定することもできます｡
	 * この場合､出力ﾌｧｲﾙ名は､入力ﾌｧｲﾙ名と同じになります｡(拡張子のみが変換される)
	 *
	 * @param	fname 出力ﾌｧｲﾙ名
	 */
	public void setOutputFile( final String fname ) {
		outputFile = nval( getRequestParameter( fname ),outputFile );
	}

	/**
	 * 【TAG】ﾄﾞｷｭﾒﾝﾄ変換後､元のﾌｧｲﾙを削除するかどうかを指定します(初期値:false[=削除しない])｡
	 *
	 * @og.tag
	 * ﾄﾞｷｭﾒﾝﾄ変換後､元のﾌｧｲﾙを削除するかどうかを指定します｡
	 * 初期値は､false(削除しない)です｡
	 *
	 * @param	flag 元ﾌｧｲﾙの削除 [true:する/false:しない]
	 */
	public void setDelInput( final String flag ) {
		delInput = nval( getRequestParameter( flag ),delInput );
	}

//	/**
//	 * 【TAG】ｼｽﾃﾑ定数でｸﾗｳﾄﾞ設定されていても､ｸﾗｳﾄﾞ環境を使用しない場合､trueを指定します(初期値:false)｡
//	 *
//	 * @og.tag
//	 * ｸﾗｳﾄﾞ設定は､ｼｽﾃﾑ定数の『CLOUD_TARGET』と『CLOUD_BUCKET』の設定で自動的に使用しますが､
//	 * どうしてもﾛｰｶﾙでのみ使いたい場合は､この属性を true に設定します｡
//	 * 標準はfalse:設定どおりとなります｡
//	 *
//	 * true/false以外を指定した場合はfalse扱いとします｡
//	 *
//	 * @og.rev 8.0.1.0 (2021/10/29) useLocal 属性を追加｡
//	 *
//	 * @param flag ﾛｰｶﾙ環境のみ [true:ﾛｰｶﾙのみ/false:設定どおり]
//	 */
//	public void setUseLocal( final String flag ) {
//		useLocal = nval( getRequestParameter( flag ),useLocal );
//	}

	/**
	 * このｵﾌﾞｼﾞｪｸﾄの文字列表現を返します｡
	 * 基本的にﾃﾞﾊﾞｯｸﾞ目的に使用します｡
	 *
	 * @return このｸﾗｽの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "fileURL"		,fileURL	)
				.println( "inputFile"	,inputFile		)
				.println( "outputFile"	,outputFile		)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
