/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.StringUtil;					// 6.9.8.0 (2018/05/28)
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.util.ArraySet;						// 6.4.3.4 (2016/03/11)
import static org.opengion.fukurou.util.StringUtil.nval;

import java.util.Set;											// 6.4.3.4 (2016/03/11)
import java.util.stream.Stream;									// 6.4.3.4 (2016/03/11)
import java.util.stream.Collectors;								// 6.4.3.4 (2016/03/11)

/**
 * 入力ﾌｨｰﾙﾄﾞやﾎﾞﾀﾝなどを作成するHTML拡張ﾀｸﾞです。
 *
 * 簡易的な入力用ﾀｸﾞとして、columnﾀｸﾞがありますので、通常は、columnﾀｸﾞをお使い下さい。
 * name 属性に ﾗﾍﾞﾙﾘｿｰｽ のｷｰを与えることで、ﾛｹｰﾙにあわせたﾘｿｰｽを
 * 使用して、画面に表示します。
 * ﾛｹｰﾙは、session 登録項目の HybsSystem#LANG_KEY を初期値で使用し、
 * language 属性で定義されている場合は、そちらを優先します。
 *
 * @og.formSample
 * ●形式：&lt;og:input name=… value=… /&gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:input
 *       type               【HTML】表示形式[text/password/checkbox/radio/submit/reset/button/image/file]を指定します(初期値:text)
 *                                  HTML5追加分[search/tel/url/email/datetime/date/month/week/time/datetime-local/number/range/color]
 *       name               【HTML】名前を指定します
 *       value              【HTML】このﾌｨｰﾙﾄﾞに設定する値を指定します
 *       caseKey            【TAG】このﾀｸﾞ自体を利用するかどうかの条件ｷｰを指定します(初期値:null)
 *       caseVal            【TAG】このﾀｸﾞ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ｾﾞﾛ文字列 でない場合(Not Null=NN)は、このﾀｸﾞは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ｾﾞﾛ文字列 の場合は、このﾀｸﾞは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このﾀｸﾞは使用されます(初期値:判定しない)
 *       lbl                【TAG】ﾗﾍﾞﾙﾘｿｰｽのﾗﾍﾞﾙIDを指定します
 *       lbls               【TAG】ﾗﾍﾞﾙをCSV形式で複数指定します
 *       td                 【TAG】ﾃｰﾌﾞﾙ形式の &lt;td&gt; ﾀｸﾞを使用するかどうか[yes/no/false]を指定します(初期値:yes)
 *       colspan            【TAG】ﾌｨｰﾙﾄﾞ部分の colspan を指定します
 *       rowspan            【TAG】ﾗﾍﾞﾙ、ﾌｨｰﾙﾄﾞ共に rowspan を指定します
 *       nextForm           【TAG】入力ｶｰｿﾙを指定されたname属性のﾌｨｰﾙﾄﾞへ自動的に移動します
 *       optionAttributes   【TAG】JavaScript などの HTML基本ﾀｸﾞ以外の属性を、そのままﾀｸﾞとして使用します
 *       must               【TAG】必須入力を表す色に変えるかどうか[true/false]を指定します(初期値:false)
 *       mustAny            【TAG】選択必須入力(どれかひとつ必須)を表す色[true/mustAny/その他]を指定します(初期値:無指定)
 *       useMustHidden      【TAG】必須の自動ﾁｪｯｸ用Hiddenを出力するかどうか[true/false]を指定します(初期値:true)
 *       aimai              【TAG】曖昧検索可能ﾌｨｰﾙﾄﾞとして、曖昧検索方法を指定します(初期値:null)
 *       size               【HTML】横幅を指定します(typeがtextまたはpasswordの場合は文字数で、それ以外はﾋﾟｸｾﾙ)
 *       maxlength          【HTML】最大入力文字数を指定します(typeがtextまたはpasswordの場合)
 *       checked            【HTML】type が checkbox か radio のとき、初めから選択された状態で表示します("checked"のみ指定可)
 *       src                【HTML】type が image の場合、送信ﾎﾞﾀﾝとして利用する画像のURLをｾｯﾄします
 *       alt                【HTML】type が image の場合で、画像が表示できない場合の代替ﾃｷｽﾄをｾｯﾄします
 *       accept             【HTML】type が file の場合に、MIMEﾀｲﾌﾟをCSV形式で複数指定出来ます
 *       usemap             【HTML】type が image の場合、ｸﾗｲｱﾝﾄｻｲﾄﾞ･ｲﾒｰｼﾞﾏｯﾌﾟのURLを指定します
 *       ismap              【HTML】type が image の場合に、ｻｰﾊﾞｰｻｲﾄﾞ･ｲﾒｰｼﾞﾏｯﾌﾟを使用する場合に指定します
 *       id                 【HTML】要素に対して固有の名前(id)をつける場合に設定します
 *       lang               【HTML】要素の内容と他の属性値の言語(lang,xml:lang)を指定します
 *       dir                【HTML】文字表記の方向(dir)を指定します
 *       title              【HTML】要素に対する補足的情報(title)を設定します
 *       style              【HTML】この要素に対して適用させるｽﾀｲﾙｼｰﾄ(style)を設定します
 *       readonly           【TAG】その部品に対して変更が出来ないように(readonly)指定します(ｻｰﾊﾞｰに送信される)
 *       disabled           【TAG】その部品に対して、選択や変更が出来ないように(disabled)指定します(ｻｰﾊﾞｰに送信されない)
 *       tabindex           【HTML】ﾀﾌﾞの移動順(tabindex)を指定します(0 ～ 32767)
 *       accesskey          【HTML】ｱｸｾｽｷｰ(alt+ｷｰで直接指定)を割り当てます
 *       clazz              【HTML】要素に対して class 属性を設定します
 *       language           【TAG】ﾀｸﾞ内部で使用する言語ｺｰﾄﾞ[ja/en/zh/…]を指定します
 *       onClick            【HTML】JavaScriptのｲﾍﾞﾝﾄ onClick を設定します(例:onClick="renew('query.jsp','QUERY');")
 *       onChange           【HTML】JavaScriptのｲﾍﾞﾝﾄ onChange を設定します(例:onChange="renew('query.jsp','QUERY');")
 *       onBlur             【HTML】JavaScriptのｲﾍﾞﾝﾄ onBlur を設定します(例:onBlur="this.value=value.toUpperCase();")
 *       onFocus            【HTML】JavaScriptのｲﾍﾞﾝﾄ onFocus を設定します
 *       ondblClick         【HTML】JavaScriptのｲﾍﾞﾝﾄ ondblClick を設定します
 *       onMouseDown        【HTML】JavaScriptのｲﾍﾞﾝﾄ onMouseDown を設定します
 *       onMouseUp          【HTML】JavaScriptのｲﾍﾞﾝﾄ onMouseUp を設定します
 *       onMouseMove        【HTML】JavaScriptのｲﾍﾞﾝﾄ onMouseMove を設定します
 *       onMouseOut         【HTML】JavaScriptのｲﾍﾞﾝﾄ onMouseOut を設定します
 *       onMouseOver        【HTML】JavaScriptのｲﾍﾞﾝﾄ onMouseOver を設定します
 *       onSelect           【HTML】JavaScriptのｲﾍﾞﾝﾄ onSelect を設定します
 *       onKeydown          【HTML】JavaScriptのｲﾍﾞﾝﾄ onKeydown を設定します
 *       onKeypress         【HTML】JavaScriptのｲﾍﾞﾝﾄ onKeypress を設定します
 *       onKeyup            【HTML】JavaScriptのｲﾍﾞﾝﾄ onKeyup を設定します
 *       autocomplete       【HTML5】入力候補を提示して入力内容を自動補完する[on/off]（初期値はon）。
 *       autofocus          【HTML5】指定した入力欄にｶｰｿﾙが当たって自動的にﾌｫｰｶｽされます。
 *       pattern            【HTML5】正規表現で入力値のﾊﾟﾀｰﾝを指定します。
 *       placeholder        【HTML5】入力欄に初期表示する内容を指定します。
 *       required           【HTML5】入力必須を指定します
 *       list               【HTML5】ﾕｰｻﾞｰに入力候補として提案するﾃﾞｰﾀﾘｽﾄの要素のid属性の値を指定します。
 *       min                【HTML5】入力欄で入力できる最小値を指定します。
 *       max                【HTML5】入力欄で入力できる最大値を指定します。
 *       step               【HTML5】入力欄で刻むｽﾃｯﾌﾟ値を指定する。
 *       roles              【TAG】ﾛｰﾙをｾｯﾄします
 *       useRealTimeCheck	【TAG】(通常は使いません)ﾘｱﾙﾀｲﾑﾁｪｯｸを行うかどうか(初期値:true:行う) 5.9.32.2 (2018/05/18)
 *       debug              【TAG】ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:input&gt;
 *
 * ●使用例
 *    &lt;og:input name="OYA" /&gt;                             lbl属性が指定されていない場合は、name属性をｷｰにしてLabelResourceを読み取ります。
 *    &lt;og:input name="PN" lbl="KO" /&gt;                     lbl属性を指定することで、ﾗﾍﾞﾙだけを付替えることができます。
 *    &lt;og:input name="PN" size="13" maxlength="11" /&gt;     強制的にｻｲｽﾞの変更をするときは、size属性、maxlength属性を使用できます。
 *    &lt;og:input name="OYA" td="false" /&gt;                  自動作成されるtdﾀｸﾞやﾗﾍﾞﾙを外すときはtd属性を使います。
 *    &lt;og:input name="OYA" td="no" /&gt;                     自動作成されるtdﾀｸﾞを外し、ﾗﾍﾞﾙとﾌｨｰﾙﾄﾞは残します。
 *    &lt;og:input name="PN" optionAttributes="onBlur='blr()'
 *                            onFocus='fcs()'" /&gt;
 *                                                               JavaScriptのｲﾍﾞﾝﾄをｺｰﾃﾞｨﾝｸﾞするときは、optionAttributes属性を使います。
 *    &lt;og:input type="radio" name="OYA" /&gt;                ﾗﾍﾞﾙ部分と入力ﾌｨｰﾙﾄﾞ部分がﾃｰﾌﾞﾙﾀｸﾞの&lt;td&gt;により左右に分割されます。
 *
 *    &lt;table&gt;
 *        &lt;tr&gt;&lt;og:input name=&quot;PN&quot; value=&quot;{&#064;PN}&quot; /&gt;&lt;/tr&gt;
 *        &lt;tr&gt;&lt;og:input name=&quot;CD&quot; value=&quot;{&#064;CD}&quot; /&gt;&lt;/tr&gt;
 *    &lt;/table&gt;
 *
 *    &lt;table&gt;
 *        &lt;tr&gt;&lt;og:input name=&quot;PN&quot; value=&quot;{&#064;PN}&quot; &gt;部品入力ﾌｨｰﾙﾄﾞ&lt;/og:input&gt;&lt;/tr&gt;
 *        &lt;tr&gt;&lt;og:input name=&quot;CD&quot; value=&quot;{&#064;CD}&quot; &gt;ｺｰﾄﾞﾌｨｰﾙﾄﾞ&lt;/og:input&gt;&lt;/tr&gt;
 *    &lt;/table&gt;
 *                                                               HTML 表示時は、前後に&lt;tr&gt;ﾀｸﾞで囲って、整形できます。
 *
 *    &lt;og:input name="A"  nextForm="B" /&gt;                最大桁数入力後、ﾌｫｰﾑ B にﾌｫｰｶｽが移動します。
 *    &lt;og:input name="B" /&gt;
 *
 *    BODY 部分に記述した値は、入力ﾌｨｰﾙﾄﾞの直後にｾｯﾄされます。
 *    &lt;og:input name="PN" &gt;
 *      &lt;button type="button" onclick="popup(･･･);return false;"&gt;
 *          &lt;og:message lbl="POPUP" comment="▼" /&gt;
 *      &lt;/button&gt;
 *    &lt;/og:input&gt;
 *
 * @og.group 画面部品
 *
 * @version	4.0
 * @author	Kazuhiko Hasegawa
 * @since	JDK5.0,
 */
public class InputTag extends HTMLTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "8.4.3.0 (2023/03/31)" ;
	private static final long serialVersionUID = 843020230331L ;

	// 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	private static final Set<String> TYPE_SET = new ArraySet<>(
									 "text","password","checkbox","radio","submit","reset","button"
									,"image","file","hidden","search","tel","url","email","datetime"
									,"date","month","week","time","datetime-local","number","range","color" );

	private static final Set<String> TD_SET = new ArraySet<>( "yes" , "no" , "false" );

	/** HTML_LABEL_SEPARATOR を boolean 変数として取得します。 */
	private final String CLM = HybsSystem.sysBool( "HTML_LABEL_SEPARATOR" ) ? ":" : "" ;	// 4.0.0 (2005/01/31)

	/** yes:&lt;td&gt; ﾀｸﾞを使用 / false:ﾌｨｰﾙﾄﾞのみ / no:ﾗﾍﾞﾙとﾌｨｰﾙﾄﾞ */
	private String tdFlag		= "yes";
	/** colspan 属性 */
	private String colspan		= "";											// 3.0.1.4 (2003/03/17)
	/** rowspan 属性 */
	private String rowspan		= "";											// 6.2.0.0 (2015/02/27)
	/** nextForm 属性 */
	private String nextForm		;												// 3.5.4.2 (2003/12/15)

	/** 表示ﾌｨｰﾙﾄﾞの大きさ */
	private final int COLUMNS_MAXSIZE = HybsSystem.sysInt( "HTML_COLUMNS_MAXSIZE" );	// 4.0.0 (2005/01/31)
	/** accesskey の設定の有効(true)/無効(false)を設定します(要再起動) */
	private static final boolean USE_ACCESSKEY = HybsSystem.sysBool( "USE_ACCESSKEY" );	// 8.4.3.0 (2023/03/31)

	/** BODY 部分に記述した値は、入力ﾌｨｰﾙﾄﾞの直後にｾｯﾄされます。 */
	private String body			;												// 4.0.0 (2007/05/30)

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰ
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public InputTag() { super(); }			// これも、自動的に呼ばれるが、空のﾒｿｯﾄﾞを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始ﾀｸﾞが見つかったときに処理する doStartTag() を ｵｰﾊﾞｰﾗｲﾄﾞします。
	 *
	 * @og.rev 4.0.0.0 (2007/05/30) 新規追加(BODY部の評価)
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		// 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		return useTag()
				? EVAL_BODY_BUFFERED		// Body を評価する。( extends BodyTagSupport 時)
				: SKIP_BODY ;				// Body を評価しない
	}

	/**
	 * Taglibのﾀｸﾞ本体を処理する doAfterBody() を ｵｰﾊﾞｰﾗｲﾄﾞします。
	 *
	 * @og.rev 4.0.0.0 (2007/05/30) 新規追加(BODY部の評価)
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		body = getBodyString();
		return SKIP_BODY ;
	}

	/**
	 * ﾀｸﾞﾘﾌﾞｵﾌﾞｼﾞｪｸﾄをﾘﾘｰｽします。
	 * ｷｬｯｼｭされて再利用されるので、ﾌｨｰﾙﾄﾞの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) ｶｽﾀﾑﾀｸﾞの release() ﾒｿｯﾄﾞを、追加
	 * @og.rev 2.0.0.8 (2002/10/09) yes/no/false で指定するように変更
	 * @og.rev 3.0.1.4 (2003/03/17) colspan 属性を追加。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.4.2 (2003/12/15) nextForm 属性を追加。
	 * @og.rev 6.2.0.0 (2015/02/27) rowspan 属性を追加
	 * @og.rev 6.2.4.0 (2015/05/15) body 属性のｸﾘｱが抜けていた。
	 */
	@Override
	protected void release2() {
		super.release2();
		tdFlag		= "yes";		// table形式の <td> ﾀｸﾞを使用するかどうか
		colspan		= "";
		rowspan		= "";			// 6.2.0.0 (2015/02/27) rowspan 属性を追加
		nextForm	= null;			// 3.5.4.2 (2003/12/15)
		body		= null;			// 6.2.4.0 (2015/05/15)
	}

	/**
	 * 検索条件の入力用のﾃｷｽﾄﾌｨｰﾙﾄﾞを作成します。
	 *
	 * ﾃｰﾌﾞﾙﾀｸﾞによりﾌｨｰﾙﾄﾞは分割されます。
	 * 使用時は、ﾃｰﾌﾞﾙﾀｸﾞを前後に使用して下さい。
	 *
	 * @og.rev 2.0.0.8 (2002/10/09) yes/no/false で指定するように変更
	 * @og.rev 2.0.1.0 (2002/10/10) ﾗﾍﾞﾙとﾌｨｰﾙﾄﾞのｾﾊﾟﾚｰﾀｰとして、ｺﾛﾝ(：)を使用するかどうかを指定できる
	 * @og.rev 3.0.1.2 (2003/03/07) forward.jsp の代替用 CommonForwardTag を新規作成
	 * @og.rev 3.1.0.1 (2003/03/26) ｷｬｯｼｭｴﾝﾄﾘｰは、type="submit" の場合のみ、登録する。
	 * @og.rev 3.1.1.0 (2003/03/28) forward.jsp 関係の処理を削除する。
	 * @og.rev 3.1.1.0 (2003/03/28) radio ﾎﾞﾀﾝ等で、ﾗﾍﾞﾙをｸﾘｯｸしても値をｾｯﾄできるようにする。
	 * @og.rev 3.5.4.2 (2003/12/15) 入力ｶｰｿﾙを自動的に次のﾌｨｰﾙﾄﾞへ移動する機能を追加する。
	 * @og.rev 4.0.0.0 (2007/05/30) BODY 部分に記述した値は、入力ﾌｨｰﾙﾄﾞの直後にｾｯﾄされます。
	 * @og.rev 4.3.7.1 (2009/06/08) id=labelのclass化
	 * @og.rev 5.2.1.0 (2010/10/01) must , mustAny 属性を自動化します。
	 * @og.rev 5.6.2.2 (2013/03/15) 自動must処理の出力位置を変更
	 * @og.rev 6.2.0.0 (2015/02/27) rowspan 属性を追加
	 * @og.rev 6.2.0.0 (2015/02/27) aimai 属性を追加。
	 * @og.rev 6.2.4.0 (2015/05/15) aimai 属性は、BODYのincludeより前に追加。
	 * @og.rev 8.2.0.2 (2022/06/24) HTML5廃止対応
	 * @og.rev 8.4.3.0 (2023/03/31) accesskey 有無を設定
	 *
	 * @return	入力用のﾃｷｽﾄﾌｨｰﾙﾄﾞﾀｸﾞ
	 * @og.rtnNotNull
	 */
	@Override
	protected String makeTag() {
		if( get( "value" ) == null ) { set( "value","" ); }

		final String name = get( "name" );							// 6.2.0.0 (2015/02/27) 後で使うので変数化
		if( getMsglbl() == null ) { setLbl( name ); }

		// 8.4.3.0 (2023/03/31) Add
		if( USE_ACCESSKEY ) {
			final String accesskey = get( "accesskey" );
			if( accesskey != null && ! accesskey.isEmpty() ) {
				setLbl( getMsglbl() + "(" + accesskey + ")" );
			}
		}

		// 3.5.4.2 (2003/12/15) 入力ｶｰｿﾙを自動的に次のﾌｨｰﾙﾄﾞへ移動する機能
		if( nextForm != null ) {
			final String onKeyup = "nextForm(this,'" + nextForm + "'," + get( "maxlength" ) + ");" ;
			set( "onKeyup",onKeyup );
		}

		// 6.2.4.0 (2015/05/15) aimai 属性は、BODYのincludeより前に追加。
		if( body == null ) { body = ""; }

		final String type = get( "type" );
		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE );
		// もう少しましな、別の方法を考える必要があります。
		if( "yes".equals( tdFlag ) ) {
			if( type != null && ( "radio".equals( type ) || "checkbox".equals( type ) )) {
				if( get( "id" ) == null ) { set( "id",TaglibUtil.getTagId() ); }

//				rtn.append( "<td align=\"right\"" ).append( rowspan ).append( '>' )				// 6.2.0.0 (2015/02/27)
				rtn.append( "<td style=\"text-align:right;\"" ).append( rowspan ).append( '>' )	// 8.2.0.2 (2022/06/24) Modify
					.append( XHTMLTag.input( getAttributes() ) )
					.append( body )									// 6.2.4.0 (2015/05/15)
					.append( "</td><td" )
					.append( colspan )
					.append( rowspan )								// 6.2.0.0 (2015/02/27)
					.append( "><label for=\"" )
					.append( get( "id" ) )
					.append( "\">" )
					.append( getLongLabel() )						// 4.0.0.0 (2005/01/31)
					.append( "</label></td>" );						// 6.2.4.0 (2015/05/15) aimai 属性は、入らない
			}
			else if( type != null && ( "submit".equals( type ) || "reset".equals( type ) || "button".equals( type ) ) ) {
				set( "value",getMsglbl() );
				rtn.append( "<td" )
					.append( colspan )
					.append( rowspan )											// 6.2.0.0 (2015/02/27)
					.append( '>' )												// 6.0.2.5 (2014/10/31) char を append する。
					.append( XHTMLTag.input( getAttributes() ) )
					.append( body )												// 6.2.4.0 (2015/05/15)
					.append( "</td><td" ).append( rowspan ).append( "></td>" );	// 6.2.4.0 (2015/05/15)
			}
			else {
				// 4.3.7.1 (2009/06/08) id=labelのclass化
				// rtn.append( "<td id=\"label\">" );
				rtn.append( "<td class=\"label\"" )
					.append( rowspan ).append( '>' )				// 6.2.0.0 (2015/02/27)
					.append( getLongLabel() )						// 4.0.0 (2005/01/31)
					.append( CLM )
					.append( "</td><td" )
					.append( colspan )
					.append( rowspan )								// 6.2.0.0 (2015/02/27)
					.append( '>' )									// 6.0.2.5 (2014/10/31) char を append する。
					.append( XHTMLTag.input( getAttributes() ) )
					.append( makeAimaiPicker( name ) )				// 6.2.0.0 (2015/02/27) aimai 属性を追加
					.append( makeMustHidden( name ) )				// 6.2.0.0 (2015/02/27) , 5.6.2.2 (2013/03/15)
					.append( body )									// 6.2.4.0 (2015/05/15)
					.append( "</td>" );
			}
		}
		else if( "no".equals( tdFlag ) ) {
			if( type != null && ( "radio".equals( type ) || "checkbox".equals( type ) )) {
				rtn.append( "<label>" )
					.append( XHTMLTag.input( getAttributes() ) )
					.append( body )									// 6.2.4.0 (2015/05/15)
				// 4.3.7.1 (2009/06/08) id=labelのclass化
				// rtn.append( "<span id=\"label\">" );
					.append( "<span class=\"label\">" )
					.append( getLongLabel() )						// 4.0.0 (2005/01/31)
					.append( "</span></label>" );
			}
			else if( type != null && ( "submit".equals( type ) || "reset".equals( type ) || "button".equals( type ) ) ) {
				set( "value",getMsglbl() );
				rtn.append( XHTMLTag.input( getAttributes() ) )
					.append( body );								// 6.2.4.0 (2015/05/15)
			}
			else {
				// 4.3.7.1 (2009/06/08) id=labelのclass化
				// rtn.append( "<span id=\"label\">" );
				rtn.append( "<span class=\"label\">" )
					.append( getLongLabel() )		// 4.0.0 (2005/01/31)
					.append( CLM )
					.append( "</span>" )
					.append( XHTMLTag.input( getAttributes() ) )
					.append( makeAimaiPicker( name ) )				// 6.2.0.0 (2015/02/27) aimai 属性を追加
					.append( makeMustHidden( name ) )				// 6.2.0.0 (2015/02/27) , 5.6.2.2 (2013/03/15)
					.append( body );								// 6.2.4.0 (2015/05/15)
			}
		}
		// "flase".equals( tdFlag ) の条件
		else {
			if( type != null && ( "submit".equals( type ) || "reset".equals( type ) || "button".equals( type ) ) ) {
				set( "value",getMsglbl() );
				rtn.append( XHTMLTag.input( getAttributes() ) )
					.append( body );								// 6.2.4.0 (2015/05/15)
			}
			else {
				rtn.append( XHTMLTag.input( getAttributes() ) )
					.append( makeAimaiPicker( name ) )				// 6.2.0.0 (2015/02/27) aimai 属性を追加
					.append( makeMustHidden( name ) )				// 6.2.0.0 (2015/02/27) , 5.6.2.2 (2013/03/15)
					.append( body );								// 6.2.4.0 (2015/05/15)
			}
		}

		return rtn.toString() ;
	}

	/**
	 * 【HTML】名前を指定します。
	 *
	 * @og.tag
	 *  &lt;og:input name="PN" /&gt;
	 *
	 * @param	name	名前
	 */
	public void setName( final String name ) {
		set( "name",getRequestParameter( name ) );
	}

	/**
	 * 【HTML】表示形式を指定します(初期値:text)。
	 *
	 * @og.tag
	 * 通常は、表示形式[text/password/checkbox/radio/submit/reset/button/image/file]を指定します。
	 * HTML5 で追加された形式[search/tel/url/email/datetime/date/month/week/time/datetime-local/number/range/color]
	 * も、設定できます。
	 *
	 * HTML5 の機能を有効にするには、ﾈｲﾃｨﾌﾞﾓｰﾄﾞで動作させる必要があります。
//	 *   ① USE_IE7_HEADER = "false" に設定する。   6.9.5.0 (2018/04/23) 廃止(false固定)
//	 *   ② USE_HTML5_HEADER = "true" に設定する。  6.9.5.0 (2018/04/23) 廃止(true固定)
	 *   ③ IEのﾂｰﾙ⇒互換表示設定で、互換表示に追加したWebｻｲﾄから削除する。
	 *   ④ 同上の設定で、ｲﾝﾄﾗｻｲﾄを互換表示で表示するのﾁｪｯｸを外す。
	 * 必要があります。
	 *
	 * &lt;og:input type="text"				/&gt; ﾃｷｽﾄﾎﾞｯｸｽを表示
	 * &lt;og:input type="password"			/&gt; ﾊﾟｽﾜｰﾄﾞ専用ﾃｷｽﾄﾎﾞｯｸｽを表示
	 * &lt;og:input type="checkbox"			/&gt; ﾁｪｯｸﾎﾞｯｸｽを表示
	 * &lt;og:input type="radio"			/&gt; ﾗｼﾞｵﾎﾞﾀﾝを表示
	 * &lt;og:input type="submit"			/&gt; 送信実行ﾎﾞﾀﾝを表示
	 * &lt;og:input type="reset"			/&gt; 入力したﾌｨｰﾙﾄﾞ内容を全てﾘｾｯﾄ
	 * &lt;og:input type="button"			/&gt; ﾎﾞﾀﾝを表示
	 * &lt;og:input type="image"			/&gt; ｲﾒｰｼﾞを表示してｸﾘｯｸした座標を返す
	 * &lt;og:input type="file"				/&gt; ﾌｧｲﾙ名入力ﾎﾞｯｸｽを表示
	 * &lt;og:input type="hidden"			/&gt; 表示させずにｻｰﾊﾞｰに送信するﾃﾞｰﾀ
	 *
	 * HTML5 で追加されたﾀｲﾌﾟ
	 * &lt;og:input type="search"			/&gt; 検索ﾃｷｽﾄの入力欄を作成する
	 * &lt;og:input type="tel"				/&gt; 電話番号の入力欄を作成する
	 * &lt;og:input type="url"				/&gt; URLの入力欄を作成する
	 * &lt;og:input type="email"			/&gt; ﾒｰﾙｱﾄﾞﾚｽの入力欄を作成する
	 * &lt;og:input type="datetime"			/&gt; UTC（協定世界時）による日時の入力欄を作成する
	 * &lt;og:input type="date"				/&gt; 日付の入力欄を作成する
	 * &lt;og:input type="month"			/&gt; 月の入力欄を作成する
	 * &lt;og:input type="week"				/&gt; 週の入力欄を作成する
	 * &lt;og:input type="time"				/&gt; 時間の入力欄を作成する
	 * &lt;og:input type="datetime-local"	/&gt; UTC（協定世界時）によらないﾛｰｶﾙ日時の入力欄を作成する
	 * &lt;og:input type="number"			/&gt; 数値の入力欄を作成する
	 * &lt;og:input type="range"			/&gt; ﾚﾝｼﾞの入力欄を作成する
	 * &lt;og:input type="color"			/&gt; 色の入力欄を作成する
	 *
	 * @og.rev 5.7.1.0 (2013/12/06) HTML5 対応
	 * @og.rev 6.3.4.0 (2015/08/01) Arrays.toString から String.join に置き換え。
	 * @og.rev 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	 *
	 * @param	type	表示形式 [text/password/checkbox/radio/submit/reset/button/image/file]、HTML5用 [search/tel/url/email/datetime/date/month/week/time/datetime-local/number/range/color]
	 */
	public void setType( final String type ) {
		final String tp = getRequestParameter( type );

		if( check( tp , TYPE_SET ) ) {
			set( "type",tp );
		}
		else {
			final String errMsg = "type 属性は、下記の中から選択してください。type=["
						 + tp + "] in [" + String.join( ", " , TYPE_SET ) + "]";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【HTML】このﾌｨｰﾙﾄﾞに設定する値を指定します。
	 *
	 * @og.tag
	 * type属性の値が"radio"か"checkbox"以外の場合、この属性は省略可能です。
	 *&lt;og:input name="PN" value="syokichi" /&gt;
	 *
	 * @param	value	初期値
	 */
	public void setValue( final String value ) {
		set( "value",getRequestParameter( value ) );
	}

	/**
	 * 【HTML】横幅を指定します(typeがtextまたはpasswordの場合は文字数で、それ以外はﾋﾟｸｾﾙ)。
	 *
	 * @og.tag
	 *  &lt;og:input name="PN" size="20" /&gt;
	 *
	 * @param	size	横幅
	 */
	public void setSize( final String size ) {
		set( "size",getRequestParameter( size ) );
	}

	/**
	 * 【HTML】最大入力文字数を指定します(入力可能文字数で、ﾊﾞｲﾄ数ではありません)。
	 *
	 * @og.tag
	 * 最大文字数はsize指定を超え得るが、超えた場合ﾕｰｻﾞｴｰｼﾞｪﾝﾄはｽｸﾛｰﾙ機構を提供する必要があります。
	 * この属性の初期値は｢無制限｣です。
	 * &lt;og:input name="PN" maxlength="20" /&gt;
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 最大入力制限数に応じたｻｲｽﾞ自動生成を入れます。
	 * @og.rev 3.5.4.5 (2004/01/23) CSSで処理する場合のｸﾗｽ属性は、maxlength が ５ 以下の場合とする。
	 * @og.rev 4.0.0.0 (2005/01/31) getFieldSize ﾒｿｯﾄﾞを XHTMLTag から DBColumn へ移動
	 * @og.rev 6.2.0.0 (2015/02/27) 小さなｶﾗﾑのｻｲｽﾞ指定は、すでに廃止 ("S0" + 桁数(1～5))
	 *
	 * @param	maxlength	最大入力文字数
	 */
	public void setMaxlength( final String maxlength ) {
		final String mlen = nval( getRequestParameter( maxlength ),null ) ;
		set( "maxlength",mlen );
		if( get( "size" ) == null && mlen != null ) {
			// 4.0.0 (2005/01/31) getFieldSize ﾒｿｯﾄﾞを XHTMLTag から DBColumn へ移動
			int size = Integer.parseInt( mlen );
			if( size > COLUMNS_MAXSIZE ) { size = COLUMNS_MAXSIZE; }
			set( "size",String.valueOf( size ) );
		}
	}

	/**
	 * 【HTML】type が checkbox か radio のとき、初めから選択された状態で表示します("checked"のみ指定可)。
	 *
	 * @og.tag
	 *
	 * &lt;og:input name="PN" type="checkbox" checked="checked" /&gt;
	 *
	 * @param	ch	選択された状態にするかどうか [checked:選択状態/その他:何もしない]
	 */
	public void setChecked( final String ch ) {
		final String checked = getRequestParameter( ch );
		if( "checked".equals( checked )) {
			set( "checked","checked" );
		}
	}

	/**
	 * 【HTML】type が image の場合、送信ﾎﾞﾀﾝとして利用する画像のURLをｾｯﾄします。
	 *
	 * @og.tag src ｾｯﾄ
	 *
	 * @param	src	画像のURL
	 */
	public void setSrc( final String src ) {
		set( "src",getRequestParameter( src ) );
	}

	/**
	 * 【HTML】画像が表示されない場合、代わりに表示させるﾃｷｽﾄを指定します。
	 *
	 * @og.tag
	 * type が、image の場合に、画像が表示されない場合、代わりに表示させるﾃｷｽﾄを指定します。
	 *
	 * @param	alt	代替ﾃｷｽﾄ
	 */
	public void setAlt( final String alt ) {
		set( "alt",getRequestParameter( alt ) );
	}

	/**
	 * 【HTML】type が file の場合に、MIMEﾀｲﾌﾟをCSV形式で複数指定出来ます。
	 *
	 * @og.tag accept ｾｯﾄ
	 *
	 * @param	accept	MIMEﾀｲﾌﾟ(CSV形式)
	 */
	public void setAccept( final String accept ) {
		set( "accept",getRequestParameter( accept ) );
	}

	/**
	 * 【HTML】ｸﾗｲｱﾝﾄｻｲﾄﾞｸﾘｯｶﾌﾞﾙを指定します。
	 *
	 * @og.tag
	 * type が image の場合、usemapの属性を設定します。
	 *
	 * @param	usemap	ｸﾗｲｱﾝﾄｻｲﾄﾞ・ｲﾒｰｼﾞﾏｯﾌﾟのURL
	 */
	public void setUsemap( final String usemap ) {
		set( "usemap",getRequestParameter( usemap ) );
	}

	/**
	 * 【HTML】ｻｰﾊﾞｰｻｲﾄﾞ・ｲﾒｰｼﾞﾏｯﾌﾟを使用する場合に指定します。
	 *
	 * @og.tag
	 * type が image の場合に、ｻｰﾊﾞｰｻｲﾄﾞ・ｲﾒｰｼﾞﾏｯﾌﾟを使用する ismap 属性を設定します。
	 *
	 * @param	ismap	ｻｰﾊﾞｰｻｲﾄﾞ・ｲﾒｰｼﾞﾏｯﾌﾟ使用可否 [true:する/false:しない]
	 */
	public void setIsmap( final String ismap ) {
		set( "ismap",getRequestParameter( ismap ) );
	}

	/**
	 * 【TAG】ﾃｰﾌﾞﾙ形式の &lt;td&gt; ﾀｸﾞを使用するかどうか[yes/no/false]を指定します(初期値:yes)。
	 *
	 * @og.tag
	 * 通常(初期値:yes)は、ﾃｰﾌﾞﾙ形式の&lt;td&gt; ﾀｸﾞを使用して、ﾗﾍﾞﾙとﾌｨｰﾙﾄﾞを整列させます。
	 * no の場合は、tdﾀｸﾞを使用せず、ﾗﾍﾞﾙとﾌｨｰﾙﾄﾞを連続して出力します。
	 * false は、ﾗﾍﾞﾙを出力せず、ﾌｨｰﾙﾄﾞのみ出力します。
	 *  &lt;og:input name="PN" td="false" /&gt;
	 *
	 * @og.rev 2.0.0.8 (2002/10/09) yes/no/false で指定するように変更
	 * @og.rev 6.3.4.0 (2015/08/01) Arrays.toString から String.join に置き換え。
	 * @og.rev 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	 *
	 * @param	flag	tdﾀｸﾞ使用 [yes:tdﾀｸﾞを使用/no:ﾗﾍﾞﾙとﾌｨｰﾙﾄﾞ/false:ﾌｨｰﾙﾄﾞのみ]
	 */
	public void setTd( final String flag ) {
		final String td = nval( getRequestParameter( flag ),tdFlag );

		if( check( td , TD_SET ) ) {
			tdFlag = td;
		}
		else {
			final String errMsg = "Error in SelectTag [td]: "
						 + td + " in [" + String.join( ", " , TD_SET ) + "] only used.";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】ﾗﾍﾞﾙをCSV形式で複数指定します。
	 *
	 * @og.tag
	 * ｼﾝｸﾞﾙﾗﾍﾞﾙの lbl 属性との違いは、ここではﾗﾍﾞﾙを複数 CSV形式で
	 * 渡すことが可能であることです。これにより、"A,B,C" という値に対して、
	 * "Aのﾗﾍﾞﾙ表示,Bのﾗﾍﾞﾙ表示,Cのﾗﾍﾞﾙ表示" という具合に文字列を
	 * 再合成します。
	 * これは、SQL文のOrder By 句で、ｿｰﾄ順を指定する場合などに便利です。
	 *
	 *  &lt;og:orderBy columns="MKNMJP,MKCD,MKNMEN" lbls="MKNMJP,MKCD,MKNMEN" /&gt;
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 先に配列に分解してからﾘｸｴｽﾄ変数の値を取得
	 * @og.rev 5.2.2.0 (2010/11/01) setMsglbl 廃止 ⇒ setLbl に置換え
	 * @og.rev 6.4.3.4 (2016/03/11) CSV形式の文字連結を、stream 経由で行います。
	 *
	 * @param	lbls	複数ﾗﾍﾞﾙID(CSV形式)
	 */
	public void setLbls( final String lbls ) {

		final String[] array = getCSVParameter( lbls );

		final String lblCSV = Stream.of( array )
								.map( lbl -> getLabel( lbl ) )
								.collect( Collectors.joining( "," ) );

		setLbl( lblCSV );		// 5.2.2.0 (2010/11/01) setMsglbl 廃止 ⇒ setLbl に置換え
	}

	/**
	 * 【TAG】ﾌｨｰﾙﾄﾞ部分の colspan を指定します。
	 *
	 * @og.tag
	 * 通常は、ﾗﾍﾞﾙとﾌｨｰﾙﾄﾞが、別のtdで囲われます。しかし、場合によっては、
	 * ﾌｨｰﾙﾄﾞ部の長いｶﾗﾑがある場合、上下のｶﾗﾑとの位置関係がおかしくなり、
	 * ﾗﾍﾞﾙとﾌｨｰﾙﾄﾞが離れて表示されるｹｰｽがあります。
	 * この場合、長いﾌｨｰﾙﾄﾞ部を持つｶﾗﾑに、colspan 属性を指定すると、
	 * その分のｶﾗﾑを割り当てることにより、上下のｶﾗﾑとの位置関係を
	 * 調節することが可能になります。
	 * 通常は、3 を指定して、tdが、ﾌｨｰﾙﾄﾞ、ﾗﾍﾞﾙ、ﾌｨｰﾙﾄﾞの
	 * 組み合わせ部分に、長いﾌｨｰﾙﾄﾞが適用されるように使用します。
	 *
	 *  &lt;og:input name="PN" colspan="3" /&gt;
	 *
	 * @og.rev 3.0.1.4 (2003/03/17) colspan 属性を追加。
	 *
	 * @param	sp	ﾌｨｰﾙﾄﾞ部分のcolspan
	 */
	public void setColspan( final String sp ) {
		colspan = nval( getRequestParameter( sp ),colspan );
		if( colspan.length() > 0 ) {
			colspan = " colspan=\"" + colspan + "\" ";
		}
	}

	/**
	 * 【TAG】ﾗﾍﾞﾙ、ﾌｨｰﾙﾄﾞ共に rowspan を指定します。
	 *
	 * @og.tag
	 * 通常は、ﾗﾍﾞﾙとﾌｨｰﾙﾄﾞが、別のtdで囲われます。rowspan は、
	 * そのどちらの TDﾌｨｰﾙﾄﾞにも、設定します。
	 * 初期値は、rowspan を付けないです。
	 *
	 * &lt;og:input name="PN" rowspan="3" /&gt;
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) rowspan 属性を追加
	 *
	 * @param	sp	ﾗﾍﾞﾙ、ﾌｨｰﾙﾄﾞ部分のrowspan
	 */
	public void setRowspan( final String sp ) {
		rowspan = nval( getRequestParameter( sp ),rowspan );
		if( rowspan.length() > 0 ) {
			rowspan = " rowspan=\"" + rowspan + "\" ";
		}
	}

	/**
	 * 【TAG】入力ｶｰｿﾙを、指定されたname属性のﾌｨｰﾙﾄﾞへ自動的に移動させます。
	 *
	 * @og.tag
	 * JavaScript の onKeyup ｲﾍﾞﾝﾄを利用して、最大桁数まで入力すると、
	 * 次のﾌｨｰﾙﾄﾞに、自動的に飛ぶように設定します。
	 * 引数は、次に飛ばすﾌｨｰﾙﾄﾞの name 属性です。
	 * 実行するには、JavaScript の nextForm ﾌｧﾝｸｼｮﾝが必要です。(default.js で指定)
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @param	name	次にﾌｫｰｶｽを移すﾀｸﾞのname属性
	 */
	public void setNextForm( final String name ) {
		nextForm = nval( getRequestParameter( name ),nextForm );
	}

	/**
	 * 【TAG】ﾘｱﾙﾀｲﾑﾁｪｯｸを行うかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * ｶﾗﾑ単位でﾘｱﾙﾀｲﾑﾁｪｯｸを行うかどうかを設定をします。
	 * ﾀｸﾞに独自属性としてrealTimeCheck="true/false"を追記します。
	 * falseを設定した場合にﾁｪｯｸが行われなくなります。
	 * 初期値はﾘｱﾙﾀｲﾑﾁｪｯｸを行う（true）です。
	 * 
	 * この属性は、optionAttributesへの設定と同様の動作を行います。
	 * 引数の文字列の、true/false 判定を行っていません。そのままの文字列が設定されます。
	 * JavaScript 側では、false 判定のみ行っているので、不正な文字列の場合は、
	 * 初期値(true:ﾘｱﾙﾀｲﾑﾁｪｯｸを行う)が適用されます。
	 * これは、Ver5 とﾛｼﾞｯｸを合わせておくための処置です。
	 *
	 * @og.rev 5.9.32.2 (2018/05/18) 新規追加
	 * @og.rev 6.9.8.0 (2018/05/28) Ver5 とﾛｼﾞｯｸを合わせます。
	 *
	 * @param	flag	ﾘｱﾙﾀｲﾑﾁｪｯｸを行うかどうか [true:行う/false:行わない]
	 */
	public void setUseRealTimeCheck( final String flag ) {
		// optionAttributes扱いで登録します。
		final String rtcf = getRequestParameter( flag );

		if( StringUtil.isNotNull( rtcf ) ){
			add( "optionAttributes", "realTimeChk=\"" + rtcf + "\"" );			// 連結時のｽﾍﾟｰｽは、Attributes ｸﾗｽで処理
		}

//		// optionAttributes扱いで登録します。
//		final boolean useRtc = nval( getRequestParameter( flag ) , true );
//
//		if( !useRtc ) {		// true 以外の場合のみ、optionAttributes に属性を追加します。
//			add( "optionAttributes", "realTimeChk=\"false\"" );			// 連結時のｽﾍﾟｰｽは、Attributes ｸﾗｽで処理
//		}
	}

	/**
	 * このｵﾌﾞｼﾞｪｸﾄの文字列表現を返します。
	 * 基本的にﾃﾞﾊﾞｯｸﾞ目的に使用します。
	 *
	 * @return	このｸﾗｽの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"					,VERSION			)
				.println( "tdFlag"					,tdFlag				)
				.println( "colspan"					,colspan			)
				.println( "rowspan"					,rowspan			)
				.println( "nextForm"				,nextForm			)
				.println( "HTML_LABEL_SEPARATOR"	,CLM				)
				.println( "HTML_COLUMNS_MAXSIZE"	,COLUMNS_MAXSIZE	)
				.println( "USE_ACCESSKEY"			,USE_ACCESSKEY		)
				.println( "Other..."				,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
